# -*- coding: utf-8 -*-
#
#  ghostmanager.py - The Manager of Ghosts
#  Copyright (C) 2004 by Takuya KAWAHARA <num@sann.ne.jp>
#  Copyright (C) 2004 by Atzm WATANABE <sitosito@p.chan.ne.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: ghostmanager.py,v 1.30 2009/12/13 19:06:08 atzm Exp $
#

__version__ = '$Revision: 1.30 $'
__author__ = '$Author: atzm $'

import os, sys, time
import re, string

from ghost import Ghost
from viewercommon import *

def _debugprint(fmt, *args):
	if __debug__:
		try:
			print >> sys.stderr, fmt % args
		except:
			pass

def _unicoding(text):
	try:
		return unicode(text, 'utf-8')
	except:
		pass
	return text

def create_ghost(path):
	if not path:
		return None
	if not os.path.isabs(path):
		return None
	if not os.path.isfile(path):
		return None
	return Ghost(path)

class GhostManager:
	def __init__(self, ghost_text):
		self.ghost_text = ghost_text                  # abspath to ~/.bottlecase/gviewer/ghost.txt
		self.basedir    = os.path.dirname(ghost_text) # abspath to ~/.bottlecase/gviewer

		self.ghost_dict      = OrderedDictionary()    # key: path
		self.ghost_dict_r    = OrderedDictionary()    # key: sakura_name
		self.fmo_ghost_names = []

		if os.path.exists(self.ghost_text):
			self.load_ghost_text()
		else:
			try:
				os.makedirs(self.get_basedir())
			except OSError:
				pass
			self.save_ghost_text([])

	def get_basedir(self):
		return self.basedir

	def get_ghosts(self):
		return self.ghost_dict

	def get_fmo_ghostnames(self):
		return self.fmo_ghost_names

	def search_ghost(self, ifghost):
		ifghost = _unicoding(ifghost)
		try:
			if self.ghost_dict_r[ifghost][0]:
				return self.ghost_dict_r[ifghost]
		except KeyError:
			t = time.strftime("%y/%m/%d %H:%M:%S", time.localtime(time.time()))
			_debugprint('[%s] search_ghost: no such ghost "%s"', t, ifghost.encode('euc-jp', 'replace'))
		return None

	def nominate_ghost(self, ifghost):
		data = self.search_ghost(ifghost)
		if data is None:
			return None
		return self._get_ghost(expand_quote(data[3]))

	def load_ghost_text(self):
		file_ = None
		try:
			file_ = open(self.ghost_text)
		except IOError:
			pass
		if not file_:
			return False

		self.ghost_dict.clear()
		self.ghost_dict_r.clear()
		self.fmo_ghost_names = []

		for line in file_:
			if not line:
				break
			if line[:4] == '/EOF':
				break
			if line[:7] == 'INSTALL':
				continue

			line = line.strip()
			if not line:
				continue

			items = line.split(',', 3)
			if len(items) != 4:
				_debugprint('invalid line: %s', line)
				continue

			[define, ghostname, fmo, path] = items
			if define[:5] == 'GHOST':
				define = True
			else:
				define = False

			if fmo[:3] == "FMO":
				fmo = True
			else:
				fmo = False

			path = os.path.normpath(path.replace('\\', r'/'))
			ghostname = unicode(ghostname, 'sjis', 'replace')
			self.ghost_dict[path] = [define, ghostname, fmo, path]

			if define:
				ghost  = self._get_ghost(path)
				sakura = ghost.get_sakura()
				self.ghost_dict_r[sakura] = self.ghost_dict[path]

				if fmo:
					self.fmo_ghost_names.append(sakura)

		return True

	# gtk.ListStore can use as the List type
	def save_ghost_text(self, list_):
		tmpfile = '%s.%u' % (self.ghost_text, os.getpid())
		file_   = None
		try:
			file_ = open(tmpfile, 'w')
		except IOError:
			pass
		if not file_:
			return False

		self.ghost_dict.clear()
		self.ghost_dict_r.clear()
		self.fmo_ghost_names = []

		file_.write('INSTALL,,,"Ghost\\"\r\n')

		for i in xrange(len(list_)):
			[define, name, fmo, path] = list_[i]
			name = _unicoding(name)
			path = _unicoding(path)

			self.ghost_dict[path] = [define, name, fmo, path]

			if define:
				ghost  = self._get_ghost(path)
				sakura = ghost.get_sakura()
				self.ghost_dict_r[sakura] = self.ghost_dict[path]

				if fmo:
					self.fmo_ghost_names.append(sakura)

			if define:
				define = 'GHOST'
			else:
				define = '#GHOST'

			if fmo:
				fmo = 'FMO'
			else:
				fmo = ''

			path = path.replace('/', '\\')
			line = ','.join([define, name, fmo, path]).encode('sjis')
			file_.write(line + '\r\n')

		file_.write('\r\n')
		file_.write('/EOF\r\n')
		file_.close()

		try:
			os.rename(tmpfile, self.ghost_text)
		except OSError:
			os.remove(tmpfile)
			_debugprint('could not write %s', self.ghost_text)
			return False

		return True

	def _get_ghost(self, path):
		fullpath = expand_quote(path)
		if not os.path.isabs(fullpath):
			fullpath = os.path.join(self.get_basedir(), fullpath)
		return create_ghost(fullpath)


# from Python Cookbook
#   URL: http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/107747
class OrderedDictionary(dict):
	def __init__(self, dictionary={}):
		self._keys = []
		dict.__init__(self, dictionary)

	def __delitem__(self, key):
		dict.__delitem__(self, key)
		self._keys.remove(key)

	def __setitem__(self, key, item):
		dict.__setitem__(self, key, item)
		if key not in self._keys:
			self._keys.append(key)

	def copy(self):
		newdict = OrderedDictionary()
		newdict.update(self)
		return newdict

	def clear(self):
		dict.clear(self)
		self._keys[:] = []

	def items(self):
		return zip(self._keys, self.values())

	def keys(self):
		return self._keys[:]

	def popitem(self):
		try:
			key = self._keys[-1]
		except IndexError:
			raise KeyError('dictionary is empty')

		val = self[key]
		del self[key]

		return (key, val)

	def setdefault(self, key, failobj=None):
		dict.setdefault(self, key, failobj)
		if key not in self._keys:
			self._keys.append(key)

	def update(self, dictionary):
		for (key, val) in dictionary.items():
			self.__setitem__(key, val)

	def values(self):
		return map(self.get, self._keys)
