unit backpack;
	{ This unit handles both the inventory display and the }
	{ FieldHQ interface, which uses many of the same things. }
{
	GearHead: Arena, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}

interface

uses
{$IFDEF PATCH_GH}
	gears_base,
{$ENDIF PATCH_GH}
	gears,locale
{$IFDEF SDLMODE}
  {$IFDEF PATCH_CHEAT}
	,SDL
  {$ENDIF PATCH_CHEAT}
	,sdlgfx
{$ENDIF SDLMODE}
	;

const
	TRIGGER_GetItem = 'GET';

{$IFDEF SDLMODE}
Procedure SelectColors( M: GearPtr; Redrawer: RedrawProcedureType );
Procedure SelectSprite( M: GearPtr; Redrawer: RedrawProcedureType );
{$ENDIF}

Procedure GivePartToPC( GB: GameBoardPtr; Part, PC: GearPtr );

Function SelectRobotParts( GB: GameBoardPtr; PC: GearPtr ): GearPtr;

Procedure DoFieldRepair( GB: GameBoardPtr; PC , Item: GearPtr; Skill: Integer );

Function Handless( Mek: GearPtr ): Boolean;
Function ShakeDown( GB: GameBoardPtr; Part: GearPtr; X,Y: Integer ): LongInt;
{$IFDEF PATCH_GH}
Procedure InstallAmmo( GB: GameBoardPtr; PC , Gun , Ammo: GearPtr );
Function NeedAmmo( Weapon: GearPtr ): Boolean;
Function ReloadAmmo( GB: GameBoardPtr; User,Weapon: GearPtr ): Boolean;
{$ENDIF PATCH_GH}
Procedure PCGetItem( GB: GameBoardPtr; PC: GearPtr );
Procedure StartContinuousUseItem( GB: GameBoardPtr; TruePC , Item: GearPtr );

Procedure FHQ_SelectMechaForPilot( GB: GameBoardPtr; NPC: GearPtr );
Procedure LancemateBackpack( GB: GameBoardPtr; PC,NPC: GearPtr );
Procedure BackpackMenu( GB: GameBoardPtr; PC: GearPtr; StartWithInv: Boolean );
{$IFDEF PATCH_CHEAT}
Procedure RealBackpack( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr; StartWithInv: Boolean );
Procedure MechaPartEditor( GB: GameBoardPtr; var LList: GearPtr; PC,Mek: GearPtr );
{$ENDIF PATCH_CHEAT}

{$IFDEF PATCH_GH}
  {$IFDEF SDLMODE}
Procedure MechaPartBrowser( Mek: GearPtr; DebugMode: Boolean; RDP: RedrawProcedureType );
Procedure MechaPartBrowser( Mek: GearPtr; RDP: RedrawProcedureType );
  {$ELSE}
Procedure MechaPartBrowser( Mek: GearPtr; DebugMode: Boolean );
Procedure MechaPartBrowser( Mek: GearPtr );
  {$ENDIF}
{$ELSE PATCH_GH}
  {$IFDEF SDLMODE}
Procedure MechaPartBrowser( Mek: GearPtr; RDP: RedrawProcedureType );
  {$ELSE}
Procedure MechaPartBrowser( Mek: GearPtr );
  {$ENDIF}
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
Procedure SelectPortrait( M: GearPtr );
Procedure Rename_Mecha( GB: GameBoardPtr; NPC: GearPtr );
{$IFDEF SDLMODE}
Function SwapMenu_NoParent( var FirstPart: GearPtr; Z: TSDL_Rect; Part: GearPtr ):Boolean;
{$ELSE SDLMODE}
Function SwapMenu_NoParent( var FirstPart: GearPtr; Z: Integer; Part: GearPtr ):Boolean;
{$ENDIF SDLMODE}
{$IFDEF SDLMODE}
Function SwapMenu( Z: TSDL_Rect; Part: GearPtr ):Boolean;
{$ELSE SDLMODE}
Function SwapMenu( Z: Integer; Part: GearPtr ):Boolean;
{$ENDIF SDLMODE}
{$ENDIF PATCH_CHEAT}

Procedure FHQ_ThisWargearWasSelected( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr );


implementation

uses
{$IFDEF DEBUG}
	errmsg,
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
	pseudosmartpointer,
{$ENDIF PATCH_GH}
{$IFDEF PATCH_I18N}
	i18nmsg,
{$ENDIF PATCH_I18N}
{$IFDEF PATCH_CHEAT}
	ui4gh,
	pcaction,
{$ENDIF PATCH_CHEAT}
	ability,action,arenacfe,arenascript,damage,gearutil,ghchars,ghholder,
	ghmodule,ghprop,ghswag,interact,menugear,rpgdice,skilluse,texutil,
{$IFDEF SDLMODE}
	sdlinfo,sdlmap,sdlmenus,
{$ELSE}
	congfx,coninfo,conmap,conmenus,context,
{$ENDIF}
	ghweapon;

var
	ForceQuit: Boolean;
	EqpRPM,InvRPM: RPGMenuPtr;
{$IFDEF DEBUG}
	EqpRPM_MaxNum, InvRPM_MaxNum: LongInt;
{$ENDIF DEBUG}
{$IFDEF SDLMODE}
	InfoGear: GearPtr;	{ Gear to appear in the INFO menu. }
	InfoGB: GameBoardPtr;
	MPB_Redraw: RedrawProcedureType;
	MPB_Gear: GearPtr;
  {$IFDEF PATCH_GH}
	MPR_InvMenu: RPGMenuPtr;
	MPR_InvMenuLGBN: GearPtr;
	MPR_InvMenuRGS: GearPtr;
  {$ENDIF PATCH_GH}

Procedure PlainRedraw;
	{ Miscellaneous menu redraw procedure. }
{$IFDEF PATCH_GH}
var
	Mek: GearPtr;
	MekNum: LongInt;
{$ENDIF PATCH_GH}
begin
	if InfoGB <> Nil then QuickCombatDisplay( InfoGB );
{$IFDEF PATCH_GH}
	Mek := NIL;
	if (NIL <> MPR_InvMenu) then begin
		MekNum := RPMLocateByPosition(MPR_InvMenu,MPR_InvMenu^.selectitem)^.value;
		if (0 <= MekNum) then begin
			if (NIL <> MPR_InvMenuRGS) then begin
				Mek := RetrieveGearSib( MPR_InvMenuRGS, MekNum );
			end;
		end;
	end else if (NIL <> InfoGear) then begin
		Mek := InfoGear;
	end;
	if (NIL <> Mek) and (GG_DisposeGear < Mek^.G) then begin
		DisplayGearInfo( Mek, InfoGB );
	end;
{$ELSE PATCH_GH}
	if InfoGear <> Nil then DisplayGearInfo( InfoGear , InfoGB );
{$ENDIF PATCH_GH}
end;

Procedure MiscProcRedraw;
	{ Miscellaneous menu redraw procedure. The Eqp display will be shown; }
	{ the INV display won't be. }
{$IFDEF PATCH_GH}
var
	Mek: GearPtr;
	MekNum: LongInt;
{$ENDIF PATCH_GH}
begin
	if InfoGB <> Nil then QuickCombatDisplay( InfoGB );
	DrawBPBorder;
{$IFDEF PATCH_GH}
	Mek := NIL;
	if (NIL <> MPR_InvMenu) then begin
		MekNum := RPMLocateByPosition(MPR_InvMenu,MPR_InvMenu^.selectitem)^.value;
		if (0 <= MekNum) then begin
			if (NIL <> MPR_InvMenuLGBN) then begin
  {$IFDEF DEBUG}
				Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum, False, 0, '' );
  {$ELSE DEBUG}
				Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum );
  {$ENDIF DEBUG}
			end else if (NIL <> MPR_InvMenuRGS) then begin
				Mek := RetrieveGearSib( MPR_InvMenuRGS , MekNum );
			end;
		end;
	end else if (NIL <> InfoGear) then begin
		Mek := InfoGear;
	end;
	if (NIL <> Mek) and (GG_DisposeGear < Mek^.G) then begin
		DisplayGearInfo( Mek, InfoGB );
	end;
{$ELSE PATCH_GH}
	if InfoGear <> Nil then DisplayGearInfo( InfoGear , InfoGB );
{$ENDIF PATCH_GH}
	if EqpRPM <> Nil then begin
		DisplayMenu( EqpRPM , Nil );
		NFGameMsg( MsgString( 'BACKPACK_Directions' ) , ZONE_Menu , MenuItem );
	end;
end;

Procedure RobotPartRedraw;
	{ Redraw procedure for the robot part selector. }
begin
	if InfoGB <> Nil then QuickCombatDisplay( InfoGB );
	DrawBPBorder;
{$IFDEF PATCH_GH}
	if (NIL <> InfoGear) and (GG_DisposeGear < InfoGear^.G) then begin
		DisplayGearInfo( InfoGear , InfoGB );
	end;
{$ELSE PATCH_GH}
	if InfoGear <> Nil then DisplayGearInfo( InfoGear , InfoGB );
{$ENDIF PATCH_GH}
	NFGameMsg( MsgString( 'SELECT_ROBOT_PARTS' ) , ZONE_EqpMenu , MenuItem );
end;


Procedure SelectColors( M: GearPtr; Redrawer: RedrawProcedureType );
	{ The player wants to change the colors for this part. Make it so. }
	{ The menu will be placed in the Menu area; assume the redrawer will }
	{ show whatever changes are made here. }
var
	RPM,CMenu: RPGMenuPtr;
	N,T,T2: Integer;
	C: SAttPtr;
	oldcolor,msg,newcolor: String;
begin
{$IFDEF PATCH_GH}
	if (NIL = M) or (M^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
	if M^.G = GG_Character then begin
		AddRPGMenuItem( RPM , MsgString( 'EDITCOLOR_1' ) , 1 );
		AddRPGMenuItem( RPM , MsgString( 'EDITCOLOR_2' ) , 2 );
		AddRPGMenuItem( RPM , MsgString( 'EDITCOLOR_3' ) , 3 );
	end else begin
		AddRPGMenuItem( RPM , MsgString( 'EDITCOLOR_4' ) , 4 );
		AddRPGMenuItem( RPM , MsgString( 'EDITCOLOR_5' ) , 5 );
		AddRPGMenuItem( RPM , MsgString( 'EDITCOLOR_6' ) , 6 );
	end;
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , -1 );

	repeat
		N := SelectMenu( RPM , Redrawer );
		if N > -1 then begin
			{ Create the list of colors. }
			CMenu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
			C := MasterColorList;
			T := 1;
			while C <> Nil do begin
				if ( Length( C^.Info ) > 6 ) and ( C^.Info[ N ] = '+' ) then begin
					AddRPGMenuItem( CMenu , Copy( RetrieveAPreamble( C^.Info ) , 7 , 255 ) , T );
				end;
				C := C^.Next;
				Inc( T );
			end;
			if CMenu^.NumItem > 0 then begin
				RPMSortAlpha( CMenu );
				T := SelectMenu( CMenu , Redrawer );
			end else begin
				T := -1;
			end;
			DisposeRPGMenu( CMenu );
			if T > 0 then begin
				C := RetrieveSAtt( MasterColorList , t );
				msg := RetrieveAString( C^.Info );

				oldcolor := SAttValue( M^.SA , 'SDL_COLORS' );
				newcolor := '';
				for t := 0 to 2 do begin
					if t = ( (N-1) mod 3 ) then begin
						for t2 := 1 to 3 do begin
							newcolor := newcolor + ' ' + BStr( ExtractValue( msg ) );
							ExtractValue( oldcolor );
						end;
					end else begin
						for t2 := 1 to 3 do begin
							newcolor := newcolor + ' ' + BStr( ExtractValue( oldcolor ) );
						end;
					end;
				end;
				SetSAtt( M^.SA , 'SDL_COLORS <' + newcolor + '>' );
			end;
		end;
	until N = -1;

	DisposeRPGMenu( RPM );
end;

Procedure SelectSprite( M: GearPtr; Redrawer: RedrawProcedureType );
	{ The player wants to change the colors for sprite for this character. }
	{ The menu will be placed in the Menu area; assume the redrawer will }
	{ show whatever changes are made here. }
var
	RPM: RPGMenuPtr;
	fname: String;
begin
{$IFDEF PATCH_GH}
	if (NIL = M) or (M^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
	if NAttValue( M^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Female then begin
		BuildFileMenu( RPM , Graphics_Directory + 'cha_f_*.*' );
	end else begin
		BuildFileMenu( RPM , Graphics_Directory + 'cha_m_*.*' );
	end;
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , -1 );

{$IFDEF PATCH_GH}
	fname := SAttValue( M^.SA , 'SDL_SPRITE' );
	while (RPM^.SelectItem < RPM^.NumItem) do begin
		if RPMLocateByPosition(RPM,RPM^.SelectItem)^.msg = fname then begin
			break;
		end;
		Inc( RPM^.SelectItem );
	end;
	RPM^.TopItem := -1;
{$ENDIF PATCH_GH}

	fname := SelectFile( RPM , Redrawer );

	if fname <> '' then begin
		SetSAtt( M^.SA , 'SDL_SPRITE <' + fname + '>' );
	end;

	DisposeRPGMenu( RPM );
end;

{$ENDIF}

Function SelectRobotParts( GB: GameBoardPtr; PC: GearPtr ): GearPtr;
	{ Select up to 10 parts to build a robot with. }
	{ Delink them from the INVENTORY and return them as a list. }
var
	Ingredients,Part,P2: GearPtr;
	RPM: RPGMenuPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit(NIL);
{$ENDIF PATCH_GH}

{$IFNDEF SDLMODE}
	DrawBPBorder;
	GameMsg( MsgString( 'SELECT_ROBOT_PARTS' ) , ZONE_EqpMenu , MenuItem );
{$ELSE}
	InfoGB := GB;
	InfoGear := PC;
{$ENDIF}
	Ingredients := Nil;
	repeat
		RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
		RPM^.Mode := RPMNoCleanup;

		Part := PC^.InvCom;
		N := 1;
		while Part <> Nil do begin
{$IFDEF PATCH_GH}
			if (GG_DisposeGear < Part^.G) then begin
				if ( Part^.G = GG_Weapon ) or ( Part^.G = GG_Shield ) or ( Part^.G = GG_ExArmor ) or ( Part^.G = GG_Sensor ) or ( Part^.G = GG_Electronics ) then begin
					AddRPGMenuItem( RPM , GearName( Part ) , N );
				end else if ( Part^.G = GG_RepairFuel ) and ( ( Part^.S = 15 ) or ( Part^.S = 23 ) ) then begin
					AddRPGMenuItem( RPM , GearName( Part ) , N );
				end;
				Inc( N );
			end;
			Part := Part^.Next;
{$ELSE PATCH_GH}
			if ( Part^.G = GG_Weapon ) or ( Part^.G = GG_Shield ) or ( Part^.G = GG_ExArmor ) or ( Part^.G = GG_Sensor ) or ( Part^.G = GG_Electronics ) then begin
				AddRPGMenuItem( RPM , GearName( Part ) , N );
			end else if ( Part^.G = GG_RepairFuel ) and ( ( Part^.S = 15 ) or ( Part^.S = 23 ) ) then begin
				AddRPGMenuItem( RPM , GearName( Part ) , N );
			end;
			Part := Part^.Next;
			Inc( N );
{$ENDIF PATCH_GH}
		end;
		RPMSortAlpha( RPM );
		AlphaKeyMenu( RPM );
		AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , -1 );

{$IFDEF SDLMODE}
		N := SelectMenu( RPM , @RobotPartRedraw );
{$ELSE}
		N := SelectMenu( RPM );
{$ENDIF}
		DisposeRPGMenu( RPM );

		if N > -1 then begin
			Part := RetrieveGearSib( PC^.InvCom , N );
			DelinkGear( PC^.InvCom , Part );
			while Part^.InvCom <> Nil do begin
				P2 := Part^.InvCom;
				DelinkGear( Part^.InvCom , P2 );
				InsertInvCom( PC , P2 );
			end;
			AppendGear( Ingredients , Part );
		end;

	until ( NumSiblingGears( Ingredients ) > 9 ) or ( N = -1 );

	SelectRobotParts := Ingredients;
end;


Procedure AddRepairOptions( RPM: RPGMenuPtr; PC,Item: GearPtr );
	{ Check the object in question, then add options to the }
	{ provided menu if the item is in need of repairs which the }
	{ PC can provide. Repair items will be numbered 100 + RSN }
var
	N: Integer;
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	PC := LocatePilot( PC );
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}
	if PC <> Nil then begin
		for N := 1 to NumRepairSkills do begin
			{ The repair option will only be added to the menu if: }
			{ - The PC has the required skill. }
			{ - The item is in need of repair (using this skill). }
			if ( NAttValue( PC^.NA , NAG_Skill , RepairSkillIndex[N] ) > 0 ) and ( TotalRepairableDamage( Item , RepairSkillIndex[N] ) > 0 ) then begin
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( RPM ,
					ReplaceHash( I18N_MsgString('BACKPACK_Repair'), I18N_Name('SkillMan',SkillMan[ RepairSkillIndex[N] ].Name) ),
					100 + N );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( RPM , MsgString( 'BACKPACK_Repair' ) + SkillMan[ RepairSkillIndex[N] ].Name , 100 + N );
{$ENDIF PATCH_I18N}
			end;
		end;
	end;
end;

Procedure DoFieldRepair( GB: GameBoardPtr; PC , Item: GearPtr; Skill: Integer );
	{ The PC is going to use one of the repair skills. Call the }
	{ standard procedure, then print output. }
var
	msg: String;
	N: LongInt;
	RepairFuel,RMaster: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then begin
		DialogMsg( MsgString( 'PCREPAIR_NoRepairFuel' ) );
		Exit;
	end;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then begin
		DialogMsg( MsgString( 'PCREPAIR_NoDamageDone' ) );
		Exit;
	end;
{$ENDIF PATCH_GH}

	{ Error check - if no repair is needed, display an appropraite }
	{ response. }
	if TotalRepairableDamage( Item , Skill ) < 1 then begin
		DialogMsg( MsgString( 'PCREPAIR_NoDamageDone' ) );
		Exit;
	end;

	{ Locate the "repair fuel". }
	RepairFuel := SeekGear( PC , GG_RepairFuel , Skill );
	if RepairFuel = Nil then begin
		DialogMsg( MsgString( 'PCREPAIR_NoRepairFuel' ) );
		Exit;
	end;

	{ Locate the root item. If this is a character, and the repair attempt }
	{ fails, and the master is destroyed, that's a bad thing. }
	RMaster := FindRoot( Item );

	N := UseRepairSkill( GB , PC , Item , Skill );
	msg := ReplaceHash( MsgString( 'PCREPAIR_UseSkill' + BStr( Skill ) ) , GearName( Item ) );

	{ Inform the user of the success. }
	if ( RMaster^.G = GG_Character ) and Destroyed( RMaster ) then begin
		AddNAtt( RMaster^.NA , NAG_Damage , NAS_StrucDamage , 30 );
		msg := msg + ReplaceHash( MsgString( 'PCREPAIR_DEAD' ) , GearName( RMaster ) );
	end else if N > 0 then begin
		msg := msg + BStr( N ) + MsgString( 'PCREPAIR_Success' + BStr( Skill ) );
	end else begin
		msg := msg + MsgString( 'PCREPAIR_Failure' + BStr( Skill ) );
	end;

	DialogMsg( msg );

	{ Deplete the fuel. }
	RepairFuel^.V := RepairFuel^.V - N;
	if RepairFuel^.V < 1 then begin
		DialogMsg( ReplaceHash( MsgString( 'PCREPAIR_FuelUsedUp' ) , GearName( RepairFuel ) ) );
		if IsSubCom( RepairFuel ) then begin
			RemoveGear( RepairFuel^.Parent^.SubCom , RepairFuel );
		end else if IsInvCom( RepairFuel ) then begin
			RemoveGear( RepairFuel^.Parent^.InvCom , RepairFuel );
		end;
	end;
end;

Function ShakeDown( GB: GameBoardPtr; Part: GearPtr; X,Y: Integer ): LongInt;
	{ This is the workhorse for this function. It does the }
	{ dirty work of separating inventory from (former) owner. }
{$IFDEF PATCH_GH}
const
	V_MAX = 2147483647;
	V_MIN = -2147483648;

	Function ShakeDown_internal( GB: GameBoardPtr; Part: GearPtr; X,Y: Integer ): Int64;
	var
		Cash: Int64;
		SPart: GearPtr;		{ Sub-Part }
	begin
		{ Start by removing the cash from this part. }
		Cash := NAttValue( Part^.NA , NAG_Experience , NAS_Credits );
		SetNAtt( Part^.NA , NAG_Experience , NAS_Credits , 0 );
		SetNAtt( Part^.NA , NAG_EpisodeData , NAS_Ransacked , 1 );

		{ Remove all InvComs, and place them on the map. }
		While (NIL <> Part^.InvCom) do begin
			SPart := Part^.InvCom;
			DelinkGear( Part^.InvCom , SPart );
			{ If this invcom isn't destroyed, put it on the }
			{ ground for the PC to pick up. Otherwise delete it. }
			if NotDestroyed( SPart ) then begin
				SetNAtt( SPart^.NA , NAG_Location , NAS_X , X );
				SetNAtt( SPart^.NA , NAG_Location , NAS_Y , Y );
				SPart^.Next := GB^.Meks;
				GB^.Meks := SPart;
			end else begin
				DisposeGear( SPart );
			end;
		end;

		{ Shake down this gear's subcoms. }
		SPart := Part^.SubCOm;
		while (NIL <> SPart) do begin
			if (GG_DisposeGear < SPart^.G) then begin
				if SPart^.G <> GG_Cockpit then begin
					Cash := Cash + ShakeDown_internal( GB , SPart , X , Y );
				end;
			end;
			SPart := SPart^.Next;
		end;

		if (V_MAX < Cash) then begin
			Cash := V_MAX;
		end else if (Cash < V_MIN) then begin
			Cash := V_MIN;
		end;
		ShakeDown_internal := Cash;
	end;
begin
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit(0);
	ShakeDown := ShakeDown_internal( GB, Part, X, Y );
{$ELSE PATCH_GH}
var
	cash: LongInt;
	SPart: GearPtr;		{ Sub-Part }
begin
	{ Start by removing the cash from this part. }
	cash := NAttValue( Part^.NA , NAG_Experience , NAS_Credits );
	SetNAtt( Part^.NA , NAG_Experience , NAS_Credits , 0 );
	SetNAtt( Part^.NA , NAG_EpisodeData , NAS_Ransacked , 1 );

	{ Remove all InvComs, and place them on the map. }
	While Part^.InvCom <> Nil do begin
		SPart := Part^.InvCom;
		DelinkGear( Part^.InvCom , SPart );
		{ If this invcom isn't destroyed, put it on the }
		{ ground for the PC to pick up. Otherwise delete it. }
		if NotDestroyed( SPart ) then begin
			SetNAtt( SPart^.NA , NAG_Location , NAS_X , X );
			SetNAtt( SPart^.NA , NAG_Location , NAS_Y , Y );
			SPart^.Next := GB^.Meks;
			GB^.Meks := SPart;
		end else begin
			DisposeGear( SPart );
		end;
	end;

	{ Shake down this gear's subcoms. }
	SPart := Part^.SubCOm;
	while SPart <> Nil do begin
		if SPart^.G <> GG_Cockpit then cash := cash + ShakeDown( GB , SPart , X , Y );
		SPart := SPart^.Next;
	end;

	ShakeDown := Cash;
{$ENDIF PATCH_GH}
end;


Function Ransack( GB: GameBoardPtr; X,Y: Integer ): LongInt;
	{ Yay! Loot and pillage! This function has two purposes: }
	{ first, it separates all Inventory gears from any non-operational }
	{ masters standing in this tile. Secondly, it collects the }
	{ money from all those non-operational masters and returns the }
	{ total amount as the function result. }
var
	it: LongInt;
	Mek: GearPtr;
begin
	it := 0;

	Mek := GB^.Meks;

	while Mek <> Nil do begin
{$IFDEF PATCH_GH}
		if (GG_DisposeGear < Mek^.G) then begin
{$ENDIF PATCH_GH}
		{ If this is a broken-down master, check to see if it's }
		{ one we want to pillage. }
		if IsMasterGear( Mek ) and not GearOperational( Mek ) then begin
			{ We will ransack this gear if it's in the correct location. }
			if ( NAttValue( Mek^.NA , NAG_Location , NAS_X ) = X ) and ( NAttValue( Mek^.NA , NAG_Location , NAS_Y ) = Y ) then begin
				it := it + ShakeDown( GB , Mek , X , Y );
			end;
		end else if ( Mek^.G = GG_MetaTerrain ) and ( ( Mek^.Stat[ STAT_Lock ] = 0 ) or Destroyed( Mek ) ) then begin
			{ Metaterrain gets ransacked if it's unlocked, }
			{ or wrecked. }
			if ( NAttValue( Mek^.NA , NAG_Location , NAS_X ) = X ) and ( NAttValue( Mek^.NA , NAG_Location , NAS_Y ) = Y ) then begin
				it := it + ShakeDown( GB , Mek , X , Y );
			end;
		end;
{$IFDEF PATCH_GH}
		end;
{$ENDIF PATCH_GH}
		Mek := Mek^.Next;
	end;

	Ransack := it;
end;

Function Handless( Mek: GearPtr ): Boolean;
	{ Return TRUE if Mek either has no hands or can't use its hands }
	{ at the moment (say, because it's transformed into tank mode). }
	{ Return TRUE if Mek has hands and they are in perfect working order. }
var
	Hand: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = Mek) or (Mek^.G <= GG_DisposeGear) then Exit(True);
  {$IFDEF PATCH_CHEAT}
	Hand := SeekActiveIntrinsic( Mek , GG_Holder , GS_Hand , Cheat_EnableCockpitBarrier_Hand );
  {$ELSE PATCH_CHEAT}
	Hand := SeekActiveIntrinsic( Mek , GG_Holder , GS_Hand );
  {$ENDIF PATCH_CHEAT}
	if (NIL = Hand) or (Hand^.G <= GG_DisposeGear) then Exit(True);
	Handless := not InGoodModule( Hand );
{$ELSE PATCH_GH}
	Hand := SeekActiveIntrinsic( Mek , GG_Holder , GS_Hand );
	if Hand = Nil then Handless := True
	else Handless := not InGoodModule( Hand );
{$ENDIF PATCH_GH}
end;

{$IFDEF SDLMODE}
	Procedure GetItemRedraw;
	begin
{$IFDEF PATCH_GH}
		if (NIL <> InfoGB) then begin
			QuickCombatDisplay( InfoGB );
		end;
		if (NIL <> InfoGear) and (GG_DisposeGear < InfoGear^.G) then begin
			DisplayGearInfo( InfoGear , InfoGB );
		end;
{$ELSE PATCH_GH}
		QuickCombatDisplay( InfoGB );
		DisplayGearInfo( InfoGear , InfoGB );
{$ENDIF PATCH_GH}
	end;
{$ENDIF}

{$IFDEF PATCH_GH}
Function SelectVisibleItem( GB: GameBoardPtr; PC: GearPtr; X,Y: Integer; var FlagAll: Boolean ): GearPtr;
{$ELSE PATCH_GH}
Function SelectVisibleItem( GB: GameBoardPtr; PC: GearPtr; X,Y: Integer ): GearPtr;
{$ENDIF PATCH_GH}
	{ Attempt to select a visible item from gameboard tile X,Y. }
	{ If more than one item is present, prompt the user for which one }
	{ to pick up. }
var
	N,T: Integer;
	RPM: RPGMenuPtr;
begin
{$IFDEF PATCH_GH}
	FlagAll := False;
{$ENDIF PATCH_GH}

	{ First count the number of items in this spot. }
	N := NumVisibleItemsAtSpot( GB , X , Y );

	{ If it's just 0 or 1, then our job is simple... }
	if N = 0 then begin
		SelectVisibleItem := Nil;
	end else if N = 1 then begin
		SelectVisibleItem := FindVisibleItemAtSpot( GB , X , Y );

	{ If it's more than one, better create a menu and let the user }
	{ pick one. }
	end else if N > 1 then begin
		DialogMsg( MsgString( 'GET_WHICH_ITEM?' ) );
		RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
		for t := 1 to N do begin
			AddRPGMenuItem( RPM , GearName( GetVisibleItemAtSpot( GB , X , Y , T ) ) , T );
		end;
{$IFDEF PATCH_GH}
		if ( 1 < RPM^.NumItem ) then begin
			AddRPGMenuItem_Top( RPM , I18N_MsgString( 'SelectVisibleItem', 'PickUpAll' ) , -5 );
		end;
{$ENDIF PATCH_GH}
{$IFDEF SDLMODE}
  {$IFDEF PATCH_GH}
		if (NIL = PC) or (PC^.G <= GG_DisposeGear) then InfoGear := NIL else InfoGear := PC;
  {$ELSE PATCH_GH}
		InfoGear := PC;
  {$ENDIF PATCH_GH}
		InfoGB := GB;
		N := SelectMenu( RPM , @GetItemRedraw );
{$ELSE}
		N := SelectMenu( RPM );
{$ENDIF}
		DisposeRPGMenu( RPM );
		if N > -1 then begin
			SelectVisibleItem := GetVisibleItemAtSpot( GB , X , Y , N );
{$IFDEF PATCH_GH}
		end else if ( -5 = N ) then begin
			FlagAll := True;
			SelectVisibleItem := NIL;
{$ENDIF PATCH_GH}
		end else begin
			SelectVisibleItem := Nil;
		end;
	end;
end;

Procedure PCGetItem( GB: GameBoardPtr; PC: GearPtr );
	{ The PC will attempt to pick up something lying on the ground. }

{$IFDEF PATCH_GH}
var
	Cash: LongInt;

	Procedure GetItem( Item: GearPtr );
	var
		NID: LongInt;
	begin
		if ( NIL <> Item ) then begin
			if IsLegalSlot( PC , Item ) then begin
				DelinkGear( GB^.Meks , Item );

				{ Clear the item's location values. }
				StripNAtt( Item , NAG_Location );
				if NAttValue( Item^.NA, NAG_ParaLocation, NAS_OriginalHome ) < 0 then begin
					StripNAtt( Item, NAG_ParaLocation );
				end;

				InsertInvCom( PC , Item );
				{ Clear the home, to prevent wandering items. }
				SetSAtt( Item^.SA , 'HOME <>' );
				DialogMsg( ReplaceHash( MsgString( 'YOU_GET_?' ) , GearName( Item ) ) );

				NID := NAttValue( Item^.NA , NAG_Narrative , NAS_NID );
				if NID <> 0 then SetTrigger( GB , TRIGGER_GetItem + BStr( NID ) );
			end else if Cash = 0 then begin
				DialogMsg( ReplaceHash( MsgString( 'CANT_GET_?' ) , GearName( Item ) ) );
			end;
		end else if Cash = 0 then begin
{$IFDEF PATCH_I18N}
			DialogMSG( I18N_MsgString('PCGetItem','No item found') );
{$ELSE PATCH_I18N}
			DialogMSG( 'No item found.' );
{$ENDIF PATCH_I18N}
		end;

		if Cash > 0 then begin
			DialogMsg( ReplaceHash( MsgString( 'YouFind$' ) , BStr( Cash ) ) );
			AddNAtt( LocatePilot( PC )^.NA , NAG_Experience , NAS_Credits , Cash );
			Cash := 0;
		end;

		{ Picking up an item takes time. }
		WaitAMinute( GB , PC , ReactionTime( PC ) );
	end;
	Procedure GetAllItem( P: Point );
	var
		Match: LPattern;
		M, M_Next: GearPtr;
	begin
		Match.X := P.X;
		Match.Y := P.Y;
		Match.Z := -10;
		Match.Only_Visibles := True;
		Match.Only_Masters := LP_MustNotBeMaster;

		M := GB^.Meks;
		while ( NIL <> M ) do begin
			M_Next := M^.Next;
			if GearMatchesLPattern( GB , M , Match ) then begin
				GetItem( M );
			end;
			M := M_Next;
		end;
	end;
var
	P: Point;
	item: GearPtr;
	FlagAll: Boolean;
{$ELSE PATCH_GH}
var
	Cash,NID: LongInt;
	P: Point;
	item: GearPtr;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) or Handless(PC) then begin
{$ELSE PATCH_GH}
	if Handless( PC ) then begin
{$ENDIF PATCH_GH}
		{ Start by checking something that other RPGs would }
		{ just assume- does the PC have any hands? }
{$IFDEF PATCH_I18N}
		DialogMsg( I18N_MsgString('PCGetItem','you need hands') );
{$ELSE PATCH_I18N}
		DialogMsg( 'You need hands in order to use this command.' );
{$ENDIF PATCH_I18N}

	end else begin
		P := GearCurrentLocation( PC );

		{ Before attempting to get an item, ransack whatever }
		{ fallen enemies lie in this spot. }
		Cash := Ransack( GB , P.X , P.Y );

		{ Perform an immediate vision check- without it, items }
		{ freed by the Ransack procedure above will remain unseen. }
		VisionCheck( GB , PC );

{$IFDEF PATCH_GH}
		Item := SelectVisibleItem( GB , PC , P.X , P.Y , FlagAll );

		if FlagAll then begin
			GetAllItem( P );
		end else begin
			GetItem( Item );
		end;
{$ELSE PATCH_GH}
		Item := SelectVisibleItem( GB , PC , P.X , P.Y );

		if Item <> Nil then begin
			if IsLegalSlot( PC , Item ) then begin
				DelinkGear( GB^.Meks , Item );

				{ Clear the item's location values. }
				StripNAtt( Item , NAG_Location );
{$IFDEF PATCH_GH}
				if NAttValue( Item^.NA, NAG_ParaLocation, NAS_OriginalHome ) < 0 then begin
					StripNAtt( Item, NAG_ParaLocation );
				end;
{$ENDIF PATCH_GH}

				InsertInvCom( PC , Item );
				{ Clear the home, to prevent wandering items. }
				SetSAtt( Item^.SA , 'HOME <>' );
				DialogMsg( ReplaceHash( MsgString( 'YOU_GET_?' ) , GearName( Item ) ) );

				NID := NAttValue( Item^.NA , NAG_Narrative , NAS_NID );
				if NID <> 0 then SetTrigger( GB , TRIGGER_GetItem + BStr( NID ) );
			end else if Cash = 0 then begin
				DialogMsg( ReplaceHash( MsgString( 'CANT_GET_?' ) , GearName( Item ) ) );
			end;
		end else if Cash = 0 then begin
{$IFDEF PATCH_I18N}
			DialogMSG( I18N_MsgString('PCGetItem','No item found') );
{$ELSE PATCH_I18N}
			DialogMSG( 'No item found.' );
{$ENDIF PATCH_I18N}
		end;

		if Cash > 0 then begin
			DialogMsg( ReplaceHash( MsgString( 'YouFind$' ) , BStr( Cash ) ) );
			AddNAtt( LocatePilot( PC )^.NA , NAG_Experience , NAS_Credits , Cash );
		end;

		{ Picking up an item takes time. }
		WaitAMinute( GB , PC , ReactionTime( PC ) );
{$ENDIF PATCH_GH}
	end;
end;

Procedure CreateInvMenu( PC: GearPtr );
	{ Allocate the Inventory menu and fill it up with the PC's inventory. }
begin
	if InvRPM <> Nil then DisposeRPGMenu( InvRPM );
	InvRPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
	InvRPM^.Mode := RPMNoCleanup;
{$IFDEF PATCH_CHEAT}
  {$IFDEF DEBUG}
	InvRPM_MaxNum := BuildInventoryMenu( InvRPM , PC , Cheat_InvMenu_ShowSubItem );
  {$ELSE DEBUG}
	BuildInventoryMenu( InvRPM , PC , Cheat_InvMenu_ShowSubItem );
  {$ENDIF DEBUG}
{$ELSE PATCH_CHEAT}
  {$IFDEF DEBUG}
	InvRPM_MaxNum := BuildInventoryMenu( InvRPM , PC );
  {$ELSE DEBUG}
	BuildInventoryMenu( InvRPM , PC );
  {$ENDIF DEBUG}
{$ENDIF PATCH_CHEAT}
	AttachMenuDesc( InvRPM , ZONE_Menu2 );
{$IFDEF PATCH_CHEAT}
	if not(Cheat_InvMenu_NoSort) then begin
		if Cheat_InvMenu_ShowSubItem then begin
			RPMSortAlpha_withSubItem( InvRPM );
		end else begin
			RPMSortAlpha( InvRPM );
		end;
	end;
{$ELSE PATCH_CHEAT}
	RPMSortAlpha( InvRPM );
{$ENDIF PATCH_CHEAT}

	{ If the menu is empty, add a message saying so. }
{$IFDEF PATCH_I18N}
	If InvRPM^.NumItem < 1 then AddRPGMenuItem( InvRPM , I18N_MsgString('CreateInvMenu','no inventory items') , -1 )
{$ELSE PATCH_I18N}
	If InvRPM^.NumItem < 1 then AddRPGMenuItem( InvRPM , '[no inventory items]' , -1 )
{$ENDIF PATCH_I18N}
	else AlphaKeyMenu( InvRPM );

	{ Add the menu keys. }
	AddRPGMenuKey(InvRPM,'/',-2);
{$IFDEF PATCH_CHEAT}
	if Cheat_MenuOrder_Edit then begin
		AddRPGMenuKey( InvRPM , KeyMap[ KMC_EditMenuOrder ].KCode , -128 );
	end;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_GH}
	if ( 1 < InvRPM^.NumItem ) then begin
		AddRPGMenuItem_Top( InvRPM , I18N_MsgString( 'ThisItemWasSelected', 'DropAll' ), -3 );
		AddRPGMenuItem_Top( InvRPM , I18N_MsgString( 'ThisItemWasSelected', 'TransferAll' ), -4 );
	end;
{$ENDIF PATCH_GH}
end;

Procedure CreateEqpMenu( PC: GearPtr );
	{ Allocate the equipment menu and fill it up with the PC's gear. }
{$IFDEF PATCH_CHEAT}
	Function AddRPGMenu_MassMeter( RPM: RPGMenuPtr; PC: GearPtr): String;
		{ On top of the backpack seperator, display the current inventory mass, and the carrying capacity }
		{ Origin : Michael }
		{ http://gearheadrpg.com/forum/index.php?action=vthread&forum=1&topic=789 }
		{ ftp://ftp.ocis.net/pub/users/ldeutsch/ghpatches/gh-1100-massmeter1.diff }
	var
		MassString: String;
		CurrentInv,CurrentEqp,Limit,Limit2: LongInt;
	begin
		CurrentInv := IntrinsicMass(PC);
		CurrentEqp := EquipmentMass(PC);
		Limit2 := GearEncumberance(PC);
	
		Limit := Limit2 * 2 - 1;	{Maximum weight before penalty starts}
	
		if PC^.G = GG_Character then begin
			Limit := Limit + NAttValue(PC^.NA,NAG_Skill,NAS_WeightLifting);
		end;
	
		Limit2 := Limit2 + Limit;	{Where penalty gets worse}
	
		MassString := ReplaceHash( I18N_MsgString('CreateEqpMenu','MassMeter'),
					MakeMassString(CurrentInv,PC^.Scale),
					MakeMassString(CurrentEqp,PC^.Scale) );
		AddRPGMenuItem_Top( RPM, #$0 + MassString, 0 );

		MassString := ReplaceHash( I18N_MsgString('CreateEqpMenu','MassMeter_Limit'),
					MakeMassString(Limit,PC^.Scale) );
		if Limit2 < CurrentEqp then begin
			MassString := MassString + I18N_MsgString('CreateEqpMenu','MassMeter_LimitOver2');
		end else if Limit < CurrentEqp then begin
			MassString := MassString + I18N_MsgString('CreateEqpMenu','MassMeter_LimitOver');
		end else begin
		end;
		AddRPGMenuItem_Top( RPM, #$0 + MassString, 0 );
	end;
{$ENDIF PATCH_CHEAT}
begin
	if EqpRPM <> Nil then DisposeRPGMenu( EqpRPM );
	EqpRPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_EqpMenu );
	EqpRPM^.Mode := RPMNoCleanup;
	AttachMenuDesc( EqpRPM , ZONE_Menu2 );
{$IFDEF DEBUG}
	EqpRPM_MaxNum := BuildEquipmentMenu( EqpRPM , PC );
{$ELSE DEBUG}
	BuildEquipmentMenu( EqpRPM , PC );
{$ENDIF DEBUG}

	{ If the menu is empty, add a message saying so. }
{$IFDEF PATCH_I18N}
	If EqpRPM^.NumItem < 1 then AddRPGMenuItem( EqpRPM , #$0 + I18N_MsgString('CreateEqpMenu','no equipped items') , -1 );
{$ELSE PATCH_I18N}
	If EqpRPM^.NumItem < 1 then AddRPGMenuItem( EqpRPM , '[no equipped items]' , -1 );
{$ENDIF PATCH_I18N}

	{ Add the menu keys. }
	AddRPGMenuKey(EqpRPM,'/',-2);

{$IFDEF PATCH_CHEAT}
	if Cheat_EqpMenu_AddMenuKey then begin
		AlphaKeyMenu( EqpRPM );
	end;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_GH}
	if ( 1 < EqpRPM^.NumItem ) then begin
		AddRPGMenuItem_Top( EqpRPM , I18N_MsgString( 'ThisItemWasSelected', 'DropAll' ), -3 );
		AddRPGMenuItem_Top( EqpRPM , I18N_MsgString( 'ThisItemWasSelected', 'TransferAll' ), -4 );
	end;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
	If Cheat_EqpMenu_ShowMassMeter then begin
		if (NIL <> PC) and (GG_DisposeGear < PC^.G) then begin
			AddRPGMenu_MassMeter( EqpRPM, PC );
		end;
	end;
{$ENDIF PATCH_CHEAT}
end;

Procedure UpdateBackpack( PC: GearPtr );
	{ Redo all the menus, and display them on the screen. }
begin
	CreateInvMenu( PC );
	CreateEqpMenu( PC );
{$IFNDEF SDLMODE}
	DisplayMenu( InvRPM );
	DisplayMenu( EqpRPM );
{$ENDIF}
end;

Procedure GivePartToPC( GB: GameBoardPtr; Part, PC: GearPtr );
	{ Give the specified part to the PC. If the part cannot be }
	{ held by the PC, store it so that it can be recovered using }
	{ the FieldHQ Wargear Explorer. }
var
	team: Integer;
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit;
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = GB) then begin
		InsertInvCom( PC, Part );
	end else
{$ENDIF PATCH_GH}
	if ( PC <> Nil ) and IsLegalSlot( PC , Part ) then begin
		InsertInvCom( PC , Part );
	end else begin
		{ If the PC can't carry this equipment, }
		{ stick it off the map. }
		team := NattValue( PC^.NA , NAG_Location , NAS_Team );
		if team = NAV_LancemateTeam then team := NAV_DefPlayerTeam;
		SetNAtt( Part^.NA , NAG_Location , NAS_Team , team );
		DeployMek( GB , Part , False );
	end;
end;

Procedure UnequipItem( GB: GameBoardPtr; PC , Item: GearPtr );
	{ Delink ITEM from its parent, and stick it in the general inventory... }
	{ If possible. Otherwise drop it. }
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}
	{ First, delink Item from its parent. }
	DelinkGear( Item^.Parent^.InvCom , Item );
	{ HOW'D YA LIKE THEM CARROT DOTS, EH!?!? }

	{ Next, link ITEM into the general inventory. }
	GivePartToPC( GB , Item , PC );

	{ Unequipping takes time. }
	if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
end;

Procedure UnequipFrontend( GB: GameBoardPtr; PC , Item: GearPtr );
	{ Simply unequip the provided item. }
	{ PRECOND: PC and ITEM had better be correct, dagnabbit... }
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_I18N}
	DialogMsg( ReplaceHash( I18N_MsgString('UnequipFrontend','You unequip'), GearName(Item) ) );
{$ELSE PATCH_I18N}
	DialogMsg( 'You unequip ' + GearName( Item ) + '.' );
{$ENDIF PATCH_I18N}
	UnequipItem( GB , PC , Item );
end;


Function CanBeExtracted( Item: GearPtr ): Boolean;
	{ Return TRUE if the listed part can be extracted from a mecha, }
	{ or FALSE if it cannot normally be extracted. }
begin
{$IFDEF PATCH_GH}
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit(False);
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
	if ( Item^.G = GG_Support ) then begin
		CanBeExtracted := False;
		if Cheat_MechaCustomize_FreeSupport then begin
			CanBeExtracted := True;
		end;
	end else if ( Item^.G = GG_Cockpit ) then begin
		CanBeExtracted := False;
		if Cheat_MechaCustomize_FreeCockpit then begin
			CanBeExtracted := True;
		end;
	end else if IsMasterGear( Item ) then begin
		CanBeExtracted := False;
		if Cheat_MechaCustomize_FreeMasterGear then begin
			CanBeExtracted := True;
		end;
	end else if ( Item^.Parent = Nil ) then begin
		CanBeExtracted := False;
		if Cheat_MechaCustomize_FreeParent then begin
			CanBeExtracted := True;
		end;
	end else if ( Item^.Parent^.Scale = 0 ) then begin
		CanBeExtracted := False;
		if Cheat_MechaCustomize_FreeScale then begin
			CanBeExtracted := True;
		end;
	end else if ( Item^.G = GG_Modifier ) then begin
		CanBeExtracted := False;
		if Cheat_MechaCustomize_FreeModifier then begin
			CanBeExtracted := True;
		end;
	end else if ( Item^.G = GG_Module ) and ( Item^.S = GS_Body ) then begin
		CanBeExtracted := False;
		if Cheat_MechaCustomize_FreeBodyModule then begin
			CanBeExtracted := True;
		end;
	end else begin
		CanBeExtracted := True;
	end;
	if Cheat_MechaCustomize_Limitless then begin
		CanBeExtracted := True;
	end;
{$ELSE PATCH_CHEAT}
	if ( Item^.G = GG_Support ) or ( Item^.G = GG_Cockpit ) or IsMasterGear( Item ) or ( Item^.Parent = Nil ) or ( Item^.Parent^.Scale = 0 ) or ( Item^.G = GG_Modifier ) then begin
		CanBeExtracted := False;
	end else if ( Item^.G = GG_Module ) and ( Item^.S = GS_Body ) then begin
		CanBeExtracted := False;
	end else begin
		CanBeExtracted := True;
	end;
{$ENDIF PATCH_CHEAT}
end;

{$IFDEF PATCH_CHEAT}
Function ExtractItem( GB: GameBoardPtr; LList, TruePC, PC: GearPtr; var Item: GearPtr ): Boolean;
{$ELSE PATCH_CHEAT}
Function ExtractItem( GB: GameBoardPtr; TruePC , PC: GearPtr; var Item: GearPtr ): Boolean;
{$ENDIF PATCH_CHEAT}
	{ Delink ITEM from its parent, and stick it in the general inventory. }
	{ Note that pulling a gear out of its mecha may well wreck it }
	{ beyond any repair! Therefore, after this call, ITEM might no }
	{ longer exist... i.e. it may equal NIL. }
var
	it: Boolean;
	SkTarget,SkRoll,WreckTarget: Integer;
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the NIL of GB at here. }
	{ Don't kick out the GG_DisposeGear of LList at here. }
	if (NIL = TruePC) or (TruePC^.G <= GG_DisposeGear) then Exit(False);
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit(False);
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit(False);
{$ENDIF PATCH_GH}

	{ First, calculate the skill target. }
	SkTarget := 2 + ComponentComplexity( Item );
	if Item^.G = GG_Module then begin
		WreckTarget := SkTarget + 8 - Item^.V;
	end else if Item^.Scale < Item^.Parent^.Scale then begin
		WreckTarget := SkTarget + 5;
	end else begin
		WreckTarget := SkTarget + 10 - UnscaledMaxDamage( Item );
	end;
	if WreckTarget < SkTarget then WreckTarget := SkTarget + 1;

{$IFDEF PATCH_GH}
	if (NIL <> GB) then begin
		SkRoll := RollStep( TeamSkill( GB , NAV_DefPlayerTeam , 31 ) );
	end else if (NIL <> LList) then begin
		SkRoll := RollStep( TeamSkill( LList, 31 ) );
	end else begin
		SkRoll := RollStep( SkillValue( TruePC, 31 ) );
	end;
{$ELSE PATCH_GH}
	SkRoll := RollStep( TeamSkill( GB , NAV_DefPlayerTeam , 31 ) );
{$ENDIF PATCH_GH}

	DoleSkillExperience( TruePC , 31 , 1 );
	AddMentalDown( TruePC , 1 );
{$IFDEF PATCH_GH}
	if (NIL <> GB) then begin
		WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );
	end;
{$ELSE PATCH_GH}
	WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );
{$ENDIF PATCH_GH}

	if SkRoll > WreckTarget then begin
		{ First, delink Item from its parent. }
		DelinkGear( Item^.Parent^.SubCom , Item );

		{ Stick the part in the general inventory, if legal. }
		GivePartToPC( GB , Item , PC );

		DoleSkillExperience( TruePC , 31 , 2 );
		DoleExperience( TruePC , 1 );
		it := True;

	end else if SkRoll > SkTarget then begin
		RemoveGear( Item^.Parent^.SubCom , Item );
		Item := Nil;
		it := True;

	end else begin
		it := False;
	end;

	ExtractItem := it;
end;

{$IFDEF PATCH_CHEAT}
Procedure ExtractFrontend( GB: GameBoardPtr; LList, TruePC, PC, Item: GearPtr );
{$ELSE PATCH_CHEAT}
Procedure ExtractFrontend( GB: GameBoardPtr; TruePC , PC , Item: GearPtr );
{$ENDIF PATCH_CHEAT}
	{ Simply remove the provided item. }
	{ PRECOND: PC and ITEM had better be correct, dagnabbit... }
var
	name: String;
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the NIL of GB at here. }
	{ Don't kick out the GG_DisposeGear of LList at here. }
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	name := GearName( Item );
	if GearActive( PC ) then begin
		DialogMsg( MsgString( 'EXTRACT_NOTACTIVE' ) );
{$IFDEF PATCH_CHEAT}
	end else if ExtractItem( GB, LList, TruePC, PC, Item ) then begin
{$ELSE PATCH_CHEAT}
	end else if ExtractItem( GB , TruePC , PC , Item ) then begin
{$ENDIF PATCH_CHEAT}
		if Item = Nil then begin
			DialogMsg( ReplaceHash( MsgString( 'EXTRACT_WRECK' ) , name ) );
		end else begin
			DialogMsg( ReplaceHash( MsgString( 'EXTRACT_OK' ) , name ) );
		end;
	end else begin
		DialogMsg( ReplaceHash( MsgString( 'EXTRACT_FAIL' ) , name ) );
	end;
end;


Procedure EquipItem( GB: GameBoardPtr; PC , Slot , Item: GearPtr );
	{ This is the real equipping procedure. Stuff ITEM into SLOT. }
	{ As noted in TheRules.txt, any nonmaster gear can only have one }
	{ item of any particular "G" type equipped at a time. So, if }
	{ SLOT already has equipment of type ITEM^.G, unequip that and }
	{ stuff it into PC's general inventory. }
var
	I2,I3: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Slot) or (Slot^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ First, check for already equipped items. }
	I2 := Slot^.InvCom;
	while I2 <> Nil do begin
		I3 := I2^.Next;		{ This next step might delink I2, so... }
		if I2^.G = Item^.G then begin
			UnequipItem( GB , PC , I2 );
		end;
		I2 := I3;
	end;

	{ Next, delink Item from PC. }
	DelinkGear( PC^.InvCom , Item );

	{ Next, link ITEM into SLOT. }
	InsertInvCom( Slot , Item );

	{ Equipping an item takes time. }
	if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
end;

Procedure EquipItemFrontend( GB: GameBoardPtr; PC , Item: GearPtr );
	{ Assign ITEM to a legal equipment slot. Move it from the }
	{ general inventory into its new home. }
var
	EI_Menu: RPGMenuPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
	ReBrowse: Boolean;
{$ENDIF PATCH_CHEAT}
{$IFDEF DEBUG}
	MaxNum: LongInt;
{$ENDIF DEBUG}
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Build the slot selection menu. }
	EI_Menu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
{$IFDEF DEBUG}
	MaxNum := BuildSlotMenu( EI_Menu , PC , Item );
{$ELSE DEBUG}
	BuildSlotMenu( EI_Menu , PC , Item );
{$ENDIF DEBUG}
{$IFDEF PATCH_I18N}
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , ReplaceHash( I18N_MsgString('EquipItemFrontend','cannot equip'), GearName(Item) ) , -1 );
{$ELSE PATCH_I18N}
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , '[cannot equip ' + GearName( Item ) + ']' , -1 );
{$ENDIF PATCH_I18N}

{$IFDEF PATCH_CHEAT}
	repeat
		ReBrowse := False;
  {$IFDEF SDLMODE}
		N := SelectMenu( EI_Menu , @MiscProcRedraw);
  {$ELSE SDLMODE}
		N := SelectMenu( EI_Menu );
  {$ENDIF SDLMODE}

		if N < -1 then begin
  {$IFDEF PATCH_I18N}
			DialogMSG( I18N_MsgString('EquipItemFrontend','You can not install to it') );
  {$ELSE PATCH_I18N}
			DialogMSG( 'You can not install to it.' );
  {$ENDIF PATCH_I18N}
			ReBrowse := True;
		end else if 0 <= N then begin
  {$IFDEF PATCH_I18N}
			DialogMsg( ReplaceHash( I18N_MsgString('EquipItemFrontend','You equip'), GearName(Item) ) );
  {$ELSE PATCH_I18N}
			DialogMsg( 'You equip ' + GearName( Item ) + '.' );
  {$ENDIF PATCH_I18N}
  {$IFDEF DEBUG}
			EquipItem( GB , PC , LocateGearByNumber( PC , N, False, MaxNum, 'EquipItemFrontend' ) , Item );
  {$ELSE DEBUG}
			EquipItem( GB , PC , LocateGearByNumber( PC , N ) , Item );
  {$ENDIF DEBUG}
		end;
	until False = ReBrowse;
	DisposeRPGMenu( EI_Menu );
{$ELSE PATCH_CHEAT}
	{ Select a slot for the item to go into. }
  {$IFDEF SDLMODE}
	N := SelectMenu( EI_Menu , @MiscProcRedraw);
  {$ELSE}
	N := SelectMenu( EI_Menu );
  {$ENDIF}
	DisposeRPGMenu( EI_Menu );

	{ If a slot was selected, pass that info on to the workhorse. }
	if N <> -1 then begin
  {$IFDEF PATCH_I18N}
		DialogMsg( ReplaceHash( I18N_MsgString('EquipItemFrontend','You equip'), GearName(Item) ) );
  {$ELSE PATCH_I18N}
		DialogMsg( 'You equip ' + GearName( Item ) + '.' );
  {$ENDIF PATCH_I18N}
  {$IFDEF DEBUG}
		EquipItem( GB , PC , LocateGearByNumber( PC , N, False, MaxNum, 'EquipItemFrontend' ) , Item );
  {$ELSE DEBUG}
		EquipItem( GB , PC , LocateGearByNumber( PC , N ) , Item );
  {$ENDIF DEBUG}
	end;
{$ENDIF PATCH_CHEAT}
end;

{$IFDEF PATCH_CHEAT}
Function InstallItem( GB: GameBoardPtr; LList, TruePC, Slot: GearPtr; var Item: GearPtr ): Boolean;
{$ELSE PATCH_CHEAT}
Function InstallItem( GB: GameBoardPtr; TruePC , Slot: GearPtr; var Item: GearPtr ): Boolean;
{$ENDIF PATCH_CHEAT}
	{ Attempt the skill rolls needed to install ITEM into the }
	{ requested slot. }
var
	SlotCom,ItemCom,UsedCom: Integer;
	SkTarget,WreckTarget,SkRoll: Integer;
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the NIL of GB at here. }
	{ Don't kick out the GG_DisposeGear of LList at here. }
	if (NIL = TruePC) or (TruePC^.G <= GG_DisposeGear) then Exit(False);
	if (NIL = Slot) or (Slot^.G <= GG_DisposeGear) then Exit(False);
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit(False);
{$ENDIF PATCH_GH}

	{ Error Check - no circular references! }
	if ( FindGearIndex( Item , Slot ) <> -1 ) then Exit( False );

	{ Also, can't engineer things when you're exhausted. }
	if CurrentMental( TruePC ) < 1 then Exit( False );

	{ Can't install into a personal-scale slot. }
{$IFDEF PATCH_CHEAT}
	if (Slot^.Scale = 0) and not(Cheat_MechaCustomize_FreeScale) then Exit( False );
{$ELSE PATCH_CHEAT}
	if Slot^.Scale = 0 then Exit( False );
{$ENDIF PATCH_CHEAT}

	SlotCom := ComponentComplexity( Slot );
	ItemCom := ComponentComplexity( Item );
	UsedCom := SubComComplexity( Slot );

	case Item^.G of
		GG_Weapon,GG_MoveSys: SkTarget := 10;
		GG_Module: SkTarget := 15;
		GG_Sensor: SkTarget := 30;
		GG_Modifier: SkTarget := 25;
	else SkTarget := 20;
	end;
	if Item^.Scale < Slot^.Scale then SkTarget := SkTarget div 2;

	{ The WreckTarget is the target number that must be beat }
	{ in order to avoid accidentally destroying the part... }
	if ( Item^.G = GG_Module ) then begin
		WreckTarget := 8 - Item^.V;
	end else if ( UnscaledMaxDamage( Item ) < 1 ) or ( Item^.Scale < Slot^.Scale ) then begin
		WreckTarget := 7;
	end else begin
		WreckTarget := 10 - UnscaledMaxDamage( Item );
	end;
	if WreckTarget < 3 then WreckTarget := 3;

	{ If the SLOT is going to be overstuffed, better raise the }
	{ number of successes and the target number drastically. }
	if ( ( ItemCom + UsedCom ) > SlotCom ) and ( Not IsMasterGear( Slot ) ) then begin
		SkTarget := SkTarget + ItemCom + UsedCom - SlotCom + 5;
	end;

{$IFDEF PATCH_GH}
	if (NIL <> GB) then begin
		WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );
	end;
{$ELSE PATCH_GH}
	WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * 5 );
{$ENDIF PATCH_GH}

{$IFDEF PATCH_GH}
	if (NIL <> GB) then begin
		SkRoll := RollStep( TeamSkill( GB , NAV_DefPlayerTeam , 31 ) );
	end else if (NIL <> LList) then begin
		SkRoll := RollStep( TeamSkill( LList, 31 ) );
	end else begin
		SkRoll := RollStep( SkillValue( TruePC, 31 ) );
	end;
{$ELSE PATCH_GH}
	SkRoll := RollStep( TeamSkill( GB , NAV_DefPlayerTeam , 31 ) );
{$ENDIF PATCH_GH}
	if SkRoll > SkTarget then begin
		{ Install the item. }
		DoleSkillExperience( TruePC , 31 , 5 );
		DoleExperience( TruePC , 10 );
		DelinkGear( Item^.Parent^.InvCom , Item );
		InsertSubCom( Slot , Item );
	end else if SkRoll < WreckTarget then begin
		RemoveGear( Item^.Parent^.InvCom , Item );
		Item := Nil;
	end;

	AddMentalDown( TruePC , 1 );
	DoleSkillExperience( TruePC , 31 , 1 );

	InstallItem := SkRoll > SkTarget;
end;

{$IFDEF PATCH_CHEAT}
Procedure InstallFrontend( GB: GameBoardPtr; LList, TruePC, PC, Item: GearPtr );
{$ELSE PATCH_CHEAT}
Procedure InstallFrontend( GB: GameBoardPtr; TruePC , PC , Item: GearPtr );
{$ENDIF PATCH_CHEAT}
	{ Assign ITEM to a legal equipment slot. Move it from the }
	{ general inventory into its new home. }
var
	EI_Menu: RPGMenuPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
	name: String;
{$IFDEF PATCH_CHEAT}
	ReBrowse: Boolean;
{$ENDIF PATCH_CHEAT}
{$IFDEF DEBUG}
	MaxNum: LongInt;
{$ENDIF DEBUG}
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the NIL of GB at here. }
	{ Don't kick out the GG_DisposeGear of LList at here. }
	if (NIL = TruePC) or (TruePC^.G <= GG_DisposeGear)
		or (NIL = PC) or (PC^.G <= GG_DisposeGear)
		or (NIL = Item) or (Item^.G <= GG_DisposeGear)
	then begin
  {$IFDEF PATCH_I18N}
		DialogMSG( I18N_MsgString('InstallFrontend','You can not install to it') );
  {$ELSE PATCH_I18N}
		DialogMSG( 'You can not install to it.' );
  {$ENDIF PATCH_I18N}
		Exit;
	end;
{$ENDIF PATCH_GH}

	{ Error check- can't install into an active master. }
	if GearActive( PC ) then begin
		DialogMsg( MsgString( 'INSTALL_NOTACTIVE' ) );
		Exit;
	end;

	{ Build the slot selection menu. }
	EI_Menu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
{$IFDEF PATCH_CHEAT}
  {$IFDEF DEBUG}
	MaxNum := BuildSubMenu( EI_Menu , PC , Item , True, Cheat_Install_ShowSubItem );
  {$ELSE DEBUG}
	BuildSubMenu( EI_Menu , PC , Item , True, Cheat_Install_ShowSubItem );
  {$ENDIF DEBUG}
  {$IFDEF PATCH_I18N}
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , ReplaceHash( I18N_MsgString('InstallFrontend','cannot install'), GearName(Item) ) , -1 );
  {$ELSE PATCH_I18N}
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , '[cannot install ' + GearName( Item ) + ']' , -1 );
  {$ENDIF PATCH_I18N}

	repeat
		ReBrowse := False;
		DialogMsg( GearName( Item ) + ' cmx:' + BStr( ComponentComplexity( Item ) ) + '. ' + MsgSTring( 'BACKPACK_InstallInfo' ) );
  {$IFDEF SDLMODE}
		N := SelectMenu( EI_Menu , @MiscProcRedraw);
  {$ELSE}
		N := SelectMenu( EI_Menu );
  {$ENDIF}

		if N < -1 then begin
  {$IFDEF PATCH_I18N}
			DialogMSG( I18N_MsgString('InstallFrontend','You can not install to it') );
  {$ELSE PATCH_I18N}
			DialogMSG( 'You can not install to it.' );
  {$ENDIF PATCH_I18N}
			ReBrowse := True;
		end else if 0 <= N then begin
			name := GearName( Item );
  {$IFDEF PATCH_CHEAT}
    {$IFDEF DEBUG}
			if InstallItem( GB, LList, TruePC, LocateGearByNumber( PC, N, False, MaxNum, 'InstallFrontend' ), Item ) then begin
    {$ELSE DEBUG}
			if InstallItem( GB, LList, TruePC, LocateGearByNumber( PC, N ), Item ) then begin
    {$ENDIF DEBUG}
  {$ELSE PATCH_CHEAT}
    {$IFDEF DEBUG}
			if InstallItem( GB , TruePC , LocateGearByNumber( PC , N, False, MaxNum, 'InstallFrontend' ) , Item ) then begin
    {$ELSE DEBUG}
			if InstallItem( GB , TruePC , LocateGearByNumber( PC , N ) , Item ) then begin
    {$ENDIF DEBUG}
  {$ENDIF PATCH_CHEAT}
				DialogMsg( ReplaceHash( MsgString( 'INSTALL_OK' ) , name ) );
			end else begin
				if Item = Nil then begin
					DialogMsg( ReplaceHash( MsgString( 'INSTALL_WRECK' ) , name ) );
				end else begin
					DialogMsg( ReplaceHash( MsgString( 'INSTALL_FAIL' ) , name ) );
				end;
			end;
		end;
	until False = ReBrowse;
	DisposeRPGMenu( EI_Menu );
{$ELSE PATCH_CHEAT}
  {$IFDEF DEBUG}
	MaxNum := BuildSubMenu( EI_Menu , PC , Item , True );
  {$ELSE DEBUG}
	BuildSubMenu( EI_Menu , PC , Item , True );
  {$ENDIF DEBUG}
  {$IFDEF PATCH_I18N}
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , ReplaceHash( I18N_MsgString('InstallFrontend','cannot install'), GearName(Item) ) , -1 );
  {$ELSE PATCH_I18N}
	if EI_Menu^.NumItem < 1 then AddRPGMenuItem( EI_Menu , '[cannot install ' + GearName( Item ) + ']' , -1 );
  {$ENDIF PATCH_I18N}

	{ Select a slot for the item to go into. }
	DialogMsg( GearName( Item ) + ' cmx:' + BStr( ComponentComplexity( Item ) ) + '. ' + MsgSTring( 'BACKPACK_InstallInfo' ) );
  {$IFDEF SDLMODE}
	N := SelectMenu( EI_Menu , @MiscProcRedraw);
  {$ELSE}
	N := SelectMenu( EI_Menu );
  {$ENDIF}
	DisposeRPGMenu( EI_Menu );

	{ If a slot was selected, pass that info on to the workhorse. }
	if N <> -1 then begin
		{ Store the name here, since the item might get destroyed }
		{ during the installation process. }
		name := GearName( Item );
  {$IFDEF PATCH_CHEAT}
    {$IFDEF DEBUG}
		if InstallItem( GB, LList, TruePC, LocateGearByNumber( PC, N, False, MaxNum, 'InstallFrontend' ), Item ) then begin
    {$ELSE DEBUG}
		if InstallItem( GB, LList, TruePC, LocateGearByNumber( PC, N ), Item ) then begin
    {$ENDIF DEBUG}
  {$ELSE PATCH_CHEAT}
    {$IFDEF DEBUG}
		if InstallItem( GB , TruePC , LocateGearByNumber( PC , N, False, MaxNum, 'InstallFrontend' ) , Item ) then begin
    {$ELSE DEBUG}
		if InstallItem( GB , TruePC , LocateGearByNumber( PC , N ) , Item ) then begin
    {$ENDIF DEBUG}
  {$ENDIF PATCH_CHEAT}
			DialogMsg( ReplaceHash( MsgString( 'INSTALL_OK' ) , name ) );
		end else begin
			if Item = Nil then begin
				DialogMsg( ReplaceHash( MsgString( 'INSTALL_WRECK' ) , name ) );
			end else begin
				DialogMsg( ReplaceHash( MsgString( 'INSTALL_FAIL' ) , name ) );
			end;
		end;
	end;
{$ENDIF PATCH_CHEAT}
end;

{$IFDEF PATCH_GH}
Procedure InstallAmmo( GB: GameBoardPtr; PC , Gun , Ammo: GearPtr );
	{ Place the ammunition gear into the gun. }
var
	A,A2: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Gun) or (Gun^.G <= GG_DisposeGear) then Exit;
	if (NIL = Ammo) or (Ammo^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ To start with, unload any ammo currently in the gun. }
	A := Gun^.SubCom;
	while A <> Nil do begin
		A2 := A^.Next;

{$IFDEF PATCH_GH}
		if (GG_DisposeGear < A^.G) then begin
{$ENDIF PATCH_GH}
		if A^.G = GG_Ammo then begin
			DelinkGear( Gun^.SubCom , A );
			InsertInvCom( PC , A );
		end;
{$IFDEF PATCH_GH}
		end;
{$ENDIF PATCH_GH}

		A := A2;
	end;

	{ Delink the magazine from wherever it currently resides. }
	if IsInvCom( Ammo ) then begin
		DelinkGear( Ammo^.Parent^.InvCom , Ammo );
	end else if IsSubCom( Ammo ) then begin
		DelinkGear( Ammo^.Parent^.SubCom , Ammo );
	end;

	{ Stick the new magazine into the gun. }
	InsertSubCom( Gun , Ammo );

	{ Loading a gun takes time. }
	if GB <> Nil then begin
		WaitAMinute_Part( GB , PC , ReactionTime( PC ) );
	end;
end;

Function NeedAmmo( Weapon: GearPtr ): Boolean;
begin
{$IFDEF PATCH_GH}
	if (NIL = Weapon) or (Weapon^.G <= GG_DisposeGear) then Exit(False);
{$ENDIF PATCH_GH}
	if ( GG_Weapon <> Weapon^.G ) then exit( False );
	if ( GS_Ballistic <> Weapon^.S ) and ( GS_Missile <> Weapon^.S ) then exit( False );
	NeedAmmo := True;
end;

Function ReloadAmmo( GB: GameBoardPtr; User,Weapon: GearPtr ): Boolean;
var
	Ammo: GearPtr;
	msg: String;
begin
{$IFDEF PATCH_GH}
	if (NIL = User) or (User^.G <= GG_DisposeGear) then Exit(False);
	if (NIL = Weapon) or (Weapon^.G <= GG_DisposeGear) then Exit(False);
{$ENDIF PATCH_GH}
	if ( GG_Weapon <> Weapon^.G ) then exit( False );
	if ( GS_Ballistic <> Weapon^.S ) and ( GS_Missile <> Weapon^.S ) then exit( False );
	if ( NIL <> LocateGoodAmmo( Weapon ) ) then exit( True );

	Ammo := SearchGoodAmmo( User , Weapon );
	if ( NIL <> Ammo ) then begin
		DialogMsg( ReplaceHash( I18N_MsgString('ReloadAmmo','Ammo loaded'), GearName( LocatePilot( User ) ) , GearName(Weapon) , GearName(Ammo) ) );

		{ Loading a gun takes time. }
		InstallAmmo( GB , User , Weapon , Ammo );
	end;
	ReloadAmmo := ( NIL <> LocateGoodAmmo( Weapon ) );
end;
{$ELSE PATCH_GH}
Procedure InstallAmmo( GB: GameBoardPtr; PC , Gun , Ammo: GearPtr );
	{ Place the ammunition gear into the gun. }
var
	A,A2: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Gun) or (Gun^.G <= GG_DisposeGear) then Exit;
	if (NIL = Ammo) or (Ammo^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ To start with, unload any ammo currently in the gun. }
	A := Gun^.SubCom;
	while A <> Nil do begin
		A2 := A^.Next;

{$IFDEF PATCH_GH}
		if (GG_DisposeGear < A^.G) then begin
{$ENDIF PATCH_GH}
		if A^.G = GG_Ammo then begin
			DelinkGear( Gun^.SubCom , A );
			InsertInvCom( PC , A );
		end;
{$IFDEF PATCH_GH}
		end;
{$ENDIF PATCH_GH}

		A := A2;
	end;

	{ Delink the magazine from wherever it currently resides. }
	if IsInvCom( Ammo ) then begin
		DelinkGear( Ammo^.Parent^.InvCom , Ammo );
	end else if IsSubCom( Ammo ) then begin
		DelinkGear( Ammo^.Parent^.SubCom , Ammo );
	end;

	{ Stick the new magazine into the gun. }
	InsertSubCom( Gun , Ammo );

	{ Loading a gun takes time. }
	if GB <> Nil then WaitAMinute( GB , PC , ReactionTime( PC ) );
end;
{$ENDIF PATCH_GH}

Procedure InstallAmmoFrontend( GB: GameBoardPtr; PC , Item: GearPtr );
	{ Assign ITEM to a legal projectile weapon. Move it from the }
	{ general inventory into its new home. }
var
	IA_Menu: RPGMenuPtr;
	Gun: GearPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
	ReBrowse: Boolean;
{$ENDIF PATCH_CHEAT}
{$IFDEF DEBUG}
	MaxNum: LongInt;
{$ENDIF DEBUG}
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Build the slot selection menu. }
	IA_Menu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
{$IFDEF PATCH_CHEAT}
  {$IFDEF DEBUG}
	MaxNum := BuildSubMenu( IA_Menu , PC , Item , False, Cheat_InstallAmmo_ShowSubItem );
  {$ELSE DEBUG}
	BuildSubMenu( IA_Menu , PC , Item , False, Cheat_InstallAmmo_ShowSubItem );
  {$ENDIF DEBUG}
  {$IFDEF PATCH_I18N}
	if IA_Menu^.NumItem < 1 then AddRPGMenuItem( IA_Menu , ReplaceHash( I18N_MsgString('InstallAmmoFrontend','no weapon'), GearName(Item) ) , -1 );
  {$ELSE PATCH_I18N}
	if IA_Menu^.NumItem < 1 then AddRPGMenuItem( IA_Menu , '[no weapon for ' + GearName( Item ) + ']' , -1 );
  {$ENDIF PATCH_I18N}

	repeat
		ReBrowse := False;
  {$IFDEF SDLMODE}
		N := SelectMenu( IA_Menu , @MiscProcRedraw);
  {$ELSE}
		N := SelectMenu( IA_Menu );
  {$ENDIF}

		if N < -1 then begin
  {$IFDEF PATCH_I18N}
			DialogMSG( I18N_MsgString('InstallAmmoFrontend','You can not install to it') );
  {$ELSE PATCH_I18N}
			DialogMSG( 'You can not install to it.' );
  {$ENDIF PATCH_I18N}
			ReBrowse := True;
		end else if 0 <= N then begin
  {$IFDEF DEBUG}
			Gun := LocateGearByNumber( PC , N, False, MaxNum, 'InstallAmmoFrontend' );
  {$ELSE DEBUG}
			Gun := LocateGearByNumber( PC , N );
  {$ENDIF DEBUG}
  {$IFDEF PATCH_I18N}
			DialogMsg( ReplaceHash( I18N_MsgString('InstallAmmoFrontend','You load'), GearName(Gun), GearName(Item) ) );
  {$ELSE PATCH_I18N}
			DialogMsg( 'You load ' + GearName( Item ) + ' into ' + GearName( Gun ) + '.' );
  {$ENDIF PATCH_I18N}
			InstallAmmo( GB , PC , Gun , Item );
		end;
	until False = ReBrowse;
	DisposeRPGMenu( IA_Menu );
{$ELSE PATCH_CHEAT}
  {$IFDEF DEBUG}
	MaxNum := BuildSubMenu( IA_Menu , PC , Item , False );
  {$ELSE DEBUG}
	BuildSubMenu( IA_Menu , PC , Item , False );
  {$ENDIF DEBUG}
  {$IFDEF PATCH_I18N}
	if IA_Menu^.NumItem < 1 then AddRPGMenuItem( IA_Menu , ReplaceHash( I18N_MsgString('InstallAmmoFrontend','no weapon'), GearName(Item) ) , -1 );
  {$ELSE PATCH_I18N}
	if IA_Menu^.NumItem < 1 then AddRPGMenuItem( IA_Menu , '[no weapon for ' + GearName( Item ) + ']' , -1 );
  {$ENDIF PATCH_I18N}

	{ Select a slot for the item to go into. }
  {$IFDEF SDLMODE}
	N := SelectMenu( IA_Menu , @MiscProcRedraw);
  {$ELSE}
	N := SelectMenu( IA_Menu );
  {$ENDIF}
	DisposeRPGMenu( IA_Menu );

	{ If a slot was selected, pass that info on to the workhorse. }
	if N <> -1 then begin
  {$IFDEF DEBUG}
		Gun := LocateGearByNumber( PC , N, False, MaxNum, 'InstallAmmoFrontend' );
  {$ELSE DEBUG}
		Gun := LocateGearByNumber( PC , N );
  {$ENDIF DEBUG}
  {$IFDEF PATCH_I18N}
		DialogMsg( ReplaceHash( I18N_MsgString('InstallAmmoFrontend','You load'), GearName(Gun), GearName(Item) ) );
  {$ELSE PATCH_I18N}
		DialogMsg( 'You load ' + GearName( Item ) + ' into ' + GearName( Gun ) + '.' );
  {$ENDIF PATCH_I18N}
		InstallAmmo( GB , PC , Gun , Item );
	end;
{$ENDIF PATCH_CHEAT}
end;


Procedure DropFrontEnd( PC , Item: GearPtr );
	{ How to drop an item: Make sure PC is a root-level gear. }
	{ Delink ITEM from its current location. }
	{ Copy PC's location variables to ITEM. }
	{ Install ITEM as the next sibling of PC. }
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Make sure PC is at root level... }
	PC := FindRoot( PC );
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Delink ITEM from its parent... }
	DelinkGear( Item^.Parent^.InvCom , Item );

	{ Copy the location variables to ITEM... }
	SetNAtt( Item^.NA , NAG_Location , NAS_X , NAttValue( PC^.NA , NAG_Location , NAS_X ) );
	SetNAtt( Item^.NA , NAG_Location , NAS_Y , NAttValue( PC^.NA , NAG_Location , NAS_Y ) );
	if not OnTheMap( PC ) then SetNAtt( Item^.NA , NAG_Location , NAS_Team , NAV_DefPlayerTeam );

	{ Install ITEM as PC's sibling... }
	Item^.Next := PC^.Next;
	PC^.Next := Item;

	{ Do display stuff. }
{$IFDEF PATCH_I18N}
	DialogMsg( ReplaceHash( I18N_MsgString('DropFrontEnd','You drop'), GearName(Item) ) );
{$ELSE PATCH_I18N}
	DialogMsg( 'You drop ' + GearName( Item ) + '.' );
{$ENDIF PATCH_I18N}
end;

Procedure DropAllFrontEnd( GB: GameBoardPtr; var LList: GearPtr; PC , Part: GearPtr );
var
	Part_Next: GearPtr;
begin
	while ( NIL <> Part ) do begin
		Part_Next := Part^.Next;
		if CheckAlongPath_DisallowDropping( Part ) then begin
		end else begin
			DropFrontEnd( PC , Part );
		end;
		Part := Part_Next;
	end;
end;

Procedure DropEqpAllFrontEnd( GB: GameBoardPtr; var LList: GearPtr; PC , Master: GearPtr );
	Procedure CheckAlongPath( Part: GearPtr; IsInv: Boolean );
	var
		Part_Next: GearPtr;
	begin
		while ( NIL <> Part ) do begin
			Part_Next := Part^.Next;
			if IsInv and ( not CheckAlongPath_DisallowDropping( Part ) ) then begin
				DropFrontEnd( PC , Part );
			end;
			CheckAlongPath( Part^.InvCom , True );
			CheckAlongPath( Part^.SubCom , False );
			Part := Part_Next;
		end;
	end;
begin
	CheckAlongPath( Master^.InvCom , False );
	CheckAlongPath( Master^.SubCom , False );
end;

{$IFDEF PATCH_GH}
Function TradeFrontEnd_GetDest( GB: GameBoardPtr; PC , LList: GearPtr ): GearPtr;
var
	TI_Menu: RPGMenuPtr;
	M: GearPtr;
	Team: Integer;
	N: LongInt;
begin
{$IFDEF PATCH_CHEAT}
	if (NIL <> GB) then begin
		if not (Cheat_Trade_NotSafeArea or IsSafeArea( GB )) then begin
			DialogMsg( MsgSTring('TRANSFER_NOTHERE') );
			Exit( NIL );
		end;

		if not (IsSafeArea( GB )) then begin
			WaitAMinute( GB , PC , ReactionTime( PC ) );
		end;
	end;
{$ELSE PATCH_CHEAT}
	if not IsSafeArea( GB ) then begin
		DialogMsg( MsgSTring( 'TRANSFER_NOTHERE' ) );
		Exit( NIL );
	end;
{$ENDIF PATCH_CHEAT}

	{ Build the slot selection menu. }
	TI_Menu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
	N := 1;
	M := LList;
	Team := NAttValue( PC^.NA , NAG_Location , NAS_Team );

	{ This menu should contain all the masters from LList which }
	{ belong to Team 1. }
	while ( NIL <> M ) do begin
		if (GG_DisposeGear < M^.G) then begin
			if ( Team = NAV_DefPlayerTeam ) or ( Team = NAV_LancemateTeam ) then begin
				if IsMasterGear( M ) and ( ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) or ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) ) and ( M <> PC ) then begin
					AddRPGMenuItem( TI_Menu , GearName( M ) , N );
				end;
			end else begin
				if IsMasterGear( M ) and ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = Team ) and ( M <> PC ) then begin
					AddRPGMenuItem( TI_Menu , GearName( M ) , N );
				end;
			end;
			Inc( N );
		end;
		M := M^.Next;
	end;
	AlphaKeyMenu( TI_Menu );

{$IFDEF PATCH_I18N}
	if TI_Menu^.NumItem < 1 then AddRPGMenuItem( TI_Menu , I18N_MsgString( 'TradeFrontEnd' , 'CannotTrade' ) , -1 );
{$ELSE PATCH_I18N}
	if TI_Menu^.NumItem < 1 then AddRPGMenuItem( TI_Menu , '[cannot trade ' + GearName( Item ) + ']' , -1 );
{$ENDIF PATCH_I18N}

	{ Select a slot for the item to go into. }
{$IFDEF SDLMODE}
	if Cheat_DisplayGearInfo then begin
		MPR_InvMenu     := TI_Menu;
		MPR_InvMenuRGS  := LList;
	end;
	N := SelectMenu( TI_Menu , @MiscProcRedraw);
	MPR_InvMenuRGS  := NIL;
	MPR_InvMenu     := NIL;
{$ELSE}
	N := SelectMenu( TI_Menu );
{$ENDIF}
	DisposeRPGMenu( TI_Menu );

	TradeFrontEnd_GetDest := RetrieveGearSib( LList , N );
end;
{$ENDIF PATCH_GH}

Procedure TradeFrontend( GB: GameBoardPtr; PC , Item, LList: GearPtr );
	{ Assign ITEM to a different master. Move it from the }
	{ general inventory of PC into its new home. }
var
{$IFDEF PATCH_GH}
	M: GearPtr;
{$ELSE PATCH_GH}
	TI_Menu: RPGMenuPtr;
	M: GearPtr;
	Team,N: Integer;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
	{ Don't kick out the GG_DisposeGear of LList at here. }
{$ENDIF PATCH_GH}

{$IFDEF PATCH_GH}
	M := TradeFrontEnd_GetDest( GB , PC , LList );

	{ If a slot was selected, pass that info on to the workhorse. }
	if ( NIL <> M ) then begin
		if IsLegalSlot( M , Item ) then begin
			DelinkGear( Item^.Parent^.InvCom , Item );
			InsertInvCom( M , Item );
			DialogMsg( MsgString( 'BACKPACK_ItemTraded' ) );
		end else begin
			DialogMsg( MsgString( 'BACKPACK_NotTraded' ) );
		end;
	end;
{$ELSE PATCH_GH}
{$IFDEF PATCH_CHEAT}
	if (NIL <> GB) then begin
		if not (Cheat_Trade_NotSafeArea or IsSafeArea( GB )) then begin
			DialogMsg( MsgSTring('TRANSFER_NOTHERE') );
			Exit;
		end;

		if not (IsSafeArea( GB )) then begin
			WaitAMinute( GB , PC , ReactionTime( PC ) );
		end;
	end;
{$ELSE PATCH_CHEAT}
	if not IsSafeArea( GB ) then begin
		DialogMsg( MsgSTring( 'TRANSFER_NOTHERE' ) );
		Exit;
	end;
{$ENDIF PATCH_CHEAT}

	{ Build the slot selection menu. }
	TI_Menu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
	N := 1;
	M := LList;
	Team := NAttValue( PC^.NA , NAG_Location , NAS_Team );

	{ This menu should contain all the masters from LList which }
	{ belong to Team 1. }
	while M <> Nil do begin
		if ( Team = NAV_DefPlayerTeam ) or ( Team = NAV_LancemateTeam ) then begin
			if IsMasterGear( M ) and ( ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) or ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NaV_LancemateTeam ) ) and ( M <> PC ) then begin
				AddRPGMenuItem( TI_Menu , GearName( M ) , N );
			end;
		end else begin
			if IsMasterGear( M ) and ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = Team ) and ( M <> PC ) then begin
				AddRPGMenuItem( TI_Menu , GearName( M ) , N );
			end;
		end;
		M := M^.Next;
		Inc( N );
	end;
	AlphaKeyMenu( TI_Menu );

{$IFDEF PATCH_I18N}
	if TI_Menu^.NumItem < 1 then AddRPGMenuItem( TI_Menu , ReplaceHash( I18N_MsgString('TradeFrontend','cannot trade'), GearName(Item) ) , -1 );
{$ELSE PATCH_I18N}
	if TI_Menu^.NumItem < 1 then AddRPGMenuItem( TI_Menu , '[cannot trade ' + GearName( Item ) + ']' , -1 );
{$ENDIF PATCH_I18N}

	{ Select a slot for the item to go into. }
{$IFDEF SDLMODE}
	N := SelectMenu( TI_Menu , @MiscProcRedraw);
{$ELSE}
	N := SelectMenu( TI_Menu );
{$ENDIF}
	DisposeRPGMenu( TI_Menu );

	{ If a slot was selected, pass that info on to the workhorse. }
	if N <> -1 then begin
		M := RetrieveGearSib( LList , N );
		if IsLegalSlot( M , Item ) then begin
			DelinkGear( Item^.Parent^.InvCom , Item );
			InsertInvCom( M , Item );
			DialogMsg( MsgString( 'BACKPACK_ItemTraded' ) );
		end else begin
			DialogMsg( MsgString( 'BACKPACK_NotTraded' ) );
		end;
	end;
{$ENDIF PATCH_GH}
end;

{$IFDEF PATCH_GH}
Procedure TradeAllFrontend( GB: GameBoardPtr; PC , Item, LList: GearPtr );
	{ Assign ITEM to a different master. Move it from the }
	{ general inventory of PC into its new home. }
var
	M: GearPtr;
	Item_Next: GearPtr;
begin
	M := TradeFrontEnd_GetDest( GB , PC , LList );

	if ( NIL <> M ) then begin
		while ( NIL <> Item ) do begin
			Item_Next := Item^.Next;
			if CheckAlongPath_DisallowTransfering( Item ) then begin
			end else begin
				if IsLegalSlot( M , Item ) then begin
					DelinkGear( Item^.Parent^.InvCom , Item );
					InsertInvCom( M , Item );
{$IFDEF PATCH_I18N}
					DialogMsg( ReplaceHash( I18N_MsgString( 'TradeAllFrontend' , 'ItemTraded' ) , GearName( Item ) ) );
{$ELSE PATCH_I18N}
					DialogMsg( 'Item traded.' );
{$ENDIF PATCH_I18N}
				end else begin
{$IFDEF PATCH_I18N}
					DialogMsg( ReplaceHash( I18N_MsgString( 'TradeAllFrontend' , 'NotTraded' ) , GearName( Item ) ) );
{$ELSE PATCH_I18N}
					DialogMsg( 'Not traded.' );
{$ENDIF PATCH_I18N}
				end;
			end;
			Item := Item_Next;
		end;
	end;
end;

Procedure TradeEqpAllFrontEnd( GB: GameBoardPtr; PC , Item, LList: GearPtr );
	{ Assign ITEM to a different master. Move it from the }
	{ general inventory of PC into its new home. }
var
	M: GearPtr;

	Procedure CheckAlongPath( Part: GearPtr; IsInv: Boolean );
	var
		Part_Next: GearPtr;
	begin
		while ( NIL <> Part ) do begin
			Part_Next := Part^.Next;
			if IsInv and ( not CheckAlongPath_DisallowTransfering( Part ) ) then begin
				if IsLegalSlot( M , Part ) then begin
					DelinkGear( Part^.Parent^.InvCom , Part );
					InsertInvCom( M , Part );
{$IFDEF PATCH_I18N}
					DialogMsg( ReplaceHash( I18N_MsgString( 'TradeAllFrontend' , 'ItemTraded' ) , GearName( Part ) ) );
{$ELSE PATCH_I18N}
					DialogMsg( 'Item traded.' );
{$ENDIF PATCH_I18N}
				end else begin
{$IFDEF PATCH_I18N}
					DialogMsg( ReplaceHash( I18N_MsgString( 'TradeAllFrontend' , 'NotTraded' ) , GearName( Part ) ) );
{$ELSE PATCH_I18N}
					DialogMsg( 'Not traded.' );
{$ENDIF PATCH_I18N}
				end;
			end;
			CheckAlongPath( Part^.InvCom , True );
			CheckAlongPath( Part^.SubCom , False );
			Part := Part_Next;
		end;
	end;
begin
	M := TradeFrontEnd_GetDest( GB , PC , LList );

	if ( NIL <> M ) then begin
		CheckAlongPath( Item^.InvCom , False );
		CheckAlongPath( Item^.SubCom , False );
	end;
end;
{$ENDIF PATCH_GH}

Procedure FHQ_AssociatePilotMek( PC , M , LList: GearPtr );
	{ Associate the mecha with the pilot. }
begin
	AssociatePilotMek( LList , PC , M );
	DialogMsg( ReplaceHash( MsgString( 'FHQ_AssociatePM' ) , GearName( PC ) ) );
end;

Procedure FHQ_SelectPilotForMecha( GB: GameBoardPtr; Mek: GearPtr );
	{ Select a pilot for the mecha in question. }
	{ Pilots must be characters- they must either belong to the default }
	{ player team or, if they're lancemates, they must have a CID. }
	{ This is to prevent the PC from dominating some sewer rats and }
	{ training them to be pilots. }
var
	RPM: RPGMenuPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
	M: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = Mek) or (Mek^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Create the menu. }
	RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
	M := GB^.Meks;
	N := 1;
	while M <> Nil do begin
{$IFDEF PATCH_GH}
		if (GG_DisposeGear < M^.G) then begin
			if M^.G = GG_Character then begin
				if ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) and ( NAttValue( M^.NA , NAG_Personal , NAS_CID ) <> 0 ) then begin
					AddRPGMenuItem( RPM , GearName( M ) , N );
				end else if NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam then begin
					AddRPGMenuItem( RPM , GearName( M ) , N );
				end;
			end;
			Inc( N );
		end;
		M := M^.Next;
{$ELSE PATCH_GH}
		if M^.G = GG_Character then begin
			if ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_LancemateTeam ) and ( NAttValue( M^.NA , NAG_Personal , NAS_CID ) <> 0 ) then begin
				AddRPGMenuItem( RPM , GearName( M ) , N );
			end else if NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam then begin
				AddRPGMenuItem( RPM , GearName( M ) , N );
			end;
		end;
		M := M^.Next;
		Inc( N );
{$ENDIF PATCH_GH}
	end;
	RPMSortAlpha( RPM );
	AddRPGMenuItem( RPM , MSgString( 'EXIT' ) , -1 );

	{ Get a selection from the menu. }
{$IFDEF SDLMODE}
  {$IFDEF PATCH_GH}
	if Cheat_DisplayGearInfo then begin
		MPR_InvMenu     := RPM;
		MPR_InvMenuRGS  := GB^.Meks;
	end;
  {$ENDIF PATCH_GH}
	n := SelectMenu( RPM , @PlainRedraw );
  {$IFDEF PATCH_GH}
	MPR_InvMenuRGS  := NIL;
	MPR_InvMenu     := NIL;
  {$ENDIF PATCH_GH}
{$ELSE}
	n := SelectMenu( RPM );
{$ENDIF}
	DisposeRPGMenu( RPM );

	if N > 0 then begin
		M := RetrieveGearSib( GB^.Meks , N );
		FHQ_AssociatePilotMek( M , Mek , GB^.Meks );
	end;
end;

Procedure FHQ_SelectMechaForPilot( GB: GameBoardPtr; NPC: GearPtr );
	{ Select a pilot for the mecha in question. }
	{ Pilots must be characters- they must either belong to the default }
	{ player team or, if they're lancemates, they must have a CID. }
	{ This is to prevent the PC from dominating some sewer rats and }
	{ training them to be pilots. }
var
	RPM: RPGMenuPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
	M: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = NPC) or (NPC^.G <= GG_DisposeGear) then begin
  {$IFDEF SDLMODE}
		INFOGear := NIL;
  {$ENDIF}
		Exit;
	end;
{$ENDIF PATCH_GH}
{$IFDEF SDLMODE}
	INFOGear := NPC;
	INFOGB := GB;
{$ENDIF}

	{ Error check- only characters can pilot mecha! Pets can't. }
	if ( NAttValue( NPC^.NA , NAG_Personal , NAS_CID ) = 0 ) then begin
		DialogMsg( ReplaceHash( MsgString( 'FHQ_SMFP_NoPets' ) , GearName( NPC ) ) );
		Exit;
	end;

	{ Create the menu. }
	RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
	M := GB^.Meks;
	N := 1;
	while M <> Nil do begin
{$IFDEF PATCH_GH}
		if (GG_DisposeGear < M^.G) then begin
			if ( M^.G = GG_Mecha ) and ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) then begin
				AddRPGMenuItem( RPM , GearName( M ) , N );
			end;
			Inc( N );
		end;
		M := M^.Next;
{$ELSE PATCH_GH}
		if ( M^.G = GG_Mecha ) and ( NAttValue( M^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) then begin
			AddRPGMenuItem( RPM , GearName( M ) , N );
		end;
		M := M^.Next;
		Inc( N );
{$ENDIF PATCH_GH}
	end;
	RPMSortAlpha( RPM );
	AddRPGMenuItem( RPM , MSgString( 'EXIT' ) , -1 );

	{ Get a selection from the menu. }
{$IFDEF SDLMODE}
  {$IFDEF PATCH_GH}
	if Cheat_DisplayGearInfo then begin
		MPR_InvMenu     := RPM;
		MPR_InvMenuRGS  := GB^.Meks;
	end;
  {$ENDIF PATCH_GH}
	n := SelectMenu( RPM , @PlainRedraw );
  {$IFDEF PATCH_GH}
	MPR_InvMenuRGS  := NIL;
	MPR_InvMenu     := NIL;
  {$ENDIF PATCH_GH}
{$ELSE}
	n := SelectMenu( RPM );
{$ENDIF}
	DisposeRPGMenu( RPM );

	if N > 0 then begin
		M := RetrieveGearSib( GB^.Meks , N );
		FHQ_AssociatePilotMek( NPC , M , GB^.Meks );
	end;
end;

Procedure StartContinuousUseItem( GB: GameBoardPtr; TruePC , Item: GearPtr );
	{ The PC wants to use this item. Give it a try. }
var
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	if (NIL = TruePC) or (TruePC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Find the item's index number. If the item cannot be found }
	{ on the TRUEPC, then this item cannot be used. }
	N := FindGearIndex( TruePC , Item );
	if N > 0 then begin
		WaitAMinute( GB , TruePC , 1 );
		SetNAtt( TruePC^.NA , NAG_Location , NAS_SmartAction , NAV_UseItem );
		SetNAtt( TruePC^.NA , NAG_Location , NAS_SmartWeapon , N );
		SetNAtt( TruePC^.NA , NAG_Location , NAS_SmartCount , 3 );

		{ When an item is used in this way, exit the menu. }
		ForceQuit := True;
	end else begin
		DialogMsg( MsgString( 'BACKPACK_CantUse' ) );
	end;
end;

Procedure UseScriptItem( GB: GameBoardPtr; TruePC, Item: GearPtr; T: String );
	{ This item has a script effect. Exit the backpack and use it. }
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the NIL of GB at here. }
	{ Don't kick out the GG_DisposeGear of TruePC at here. }
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	if SAttValue( Item^.SA , T ) <> '' then begin
		{ Announce the intention. }
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_Script_' + T ) , GearName( Item ) ) );

		{ Using items takes time... }
{$IFDEF PATCH_GH}
		if (NIL <> TruePC) and (GG_DisposeGear < TruePC^.G) then begin
			WaitAMinute( GB , TruePC , ReactionTime( TruePC ) );
		end;
{$ELSE PATCH_GH}
		WaitAMinute( GB , TruePC , ReactionTime( TruePC ) );
{$ENDIF PATCH_GH}

		{ ...and also exits the backpack. }
		ForceQuit := True;
{$IFDEF PATCH_GH}
		if (NIL <> GB) then begin
			GFCombatDisplay( GB );
		end;
{$ELSE PATCH_GH}
		GFCombatDisplay( GB );
{$ENDIF PATCH_GH}

		{ Finally, trigger the script. }
		TriggerGearScript( GB , Item , T );
	end else begin
		{ Announce the lack of a valid script. }
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_CannotUseScript' ) , GearName( Item ) ) );
	end;
end;

{$IFDEF PATCH_CHEAT}
Procedure UseSkillOnItem( GB: GameBoardPtr; LList, TruePC, Item: GearPtr );
{$ELSE PATCH_CHEAT}
Procedure UseSkillOnItem( GB: GameBoardPtr; TruePC, Item: GearPtr );
{$ENDIF PATCH_CHEAT}
	{ The PC will have the option to use a CLUE-type skill on this }
	{ item, maybe to gain some new information, activate an effect, }
	{ or whatever else. }
var
	SkMenu: RPGMenuPtr;
	T: Integer;
	msg: String;
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the NIL of GB at here. }
	{ Don't kick out the GG_DisposeGear of LList at here. }
	{ Don't kick out the GG_DisposeGear of TruePC at here. }
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	SkMenu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );

	{ Add the usable skills. }
	for t := 1 to NumSkill do begin
		{ In order to be usable, it must be a CLUE type skill, }
		{ and the PC must have ranks in it. }
{$IFDEF PATCH_CHEAT}
		if ((USAGE_Clue = SkillMan[T].Usage)
			and (
				((NIL <> GB) and TeamHasSkill( GB, NAV_DefPlayerTeam, T ))
				or ((NIL <> LList) and TeamHasSkill( LList, T ))
				or HasTalent( TruePC, NAS_JackOfAll )
				)
			) then begin
{$ELSE PATCH_CHEAT}
		if ( SkillMan[ T ].Usage = USAGE_Clue ) and ( TeamHasSkill( GB , NAV_DefPlayerTeam , T ) or HasTalent( TruePC , NAS_JackOfAll ) ) then begin
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_I18N}
			msg := ReplaceHash( I18N_MsgString('BACKPACK','ClueSkillPrompt'), I18N_Name('SkillMan',SkillMan[ T ].Name), GearName(Item) );
{$ELSE PATCH_I18N}
			msg := ReplaceHash( MsgString( 'BACKPACK_ClueSkillPrompt' ) , SkillMan[ T ].Name );
			msg := ReplaceHash( msg , GearName( Item ) );
{$ENDIF PATCH_I18N}
			AddRPGMenuItem( SkMenu , msg , T );
		end;
	end;
{$IFDEF PATCH_I18N}
{$ELSE PATCH_I18N}
	RPMSortAlpha( SkMenu );
{$ENDIF PATCH_I18N}
	AddRPGMenuItem( SkMenu , MsgSTring( 'BACKPACK_CancelSkillUse' ) , -1 );

{$IFDEF SDLMODE}
	InfoGear := TruePC;
	InfoGB := GB;
	T := SelectMenu( SkMenu , @MiscProcRedraw);
{$ELSE}
	T := SelectMenu( SkMenu );
{$ENDIF}
	DisposeRPGMenu( SkMenu );

	if T <> -1 then begin
		UseScriptItem( GB , TruePC , Item , 'CLUE' + BStr( T ) );
	end;
end;

Procedure EatItem( GB: GameBoardPtr; TruePC , Item: GearPtr );
	{ The PC wants to eat this item. Give it a try. }
var
	effect: String;
{$IFDEF PATCH_GH}
	WaitTime: Int64;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	if (NIL = TruePC) or (TruePC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	TruePC := LocatePilot( TruePC );

{$IFDEF PATCH_GH}
	if (NIL = TruePC) or (TruePC^.G <= GG_DisposeGear) then begin
{$ELSE PATCH_GH}
	if TruePC = Nil then begin
{$ENDIF PATCH_GH}
		DialogMsg( ReplaceHash( MsgString( 'BACKPACK_CantBeEaten' ) , GearName( Item ) ) );

	end else if ( NAttValue( TruePC^.NA , NAG_Condition , NAS_Hunger ) > ( Item^.V div 2 ) ) or ( Item^.V = 0 ) then begin
		{ Show a message. }
{$IFDEF PATCH_I18N}
		DialogMsg( ReplaceHash( I18N_MsgString('BACKPACK','YouAreEating'), GearName(TruePC), GearName(Item) ) );
{$ELSE PATCH_I18N}
		DialogMsg( ReplaceHash( ReplaceHash( MsgString( 'BACKPACK_YouAreEating' ) , GearName( TruePC ) ) , GearName( Item ) ) );
{$ENDIF PATCH_I18N}

		{ Eating takes time... }
{$IFDEF PATCH_GH}
		WaitTime := ReactionTime( TruePC ) * GearMass( Item ) + 1;
		while (32767 < WaitTime) do begin
			WaitAMinute( GB, TruePC, 32767 );
			WaitTime := WaitTime - 32767;
		end;
		WaitAMinute( GB, TruePC, WaitTime );
{$ELSE PATCH_GH}
		WaitAMinute( GB , TruePC , ReactionTime( TruePC ) * GearMass( Item ) + 1 );
{$ENDIF PATCH_GH}

		{ ...and also exits the backpack. }
		ForceQuit := True;

		{ Locate the PC's Character record, then adjust hunger values. }
		AddNAtt( TruePC^.NA , NAG_Condition , NAS_Hunger , -Item^.V );
		AddMoraleDmg( TruePC , -( Item^.Stat[ STAT_MoraleBoost ] * FOOD_MORALE_FACTOR ) );

		{ Invoke the item's effect, if any. }
		effect := SAttValue( Item^.SA , 'EFFECT' );
		if effect <> '' then begin
			GFCombatDisplay( GB );
{$IFDEF PATCH_GH}
			EffectFrontEnd( GB , Item , TruePC , effect , '' );
{$ELSE PATCH_GH}
			EffectFrontEnd( GB , TruePC , effect , '' );
{$ENDIF PATCH_GH}
		end;

		{ Destroy the item, if appropriate. }
		Dec( Item^.Stat[ STAT_FoodQuantity ] );
		if Item^.Stat[ STAT_FoodQuantity ] < 1 then begin
			if IsInvCom( Item ) then begin
				RemoveGEar( Item^.Parent^.InvCom , Item );
			end else if IsSubCom( Item ) then begin
				RemoveGEar( Item^.Parent^.SubCom , Item );
			end;
		end;
	end else begin
		DialogMsg( MsgString( 'BACKPACK_NotHungry' ) );
	end;
end;


{$IFDEF PATCH_CHEAT}
Procedure ThisItemWasSelected( GB: GameBoardPtr; var LList: GearPtr; TruePC , PC , Item: GearPtr; ShowOnly: Boolean );
{$ELSE PATCH_CHEAT}
Procedure ThisItemWasSelected( GB: GameBoardPtr; var LList: GearPtr; TruePC , PC , Item: GearPtr );
{$ENDIF PATCH_CHEAT}
	{ TruePC is the primary character, who may be doing repairs }
	{  and stuff. }
	{ PC is the current master being examined, which may well be }
	{  a mecha belonging to the TruePC rather than the TruePC itself. }
	{ LList is a list of mecha and other things which may or may not }
	{  belong to the same team as TruePC et al. }
	{ Item is the piece of wargear currently being examined. }
var
	TIWS_Menu: RPGMenuPtr;
{$IFDEF PATCH_GH}
	SI,TI: Integer;
{$ENDIF PATCH_GH}
	N: Integer;
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the GG_DisposeGear of LList at here. }
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
	{if (NIL = TruePC) or (TruePC^.G <= GG_DisposeGear) then Exit;}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	TIWS_Menu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );

{$IFDEF PATCH_GH}
	SI := 1;
	TI := 1;
{$ELSE PATCH_GH}
{$IFDEF PATCH_CHEAT}
    if not(ShowOnly) then begin
{$ENDIF PATCH_CHEAT}
	if Item^.G = GG_Usable then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_UseItem' ) , GearName( Item ) ) , -9 );
	if Item^.G = GG_Consumable then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_EatItem' ) , GearName( Item ) ) , -10 );

	if SATtValue( Item^.SA , 'USE' ) <> '' then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_UseItemScript' ) , GearName( Item ) ) , -11 );

	if Item^.G = GG_Ammo then AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_LoadAmmo' ) , -5 );
	if IsInvCom( Item ) then begin
		if Item^.Parent = PC then begin
{$IFDEF PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('ThisItemWasSelected','Equip'), GearName(Item) ), -2 );
{$ELSE PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , 'Equip ' + GearName( Item ) , -2 );
{$ENDIF PATCH_I18N}
			if ( FindMaster( Item ) <> Nil ) and ( FindMaster( Item )^.G = GG_Mecha ) then begin
{$IFDEF PATCH_CHEAT}
				if (NIL <> TruePC) then begin
{$IFDEF PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Install'), GearName(Item) ) , -8 );
{$ELSE PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_Install' ) + GearName( Item ) , -8 );
{$ENDIF PATCH_I18N}
					if 0 < SAttValueToInt(Item^.SA,SATT_TRANSFORMABLE) then begin
{$IFDEF PATCH_I18N}
						AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Transformation'), GearName(Item) ) , -13 );
{$ELSE PATCH_I18N}
						AddRPGMenuItem( TIWS_Menu , MsgString( 'Transformation' ) + GearName( Item ) , -13 );
{$ENDIF PATCH_I18N}
					end;
				end;
{$ELSE PATCH_CHEAT}
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Install'), GearName(Item) ) , -8 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_Install' ) + GearName( Item ) , -8 );
{$ENDIF PATCH_I18N}
{$ENDIF PATCH_CHEAT}
			end;
		end else begin
{$IFDEF PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('ThisItemWasSelected','Unequip'), GearName(Item) ), -3 );
{$ELSE PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , 'Unequip ' + GearName( Item ) , -3 );
{$ENDIF PATCH_I18N}
		end;
{$IFDEF PATCH_CHEAT}
		if (NIL <> LList) then begin
			if (NIL = GB) or ((NIL <> GB) and (Cheat_Trade_NotSafeArea or IsSafeArea(GB))) then begin
				AddRPGMenuItem( TIWS_Menu, MsgString('BACKPACK_TradeItem'), -6 );
			end;
		end;
{$ELSE PATCH_CHEAT}
		if ( LList <> Nil ) and ( GB <> Nil ) and IsSafeArea( GB ) then AddRPGMenuItem ( TIWS_Menu , MsgString( 'BACKPACK_TradeItem' ) , -6 );
{$ENDIF PATCH_CHEAT}
		AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_DropItem' ) , -4 );
	end else if ( FindMaster( Item ) <> Nil ) and ( FindMaster( Item )^.G = GG_Mecha ) and CanBeExtracted( Item ) then begin
{$IFDEF PATCH_CHEAT}
		if (NIL <> TruePC) then
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_I18N}
		AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Remove'), GearName(Item) ) , -7 );
{$ELSE PATCH_I18N}
		AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_Remove' ) + GearName( Item ) , -7 );
{$ENDIF PATCH_I18N}
	end;
{$IFDEF PATCH_CHEAT}
	if not(IsInvCom(Item)) and (NIL = Item^.Parent) and (NIL <> TruePC) then begin
		if 0 < SAttValueToInt(Item^.SA,SATT_TRANSFORMABLE) then begin
{$IFDEF PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Transformation'), GearName(Item) ) , -13 );
{$ELSE PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , MsgString( 'Transformation' ) + GearName( Item ) , -13 );
{$ENDIF PATCH_I18N}
		end;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_CHEAT}
		if 0 < SAttValueToInt(Item^.SA,SATT_SEPARABLE) then begin
{$IFDEF PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','PurgeParts'), GearName(Item) ) , -14 );
{$ELSE PATCH_I18N}
			AddRPGMenuItem( TIWS_Menu , MsgString( 'PurgeParts' ) + GearName( Item ) , -14 );
{$ENDIF PATCH_I18N}
		end;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_CHEAT}
	end;
{$ENDIF PATCH_CHEAT}
	AddRepairOptions( TIWS_Menu , TruePC , Item );

	if ( Item^.G = GG_Weapon ) or ( ( Item^.G = GG_Ammo ) and ( Item^.S = GS_Grenade ) ) then begin
		if NAttValue( Item^.NA , NAG_WeaponModifier , NAS_SafetySwitch ) = 0 then begin
			AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_EngageSafety' ) , -12 );
		end else begin
			AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_DisengageSafety' ) , -12 );
		end;
	end;

	AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_UseSkillOnItem' ) , 1 );
{$IFDEF PATCH_CHEAT}
    end;
{$ENDIF PATCH_CHEAT}
	AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_ExitTIWS' ) , -1 );
{$IFDEF PATCH_CHEAT}
	if Cheat_ThisItemWasSelected_AddMenuKey then begin
		AlphaKeyMenu( TIWS_Menu );
	end;
{$ENDIF PATCH_CHEAT}
{$ENDIF PATCH_GH}

	repeat
		DisplayGearInfo( Item );
{$IFDEF PATCH_GH}
		TIWS_Menu := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );

{$IFDEF PATCH_CHEAT}
		if not(ShowOnly) then begin
{$ENDIF PATCH_CHEAT}
			if Item^.G = GG_Usable then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_UseItem' ) , GearName( Item ) ) , -9 );
			if Item^.G = GG_Consumable then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_EatItem' ) , GearName( Item ) ) , -10 );

			if SATtValue( Item^.SA , 'USE' ) <> '' then AddRPGMenuItem( TIWS_Menu , ReplaceHash( MsgString( 'BACKPACK_UseItemScript' ) , GearName( Item ) ) , -11 );

			if Item^.G = GG_Ammo then AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_LoadAmmo' ) , -5 );
			if IsInvCom( Item ) then begin
				if Item^.Parent = PC then begin
{$IFDEF PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('ThisItemWasSelected','Equip'), GearName(Item) ), -2 );
{$ELSE PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , 'Equip ' + GearName( Item ) , -2 );
{$ENDIF PATCH_I18N}
					if ( FindMaster( Item ) <> Nil ) and ( FindMaster( Item )^.G = GG_Mecha ) then begin
{$IFDEF PATCH_CHEAT}
						if (NIL <> TruePC) then begin
{$IFDEF PATCH_I18N}
							AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Install'), GearName(Item) ) , -8 );
{$ELSE PATCH_I18N}
							AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_Install' ) + GearName( Item ) , -8 );
{$ENDIF PATCH_I18N}
							if 0 < SAttValueToInt(Item^.SA,SATT_TRANSFORMABLE) then begin
{$IFDEF PATCH_I18N}
								AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Transformation'), GearName(Item) ) , -13 );
{$ELSE PATCH_I18N}
								AddRPGMenuItem( TIWS_Menu , MsgString( 'Transformation' ) + GearName( Item ) , -13 );
{$ENDIF PATCH_I18N}
							end;
						end;
{$ELSE PATCH_CHEAT}
{$IFDEF PATCH_I18N}
						AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Install'), GearName(Item) ) , -8 );
{$ELSE PATCH_I18N}
						AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_Install' ) + GearName( Item ) , -8 );
{$ENDIF PATCH_I18N}
{$ENDIF PATCH_CHEAT}
					end;
				end else begin
{$IFDEF PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('ThisItemWasSelected','Unequip'), GearName(Item) ), -3 );
{$ELSE PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , 'Unequip ' + GearName( Item ) , -3 );
{$ENDIF PATCH_I18N}
				end;
{$IFDEF PATCH_CHEAT}
				if (NIL <> LList) then begin
					if (NIL = GB) or ((NIL <> GB) and (Cheat_Trade_NotSafeArea or IsSafeArea(GB))) then begin
						if CheckAlongPath_DisallowTransfering( Item ) then begin
{$IFDEF PATCH_I18N}
							AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'YouCannotTransferIt' ) , 0 );
{$ELSE PATCH_I18N}
							AddRPGMenuItem( TIWS_Menu , 'You cannot transfer it.' , 0 );
{$ENDIF PATCH_I18N}
						end else begin
							AddRPGMenuItem( TIWS_Menu, MsgString('BACKPACK_TradeItem'), -6 );
						end;
					end;
				end;
{$ELSE PATCH_CHEAT}
				if ( LList <> Nil ) and ( GB <> Nil ) and IsSafeArea( GB ) then begin
					if CheckAlongPath_DisallowTransfering( Item ) then begin
{$IFDEF PATCH_I18N}
						AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'YouCannotTransferIt' ) , 0 );
{$ELSE PATCH_I18N}
						AddRPGMenuItem( TIWS_Menu , 'You cannot transfer it.' , 0 );
{$ENDIF PATCH_I18N}
					end else begin
						AddRPGMenuItem ( TIWS_Menu , MsgString( 'BACKPACK_TradeItem' ) , -6 );
					end;
				end;
{$ENDIF PATCH_CHEAT}
				if CheckAlongPath_DisallowDropping( Item ) then begin
					AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'YouCannotDropIt' ) , 0 );
				end else begin
					AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_DropItem' ) , -4 );
				end;
			end else if ( FindMaster( Item ) <> Nil ) and ( FindMaster( Item )^.G = GG_Mecha ) and CanBeExtracted( Item ) then begin
{$IFDEF PATCH_CHEAT}
				if (NIL <> TruePC) then
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Remove'), GearName(Item) ) , -7 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_Remove' ) + GearName( Item ) , -7 );
{$ENDIF PATCH_I18N}
			end;
{$IFDEF PATCH_CHEAT}
			if not(IsInvCom(Item)) and (NIL = Item^.Parent) and (NIL <> TruePC) then begin
				if 0 < SAttValueToInt(Item^.SA,SATT_TRANSFORMABLE) then begin
{$IFDEF PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','Transformation'), GearName(Item) ) , -13 );
{$ELSE PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , MsgString( 'Transformation' ) + GearName( Item ) , -13 );
{$ENDIF PATCH_I18N}
				end;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_CHEAT}
				if 0 < SAttValueToInt(Item^.SA,SATT_SEPARABLE) then begin
{$IFDEF PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , ReplaceHash( I18N_MsgString('BACKPACK','PurgeParts'), GearName(Item) ) , -14 );
{$ELSE PATCH_I18N}
					AddRPGMenuItem( TIWS_Menu , MsgString( 'PurgeParts' ) + GearName( Item ) , -14 );
{$ENDIF PATCH_I18N}
				end;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_CHEAT}
			end;
{$ENDIF PATCH_CHEAT}
			AddRepairOptions( TIWS_Menu , TruePC , Item );

			if ( Item^.G = GG_Weapon ) or ( ( Item^.G = GG_Ammo ) and ( Item^.S = GS_Grenade ) ) then begin
				if NAttValue( Item^.NA , NAG_WeaponModifier , NAS_SafetySwitch ) = 0 then begin
					AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_EngageSafety' ) , -12 );
				end else begin
					AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_DisengageSafety' ) , -12 );
				end;
			end;

			AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_UseSkillOnItem' ) , 1 );
			if DisallowSelling( Item ) then begin
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'AllowSelling' ) , 7 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , 'AllowSelling' , 7 );
{$ENDIF PATCH_I18N}
			end else begin
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'DisallowSelling' ) , 6 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , 'DisallowSelling' , 6 );
{$ENDIF PATCH_I18N}
			end;
{$IFDEF PATCH_CHEAT}
			if DisallowDropping( Item ) then begin
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'AllowDropping' ) , 9 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , 'AllowDropping' , 9 );
{$ENDIF PATCH_I18N}
			end else begin
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'DisallowDropping' ) , 8 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , 'DisallowDropping' , 8 );
{$ENDIF PATCH_I18N}
			end;
			if DisallowTransfering( Item ) then begin
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'AllowTransfering' ) , 11 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , 'AllowTransfering' , 11 );
{$ENDIF PATCH_I18N}
			end else begin
{$IFDEF PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , I18N_MsgString( 'ThisItemWasSelected' , 'DisallowTransfering' ) , 10 );
{$ELSE PATCH_I18N}
				AddRPGMenuItem( TIWS_Menu , 'DisallowTransfering' , 10 );
{$ENDIF PATCH_I18N}
			end;
		end;
{$ENDIF PATCH_CHEAT}
		AddRPGMenuItem( TIWS_Menu , MsgString( 'BACKPACK_ExitTIWS' ) , -1 );
{$IFDEF PATCH_CHEAT}
		if Cheat_ThisItemWasSelected_AddMenuKey then begin
			AlphaKeyMenu( TIWS_Menu );
		end;
{$ENDIF PATCH_CHEAT}
		TIWS_Menu^.SelectItem := SI;
		TIWS_Menu^.TopItem    := TI;
{$ENDIF PATCH_GH}
{$IFDEF SDLMODE}
		InfoGear := Item;
		InfoGB := GB;
		N := SelectMenu( TIWS_Menu , @MiscProcRedraw );
{$ELSE}
		N := SelectMenu( TIWS_Menu );
{$ENDIF}
{$IFDEF PATCH_GH}
		SI := TIWS_Menu^.SelectItem;
		TI := TIWS_Menu^.TopItem;
		DisposeRPGMenu( TIWS_Menu );
{$ENDIF PATCH_GH}
		if N > 100 then begin
			DoFieldRepair( GB , TruePC , Item , RepairSkillIndex[N-100] );
		end else begin
			case N of
{$IFDEF PATCH_CHEAT}
				1: UseSkillOnItem( GB, LList, TruePC, Item );
{$ELSE PATCH_CHEAT}
				1: UseSkillOnItem( GB , TruePC , Item );
{$ENDIF PATCH_CHEAT}
				-2: EquipItemFrontend( GB , PC , Item );
				-3: UnequipFrontEnd( GB , PC , Item );
				-4: DropFrontEnd( PC , Item );
				-5: InstallAmmoFrontEnd( GB , PC , Item );
				-6: TradeFrontEnd( GB , PC, Item , LList );
{$IFDEF PATCH_CHEAT}
				-7: ExtractFrontEnd( GB, LList, TruePC, PC, Item );
				-8: InstallFrontEnd( GB, LList, TruePC, PC, Item );
{$ELSE PATCH_CHEAT}
				-7: ExtractFrontEnd( GB , TruePC , PC , Item );
				-8: InstallFrontEnd( GB , TruePC , PC , Item );
{$ENDIF PATCH_CHEAT}
				-9: StartContinuousUseItem( GB , TruePC , Item );
				-10: EatItem( GB , PC , Item );
				-11: UseScriptItem( GB , TruePC , Item , 'USE' );
				-12: SetNAtt( Item^.NA , NAG_WeaponModifier , NAS_SafetySwitch , 1 - NAttValue( Item^.NA , NAG_WeaponModifier , NAS_SafetySwitch ) );
{$IFDEF PATCH_CHEAT}
				-13: UserTransformation( GB , Item , True );
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_CHEAT}
				-14: UserPurgeParts( GB , Item );
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_GH}
				6: SetSAtt( Item^.SA , TAG_DisallowSelling + ' <true>' );
				7: SetSAtt( Item^.SA , TAG_DisallowSelling + ' <>' );
				8: SetSAtt( Item^.SA , TAG_DisallowDropping + ' <true>' );
				9: SetSAtt( Item^.SA , TAG_DisallowDropping + ' <>' );
				10: SetSAtt( Item^.SA , TAG_DisallowTransfering + ' <true>' );
				11: SetSAtt( Item^.SA , TAG_DisallowTransfering + ' <>' );
{$ENDIF PATCH_GH}
			end;
		end;
	until ( N < 0 ) or ForceQuit;

{$IFDEF PATCH_GH}
{$ELSE PATCH_GH}
	DisposeRPGMenu( TIWS_Menu );
{$ENDIF PATCH_GH}
end;

{$IFDEF PATCH_CHEAT}
Procedure ThisItemWasSelected( GB: GameBoardPtr; var LList: GearPtr; TruePC , PC , Item: GearPtr );
begin
	ThisItemWasSelected( GB, LList, TruePC, PC, Item, False );
end;
{$ENDIF PATCH_CHEAT}

Function DoInvMenu( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr ): Boolean;
	{ Return TRUE if the user selected Quit. }
var
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
	top, sel: Integer;
{$ENDIF PATCH_CHEAT}
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the GG_DisposeGear of LList at here. }
	{if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit(True);}
	if (NIL = M) or (M^.G <= GG_DisposeGear) then Exit(True);
{$ENDIF PATCH_GH}

{$IFDEF PATCH_CHEAT}
	top := 0;
	sel := 0;
{$ENDIF PATCH_CHEAT}
	Repeat
{$IFDEF SDLMODE}
		InfoGear := M;
		InfoGB := GB;
{$ENDIF}
{$IFDEF PATCH_CHEAT}
		if Cheat_InvMenu_KeepPosition and (0 < sel) then begin
			InvRPM^.TopItem := top;
			SetItemByPosition( InvRPM, sel );
		end;
{$ENDIF PATCH_CHEAT}
{$IFDEF SDLMODE}
  {$IFDEF PATCH_GH}
		if Cheat_DisplayGearInfo then begin
			MPR_InvMenu     := InvRPM;
			MPR_InvMenuLGBN := M;
		end;
  {$ENDIF PATCH_GH}
		N := SelectMenu( INVRPM , @MiscProcRedraw);
  {$IFDEF PATCH_GH}
		MPR_InvMenuLGBN := NIL;
		MPR_InvMenu     := NIL;
  {$ENDIF PATCH_GH}
{$ELSE}
		N := SelectMenu( InvRPM );
{$ENDIF}
{$IFDEF PATCH_CHEAT}
		top := InvRPM^.TopItem;
		sel := InvRPM^.SelectItem;
{$ENDIF PATCH_CHEAT}

		{ If an item was selected, pass it along to the appropriate }
		{ procedure. }
		if N > 0 then begin
{$IFDEF PATCH_CHEAT}
			if ' ' = RPMLocateByPosition( InvRPM, sel )^.msg[1] then begin
  {$IFDEF DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N, False, InvRPM_MaxNum, 'DoInvMenu' ), True );
  {$ELSE DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ), True );
  {$ENDIF DEBUG}
			end else begin
  {$IFDEF DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N, False, InvRPM_MaxNum, 'DoInvMenu' ) );
  {$ELSE DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ) );
  {$ENDIF DEBUG}
			end;
{$ELSE PATCH_CHEAT}
  {$IFDEF DEBUG}
			ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N, False, InvRPM_MaxNum, 'DoInvMenu' ) );
  {$ELSE DEBUG}
			ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ) );
  {$ENDIF DEBUG}
{$ENDIF PATCH_CHEAT}
			{ Restore the display. }
			UpdateBackpack( M );
			DisplayGearInfo( M );
{$IFDEF PATCH_GH}
		end else if ( -3 = N ) then begin
			DropAllFrontEnd( GB , LList , M , M^.InvCom );
			UpdateBackpack( M );
			DisplayGearInfo( M );
			N := 0;
		end else if ( -4 = N ) then begin
			TradeAllFrontEnd( GB , M , M^.InvCom , LList );
			UpdateBackpack( M );
			N := 0;
{$ENDIF PATCH_GH}
		end;
{$IFDEF PATCH_CHEAT}
		if N = -128 then begin
  {$IFDEF DEBUG}
			SwapMenu( ZONE_InvMenu , LocateGearByNumber( M , RPMLocateByPosition( InvRPM , InvRPM^.SelectItem )^.value, False, InvRPM_MaxNum, 'DoInvMenu' ) );
  {$ELSE DEBUG}
			SwapMenu( ZONE_InvMenu , LocateGearByNumber( M , RPMLocateByPosition( InvRPM , InvRPM^.SelectItem )^.value ) );
  {$ENDIF DEBUG}
			UpdateBackpack( M );
			DisplayGearInfo( M );
			N := 0;
		end;
{$ENDIF PATCH_CHEAT}
	until ( N < 0 ) or ForceQuit;

{$IFNDEF SDLMODE}
	DisplayMenu( InvRPM );
{$ENDIF}

	DoInvMenu := N=-1;
end;

{$IFDEF SDLMODE}
Procedure EqpRedraw;
	{ Show Inventory, select Equipment. }
{$IFDEF PATCH_GH}
var
	Mek: GearPtr;
	MekNum: LongInt;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	if (NIL <> InfoGB) then begin
		QuickCombatDisplay( InfoGB );
	end;
	DrawBPBorder;
	Mek := InfoGear;
	if (NIL <> MPR_InvMenu) then begin
		MekNum := RPMLocateByPosition(MPR_InvMenu,MPR_InvMenu^.selectitem)^.value;
		if (0 <= MekNum) then begin
  {$IFDEF DEBUG}
			Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum, False, 0, '' );
  {$ELSE DEBUG}
			Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum );
  {$ENDIF DEBUG}
		end;
	end;
	if (NIL <> Mek) and (GG_DisposeGear < Mek^.G) then begin
		DisplayGearInfo( Mek, InfoGB );
	end;
{$ELSE PATCH_GH}
	QuickCombatDisplay( InfoGB );
	DrawBPBorder;
	DisplayGearInfo( InfoGear , InfoGB );
{$ENDIF PATCH_GH}
	DisplayMenu( InvRPM , Nil );
	NFGameMsg( MsgString( 'BACKPACK_Directions' ) , ZONE_Menu , MenuItem );
end;
{$ENDIF}

Function DoEqpMenu( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr ): Boolean;
	{ Return TRUE if the user selected Quit. }
var
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
	top, sel: Integer;
{$ENDIF PATCH_CHEAT}
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the GG_DisposeGear of LList at here. }
	{if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit(True);}
	if (NIL = M) or (M^.G <= GG_DisposeGear) then Exit(True);
{$ENDIF PATCH_GH}

{$IFDEF PATCH_CHEAT}
	top := 0;
	sel := 0;
{$ENDIF PATCH_CHEAT}
	Repeat
{$IFDEF SDLMODE}
		InfoGear := M;
		InfoGB := GB;
{$ENDIF}
{$IFDEF PATCH_CHEAT}
		if Cheat_EqpMenu_KeepPosition and (0 < sel) then begin
			EqpRPM^.TopItem := top;
			SetItemByPosition( EqpRPM, sel );
		end;
{$ENDIF PATCH_CHEAT}
{$IFDEF SDLMODE}
  {$IFDEF PATCH_GH}
		if Cheat_DisplayGearInfo then begin
			MPR_InvMenu     := EqpRPM;
			MPR_InvMenuLGBN := M;
		end;
  {$ENDIF PATCH_GH}
		N := SelectMenu( EqpRPM , @EqpRedraw);
  {$IFDEF PATCH_GH}
		MPR_InvMenuLGBN := NIL;
		MPR_InvMenu     := NIL;
  {$ENDIF PATCH_GH}
{$ELSE}
		N := SelectMenu( EqpRPM );
{$ENDIF}
{$IFDEF PATCH_CHEAT}
		top := EqpRPM^.TopItem;
		if 0 < N then begin
			SetItemByValue( EqpRPM, N );
		end;
		sel := EqpRPM^.SelectItem;
{$ENDIF PATCH_CHEAT}

		{ If an item was selected, pass it along to the appropriate }
		{ procedure. }
		if N > 0 then begin
{$IFDEF PATCH_CHEAT}
			if ' ' = RPMLocateByPosition( EqpRPM, sel )^.msg[1] then begin
  {$IFDEF DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N, False, EqpRPM_MaxNum, 'DoEqpMenu' ), True );
  {$ELSE DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ), True );
  {$ENDIF DEBUG}
			end else begin
  {$IFDEF DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N, False, EqpRPM_MaxNum, 'DoEqpMenu' ) );
  {$ELSE DEBUG}
				ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ) );
  {$ENDIF DEBUG}
			end;
{$ELSE PATCH_CHEAT}
  {$IFDEF DEBUG}
			ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N, False, EqpRPM_MaxNum, 'DoEqpMenu' ) );
  {$ELSE DEBUG}
			ThisItemWasSelected( GB , LList , PC , M , LocateGearByNumber( M , N ) );
  {$ENDIF DEBUG}
{$ENDIF PATCH_CHEAT}
			{ Restore the display. }
			UpdateBackpack( M );
			DisplayGearInfo( M );
{$IFDEF PATCH_GH}
		end else if ( -3 = N ) then begin
			DropEqpAllFrontEnd( GB , LList , M , M );
			UpdateBackpack( M );
			DisplayGearInfo( M );
			N := 0;
		end else if ( -4 = N ) then begin
			TradeEqpAllFrontEnd( GB , M , M , LList );
			UpdateBackpack( M );
			N := 0;
{$ENDIF PATCH_GH}
		end;
	until ( N < 0 ) or ForceQuit;

{$IFNDEF SDLMODE}
	DisplayMenu( EqpRPM );
{$ENDIF}

	DoEqpMenu := N=-1;
end;


Procedure RealBackpack( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr; StartWithInv: Boolean );
	{ This is the backpack routine which should allow the player to go }
	{ through all the stuff in his/her inventory, equip items, drop them, }
	{ reload weapons, and whatnot. It is based roughly upon the procedures }
	{ from DeadCold. }
var
	QuitBP: Boolean;
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the GG_DisposeGear of LList at here. }
	{if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;}
	if (NIL = M) or (M^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Set up the display. }
	DrawBPBorder;
	ForceQuit := False;

	{ Initialize menus to NIL, then create them. }
	InvRPM := Nil;
	EqpRPM := Nil;
	UpdateBackpack( M );

	repeat
{$IFNDEF SDLMODE}
		GameMsg( MsgString( 'BACKPACK_Directions' ) , ZONE_Menu , MenuItem );
{$ENDIF}
		if StartWithInv then begin
			QuitBP := DoInvMenu( GB , LList , PC , M );
		end else begin
			QuitBP := DoEqpMenu( GB , LList , PC , M );
		end;

		{ If we have not been ordered to exit the loop, we must }
		{ have been ordered to switch menus. }
		StartWithInv := Not StartWithInv;
	until QuitBP or ForceQuit;

	DisposeRPGMenu( InvRPM );
	DisposeRPGMenu( EqpRPM );
end;

Procedure LancemateBackpack( GB: GameBoardPtr; PC,NPC: GearPtr );
	{ This is a header for the REALBACKPACK function. }
begin
	RealBackPack( GB , GB^.Meks , PC , NPC , True );
end;

Procedure BackpackMenu( GB: GameBoardPtr; PC: GearPtr; StartWithInv: Boolean );
	{ This is a header for the REALBACKPACK function. }
begin
{$IFDEF PATCH_GH}
	if (NIL = GB) then Exit;
{$ENDIF PATCH_GH}
	RealBackPack( GB , GB^.Meks , PC , PC , StartWithInv );
end;

{$IFDEF SDLMODE}
	Procedure MPERedraw;
		{ Show Inventory, select Equipment. }
  {$IFDEF PATCH_GH}
	var
		Mek: GearPtr;
		MekNum: LongInt;
	begin
		if (NIL <> InfoGB) then begin
			QuickCombatDisplay( InfoGB );
		end;
		DrawBPBorder;
		Mek := NIL;
		if (NIL <> MPR_InvMenu) then begin
			MekNum := RPMLocateByPosition(MPR_InvMenu,MPR_InvMenu^.selectitem)^.value;
			if (0 <= MekNum) then begin
  {$IFDEF DEBUG}
				Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum, False, 0, '' );
  {$ELSE DEBUG}
				Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum );
  {$ENDIF DEBUG}
			end;
		end else if (NIL <> InfoGear) then begin
			Mek := InfoGear;
		end;
		if (NIL <> InfoGear) and (GG_DisposeGear < InfoGear^.G) then begin
			NFGameMsg( FullGearName(InfoGear) + ' '  + MechaDescription(InfoGear), ZONE_EqpMenu, InfoGreen );
		end;
		if (NIL <> Mek) and (GG_DisposeGear < Mek^.G) then begin
			DisplayGearInfo( Mek );
		end;
	end;
  {$ELSE PATCH_GH}
	begin
		QuickCombatDisplay( InfoGB );
		DrawBPBorder;
		NFGameMsg( FullGearName( INFOGear ) + ' '  + MechaDescription( InfoGear) , ZONE_EqpMenu , InfoGreen );
		DisplayGearInfo( InfoGear );
	end;
  {$ENDIF PATCH_GH}
{$ENDIF}

Procedure MechaPartEditor( GB: GameBoardPtr; var LList: GearPtr; PC,Mek: GearPtr );
	{ This procedure may be used to browse through all the various }
	{ bits of a mecha and examine each one individually. }
var
	RPM: RPGMenuPtr;
{$IFDEF PATCH_CHEAT}
	N: LongInt;
	top,sel: Integer;
{$ELSE PATCH_CHEAT}
{$IFDEF PATCH_GH}
	N: LongInt;
	I: Integer;
{$ELSE PATCH_GH}
	N,I: Integer;
{$ENDIF PATCH_GH}
{$ENDIF PATCH_CHEAT}
{$IFDEF DEBUG}
	MaxNum: LongInt;
{$ENDIF DEBUG}
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the GG_DisposeGear of LList at here. }
	{if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;}
	if (NIL = Mek) or (Mek^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Set up the display. }
	DrawBPBorder;
{$IFDEF PATCH_CHEAT}
	top := 0;
	sel := 0;
{$ELSE PATCH_CHEAT}
	I := 0;
{$ENDIF PATCH_CHEAT}

	Repeat
		RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_InvMenu );
{$IFDEF DEBUG}
		MaxNum := BuildGearMenu( RPM , Mek );
{$ELSE DEBUG}
		BuildGearMenu( RPM , Mek );
{$ENDIF DEBUG}
{$IFDEF PATCH_CHEAT}
		if Cheat_MechaPartEditor_KeepPosition and (0 < sel) then begin
			RPM^.TopItem := top;
			SetItemByPosition( RPM, sel );
		end;
{$ELSE PATCH_CHEAT}
		if I > 0 then SetItemByPosition( RPM , I );
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_I18N}
		AddRPGMenuItem( RPM , I18N_MsgString('MechaPartEditor','Exit Editor') , -1 );
{$ELSE PATCH_I18N}
		AddRPGMenuItem( RPM , 'Exit Editor' , -1 );
{$ENDIF PATCH_I18N}
{$IFDEF PATCH_CHEAT}
		if Cheat_MenuOrder_Edit then begin
			AddRPGMenuKey( RPM , KeyMap[ KMC_EditMenuOrder ].KCode , -128 );
		end;
		if Cheat_MechaPartEditor_AddMenuKey then begin
			AlphaKeyMenu( RPM );
		end;
{$ENDIF PATCH_CHEAT}

{$IFNDEF SDLMODE}
		GameMsg( FullGearName( Mek ) + ' '  + MechaDescription( Mek ) , ZONE_EqpMenu , InfoGreen );
{$ENDIF}
		DisplayGearInfo( Mek );
{$IFDEF SDLMODE}
		InfoGear := Mek;
		InfoGB := GB;
  {$IFDEF PATCH_GH}
		if Cheat_DisplayGearInfo then begin
			MPR_InvMenu     := RPM;
			MPR_InvMenuLGBN := Mek;
		end;
  {$ENDIF PATCH_GH}
		N := SelectMenu( RPM , @MPERedraw);
  {$IFDEF PATCH_GH}
		MPR_InvMenuLGBN := NIL;
		MPR_InvMenu     := NIL;
  {$ENDIF PATCH_GH}
{$ELSE}
		N := SelectMenu( RPM );
{$ENDIF}
{$IFDEF PATCH_CHEAT}
		top := RPM^.TopItem;
		if 0 < N then begin
			SetItemByValue( RPM, N );
		end;
		sel := RPM^.SelectItem;
		if N = -128 then begin
  {$IFDEF SDLMODE}
    {$IFDEF DEBUG}
			InfoGear := LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.SelectItem )^.value, False, MaxNum, 'MechaPartEditor' );
    {$ELSE DEBUG}
			InfoGear := LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.SelectItem )^.value );
    {$ENDIF DEBUG}
  {$ENDIF SDLMODE}
  {$IFDEF DEBUG}
			SwapMenu( ZONE_InvMenu , LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.SelectItem )^.value, False, MaxNum, 'MechaPartEditor'  ) );
  {$ELSE DEBUG}
			SwapMenu( ZONE_InvMenu , LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.SelectItem )^.value ) );
  {$ENDIF DEBUG}
		end;
{$ELSE PATCH_CHEAT}
		I := RPM^.SelectItem;
{$ENDIF PATCH_CHEAT}
		DisposeRPGMenu( RPM );

		if N > -1 then begin
{$IFDEF DEBUG}
			ThisItemWasSelected( GB , LList , PC , Mek , LocateGearByNumber( Mek , N, False, MaxNum, 'MechaPartEditor' ) );
{$ELSE DEBUG}
			ThisItemWasSelected( GB , LList , PC , Mek , LocateGearByNumber( Mek , N ) );
{$ENDIF DEBUG}
		end;
	until N = -1;

end;

{$IFDEF SDLMODE}
Procedure PartBrowserRedraw;
	{ Redraw the screen for the part browser. }
{$IFDEF PATCH_GH}
var
	Mek: GearPtr;
	MekNum: LongInt;
{$ENDIF PATCH_GH}
begin
	if MPB_Redraw <> Nil then MPB_Redraw;
	{ Don't kick out the GG_DisposeGear at here. }
  {$IFDEF PATCH_GH}
	Mek := NIL;
	if (NIL <> MPR_InvMenu) then begin
		MekNum := RPMLocateByPosition(MPR_InvMenu,MPR_InvMenu^.selectitem)^.value;
		if (0 <= MekNum) then begin
  {$IFDEF DEBUG}
			Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum, False, 0, '' );
  {$ELSE DEBUG}
			Mek := LocateGearByNumber( MPR_InvMenuLGBN, MekNum );
  {$ENDIF DEBUG}
		end;
	end else if (NIL <> MPB_Gear) then begin
		Mek := MPB_Gear;
	end;
	if (NIL <> Mek) and (GG_DisposeGear < Mek^.G) then begin
		DisplayGearInfo( Mek, True );
	end;
  {$ELSE PATCH_GH}
	if MPB_Gear <> Nil then DisplayGearInfo( MPB_Gear );
  {$ENDIF PATCH_GH}
end;
{$ENDIF}

{$IFDEF PATCH_GH}
  {$IFDEF SDLMODE}
Procedure MechaPartBrowser( Mek: GearPtr; RDP: RedrawProcedureType );
begin
	MechaPartBrowser( Mek, False, RDP );
end;
  {$ELSE SDLMODE}
Procedure MechaPartBrowser( Mek: GearPtr );
begin
	MechaPartBrowser( Mek, False );
end;
  {$ENDIF SDLMODE}
{$ENDIF PATCH_GH}

{$IFDEF PATCH_GH}
  {$IFDEF SDLMODE}
Procedure MechaPartBrowser( Mek: GearPtr; DebugMode: Boolean; RDP: RedrawProcedureType );
  {$ELSE SDLMODE}
Procedure MechaPartBrowser( Mek: GearPtr; DebugMode: Boolean );
  {$ENDIF SDLMODE}
{$ELSE PATCH_GH}
  {$IFDEF SDLMODE}
Procedure MechaPartBrowser( Mek: GearPtr; RDP: RedrawProcedureType );
  {$ELSE}
Procedure MechaPartBrowser( Mek: GearPtr );
  {$ENDIF}
{$ENDIF PATCH_GH}
	{ This procedure may be used to browse through all the various }
	{ bits of a mecha and examine each one individually. }
var
	RPM: RPGMenuPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
	ReBrowse: Boolean;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_GH}
	top, sel: Integer;
  {$IFDEF SDLMODE}
  {$ELSE SDLMODE}
	MPB_Gear: GearPtr;
  {$ENDIF SDLMODE}
{$ENDIF PATCH_GH}
{$IFDEF DEBUG}
	MaxNum: LongInt;
{$ENDIF DEBUG}

{$IFDEF PATCH_CHEAT}
	Key: String;

	Procedure MechaPartBrowser_SearchForward;
	var
		RPM_Item: RPGMenuItemPtr;
	begin
	{$IFDEF SDLMODE}
		Key := GetStringFromUser( 'Search String Forward', NIL, Key );
	{$ELSE SDLMODE}
		Key := GetStringFromUser( 'Search String Forward', Key );
	{$ENDIF SDLMODE}
		sel := RPM^.SelectItem;
		RPM_Item := RPMLocateByPosition( RPM, sel );
		repeat
			Inc( sel );
			RPM_Item := RPM_Item^.Next;
			if (RPM^.NumItem < sel) then begin
				sel := 1;
				RPM_Item := RPM^.FirstItem;
			end;
			if 0 < Pos( Key, RPM_Item^.msg ) then begin
				RPM^.SelectItem := sel;
				break;
			end;
		until sel = RPM^.SelectItem;
	end;

	Procedure MechaPartBrowser_SearchBackward;
	var
		RPM_Item: RPGMenuItemPtr;
	begin
	{$IFDEF SDLMODE}
		Key := GetStringFromUser( 'Search String Backward', NIL, Key );
	{$ELSE SDLMODE}
		Key := GetStringFromUser( 'Search String Backward', Key );
	{$ENDIF SDLMODE}
		sel := RPM^.SelectItem;
		repeat
			Dec( sel );
			if (sel < 1) then begin
				sel := RPM^.NumItem;
			end;
			RPM_Item := RPMLocateByPosition( RPM, sel );
			if 0 < Pos( Key, RPM_Item^.msg ) then begin
				RPM^.SelectItem := sel;
				break;
			end;
		until sel = RPM^.SelectItem;
	end;
{$ENDIF PATCH_CHEAT}
begin
{$IFDEF PATCH_GH}
	if not(DebugMode) then begin
		if (NIL = Mek) or (Mek^.G <= GG_DisposeGear) then Exit;
	end;
	top := 0;
	sel := 0;
{$ENDIF PATCH_GH}

{$IFDEF PATCH_CHEAT}
	Key := '';

    repeat
	ReBrowse := False;
{$ENDIF PATCH_CHEAT}
{$IFDEF SDLMODE}
	MPB_Redraw := RDP;
	MPB_Gear := Mek;
{$ENDIF}
{$IFDEF PATCH_GH}
	MPB_Gear := Mek;
	if not(DebugMode) then begin
		if (NIL = MPB_Gear) or (MPB_Gear^.G <= GG_DisposeGear) then MPB_Gear := NIL;
	end;
{$ENDIF PATCH_GH}

	RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
{$IFDEF PATCH_GH}
  {$IFDEF DEBUG}
	MaxNum := BuildGearMenu( RPM, Mek, DebugMode );
  {$ELSE DEBUG}
	BuildGearMenu( RPM, Mek, DebugMode );
  {$ENDIF DEBUG}
{$ELSE PATCH_GH}
  {$IFDEF DEBUG}
	MaxNum := BuildGearMenu( RPM , Mek );
  {$ELSE DEBUG}
	BuildGearMenu( RPM , Mek );
  {$ENDIF DEBUG}
{$ENDIF PATCH_GH}
{$IFDEF PATCH_I18N}
	AddRPGMenuItem( RPM , I18N_MsgString('MechaPartBrowser','Exit Browser') , -1 );
{$ELSE PATCH_I18N}
	AddRPGMenuItem( RPM , 'Exit Browser' , -1 );
{$ENDIF PATCH_I18N}
{$IFDEF PATCH_CHEAT}
	if Cheat_MenuOrder_Edit then begin
		AddRPGMenuKey( RPM , KeyMap[ KMC_EditMenuOrder ].KCode , -128 );
	end;
	if DebugMode and Cheat_MechaPartBrowser_Delete then begin
		AddRPGMenuKey( RPM , KeyMap[ KMC_Stop ].KCode , -3 );
	end;
	if DebugMode then begin
		AddRPGMenuKey( RPM , KeyMap[ KMC_ExamineMap ].KCode , -4 );
		AddRPGMenuKey( RPM , KeyMap[ KMC_Search ].KCode , -5 );
	end;
{$ENDIF PATCH_CHEAT}


{$IFDEF PATCH_GH}
	if 0 < sel then begin
		RPM^.TopItem := top;
		SetItemByPosition( RPM, sel );
	end;
{$ENDIF PATCH_GH}
	Repeat
{$IFDEF PATCH_GH}
		DisplayGearInfo( MPB_Gear, DebugMode );
  {$IFDEF SDLMODE}
		if Cheat_DisplayGearInfo then begin
			MPR_InvMenu     := RPM;
			MPR_InvMenuLGBN := Mek;
		end;
		N := SelectMenu( RPM , @PartBrowserRedraw );
		MPR_InvMenuLGBN := NIL;
		MPR_InvMenu     := NIL;
  {$ELSE SDLMODE}
		N := SelectMenu( RPM );
  {$ENDIF SDLMODE}
{$ELSE PATCH_GH}
  {$IFDEF SDLMODE}
		N := SelectMenu( RPM , @PartBrowserRedraw );
  {$ELSE}
		DisplayGearInfo( Mek );
		N := SelectMenu( RPM );
  {$ENDIF}
{$ENDIF PATCH_GH}

		if N > -1 then begin
{$IFDEF PATCH_GH}
  {$IFDEF DEBUG}
			MPB_Gear := LocateGearByNumber( Mek, N, DebugMode, MaxNum, 'MechaPartBrowser' );
  {$ELSE DEBUG}
			MPB_Gear := LocateGearByNumber( Mek, N, DebugMode );
  {$ENDIF DEBUG}
{$ELSE PATCH_GH}
  {$IFDEF SDLMODE}
    {$IFDEF DEBUG}
			MPB_Gear := LocateGearByNumber( Mek , N, False, MaxNum, 'MechaPartBrowser' );
    {$ELSE DEBUG}
			MPB_Gear := LocateGearByNumber( Mek , N );
    {$ENDIF DEBUG}
  {$ELSE}
    {$IFDEF DEBUG}
			DisplayGearInfo( LocateGearByNumber( Mek , N, False, MaxNum, 'MechaPartBrowser' ) );
    {$ELSE DEBUG}
			DisplayGearInfo( LocateGearByNumber( Mek , N ) );
    {$ENDIF DEBUG}
			EndOFGameMoreKey;
  {$ENDIF}
{$ENDIF PATCH_GH}

{$IFDEF PATCH_CHEAT}
		end else if N = -128 then begin
  {$IFDEF PATCH_GH}
    {$IFDEF DEBUG}
			MPB_Gear := LocateGearByNumber( Mek, RPMLocateByPosition( RPM , RPM^.selectitem )^.value, DebugMode, MaxNum, 'MechaPartBrowser' );
    {$ELSE DEBUG}
			MPB_Gear := LocateGearByNumber( Mek, RPMLocateByPosition( RPM , RPM^.selectitem )^.value, DebugMode );
    {$ENDIF DEBUG}
  {$ELSE PATCH_GH}
    {$IFDEF SDLMODE}
      {$IFDEF DEBUG}
			MPB_Gear := LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.selectitem )^.value, False, MaxNum, 'MechaPartBrowser' );
      {$ELSE DEBUG}
			MPB_Gear := LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.selectitem )^.value );
      {$ENDIF DEBUG}
    {$ENDIF SDLMODE}
  {$ENDIF PATCH_GH}
  {$IFDEF PATCH_GH}
			if SwapMenu( ZONE_Menu, MPB_Gear ) then begin
  {$ELSE PATCH_GH}
    {$IFDEF DEBUG}
			if SwapMenu( ZONE_Menu , LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.selectitem )^.value, False, MaxNum, 'MechaPartBrowser' ) ) then begin
    {$ELSE DEBUG}
			if SwapMenu( ZONE_Menu , LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.selectitem )^.value ) ) then begin
    {$ENDIF DEBUG}
  {$ENDIF PATCH_GH}
				N := -1;
				ReBrowse := True;
			end;

		end else if N = -3 then begin
  {$IFDEF PATCH_GH}
    {$IFDEF DEBUG}
			MPB_Gear := LocateGearByNumber( Mek, RPMLocateByPosition( RPM , RPM^.selectitem )^.value, DebugMode, MaxNum, 'MechaPartBrowser' );
    {$ELSE DEBUG}
			MPB_Gear := LocateGearByNumber( Mek, RPMLocateByPosition( RPM , RPM^.selectitem )^.value, DebugMode );
    {$ENDIF DEBUG}
			Mark_GG_DisposeGear( MPB_Gear );
  {$ELSE PATCH_GH}
    {$IFDEF DEBUG}
			Mark_GG_DisposeGear( LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.selectitem )^.value, False, MaxNum, 'MechaPartBrowser' ) );
    {$ELSE DEBUG}
			Mark_GG_DisposeGear( LocateGearByNumber( Mek , RPMLocateByPosition( RPM , RPM^.selectitem )^.value ) );
    {$ENDIF DEBUG}
  {$ENDIF PATCH_GH}
			N := -1;
			ReBrowse := True;

		end else if N = -4 then begin
			MechaPartBrowser_SearchForward();
			N := -1;
			ReBrowse := True;
		end else if N = -5 then begin
			MechaPartBrowser_SearchBackward();
			N := -1;
			ReBrowse := True;
{$ENDIF PATCH_CHEAT}
		end;
	until N = -1;
{$IFDEF PATCH_GH}
	top := RPM^.TopItem;
	sel := RPM^.SelectItem;
{$ENDIF PATCH_GH}
	DisposeRPGMenu( RPM );
{$IFDEF PATCH_CHEAT}
    until False = ReBrowse;
{$ENDIF PATCH_CHEAT}
end;

{$IFDEF SDLMODE}
	Procedure FHQRedraw;
	begin
		if InfoGB <> Nil then QuickCombatDisplay( InfoGB );
{$IFDEF PATCH_GH}
		if (NIL <> InfoGear) and (GG_DisposeGear < InfoGear^.G) then begin
			DisplayGearInfo( InfoGear );
		end;
{$ELSE PATCH_GH}
		DisplayGearInfo( InfoGear );
{$ENDIF PATCH_GH}
	end;
{$ENDIF}

Procedure FHQ_Transfer( var LList: GearPtr; PC,Item: GearPtr );
	{ An item has been selected. Allow it to be transferred to }
	{ one of the team's master gears. }
var
	RPM: RPGMenuPtr;
	M: GearPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
	Team: Integer;
{$ELSE PATCH_GH}
	N,Team: Integer;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the GG_DisposeGear of LList at here. }
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = Item) or (Item^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Show the item's stats. }
	DisplayGearInfo( Item );

	{ Create the menu. }
	RPM := CreateRPGMenu( MenuItem, MenuSelect, ZONE_Menu );
	M := LList;
	N := 1;
	Team := NAttValue( PC^.NA , NAG_LOcation , NAS_Team );
	while M <> Nil do begin
{$IFDEF PATCH_GH}
		if (GG_DisposeGear < M^.G) then begin
			if ( ( NAttValue( M^.NA , NAG_LOcation , NAS_Team ) = Team ) or ( NAttValue( M^.NA , NAG_LOcation , NAS_Team ) = NAV_LancemateTeam ) ) and IsMasterGear( M ) and IsLegalSlot( M , Item ) then begin
				AddRPGMenuItem( RPM , GearName( M ) , N );
			end;
			Inc( N );
		end;
		M := M^.Next;
{$ELSE PATCH_GH}
		if ( ( NAttValue( M^.NA , NAG_LOcation , NAS_Team ) = Team ) or ( NAttValue( M^.NA , NAG_LOcation , NAS_Team ) = NAV_LancemateTeam ) ) and IsMasterGear( M ) and IsLegalSlot( M , Item ) then begin
			AddRPGMenuItem( RPM , GearName( M ) , N );
		end;

		M := M^.Next;
		Inc( N );
{$ENDIF PATCH_GH}
	end;

	{ Sort the menu, then add an exit option. }
	RPMSortAlpha( RPM );
	AddRPGMenuItem( RPM , MsgString( 'FHQ_ReturnToMain' ) , -1 );

	{ Get a menu selection, then exit the menu. }
	DialogMSG( MsgString( 'FHQ_SelectDestination' ) );
{$IFDEF SDLMODE}
	InfoGear := Item;
	N := SelectMenu( RPM , @FHQRedraw );
{$ELSE}
	N := SelectMenu( RPM );
{$ENDIF}
	DisposeRPGMenu( RPM );

	if N > -1 then begin
		M := RetrieveGearSib( LList , N );
		DelinkGear( LList , Item );
		InsertInvCom( M , Item );
		DialogMSG( MsgString( 'FHQ_ItemMoved' ) );
	end else begin
		DialogMSG( MsgString( 'Cancelled' ) );
	end;
end;

Procedure Rename_Mecha( GB: GameBoardPtr; NPC: GearPtr );
	{ Enter a new name for NPC. }
var
	name: String;
{$IFDEF PATCH_CHEAT}
	it: SAttPtr;
{$ENDIF PATCH_CHEAT}
begin
{$IFDEF PATCH_GH}
	if (NIL = NPC) or (NPC^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

{$IFDEF SDLMODE}
  {$IFDEF PATCH_GH}
	name := GetStringFromUser( ReplaceHash( MsgString( 'FHQ_Rename_Prompt' ) , GearName( NPC ) ) , @FHQRedraw , GearName( NPC ) );
  {$ELSE PATCH_GH}
	name := GetStringFromUser( ReplaceHash( MsgString( 'FHQ_Rename_Prompt' ) , GearName( NPC ) ) , @FHQRedraw );
  {$ENDIF PATCH_GH}
{$ELSE SDLMODE}
  {$IFDEF PATCH_GH}
	name := GetStringFromUser( ReplaceHash( MsgString( 'FHQ_Rename_Prompt' ) , GearName( NPC ) ) , GearName( NPC ) );
	if (NIL <> GB) then begin
		GFCombatDisplay( GB );
	end;
  {$ELSE PATCH_GH}
	name := GetStringFromUser( ReplaceHash( MsgString( 'FHQ_Rename_Prompt' ) , GearName( NPC ) ) );
	GFCombatDisplay( GB );
  {$ENDIF PATCH_GH}
{$ENDIF SDLMODE}
{$IFDEF PATCH_CHEAT}
	if Cheat_RenameBackup then begin
		if name ='.' then begin
			name := SAttValue( NPC^.SA , 'NameDef' );
			if name = '.' then begin
				it := FindSAtt(NPC^.SA ,'Name');
				RemoveSAtt( NPC^.SA , it );
				it := FindSAtt(NPC^.SA ,'NameDef');
				RemoveSAtt( NPC^.SA , it );
			end else if name <> '' then begin
				SetSAtt( NPC^.SA , 'Name <' + SAttValue( NPC^.SA , 'NameDef' ) + '>' );
				it := FindSAtt(NPC^.SA ,'NameDef');
				RemoveSAtt( NPC^.SA , it );
			end;
		end else if (name <> '') then begin
			if (SAttValue( NPC^.SA , 'NameDef' ) <> '') then begin
				SetSAtt( NPC^.SA , 'Name <' + name + '>' );
			end else begin
				if (SAttValue( NPC^.SA , 'Name' ) <> '') then begin
					SetSAtt( NPC^.SA , 'NameDef <' + SAttValue(NPC^.SA , 'Name') + '>' );
				end else begin
					SetSAtt( NPC^.SA , 'NameDef <.>' );
				end;
				SetSAtt( NPC^.SA , 'Name <' + name + '>' );
			end;
		end;
	end else begin
		if name <> '' then SetSAtt( NPC^.SA , 'name <' + name + '>' );
	end;
{$ELSE PATCH_CHEAT}
	if name <> '' then SetSAtt( NPC^.SA , 'name <' + name + '>' );
{$ENDIF PATCH_CHEAT}
end;

Procedure FHQ_ThisWargearWasSelected( GB: GameBoardPtr; var LList: GearPtr; PC,M: GearPtr );
	{ A mecha has been selected by the PC from the FHQ main menu. }
	{ Offer up all the different choices of things the PC can }
	{ do with mecha - select pilot, repair, check inventory, etc. }
var
	RPM: RPGMenuPtr;
	N: Integer;
begin
{$IFDEF PATCH_GH}
	{ Don't kick out the GG_DisposeGear of LList at here. }
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
	if (NIL = M) or (M^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	repeat
		{ Show the mecha's stats. }
		DisplayGearInfo( M );

		{ Create the FHQ menu. }
		RPM := CreateRPGMenu( MenuItem, MenuSelect, ZONE_Menu );
		RPM^.Mode := RPMNoCleanup;

		if IsMasterGear( M ) then begin
			if IsSafeArea( GB ) or OnTheMap( M ) then AddRPGMenuItem( RPM , MsgString( 'FHQ_GoBackpack' ) , 1 );
		end else if IsSafeArea( GB ) then begin
			AddRPGMenuItem( RPM , MsgString( 'FHQ_Transfer' ) , -3 );
		end;

		if IsSafeArea( GB ) then AddRepairOptions( RPM , PC , M );

		if M^.G = GG_Mecha then begin
			AddRPGMenuItem( RPM , MsgString( 'FHQ_SelectMecha' ) , 2 );
			AddRPGMenuItem( RPM , MsgString( 'FHQ_Rename' ) , 6 );
		end;
		if IsSafeArea( GB ) then AddRPGMenuItem( RPM , MsgString( 'FHQ_PartEditor' ) , 4 );

{$IFDEF SDLMODE}
		if M^.G = GG_Mecha then AddRPGMenuItem( RPM , MsgString( 'FHQ_EditColor' ) , 5 );
{$ENDIF}

		AddRPGMenuItem( RPM , MsgString( 'FHQ_ReturnToMain' ) , -1 );
{$IFDEF PATCH_CHEAT}
		if Cheat_FieldHQ_AddMenuKey then begin
			AlphaKeyMenu( RPM );
		end;
{$ENDIF PATCH_CHEAT}

		{ Get a selection from the menu, then dispose of it. }
{$IFDEF SDLMODE}
		InfoGB := GB;
		infoGear := M;
		N := SelectMenu( RPM , @FHQRedraw );
{$ELSE}
		N := SelectMenu( RPM );
{$ENDIF}
		DisposeRPGMenu( RPM );

		if N > 100 then begin
			{ A repair option must have been selected. }
			DoFieldRepair( GB , PC , M , RepairSkillIndex[N-100] );

		end else begin
			case N of
				1: RealBackpack( GB , LList , PC , M , False );
				2: FHQ_SelectPilotForMecha( GB , M );
				-3: FHQ_Transfer( LList , PC , M );
				4: MechaPartEditor( GB , LList , PC , M );
{$IFDEF SDLMODE}
				5: SelectColors( M , @FHQRedraw );
{$ENDIF}
				6: Rename_Mecha( GB , M );
			end;

		end;

	until N < 0;
{$IFDEF SDLMODE}
	CleanSpriteList;
{$ENDIF}
	GFCombatDisplay( GB );
end;

{$IFDEF PATCH_CHEAT}
Procedure SelectPortrait( M: GearPtr );
	{ The player wants to change the colors for sprite for this character. }
	{ The menu will be placed in the Menu area; assume the redrawer will }
	{ show whatever changes are made here. }
var
	RPM: RPGMenuPtr;
	fname: String;
begin
{$IFDEF PATCH_GH}
	if (NIL = M) or (M^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
	if NAttValue( M^.NA , NAG_CharDescription , NAS_Gender ) = NAV_Female then begin
		BuildFileMenu( RPM , Graphics_Directory + 'por_f_*.*' );
	end else begin
		BuildFileMenu( RPM , Graphics_Directory + 'por_m_*.*' );
	end;
	AddRPGMenuItem( RPM , MsgString( 'EXIT' ) , -1 );

{$IFDEF PATCH_GH}
	fname := SAttValue( M^.SA , 'SDL_PORTRAIT' );
	while (RPM^.SelectItem < RPM^.NumItem) do begin
		if RPMLocateByPosition(RPM,RPM^.SelectItem)^.msg = fname then begin
			break;
		end;
		Inc( RPM^.SelectItem );
	end;
	RPM^.TopItem := -1;
{$ENDIF PATCH_GH}

{$IFDEF SDLMODE}
	fname := SelectFile( RPM , Nil );
{$ELSE SDLMODE}
	fname := SelectFile( RPM );
{$ENDIF SDLMODE}

	if fname <> '' then begin
		SetSAtt( M^.SA , 'SDL_PORTRAIT <' + fname + '>' );
	end;

	DisposeRPGMenu( RPM );
end;
{$ENDIF PATCH_CHEAT}

{$IFDEF PATCH_CHEAT}
Procedure BuildSiblingMenu( RPM: RPGMenuPtr; First , Item: GearPtr );
var
	N: Integer;
	Part: GearPtr;
begin
	{ Don't kick out the GG_DisposeGear of Item at here. }
	N := 0;
	Part := First;
	while Part <> Nil do begin
{$IFDEF PATCH_GH}
		if (GG_DisposeGear < Part^.G) then begin
			Inc(N);
			if (Part <> Item) then begin
				AddRPGMenuItem( RPM , '   ' + GearName( Part ) , N )
			end else begin
				AddRPGMenuItem( RPM , '< ' + GearName( Part ) , N );
			end;
		end;
		Part := Part^.Next;
{$ELSE PATCH_GH}
		Inc(N);
		if (Part <> Item) then AddRPGMenuItem( RPM , '   ' + GearName( Part ) , N )
		else AddRPGMenuItem( RPM , '< ' + GearName( Part ) , N );
		Part := Part^.Next;
{$ENDIF PATCH_GH}
	end;
end;

Procedure SwapParts( var F: GearPtr; A,B: GearPtr );
var
	t,pa,pb: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = F) then Exit;
	if (NIL = A) or (A^.G <= GG_DisposeGear) then Exit;
	if (NIL = B) or (B^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	t := F;
	while (t <> NIL) and (t <> A) and (t <> B) do begin
		t := t^.Next;
	end;
	if t = B then begin
		B := A;
		A := t;
	end;
	
	if F = A then begin
		pa := NIL;
	end else begin
		pa := F;
		while (pa^.Next <> A) do begin
			pa := pa^.next;
		end;
	end;

	pb := A;
	while (pb^.Next <> B) do begin
		pb := pb^.next;
	end;

	{ F ... pa A ... pb B ... => F ... pa B ... pb A ... }
	{ F ... pa A B ...        => F ... pa B A ...        }
	{ A ... pb B ...          => B ... pb A ...          }
	{ A B ...                 => B A ...                 }
	if A^.Next <> B then begin
		t := A^.Next;
	end else begin
		t := A;
	end;
	A^.Next := B^.Next;
	B^.Next := t;

	if pb <> A then begin
		pb^.Next := A;
	end;
	if (NIL = pa) then begin
		F := B;
	end else begin
		pa^.Next := B;
	end;
end;

{$IFDEF SDLMODE}
Function SwapMenu_NoParent( var FirstPart: GearPtr; Z: TSDL_Rect; Part: GearPtr ):Boolean;
{$ELSE SDLMODE}
Function SwapMenu_NoParent( var FirstPart: GearPtr; Z: Integer; Part: GearPtr ):Boolean;
{$ENDIF SDLMODE}
var
	done: Boolean;
	RPM: RPGMenuPtr;
	A,B: GearPtr;
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
	top: Integer = 1;
	sel: Integer = 1;
begin
	done := False;
	{ Don't kick out the GG_DisposeGear at here. }
	if (NIL <> FirstPart) and (NIL <> Part) then begin
		if 1 < NumSiblingGears(FirstPart) then begin
			RPM := CreateRPGMenu( MenuItem , MenuSelect , Z );
			A := Part;

			repeat
				BuildSiblingMenu( RPM, FirstPart, A );
				if done then AddRPGMenuItem( RPM , I18N_MsgString('BACKPACK_SwapMenu','Exit') , -1 )
				else AddRPGMenuItem( RPM , I18N_MsgString('BACKPACK_SwapMenu','Cancel') , -1 );
				RPM^.SelectItem := sel;
				RPM^.TopItem := top;
{$IFDEF SDLMODE}
				N := SelectMenu( RPM , Nil );
{$ELSE SDLMODE}
				N := SelectMenu( RPM );
{$ENDIF SDLMODE}

				if 0 < N then begin
					B := RetrieveGearSib( FirstPart, N );
					if A = Nil then begin
						A := B;
					end else if A = B then begin
						A := Nil;
					end else begin
						SwapParts( FirstPart, A, B );
						done := True;
						A := Nil;
					end;

					sel := RPM^.SelectItem;
					top := RPM^.TopItem;
					ClearMenu( RPM );
				end;
			until N = -1;

			DisposeRPGMenu( RPM );
		end;
	end;
	SwapMenu_NoParent := done;
end;

{$IFDEF SDLMODE}
Function SwapMenu( Z: TSDL_Rect; Part: GearPtr ):Boolean;
{$ELSE SDLMODE}
Function SwapMenu( Z: Integer; Part: GearPtr ):Boolean;
{$ENDIF SDLMODE}
var
	done: Boolean;
begin
	done := False;
	{ Don't kick out the GG_DisposeGear at here. }
	if (NIL <> Part) and (NIL <> Part^.Parent) then begin
		if IsInvCom(Part) then begin
			done := SwapMenu_NoParent( Part^.Parent^.InvCom, Z, Part );
		end else begin
			done := SwapMenu_NoParent( Part^.Parent^.SubCom, Z, Part );
		end;
	end;
	SwapMenu := done;
end;
{$ENDIF PATCH_CHEAT}



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: backpack.pp');
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
  {$IFDEF SDLMODE}
	InfoGear := NIL;
	InfoGB   := NIL;
	MPB_Redraw := NIL;
	MPB_Gear := NIL;
	MPR_InvMenu     := NIL;
	MPR_InvMenuLGBN := NIL;
	MPR_InvMenuRGS  := NIL;
	Attach_SmartPointer( 'InfoGear: GearPtr', @InfoGear );
	Attach_SmartPointer( 'InfoGB: GameBoardPtr', @InfoGB );
	Attach_SmartPointer( 'MPB_Gear: GearPtr', @MPB_Gear );
  {$ENDIF SDLMODE}
{$ENDIF PATCH_GH}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: backpack.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
