unit ConInfo;
{
	GearHead: Arena, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}

interface

uses
{$IFDEF PATCH_GH}
	gears_base,
{$ENDIF PATCH_GH}
	gears,locale;

Procedure GearInfo( Part: GearPtr; X1,Y1,X2,Y2,BorColor: Byte );
Procedure LocationInfo( Part: GearPtr; gb: GameBoardPtr );
{$IFDEF PATCH_GH}
Procedure DisplayGearInfo( Part: GearPtr; DebugMode: Boolean );
{$ENDIF PATCH_GH}
Procedure DisplayGearInfo( Part: GearPtr );
Procedure DisplayGearInfo( Part: GearPtr; gb: GameBoardPtr; Z: Integer );
Procedure DisplayGearInfo( Part: GearPtr; gb: GameBoardPtr );

Function JobAgeGenderDesc( NPC: GearPtr ): String;
Procedure DisplayInteractStatus( GB: GameBoardPtr; NPC: GearPtr; React,Endurance: Integer );
Procedure QuickWeaponInfo( Part: GearPtr );
Procedure CharacterDisplay( PC: GearPtr; GB: GameBoardPtr );
Procedure InjuryViewer( PC: GearPtr );

Procedure MapEditInfo( Pen,Palette,X,Y: Integer );

implementation

uses
{$IFDEF GUIMSWINMODE}
{$ELSE GUIMSWINMODE}
	crt,
{$ENDIF GUIMSWINMODE}
{$IFDEF DEBUG}
	errmsg,
{$ENDIF DEBUG}
{$IFDEF PATCH_I18N}
	i18nmsg,
{$ENDIF PATCH_I18N}
{$IFDEF PATCH_CHEAT}
	ui4gh,
{$ENDIF PATCH_CHEAT}
{$IFDEF GUIMSWINMODE}
	w32crt,
	conoutput,
{$ENDIF GUIMSWINMODE}
	ability,damage,gearutil,ghchars,ghmecha,ghmodule,ghweapon,
	interact,movement,texutil,congfx,conmap,context;

var
	CX,CY: Byte;			{ Cursor Position }
	ZX1,ZY1,ZX2,ZY2: Byte;		{ Info Zone coords }
{$IFDEF PATCH_GH}
{$ELSE PATCH_GH}
	LastGearShown: GearPtr;
{$ENDIF PATCH_GH}

const
	SX_Char: Array [1..Num_Status_FX] of Char = (
		'P','B','R','S','H',
		'V','T','D','R','P',
		'A','G','L','N','Z',
		'X','S','R','S','I',
		'@','@','@','@','@'
	);
	SX_Color: Array [1..Num_Status_FX] of Byte = (
		Magenta, LightRed, LightGreen, Magenta, Yellow,
		Cyan, Cyan, Cyan, Cyan, Cyan,
		Cyan, Cyan, Cyan, Cyan, Cyan,
		Cyan, Cyan, Red, Yellow, Magenta,
		Red, Red, Red, Red, Red
	);


Function MaxTArmor( Part: GearPtr ): LongInt;
	{ Find the max amount of armor on this gear, counting external armor. }
var
	it: LongInt;
	S: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit(0);
{$ENDIF PATCH_GH}

	it := GearMaxArmor( Part );
	S := Part^.InvCom;
	while S <> Nil do begin
		if S^.G = GG_ExArmor then it := it + GearMaxArmor( S );
		S := S^.Next;
	end;
	MaxTArmor := it;
end;

Function CurrentTArmor( Part: GearPtr ): LongInt;
	{ Find the current amount of armor on this gear, counting external armor. }
var
	it: LongInt;
	S: GearPtr;
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit(0);
{$ENDIF PATCH_GH}

	it := GearCurrentArmor( Part );
	S := Part^.InvCom;
	while S <> Nil do begin
		if S^.G = GG_ExArmor then it := it + GearCurrentArmor( S );
		S := S^.Next;
	end;
	CurrentTArmor := it;
end;

Procedure AI_Title( msg: String; C: Byte );
	{ Draw a centered message on the current line. }
var
	X: Integer;
begin
{$IFDEF PATCH_I18N}
	X := ( ( ZX2 - ZX1 ) div 2 ) - ( WidthMBcharStr( msg ) div 2 ) + 1;
{$ELSE PATCH_I18N}
	X := ( ( ZX2 - ZX1 ) div 2 ) - ( Length( msg ) div 2 ) + 1;
{$ENDIF PATCH_I18N}
	if X < 1 then X := 1;
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( X , CY );
	w32crt.TextColor( C );
{$ELSE GUIMSWINMODE}
	GotoXY( X , CY );
	TextColor( C );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	WriteMBCharStr( msg, 0 );
{$ELSE PATCH_I18N}
	Write( msg );
{$ENDIF PATCH_I18N}
	CX := 1;
	CY := CY + 1;
end;

Procedure AI_Line( msg: String; C: Byte );
	{ Draw a left justified message on the current line. }
begin
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( ZX1 , CY );
	w32crt.TextColor( C );
{$ELSE GUIMSWINMODE}
	GotoXY( ZX1 , CY );
	TextColor( C );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	WriteMBCharStr( msg, (ZX2-ZX1) );
{$ELSE PATCH_I18N}
	Write( msg );
{$ENDIF PATCH_I18N}
	CX := 1;
	Inc( CY );
end;

Procedure AI_PrintFromRight( msg: String; Tab,C: Byte );
	{ Draw a left justified message on the current line. }
begin
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( Tab , CY );
	w32crt.TextColor( C );
{$ELSE GUIMSWINMODE}
	GotoXY( Tab , CY );
	TextColor( C );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	WriteMBCharStr( msg, 0 );
{$ELSE PATCH_I18N}
	Write( msg );
{$ENDIF PATCH_I18N}
{$IFDEF GUIMSWINMODE}
	CX := w32crt.WhereX;
{$ELSE GUIMSWINMODE}
	CX := WhereX;
{$ENDIF GUIMSWINMODE}
end;

Procedure AI_PrintFromLeft( msg: String; Tab,C: Byte );
	{ Draw a left justified message on the current line. }
var
	TP: Integer;
begin
{$IFDEF PATCH_I18N}
	TP := Tab - WidthMBcharStr( msg );
{$ELSE PATCH_I18N}
	TP := Tab - Length( msg );
{$ENDIF PATCH_I18N}
	if TP < 1 then TP := 1;
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( TP , CY );
	w32crt.TextColor( C );
{$ELSE GUIMSWINMODE}
	GotoXY( TP , CY );
	TextColor( C );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	WriteMBCharStr( msg, 0 );
{$ELSE PATCH_I18N}
	Write( msg );
{$ENDIF PATCH_I18N}
{$IFDEF GUIMSWINMODE}
	CX := w32crt.WhereX;
{$ELSE GUIMSWINMODE}
	CX := WhereX;
{$ENDIF GUIMSWINMODE}
end;

Procedure AI_PrintChar( msg: Char; C: Byte );
	{ Print a character on the current line, unless doing so would }
	{ cause the line to spread onto the next line. }
begin
{$IFDEF GUIMSWINMODE}
	if w32crt.WhereX < ( ZX2 - ZX1 - 1 ) then begin
		w32crt.TextColor( C );
{$ELSE GUIMSWINMODE}
	if WhereX < ( ZX2 - ZX1 - 1 ) then begin
		TextColor( C );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
		WriteMBCharStr( msg, 0 );
{$ELSE PATCH_I18N}
		Write( msg );
{$ENDIF PATCH_I18N}
{$IFDEF GUIMSWINMODE}
		CX := w32crt.WhereX;
{$ELSE GUIMSWINMODE}
		CX := WhereX;
{$ENDIF GUIMSWINMODE}
	end;
end;

Procedure AI_NextLine;
	{ Move the cursor to the next line. }
begin
	Inc( CY );
end;

Function StatusColor( Full , Current: LongInt ): Byte;
	{ Given a part's Full and Current hit ratings, decide on a good status color. }
begin
	if Full = Current then StatusColor := LightGreen
	else if Current > ( Full div 2 ) then StatusColor := Green
	else if Current > ( Full div 4 ) then StatusColor := Yellow
	else if Current > ( Full div 8 ) then StatusColor := LightRed
	else if Current > 0 then StatusColor := Red
	else StatusColor := DarkGray;
end;

Function EnduranceColor( Full , Current: LongInt ): Byte;
	{ Choose colour to show remaining endurance (Stamina or Mental points)}
begin
        { This is absolute rather than relative. }
	if Full = Current then EnduranceColor := LightGreen
	else if Current > 5 then EnduranceColor := Green
	else if Current > 0 then EnduranceColor := Yellow
	else EnduranceColor := LightRed;
end;

Function HitsColor( Part: GearPtr ): LongInt;
	{ Decide upon a nice color to represent the hits of this part. }
begin
	if PartActive( Part ) then
		HitsColor := StatusColor( GearMaxDamage( Part ) , GearCurrentDamage( Part ) )
	else
		HitsColor := StatusColor( 100 , 0 );
end;

Function ArmorColor( Part: GearPtr ): LongInt;
	{ Decide upon a nice color to represent the armor of this part. }
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit(StatusColor(100,0));
{$ENDIF PATCH_GH}
	ArmorColor := StatusColor( MaxTArmor( Part ) , CurrentTArmor( Part ) );
end;

Function ArmorDamageColor( Part: GearPtr ): LongInt;
	{ Decide upon a nice color to represent the armor of this part. }
var
	MA,CA: LongInt;	{ Max Armor, Current Armor }
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit(StatusColor(100,0));
{$ENDIF PATCH_GH}

	MA := MaxTArmor( Part );
	CA := CurrentTArmor( Part );

	if MA = 0 then begin
		ArmorDamageColor := Magenta;
	end else if ( CA >= ( MA * 3 div 4 ) ) then begin
		ArmorDamageColor := Black;
	end else if ( CA > MA div 4 ) then begin
		ArmorDamageColor := Blue;
	end else begin
		ArmorDamageColor := LightGray;
	end;
end;

Procedure ShowStatus( Part: GearPtr );
	{ Display all this part's status conditions. }
var
	T: LongInt;
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Show the character's status conditions. }
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( 2 , CY );
{$ELSE GUIMSWINMODE}
	GotoXY( 2 , CY );
{$ENDIF GUIMSWINMODE}

	{ Hunger and morale come first. }
	if Part^.G = GG_Character then begin
		T := NAttValue( Part^.NA , NAG_Condition , NAS_Hunger ) - Hunger_Penalty_Starts;
		if T > ( NumGearStats * 3 ) then begin
			AI_PrintChar( 'H' , LightRed );
		end else if T > ( NumGearStats * 2 ) then begin
			AI_PrintChar( 'H' , Yellow );
		end else if T > 0 then begin
			AI_PrintChar( 'H' , Green );
		end;

		T := NAttValue( Part^.NA , NAG_Condition , NAS_MoraleDamage );
		if T < -20 then begin
			AI_PrintChar( '+' , LightGreen );
		end else if T > ( 65 ) then begin
			AI_PrintChar( '-' , LightRed );
		end else if T > ( 40 ) then begin
			AI_PrintChar( '-' , Yellow );
		end else if T > 20 then begin
			AI_PrintChar( '-' , Green );
		end;

	end else if Part^.G = GG_Mecha then begin
		{ Mecha may be overloaded. }
		T := NAttValue( Part^.NA , NAG_Condition , NAS_Overload ) - OverloadCapacity( Part );
		if T > 10 then begin
			AI_PrintChar( 'O' , LightRed );
		end else if T > 0 then begin
			AI_PrintChar( 'O' , DarkGray );
		end;
	end;

	for t := 1 to Num_Status_FX do begin
		if NAttValue( Part^.NA , NAG_StatusEffect , T ) <> 0 then begin
			AI_PrintChar( SX_Char[ T ] , SX_Color[ T ] );
		end;
	end;

	if NAttValue( Part^.NA , NAG_EpisodeData , NAS_Ransacked ) = 1
	then AI_PrintChar( '$' , DarkGray );
end;

Procedure DisplayModules( Mek: GearPtr );
	{ Draw a lovely little diagram detailing this mek's modules. }
var
	MM,N,X0: Integer;
	MD: GearPtr;
	Flayed, Gutted : Boolean;
	Procedure AddPartsToDiagram( GS: Integer );
		{ Add parts to the status diagram whose gear S value }
		{ is equal to the provided number. }
	var
		X: Integer;
		FG, BG: Byte;
	begin
		MD := Mek^.SubCom;
		while ( MD <> Nil ) do begin
			if ( MD^.G = GG_Module ) and ( MD^.S = GS ) then begin

				FG := HitsColor( MD );
				BG := ArmorDamageColor( MD );
				
				if (FG = DarkGray) And (BG <> Black)
				    then FG := Black;

				if Flayed Or (Gutted And (GS = GS_Body)) 
				then begin
				    if Gutted
				    then FG := White
				    else FG := LightMagenta;
				    BG := Red;
				end;

{$IFDEF GUIMSWINMODE}
				w32crt.TextColor(FG);
				w32crt.TextBackground(BG);
{$ELSE GUIMSWINMODE}
				TextColor(FG);
				TextBackground(BG);
{$ENDIF GUIMSWINMODE}

				if Odd( N ) then X := X0 - ( N div 2 ) - 1
				else X := X0 + ( N div 2 );
				Inc( N );
{$IFDEF GUIMSWINMODE}
				w32crt.GotoXY( X , CY );
{$ELSE GUIMSWINMODE}
				GotoXY( X , CY );
{$ENDIF GUIMSWINMODE}
				Case GS of
{$IFDEF GUIMSWINMODE}
					GS_Head:	conoutput.ConWrite('o');
					GS_Turret:	conoutput.ConWrite('=');
					GS_Storage:	conoutput.ConWrite('x');
{$IFDEF PATCH_CHEAT}
					GS_Conversion:	conoutput.ConWrite('O');
{$ENDIF PATCH_CHEAT}
					GS_Body:	begin
							conoutput.ConWrite('B');
							end;
					GS_Arm:		conoutput.ConWrite('+');
					GS_Wing:	conoutput.ConWrite('W');
					GS_Tail:	conoutput.ConWrite('t');
					GS_Leg:		conoutput.ConWrite('l');
{$ELSE GUIMSWINMODE}
					GS_Head:	write('o');
					GS_Turret:	write('=');
					GS_Storage:	write('x');
{$IFDEF PATCH_CHEAT}
					GS_Conversion:	write('O');
{$ENDIF PATCH_CHEAT}
					GS_Body:	begin
							write('B');
							end;
					GS_Arm:		write('+');
					GS_Wing:	write('W');
					GS_Tail:	write('t');
					GS_Leg:		write('l');
{$ENDIF GUIMSWINMODE}
				end;
			end;
			MD := MD^.Next;
		end;
	end;

begin
{$IFDEF PATCH_GH}
	if (NIL = Mek) or (Mek^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ this "if" is just a shortcut }
	if GearOperational(Mek)
	then begin
	    Gutted := False;
	    Flayed := False;
	end else begin
	    Gutted := (NAttValue( Mek^.NA , NAG_EpisodeData , NAS_Gutted) = 1);
	    Flayed := (NAttValue( Mek^.NA , NAG_EpisodeData , NAS_Flayed) = 1);
	end;

	{ Draw the status diagram for this mek. }
	{ Line One - Heads, Turrets, Storage }
	X0 := ( ZX2 - ZX1 ) div 2 + 2;
	MM := CY;	{ Save the CY value, since we want to print info }
			{ on these same three lines. }
	N := 0;
	AddPartsToDiagram( GS_Head );
	AddPartsToDiagram( GS_Turret );
{$IFDEF PATCH_CHEAT}
	AddPartsToDiagram( GS_Conversion );
{$ENDIF PATCH_CHEAT}
	if N < 1 then N := 1;	{ Want storage to either side of body. }
	AddPartsToDiagram( GS_Storage );
	AI_NextLine;

	{ Line Two - Torso, Arms, Wings }
	N := 0;
	AddPartsToDiagram( GS_Body );
	AddPartsToDiagram( GS_Arm );
	AddPartsToDiagram( GS_Wing );
	AI_NextLine;

	{ Line Three - Tail, Legs }
	N := 0;
	AddPartsToDiagram( GS_Tail );
	if N < 1 then N := 1;	{ Want legs to either side of body; tail in middle. }
	AddPartsToDiagram( GS_Leg );
	AI_NextLine;

	{ Restore background color to black. }
{$IFDEF GUIMSWINMODE}
	w32crt.TextBackground( Black );
{$ELSE GUIMSWINMODE}
	TextBackground( Black );
{$ENDIF GUIMSWINMODE}

	{ Restore CY. }
	CY := MM;
end;

Procedure MekStatDisplay( Mek: GearPtr );
	{ Display the stats for MEK. }
	{ MEK absolutely must be a valid mecha; otherwise }
	{ there's gonna be a strange display. }
var
	msg: String;
	MM,N,A,B: Integer;
	MD: GearPtr;
{$IFDEF PATCH_CHEAT}
	MoveOrder: Integer;
{$ENDIF PATCH_CHEAT}
begin
{$IFDEF PATCH_GH}
	if (NIL = Mek) or (Mek^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ General mecha information - Name, mass, maneuver }
{$IFDEF PATCH_CHEAT}
	if Cheat_Display_SW and Cheat_Display_DESIG then begin
		AI_PrintFromLeft(  GearName(Mek) , 0, White );
		AI_PrintFromRight( SAttValue( Mek^.SA , 'DESIG' ) , ZX2 - ZX1 - WidthMBcharStr(SAttValue(Mek^.SA,'DESIG')) , MenuItem );
		AI_NextLine;
	end else begin
		AI_Title( GearName(Mek) , White );
	end;
{$ELSE PATCH_CHEAT}
	AI_Title( GearName(Mek) , White );
{$ENDIF PATCH_CHEAT}

	DisplayModules( Mek );

	AI_PrintFromRight( 'MV:' + SgnStr(MechaManeuver(Mek)) , ZX2 - ZX1 - 5 , LightGray );
	AI_NextLine;
	AI_PrintFromRight( 'TR:' + SgnStr(MechaTargeting(Mek)) , ZX2 - ZX1 - 5 , LightGray );
	AI_NextLine;
	AI_PrintFromRight( 'SE:' + SgnStr(MechaSensorRating(Mek)) , ZX2 - ZX1 - 5 , LightGray );
	AI_NextLine;
{$IFDEF PATCH_CHEAT}
	if Cheat_Display_SW and Cheat_Display_OverLoad then begin
		MM := OverloadCapacity( Mek ) - NAttValue( Mek^.NA , NAG_Condition , NAS_Overload );
		if MM < 0 then AI_PrintFromRight( 'OL:' + SgnStr(MM) , ZX2 - ZX1 - 5 , EnemyRed )
		else AI_PrintFromRight( 'OC:' + BStr(MM) , ZX2 - ZX1 - 5 , MenuItem );
		AI_NextLine;
	end;
{$ENDIF PATCH_CHEAT}

	{ Pilot Information - Name, health, rank }
	MD := LocatePilot( Mek );
{$IFDEF PATCH_GH}
	if (NIL <> MD) and (GG_DisposeGear < MD^.G) then begin
{$ELSE PATCH_GH}
	if MD <> Nil then begin
{$ENDIF PATCH_GH}

		{ Pilot's name - Left Justified. }
		msg := GearName( MD );

		{ Color determined by exhaustion. }
		A := CharCurrentMental( MD );
		B := CharCurrentStamina( MD );
		if ( A=0 ) and ( B=0 ) then begin
			N := LightRed;
		end else if ( A=0 ) or ( B=0 ) then begin
			N := Yellow;
		end else begin
			N := LightGray;
		end;

		AI_PrintFromRight( msg , 2 , N );

		AI_PrintFromLeft( BStr( GearCurrentDamage( MD ) ) + 'HP' , ZX2 - ZX1 - 1 , HitsColor( MD ) );
		AI_NextLine;
	end;

	AI_Title( MassString( Mek ) + ' ' + FormName[Mek^.S] + '  PV:' + BStr( GearValue( Mek ) ) , DarkGray );

	{ Movement information. }
	MM := NAttValue( Mek^.NA , NAG_Action , NAS_MoveMode );
	if MM > 0 then begin
{$IFDEF PATCH_I18N}
		msg := I18N_Name('MoveModeName',MoveModeName[ MM ]);
{$ELSE PATCH_I18N}
		msg := MoveModeName[ MM ];
{$ENDIF PATCH_I18N}
{$IFDEF PATCH_CHEAT}
		if Cheat_Display_SW and Cheat_Display_DamagePercent then begin
			msg := msg + ReplaceHash( I18N_MsgString('MekStatDisplay','Broken'), BStr(100 - PercentDamaged(Mek)) );
		end else if Cheat_Display_SpeedoMeter then begin
			if Cheat_Display_SW and ( BaseMoveRate( Mek ) = 0 ) then begin
				msg := msg + ReplaceHash( I18N_MsgString('MekStatDisplay','Broken'), BStr(100 - PercentDamaged(Mek)) );
			end else begin
				MoveOrder := NAttValue( Mek^.NA , NAG_Action , NAS_MoveAction );
				if ( NAV_TurnLeft = MoveOrder ) or ( NAV_TurnRight = MoveOrder ) then begin
					msg := msg + ' (' + I18N_MsgString('MekStatDisplay','Turn') + ' ' + BStr( NAttValue( Mek^.NA , NAG_Action , NAS_SpeedoMeter ) ) + 'dpr)';
				end else begin
					msg := msg + ' (' + BStr( NAttValue( Mek^.NA , NAG_Action , NAS_SpeedoMeter ) ) + 'dpr)';
				end;
			end;
		end else begin
			msg := msg + ' (' + BStr( Speedometer( Mek ) ) + 'dpr)';
		end;
{$ELSE PATCH_CHEAT}
		msg := msg + ' (' + BStr( Speedometer( Mek ) ) + 'dpr)';
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_I18N}
	end else msg := I18N_MsgString('MekStatDisplay','Immobile');
{$ELSE PATCH_I18N}
	end else msg := 'Immobile';
{$ENDIF PATCH_I18N}
	AI_Title( msg , DarkGray );

	AI_NextLine;
	ShowStatus( Mek );
end;

Procedure CharacterInfo( Part: GearPtr );
	{ This gear is a character. Print a list of stats and skills. }
var
	T,TT,Width,S: Integer;
	C: Byte;
{$IFDEF PATCH_CHEAT}
	MPV,GV: Int64;
{$ENDIF PATCH_CHEAT}
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Show the character's name and health status. }
	AI_Title( GearName(Part) , White );

	DisplayModules( Part );

	AI_PrintFromLeft( BStr( GearCurrentDamage(Part)) + '/' + BStr( GearMaxDamage(Part)) , ZX2 - ZX1 - 2 , HitsColor( Part ) );
	AI_PrintFromLeft( 'HP' , ZX2 - ZX1 + 1 , LightGray );
	AI_NextLine;
	AI_PrintFromLeft( BStr( CharCurrentStamina(Part)) + '/' + BStr( CharStamina(Part)) , ZX2 - ZX1 - 2 , EnduranceColor( CharStamina(Part) , CharCurrentStamina(Part) ) );
	AI_PrintFromLeft( 'St' , ZX2 - ZX1 + 1 , LightGray );
	AI_NextLine;
	AI_PrintFromLeft( BStr( CharCurrentMental(Part)) + '/' + BStr( CharMental(Part)) , ZX2 - ZX1 - 2 , EnduranceColor( CharMental(Part) , CharCurrentMental(Part) ) );
	AI_PrintFromLeft( 'Me' , ZX2 - ZX1 + 1 , LightGray );
	AI_NextLine;
{$IFDEF PATCH_CHEAT}
	if Cheat_Display_SW and Cheat_Display_PV then begin
		GV := Int64(GearValue(Part)) * Int64(Part^.Scale +1);
		MPV := Int64(Part^.V) * Int64(Part^.V * 150 - 100) * Int64(Part^.Scale +1);
		if MPV > GV then GV := MPV;
		AI_PrintFromLeft( BStr( GV ) , ZX2 - ZX1 - 2 , MenuItem );
		AI_PrintFromLeft( 'PV' , ZX2 - ZX1 + 1 , MenuItem );
	end;
{$ENDIF PATCH_CHEAT}
	AI_NextLine;


	{ Determine the spacing for the character's stats. }
	Width := ( ZX2 - ZX1 ) div 4;

	{ Show the character's stats. }
	for t := 1 to ( NumGearStats div 4 ) do begin
		for tt := 1 to 4 do begin
{$IFDEF PATCH_I18N}
			AI_PrintFromRight( HeadMBChar( I18N_Name('StatName', StatName[ T * 4 + TT - 4 ]) ) + ':' , ( TT-1 ) * Width + 1 , LightGray );
{$ELSE PATCH_I18N}
			AI_PrintFromRight( StatName[ T * 4 + TT - 4 ][1] + StatName[ T * 4 + TT - 4 ][2] + ':' , ( TT-1 ) * Width + 1 , LightGray );
{$ENDIF PATCH_I18N}

			{ Determine the stat value. This may be higher or lower than natural... }
			S := CStat( Part , T * 4 + TT - 4 );
			if S > Part^.Stat[ T * 4 + TT - 4 ] then C := LighTGreen
			else if S < Part^.Stat[ T * 4 + TT - 4 ] then C := LightRed
			else C := Green;
			AI_PrintFromLeft( BStr( S ) , ( TT - 1 ) * Width + 6 , C );
		end;
		AI_NextLine;
	end;

	ShowStatus( Part );
end;

{$IFDEF PATCH_GH}
Procedure MiscInfo( Part: GearPtr; DebugMode: Boolean );
{$ELSE PATCH_GH}
Procedure MiscInfo( Part: GearPtr );
{$ENDIF PATCH_GH}
	{ Display info for any gear that doesn't have its own info }
	{ procedure. }
var
	N: LongInt;
	msg: String;
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (not(DebugMode) and (Part^.G <= GG_DisposeGear)) then Exit;
{$ENDIF PATCH_GH}

	{ Show the part's name. }
{$IFDEF PATCH_GH}
	AI_Title( GearName(Part,DebugMode) , White );
{$ELSE PATCH_GH}
	AI_Title( GearName(Part) , White );
{$ENDIF PATCH_GH}

	{ Display the part's armor rating. }
	N := GearCurrentArmor( Part );
	if N > 0 then msg := '[' + BStr( N )
	else msg := '[-';
	msg := msg + '] ';
	AI_PrintFromRight( msg , 1 , ArmorColor( Part ) );

	{ Display the part's damage rating. }
	N := GearCurrentDamage( Part );
	if N > 0 then msg := BStr( N )
	else msg := '-';
	AI_PrintFromRight( msg + ' DP' , CX , HitsColor( Part ) );

{$IFDEF PATCH_GH}
	N := ( Int64(GearMass( Part )) + 1 ) div 2;
{$ELSE PATCH_GH}
	N := ( GearMass( Part ) + 1 ) div 2;
{$ENDIF PATCH_GH}
	if N > 0 then AI_PrintFromLeft( MassString( Part ) , ZX2 - ZX1 + 2 , LightGray );

{$IFDEF PATCH_GH}
	if Part^.G < 0 then begin
		AI_NextLine;
		AI_PrintFromRight( Bstr( Part^.G ) + ',' + BStr( Part^.S ) + ',' + BStr( Part^.V ), CX, LightGray );
	end;
{$ENDIF PATCH_GH}

	GameMsg( ExtendedDescription( Part ) , ZX1 , ZY1 + 3 , ZX2 , ZY2 , LightGray );
end;

Procedure SetInfoZone( X1,Y1,X2,Y2,BorColor: Byte );
	{ Copy the provided coordinates into this unit's global }
	{ variables, then draw a nice little border and clear the }
	{ selected area. }
begin
	{ Copy the dimensions provided into this unit's global variables. }
	ZX1 := X1 + 1;
	ZY1 := Y1 + 1;
	ZX2 := X2 - 1;
	ZY2 := Y2 - 1;

	DrawZoneBorder( X1 , Y1 , X2 , Y2 , BorColor );
	CX := 1;
	CY := 1;

{$IFDEF GUIMSWINMODE}
	w32crt.Window( ZX1 , ZY1 , ZX2 , ZY2 );
	w32crt.ClrScr;
{$ELSE GUIMSWINMODE}
	Window( ZX1 , ZY1 , ZX2 , ZY2 );
	ClrScr;
{$ENDIF GUIMSWINMODE}
end;

Procedure MetaTerrainInfo( Part: GearPtr );
	{ Display info for any gear that doesn't have its own info }
	{ procedure. }
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}
	AI_Title( GearName(Part) , TerrainGreen );
end;

Procedure RepairFuelInfo( Part: GearPtr );
	{ Display info for any gear that doesn't have its own info }
	{ procedure. }
var
{$IFDEF PATCH_GH}
	N: LongInt;
{$ELSE PATCH_GH}
	N: Integer;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Show the part's name. }
	AI_Title( GearName(Part) , White );

{$IFDEF PATCH_GH}
	N := ( Int64(GearMass( Part )) + 1 ) div 2;
{$ELSE PATCH_GH}
	N := ( GearMass( Part ) + 1 ) div 2;
{$ENDIF PATCH_GH}
	if N > 0 then AI_PrintFromLeft( MassString( Part ) , ZX2 - ZX1 + 2 , LightGray );

	AI_NextLine;
{$IFDEF PATCH_I18N}
	AI_Title( I18N_Name( 'SkillMan', SkillMan[ Part^.S ].Name ) , Yellow );
{$ELSE PATCH_I18N}
	AI_Title( SkillMan[ Part^.S ].Name , Yellow );
{$ENDIF PATCH_I18N}
	AI_Title( BStr( Part^.V ) + ' DP' , Green );
end;


{$IFDEF PATCH_GH}
Procedure GearInfo( Part: GearPtr; X1,Y1,X2,Y2,BorColor: Byte; DebugMode: Boolean );
{$ELSE PATCH_GH}
Procedure GearInfo( Part: GearPtr; X1,Y1,X2,Y2,BorColor: Byte );
{$ENDIF PATCH_GH}
	{ Display some information for this gear inside the screen area }
	{ X1,Y1,X2,Y2. }
begin
	{ draw info window no larger than necessary }
	if Y2 - Y1 > 9 then Y2 := Y1 + 9;

	SetInfoZone( X1,Y1,X2,Y2,BorColor );

{$IFDEF PATCH_GH}
	if (NIL = Part) or (not(DebugMode) and (Part^.G <= GG_DisposeGear)) then Exit;
{$ELSE PATCH_GH}
	{ Record this gear's address. }
	LastGearShown := Part;

	{ Error check }
	{ Note that we want the area cleared, even in case of an error. }
	if Part = Nil then exit;
{$ENDIF PATCH_GH}

	{ Depending upon PART's type, branch to an appropriate procedure. }
	case Part^.G of
		GG_Mecha:	MekStatDisplay( Part );
		GG_Character:	CharacterInfo( Part );
		GG_MetaTerrain:	MetaTerrainInfo( Part );
		GG_RepairFuel:	RepairFuelInfo( Part );
{$IFDEF PATCH_GH}
	else MiscInfo( Part, DebugMode );
{$ELSE PATCH_GH}
	else MiscInfo( Part );
{$ENDIF PATCH_GH}
	end;

	{ Restore the clip area to the full screen. }
	maxclipzone;
end;

{$IFDEF PATCH_GH}
Procedure GearInfo( Part: GearPtr; X1,Y1,X2,Y2,BorColor: Byte );
begin
	GearInfo( Part, X1,Y1,X2,Y2,BorColor, False );
end;
{$ENDIF PATCH_GH}

Procedure LocationInfo( Part: GearPtr; gb: GameBoardPtr );
	{ Display location info for this part, if it is on the map. }
	{ This procedure is meant to be called after a GearInfo call, }
	{ since it assumes that ZX1,ZY1...etc will have been set up }
	{ properly beforehand. }
const
	OX = 3;
	OY = 2;
var
	D,Z: Integer;
begin
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	{ Props are master gears, but they don't get location info. }
	if OnTheMap( Part ) and IsMasterGear( Part ) and ( Part^.G <> GG_Prop ) then begin
		{ Clear the compass area. }
{$IFDEF GUIMSWINMODE}
		w32crt.gotoXY( ZX1 + OX - 1 , ZY1 + OY - 1 );
		conoutput.ConWrite( '   ' );
		w32crt.gotoXY( ZX1 + OX - 1 , ZY1 + OY );
		conoutput.ConWrite( '   ' );
		w32crt.gotoXY( ZX1 + OX - 1 , ZY1 + OY + 1 );
		conoutput.ConWrite( '   ' );
{$ELSE GUIMSWINMODE}
		gotoXY( ZX1 + OX - 1 , ZY1 + OY - 1 );
		write( '   ' );
		gotoXY( ZX1 + OX - 1 , ZY1 + OY );
		write( '   ' );
		gotoXY( ZX1 + OX - 1 , ZY1 + OY + 1 );
		write( '   ' );
{$ENDIF GUIMSWINMODE}

		D := NAttValue( Part^.NA , NAG_Location , NAS_D );
		Z := MekAltitude( gb , Part );
		if Z >= 0 then begin
{$IFDEF GUIMSWINMODE}
			w32crt.GotoXY( ZX1 + OX , ZY1 + OY );
			w32crt.TextColor( NeutralGrey );
			conoutput.ConWrite( BStr ( Z ) );
{$ELSE GUIMSWINMODE}
			GotoXY( ZX1 + OX , ZY1 + OY );
			TextColor( NeutralGrey );
			Write( BStr ( Z ) );
{$ENDIF GUIMSWINMODE}

		end else begin
{$IFDEF GUIMSWINMODE}
			w32crt.GotoXY( ZX1 + OX , ZY1 + OY );
			w32crt.TextColor( PlayerBlue );
			conoutput.ConWrite( BStr ( Abs( Z ) ) );
{$ELSE GUIMSWINMODE}
			GotoXY( ZX1 + OX , ZY1 + OY );
			TextColor( PlayerBlue );
			Write( BStr ( Abs( Z ) ) );
{$ENDIF GUIMSWINMODE}

		end;

{$IFDEF GUIMSWINMODE}
		w32crt.TextColor( White );
		w32crt.GotoXY( ZX1 + OX + AngDir[D,1] , ZY1 + OY + AngDir[D,2] );
		conoutput.ConWrite( '+' );
		w32crt.TextColor( DarkGray );
		w32crt.GotoXY( ZX1 + OX - AngDir[D,1] , ZY1 + OY - AngDir[D,2] );
		conoutput.ConWrite( '=' );
{$ELSE GUIMSWINMODE}
		TextColor( White );
		GotoXY( ZX1 + OX + AngDir[D,1] , ZY1 + OY + AngDir[D,2] );
		Write( '+' );
		TextColor( DarkGray );
		GotoXY( ZX1 + OX - AngDir[D,1] , ZY1 + OY - AngDir[D,2] );
		Write( '=' );
{$ENDIF GUIMSWINMODE}

		{ Speedometer. }
		if Speedometer( Part ) > 0 then begin
{$IFDEF GUIMSWINMODE}
			w32crt.GotoXY( ZX1 + OX - 3 , ZY1 + OY );
{$ELSE GUIMSWINMODE}
			GotoXY( ZX1 + OX - 3 , ZY1 + OY );
{$ENDIF GUIMSWINMODE}
			if NAttValue( Part^.NA , NAG_Action , NAS_MoveAction ) = NAV_FullSPeed then begin
{$IFDEF GUIMSWINMODE}
				w32crt.TextColor( LightCyan );
{$ELSE GUIMSWINMODE}
				TextColor( LightCyan );
{$ENDIF GUIMSWINMODE}
			end else begin
{$IFDEF GUIMSWINMODE}
				w32crt.TextColor( Cyan );
{$ELSE GUIMSWINMODE}
				TextColor( Cyan );
{$ENDIF GUIMSWINMODE}
			end;

{$IFDEF GUIMSWINMODE}
			conoutput.ConWrite( 'G' );
			w32crt.GotoXY( ZX1 + OX - 3 , ZY1 + OY + 1 );
			w32crt.TextColor( DarkGray );
			conoutput.ConWrite( 'S' );
{$ELSE GUIMSWINMODE}
			Write( 'G' );
			GotoXY( ZX1 + OX - 3 , ZY1 + OY + 1 );
			TextColor( DarkGray );
			Write( 'S' );
{$ENDIF GUIMSWINMODE}
		end else begin
{$IFDEF GUIMSWINMODE}
			w32crt.GotoXY( ZX1 + OX - 3 , ZY1 + OY );
			w32crt.TextColor( DarkGray );
			conoutput.ConWrite( 'G' );
			w32crt.GotoXY( ZX1 + OX - 3 , ZY1 + OY + 1 );
			w32crt.TextColor( Cyan );
			conoutput.ConWrite( 'S' );
{$ELSE GUIMSWINMODE}
			GotoXY( ZX1 + OX - 3 , ZY1 + OY );
			TextColor( DarkGray );
			Write( 'G' );
			GotoXY( ZX1 + OX - 3 , ZY1 + OY + 1 );
			TextColor( Cyan );
			Write( 'S' );
{$ENDIF GUIMSWINMODE}
		end;
	end;
end;


{$IFDEF PATCH_GH}
Procedure DisplayGearInfo( Part: GearPtr );
begin
	DisplayGearInfo( Part, False );
end;
{$ENDIF PATCH_GH}

{$IFDEF PATCH_GH}
Procedure DisplayGearInfo( Part: GearPtr; DebugMode: Boolean );
{$ELSE PATCH_GH}
Procedure DisplayGearInfo( Part: GearPtr );
{$ENDIF PATCH_GH}
	{ Show some stats for whatever sort of thing PART is. }
begin
	{ All this procedure does is call the ArenaInfo unit procedure }
	{ with the dimensions of the Info Zone. }
{$IFDEF PATCH_GH}
	GearInfo( Part, ScreenZone[ ZONE_Info, 1 ], ScreenZone[ ZONE_Info, 2 ], ScreenZone[ ZONE_Info, 3 ], ScreenZone[ ZONE_Info, 4 ], NeutralGrey, DebugMode );
{$ELSE PATCH_GH}
	GearInfo( Part , ScreenZone[ ZONE_Info , 1 ] , ScreenZone[ ZONE_Info , 2 ] , ScreenZone[ ZONE_Info , 3 ] , ScreenZone[ ZONE_Info , 4 ] , NeutralGrey );
{$ENDIF PATCH_GH}
end;

Procedure DisplayGearInfo( Part: GearPtr; gb: GameBoardPtr; Z: Integer );
	{ Show some stats for whatever sort of thing PART is. }
begin
	{ All this procedure does is call the ArenaInfo unit procedure }
	{ with the dimensions of the provided Zone. }
	GearInfo( Part , ScreenZone[ Z , 1 ] , ScreenZone[ Z , 2 ] , ScreenZone[ Z , 3 ] , ScreenZone[ Z , 4 ] , TeamColor( GB , Part ) );

	LocationInfo( Part , gb );
end;

Procedure DisplayGearInfo( Part: GearPtr; gb: GameBoardPtr );
	{ Show some stats for whatever sort of thing PART is. }
begin
	DisplayGearInfo( Part , GB , ZONE_Info );
end;

Function JobAgeGenderDesc( NPC: GearPtr ): String;
	{ Return the Job, Age, and Gender of the provided character in }
	{ a nicely formatted string. }
var
	msg,job: String;
begin
{$IFDEF PATCH_GH}
	if (NIL = NPC) or (NPC^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

{$IFDEF PATCH_I18N}
	JobAgeGenderDesc := ReplaceHash( I18N_MsgString('JobAgeGenderDesc'),
				BStr( NAttValue( NPC^.NA , NAG_CharDescription , NAS_DAge ) + 20 ),
				I18N_Name('GenderName',GenderName[ NAttValue( NPC^.NA , NAG_CharDescription , NAS_Gender ) ]),
				I18N_Name('Jobs',SAttValue( NPC^.SA , 'JOB' )) );
{$ELSE PATCH_I18N}
	msg := BStr( NAttValue( NPC^.NA , NAG_CharDescription , NAS_DAge ) + 20 );
	msg := msg + ' year old ' + LowerCase( GenderName[ NAttValue( NPC^.NA , NAG_CharDescription , NAS_Gender ) ] );
	job := SAttValue( NPC^.SA , 'JOB' );
	if job <> '' then msg := msg + ' ' + LowerCase( job );
	msg := msg + '.';
	JobAgeGenderDesc := msg;
{$ENDIF PATCH_I18N}
end;

Procedure DisplayInteractStatus( GB: GameBoardPtr; NPC: GearPtr; React,Endurance: Integer );
	{ Show the needed information regarding this conversation. }
var
	msg: String;
	C: Byte;
	T: Integer;
begin
{$IFDEF PATCH_GH}
	if (NIL = NPC) or (NPC^.G <= GG_DisposeGear) then Exit;
{$ENDIF PATCH_GH}

	ZX1 := ScreenZone[ ZONE_InteractStatus , 1 ];
	ZY1 := ScreenZone[ ZONE_InteractStatus , 2 ];
	ZX2 := ScreenZone[ ZONE_InteractStatus , 3 ];
	ZY2 := ScreenZone[ ZONE_InteractStatus , 4 ];
	CX := 1;
	CY := 1;
{$IFDEF GUIMSWINMODE}
	w32crt.Window( ZX1 , ZY1 , ZX2 , ZY2 );
	w32crt.ClrScr;
{$ELSE GUIMSWINMODE}
	Window( ZX1 , ZY1 , ZX2 , ZY2 );
	ClrScr;
{$ENDIF GUIMSWINMODE}

	{ First the name, then the description. }
	AI_Title( GearName( NPC ) , InfoHiLight );

	AI_Title( JobAgeGenderDesc( NPC ) , InfoGreen );

	if React > 0 then begin
		msg := '';
		for T := 0 to ( React div 4 ) do msg := msg + '+';
		C := LightGreen;
	end else if React < 0 then begin
		msg := '';
		for T := 0 to ( Abs(React) div 4 ) do msg := msg + '-';
		C := LightRed;
	end else begin
		msg := '~~~';
		C := Yellow;
	end;
	AI_PrintFromRight( '[:)]' , 1 , Green );
	AI_PrintFromRight( msg , 5 , C );

	msg := '';
	if Endurance > 10 then Endurance := 10;
	for t := 1 to Endurance do msg := msg + '>';
	AI_PrintFromRight( '[Zz]' , ZX2 - ZX1 - 12 , Green );
	AI_PrintFromRight( msg , ZX2 - ZX1 - 8 , LightGreen );

	{ Restore the clip area to the full screen. }
	maxclipzone;
end;

Procedure QuickWeaponInfo( Part: GearPtr );
	{ Provide quick info for this weapon in the MENU1 zone. }
begin
	{ Error check }
	{ Note that we want the area cleared, even in case of an error. }
{$IFDEF PATCH_GH}
	if (NIL = Part) or (Part^.G <= GG_DisposeGear) then Exit;
{$ELSE PATCH_GH}
	if Part = Nil then exit;
{$ENDIF PATCH_GH}

	{ Display the weapon description. }
	GameMsg( GearName( Part ) + ' ' + WeaponDescription( Part ) , ZONE_Menu1 , InfoGreen );
end;

Procedure CharacterDisplay( PC: GearPtr; GB: GameBoardPtr );
	{ Display the important stats for this PC in the map zone. }
var
	msg: String;
	T,FID: Integer;
	S: LongInt;
	C: Byte;
	CY0,R: Integer;
	Mek: GearPtr;
begin
	{ Begin with one massive error check... }
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
{$ELSE PATCH_GH}
	if PC = Nil then Exit;
{$ENDIF PATCH_GH}
	if PC^.G <> GG_Character then PC := LocatePilot( PC );
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
{$ELSE PATCH_GH}
	if PC = Nil then Exit;
{$ENDIF PATCH_GH}

	SetInfoZone( ScreenZone[ ZONE_Map , 1 ] - 1 , ScreenZone[ ZONE_Map , 2 ] - 1 , ScreenZone[ ZONE_Map , 3 ] + 1 , ScreenZone[ ZONE_Map , 4 ] + 1 , PlayerBlue );

	AI_Title( GearName( PC ) , White );
	AI_Title( JobAgeGenderDesc( PC ) , InfoGreen );
	AI_NextLine;

	{ Print the stats. }
	{ Save the CY value, since we'll want to come back and add more }
	{ info to the right side of the screen. }
	CY0 := CY;
	for t := 1 to NumGearStats do begin
		{ Find the adjusted stat value for this stat. }
		S := CStat( PC , T );
		R := ( S + 2 ) div 3;
		if R > 7 then R := 7;

		{ Determine an appropriate color for the stat, depending }
		{ on whether its adjusted value is higher or lower than }
		{ the basic value. }
		if S > PC^.Stat[ T ] then C := LighTGreen
		else if S < PC^.Stat[ T ] then C := LightRed
		else C := Green;

		{ Do the output. }
{$IFDEF PATCH_I18N}
		AI_PrintFromRight( I18N_Name( 'StatName', StatName[ T ] ), 2 , LightGray );
{$ELSE PATCH_I18N}
		AI_PrintFromRight( StatName[ T ] , 2 , LightGray );
{$ENDIF PATCH_I18N}
		AI_PrintFromLeft( BStr( S ) , 15 , C );
		AI_PrintFromRight( MsgString( 'STATRANK' + BStr( R ) ) , 16 , C );

		AI_NextLine;
	end;

	{ Retsore CY. }
	CY := CY0;

	{ Calculate the mid point at which to print the second column. }
	T := ( ZX2 - ZX1 ) div 2 + 3;

	AI_PrintFromRight( MsgString( 'INFO_XP' ) , T , LightGray );
	S := NAttVAlue( PC^.NA , NAG_Experience , NAS_TotalXP );
	AI_PrintFromLeft( BStr( S ) , ZX2 - ZX1 + 1 , Green );
	AI_NextLine;

	AI_PrintFromRight( MsgString( 'INFO_XPLeft' ) , T , LightGray );
	S := S - NAttVAlue( PC^.NA , NAG_Experience , NAS_SpentXP );
	AI_PrintFromLeft( BStr( S ) , ZX2 - ZX1 + 1 , Green );
	AI_NextLine;

	AI_PrintFromRight( MsgString( 'INFO_Credits' ) , T , LightGray );
	S := NAttVAlue( PC^.NA , NAG_Experience , NAS_Credits );
	AI_PrintFromLeft( '$' + BStr( S ) , ZX2 - ZX1 + 1 , Green );
	AI_NextLine;

	{ Print info on the PC's mecha, if appropriate. }
{$IFDEF PATCH_GH}
	if ( GB <> Nil ) then begin
		Mek := FindPilotsMecha( GB^.Meks , PC );
		if (NIL <> Mek) and (GG_DisposeGear < Mek^.G) then begin
			AI_NextLine;
			AI_PrintFromRight( MsgString( 'INFO_MekSelect' ) , T , LightGray );
			AI_NextLine;

			msg := FullGearName( Mek );
			AI_PrintFromLeft( Msg , ZX2 - ZX1 + 1 , Green );
		end;

		{ Print info on the PC's faction, if appropriate. }
		FID := NAttValue( PC^.NA , NAG_Personal , NAS_FactionID );
		if ( FID <> 0 ) and ( GB^.Scene <> Nil ) then begin
			Mek := SeekFaction( GB^.Scene , FID );
			if (NIL <> Mek) and (GG_DisposeGear < Mek^.G) then begin
				AI_NextLine;
				AI_PrintFromRight( MsgString( 'INFO_Faction' ) , T , LightGray );
				AI_NextLine;

				msg := GearName( Mek );
				AI_PrintFromLeft( Msg , ZX2 - ZX1 + 1 , Green );
			end;
		end;
	end;
{$ELSE PATCH_GH}
	if ( GB <> Nil ) then begin
		Mek := FindPilotsMecha( GB^.Meks , PC );
		if Mek <> Nil then begin
			AI_NextLine;
			AI_PrintFromRight( MsgString( 'INFO_MekSelect' ) , T , LightGray );
			AI_NextLine;

			msg := FullGearName( Mek );

			AI_PrintFromLeft( Msg , ZX2 - ZX1 + 1 , Green );
		end;
	end;

	{ Print info on the PC's faction, if appropriate. }
	FID := NAttValue( PC^.NA , NAG_Personal , NAS_FactionID );
	if ( FID <> 0 ) and ( GB <> Nil ) and ( GB^.Scene <> Nil ) then begin
		Mek := SeekFaction( GB^.Scene , FID );
		if Mek <> Nil then begin
			AI_NextLine;
			AI_PrintFromRight( MsgString( 'INFO_Faction' ) , T , LightGray );
			AI_NextLine;

			msg := GearName( Mek );

			AI_PrintFromLeft( Msg , ZX2 - ZX1 + 1 , Green );
		end;
	end;
{$ENDIF PATCH_GH}

	msg := SAttValue( PC^.SA , 'BIO1' );
	if msg <> '' then begin
		GameMsg( msg , ZONE_Biography , Green );
	end;

	{ Restore the display. }
	MaxClipZone;
end;

Procedure InjuryViewer( PC: GearPtr );
	{ Display a brief listing of all the PC's major health concerns. }
	Procedure ShowSubInjuries( Part: GearPtr );
		{ Show the injuries of this part, and also for its subcoms. }
	var
		MD,CD: Integer;
	begin
		while Part <> Nil do begin
{$IFDEF PATCH_GH}
			if (GG_DisposeGear < Part^.G) then begin
{$ENDIF PATCH_GH}
			MD := GearMaxDamage( Part );
			CD := GearCurrentDamage( Part );
			if not PartActive( Part ) then begin
				AI_PrintFromRight( GearName( Part ) + MsgString( 'INFO_IsDisabled' ) , 2 , StatusColor( MD , CD ) );
				AI_NextLine;
			end else if CD < MD then begin
				AI_PrintFromRight( GearName( Part ) + MsgString( 'INFO_IsHurt' ) , 2 , StatusColor( MD , CD ) );
				AI_NextLine;
			end;
			ShowSubInjuries( Part^.SubCom );
{$IFDEF PATCH_GH}
			end;
{$ENDIF PATCH_GH}
			Part := Part^.Next;
		end;
	end;
var
	SP,MP,T: Integer;
begin
	{ Begin with one massive error check... }
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
{$ELSE PATCH_GH}
	if PC = Nil then Exit;
{$ENDIF PATCH_GH}
	if PC^.G <> GG_Character then PC := LocatePilot( PC );
{$IFDEF PATCH_GH}
	if (NIL = PC) or (PC^.G <= GG_DisposeGear) then Exit;
{$ELSE PATCH_GH}
	if PC = Nil then Exit;
{$ENDIF PATCH_GH}

	SetInfoZone( ScreenZone[ ZONE_Map , 1 ] - 1 , ScreenZone[ ZONE_Map , 2 ] - 1 , ScreenZone[ ZONE_Map , 3 ] + 1 , ScreenZone[ ZONE_Map , 4 ] + 1 , PlayerBlue );

	AI_Title( MsgString( 'INFO_InjuriesTitle' ) , White );

	{ Show exhaustion status first. }
	SP := CharCurrentStamina( PC );
	MP := CharCurrentMental( PC );
	if ( SP = 0 ) and ( MP = 0 ) then begin
		AI_PrintFromRight( MsgString( 'INFO_FullExhausted' ) , 2 , LightRed );
		AI_NextLine;
	end else if ( SP = 0 ) or ( MP = 0 ) then begin
		AI_PrintFromRight( MsgString( 'INFO_PartExhausted' ) , 2 , Yellow );
		AI_NextLine;
	end;

	{ Hunger next. }
	T := NAttValue( PC^.NA , NAG_Condition , NAS_Hunger ) - Hunger_Penalty_Starts;
	if T > ( NumGearStats * 3 ) then begin
		AI_PrintFromRight( MsgString( 'INFO_ExtremeHunger' ) , 2 , LightRed );
		AI_NextLine;
	end else if T > ( NumGearStats * 2 ) then begin
		AI_PrintFromRight( MsgString( 'INFO_Hunger' ) , 2 , Yellow );
		AI_NextLine;
	end else if T > 0 then begin
		AI_PrintFromRight( MsgString( 'INFO_MildHunger' ) , 2 , Green );
		AI_NextLine;
	end;

	{ Low morale next. }
	T := NAttValue( PC^.NA , NAG_Condition , NAS_MoraleDamage );
	if T > 65 then begin
		AI_PrintFromRight( MsgString( 'INFO_ExtremeMorale' ) , 2 , LightRed );
		AI_NextLine;
	end else if T > 40 then begin
		AI_PrintFromRight( MsgString( 'INFO_Morale' ) , 2 , Yellow );
		AI_NextLine;
	end else if T > 20 then begin
		AI_PrintFromRight( MsgString( 'INFO_MildMorale' ) , 2 , Green );
		AI_NextLine;
	end;


	for t := 1 to Num_Status_FX do begin
		if NAttValue( PC^.NA , NAG_StatusEffect , T ) <> 0 then begin
			AI_PrintFromRight( MsgString( 'INFO_Status' + BStr( T ) ) , 2 , LightRed );
			AI_NextLine;
		end;
	end;

	{ Show limb injuries. }
	ShowSubInjuries( PC^.SubCom );

	{ Restore the display. }
	MaxClipZone;
end;

Procedure MapEditInfo( Pen,Palette,X,Y: Integer );
	{ Show the needed info for the map editor- the current pen }
	{ terrain, the terrain palette, and the cursor position. }
begin
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( ScreenZone[ ZONE_Info , 1 ] + 1 , ScreenZone[ ZONE_Info , 2 ] + 1 );
	w32crt.TextBackground( Black );
	w32crt.ClrEOL;
	w32crt.TextColor( White );
	conoutput.ConWrite( '[' );
	w32crt.TextColor( TerrColor[ Pen ] );
	conoutput.ConWrite( TerrGfx[ Pen ] );
	w32crt.TextColor( White );
{$ELSE GUIMSWINMODE}
	GotoXY( ScreenZone[ ZONE_Info , 1 ] + 1 , ScreenZone[ ZONE_Info , 2 ] + 1 );
	TextBackground( Black );
	ClrEOL;
	TextColor( White );
	Write( '[' );
	TextColor( TerrColor[ Pen ] );
	Write( TerrGfx[ Pen ] );
	TextColor( White );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	WriteMBCharStr( '] ' + I18N_Name('TerrMan',TerrMan[Pen].Name), 0 );
{$ELSE PATCH_I18N}
	Write( '] ' + TerrMan[ Pen ].Name );
{$ENDIF PATCH_I18N}

	CMessage( BStr( X ) + ',' + BStr( Y ) , ZONE_Clock , White );
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: coninfo.pp');
{$ENDIF DEBUG}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: coninfo.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
