unit context;
	{ This unit deals with text output & keyboard input. }
{
	GearHead: Arena, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}

interface

uses
{$IFDEF DEBUG}
	sysutils,
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
	gears_base,
{$ENDIF PATCH_GH}
	gears;

const
	Console_History_Length = 240;

var
	Text_Messages: SAttPtr;
	Console_History: SAttPtr;

{$IFDEF PATCH_I18N}
Procedure WriteMBCharStr( const arg_msg: String; Xwidth: Integer );
{$ENDIF PATCH_I18N}

Function RPGKey: Char;
Function DirKey: Integer;
Procedure EndOfGameMoreKey;
Procedure CMessage( const msg: String; Z: Integer; C: Byte );
Procedure GameMSG( msg: string; X1,Y1,X2,Y2,C: Byte ); {no const}
Procedure GameMSG( const msg: string; Z,C: Byte );
{$IFDEF PATCH_GH}
Procedure RedrawConsole;
{$ENDIF PATCH_GH}
Procedure DialogMSG(msg: string); {no const}
{$IFDEF PATCH_GH}
Function GetStringFromUser( const Prompt, Init_text: String ): String;
{$ENDIF PATCH_GH}
Function GetStringFromUser( const Prompt: String ): String;
Function MsgString( const MsgLabel: String ): String;

Function MoreHighFirstLine( LList: SAttPtr ): Integer;
Procedure MoreText( LList: SAttPtr; FirstLine: Integer );

{$IFDEF PATCH_GH}
	{ SetupHQDisplay was moved from congfx.pp. }
Procedure SetupHQDisplay;
{$ELSE PATCH_GH}
{$ENDIF PATCH_GH}

implementation

uses
{$IFDEF PATCH_I18N}
  {$IFDEF WITH_WIDECHAR}
	strings,
  {$ELSE WITH_WIDECHAR}
    {$IFDEF GUIMSWINMODE}
	strings,
    {$ENDIF GUIMSWINMODE}
  {$ENDIF WITH_WIDECHAR}
{$ENDIF PATCH_I18N}
{$IFDEF GUIMSWINMODE}
{$ELSE GUIMSWINMODE}
	crt,
{$ENDIF GUIMSWINMODE}
{$IFDEF DEBUG}
	errmsg,
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
	pseudosmartpointer,
{$ENDIF PATCH_GH}
{$IFDEF PATCH_I18N}
  {$IFDEF PATCH_GH}
	i18nmsg,
  {$ENDIF PATCH_GH}
  {$IFDEF WITH_TENC}
	iconv,
  {$ENDIF WITH_TENC}
  {$IFDEF GUIMSWINMODE}
	w32crt,
	conoutput,
  {$ENDIF GUIMSWINMODE}
{$ENDIF PATCH_I18N}
	texutil,ui4gh,congfx
{$IFDEF PATCH_GH}
	,conmenus
{$ENDIF PATCH_GH}
	;

{$IFDEF PATCH_I18N}
Procedure WriteMBCharStr( const arg_msg: String; Xwidth: Integer );
	{ NOTE: In CJK, there are many charctors, one charactor have double size for one ANK charactor and data length is 3 or 4 bytes. }
	{ NOTE: But, a function Write() clipped by data length. }
	{ NOTE: FPC's unicode functions is not stable, is its? }
  {$IFDEF GUIMSWINMODE}
const
	WCLen = 288; { 254; }
  {$ELSE GUIMSWINMODE}
    {$IFDEF WITH_WIDECHAR}
const
	WCLen = 512;
    {$ELSE WITH_WIDECHAR}
      {$IFDEF WITH_TENC}
const
	tmsgLen = 288; { 254; }
      {$ENDIF WITH_TENC}
    {$ENDIF WITH_WIDECHAR}
  {$ENDIF GUIMSWINMODE}

var
  {$IFDEF GUIMSWINMODE}
	pmsg: PChar;
	pdst: PChar;
	dstlen: Integer;
	dst: String;
  {$ELSE GUIMSWINMODE}
    {$IFDEF WITH_WIDECHAR}
	pmsg: PChar;
	pdst: PWideChar;
	dstlen: Integer;
	wdst: WideString;
    {$ELSE WITH_WIDECHAR}
	msg: String;
      {$IFDEF WITH_TENC}
	tmsg: Array[0..tmsgLen] of Char;
	pmsg, ptmsg: PChar;
      {$ENDIF WITH_TENC}
	MaxLen: Integer;
	P, lastP: Integer;
	X, Y: Integer;
	Len: Integer;
    {$ENDIF WITH_WIDECHAR}
  {$ENDIF GUIMSWINMODE}

begin
  {$IFDEF GUIMSWINMODE}
	pmsg := QuickPCopy(arg_msg);
	pdst := StrAlloc(WCLen);
	dstlen := Conv_ToTenc( pmsg, Length(arg_msg), pdst, WCLen );
	dst := pdst;
	conoutput.ConWrite( dst );
	Dispose( pdst );
	Dispose( pmsg );
  {$ELSE GUIMSWINMODE}
    {$IFDEF WITH_WIDECHAR}
	pmsg := QuickPCopy(arg_msg);
	pdst := PWideChar(StrAlloc(WCLen));
	dstlen := Conv_ToUni16( pmsg, Length(arg_msg), PWord(pdst), WCLen );
	wdst := pdst;
	Write( wdst );
	Dispose( pdst );
	Dispose( pmsg );
    {$ELSE WITH_WIDECHAR}
      {$IFDEF WITH_TENC}
	if SYSTEM_CHARSET = TERMINAL_CHARSET then begin
		if TERMINAL_bidiRTL then begin
			msg := Conv_bidiRTL(arg_msg);
		end else begin
			msg := arg_msg;
		end;
	end else begin
		ptmsg := tmsg;
		if TERMINAL_bidiRTL then begin
			pmsg := QuickPCopy(Conv_bidiRTL(arg_msg));
		end else begin
			pmsg := QuickPCopy(arg_msg);
		end;
		Conv_ToTenc( pmsg, Length(arg_msg), ptmsg, tmsgLen );
		Dispose( pmsg );
		msg := StrPas( tmsg );
	end;
      {$ELSE WITH_TENC}
	msg := arg_msg;
      {$ENDIF WITH_TENC}

	MaxLen := Length(msg);
	P := 1;	lastP := 1;
	X := WhereX;	Y := WhereY;
	if TERMINAL_bidiRTL and (0 < Xwidth) then begin
		X := X + Xwidth - WidthMBcharStr(msg);
		if X < GOTOXY_MIN then begin
			X := GOTOXY_MIN;
		end;
		GotoXY(X,Y);
	end;

      {$IFDEF PASCAL_WRITE_BUG_HACK}
	while (P <= MaxLen) do begin
        {$IFDEF WITH_TENC}
		Len := LengthMBChar( msg[P], TENC );
        {$ELSE WITH_TENC}
		Len := LengthMBChar( msg[P] );
        {$ENDIF WITH_TENC}
		if 0 < Len then begin
			if Len <= 2 then begin
				{ BUG: Display width is mistook if use JISx0201-KANA. }
				P := P + Len;
				X := X + Len;
			end else begin
				Write(Copy(msg,lastP,P-lastP+len));
				P := P + Len;
				lastP := P;
				X := X + 2;
				GotoXY(X,Y+1);
				if (Y+1) <> WhereY then begin
					GotoXY(X,Y-1);
					if (Y-1) <> WhereY then begin
						GotoXY(X+2,Y);
					end;
				end;
				GotoXY(X,Y);
			end;
		end else begin
			Inc(P);
			Inc(X);
		end;
	end;
	if lastP < P then Write(Copy(msg,lastP,MaxLen-lastP+1));
      {$ELSE PASCAL_WRITE_BUG_HACK}
	Write( msg );
      {$ENDIF PASCAL_WRITE_BUG_HACK}
    {$ENDIF WITH_WIDECHAR}
  {$ENDIF GUIMSWINMODE}
end;
{$ENDIF PATCH_I18N}


Function RPGKey: Char;
	{Read a keypress from the keyboard. Convert it into a form}
	{that my other procedures would be willing to call useful.}
var
	rk,getit: Char;
begin
{$IFDEF GUIMSWINMODE}
	RK := w32crt.ReadKey;
{$ELSE GUIMSWINMODE}
	RK := ReadKey;
{$ENDIF GUIMSWINMODE}
{$IFDEF DEBUG}
	WriteLn(IntToHex(Ord(RK),2));
{$ENDIF DEBUG}

	Case RK of
		{ 45 5B 1B: 5 in TenKey on Unix-Console }
		#0: begin	{We have a two-part special key.}
			{Obtain the scan code.}
{$IFDEF GUIMSWINMODE}
			getit := w32crt.ReadKey;
{$ELSE GUIMSWINMODE}
			getit := ReadKey;
{$ENDIF GUIMSWINMODE}
{$IFDEF DEBUG}
			WriteLn(IntToHex(Ord(getit),2));
{$ENDIF DEBUG}
			case getit of
{$IFDEF PATCH_GH}
				#$52: RK := '0';	{ 0 in TenKey }
				#$53: RK := '.';	{ . in TenKey on X-Window/MS-Windows }
				#$5A: RK := KeyMap[ KMC_SouthWest ].KCode;	{ End Cursor Key on X-Window }
				#$4F: RK := KeyMap[ KMC_SouthWest ].KCode;	{ End Cursor Key on Unix-Console, MS-Windows }
				#$50: RK := KeyMap[ KMC_South ].KCode;		{ Down Cursor Key }
				#$51: RK := KeyMap[ KMC_SouthEast ].KCode;	{ PageDown Cursor Key }
				#$4B: RK := KeyMap[ KMC_West ].KCode;		{ Left Cursor Key }
				#$5F: RK := '5';				{ 5 in TenKey on X-Window }
				#$4C: RK := '5';				{ 5 in TenKey on MS-Windows }
				#$4D: RK := KeyMap[ KMC_East ].KCode;		{ Right Cursor Key }
				#$5B: begin
						getit := ReadKey;
						case getit of
							#$1B: RK := KeyMap[ KMC_NorthWest ].KCode;	{ Home Cursor Key on X-Window }
							else RK := #0;
						end;
					end;
				#$47: RK := KeyMap[ KMC_NorthWest ].KCode;	{ Home Cursor Key on Unix-Console, MS-Windows }
				#$48: RK := KeyMap[ KMC_North ].KCode;		{ Up Cursor Key }
				#$49: RK := KeyMap[ KMC_NorthEast ].KCode;	{ PageUp Cursor Key }
				#$35: RK := '/';	{ Slash in TenKey on MS-Windows }
				#$1C: RK := ' ';	{ Enter in TenKey on MS-Windows => Altanative-RET }
				{ #$3B: }{ F1  on Unix-Console, MS-Windows }
				{ #$3C: }{ F2  on Unix-Console, MS-Windows }
				{ #$3D: }{ F3  on Unix-Console, MS-Windows }
				{ #$3E: }{ F4  on Unix-Console, MS-Windows }
				{ #$3F: }{ F5  on Unix-Console, MS-Windows }
				{ #$40: }{ F6  }
				{ #$41: }{ F7  }
				{ #$42: }{ F8  }
				{ #$43: }{ F9  }
				{ #$44: }{ F10 }
				{ #$85: }{ F11 }
				{ #$86: }{ F12 }
				#$29: RK := #$1B;	{ JP109key-Hankaku/Zenkaku on MS-Windows => ESC }
{$ELSE PATCH_GH}
				#72: RK := KeyMap[ KMC_North ].KCode; {Up Cursor Key}
				#71: RK := KeyMap[ KMC_NorthWest ].KCode; {Home Cursor Key}
				#73: RK := KeyMap[ KMC_NorthEast ].KCode; {PageUp Cursor Key}
				#80: RK := KeyMap[ KMC_South ].KCode; {Down Cursor Key}
				#79: RK := KeyMap[ KMC_SouthWest ].KCode; {End Cursor Key}
				#81: RK := KeyMap[ KMC_SouthEast ].KCode; {PageDown Cursor Key}
				#75: RK := KeyMap[ KMC_West ].KCode; {Left Cursor Key}
				#77: RK := KeyMap[ KMC_East ].KCode; {Right Cursor Key}
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
				#$86: if Cheat_Display then if Cheat_Display_SW then Cheat_Display_SW := False else Cheat_Display_SW := True;
{$ENDIF PATCH_CHEAT}
{$IFDEF PATCH_GH}
				else RK := #0;
{$ENDIF PATCH_GH}
			end;
		end;

		{Convert the Backspace character to ESCape.}
		#8: RK := #27;	{ Backspace, DEL in TenKey on Unix-Console => ESC }

		{Normally, SPACE is the selection button, but ENTER should}
		{work as well. Therefore, convert all enter codes to spaces.}
		#10: RK := ' ';
		#13: RK := ' ';	{ Enter, Enter in TenKey on Unix-Console/X-Window }
	end;

	RPGKey := RK;
end;

Function DirKey: Integer;
	{ Get a direction selection from the user. If a standard direction }
	{ key was selected, return its direction (0 is East, increase }
	{ clockwise). See Locale.pp for details. }
	{ Return -1 if no good direction was chosen. }
var
	K: Char;
begin
	K := RPGKey;
	if K = KeyMap[ KMC_East ].KCode then begin
		DirKey := 0;
	end else if K = KeyMap[ KMC_SouthEast ].KCode then begin
		DirKey := 1;
	end else if K = KeyMap[ KMC_South ].KCode then begin
		DirKey := 2;
	end else if K = KeyMap[ KMC_SouthWest ].KCode then begin
		DirKey := 3;
	end else if K = KeyMap[ KMC_West ].KCode then begin
		DirKey := 4;
	end else if K = KeyMap[ KMC_NorthWest ].KCode then begin
		DirKey := 5;
	end else if K = KeyMap[ KMC_North ].KCode then begin
		DirKey := 6;
	end else if K = KeyMap[ KMC_NorthEast ].KCode then begin
		DirKey := 7;
{$IFDEF PATCH_GH}
	end else if K = KeyMap[ KMC_Enter ].KCode then begin
		DirKey := 8;
	end else if K = KeyMap[ KMC_Enter2 ].KCode then begin
		DirKey := 8;
	end else if K = ' ' then begin
		DirKey := 8;
	end else if K = #27 then begin
		DirKey := -1;
	end else if K = KeyMap[ KMC_QuitGame ].KCode then begin
		DirKey := -1;
	end else if K = KeyMap[ KMC_Eject ].KCode then begin
		DirKey := -1;
	end else begin
		DirKey := 8;
{$ELSE PATCH_GH}
	end else begin
		DirKey := -1;
{$ENDIF PATCH_GH}
	end;

end;

Procedure EndOfGameMoreKey;
	{ The end of the game has been reached. Wait for the user to }
	{ press either the space bar or the ESC key. }
var
	A: Char;
begin
{$IFDEF PATCH_GH}
  {$IFDEF PATCH_I18N}
	DialogMSG( I18N_MsgString('EndOfGameMoreKey','Hit space bar') );
  {$ELSE PATCH_I18N}
	DialogMSG( '[Hit space bar.]' );
  {$ENDIF PATCH_I18N}
{$ENDIF PATCH_GH}

	{ First, get rid of any pending keypresses. }
{$IFDEF GUIMSWINMODE}
	while w32crt.keypressed do w32crt.readkey;
{$ELSE GUIMSWINMODE}
	while keypressed do readkey;
{$ENDIF GUIMSWINMODE}

	{ Keep reading keypresses until either a space or an ESC is found. }
	repeat
		A := RPGKey;
	until ( A = ' ' ) or ( A = #27 );
end;

Procedure CMessage( const msg: String; Z: Integer; C: Byte );
	{ Display MSG centered in zone Z. }
var
	X,Y: Integer;
begin
	{ Figure out the coordinates for centered display. }
{$IFDEF PATCH_I18N}
	X := ( ScreenZone[Z,3] + ScreenZone[Z,1] ) div 2 - ( WidthMBcharStr( msg ) div 2 ) + 1;
{$ELSE PATCH_I18N}
	X := ( ScreenZone[Z,3] + ScreenZone[Z,1] ) div 2 - ( Length( msg ) div 2 ) + 1;
{$ENDIF PATCH_I18N}
	Y := ( ScreenZone[Z,4] + ScreenZone[Z,2] ) div 2;

	{ Actually do the output. }
	ClrZone( Z );
	if X < 1 then X := 1;
	if Y < 1 then Y := 1;
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( X , Y );
	w32crt.TextColor( C );
{$ELSE GUIMSWINMODE}
	GotoXY( X , Y );
	TextColor( C );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	WriteMBCharStr(msg,0);
{$ELSE PATCH_I18N}
	Write(msg);
{$ENDIF PATCH_I18N}
end;

Procedure GameMSG( msg: string; X1,Y1,X2,Y2,C: Byte );  {not const-able}
	{Prettyprint the string MSG with color C in screen zone Z.}
var
	Width: Integer;		{Our pixel width.}
	NextWord: String;
	THELine: String;	{The line under construction.}
	LC: Boolean;		{Loop Condition.}
{$IFDEF PATCH_I18N}
	LW_I18N: Boolean;	{Is the last word I18N character?}
	CW_I18N: Boolean;	{Is the current word I18N character?}
	DItS: Boolean;		{Do insert the space, or not.}
{$ENDIF PATCH_I18N}
begin
	{ CLean up the message a bit. }
	DeleteWhiteSpace( msg );
{$IFDEF GUIMSWINMODE}
	w32crt.TextColor( C );
	w32crt.TextBackground( StdBlack );
{$ELSE GUIMSWINMODE}
	TextColor( C );
	TextBackground( StdBlack );
{$ENDIF GUIMSWINMODE}

	{Clear the message area, and set clipping bounds.}
{$IFDEF GUIMSWINMODE}
	w32crt.Window( X1 , Y1 , X2 , Y2 );
	w32crt.ClrScr;
{$ELSE GUIMSWINMODE}
	Window( X1 , Y1 , X2 , Y2 );
	ClrScr;
{$ENDIF GUIMSWINMODE}

	{Calculate the width of the text area.}
	Width := X2 - X1;

	{THELine = The first word in this iteration}
{$IFDEF PATCH_I18N}
	LW_I18N := False;
	THELine := ExtractWord( msg, DItS, CW_I18N );
{$ELSE PATCH_I18N}
	THELine := ExtractWord( msg );
{$ENDIF PATCH_I18N}

	{Start the main processing loop.}
	while TheLine <> '' do begin
		{Set the LoopCondition to True.}
		LC := True;

		{ Start building the line. }
		repeat
{$IFDEF PATCH_I18N}
			NextWord := ExtractWord( Msg, DItS, CW_I18N );

			if (False = LW_I18N) and (False = CW_I18N) then begin
				DItS := True;
			end;
			LW_I18N := CW_I18N;
			if DItS then begin
				if WidthMBcharStr(THEline + ' ') <= Width then begin
					THEline := THEline + ' ';
				end;
			end;
			if WidthMBcharStr(THEline + NextWord) <= Width then begin
				THEline := THEline + NextWord;
			end else
				LC := False;
{$ELSE PATCH_I18N}
			NextWord := ExtractWord( Msg );

			if Length(THEline + ' ' + NextWord) < Width then
				THEline := THEline + ' ' + NextWord
			else
				LC := False;
{$ENDIF PATCH_I18N}

		until (not LC) or (NextWord = '') or ( TheLine[Length(TheLine)] = #13 );

		{ If the line ended due to a line break, deal with it. }
		if ( TheLine[Length(TheLine)] = #13 ) then begin
			{ Display the line break as a space. }
			TheLine[Length(TheLine)] := ' ';
{$IFDEF PATCH_I18N}
			NextWord := ExtractWord( msg, DItS, CW_I18N );
{$ELSE PATCH_I18N}
			NextWord := ExtractWord( msg );
{$ENDIF PATCH_I18N}
		end;

		{ Output the line. }
		if NextWord = '' then begin
{$IFDEF PATCH_I18N}
			WriteMBCharStr(THELine,Width);
{$ELSE PATCH_I18N}
			Write(THELine);
{$ENDIF PATCH_I18N}
		end else begin
{$IFDEF PATCH_I18N}
			WriteMBCharStr(THELine,Width);
  {$IFDEF GUIMSWINMODE}
			conoutput.ConWriteLn;
  {$ELSE GUIMSWINMODE}
			WriteLn;
  {$ENDIF GUIMSWINMODE}
{$ELSE PATCH_I18N}
			WriteLn(THELine);
{$ENDIF PATCH_I18N}
		end;

		{ Prepare for the next iteration. }
		TheLine := NextWord;

	end; { while msg <> '' }

	{Restore the clip window to its maximum size.}
	MaxClipZone;
end;

Procedure GameMSG( const msg: string; Z,C: Byte );
	{ Print a message in zone Z. }
begin
	GameMSG( msg , ScreenZone[Z,1], ScreenZone[Z,2], ScreenZone[Z,3], ScreenZone[Z,4], C );
end;

{$IFDEF PATCH_GH}
Procedure RedrawConsole;
	{ Redraw the console. }
var
	SL: SAttPtr;
{$IFDEF PATCH_I18N}
	MaxWidth: Integer;
{$ENDIF PATCH_I18N}
begin
	{ Restore the console display. }
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( ScreenZone[ ZONE_Dialog , 1 ] , ScreenZone[ ZONE_Dialog , 2 ] -1 );
	w32crt.TextColor( Green );
{$ELSE GUIMSWINMODE}
	GotoXY( ScreenZone[ ZONE_Dialog , 1 ] , ScreenZone[ ZONE_Dialog , 2 ] -1 );
	TextColor( Green );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	MaxWidth := ScreenZone[ZONE_Dialog,3] - ScreenZone[ZONE_Dialog,1];
{$ENDIF PATCH_I18N}
	SL := RetrieveSAtt( Console_History , NumSAtts( Console_History ) - ScreenRows + ScreenZone[ ZONE_Dialog , 2 ] );
	if SL = Nil then SL := Console_History;
	while SL <> Nil do begin
{$IFDEF GUIMSWINMODE}
		conoutput.ConWriteLn;
{$ELSE GUIMSWINMODE}
		writeln;
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
		WriteMBCharStr( SL^.Info, MaxWidth );
{$ELSE PATCH_I18N}
		write( SL^.Info );
{$ENDIF PATCH_I18N}
		SL := SL^.Next;
	end;
end;
{$ENDIF PATCH_GH}

Procedure DialogMSG(msg: string); {not const-able}
	{ Print a message in the scrolling dialog box. }
var
	Width: Integer;		{Our pixel width.}
	NextWord: String;
	THELine: String;	{The line under construction.}
	LC: Boolean;		{Loop Condition.}
	SA: SAttPtr;
{$IFDEF PATCH_I18N}
	LW_I18N: Boolean;	{Is the last word I18N ?}
	CW_I18N: Boolean;	{Is the current word I18N ?}
	DItS: Boolean;		{Do insert the space, or not.}
	SL: SAttPtr;
{$ENDIF PATCH_I18N}
begin
{$IFDEF DEBUG}
	if DEBUG_TraceMacro then begin
		ErrorMessage_fork( 'TRACE: DialogMSG() "' + msg + '"');
	end;
{$ENDIF DEBUG}
	{ CLean up the message a bit. }
	DeleteWhiteSpace( msg );
{$IFDEF GUIMSWINMODE}
	w32crt.TextColor( InfoGreen );
	w32crt.TextBackground( StdBlack );
{$ELSE GUIMSWINMODE}
	TextColor( InfoGreen );
	TextBackground( StdBlack );
{$ENDIF GUIMSWINMODE}
	msg := '> ' + msg;

	{Clear the message area, and set clipping bounds.}
	ClipZone( ZONE_Dialog );

	{Set initial cursor position.}
{$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( 1 , ScreenZone[ZONE_Dialog,4] - ScreenZone[ZONE_Dialog,2] + 1 );
{$ELSE GUIMSWINMODE}
	GotoXY( 1 , ScreenZone[ZONE_Dialog,4] - ScreenZone[ZONE_Dialog,2] + 1 );
{$ENDIF GUIMSWINMODE}

	{Calculate the width of the text area.}
	Width := ScreenZone[ZONE_Dialog,3] - ScreenZone[ZONE_Dialog,1];

	{THELine = The first word in this iteration}
{$IFDEF PATCH_I18N}
	LW_I18N := False;
	THELine := ExtractWord( msg, DItS, CW_I18N );
{$ELSE PATCH_I18N}
	THELine := ExtractWord( msg );
{$ENDIF PATCH_I18N}

	{Start the main processing loop.}
	while TheLine <> '' do begin
		{Set the LoopCondition to True.}
		LC := True;

		{ Start building the line. }
		repeat
{$IFDEF PATCH_I18N}
			NextWord := ExtractWord( Msg, DItS, CW_I18N );

			if (False = LW_I18N) and (False = CW_I18N) then begin
				DItS := True;
			end;
			LW_I18N := CW_I18N;
			if DItS then begin
				if WidthMBcharStr(THEline + ' ') <= Width then begin
					THEline := THEline + ' ';
				end;
			end;
			if WidthMBcharStr(THEline + NextWord) <= Width then begin
				THEline := THEline + NextWord;
			end else
				LC := False;
{$ELSE PATCH_I18N}
			NextWord := ExtractWord( Msg );

			if Length(THEline + ' ' + NextWord) < Width then
				THEline := THEline + ' ' + NextWord
			else
				LC := False;
{$ENDIF PATCH_I18N}

		until (not LC) or (NextWord = '') or ( TheLine[Length(TheLine)] = #13 );

		{ If the line ended due to a line break, deal with it. }
		if ( TheLine[Length(TheLine)] = #13 ) then begin
			{ Display the line break as a space. }
			TheLine[Length(TheLine)] := ' ';
{$IFDEF PATCH_I18N}
			NextWord := ExtractWord( msg, DItS, CW_I18N );
{$ELSE PATCH_I18N}
			NextWord := ExtractWord( msg );
{$ENDIF PATCH_I18N}
		end;

		{ Output the line. }
		if TheLine <> '' then begin
{$IFDEF PATCH_I18N}
{$ELSE PATCH_I18N}
			writeln;
			write( TheLine );
{$ENDIF PATCH_I18N}
			if NumSAtts( Console_History ) >= Console_History_Length then begin
				SA := Console_History;
				RemoveSAtt( Console_History , SA );
{$IFDEF PATCH_GH}
				PurgeSAtt( Console_History );
{$ENDIF PATCH_GH}
			end;
			StoreSAtt( Console_History , TheLine );
		end;

		{ Prepare for the next iteration. }
		TheLine := NextWord;

	end; { while msg <> '' }

{$IFDEF PATCH_I18N}
	{ NOTE: In CJK, there are many charctors, one charactor have double size for one ANK charactor and data length is 3 or 4 bytes. }
	{ NOTE: But, a function Writeln() fail scrolling these charactors. }
	{ NOTE: FPC's unicode functions is not stable, is its? }

	ClrZone( ZONE_Dialog );
	MaxClipZone;

	{ Restore the console display. }
  {$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( ScreenZone[ ZONE_Dialog , 1 ] , ScreenZone[ ZONE_Dialog , 2 ] -1 );
	w32crt.TextColor( Green );
  {$ELSE GUIMSWINMODE}
	GotoXY( ScreenZone[ ZONE_Dialog , 1 ] , ScreenZone[ ZONE_Dialog , 2 ] -1 );
	TextColor( Green );
  {$ENDIF GUIMSWINMODE}
	SL := RetrieveSAtt( Console_History , NumSAtts( Console_History ) - ScreenRows + ScreenZone[ ZONE_Dialog , 2 ] );
	if SL = Nil then SL := Console_History;
	while SL <> Nil do begin
  {$IFDEF GUIMSWINMODE}
		conoutput.ConWriteLn;
  {$ELSE GUIMSWINMODE}
		writeln;
  {$ENDIF GUIMSWINMODE}
		WriteMBCharStr( SL^.Info, Width );
		SL := SL^.Next;
	end;
{$ELSE PATCH_I18N}
	{Restore the clip window to its maximum size.}
	MaxClipZone;
{$ENDIF PATCH_I18N}

end;

{$IFDEF PATCH_GH}
Function GetStringFromUser( const Prompt: String ): String;
begin
	GetStringFromUser := GetStringFromUser( Prompt, '' );
end;

Function GetStringFromUser( const Prompt, Init_text: String ): String;
{$ELSE PATCH_GH}
Function GetStringFromUser( const Prompt: String ): String;
{$ENDIF PATCH_GH}
	{ Does what it says. }
{$IFDEF GUIMSWINMODE}
const
	WCLen = 288; { 254; }
{$ENDIF GUIMSWINMODE}
var
	it: String;
{$IFDEF PATCH_I18N}
  {$IFDEF GUIMSWINMODE}
	pmsg: PChar;
	pdst: PChar;
	dstlen: Integer;
	dst: String;
  {$ELSE GUIMSWINMODE}
	RK: Char;
	state: ShortInt = 0;
	mbchar_work: String = '';
  {$ENDIF GUIMSWINMODE}
	MaxInputWidth: Integer = 0;
  {$IFDEF PATCH_GH}
	trimedlength: integer;
	getit: Char;
  {$ENDIF PATCH_GH}
	X: Integer;
{$ELSE PATCH_I18N}
  {$IFDEF PATCH_GH}
	MaxInputWidth: Integer = 0;
	X: Integer;
  {$ENDIF PATCH_GH}
{$ENDIF PATCH_I18N}
begin
	DrawZoneBorder( ScreenZone[ ZONE_TextInput , 1 ] - 1 , ScreenZone[ ZONE_TextInput , 2 ] -1 , ScreenZone[ ZONE_TextInput , 3 ] + 1 , ScreenZone[ ZONE_TextInput , 4 ] + 1 , LightCyan );
	ClrZone( ZONE_TextInput );
{$IFDEF PATCH_I18N}
	X := ( ScreenZone[ZONE_TextInput,3] + ScreenZone[ZONE_TextInput,1] ) div 2 - ( WidthMBcharStr( Prompt ) div 2 ) + 1;
	if X < GOTOXY_MIN then begin
		X := GOTOXY_MIN;
	end;
  {$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( X, ScreenZone[ ZONE_TextInput , 4 ] );
	w32crt.TextColor( InfoGreen );
	WriteMBCharStr( Prompt, 0 );
	w32crt.TextColor( InfoHilight );
	w32crt.CursorOn;
  {$ELSE GUIMSWINMODE}
	GotoXY( X, ScreenZone[ ZONE_TextInput , 4 ] );
	TextColor( InfoGreen );
	WriteMBCharStr( Prompt, 0 );
	TextColor( InfoHilight );
	CursorOn;
  {$ENDIF GUIMSWINMODE}
{$ELSE PATCH_I18N}
  {$IFDEF PATCH_GH}
	X := ( ScreenZone[ZONE_TextInput,3] + ScreenZone[ZONE_TextInput,1] ) div 2 - ( WidthMBcharStr( Prompt ) div 2 ) + 1;
	if X < GOTOXY_MIN then begin
		X := GOTOXY_MIN;
	end;
	GotoXY( X, ScreenZone[ ZONE_TextInput , 4 ] );
  {$ELSE PATCH_GH}
	GotoXY( ( ScreenZone[ZONE_TextInput,3] + ScreenZone[ZONE_TextInput,1] ) div 2 - ( Length( Prompt ) div 2 ) + 1 , ScreenZone[ ZONE_TextInput , 4 ] );
  {$ENDIF PATCH_GH}
	TextColor( InfoGreen );
	Write( Prompt );
	TextColor( InfoHilight );
	CursorOn;
{$ENDIF PATCH_I18N}
	ClipZone( ZONE_TextInput );

{$IFDEF PATCH_GH}
	MaxInputWidth := ScreenZone[ ZONE_TextInput , 3 ] - ScreenZone[ ZONE_TextInput , 1 ];
	if 127 < MaxInputWidth then MaxInputWidth := 127;

  {$IFDEF PATCH_I18N}
	trimedlength := MBCharTrimedLength( Init_text, MaxInputWidth );
	if (0 < trimedlength) then begin
		it := Copy(Init_text,1,trimedlength);
	end else begin
		it := '';
	end;
  {$ELSE PATCH_I18N}
	it := Copy(Init_text,1,MaxInputWidth);
  {$ENDIF PATCH_I18N}

  {$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( 1 , 1 );
	w32crt.ClrEOL;

	pdst := StrAlloc(WCLen);
	pmsg := QuickPCopy(it);
	dstlen := Conv_ToTenc( pmsg, Length(it), pdst, WCLen );
	dst := pdst;
	it := w32crt.W32GetLine( dst );
	Dispose( pmsg );
	pmsg := QuickPCopy(it);
	dstlen := Conv_FromTenc( pmsg, Length(it), pdst, WCLen );
	it := pdst;
	Dispose( pmsg );
	Dispose( pdst );
  {$ELSE GUIMSWINMODE}
    {$IFDEF PATCH_I18N}
	GotoXY( 1 , 1 );
	ClrEOL;
	WriteMBCharStr( it, MaxInputWidth );

	repeat
		if TERMINAL_bidiRTL then begin
			GotoXY( 1, 1 );
		end else begin
			GotoXY( 1 + WidthMBcharStr(it) , 1 );
		end;
		repeat
			RK := ReadKey;
{$IFDEF DEBUG}
			WriteLn(IntToHex(Ord(RK),2));
{$ENDIF DEBUG}
			{ 45 5B 1B: 5 in TenKey on Unix-Console }
			if #0 = RK then begin
				{We have a two-part special key.}
				getit := ReadKey;
{$IFDEF DEBUG}
				WriteLn(IntToHex(Ord(getit),2));
{$ENDIF DEBUG}
				case getit of
					#$52: RK := '0';	{ 0 in TenKey }
					#$53: RK := '.';	{ . in TenKey on X-Window/MS-Windows }
					#$5A: RK := '1';	{ End Cursor Key on X-Window }
					#$4F: RK := '1';	{ End Cursor Key on Unix-Console, MS-Windows }
					#$50: RK := '2';	{ Down Cursor Key }
					#$51: RK := '3';	{ PageDown Cursor Key }
					#$4B: RK := '4';	{ Left Cursor Key }
					#$5F: RK := '5';	{ 5 in TenKey on X-Window }
					#$4C: RK := '5';	{ 5 in TenKey on MS-Windows }
					#$4D: RK := '6';	{ Right Cursor Key }
					#$5B: begin
							getit := ReadKey;
							case getit of
								#$1B: RK := '7';	{ Home Cursor Key on X-Window }
								else RK := #0;
							end;
						end;
					#$47: RK := '7';	{ Home Cursor Key on Unix-Console, MS-Windows }
					#$48: RK := '8';	{ Up Cursor Key }
					#$49: RK := '9';	{ PageUp Cursor Key }
					#$35: RK := '/';	{ Slash in TenKey on MS-Windows }
					#$1C: RK := #$0A;	{ Enter in TenKey on MS-Windows => Altanative-RET }
					{ #$3B: }{ F1  on Unix-Console, MS-Windows }
					{ #$3C: }{ F2  on Unix-Console, MS-Windows }
					{ #$3D: }{ F3  on Unix-Console, MS-Windows }
					{ #$3E: }{ F4  on Unix-Console, MS-Windows }
					{ #$3F: }{ F5  on Unix-Console, MS-Windows }
					{ #$40: }{ F6  }
					{ #$41: }{ F7  }
					{ #$42: }{ F8  }
					{ #$43: }{ F9  }
					{ #$44: }{ F10 }
					{ #$85: }{ F11 }
					{ #$86: }{ F12 }
					#$29: RK := #$1B;	{ JP109key-Hankaku/Zenkaku on MS-Windows => ESC }
					else RK := #0;
				end;
			end;
			RK := EditMBCharStr( it, 127, MaxInputWidth, RK, NIL, state, mbchar_work );
		until not(RK = #255);
		GotoXY( 1 , 1 );
		ClrEOL;
		WriteMBCharStr( it, MaxInputWidth );
	until (RK = #10) or (RK = #13) or (RK = #27);
	if (#27 = RK) then begin
		it := '';
	end;
    {$ELSE PATCH_I18N}
	GotoXY( 1 , 1 );
	ClrEOL;
	Write( it );
	{ ***BUG*** }
	GotoXY( 1 , 1 );
	ReadLn( it );
    {$ENDIF PATCH_I18N}
  {$ENDIF GUIMSWINMODE}
{$ELSE PATCH_GH}
  {$IFDEF PATCH_I18N}
    {$IFDEF GUIMSWINMODE}
	w32crt.GotoXY( 1 , 1 );
	it := w32crt.W32GetLine;
    {$ELSE GUIMSWINMODE}
	it := '';
	MaxInputWidth := ScreenZone[ ZONE_TextInput , 3 ] - ScreenZone[ ZONE_TextInput , 1 ];
	if 127 < MaxInputWidth then MaxInputWidth := 127;
	repeat
		if TERMINAL_bidiRTL then begin
			GotoXY( 1, 1 );
		end else begin
			GotoXY( 1 + WidthMBcharStr(it) , 1 );
		end;
		repeat
			RK := ReadKey;
			RK := EditMBCharStr( it, 127, MaxInputWidth, RK, NIL, state, mbchar_work );
		until not(RK = #255);
		GotoXY( 1 , 1 );
		ClrEOL;
		WriteMBCharStr( it, MaxInputWidth );
	until (RK = #10) or (RK = #13) or (RK = #27);
    {$ENDIF GUIMSWINMODE}
  {$ELSE PATCH_I18N}
	GotoXY( 1 , 1 );
	ReadLn( it );
  {$ENDIF PATCH_I18N}
{$ENDIF PATCH_GH}

{$IFDEF GUIMSWINMODE}
	w32crt.CursorOff;
{$ELSE GUIMSWINMODE}
	CursorOff;
{$ENDIF GUIMSWINMODE}
	ClrZone( ZONE_Map );
	MaxClipZone;

	GetStringFromUser := it;
end;

Function MsgString( const MsgLabel: String ): String;
	{ Return the standard message string which has the requested }
	{ label. }
begin
	MsgString := SAttValue( Text_Messages , MsgLabel );
end;

Function MoreHighFirstLine( LList: SAttPtr ): Integer;
	{ Determine the highest possible FirstLine value. }
var
	it: Integer;
begin
{$IFDEF PATCH_GH}
	if (NIL = LList) then Exit(0);
{$ENDIF PATCH_GH}
	it := NumSAtts( LList ) - ( ScreenRows - 3 );
	if it < 1 then it := 1;
	MoreHighFirstLine := it;
end;

Procedure MoreText( LList: SAttPtr; FirstLine: Integer );
	{ Browse this text file across the majority of the screen. }
	{ Clear the screen upon exiting, though restoration of the }
	{ previous display is someone else's responsibility. }
	Procedure DisplayTextHere;
	var
		CLine: SAttPtr;	{ Current Line }
	{$IFDEF PATCH_I18N}
		trimedlength: integer;
	{$ENDIF}
	begin
		{ Error check. }
		if FirstLine < 1 then FirstLine := 1
		else if FirstLine > MoreHighFirstLine( LList ) then FirstLine := MoreHighFirstLine( LList );
	{$IFDEF GUIMSWINMODE}
		w32crt.GotoXY( 1 , 1 );
	{$ELSE GUIMSWINMODE}
		GotoXY( 1 , 1 );
	{$ENDIF GUIMSWINMODE}

		CLine := RetrieveSATt( LList , FirstLine );
	{$IFDEF GUIMSWINMODE}
		while ( w32crt.WhereY < ( ScreenRows - 1 ) ) do begin
			w32crt.ClrEOL;
	{$ELSE GUIMSWINMODE}
		while ( WhereY < ( ScreenRows - 1 ) ) do begin
			ClrEOL;
	{$ENDIF GUIMSWINMODE}
			if CLine <> Nil then begin
	{$IFDEF PATCH_I18N}
				trimedlength := MBCharTrimedLength( CLine^.Info, ScreenColumns - 2 );
				if (0 < trimedlength) then begin
					WriteMBCharStr( Copy(CLine^.Info,1,trimedlength), ScreenColumns );
				end;
	  {$IFDEF GUIMSWINMODE}
				conoutput.ConWriteLn;
	  {$ELSE GUIMSWINMODE}
				WriteLn;
	  {$ENDIF GUIMSWINMODE}
	{$ELSE PATCH_I18N}
				writeln( Copy( CLine^.Info , 1 , ScreenColumns - 2 ) );
	{$ENDIF PATCH_I18N}
				CLine := CLine^.Next;
			end else begin
	{$IFDEF GUIMSWINMODE}
				conoutput.ConWriteLn;
	{$ELSE GUIMSWINMODE}
				writeln;
	{$ENDIF GUIMSWINMODE}
			end;
		end;
	end;
var
	A: Char;
{$IFDEF PATCH_GH}
	RPM: RPGMenuPtr;
	t, t_max: Integer;
	CLine: SAttPtr;	{ Current Line }
{$ENDIF PATCH_GH}
begin
{$IFDEF GUIMSWINMODE}
	w32crt.ClrScr;
	w32crt.GotoXY( 1 , ScreenRows );
	w32crt.TextColor( LightGreen );
	w32crt.TextBackground( Black );
{$ELSE GUIMSWINMODE}
	ClrScr;
	GotoXY( 1 , ScreenROws );
	TextColor( LightGreen );
	TextBackground( Black );
{$ENDIF GUIMSWINMODE}
{$IFDEF PATCH_I18N}
	WriteMBCharStr( MsgString( 'MORETEXT_Prompt' ), ScreenColumns );
{$ELSE PATCH_I18N}
	Write( MsgString( 'MORETEXT_Prompt' ) );
{$ENDIF PATCH_I18N}

{$IFDEF PATCH_GH}
	if Show_MenuScrollbar then begin
		RPM := CreateRPGMenu( LightGray , Green , ZONE_MoreText );
		CLine := RetrieveSAtt( LList , 1 );
		t_max := NumSAtts( LList );
		for t := 1 to t_max do begin
			AddRPGMenuItem( RPM , CLine^.Info , -1 );
			CLine := CLine^.Next;
		end;
		RPM^.TopItem := FirstLine;
		SetItemByPosition( RPM, FirstLine );
		SelectMenu( RPM );
		DisposeRPGMenu( RPM );
	end else begin
{$ENDIF PATCH_GH}
	{ Display the screen. }
{$IFDEF GUIMSWINMODE}
	w32crt.TextColor( LightGray );
{$ELSE GUIMSWINMODE}
	TextColor( LightGray );
{$ENDIF GUIMSWINMODE}
	DisplayTextHere;

	repeat
		{ Get input from user. }
		A := RPGKey;

{$IFDEF PATCH_GH}
		if A = KeyMap[ KMC_MenuUp ].KCode then begin
			A := RPK_Up;
		end else if A = KeyMap[ KMC_MenuDown ].KCode then begin
			A := RPK_Down;
		end else if A = KeyMap[ KMC_PageUp ].KCode then begin
			A := RPK_UpRight;
		end else if A = KeyMap[ KMC_PageDown ].KCode then begin
			A := RPK_DownRight;
		end else if A = KeyMap[ KMC_ScrollUp ].KCode then begin
			A := RPK_UpRight;
		end else if A = KeyMap[ KMC_ScrollDown ].KCode then begin
			A := RPK_DownRight;
		end;

		{ Possibly process this input. }
		if A = RPK_Down then begin
			Inc( FirstLine );
			DisplayTextHere;
		end else if A = RPK_Up then begin
			Dec( FirstLine );
			DisplayTextHere;
		end else if A = RPK_UpRight then begin
			FirstLine := FirstLine - ScreenRows;
			DisplayTextHere;
		end else if A = RPK_DownRight then begin
			FirstLine := FirstLine + ScreenRows;
			DisplayTextHere;
		end;
{$ELSE PATCH_GH}
		{ Possibly process this input. }
		if A = KeyMap[ KMC_South ].KCode then begin
			Inc( FirstLine );
			DisplayTextHere;
		end else if A = KeyMap[ KMC_North ].KCode then begin
			Dec( FirstLine );
			DisplayTextHere;
		end;
{$ENDIF PATCH_GH}

	until ( A = #27 ) or ( A = 'Q' );
{$IFDEF PATCH_GH}
	end;
{$ENDIF PATCH_GH}

	{ CLear the display area. }
{$IFDEF GUIMSWINMODE}
	w32crt.ClrScr;
{$ELSE GUIMSWINMODE}
	ClrScr;
{$ENDIF GUIMSWINMODE}
end;

{$IFDEF PATCH_GH}
Procedure SetupHQDisplay;
	{ CLear the screen & draw boxes. }
begin
  {$IFDEF GUIMSWINMODE}
	w32crt.ClrScr;
  {$ELSE GUIMSWINMODE}
	ClrScr;
  {$ENDIF GUIMSWINMODE}
	RedrawConsole;
end;
{$ENDIF PATCH_GH}



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: context.pp');
{$ENDIF DEBUG}
	Text_Messages := LoadStringList( Standard_Message_File );
	Console_History := Nil;
{$IFDEF PATCH_GH}
	Attach_SmartPointer( 'Console_History: SAttPtr', @Console_History );
{$ENDIF PATCH_GH}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: context.pp(finalization)');
{$ENDIF DEBUG}
	DisposeSAtt( Text_Messages );
	DisposeSAtt( Console_History );
end;

end.
