unit grabgear;
	{ This unit has one purpose: To seek gears and stick them in }
	{ ArenaScript's grabbed_gear global variable. }
{
	GearHead: Arena, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}

interface

uses gears,locale;

Function GG_LocatePC( GB: GameBoardPtr ): GearPtr;
Function GG_LocateNPC( CID: LongInt; GB: GameBoardPtr; Source: GearPtr ): GearPtr;
Function GG_LocateItem( NID: LongInt; GB: GameBoardPtr; Source: GearPtr ): GearPtr;
Function GG_LocateFaction( FID: Integer; GB: GameBoardPtr; Scene: GearPtr ): GearPtr;
Function GG_LocateAdventure( GB: GameBoardPtr; Source: GearPtr ): GearPtr;

Function PlotMaster( Scene: GearPtr ): GearPtr;
Function StoryMaster( Scene: GearPtr ): GearPtr;

Function Attempt_Gear_Grab( const Cmd: String;var Event: String; GB: GameBoardPtr; Source: GearPtr ): Boolean;

implementation

uses
{$IFDEF DEBUG}
	sysutils,errmsg,
{$ENDIF DEBUG}
	ability,arenascript,gearutil,interact,
{$IFDEF PATCH_GH}
  {$IFDEF SDLMODE}
	sdlgfx,sdlmap
  {$ELSE SDLMODE}
	conmap,context
  {$ENDIF SDLMODE}
{$ELSE PATCH_GH}
  {$IFDEF SDLMODE}
	sdlmap
  {$ELSE}
	conmap
  {$ENDIF}
{$ENDIF PATCH_GH}
	;

Function GG_LocatePC( GB: GameBoardPtr ): GearPtr;
	{ Attempt to find the player character. If there's more than one }
	{ master on Team 1, return one of them. }
var
	Bits,PC: GearPtr;
begin
	{ Begin the search... }
	PC := Nil;

	{ We are going to cheat a little bit. }
	{ If the interaction menu has been defined, we already know the }
	{ location of the PC since it's stored in I_PC. }
{$IFDEF PATCH_GH}
	if (NIL <> IntMenu) and (NIL <> I_PC) and (GG_DisposeGear < I_PC^.G) then begin
{$ELSE PATCH_GH}
	if ( IntMenu <> Nil ) and ( I_PC <> Nil ) then begin
{$ENDIF PATCH_GH}
		PC := I_PC;
	end else if GB <> Nil then begin
		Bits := GB^.Meks;
		while ( Bits <> Nil ) and ( PC = Nil ) do begin
{$IFDEF PATCH_GH}
			if (GG_DisposeGear < Bits^.G) then begin
{$ENDIF PATCH_GH}
			if ( NAttValue( Bits^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) and IsMasterGear( Bits ) and OnTheMap( Bits ) and GearOperational( Bits ) then begin
				PC := Bits;
			end;
{$IFDEF PATCH_GH}
			end;
{$ENDIF PATCH_GH}
			Bits := Bits^.Next;
		end;
	end;

	{ If the PC can't be found on the map, search again... }
	{ This time take any Team1 master that has a pilot. }
	if PC = Nil then begin
		Bits := GB^.Meks;
		while ( Bits <> Nil ) and ( PC = Nil ) do begin
{$IFDEF PATCH_GH}
			if (GG_DisposeGear < Bits^.G) then begin
{$ENDIF PATCH_GH}
			if ( NAttValue( Bits^.NA , NAG_Location , NAS_Team ) = NAV_DefPlayerTeam ) and IsMasterGear( Bits ) and ( LocatePilot( Bits ) <> Nil ) then begin
				PC := Bits;
			end;
{$IFDEF PATCH_GH}
			end;
{$ENDIF PATCH_GH}
			Bits := Bits^.Next;
		end;
	end;

	GG_LocatePC := PC;
end;

Function GG_LocateNPC( CID: LongInt; GB: GameBoardPtr; Source: GearPtr ): GearPtr;
	{ ATtempt to find a NPC in either the mecha list or in the }
	{ adventure. Return NIL if no such NPC can be found. }
var
	NPC: GearPtr;
begin
{$IFDEF DEBUG}
	Show_DebugMessage_SCRIPT_DynamicEncounter('GG_LocateNPC()');
{$ENDIF DEBUG}

	{ Error check - no undefined searches!!! }
	if CID = 0 then Exit( Nil );

	NPC := Nil;
{$IFDEF PATCH_GH}
	if (NIL <> GB) then begin
		NPC := SeekGearByCID( GB^.Meks , CID );
	end;
	if (NIL = NPC) and (NIL <> GB^.Scene) then begin
		NPC := SeekGearByCID( FindRoot( GB^.Scene ) , CID );
	end;
	if (NIL = NPC) then begin
		NPC := SeekGearByCID( FindRoot( Source ) , CID );
	end;
	if (NIL = NPC) and (NIL <> SCRIPT_DynamicEncounter) then begin
		NPC := SeekGearByCID( SCRIPT_DynamicEncounter , CID );
	end;
{$ELSE PATCH_GH}
	if ( GB <> Nil ) then NPC := SeekGearByCID( GB^.Meks , CID );
	if ( NPC = Nil ) and ( GB^.Scene <> Nil ) then NPC := SeekGearByCID( FindRoot( GB^.Scene ) , CID );
	if NPC = Nil then NPC := SeekGearByCID( FindRoot( Source ) , CID );
	if ( NPC = Nil ) and ( SCRIPT_DynamicEncounter <> Nil ) then NPC := SeekGearByCID( SCRIPT_DynamicEncounter , CID );
{$ENDIF PATCH_GH}
	GG_LocateNPC := NPC;
end;

Function GG_LocateItem( NID: LongInt; GB: GameBoardPtr; Source: GearPtr ): GearPtr;
	{ ATtempt to find a item in either the mecha list or in the }
	{ adventure. Return NIL if no such item can be found. }
var
	Item: GearPtr;
begin
	{ Error check - no undefined searches!!! }
	if NID = 0 then Exit( Nil );

{$IFDEF PATCH_GH}
	Item := NIL;
{$ENDIF PATCH_GH}
	if GB <> Nil then begin
		Item := SeekGearByIDTag( GB^.Meks , NAG_Narrative , NAS_NID , NID );
{$IFDEF PATCH_GH}
		if (NIL = Item) then begin
			Item := SeekGearByIDTag( FindRoot( GB^.Scene ) , NAG_Narrative , NAS_NID , NID );
		end;
{$ELSE PATCH_GH}
		if Item = Nil then Item := SeekGearByIDTag( FindRoot( GB^.Scene ) , NAG_Narrative , NAS_NID , NID );
{$ENDIF PATCH_GH}
	end else begin
		Item := SeekGearByIDTag( FindRoot( Source ) , NAG_Narrative , NAS_NID , NID );
	end;
	GG_LocateItem := Item;
end;

Function GG_LocateFaction( FID: Integer; GB: GameBoardPtr; Scene: GearPtr ): GearPtr;
	{ Find a faction gear, given its ID number and all the regular }
	{ information passed around by ArenaScript procedures. }
begin
	{ Seek the faction. }
	if ( GB <> Nil ) and ( GB^.Scene <> Nil ) then begin
		GG_LocateFaction := SeekFaction( GB^.Scene , FID );

	end else if Scene <> Nil then begin
		GG_LocateFaction := SeekFaction( Scene , FID );

	end else begin
		GG_LocateFaction := Nil;

	end;
end;

Function GG_LocateAdventure( GB: GameBoardPtr; Source: GearPtr ): GearPtr;
	{ Find the adventure. }
begin
	if ( GB <> Nil ) and ( GB^.Scene <> Nil ) then begin
		GG_LocateAdventure := FindRoot( GB^.Scene );
	end else begin
		GG_LocateAdventure := FindRoot( Source );
	end;
end;

Function PlotMaster( Scene: GearPtr ): GearPtr;
	{ Given a scene gear, find the PLOT that it is based off of, }
	{ returning NIL if no such plot exists. Assuming that SCENE is }
	{ based on a plot in the first place, it must be either the }
	{ plot itself, or a descendant of the plot. }
begin
	{ Note that the master plot may have a G of GG_AbsolutelyNothing, }
	{ if a previous command in the script has set this plot to be }
	{ advanced. }
{$IFDEF PATCH_GH}
	{ Before "AdvancePlot 0" : GG_Adventure -> GG_Plot -> GG_Persona }
	{ After  "AdvancePlot 0" : GG_Adventure -> GG_AbsolutelyNothing -> GG_Persona }
	if (NIL = Scene) then exit(NIL);
	if (GG_AbsolutelyNothing = Scene^.G) or (Scene^.G <= GG_DisposeGear) then begin
  {$IFDEF DEBUG}
		if (Scene^.G <= GG_DisposeGear) then begin
			ErrorMessage_fork('ERROR: PlotMaster(): Scene is GG_DisposeGear ' + IntToHex(Int64(Scene), 16) + ', Replaced by ' + IntToHex(Int64(Current_PlotMaster), 16) );
			DialogMsg('ERROR: PlotMaster(): Scene is GG_DisposeGear ' + IntToHex(Int64(Scene), 16) + ', Replaced by ' + IntToHex(Int64(Current_PlotMaster), 16) );
		end else begin
			ErrorMessage_fork('WARNING: PlotMaster(): Scene is GG_AbsolutelyNothing ' + IntToHex(Int64(Scene), 16) + ', Replaced by ' + IntToHex(Int64(Current_PlotMaster), 16) );
		end;
  {$ENDIF DEBUG}
		while (NIL <> Scene^.Parent) and ((GG_AbsolutelyNothing = Scene^.Parent^.G) or (Scene^.Parent^.G <= GG_DisposeGear)) do begin
			Scene := Scene^.Parent;
		end;
		Scene := Current_PlotMaster;
	end else begin
		while (NIL <> Scene) and (GG_Plot <> Scene^.G) and (GG_AbsolutelyNothing <> Scene^.G) do begin
			Scene := Scene^.Parent;
		end;
	end;
{$ELSE PATCH_GH}
	while ( Scene <> Nil ) and (Scene^.G <> GG_Plot ) and ( Scene^.G <> GG_AbsolutelyNothing ) do Scene := Scene^.Parent;
{$ENDIF PATCH_GH}
	PlotMaster := Scene;
end;

Function StoryMaster( Scene: GearPtr ): GearPtr;
	{ Given a source gear, find the STORY that it is based off of, }
	{ returning NIL if no such story exists. }
begin
	{ Note that the master plot may have a G of GG_AbsolutelyNothing, }
	{ if a previous command in the script has set this plot to be }
	{ advanced. }
{$IFDEF PATCH_GH}
	if (NIL = Scene) then exit(NIL);
	if (GG_AbsolutelyNothing = Scene^.G) or (Scene^.G <= GG_DisposeGear) then begin
  {$IFDEF DEBUG}
		if (Scene^.G <= GG_DisposeGear) then begin
			ErrorMessage_fork('ERROR: StoryMaster(): Scene is GG_DisposeGear ' + IntToHex(Int64(Scene), 16) + ', Replaced by ' + IntToHex(Int64(Current_StoryMaster), 16) );
			DialogMsg('ERROR: StoryMaster(): Scene is GG_DisposeGear ' + IntToHex(Int64(Scene), 16) + ', Replaced by ' + IntToHex(Int64(Current_StoryMaster), 16) );
		end else begin
			ErrorMessage_fork('WARNING: StoryMaster(): Scene is GG_AbsolutelyNothing ' + IntToHex(Int64(Scene), 16) + ', Replaced by ' + IntToHex(Int64(Current_StoryMaster), 16) );
		end;
  {$ENDIF DEBUG}
		while (NIL <> Scene^.Parent) and ((GG_AbsolutelyNothing = Scene^.Parent^.G) or (Scene^.Parent^.G <= GG_DisposeGear)) do begin
			Scene := Scene^.Parent;
		end;
		Scene := Current_StoryMaster;
	end else begin
		while (NIL <> Scene) and (GG_Story <> Scene^.G) and (GG_AbsolutelyNothing <> Scene^.G) do begin
			Scene := Scene^.Parent;
		end;
	end;
{$ELSE PATCH_GH}
	while ( Scene <> Nil ) and (Scene^.G <> GG_Story ) and ( Scene^.G <> GG_AbsolutelyNothing ) do Scene := Scene^.Parent;
{$ENDIF PATCH_GH}
	StoryMaster := Scene;
end;

Function Attempt_Gear_Grab( const Cmd : String; var Event: String; GB: GameBoardPtr; Source: GearPtr ): Boolean;
	{ See whether or not CMD refers to a valid Gear-Grabbing command. }
	{ CMD is assumed to be already uppercase. }
	{ If CMD is not a gear-grabbing command, no changes are made. }
	{ Return TRUE if a gear was grabbed, FALSE otherwise. }
var
	it: Boolean;
	X: LongInt;
begin
	{ Assume this is a gear-grabbing command, for now. }
	it := True;

	if CMD = 'GRABSOURCE' then begin
		Grabbed_Gear := Source;

	end else if CMD = 'GRABADVENTURE' then begin
		Grabbed_Gear := GG_LocateAdventure( GB , Source );

	end else if CMD = 'GRABDYNAMIC' then begin
		{ Grab the dynamic scene currently under construction. }
{$IFDEF DEBUG}
		Show_DebugMessage_SCRIPT_DynamicEncounter('Attempt_Gear_Grab()');
{$ENDIF DEBUG}
		Grabbed_Gear := SCRIPT_DynamicEncounter;

	end else if ( CMD = 'GRABCURRENTSCENE' ) and ( GB <> Nil ) then begin
		Grabbed_Gear := GB^.Scene;

	end else if CMD = 'GRABFACTION' then begin
		X := ScriptValue( Event , GB , Source );
		Grabbed_Gear := GG_LocateFaction( X , GB , Source );

	end else if ( CMD = 'GRABSCENE' ) and ( GB <> Nil ) then begin
		X := ScriptValue( Event , GB , Source );
		Grabbed_Gear := FindActualScene( GB , X );

	end else if CMD = 'GRABNPC' then begin
		X := ScriptValue( Event , GB , Source );
		Grabbed_Gear := GG_LocateNPC( X , GB , Source );

	end else if CMD = 'GRABLOCAL' then begin
		X := ScriptValue( Event , GB , Source );
		Grabbed_Gear :=  LocateMekByUID( gb , X );

	end else if CMD = 'GRABTEAM' then begin
		X := ScriptValue( Event , GB , Source );
		Grabbed_Gear :=  LocateTeam( gb , X );

	end else if CMD = 'GRABITEM' then begin
		X := ScriptValue( Event , GB , Source );
		Grabbed_Gear := GG_LocateItem( X , GB , Source );

{$IFDEF PATCH_GH}
	end else if ( CMD = 'GRABCHATNPC' ) then begin
		Grabbed_Gear := I_NPC;
{$ELSE PATCH_GH}
	end else if ( CMD = 'GRABCHATNPC' ) and ( IntMenu <> Nil ) then begin
		Grabbed_Gear := I_NPC;
{$ENDIF PATCH_GH}

	end else if ( CMD = 'GRABPC' ) and ( GB <> Nil ) then begin
		Grabbed_Gear := GG_LocatePC( GB );

	end else if ( CMD = 'GRABPCPILOT' ) and ( GB <> Nil ) then begin
		Grabbed_Gear := LocatePilot( GG_LocatePC( GB ) );

	end else if ( CMD = 'GRABPLOT' ) and ( Source <> Nil ) then begin
		Grabbed_Gear := PlotMaster( Source );

	end else if ( CMD = 'GRABSTORY' ) and ( Source <> Nil ) then begin
		Grabbed_Gear := StoryMaster( Source );


	end else begin
		{ No command was found matching CMD... return False. }
		it := False;
	end;

{$IFDEF PATCH_GH}
  {$IFDEF DEBUG}
	if it then begin
		if NIL = Grabbed_Gear then begin
			Show_DebugMessage_Grabbed_Gear('TRACE: Attempt_Gear_Grab(): Grabbed_Gear is NIL.');
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1( cmd + ' (' + Event + ')' ) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2( cmd + ' (' + Event + ')' ) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3( cmd + ' (' + Event + ')' ) );
		end else if (Grabbed_Gear^.G <= GG_DisposeGear) then begin
			Show_DebugMessage_Grabbed_Gear('ERROR: Attempt_Gear_Grab(): ');
			DialogMsg('ERROR: Attempt_Gear_Grab(): Grabbed_Gear is GG_DisposeGear.');
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT1( cmd + ' (' + Event + ')' ) );
			DialogMsg( Make_ErrorMessage_ASL_CONTEXT1( cmd + ' (' + Event + ')' ) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT2( cmd + ' (' + Event + ')' ) );
			DialogMsg( Make_ErrorMessage_ASL_CONTEXT2( cmd + ' (' + Event + ')' ) );
			ErrorMessage_fork( Make_ErrorMessage_ASL_CONTEXT3( cmd + ' (' + Event + ')' ) );
			DialogMsg( Make_ErrorMessage_ASL_CONTEXT3( cmd + ' (' + Event + ')' ) );
		end else begin
			Show_DebugMessage_Grabbed_Gear('TRACE: Attempt_Gear_Grab() cmd:"' + cmd + '".');
		end;
	end;
  {$ELSE DEBUG}
	if it then begin
		if NIL = Grabbed_Gear then begin
		end else if (Grabbed_Gear^.G <= GG_DisposeGear) then begin
			DialogMsg('ERROR: Attempt_Gear_Grab(): Grabbed_Gear is GG_DisposeGear.');
			DialogMsg( Make_ErrorMessage_ASL_CONTEXT1( cmd + ' (' + Event + ')' ) );
			DialogMsg( Make_ErrorMessage_ASL_CONTEXT2( cmd + ' (' + Event + ')' ) );
			DialogMsg( Make_ErrorMessage_ASL_CONTEXT3( cmd + ' (' + Event + ')' ) );
		end else begin
		end;
	end;
  {$ENDIF DEBUG}
{$ENDIF PATCH_GH}
	Attempt_Gear_Grab := it;
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: grabgear.pp');
{$ENDIF DEBUG}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: grabgear.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
