{$IFDEF PATCH_I18N}
unit iconv;

interface

uses sysutils, libiconv;


type
	enc_type = (ENC_UNKNOWN, SINGLEBYTE, EUCJP, EUCKR, EUCCN, EUCTW, UTF8, SJIS, CP932);


const
	SENC: enc_type = ENC_UNKNOWN;
	SYSTEM_CHARSET: String = '';

{$IFDEF WITH_TENC}
	{ A conversion charset for the terminal. }
	TENC: enc_type = ENC_UNKNOWN;
	TERMINAL_CHARSET: String = '';

	TERMINAL_bidiRTL: Boolean = False;
	TERMINAL_bidiRTL_Punctuation: String = '';
	TERMINAL_bidiRTL_ConvPair1: String = '';
	TERMINAL_bidiRTL_ConvPair2: String = '';
{$ENDIF WITH_TENC}

{$IFDEF CONV_UNICODE}
	{ A conversion charset for SDL_TTF.TTF_RenderUnicode_Solid(). }
	UNICODE_CHARSET = 'UTF-16LE';
{$ENDIF CONV_UNICODE}


{$IFDEF ICONV}
var
  {$IFDEF WITH_TENC}
	{ Conversion tables for the terminal. }
	iconv_enc2tenc: iconv_t;
	iconv_tenc2enc: iconv_t;
  {$ENDIF WITH_TENC}
  {$IFDEF CONV_UNICODE}
	{ Conversion tables for SDL_TTF.TTF_RenderUnicode_Solid(). }
	iconv_enc2utf16: iconv_t;
	iconv_utf16toenc: iconv_t;
  {$ENDIF CONV_UNICODE}
{$ENDIF ICONV}



implementation

uses
{$IFDEF PATCH_GH}
	errmsg,
{$ELSE PATCH_GH}
  {$IFDEF DEBUG}
	errmsg,
  {$ENDIF DEBUG}
{$ENDIF PATCH_GH}
	i18nmsg;


Function Parse_EncType( const encoding: String; var charset: String ): enc_type;
var
	P1, P2: Integer;
	codec: String;
	encoding_order: Boolean = False;
begin
	Parse_EncType := ENC_UNKNOWN;

	P1 := Pos( '.', encoding );
	if 0 < P1 then begin
		charset := UpCase(Copy( encoding, P1+1, Length(encoding) -P1 ));
	end else begin
		charset := UpCase(encoding);
	end;

	P2 := Pos('-', charset);
	if 0 < P2 then begin
		codec := Copy( charset, 1, P2-1 );
		if ('SINGLEBYTE' = codec) then begin
			Parse_EncType := SINGLEBYTE;
			encoding_order := True;
		end else if ('MULTIBYTE' = codec) then begin
			Parse_EncType := ENC_UNKNOWN;
			encoding_order := True;
		end else if ('2BYTE' = codec) then begin
			Parse_EncType := ENC_UNKNOWN;
			encoding_order := True;
		end;
		if encoding_order then begin
			charset := Copy( charset, P2+1, Length(charset) -P2 );
		end;
	end;
	if (0 = Length(charset)) or ('SINGLEBYTE' = charset) then begin
		charset := 'ISO8859-1';
		Parse_EncType := SINGLEBYTE;
	end else if ('EUCJP' = charset) or ('EUC-JP' = charset) then begin
		Parse_EncType := EUCJP;
	end else if ('EUCKR' = charset) or ('EUC-KR' = charset) then begin
		Parse_EncType := EUCKR;
	end else if ('EUCCN' = charset) or ('EUC-CN' = charset) then begin
		Parse_EncType := EUCCN;
	end else if ('EUCTW' = charset) or ('EUC-TW' = charset) then begin
		Parse_EncType := EUCTW;
	end else if ('UTF-8' = charset) then begin
		Parse_EncType := UTF8;
	end else if ('SJIS' = charset) or ('SHIFT-JIS' = charset) or ('SHIFT_JIS' = charset) then begin
		Parse_EncType := SJIS;
	end else if ('CP932' = charset) or ('MS932' = charset) then begin
		Parse_EncType := CP932;
	end else begin
		Parse_EncType := SINGLEBYTE;
	end;
end;



Procedure Get_senc();
var
	codec: String;
begin
	codec := I18N_Settings('SYSTEM_ENCODING', '');
	SENC := Parse_EncType( codec, SYSTEM_CHARSET );
end;


Procedure Get_tenc();
{$IFDEF WITH_TENC}
var
	codec: String;
	bidiRTL: String;
begin
	codec := '';
	if '' = codec then begin
		codec := GetEnvironmentVariable('GEARHEAD_LANG');
	end;
	if '' = codec then begin
		codec := GetEnvironmentVariable('LC_ALL');
	end;
	if '' = codec then begin
		codec := GetEnvironmentVariable('LC_MESSAGES');
	end;
	if '' = codec then begin
		codec := GetEnvironmentVariable('LOCALE');
	end;
	if '' = codec then begin
		codec := GetEnvironmentVariable('LANGUAGE');
	end;
	if '' = codec then begin
		codec := GetEnvironmentVariable('LANG');
	end;
  {$IFDEF Windows}
	if '' = codec then begin
		codec := I18N_Settings('TERMINAL_ENCODING_DEFAULT_MSWIN','');
	end;
  {$ENDIF Windows}
	if '' = codec then begin
		codec := I18N_Settings('TERMINAL_ENCODING_DEFAULT','');
	end;
	if '' = codec then begin
		codec := I18N_Settings('SYSTEM_ENCODING','');
	end;

	TENC := Parse_EncType( codec, TERMINAL_CHARSET );

	bidiRTL := I18N_Settings('TERMINAL_ENCODING_CONV_bidiRTL','');
	case bidiRTL[1] of
	'T':	TERMINAL_bidiRTL := True;
	'F':	TERMINAL_bidiRTL := False;
	end;
	TERMINAL_bidiRTL_Punctuation := ' '  + I18N_Settings('bidiRTL_CONVERT_PUNCTUATION','');
	TERMINAL_bidiRTL_ConvPair1   := '< ' + I18N_Settings('bidiRTL_CONVERT_CHAR_PAIR1','');
	TERMINAL_bidiRTL_ConvPair2   := '> ' + I18N_Settings('bidiRTL_CONVERT_CHAR_PAIR2','');
{$ELSE WITH_TENC}
begin
{$ENDIF WITH_TENC}
end;



Procedure Get_enc();
var
	err: Boolean = False;
begin
	Get_senc();
	Get_tenc();

	if ENC_UNKNOWN = SENC then begin
{$IFDEF PATCH_GH}
		ErrorMessage('Unknown locale "' + SYSTEM_CHARSET + '".');
{$ELSE PATCH_GH}
		WriteLn('Unknown locale "' + SYSTEM_CHARSET + '".');
{$ENDIF PATCH_GH}
		err := True;
	end;

{$IFDEF WITH_TENC}
	if ENC_UNKNOWN = TENC then begin
  {$IFDEF PATCH_GH}
		ErrorMessage('Unknown locale "' + TERMINAL_CHARSET + '".');
  {$ELSE PATCH_GH}
		WriteLn('Unknown locale "' + TERMINAL_CHARSET + '".');
  {$ENDIF PATCH_GH}
		err := True;
	end;
{$ENDIF WITH_TENC}

	if err then begin
		halt(255);
	end;
end;



Procedure Init_tenc();
{$IFDEF WITH_TENC}
  {$IFDEF ICONV}
var
	tenc_cstr: Array[0..255] of Char;
	senc_cstr: Array[0..255] of Char;
  {$ENDIF ICONV}
begin
	{ Initialize conversion tables. }
  {$IFDEF ICONV}
	StrPCopy( tenc_cstr, TERMINAL_CHARSET );
	StrPCopy( senc_cstr, SYSTEM_CHARSET );
	if (CP932 = TENC) and (EUCJP = SENC) then begin
		StrPCopy( senc_cstr, 'EUCJP-MS' );
	end;
	if (CP932 = SENC) and (EUCJP = TENC) then begin
		StrPCopy( tenc_cstr, 'EUCJP-MS' );
	end;
	iconv_enc2tenc := libiconv.iconv_open( tenc_cstr, senc_cstr );
	iconv_tenc2enc := libiconv.iconv_open( senc_cstr, tenc_cstr );
	if (iconv_t(-1) = iconv_enc2tenc) or (iconv_t(-1) = iconv_tenc2enc) then begin
    {$IFDEF PATCH_GH}
		ErrorMessage('iconv initialization failed. (system encoding "' + SYSTEM_CHARSET + '", terminal encoding "' + TERMINAL_CHARSET + '")');
    {$ELSE PATCH_GH}
		WriteLn('iconv initialization failed. (system encoding "' + SYSTEM_CHARSET + '", terminal encoding "' + TERMINAL_CHARSET + '")');
    {$ENDIF PATCH_GH}
		halt(255);
	end;
  {$ENDIF ICONV}
{$ELSE WITH_TENC}
begin
{$ENDIF WITH_TENC}
end;


Procedure Init_unicode();
{$IFDEF CONV_UNICODE}
  {$IFDEF ICONV}
var
	uenc_cstr: Array[0..255] of Char;
	senc_cstr: Array[0..255] of Char;
  {$ENDIF ICONV}
begin
	{ Initialize conversion tables. }
  {$IFDEF ICONV}
	StrPCopy( uenc_cstr, UNICODE_CHARSET );
	StrPCopy( senc_cstr, SYSTEM_CHARSET );
	iconv_enc2utf16 := libiconv.iconv_open( uenc_cstr, senc_cstr );
	iconv_utf16toenc := libiconv.iconv_open( senc_cstr, uenc_cstr );
	if (iconv_t(-1) = iconv_enc2utf16) or (iconv_t(-1) = iconv_utf16toenc) then begin
    {$IFDEF PATCH_GH}
		ErrorMessage('iconv initialization failed. (system encoding "' + SYSTEM_CHARSET + '", unicode encoding "' + UNICODE_CHARSET + '")');
    {$ELSE PATCH_GH}
		WriteLn('iconv initialization failed. (system encoding "' + SYSTEM_CHARSET + '", unicode encoding "' + UNICODE_CHARSET + '")');
    {$ENDIF PATCH_GH}
		halt(255);
	end;
  {$ENDIF ICONV}
{$ELSE CONV_UNICODE}
begin
{$ENDIF CONV_UNICODE}
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: iconv.pp');
{$ENDIF DEBUG}

{$IFDEF ENCODING_SINGLEBYTE}
	SENC := SINGLEBYTE;
	SYSTEM_CHARSET := '';
{$ENDIF}
{$IFDEF ENCODING_EUCJP}
	SENC := EUCJP;
	SYSTEM_CHARSET := 'EUCJP';
{$ENDIF}
{$IFDEF ENCODING_EUCKR}
	SENC := EUCKR;
	SYSTEM_CHARSET := 'EUCKR';
{$ENDIF}
{$IFDEF ENCODING_EUCCN}
	SENC := EUCCN;
	SYSTEM_CHARSET := 'EUCCN';
{$ENDIF}
{$IFDEF ENCODING_EUCTW}
	SENC := EUCTW;
	SYSTEM_CHARSET := 'EUCTW';
{$ENDIF}
{$IFDEF ENCODING_UTF8}
	SENC := UTF8;
	SYSTEM_CHARSET := 'UTF-8';
{$ENDIF}
{$IFDEF ENCODING_SJIS}
	SENC := SJIS;
	SYSTEM_CHARSET := 'SJIS';
{$ENDIF}
{$IFDEF ENCODING_CP932}
	SENC := CP932;
	SYSTEM_CHARSET := 'CP932';
{$ENDIF}

	Get_enc();
	Init_tenc();
	Init_unicode();
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: iconv.pp(finalization)');
{$ENDIF DEBUG}
{$IFDEF ICONV}
  {$IFDEF CONV_UNICODE}
	libiconv.iconv_close( iconv_utf16toenc );
	libiconv.iconv_close( iconv_enc2utf16 );
  {$ENDIF CONV_UNICODE}
  {$IFDEF WITH_TENC}
	libiconv.iconv_close( iconv_tenc2enc );
	libiconv.iconv_close( iconv_enc2tenc );
  {$ENDIF WITH_TENC}
{$ENDIF ICONV}
end;

end.
{$ENDIF PATCH_I18N}
