unit navigate;
	{ This unit is the flow controller for the RPG bits of the game. }
	{ It decides where the PC is, then when the PC exits a scene it }
	{ decides where to go next. }
{
	GearHead: Arena, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}

interface

uses
{$IFDEF PATCH_GH}
	gears_base,
{$ENDIF PATCH_GH}
	gears,locale;

Const
	Max_Number_Of_Plots = 40;
	Plots_Per_Generation = 5;

Procedure Navigator( Camp: CampaignPtr; Scene: GearPtr; var PCForces: GearPtr );
Procedure RestoreCampaign;

implementation

uses
{$IFDEF PATCH_GH}
	gzio,errmsg,
{$ELSE PATCH_GH}
  {$IFDEF DEBUG}
	errmsg,
  {$ENDIF DEBUG}
{$ENDIF PATCH_GH}
{$IFDEF PATCH_I18N}
	i18nmsg,
	texutil,
{$ENDIF PATCH_I18N}
{$IFDEF PATCH_GH}
	ui4gh,
	arenaplay,arenascript,damage,interact,gearutil,
	ghchars,ghweapon,movement,randchar,
{$ELSE PATCH_GH}
	arenaplay,arenascript,damage,interact,gearutil,
	ghchars,ghweapon,movement,randchar,ui4gh,
{$ENDIF PATCH_GH}
{$IFDEF SDLMODE}
	sdlgfx,sdlmap,sdlmenus
{$ELSE SDLMODE}
	congfx,conmap,conmenus,context
{$ENDIF SDLMODE}
{$IFDEF ENABLE_ADDRESSBOOK}
	,pcaction
{$ENDIF ENABLE_ADDRESSBOOK}
	;

{$IFDEF PATCH_GH}
var
	Load_Game_Archive_Buf: packed array [0..GZ_Archive_BufLen-1] of byte;	{ Global uses BSS instead of stack }
{$ENDIF PATCH_GH}

Function NoLivingPlayers( PList: GearPtr ): Boolean;
	{ Return TRUE if the provided list of gears contains no }
	{ living characters. Return FALSE if it contains at least }
	{ one. }
var
	it: Boolean;
begin
	{ Start by assuming TRUE, then set to FALSE if a character is found. }
	it := TRUE;

	{ Loop through all the gears in the list. }
	while PList <> Nil do begin
		if ( PList^.G = GG_Character ) and NotDestroyed( PList ) and ( NAttValue( PList^.NA , NAG_CharDescription , NAS_CharType ) = 0 ) then begin
			it := False;
		end;
		PList := PList^.Next;
	end;

	{ Return the result. }
	NoLivingPlayers := it;
end;

Procedure CampaignUpkeep( Camp: CampaignPtr );
	{ Do some gardening work on this campaign. This procedure keeps }
	{ everything in the CAMP structure shiny, fresh, and working. }
	{ - Load a new PLOT, if appropriate. }
	{ - Delete dynamic scenes. }
var
	Part,Part2: GearPtr;
begin
	{ Get rid of any dynamic scenes that have outlived their usefulness. }
	{ If a SCENE is found in the InvComs, it must be dynamic. }
	Part := Camp^.Source^.InvCom;
	while Part <> Nil do begin
		Part2 := Part^.Next;

		if Part^.G = GG_Scene then RemoveGear( Camp^.Source^.InvCom , Part );

		Part := Part2;
	end;
end;

{$IFDEF PATCH_GH}
Procedure Navigator( Camp: CampaignPtr; Scene: GearPtr; var PCForces: GearPtr; RestoreMode_arg: Boolean );
{$ELSE PATCH_GH}
Procedure Navigator( Camp: CampaignPtr; Scene: GearPtr; var PCForces: GearPtr );
{$ENDIF PATCH_GH}
	{ This is the role-playing flow controller. It decides what scene }
	{ of an adventure gear to load next. }
var
	N: Integer;
{$IFDEF PATCH_GH}
	RestoreMode: Boolean;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	RestoreMode := RestoreMode_arg;
{$ENDIF PATCH_GH}
	repeat
{$IFDEF PATCH_GH}
		if (NIL <> Scene) and (GG_DisposeGear < Scene^.G) then begin
			N := ScenePlayer( Camp , Scene , PCForces, RestoreMode );
		end;
		RestoreMode := False;
{$ELSE PATCH_GH}
		if SCene <> Nil then N := ScenePlayer( Camp , Scene , PCForces );
{$ENDIF PATCH_GH}

		{ Move to the destination scene, if appropriate. }
		if N > 0 then begin
			{ Perform upkeep on the campaign- delete dynamic scenes, }
			{ load new plots, yadda yadda yadda. }
			CampaignUpkeep( Camp );

			Scene := SeekGear( Camp^.Source , GG_Scene , N );

		{ If no destination scene was implied, check to see if there's }
		{ a dynamic scene waiting to be processed. }
{$IFDEF PATCH_GH}
		end else if (NIL <> SCRIPT_DynamicEncounter) and (GG_DisposeGear < SCRIPT_DynamicEncounter^.G) then begin
{$ELSE PATCH_GH}
		end else if SCRIPT_DynamicEncounter <> Nil then begin
{$ENDIF PATCH_GH}
{$IFDEF DEBUG}
			Show_DebugMessage_SCRIPT_DynamicEncounter('Navigator()');
{$ENDIF DEBUG}

			Scene := SCRIPT_DynamicEncounter;

			{ Stick the scene into the campaign. Normally scenes }
			{ are filed under SubComs, but in this case we'll store }
			{ it as an InvCom so we'll remember to delete it later. }
			InsertInvCom( Camp^.Source , Scene );

			{ Set the DynamicEncounter var to Nil, since we've moved }
			{ the scene to the campaign and don't want the ArenaScript }
			{ procedures to try and modify or delete it any more. }
			SCRIPT_DynamicEncounter := Nil;

			{ Set N to >0, since we don't want the "until..." }
			{ condition to exit. }
			N := 1;
		end;

{$IFDEF PATCH_GH}
	until ( N < 1 ) or NoLivingPlayers( PCForces ) or ( NIL = Scene ) or ( Scene^.G <= GG_DisposeGear );
{$ELSE PATCH_GH}
	until ( N < 1 ) or NoLivingPlayers( PCForces ) or ( Scene = Nil );
{$ENDIF PATCH_GH}

	{ If the game is over because the PC died, do a [MORE] prompt. }
	if NoLivingPlayers( PCForces ) then begin
		EndOfGameMoreKey;
	end;
end;
{$IFDEF PATCH_GH}
Procedure Navigator( Camp: CampaignPtr; Scene: GearPtr; var PCForces: GearPtr );
begin
	Navigator( Camp, Scene, PCForces, False );
end;
{$ENDIF PATCH_GH}

{$IFDEF SDLMODE}
Procedure RCRedraw;

begin

end;
{$ENDIF}

Procedure RestoreCampaign;
	{ Select a previously saved unit from the menu. If no unit is }
	{ found, jump to the CreateNewUnit procedure above. }
{$IFDEF PATCH_GH}
	Function decompress( InFileName, OutFileName: String ): Boolean;
	var
		InFile: gzFile;
		OutFile: File;
		len: LongInt;
		written: cardinal;
		err: integer;
	begin
		InFile := gzopen( InFileName , 'r' );
		if ( NIL = InFile ) then begin
			ErrorMessage('RestoreCampaign Failed "' + InFileName + '".' );
			exit (False);
		end;

		Assign( OutFile , OutFileName );
		Rewrite( OutFile , 1 );

		while True do begin
			len := gzread( InFile , @Load_Game_Archive_Buf, GZ_Archive_BufLen );
			if ( 0 = len ) then begin
				decompress := True;
				break;
			end;
			if ( len < 0 ) then begin
				decompress := False;
				ErrorMessage('RestoreCampaign Failed : ' + gzerror( InFile , err ) );
				break;
			end;

			blockwrite( OutFile , Load_Game_Archive_Buf , len , written );
			if ( written <> len ) then begin
				decompress := False;
				ErrorMessage('RestoreCampaign Failed "' + InFileName + '".' );
				break;
			end;
		end;

		close( OutFile );
		if ( 0 <> gzclose( InFile ) ) then begin
			ErrorMessage('RestoreCampaign Failed "' + InFileName + '".' );
			exit (False);
		end;
	end;
{$ENDIF PATCH_GH}
var
	RPM: RPGMenuPtr;
	rpgname: String;	{ Campaign Name }
	Camp: CampaignPtr;
	F: Text;		{ A File }
	PC,Part,P2: GearPtr;
	DoSave: Boolean;
{$IFDEF ENABLE_ADDRESSBOOK}
	cname: String;
{$ENDIF}
{$IFDEF PATCH_GH}
	N: Integer;
	InFileName: String;
	OutFileName: String;
{$ENDIF PATCH_GH}
begin
	{ Create a menu listing all the units in the SaveGame directory. }
	RPM := CreateRPGMenu( MenuItem , MenuSelect , ZONE_Menu );
{$IFDEF PATCH_GH}
	N := BuildFileMenu( RPM , Save_Campaign_Base + Default_Search_Pattern );
	BuildFileMenu( RPM , Save_Campaign_Base + Archive_Search_Pattern , N );
{$ELSE PATCH_GH}
	BuildFileMenu( RPM , Save_Campaign_Base + Default_Search_Pattern );
{$ENDIF PATCH_GH}

	PC := Nil;

	{ If any units are found, allow the player to load one. }
	if RPM^.NumItem > 0 then begin
		RPMSortAlpha( RPM );
{$IFDEF PATCH_I18N}
		DialogMSG( I18N_MsgString('RestoreCampaign','Select campaign') );
{$ELSE PATCH_I18N}
		DialogMSG('Select campaign file to load.');
{$ENDIF PATCH_I18N}
{$IFDEF SDLMODE}
		rpgname := SelectFile( RPM , @RCRedraw );
{$ELSE}
		rpgname := SelectFile( RPM );
{$ENDIF}
		if rpgname <> '' then begin
{$IFDEF DEBUG}
			ErrorMessage_fork('RestoreCampaign: "' + rpgname + '".' );
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
  {$IFDEF PATCH_I18N}
			InFileName := Save_Game_Directory + TextEncode( rpgname );
  {$ELSE PATCH_I18N}
			InFileName := Save_Game_Directory + rpgname;
  {$ENDIF PATCH_I18N}
			if ( ( ( 3 + 1 + 4 + 3 ) <= Length(rpgname) ) and ( GZ_Archive_Suffix = Copy( rpgname , Length(rpgname) - Length(GZ_Archive_Suffix) + 1 , Length(GZ_Archive_Suffix) ) ) ) then begin
				OutFileName := Save_Game_Directory + TextEncode( Copy( rpgname , 1 , ( Pos( '.' , rpgname ) - 1 ) ) + Default_File_Ending );
				decompress( InFileName , OutFileName );
				InFileName := OutFileName;
			end;
			Assign(F, InFileName );
{$ELSE PATCH_GH}
  {$IFDEF PATCH_I18N}
			Assign(F, Save_Game_Directory + TextEncode(rpgname) );
  {$ELSE PATCH_I18N}
			Assign(F, Save_Game_Directory + rpgname );
  {$ENDIF PATCH_I18N}
{$ENDIF PATCH_GH}
			reset(F);
			Camp := ReadCampaign(F);
			Close(F);
{$IFDEF ENABLE_ADDRESSBOOK}
			cname := Copy( rpgname, 4, Length(rpgname) - 3 );
{$IFDEF PATCH_I18N}
			LoadAddressBook( Save_Campaign_AddressBook_Base + TextEncode(cname) );
{$ELSE PATCH_I18N}
			LoadAddressBook( Save_Campaign_AddressBook_Base + cname );
{$ENDIF PATCH_I18N}
{$ENDIF ENABLE_ADDRESSBOOK}
{$IFDEF PATCH_GH}
			Navigator( Camp , Camp^.GB^.Scene , PC, True );
{$ELSE PATCH_GH}
			Navigator( Camp , Camp^.GB^.Scene , PC );
{$ENDIF PATCH_GH}
			DoSave := Camp^.Source^.S <> 0;
			DisposeCampaign( Camp );
		end else begin
			DoSave := False;
		end;

	end else begin
		{ The menu was empty... print the info message. }
		DialogMsg( MsgString( 'NEWRPGCAMP_NoCamps' ) );
		DoSave := False;
	end;

	if ( PC <> Nil ) and ( DoSave or Always_Save_Character ) then begin
		if not NoLivingPlayers( PC ) then begin
			Part := PC;
			while Part <> Nil do begin
				P2 := Part^.Next;
{$IFDEF PATCH_GH}
				if (GG_DisposeGear < Part^.G) then begin
{$ELSE PATCH_GH}
{$ENDIF PATCH_GH}
				{ Lancemates don't get saved to the character file. }
				if NAttValue( Part^.NA , NAG_CharDescription , NAS_CharType ) = NAV_CTLancemate then begin
					RemoveGear( PC , Part );
				end else begin
					{ Everything else does get saved. }
					StripNAtt( Part , NAG_Visibility );
					StripNAtt( Part , NAG_EpisodeData );
					StripNAtt( Part , NAG_WeaponModifier );
					StripNAtt( Part , NAG_Action );
					StripNAtt( Part , NAG_Location );
					StripNAtt( Part , NAG_Damage );
					StripNAtt( Part , NAG_ReactionScore );
					StripNAtt( Part , NAG_FactionScore );
					StripNAtt( Part , NAG_Condition );
					StripNAtt( Part , NAG_StatusEffect );
					StripNAtt( Part , NAG_Narrative );
					SetNAtt( Part^.NA , NAG_Personal , NAS_FactionID , 0 );
				end;
{$IFDEF PATCH_GH}
				end;
{$ELSE PATCH_GH}
{$ENDIF PATCH_GH}
				Part := P2;
			end;
			SaveChar( PC );

		end;
		DisposeGear( PC );
	end;

	DisposeRPGMenu( RPM );
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: navigate.pp');
{$ENDIF DEBUG}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: navigate.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
