unit sdlmenus;
{$MODE FPC}

	{ BUGS - If SelectMenu is handed an empty menu, all heck will }
	{  break loose. This can be a particular problem for SelectFile. }
{
	GearHead: Arena, a roguelike mecha CRPG
	Copyright (C) 2005 Joseph Hewitt

	This library is free software; you can redistribute it and/or modify it
	under the terms of the GNU Lesser General Public License as published by
	the Free Software Foundation; either version 2.1 of the License, or (at
	your option) any later version.

	The full text of the LGPL can be found in license.txt.

	This library is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
	General Public License for more details. 

	You should have received a copy of the GNU Lesser General Public License
	along with this library; if not, write to the Free Software Foundation,
	Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
}

interface

uses SDL,SDL_TTF,dos,
{$IFDEF PLUSGL}
	glgfx,
{$ELSE}
	sdlgfx,
{$ENDIF}
	texutil,ui4gh;

const
	{These two constants are used to tell the SELECT procedure whether or not}
	{the user is allowed to cancel.}
	RPMNormal = 0;
	RPMNoCancel = 1;

	RPMNoCleanup = 2; {If you want the menu left on the screen after we've finished, use this.}

{$IFDEF PATCH_GH}
	IsMenuActive: Boolean = False;
{$ENDIF PATCH_GH}

type
	RPGMenuKeyPtr = ^RPGMenuKey;
	RPGMenuKey = Record
		k: Char;
		value: integer;		{The value returned when this key is pressed.}
		next: RPGMenuKeyPtr;
	end;

	RPGMenuItemPtr = ^RPGMenuItem;
	RPGMenuItem = Record
		msg: string;		{The text which appears in the menu}
{$IFDEF PATCH_GH}
		value: LongInt;		{A value, returned by SelectMenu. -1 is reserved for Cancel}
{$ELSE PATCH_GH}
		value: integer;		{A value, returned by SelectMenu. -1 is reserved for Cancel}
{$ENDIF PATCH_GH}
		desc: string;		{Pointer to the item description. If Nil, no desc.}
		next: RPGMenuItemPtr;
	end;
	RPGMenu = Record
		active: boolean;
		itemcolor,selcolor,dtexcolor: TSDL_Color;
		Menu_Zone,Desc_Zone: TSDL_Rect;
		mode: Byte;
		topitem,selectitem,numitem: integer; {fields holding info about the status of the menu.}
		FirstItem: RPGMenuItemPtr;
		FirstKey: RPGMenuKeyPtr;
	end;
	RPGMenuPtr = ^RPGMenu;



{$IFDEF PATCH_GH}
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: LongInt; const desc: string): RPGMenuItemPtr;
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: LongInt): RPGMenuItemPtr;
Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: LongInt; const desc: string): RPGMenuItemPtr;
Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: LongInt): RPGMenuItemPtr;
{$ELSE PATCH_GH}
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer; const desc: string): RPGMenuItemPtr;
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer): RPGMenuItemPtr;
{$ENDIF PATCH_GH}
{$IFDEF PATCH_CHEAT}
Function PushRPGMenuItemFront( var RPM: RPGMenuPtr; msg: string; value: LongInt; desc: string ): RPGMenuItemPtr;
Function PushRPGMenuItemFront( var RPM: RPGMenuPtr; msg: string; value: LongInt ): RPGMenuItemPtr;
{$ENDIF PATCH_CHEAT}
Procedure DisposeRPGMenuItem( var LList: RPGMenuItemPtr );
Procedure ClearMenu( RPM: RPGMenuPtr );
Procedure RemoveRPGMenuItem(RPM: RPGMenuPtr; var LMember: RPGMenuItemPtr);

Procedure AddRPGMenuKey(RPM: RPGMenuPtr; k: Char; value: Integer);
Function CreateRPGMenu(icolor,scolor: TSDL_Color; Z: TSDL_Rect): RPGMenuPtr;
Procedure AttachMenuDesc( RPM: RPGMenuPtr; Z: TSDL_Rect );

{$IFDEF PATCH_GH}
Procedure DisposeRPGMenu(var RPM_arg: RPGMenuPtr);
{$ELSE PATCH_GH}
Procedure DisposeRPGMenu(var RPM: RPGMenuPtr);
{$ENDIF PATCH_GH}
Procedure DisplayMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType );
Function RPMLocateByPosition(RPM: RPGMenuPtr; i: integer): RPGMenuItemPtr;
{$IFDEF PATCH_GH}
Function SelectMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): LongInt;
{$ELSE PATCH_GH}
Function SelectMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): integer;
{$ENDIF PATCH_GH}
Procedure RPMSortAlpha(RPM: RPGMenuPtr);
{$IFDEF PATCH_CHEAT}
Procedure RPMSortAlpha_withSubItem(RPM: RPGMenuPtr);
{$ENDIF PATCH_CHEAT}

{$IFDEF PATCH_GH}
Function SetItemByValue( RPM: RPGMenuPtr ; V: LongInt ): RPGMenuItemPtr;
{$ELSE PATCH_GH}
Function SetItemByValue( RPM: RPGMenuPtr ; V: Integer ): RPGMenuItemPtr;
{$ENDIF PATCH_GH}
Procedure SetItemByPosition( RPM: RPGMenuPtr ; N: Integer );

{$IFDEF PATCH_GH}
Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String; N: Integer ): Integer;
Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String ): Integer;
{$ELSE PATCH_GH}
Procedure BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String );
{$ENDIF PATCH_GH}
Function SelectFile( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): String;

implementation

{$IFDEF PATCH_I18N}
uses
  {$IFDEF PATCH_GH}
	errmsg,
  {$ELSE PATCH_GH}
    {$IFDEF DEBUG}
	errmsg,
    {$ENDIF DEBUG}
  {$ENDIF PATCH_GH}
  {$IFDEF PATCH_GH}
	pseudosmartpointer,
  {$ENDIF PATCH_GH}
  {$IFDEF DEBUG}
	gears,
  {$ENDIF DEBUG}
	iconv
  {$IFDEF PATCH_CHEAT}
	,menugear
  {$ENDIF PATCH_CHEAT}
	;
{$ELSE PATCH_I18N}
  {$IFDEF PATCH_GH}
uses errmsg, pseudosmartpointer
    {$IFDEF PATCH_CHEAT}
	,menugear
    {$ENDIF PATCH_CHEAT}
	;
  {$ELSE PATCH_GH}
    {$IFDEF DEBUG}
uses errmsg, gears
      {$IFDEF PATCH_CHEAT}
	,menugear
      {$ENDIF PATCH_CHEAT}
	;
    {$ELSE DEBUG}
      {$IFDEF PATCH_CHEAT}
uses menugear;
      {$ENDIF PATCH_CHEAT}
    {$ENDIF DEBUG}
  {$ENDIF PATCH_GH}
{$ENDIF PATCH_I18N}

Function LastMenuItem(MIList: RPGMenuItemPtr): RPGMenuItemPtr;
	{This procedure will find the last item in the linked list.}
begin
	{While the menu item is pointing to a next menu item, it's not the last. duh.}
	{So, move through the list until we hit a Nil pointer.}
	while MIList^.next <> Nil do begin
		{Check the next one.}
		MIList := MIList^.next;
	end;
	{We've found a MI.next = Nil. Yay! Return its address.}
	LastMenuItem := MIList;
end;

{$IFDEF PATCH_GH}
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: LongInt; const desc: string): RPGMenuItemPtr;
{$ELSE PATCH_GH}
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer; const desc: string): RPGMenuItemPtr;
{$ENDIF PATCH_GH}
	{This procedure will add an item to the RPGToolMenu.}
	{The new item will be added as the last item in the list.}
var
	it: ^RPGMenuItem;		{Here's a pointer for the item we're creating.}
	temp: RPGMenuItemPtr;
begin
	{Allocate memory for it.}
	New(it);
{$IFDEF DEBUG}
	Trace_MemoryLeak('AddRPGMenuItem() New',it);
{$ENDIF DEBUG}

	{Check to make sure that the allocation succeeded.}
	if it = Nil then begin
		exit( Nil );
	end;

	{Initialize it to the correct values.}
	it^.msg := msg;
	it^.value := value;
	it^.next := Nil;
	it^.desc := desc; 	{The desc field is assigned the value of PChar since it}
				{is assumed that we arent responsible for the allocation,}
				{disposal, or contents of this string.}

	{Locate the last item in the list, then assign "it" to it.}
	{If the list is currently empty, stick "it" in as the first item.}
	if RPM^.firstitem = Nil then begin
		RPM^.firstitem := it;
	end else begin
		temp := LastMenuItem(RPM^.FirstItem);
		temp^.next := it;		
	end;

	{Increment the NumItem field.}
	Inc(RPM^.numitem);

	AddRPGMenuItem := it;
end;

{$IFDEF PATCH_GH}
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: LongInt): RPGMenuItemPtr;
{$ELSE PATCH_GH}
Function AddRPGMenuItem(var RPM: RPGMenuPtr; const msg: string; value: integer): RPGMenuItemPtr;
{$ENDIF PATCH_GH}
	{ Just like the above, but no desc. }
begin
	AddRPGMenuItem := AddRPGMenuItem( RPM , msg , value , '' );
end;

{$IFDEF PATCH_GH}
Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: LongInt; const desc: string): RPGMenuItemPtr;
	{This procedure will add an item to the RPGToolMenu.}
	{The new item will be added as the last item in the list.}
var
	it: ^RPGMenuItem;		{Here's a pointer for the item we're creating.}
	temp: RPGMenuItemPtr;
begin
	{Allocate memory for it.}
	New(it);
{$IFDEF DEBUG}
	Trace_MemoryLeak('AddRPGMenuItem_Top() New',it);
{$ENDIF DEBUG}

	{Check to make sure that the allocation succeeded.}
	if it = Nil then begin
		exit( Nil );
	end;

	{Initialize it to the correct values.}
	it^.msg := msg;
	it^.value := value;
	it^.next := Nil;
	it^.desc := desc; 	{The desc field is assigned the value of PChar since it}
				{is assumed that we arent responsible for the allocation,}
				{disposal, or contents of this string.}

	it^.next := RPM^.firstitem;
	RPM^.firstitem := it;

	{Increment the NumItem field.}
	Inc(RPM^.numitem);

	AddRPGMenuItem_Top := it;
end;

Function AddRPGMenuItem_Top(var RPM: RPGMenuPtr; const msg: string; value: LongInt): RPGMenuItemPtr;
	{ Just like the above, but no desc. }
begin
	AddRPGMenuItem_Top := AddRPGMenuItem_Top( RPM , msg , value , '' );
end;
{$ENDIF PATCH_GH}

{$IFDEF PATCH_CHEAT}
Function PushRPGMenuItemFront( var RPM: RPGMenuPtr; msg: string; value: LongInt; desc: string ): RPGMenuItemPtr;
var
	it: ^RPGMenuItem;		{Here's a pointer for the item we're creating.}
	temp: RPGMenuItemPtr;
begin
	{Allocate memory for it.}
	New(it);
{$IFDEF DEBUG}
	Trace_MemoryLeak('PushRPGMenuItemFront() New',it);
{$ENDIF DEBUG}

	{Check to make sure that the allocation succeeded.}
	if it = Nil then begin
		{Oops... something went wrong. Better let the user know.}
  {$IFDEF PATCH_GH}
		ErrorMessage('Error: Popcorn Delta. AddRPGMenuItem messsed up.');
  {$ELSE PATCH_GH}
		writeln('Error: Popcorn Delta. AddRPGMenuItem messsed up.');
  {$ENDIF PATCH_GH}
	{	readkey;	}
		exit;
	end;

	{Initialize it to the correct values.}
	it^.msg := msg;
	it^.value := value;
	it^.next := Nil;
	it^.desc := desc; 	{The desc field is assigned the value of PChar since it}
				{is assumed that we arent responsible for the allocation,}
				{disposal, or contents of this string.}

	{Locate the last item in the list, then assign "it" to it.}
	{If the list is currently empty, stick "it" in as the first item.}
	temp := RPM^.firstitem;
	it^.next := temp;
	RPM^.firstitem := it;

	{Increment the NumItem field.}
	Inc(RPM^.numitem);

	PushRPGMenuItemFront := it;
end;

Function PushRPGMenuItemFront( var RPM: RPGMenuPtr; msg:string; value: LongInt ): RPGMenuItemPtr;
begin
	PushRPGMenuItemFront := PushRPGMenuItemFront( RPM, msg, value, '' );
end;
{$ENDIF PATCH_CHEAT}

Procedure DisposeRPGMenuItem( var LList: RPGMenuItemPtr );
	{ Get rid of this list of items. }
	{ WARNING - If you call this procedure for a menu, it will not }
	{ change the value of the NumItems field!!! This will cause }
	{ problems when trying to use the menu. Unless you know exactly }
	{ what you're doing, use the ClearMenu procedure instead. }
var
	NextItem: RPGMenuItemPtr;
begin
	while LList <> Nil do begin
		NextItem := LList^.Next;
{$IFDEF DEBUG}
		Trace_MemoryLeak('DisposeRPGMenuItem() Dispose',LList);
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
		CheckAndNIL_Pointer('DisposeRPGMenuItem() Dispose',LList,True);
{$ENDIF PATCH_GH}
{$IFDEF PATCH_GH_PARANOID_SAFER}
		LList^.msg[1]  := '@';
		LList^.value   := -32767;
		LList^.desc[1] := '@';
		LList^.Next    := RPGMenuItemPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
		Dispose( LList );
		LList := NextItem;
	end;
end;

Procedure ClearMenu( RPM: RPGMenuPtr );
	{ Deallocate all the items in this menu, and set the number of }
	{ items to 0. }
begin
	DisposeRPGMenuItem( RPM^.FirstItem );
	RPM^.NumItem := 0;
	RPM^.SelectItem := 1;
	RPM^.topitem := 1;
end;

Procedure RemoveRPGMenuItem(RPM: RPGMenuPtr; var LMember: RPGMenuItemPtr);
	{Locate and extract member LMember from list LList.}
	{Then, dispose of LMember.}
var
	a,b: RPGMenuItemPtr;
begin
	{Initialize A and B}
	B := RPM^.FirstItem;
	A := Nil;

	{Locate LMember in the list. A will thereafter be either Nil,}
	{if LMember if first in the list, or it will be equal to the}
	{element directly preceding LMember.}
	while (B <> LMember) and (B <> Nil) do begin
		A := B;
		B := B^.next;
	end;

	if B = Nil then begin
		{Major FUBAR. The member we were trying to remove can't}
		{be found in the list.}
{$IFDEF PATCH_GH}
		ErrorMessage_fork('ERROR- RemoveLink asked to remove a link that doesnt exist.');
{$ELSE PATCH_GH}
		writeln('ERROR- RemoveLink asked to remove a link that doesnt exist.');
{$ENDIF PATCH_GH}
		end
	else if A = Nil then begin
		{There's no element before the one we want to remove,}
		{i.e. it's the first one in the list.}
		RPM^.FirstItem := B^.Next;
{$IFDEF DEBUG}
		Trace_MemoryLeak('RemoveRPGMenuItem() Dispose',B);
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
		CheckAndNIL_Pointer('RemoveRPGMenuItem() Dispose',B,True);
{$ENDIF PATCH_GH}
{$IFDEF PATCH_GH_PARANOID_SAFER}
		B^.msg[1]  := '@';
		B^.value   := -32767;
		B^.desc[1] := '@';
		B^.Next    := RPGMenuItemPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
		Dispose(B);
		end
	else begin
		{We found the attribute we want to delete and have another}
		{one standing before it in line. Go to work.}
		A^.next := B^.next;
{$IFDEF DEBUG}
		Trace_MemoryLeak('RemoveRPGMenuItem() Dispose',B);
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
		CheckAndNIL_Pointer('RemoveRPGMenuItem() Dispose',B,True);
{$ENDIF PATCH_GH}
{$IFDEF PATCH_GH_PARANOID_SAFER}
		B^.msg[1]  := '@';
		B^.value   := -32767;
		B^.desc[1] := '@';
		B^.Next    := RPGMenuItemPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
		Dispose(B);
	end;

	{ Reduce the number of items in the menu. }
	Dec(RPM^.NumItem);
end;

Procedure AddRPGMenuKey(RPM: RPGMenuPtr; k: Char; value: Integer);
	{Add a dynamically defined RPGMenuKey to the menu.}
var
	it: RPGMenuKeyPtr;
begin
	New(it);
{$IFDEF DEBUG}
	Trace_MemoryLeak('AddRPGMenuKey() New',it);
{$ENDIF DEBUG}
	if it = Nil then begin
		exit;
	end;

	{Initialize the values.}
	it^.k := k;
	it^.value := value;
	it^.next := RPM^.FirstKey;
	RPM^.FirstKey := it;
end;

Function CreateRPGMenu(icolor,scolor: TSDL_Color; Z: TSDL_Rect): RPGMenuPtr;
	{This function creates a new RPGMenu record, and returns the address.}
var
	it: ^RPGMenu;			{Here's a pointer for the menu we're making.}
begin
	{Allocate memory for it.}
	New(it);
{$IFDEF DEBUG}
	Trace_MemoryLeak('CreateRPGMenu() New',it);
{$ENDIF DEBUG}

	{Check to make sure that we've actually initialized something.}
	if it = Nil then exit( Nil );

	{Initialize the elements of the record.}
	it^.itemcolor := icolor;
	it^.selcolor := scolor;
	it^.Menu_Zone := Z;
	it^.Desc_Zone.W := 0; {A width value of 0 means there is no desc window.}
	it^.Mode := RPMNormal;
	it^.FirstItem := Nil;
	it^.FirstKey := Nil;
	it^.dtexcolor := icolor;
	it^.active := False;

	{TopItem refers to the highest item on the screen display.}
	it^.topitem := 1;

	{SelectItem refers to the item currently being pointed at by the selector.}
	it^.selectitem := 1;

	{NumItem refers to the total number of items currently in the linked list.}
	it^.numitem := 0;

	{Return the address.}
	CreateRPGMenu := it;
end;

Procedure AttachMenuDesc( RPM: RPGMenuPtr; Z: TSDL_Rect );
	{ Set the area for description items to zone Z. }
begin
	RPM^.Desc_Zone := Z;
end;

{$IFDEF PATCH_GH}
Procedure DisposeRPGMenu(var RPM_arg: RPGMenuPtr);
{$ELSE PATCH_GH}
Procedure DisposeRPGMenu(var RPM: RPGMenuPtr);
{$ENDIF PATCH_GH}
	{This procedure is called when you want to get rid of the menu. It will deallocate}
	{the memory for the RPGMenu record and also for all of the linked RPGMenuItems.}
var
{$IFDEF PATCH_GH}
	RPM: RPGMenuPtr;
{$ENDIF PATCH_GH}
	c,d: RPGMenuKeyPtr;
begin
{$IFDEF PATCH_GH}
	RPM := RPM_arg;
{$ENDIF PATCH_GH}
	{Check to make sure that we've got a valid pointer here.}
	if RPM = Nil then begin
{$IFDEF PATCH_GH}
		ErrorMessage('ERROR: Joe is a Doofus. DisposeRPGMenu has been passed a null pointer.');
{$ENDIF PATCH_GH}
		exit;
	end;
{$IFDEF PATCH_GH}
	RPM_arg := NIL;
{$ENDIF PATCH_GH}

	{Save the location of the first menu item...}
	DisposeRPGMenuItem( RPM^.FirstItem );
	c := RPM^.FirstKey;
	{... then get rid of the menu record.}
{$IFDEF DEBUG}
	Trace_MemoryLeak('DisposeRPGMenu() Dispose',RPM);
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
	CheckAndNIL_Pointer('DisposeRPGMenu() Dispose',RPM,True);
{$ENDIF PATCH_GH}
{$IFDEF PATCH_GH_PARANOID_SAFER}
	RPM^.ItemColor.r := 126;
	RPM^.ItemColor.g := 126;
	RPM^.ItemColor.b := 126;
	RPM^.SelColor.r  := 126;
	RPM^.SelColor.g  := 126;
	RPM^.SelColor.b  := 126;
	RPM^.DtexColor.r := 126;
	RPM^.DtexColor.g := 126;
	RPM^.DtexColor.b := 126;
	RPM^.Menu_Zone.x := -32767;
	RPM^.Menu_Zone.y := -32767;
	RPM^.Menu_Zone.w := 32766;
	RPM^.Menu_Zone.h := 32766;
	RPM^.Desc_Zone.x := -32767;
	RPM^.Desc_Zone.y := -32767;
	RPM^.Desc_Zone.w := 32766;
	RPM^.Desc_Zone.h := 32766;
	RPM^.mode        := 126;
	RPM^.TopItem     := -32767;
	RPM^.SelectItem  := -32767;
	RPM^.NumItem     := -32767;
	RPM^.FirstItem   := RPGMenuItemPtr(-1);
	RPM^.FirstKey    := RPGMenuKeyPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
	Dispose(RPM);
	RPM := Nil;

	{Keep processing the menu items until we hit a Nil nextitem.}
	while c <> Nil do begin
		d := c^.next;
{$IFDEF DEBUG}
		Trace_MemoryLeak('DisposeRPGMenu() Dispose',c);
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
		CheckAndNIL_Pointer('DisposeRPGMenu() Dispose',c,True);
{$ENDIF PATCH_GH}
{$IFDEF PATCH_GH_PARANOID_SAFER}
		c^.k     := #126;
		c^.value := -32767;
		c^.Next  := RPGMenuKeyPtr(-1);
{$ENDIF PATCH_GH_PARANOID_SAFER}
		Dispose(c);
		c := d;
	end;
end;

Function RPMLocateByPosition(RPM: RPGMenuPtr; i: integer): RPGMenuItemPtr;
	{Locate the i'th element of the item list, then return its address.}
var
	a: RPGMenuItemPtr;	{Our pointer}
	t: integer;		{Our counter}
begin
	{Error check, first off.}
	if i > RPM^.numitem then begin
		exit( Nil );
	end;

	a := RPM^.FirstItem;
	t := 1;

	if i > 1 then begin
		for t := 2 to i do
			a := a^.next;
	end;

	RPMLocateByPosition := a;
end;

Function MenuHeight( RPM: RPGMenuPtr ): Integer;
	{ Return the height of the menu, in text rows. }
var
	MH: Integer;
begin
	MH := ( RPM^.Menu_Zone.h div TTF_FontLineSkip( game_font ) );
{$IFDEF PATCH_GH}
	if Show_MenuPage then begin
		Dec( MH );
	end;
{$ENDIF PATCH_GH}
	if MH < 1 then MH := 1;
	MenuHeight := MH;
end;

Procedure RPMRefreshDesc(RPM: RPGMenuPtr);
	{Refresh the menu description box, if appropriate.}
begin
	{Check to make sure that this menu has a description box, first off.}
	if RPM^.Desc_Zone.W > 0 then begin
{$IFDEF PATCH_I18N}
		NFVCMessage( RPMLocateByPosition(RPM,RPM^.selectitem)^.desc , RPM^.Desc_Zone , RPM^.dtexcolor );
{$ELSE PATCH_I18N}
		NFCMessage( RPMLocateByPosition(RPM,RPM^.selectitem)^.desc , RPM^.Desc_Zone , RPM^.dtexcolor );
{$ENDIF PATCH_I18N}
	end;
end;

Procedure DisplayMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType );
	{Display the menu on the screen.}
{$IFDEF PATCH_GH}
const
	MoreMark: String = '*';
	CurrentPageMark: String = '-';
{$ENDIF PATCH_GH}
var
	topitem: RPGMenuItemPtr;
	a: RPGMenuItemPtr;		{A pointer to be used while printing.}
	t: integer;
	height: integer;		{The width of the menu display.}
	NextColor: PSDL_Color;
	Item_Image: PSDL_Surface;
{$IFDEF PATCH_I18N}
	MyDestRTL: TSDL_Rect;
{$ELSE PATCH_I18N}
	Item_PText: PChar;
{$ENDIF PATCH_I18N}
	MyDest: TSDL_Rect;
	Y,DY: Integer;
{$IFDEF PATCH_GH}
	MyDestBase: TSDL_Rect;
	MenuItemStr: String;
	LastItem: Integer;
{$ENDIF PATCH_GH}
begin
	{Error check- make sure the menu has items in it.}
	if RPM^.FirstItem = Nil then Exit;
{$IFDEF PATCH_GH}
	if RPM^.NumItem < RPM^.SelectItem then begin
		RPM^.SelectItem := RPM^.NumItem;
	end else if RPM^.SelectItem < 1 then begin
		RPM^.SelectItem := 1;
	end;
	if RPM^.NumItem < RPM^.TopItem then begin
		RPM^.TopItem := RPM^.NumItem;
	end else if RPM^.TopItem < 1 then begin
		RPM^.TopItem := 1;
	end;
{$ENDIF PATCH_GH}

	{ If a redraw procedure has been specified, call it. }
	if ReDrawer <> Nil then ReDrawer;

{$IFNDEF PLUSGL}
	ClrZone( RPM^.Menu_Zone );
{$ENDIF}

	SDL_SetClipRect( Game_Screen , @RPM^.Menu_Zone );

	{Calculate the height of the menu.}
	height := MenuHeight( rpm );

	{Locate the top of the menu.}
	topitem := RPMLocateByPosition(RPM,RPM^.topitem);

{$IFDEF PATCH_JPSSDL}
	MyDest.X := RPM^.Menu_Zone.X + Pad_Left;
{$ELSE PATCH_JPSSDL}
	MyDest.X := RPM^.Menu_Zone.X;
{$ENDIF PATCH_JPSSDL}
	Y := RPM^.Menu_Zone.Y;
	DY := TTF_FontLineSkip( game_font );
{$IFDEF PATCH_JPSSDL}
	MyDest.W := RPM^.Menu_Zone.W - Pad_Left;
{$ELSE PATCH_JPSSDL}
	MyDest.W := RPM^.Menu_Zone.W;
{$ENDIF PATCH_JPSSDL}
{$IFDEF PATCH_I18N}
	MyDestRTL := MyDest;
{$ENDIF PATCH_I18N}
{$IFDEF PATCH_GH}
	MyDestBase := MyDest;
{$ENDIF PATCH_GH}

	a := topitem;
	for t := 1 to Height do begin
		MyDest.Y := Y;
		Y := Y + DY;

		{If we're at the currently selected item, highlight it.}
		if ((t + RPM^.topitem - 1) = RPM^.selectitem) and RPM^.Active then
			NextColor := @RPM^.selcolor
		else
			NextColor := @RPM^.itemcolor;

{$IFDEF PATCH_I18N}
		if (1 < Length(a^.msg)) and (#$0 = a^.msg[1]) then begin
			Item_Image := I18N_TTF_RenderText( game_font, Copy(a^.msg,2,Length(a^.msg)-1), NextColor^ );
		end else begin
			Item_Image := I18N_TTF_RenderText( game_font , a^.msg , NextColor^ );
		end;
		if TERMINAL_bidiRTL then begin
			MyDest.X := MyDestRTL.X + MyDestRTL.W - Item_Image^.W;
		end;
{$ELSE PATCH_I18N}
		Item_PText := QuickPCopy( a^.msg );
		Item_Image := TTF_RenderText_Solid( game_font , Item_PText , NextColor^ );
		Dispose( Item_PText );
{$ENDIF PATCH_I18N}
		SDL_BlitSurface( Item_Image , Nil , Game_Screen , @MyDest );
		SDL_FreeSurface( Item_Image );

{$IFDEF PATCH_GH}
		if not(SDL_Show_MenuScrollbar) and Show_MenuScrollbar then begin
			if ((1 = t) and (1 < RPM^.topitem)) or ((Height = t) and ((RPM^.topitem + Height - 1) < RPM^.numitem)) then begin
  {$IFDEF PATCH_I18N}
				Item_Image := I18N_TTF_RenderText( game_font , MoreMark , RPM^.selcolor );
				if TERMINAL_bidiRTL then begin
					MyDest.X := MyDestBase.X;
				end else begin
					MyDest.X := MyDestBase.X + MyDestBase.W - Item_Image^.W;
				end;
  {$ELSE PATCH_I18N}
				Item_PText := QuickPCopy( MoreMark );
				Item_Image := TTF_RenderText_Solid( game_font , Item_PText , RPM^.selcolor );
				Dispose( Item_PText );
				MyDest.X := MyDestBase.X + MyDestBase.W - Item_Image^.W;
  {$ENDIF PATCH_I18N}
				SDL_BlitSurface( Item_Image , NIL , Game_Screen , @MyDest );
				SDL_FreeSurface( Item_Image );
				MyDest.X := MyDestBase.X;
			end;
		end;
{$ENDIF PATCH_GH}

		a := a^.next;

		{Check to see if we've prematurely encountered the end of the list.}
		if a = Nil then break;
	end;

{$IFDEF PATCH_GH}
	if SDL_Show_MenuScrollbar then begin
		MyDestBase.X := RPM^.Menu_Zone.X;
		MyDestBase.W := RPM^.Menu_Zone.W;
		MyDestBase.Y := RPM^.Menu_Zone.Y + SDL_Scrollbar_Size;
		MyDestBase.H := RPM^.Menu_Zone.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size;

		{ Button: Up }
		MyDest.W := SDL_Scrollbar_Size - 2;
		MyDest.X := MyDestBase.X + MyDestBase.W - SDL_Scrollbar_Size + 1;
		MyDest.H := SDL_Scrollbar_Size - 2;
		MyDest.Y := RPM^.Menu_Zone.Y;
		if (1 < RPM^.topitem) then begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuSelect.R , MenuSelect.G , MenuSelect.B ) );
		end else begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuItem.R , MenuItem.G , MenuItem.B ) );
		end;

		{ Button: Down }
		MyDest.H := SDL_Scrollbar_Size - 2;
		MyDest.Y := RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H - SDL_Scrollbar_Size + 2;
		if (Height <= (RPM^.numitem - RPM^.topitem)) then begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuSelect.R , MenuSelect.G , MenuSelect.B ) );
		end else begin
			SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuItem.R , MenuItem.G , MenuItem.B ) );
		end;

		{ Bar: Silhouette }
		MyDest.W := SDL_Scrollbar_Size;
		MyDest.X := MyDestBase.X + MyDestBase.W - MyDest.W;
		MyDest.H := MyDestBase.H;
		MyDest.Y := MyDestBase.Y;
		SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuItem.R , MenuItem.G , MenuItem.B ) );

		{ Bar: Light }
		MyDest.Y := MyDestBase.Y + (((RPM^.topitem - 1) * MyDestBase.H) div RPM^.numitem);
		MyDest.H := ((Height * MyDestBase.H) div RPM^.numitem);
		if (MyDest.H < 1) then begin
			MyDest.H := 1;
		end;
		if (MyDestBase.H < MyDest.H) then begin
			MyDest.H := MyDestBase.H;
		end;
		SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , MenuSelect.R , MenuSelect.G , MenuSelect.B ) );

		{ Bar: Highlight }
		MyDest.X := MyDestBase.X + MyDestBase.W - (SDL_Scrollbar_Size div 2);
		MyDest.W := 1;
		MyDest.Y := MyDestBase.Y + (((RPM^.selectitem - 1) * MyDestBase.H) div RPM^.numitem);
		MyDest.H := (MyDestBase.H div RPM^.numitem);
		if (MyDest.H < 1) then begin
			MyDest.H := 1;
		end;
		SDL_FillRect( Game_Screen, @MyDest, SDL_MapRGB( Game_Screen^.Format , BorderBlue.R , BorderBlue.G , BorderBlue.B ) );
	end else if Show_MenuScrollbar and (Height < RPM^.numitem) then begin
		MyDest.Y := RPM^.Menu_Zone.Y + (((RPM^.selectitem - 1) * (Height * DY - DY)) div (RPM^.numitem - 1));
  {$IFDEF PATCH_I18N}
		Item_Image := I18N_TTF_RenderText( game_font , CurrentPageMark , RPM^.selcolor );
		if TERMINAL_bidiRTL then begin
			MyDest.X := MyDestBase.X;
		end else begin
			MyDest.X := MyDestBase.X + MyDestBase.W - Item_Image^.W;
		end;
  {$ELSE PATCH_I18N}
		Item_PText := QuickPCopy( CurrentPageMark );
		Item_Image := TTF_RenderText_Solid( game_font , Item_PText , RPM^.selcolor );
		Dispose( Item_PText );
		MyDest.X := MyDestBase.X + MyDestBase.W - Item_Image^.W;
  {$ENDIF PATCH_I18N}
		SDL_BlitSurface( Item_Image , NIL , Game_Screen , @MyDest );
		SDL_FreeSurface( Item_Image );
		MyDest.X := MyDestBase.X;
	end;
	if Show_MenuPage then begin
		LastItem := RPM^.topitem + Height - 1;
		if RPM^.numitem < LastItem then begin
			LastItem := RPM^.numitem;
		end;
		MenuItemStr := '(' + BStr(RPM^.topitem) + '-' + BStr(RPM^.selectitem) + '-' + BStr(LastItem) + '/' + BStr(RPM^.numitem) + ')';
		MyDest.X := MyDestBase.X;
		MyDest.Y := RPM^.Menu_Zone.Y + DY * Height;
  {$IFDEF PATCH_I18N}
		Item_Image := I18N_TTF_RenderText( game_font, MenuItemStr, RPM^.itemcolor );
		if TERMINAL_bidiRTL then begin
			MyDest.X := MyDestRTL.X + MyDestRTL.W - Item_Image^.W;
		end;
  {$ELSE PATCH_I18N}
		Item_PText := QuickPCopy( a^.msg );
		Item_Image := TTF_RenderText_Solid( game_font, MenuItemStr, @RPM^.itemcolor );
		Dispose( Item_PText );
  {$ENDIF PATCH_I18N}
		SDL_BlitSurface( Item_Image, NIL, Game_Screen, @MyDest );
		SDL_FreeSurface( Item_Image );
	end;
{$ENDIF PATCH_GH}

	{Restore the window to its regular size.}
	SDL_SetClipRect( Game_Screen , Nil );

	{If there's an associated Desc field, display it now.}
	RPMRefreshDesc(RPM);
end;

{$IFDEF PATCH_GH}
Procedure RPMReposition( RPM: RPGMenuPtr; FullScroll: Boolean );
	{The selected item has just changed, and is no longer visible on screen.}
	{Adjust the RPGMenu's topitem field to an appropriate value.}
begin
	{When this function is called, there are two possibilities: either the}
	{selector has moved off the bottom of the page or the top.}

	if RPM^.selectitem < RPM^.topitem then begin
		{The selector has moved off the bottom of the list. The new page}
		{display should start with SelectItem on the bottom.}
		if FullScroll then begin
			RPM^.topitem := RPM^.selectitem - MenuHeight( RPM ) + 1;
		end else begin
			RPM^.topitem := RPM^.selectitem;
		end;

		{Error check- if this moves topitem below 1, that's bad.}
		if RPM^.topitem < 1 then begin
			RPM^.topitem := 1;
		end;
	end else begin
		{The selector has moved off the top of the list. The new page should}
		{start with SelectItem at the top, unless this would make things look}
		{funny.}
		if FullScroll then begin
			if ((RPM^.selectitem + MenuHeight( RPM ) - 1) > RPM^.numitem) then begin
				{There will be whitespace at the bottom of the menu if we assign}
				{SelectItem to TopItem. Make TopItem equal to the effective last}
				{page.}
				RPM^.topitem := RPM^.numitem - MenuHeight( RPM ) + 1;
				if RPM^.topitem < 1 then begin
					RPM^.topitem := 1;
				end;
			end else begin
				RPM^.topitem := RPM^.selectitem;
			end;
		end else if ((RPM^.topitem + MenuHeight( RPM ) - 1) < RPM^.numitem) then begin
			Inc( RPM^.TopItem );
		end;
	end;

end;

Procedure RPMReposition( RPM: RPGMenuPtr );
begin
	{ Check a limit. }
	if (RPM^.selectitem < 1) then begin RPM^.selectitem := 1
	end else if (RPM^.numitem < RPM^.selectitem) then begin RPM^.selectitem := RPM^.numitem
	end;

	if (RPM^.selectitem < RPM^.topitem) then begin
		{ Page Up }
		RPM^.topitem := RPM^.selectitem;

		{ Check a limit. }
		if RPM^.topitem < 1 then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.topitem + MenuHeight(RPM) <= RPM^.selectitem) then begin
		{ Page Down }
		RPM^.topitem := RPM^.selectitem - MenuHeight(RPM) + 1;

		{ Check a limit. }
		if RPM^.numitem < RPM^.topitem then begin
			RPM^.topitem := RPM^.numitem;
		end;
	end;

end;
{$ELSE PATCH_GH}
Procedure RPMReposition( RPM: RPGMenuPtr; FullScroll: Boolean );
	{The selected item has just changed, and is no longer visible on screen.}
	{Adjust the RPGMenu's topitem field to an appropriate value.}
begin
	{When this function is called, there are two possibilities: either the}
	{selector has moved off the bottom of the page or the top.}

	if RPM^.selectitem < RPM^.topitem then begin
		{The selector has moved off the bottom of the list. The new page}
		{display should start with SelectItem on the bottom.}
		if FullScroll then begin
			RPM^.topitem := RPM^.selectitem - MenuHeight( RPM ) + 1;
		end else begin
			RPM^.topitem := RPM^.selectitem;
		end;

		{Error check- if this moves topitem below 1, that's bad.}
		if RPM^.topitem < 1 then
			RPM^.topitem := 1;
		end
	else begin
		{The selector has moved off the top of the list. The new page should}
		{start with SelectItem at the top, unless this would make things look}
		{funny.}
		if FullScroll then begin
			if ((RPM^.selectitem + MenuHeight( RPM ) - 1) > RPM^.numitem) then begin
				{There will be whitespace at the bottom of the menu if we assign}
				{SelectItem to TopItem. Make TopItem equal to the effective last}
				{page.}
				RPM^.topitem := RPM^.numitem - MenuHeight( RPM ) + 1;
				if RPM^.topitem < 1 then RPM^.topitem := 1;
				end
			else
				RPM^.topitem := RPM^.selectitem;
		end else if ((RPM^.topitem + MenuHeight( RPM ) - 1) < RPM^.numitem) then begin
			Inc( RPM^.TopItem );
		end;
	end;

end;
{$ENDIF PATCH_GH}

{$IFDEF PATCH_GH}
Procedure RPMReposition_center( RPM: RPGMenuPtr );
begin
	{ Check a limit. }
	if (RPM^.selectitem < 1) then begin
		RPM^.selectitem := 1
	end else if (RPM^.numitem < RPM^.selectitem) then begin
		RPM^.selectitem := RPM^.numitem
	end;

	{ Auto reposition mode. }
	if (RPM^.topitem < 1) then begin
		RPM^.topitem := RPM^.selectitem - ( MenuHeight(RPM) div 2 );
		if (RPM^.topitem < 1) then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.numitem < RPM^.topitem) then begin
		RPM^.topitem := RPM^.selectitem - MenuHeight(RPM) + 1;
		if (RPM^.topitem < 1) then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.selectitem < RPM^.topitem) then begin
		RPM^.topitem := RPM^.selectitem - MenuHeight(RPM) + 1;
		if RPM^.topitem < 1 then begin
			RPM^.topitem := 1;
		end;
	end else if (RPM^.topitem + MenuHeight(RPM) <= RPM^.selectitem) then begin
		RPM^.topitem := RPM^.selectitem;
		if RPM^.numitem < RPM^.topitem then begin
			RPM^.topitem := RPM^.numitem;
		end;
	end;

end;
{$ENDIF PATCH_GH}

Procedure RPMUpKey( RPM: RPGMenuPtr; FullScroll: Boolean );
	{Someone just pressed the UP key, and we're gonna process that input.}
	{PRECONDITIONS: RPM has been initialized properly, and is currently being}
	{  displayed on the screen.}
begin
	{Decrement the selected item by one.}
	Dec(RPM^.selectitem);

	{If this causes it to go beneath one, wrap around to the last item.}
	if RPM^.selectitem = 0 then
		RPM^.selectitem := RPM^.numitem;

	{If the movement takes the selected item off the screen, do a redisplay.}
	{Otherwise, indicate the newly selected item.}
	if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem) >= MenuHeight( RPM )) then begin
		{Determine an appropriate new value for topitem.}
		RPMReposition(RPM,FullScroll);
	end;
end;

Procedure RPMDownKey( RPM: RPGMenuPtr; FullScroll: Boolean );
	{Someone just pressed the DOWN key, and we're gonna process that input.}
	{PRECONDITIONS: RPM has been initialized properly, and is currently being}
	{  displayed on the screen.}
begin
	{Increment the selected item.}
	Inc(RPM^.selectitem);
	{If this takes the selection out of bounds, restart at the first item.}
	if RPM^.selectitem = RPM^.numitem + 1 then
		RPM^.selectitem := 1;

	{If the movement takes the selected item off the screen, do a redisplay.}
	{Otherwise, indicate the newly selected item.}
	if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem) >= MenuHeight( RPM ) ) then begin
		{Determine an appropriate new value for topitem.}
		RPMReposition(RPM,FullScroll);
	end;
end;

{$IFDEF PATCH_JPSSDL}
Procedure RPMPgUpKey( RPM: RPGMenuPtr; FullScroll: Boolean );
var
	p: Integer;
begin
	p := RPM^.selectitem - RPM^.topitem;
	RPM^.selectitem := RPM^.selectitem - MenuHeight( RPM );
	RPM^.topitem := RPM^.selectitem - p;

	if RPM^.selectitem < 1 then RPM^.selectitem := 1;
	if RPM^.topitem < 1 then RPM^.topitem := 1;
end;

Procedure RPMPgDownKey( RPM: RPGMenuPtr; FullScroll: Boolean );
var
	p: Integer;
begin
	p := RPM^.selectitem - RPM^.topitem;
	RPM^.selectitem := RPM^.selectitem + MenuHeight( RPM );
	RPM^.topitem := RPM^.selectitem - p;
	
	if RPM^.selectitem > RPM^.numitem then  RPM^.selectitem := RPM^.numitem;
	if (RPM^.numitem > MenuHeight( RPM )) then begin
		if RPM^.topitem > (RPM^.numitem - MenuHeight( RPM ) + 1) then RPM^.topitem := RPM^.numitem - MenuHeight( RPM ) + 1;
	end else RPM^.topitem := 1;
end;
{$ENDIF PATCH_JPSSDL}

{$IFDEF PATCH_GH}
Procedure RPMScrollUpKey( RPM: RPGMenuPtr; FullScroll: Boolean );
begin
	Dec(RPM^.selectitem);
	if RPM^.selectitem < 1 then
		RPM^.selectitem := 1;
	Dec(RPM^.topitem);
	if RPM^.topitem < 1 then
		RPM^.topitem := 1;
end;

Procedure RPMScrollDownKey( RPM: RPGMenuPtr; FullScroll: Boolean );
begin
	Inc(RPM^.selectitem);
	if RPM^.numitem < RPM^.selectitem then
		RPM^.selectitem := RPM^.numitem;
	Inc(RPM^.topitem);
	if (RPM^.numitem > MenuHeight( RPM )) then begin
		if RPM^.topitem > (RPM^.numitem - MenuHeight( RPM ) + 1) then RPM^.topitem := RPM^.numitem - MenuHeight( RPM ) + 1;
	end else RPM^.topitem := 1;
end;
{$ENDIF PATCH_GH}

{$IFDEF PATCH_GH}
Function SelectMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): LongInt;
{$ELSE PATCH_GH}
Function SelectMenu( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): integer;
{$ENDIF PATCH_GH}
	{This function will allow the user to browse through the menu and will}
	{return a value based upon the user's selection.}
var
	getit: char;		{Character used to store user input}
{$IFDEF PATCH_GH}
	r: LongInt;		{The value we'll be sending back.}
	I: integer;
{$ELSE PATCH_GH}
	r,I: integer;		{The value we'll be sending back.}
{$ENDIF PATCH_GH}
	m: RPGMenuKeyPtr;
	UK: Boolean;		{Has a special MenuKey been pressed?}
	OldMouseX, OldMouseY: Integer; { TUNGINOBI: I got sick of the mouse cursor getting }
                          { in the way of the keyboard, so this will only }
                          { change the menu item if the mouse has moved. }
{$IFDEF PATCH_GH}
	Scrollbar_DragMode: Boolean = False;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
	RPMReposition_center( RPM );
{$ENDIF PATCH_GH}

	{The menu is now active!}
	RPM^.Active := True;
	{Show the menu to the user.}
	DisplayMenu( RPM , ReDrawer );
	GhFlip;

	{TUNGINOBI: Initialise the mouse movement variables}
	OldMouseX := Mouse_X;
	OldMouseY := Mouse_Y;

	{Initialize UK}
	UK := False;

{$IFDEF PATCH_GH}
	IsMenuActive := True;
{$ENDIF PATCH_GH}

	{Start the loop. Remain in this loop until either the player makes a selection}
	{or cancels the menu using the ESC key.}
	repeat
		DisplayMenu(RPM,ReDrawer);
		GhFlip;

		{Read the input from the keyboard.}
		getit := RPGKey;

{$IFDEF PATCH_GH}
		if getit = KeyMap[ KMC_MenuUp ].KCode then begin
			getit := RPK_Up;
		end else if getit = KeyMap[ KMC_MenuDown ].KCode then begin
			getit := RPK_Down;
		end else if getit = KeyMap[ KMC_PageUp ].KCode then begin
			getit := RPK_UpRight;
		end else if getit = KeyMap[ KMC_PageDown ].KCode then begin
			getit := RPK_DownRight;
		end else if getit = KeyMap[ KMC_ScrollUp ].KCode then begin
			getit := RPK_UpLeft;
		end else if getit = KeyMap[ KMC_ScrollDown ].KCode then begin
			getit := RPK_DownLeft;
		end;
		if SelectMenu_UpDown_by_MouseWheel_UD_Reverse then begin
			if getit = KeyMap[ KMC_ButtonWUp ].KCode then begin
				getit := RPK_Down;
			end else if getit = KeyMap[ KMC_ButtonWDown ].KCode then begin
				getit := RPK_Up;
			end;
		end else if SelectMenu_UpDown_by_MouseWheel_UD then begin
			if getit = KeyMap[ KMC_ButtonWUp ].KCode then begin
				getit := RPK_Up;
			end else if getit = KeyMap[ KMC_ButtonWDown ].KCode then begin
				getit := RPK_Down;
			end;
		end;
		if SelectMenu_UpDown_by_MouseWheel_LR_Reverse then begin
			if getit = KeyMap[ KMC_ButtonWLeft ].KCode then begin
				getit := RPK_Down;
			end else if getit = KeyMap[ KMC_ButtonWRight ].KCode then begin
				getit := RPK_Up;
			end;
		end else if SelectMenu_UpDown_by_MouseWheel_LR then begin
			if getit = KeyMap[ KMC_ButtonWLeft ].KCode then begin
				getit := RPK_Up;
			end else if getit = KeyMap[ KMC_ButtonWRight ].KCode then begin
				getit := RPK_Down;
			end;
		end;
		if SelectMenu_Scroll_by_MouseWheel_UD_Reverse then begin
			if getit = KeyMap[ KMC_ButtonWUp ].KCode then begin
				getit := RPK_DownLeft;
			end else if getit = KeyMap[ KMC_ButtonWDown ].KCode then begin
				getit := RPK_UpLeft;
			end;
		end else if SelectMenu_Scroll_by_MouseWheel_UD then begin
			if getit = KeyMap[ KMC_ButtonWUp ].KCode then begin
				getit := RPK_UpLeft;
			end else if getit = KeyMap[ KMC_ButtonWDown ].KCode then begin
				getit := RPK_DownLeft;
			end;
		end;
		if SelectMenu_Scroll_by_MouseWheel_LR_Reverse then begin
			if getit = KeyMap[ KMC_ButtonWLeft ].KCode then begin
				getit := RPK_DownLeft;
			end else if getit = KeyMap[ KMC_ButtonWRight ].KCode then begin
				getit := RPK_UpLeft;
			end;
		end else if SelectMenu_Scroll_by_MouseWheel_LR then begin
			if getit = KeyMap[ KMC_ButtonWLeft ].KCode then begin
				getit := RPK_UpLeft;
			end else if getit = KeyMap[ KMC_ButtonWRight ].KCode then begin
				getit := RPK_DownLeft;
			end;
		end;
		if SelectMenu_ScrollPage_by_MouseWheel_UD_Reverse then begin
			if getit = KeyMap[ KMC_ButtonWUp ].KCode then begin
				getit := RPK_DownRight;
			end else if getit = KeyMap[ KMC_ButtonWDown ].KCode then begin
				getit := RPK_UpRight;
			end;
		end else if SelectMenu_ScrollPage_by_MouseWheel_UD then begin
			if getit = KeyMap[ KMC_ButtonWUp ].KCode then begin
				getit := RPK_UpRight;
			end else if getit = KeyMap[ KMC_ButtonWDown ].KCode then begin
				getit := RPK_DownRight;
			end;
		end;
		if SelectMenu_ScrollPage_by_MouseWheel_LR_Reverse then begin
			if getit = KeyMap[ KMC_ButtonWLeft ].KCode then begin
				getit := RPK_DownRight;
			end else if getit = KeyMap[ KMC_ButtonWRight ].KCode then begin
				getit := RPK_UpRight;
			end;
		end else if SelectMenu_ScrollPage_by_MouseWheel_LR then begin
			if getit = KeyMap[ KMC_ButtonWLeft ].KCode then begin
				getit := RPK_UpRight;
			end else if getit = KeyMap[ KMC_ButtonWRight ].KCode then begin
				getit := RPK_DownRight;
			end;
		end;
{$ENDIF PATCH_GH}

		{Certain keys need processing- if so, process them.}
		case getit of
			{Selection Movement Keys}
{$IFDEF PATCH_GH}
			RPK_Up:   begin RPMUpKey( RPM , True );   getit := #0; end;
			RPK_Down: begin RPMDownKey( RPM , True ); getit := #0; end;
			RPK_UpRight:   begin RPMPgUpKey( RPM , True );   getit := #0; end;
			RPK_DownRight: begin RPMPgDownKey( RPM , True ); getit := #0; end;
			RPK_UpLeft:   begin RPMScrollUpKey( RPM , False );   getit := #0; end;
			RPK_DownLeft: begin RPMScrollDownKey( RPM , False ); getit := #0; end;
			RPK_MouseMotion:
				if Mouse_Active then begin
					if Scrollbar_DragMode and SDL_Show_MenuScrollbar then begin
						if (RPM^.Menu_Zone.Y <= Mouse_Y) and (Mouse_Y <= (RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H)) then begin
							RPM^.selectitem := ((Mouse_Y - (RPM^.Menu_Zone.Y + SDL_Scrollbar_Size)) * RPM^.numitem) div (RPM^.Menu_Zone.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size) + 1;
							RPMReposition( RPM );
						end;
					end else
					{ If the mouse pointer is around }
					{ the menu, we may have to do something. }
					if ( Mouse_X >= RPM^.Menu_Zone.X ) and ( Mouse_X <= ( RPM^.Menu_Zone.X + RPM^.Menu_Zone.W ) ) and (( Mouse_X <> OldMouseX ) or ( Mouse_Y <> OldMouseY )) then begin
						if SDL_Show_MenuScrollbar and ((RPM^.Menu_Zone.X + RPM^.Menu_Zone.W - SDL_Scrollbar_Size) <= Mouse_X) then begin
							{ Region of ScrollBar }
						end else if ( Mouse_Y < ( RPM^.Menu_Zone.Y ) ) then begin
							if SelectMenu_Scroll_by_Edge and ( RPM^.SelectItem > 1 ) then RPMUpKey( RPM , False );
						end else if ( Mouse_Y > ( RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H ) ) then begin
							if SelectMenu_Scroll_by_Edge and ( (RPM^.selectitem - RPM^.topitem) < MenuHeight( RPM ) ) and ( RPM^.selectitem < RPM^.numitem ) then RPMDownKey( RPM , False );
						end else begin
							I := ( Mouse_Y - RPM^.Menu_Zone.Y ) div TTF_FontLineSkip( game_font );
							if (0 <= I) and (I < MenuHeight( RPM )) then begin
								SetItemByPosition( RPM , RPM^.TopItem + I );
							end;
							{ Upon setting an item directly, freeze the mouse. }
							OldMouseX := Mouse_X;
							OldMouseY := Mouse_Y;
						end;
					end;
				end;
			RPK_MouseButton:
				if Mouse_Active then begin
					{ If the mouse hit happened inside }
					{ the menu area, it was a selection. }
					{ Otherwise it was a cancel. }
					if ( Mouse_X >= RPM^.Menu_Zone.X ) and ( Mouse_X <= ( RPM^.Menu_Zone.X + RPM^.Menu_Zone.W )) and ( Mouse_Y >= RPM^.Menu_Zone.Y ) and ( Mouse_Y <= ( RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H )) then begin
						if SDL_Show_MenuScrollbar and ((RPM^.Menu_Zone.X + RPM^.Menu_Zone.W - SDL_Scrollbar_Size) <= Mouse_X) then begin
							{ Region of ScrollBar }
							if (Mouse_Y <= (RPM^.Menu_Zone.Y + SDL_Scrollbar_Size)) then begin
								RPMScrollUpKey( RPM , False );
							end else if ((RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H - SDL_Scrollbar_Size) <= Mouse_Y) then begin
								RPMScrollDownKey( RPM , False );
							end else if (Mouse_Y < (RPM^.Menu_Zone.Y + SDL_Scrollbar_Size + (((RPM^.topitem - 1) * (RPM^.Menu_Zone.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size)) div RPM^.numitem))) then begin
								RPMPgUpKey( RPM , True );
							end else if ((RPM^.Menu_Zone.Y + SDL_Scrollbar_Size + (((RPM^.topitem - 1 + MenuHeight(RPM)) * (RPM^.Menu_Zone.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size)) div RPM^.numitem)) < Mouse_Y) then begin
								RPMPgDownKey( RPM , True );
							end else begin
								Scrollbar_DragMode := True;
							end;
							getit := #0;
						end else begin
							getit := ' ';
						end;
					end else begin
						if RPM^.Mode <> RPMNoCancel then getit := #27
						else getit := ' ';
					end;
				end;
			RPK_MouseButtonRelease:
				if Mouse_Active then begin
					if Scrollbar_DragMode and SDL_Show_MenuScrollbar then begin
						if (RPM^.Menu_Zone.Y <= Mouse_Y) and (Mouse_Y <= (RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H)) then begin
							RPM^.selectitem := ((Mouse_Y - (RPM^.Menu_Zone.Y + SDL_Scrollbar_Size)) * RPM^.numitem) div (RPM^.Menu_Zone.H - SDL_Scrollbar_Size - SDL_Scrollbar_Size) + 1;
							RPMReposition( RPM );
						end;
					end;
					Scrollbar_DragMode := False;
					getit := #0;
				end;
{$ELSE PATCH_GH}
			RPK_Up: RPMUpKey( RPM , True );
			RPK_Down: RPMDownKey( RPM , True );
  {$IFDEF PATCH_JPSSDL}
			RPK_Left: RPMPgUpKey( RPM , True );
			RPK_Right: RPMPgDownKey( RPM , True );
  {$ENDIF PATCH_JPSSDL}
			RPK_TimeEvent:
				begin
					{ If the mouse pointer is around }
					{ the menu, we may have to do something. }
					if Mouse_Active and ( Mouse_X >= RPM^.Menu_Zone.X ) and ( Mouse_X <= ( RPM^.Menu_Zone.X + RPM^.Menu_Zone.W ) ) and (( Mouse_X <> OldMouseX ) or ( Mouse_Y <> OldMouseY )) then begin
						if ( Mouse_Y < ( RPM^.Menu_Zone.Y ) ) then begin
							if ( RPM^.SelectItem > 1 ) then RPMUpKey( RPM , False );
						end else if ( Mouse_Y > ( RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H ) ) then begin
							if ( (RPM^.selectitem - RPM^.topitem) < MenuHeight( RPM ) ) and ( RPM^.selectitem < RPM^.numitem ) then RPMDownKey( RPM , False );
						end else begin
							I := ( Mouse_Y - RPM^.Menu_Zone.Y ) div TTF_FontLineSkip( game_font );
							SetItemByPosition( RPM , RPM^.TopItem + I );
							{ Upon setting an item directly, freeze the mouse. }
							OldMouseX := Mouse_X;
							OldMouseY := Mouse_Y;
						end;
					end;
				end;
			RPK_MouseButton:
				if Mouse_Active then begin
					{ If the mouse hit happened inside }
					{ the menu area, it was a selection. }
					{ Otherwise it was a cancel. }
					if ( Mouse_X >= RPM^.Menu_Zone.X ) and ( Mouse_X <= ( RPM^.Menu_Zone.X + RPM^.Menu_Zone.W )) and ( Mouse_Y >= RPM^.Menu_Zone.Y ) and ( Mouse_Y <= ( RPM^.Menu_Zone.Y + RPM^.Menu_Zone.H )) then begin
						getit := ' ';
					end else begin
						if RPM^.Mode <> RPMNoCancel then getit := #27
						else getit := ' ';
					end;
				end;
{$ENDIF PATCH_GH}
			RPK_RightButton:
				if ( RPM^.Mode <> RPMNoCancel ) and Mouse_Active then getit := #27;

			{If we receive an ESC, better check to make sure we're in a}
			{cancelable menu. If not, convert the ESC to an unused key.}
			#27: If RPM^.Mode = RPMNoCancel then getit := 'Q';
			{ If we get a backspace, conver that to ESC. }
			#8: If RPM^.Mode <> RPMNoCancel then getit := #27;
			{ Convert enter to space. }
			#13,#10: getit := ' ';
		end;


		{Check to see if a special MENU KEY has been pressed.}
		if RPM^.FirstKey <> Nil then begin
			m := RPM^.FirstKey;
			while m <> Nil do begin
				if getit = m^.k then begin
					UK := True;
					r := m^.value;
				end;
				m := m^.next;
			end;
		end;

	{Check for a SPACE or ESC.}
	until (getit = ' ') or (getit = #27) or UK;

{$IFDEF PATCH_GH}
	IsMenuActive := False;
{$ENDIF PATCH_GH}

	{The menu is no longer active.}
	RPM^.Active := False;

	{We have to send back a different value depending upon whether a selection}
	{was made or the menu was cancelled. If an item was selected, return its}
	{value field. The value always returned by a cancel will be -1.}
	{If a MenuKey was pressed, r already contains the right value.}
	if getit = ' ' then begin
			r := RPMLocateByPosition(RPM,RPM^.selectitem)^.value;
		end
	else if not UK then
		r := -1;

	SelectMenu := r;
end;

Procedure RPMSortAlpha(RPM: RPGMenuPtr);
	{Given a menu, RPM, sort its items based on the alphabetical}
	{order of their msg fields.}
	{I should mention here that I haven't written a sorting}
	{algorithm in years, and only once on a linked list (CS assignment).}
	{I think this is an insertion sort... I checked on internet for}
	{examples of sorting techniques, found a bunch of contradictory}
	{information, and decided to just write the easiest thing that}
	{would work. Since we're dealing with a relatively small number}
	{of items here, speed shouldn't be that big a concern.}
var
	sorted: RPGMenuItemPtr;	{The sorted list}
	a,b,c,d: RPGMenuItemPtr;{Counters. We always need them, you know.}
	youshouldstop: Boolean;	{Can you think of a better name?}
begin
	{Initialize A and Sorted.}
	a := RPM^.firstitem;
	Sorted := Nil;

	while a <> Nil do begin
		b := a;		{b is to be added to sorted}
		a := a^.next;	{increase A to the next item in the menu}

		{Give b's Next field a value of Nil.}
		b^.next := nil;

		{Locate the correct position in Sorted to store b}
		if Sorted = Nil then
			{This is the trivial case- Sorted is empty.}
			Sorted := b
		else if UpCase( b^.msg ) < Upcase( Sorted^.msg ) then begin
			{b should be the first element in the list.}
			c := sorted;
			sorted := b;
			sorted^.next := c;
			end
		else begin
			{c and d will be used to move through Sorted.}
			c := Sorted;

			{Locate the last item lower than b}
			youshouldstop := false;
			repeat
				d := c;
				c := c^.next;

				if c = Nil then
					youshouldstop := true
				else if UpCase( c^.msg ) > UpCase( b^.msg ) then begin
					youshouldstop := true;
				end;
			until youshouldstop;
			b^.next := c;
			d^.next := b;
		end;
	end;
	RPM^.firstitem := Sorted;
end;

{$IFDEF PATCH_CHEAT}
Procedure RPMSortAlpha_withSubItem(RPM: RPGMenuPtr);
var
	Sorted: RPGMenuItemPtr;
	NextBranches: RPGMenuItemPtr;
	CurrentBranch_Top, CurrentBranch_Bottom: RPGMenuItemPtr;
	TmpBranch_Top, TmpBranch_Bottom: RPGMenuItemPtr;
	TmpNextBranches: RPGMenuItemPtr;
	youshouldstop: Boolean;	{Can you think of a better name?}
begin
	NextBranches := RPM^.firstitem;
	Sorted := NIL;

	while NextBranches <> NIL do begin
		{  Take a current branch to 'CurrentBranch_Top' and 'CurrentBranch_Bottom', }
		{ and get the top of next branches to 'NextBranches'. }
		CurrentBranch_Top    := NextBranches;
		CurrentBranch_Bottom := CurrentBranch_Top;
		while (NIL <> CurrentBranch_Bottom^.next) and (#$0 = CurrentBranch_Bottom^.next^.msg[1]) do begin
			CurrentBranch_Bottom := CurrentBranch_Bottom^.next;
		end;
		NextBranches               := CurrentBranch_Bottom^.next;
		CurrentBranch_Bottom^.next := NIL;

		{ Locate the correct position in Sorted to store 'CurrentBranch_Top'. }
		if Sorted = NIL then begin
			Sorted := CurrentBranch_Top;
		end else if CurrentBranch_Top^.msg < Sorted^.msg then begin
			{ CurrentBranch_Top/CurrentBranch_Bottom should be the first element in the list. }
			TmpNextBranches := Sorted;
			Sorted          := CurrentBranch_Top;
			CurrentBranch_Bottom^.next := TmpNextBranches;
		end else begin
			{ Locate the last item lower than CurrentBranch_Top. }
			TmpNextBranches := Sorted;
			youshouldstop := false;
			repeat
				TmpBranch_Top    := TmpNextBranches;
				TmpBranch_Bottom := TmpBranch_Top;
				while (NIL <> TmpBranch_Bottom^.next) and (#$0 = TmpBranch_Bottom^.next^.msg[1]) do begin
					TmpBranch_Bottom := TmpBranch_Bottom^.next;
				end;
				TmpNextBranches := TmpBranch_Bottom^.next;

				if TmpNextBranches = NIL then
					youshouldstop := true
				else if CurrentBranch_Top^.msg < TmpNextBranches^.msg then begin
					youshouldstop := true;
				end;
			until youshouldstop;
			CurrentBranch_Bottom^.next := TmpNextBranches;
			TmpBranch_Bottom^.next := CurrentBranch_Top;
		end;
	end;
	RPM^.firstitem := Sorted;
end;
{$ENDIF PATCH_CHEAT}

{$IFDEF PATCH_GH}
Function SetItemByValue( RPM: RPGMenuPtr ; V: LongInt ): RPGMenuItemPtr;
{$ELSE PATCH_GH}
Function SetItemByValue( RPM: RPGMenuPtr ; V: Integer ): RPGMenuItemPtr;
{$ENDIF PATCH_GH}
	{ Search through the list, and set the SelectItem }
	{ field to the first menu item which matches V. }
var
	T: Integer;
	MI: RPGMenuItemPtr;
begin
	if RPM = Nil then exit;

	MI := RPM^.FirstItem;
	T := 1;

	while (MI <> Nil) and (MI^.Value <> V) do begin
		MI := MI^.Next;
		Inc( T );
	end;

	if MI <> Nil then begin
		RPM^.SelectItem := T;

		if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem) > MenuHeight( RPM ) ) then begin
			{Determine an appropriate new value for topitem.}
			RPMReposition(RPM,True);
		end;
	end;

	SetItemByValue := MI;
end;

Procedure SetItemByPosition( RPM: RPGMenuPtr ; N: Integer );
	{ Search through the list, and set the SelectItem }
	{ field to the Nth menu item. }
begin
	if RPM = Nil then exit;

{$IFDEF PATCH_GH}
	if RPM^.NumItem < N then begin
		RPM^.SelectItem := RPM^.NumItem;
	end else if N < 1 then begin
		RPM^.SelectItem := 1;
	end else begin
		RPM^.SelectItem := N;
	end;
	if RPM^.NumItem < RPM^.TopItem then begin
		RPM^.TopItem := RPM^.NumItem;
	end else if RPM^.TopItem < 1 then begin
		RPM^.TopItem := 1;
	end;
	if (RPM^.SelectItem < RPM^.TopItem) or ((RPM^.SelectItem - RPM^.TopItem + 1) > MenuHeight( RPM ) ) then begin
		{Determine an appropriate new value for topitem.}
		RPMReposition(RPM,True);
	end;
{$ELSE PATCH_GH}
	if N <= RPM^.NumItem then begin
		RPM^.SelectItem := N;

		if (RPM^.selectitem < RPM^.topitem) or ((RPM^.selectitem - RPM^.topitem + 1) > MenuHeight( RPM ) ) then begin
			{Determine an appropriate new value for topitem.}
			RPMReposition(RPM,True);
		end;
	end;
{$ENDIF PATCH_GH}
end;


{$IFDEF PATCH_GH}
Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String; N: Integer ): Integer;
	{ Do a DosSearch for files matching SearchPattern, then add }	
	{ each of the files found to the menu. }
var
	F: SearchRec;
{$ELSE PATCH_GH}
Procedure BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String );
	{ Do a DosSearch for files matching SearchPattern, then add }	
	{ each of the files found to the menu. }
var
	F: SearchRec;
	N: Integer;
{$ENDIF PATCH_GH}
begin
{$IFDEF PATCH_GH}
{$ELSE PATCH_GH}
	N := 1;
{$ENDIF PATCH_GH}
	FindFirst( SearchPattern , AnyFile , F );

	While DosError = 0 do begin
{$IFDEF PATCH_I18N}
		AddRPGMenuItem( RPM , TextDecode(F.Name) , N );
{$ELSE PATCH_I18N}
		AddRPGMenuItem( RPM , F.Name , N );
{$ENDIF PATCH_I18N}
		Inc(N);
		FindNext( F );
	end;

{$IFDEF PATCH_GH}
	FindClose( F );
	BuildFileMenu := N;
{$ELSE PATCH_GH}
{$ENDIF PATCH_GH}
end;
{$IFDEF PATCH_GH}
Function BuildFileMenu( RPM: RPGMenuPtr; const SearchPattern: String ): Integer;
begin
	BuildFileMenu := BuildFileMenu( RPM , SearchPattern , 1 );
end;
{$ENDIF PATCH_GH}

Function SelectFile( RPM: RPGMenuPtr; ReDrawer: RedrawProcedureType ): String;
	{ RPM is a menu created by the BuildFileMenu procedure. }
	{ So, select one of the items and return the item name, which }
	{ should be a filename. }
var
	N: Integer;	{ The number of the file selected. }
	Name: String;	{ The name of the filename selected. }
{$IFDEF PATCH_CHEAT}
	P: Integer;
{$ENDIF PATCH_CHEAT}
begin
{$IFDEF PATCH_CHEAT}
	if Cheat_Restore_AddMenuKey and (Pos(' ',RPM^.FirstItem^.msg) < 1) then begin
		AlphaKeyMenu( RPM );
	end;
{$ENDIF PATCH_CHEAT}

	{ Do the menu selection first. }
	N := SelectMenu( RPM , ReDrawer );

	if N = -1 then begin
		{ Selection was canceled. So, return an empty string. }
		Name := '';
	end else begin
		{ Locate the selected element of the menu. }
{$IFDEF PATCH_CHEAT}
		Name := SetItemByValue( RPM, N )^.msg;
		if Cheat_Restore_AddMenuKey then begin
			P := Pos(' ',Name) + 1;
			Name := Copy( Name, P, Length(Name)-P+1 );
		end;
{$ELSE PATCH_CHEAT}
		Name := RPMLocateByPosition(RPM,RPM^.SelectItem)^.msg;
{$ENDIF PATCH_CHEAT}
	end;

	SelectFile := Name;
end;



initialization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: sdlmenus.pp');
{$ENDIF DEBUG}
{$IFDEF PATCH_GH}
	IsMenuActive := False;
{$ENDIF PATCH_GH}
end;

finalization
begin
{$IFDEF DEBUG}
	ErrorMessage_fork('DEBUG: sdlmenus.pp(finalization)');
{$ENDIF DEBUG}
end;

end.
