/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: image_view.c,v 1.90.2.33 2003/05/22 03:44:56 makeinu Exp $
 */

#include <stdlib.h>
#include <string.h>

#include "gimageview.h"

#include "cursors.h"
#include "dnd.h"
#include "fileutil.h"
#include "gimv_image.h"
#include "gtk2-compat.h"
#include "gimv_icon_stock.h"
#include "gtkutils.h"
#include "image_view.h"
#include "image_window.h"
#include "menu.h"
#include "nav_window.h"
#include "thumbnail.h"
#include "prefs.h"

#define MIN_IMAGE_WIDTH  8
#define MIN_IMAGE_HEIGHT 8
#define IMAGE_VIEW_ENABLE_SCALABLE_LOAD 1


typedef enum {
   IMAGE_CHANGED_SIGNAL,
   LOAD_START_SIGNAL,
   LOAD_END_SIGNAL,
   SET_LIST_SIGNAL,
   UNSET_LIST_SIGNAL,
   RENDERED_SIGNAL,
   TOGGLE_ASPECT_SIGNAL,
   TOGGLE_BUFFER_SIGNAL,
   THUMBNAIL_CREATED_SIGNAL,

   IMAGE_PRESSED_SIGNAL,
   IMAGE_RELEASED_SIGNAL,
   IMAGE_CLICKED_SIGNAL,

   LAST_SIGNAL
} ImageViewSignalType;


typedef enum {
   MOVIE_STOP,
   MOVIE_PLAY,
   MOVIE_PAUSE,
   MOVIE_FORWARD,
   MOVIE_REVERSE,
   MOVIE_EJECT
} ImageViewMovieMenu;


typedef struct ImageViewImageList_Tag
{
   GList                *list;
   GList                *current;
   gpointer              owner;
   ImageViewNextFn       next_fn;
   ImageViewPrevFn       prev_fn;
   ImageViewNthFn        nth_fn;
   ImageViewRemoveListFn remove_list_fn;
   gpointer              list_fn_user_data;
} ImageViewImageList;


struct ImageViewPrivate_Tag
{
   /* image */
   GdkPixmap       *pixmap;
   GdkBitmap       *mask;

   /* image status */
   gint             x_pos;
   gint             y_pos;
   gint             width;
   gint             height;

   /* image list and related functions */
   ImageViewImageList *image_list;

   /* information about image dragging */
   guint            button;
   gboolean         pressed;
   gboolean         dragging;
   gint             drag_startx;
   gint             drag_starty;
   gint             x_pos_drag_start;
   gint             y_pos_drag_start;

   /* */
   guint   load_end_signal_id;
   guint   loader_progress_update_signal_id;
   guint   loader_load_end_signal_id;

   GtkWindow *fullscreen;
};


/* object class methods */
static void imageview_class_init    (ImageViewClass *klass);
static void imageview_init          (ImageView *iv);
static void imageview_destroy       (GtkObject *object);

/* image view class methods */
static void imageview_image_changed (ImageView *iv);

/* call back functions for reference popup menu */
static void cb_ignore_alpha                (ImageView       *iv,
                                            guint            action,
                                            GtkWidget       *widget);
static void cb_keep_aspect                 (ImageView       *iv,
                                            guint            action,
                                            GtkWidget       *widget);
static void cb_zoom                        (ImageView       *iv,
                                            ImageViewZoomType zoom,
                                            GtkWidget       *widget);
static void cb_rotate                      (ImageView       *iv,
                                            guint            action,
                                            GtkWidget       *widget);
static void cb_toggle_scrollbar            (ImageView       *iv,
                                            guint            action,
                                            GtkWidget       *widget);
static void cb_create_thumbnail            (ImageView       *iv,
                                            guint            action,
                                            GtkWidget       *widget);
static void cb_toggle_buffer               (ImageView       *iv,
                                            guint            action,
                                            GtkWidget       *widget);

/* call back functions for player toolbar */
static void     cb_imageview_play   (GtkButton      *button,
                                     ImageView      *iv);
static void     cb_imageview_stop   (GtkButton      *button,
                                     ImageView      *iv);
static void     cb_imageview_fw     (GtkButton      *button,
                                     ImageView      *iv);
static void     cb_imageview_rw     (GtkButton      *button,
                                     ImageView      *iv);
static void     cb_imageview_eject  (GtkButton      *button,
                                     ImageView      *iv);
static gboolean cb_seekbar_pressed  (GtkWidget      *widget,
                                     GdkEventButton *event,
                                     ImageView      *iv);
static gboolean cb_seekbar_released (GtkWidget      *widget,
                                     GdkEventButton *event,
                                     ImageView      *iv);

/* other call back functions */
static void     cb_destroy_loader          (ImageLoader       *loader,
                                            gpointer           data);
static gboolean cb_image_key_press         (GtkWidget         *widget,
                                            GdkEventKey       *event,
                                            ImageView         *iv);
static gboolean cb_image_button_press      (GtkWidget         *widget,
                                            GdkEventButton    *event,
                                            ImageView         *iv);
static gboolean cb_image_button_release    (GtkWidget         *widget, 
                                            GdkEventButton    *event,
                                            ImageView         *iv);
static gboolean cb_image_motion_notify     (GtkWidget         *widget, 
                                            GdkEventMotion    *event,
                                            ImageView         *iv);
static void     cb_scrollbar_value_changed (GtkAdjustment     *adj,
                                            ImageView         *iv);
static gboolean cb_nav_button              (GtkWidget         *widget,
                                            GdkEventButton    *event,
                                            ImageView         *iv);

/* callback functions for movie menu */
static void cb_movie_menu                  (ImageView       *iv,
                                            ImageViewMovieMenu operation,
                                            GtkWidget       *widget);
static void cb_movie_menu_destroy          (GtkWidget       *widget,
                                            ImageView       *iv);
static void cb_view_modes_menu_destroy     (GtkWidget       *widget,
                                            ImageView       *iv);
static void cb_continuance                 (ImageView       *iv,
                                            guint            active,
                                            GtkWidget       *widget);
static void cb_change_view_mode            (GtkWidget       *widget,
                                            ImageView       *iv);

/* other private functions */
static void allocate_draw_buffer             (ImageView       *iv);
static void imageview_calc_image_size        (ImageView       *iv);
static void imageview_rotate_render          (ImageView       *iv,
                                              guint            action);

static void imageview_change_draw_widget     (ImageView         *iv,
                                              const gchar       *label);
static void imageview_adjust_pos_in_frame    (ImageView         *iv,
                                              gboolean           center);
static gint idle_imageview_change_image_info (gpointer           data);
static void imageview_get_request_size       (ImageView         *iv,
                                              gint              *width_ret,
                                              gint              *height_ret);

static GtkWidget * imageview_create_player_toolbar (ImageView *iv);


static GtkVBoxClass *parent_class = NULL;
static gint imageview_signals[LAST_SIGNAL] = {0};


extern ImageViewPlugin imageview_draw_vfunc_table;

GList *draw_area_list = NULL;


/* reference menu items */
GtkItemFactoryEntry imageview_popup_items [] =
{
   {N_("/tear"),                  NULL,         NULL,                0, "<Tearoff>"},
   {N_("/_Zoom"),                 NULL,         NULL,                0, "<Branch>"},
   {N_("/_Rotate"),               NULL,         NULL,                0, "<Branch>"},
   {N_("/Ignore _Alpha Channel"), NULL,         cb_ignore_alpha,     0, "<ToggleItem>"},
   {N_("/---"),                   NULL,         NULL,                0, "<Separator>"},
   {N_("/M_ovie"),                NULL,         NULL,                0, "<Branch>"},
   {N_("/---"),                   NULL,         NULL,                0, "<Separator>"},
   {N_("/_View modes"),           NULL,         NULL,                0, "<Branch>"},
   {N_("/Show _Scrollbar"),       "<shift>S",   cb_toggle_scrollbar, 0, "<ToggleItem>"},
   {N_("/---"),                   NULL,         NULL,                0, "<Separator>"},
   {N_("/Create _Thumbnail"),     "<shift>T",   cb_create_thumbnail, 0, NULL},
   {N_("/Memory _Buffer"),        "<control>B", cb_toggle_buffer,    0, "<ToggleItem>"},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "Zoom" sub menu */
GtkItemFactoryEntry imageview_zoom_items [] =
{
   {N_("/tear"),                NULL,        NULL,            0,           "<Tearoff>"},
   {N_("/Zoom _In"),            "S",         cb_zoom,         IMAGEVIEW_ZOOM_IN,     NULL},
   {N_("/Zoom _Out"),           "A",         cb_zoom,         IMAGEVIEW_ZOOM_OUT,    NULL},
   {N_("/_Fit to Widnow"),      "W",         cb_zoom,         IMAGEVIEW_ZOOM_FIT,    NULL},
   {N_("/Keep _aspect ratio"),  "<shift>A",  cb_keep_aspect,  0,           "<ToggleItem>"},
   {N_("/---"),                 NULL,        NULL,            0,           "<Separator>"},
   {N_("/10%(_1)"),             "1",         cb_zoom,         IMAGEVIEW_ZOOM_10,     NULL},
   {N_("/25%(_2)"),             "2",         cb_zoom,         IMAGEVIEW_ZOOM_25,     NULL},
   {N_("/50%(_3)"),             "3",         cb_zoom,         IMAGEVIEW_ZOOM_50,     NULL},
   {N_("/75%(_4)"),             "4",         cb_zoom,         IMAGEVIEW_ZOOM_75,     NULL},
   {N_("/100%(_5)"),            "5",         cb_zoom,         IMAGEVIEW_ZOOM_100,    NULL},
   {N_("/125%(_6)"),            "6",         cb_zoom,         IMAGEVIEW_ZOOM_125,    NULL},
   {N_("/150%(_7)"),            "7",         cb_zoom,         IMAGEVIEW_ZOOM_150,    NULL},
   {N_("/175%(_8)"),            "8",         cb_zoom,         IMAGEVIEW_ZOOM_175,    NULL},
   {N_("/200%(_9)"),            "9",         cb_zoom,         IMAGEVIEW_ZOOM_200,    NULL},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "Rotate" sub menu */
GtkItemFactoryEntry imageview_rotate_items [] =
{
   {N_("/tear"),            NULL,  NULL,       0,           "<Tearoff>"},
   {N_("/Rotate 90 deg"),   "E",   cb_rotate,  IMAGEVIEW_ROTATE_90,   NULL},
   {N_("/Rotate -90 deg"),  "R",   cb_rotate,  IMAGEVIEW_ROTATE_270,  NULL},
   {N_("/Rotate 180 deg"),  "D",   cb_rotate,  IMAGEVIEW_ROTATE_180,  NULL},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "Movie" sub menu */
GtkItemFactoryEntry imageview_playable_items [] =
{
   {N_("/tear"),             NULL,  NULL,           0,             "<Tearoff>"},
   {N_("/_Play"),            NULL,  cb_movie_menu,  MOVIE_PLAY,    NULL},
   {N_("/_Stop"),            NULL,  cb_movie_menu,  MOVIE_STOP,    NULL},
   {N_("/P_ause"),           NULL,  cb_movie_menu,  MOVIE_PAUSE,   NULL},
   {N_("/_Forward"),         NULL,  cb_movie_menu,  MOVIE_FORWARD, NULL},
   {N_("/_Reverse"),         NULL,  cb_movie_menu,  MOVIE_REVERSE, NULL},
   {N_("/---"),              NULL,  NULL,           0,             "<Separator>"},
   {N_("/_Continuance"),     NULL,  cb_continuance, 0,             "<ToggleItem>"},
   {N_("/---"),              NULL,  NULL,           0,             "<Separator>"},
   {N_("/_Eject"),           NULL,  cb_movie_menu,  MOVIE_EJECT, NULL},
   {NULL, NULL, NULL, 0, NULL},
};


static GList *ImageViewList = NULL;
static GdkPixmap *buffer = NULL;
static gboolean move_scrollbar_by_user = TRUE;


/****************************************************************************
 *
 *  Plugin Management
 *
 ****************************************************************************/
static gint
comp_func_priority (ImageViewPlugin *plugin1,
                    ImageViewPlugin *plugin2)
{
   g_return_val_if_fail (plugin1, 1);
   g_return_val_if_fail (plugin2, -1);

   return plugin1->priority_hint - plugin2->priority_hint;
}

gboolean
imageview_plugin_regist (const gchar *plugin_name,
                         const gchar *module_name,
                         gpointer impl,
                         gint     size)
{
   ImageViewPlugin *plugin = impl;

   g_return_val_if_fail (module_name, FALSE);
   g_return_val_if_fail (plugin, FALSE);
   g_return_val_if_fail (size > 0, FALSE);
   g_return_val_if_fail (plugin->if_version == GIMV_IMAGE_VIEW_IF_VERSION, FALSE);
   g_return_val_if_fail (plugin->label, FALSE);

   if (!draw_area_list)
      draw_area_list = g_list_append (draw_area_list,
                                      &imageview_draw_vfunc_table);

   draw_area_list = g_list_append (draw_area_list, plugin);
   draw_area_list = g_list_sort (draw_area_list,
                                 (GCompareFunc) comp_func_priority);

   return TRUE;
}


GList *
imageview_plugin_get_list (void)
{
   if (!draw_area_list)
      draw_area_list = g_list_append (draw_area_list,
                                      &imageview_draw_vfunc_table);
   return draw_area_list;
}


/****************************************************************************
 *
 *
 *
 ****************************************************************************/
guint
imageview_get_type (void)
{
   static guint imageview_type = 0;

   if (!imageview_type) {
      static const GtkTypeInfo imageview_info = {
         "ImageView",
         sizeof (ImageView),
         sizeof (ImageViewClass),
         (GtkClassInitFunc) imageview_class_init,
         (GtkObjectInitFunc) imageview_init,
         NULL,
         NULL,
         (GtkClassInitFunc) NULL,
      };

      imageview_type = gtk_type_unique (gtk_vbox_get_type (), &imageview_info);
   }

   return imageview_type;
}


static void
imageview_class_init (ImageViewClass *klass)
{
   GtkObjectClass *object_class;

   object_class = (GtkObjectClass *) klass;
   parent_class = gtk_type_class (gtk_vbox_get_type ());

   imageview_signals[IMAGE_CHANGED_SIGNAL]
      = gtk_signal_new ("image_changed",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, image_changed),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   imageview_signals[LOAD_START_SIGNAL]
      = gtk_signal_new ("load_start",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, load_start),
                        gtk_marshal_NONE__POINTER,
                        GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

   imageview_signals[LOAD_END_SIGNAL]
      = gtk_signal_new ("load_end",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, load_end),
                        gtk_marshal_NONE__POINTER_INT,
                        GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_INT);

   imageview_signals[SET_LIST_SIGNAL]
      = gtk_signal_new ("set_list",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, set_list),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   imageview_signals[UNSET_LIST_SIGNAL]
      = gtk_signal_new ("unset_list",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, unset_list),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   imageview_signals[RENDERED_SIGNAL]
      = gtk_signal_new ("rendered",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, rendered),
                        gtk_signal_default_marshaller,
                        GTK_TYPE_NONE, 0);

   imageview_signals[TOGGLE_ASPECT_SIGNAL]
      = gtk_signal_new ("toggle_aspect",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, toggle_aspect),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1, GTK_TYPE_INT);

   imageview_signals[TOGGLE_BUFFER_SIGNAL]
      = gtk_signal_new ("toggle_buffer",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, toggle_buffer),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1, GTK_TYPE_INT);

   imageview_signals[THUMBNAIL_CREATED_SIGNAL]
      = gtk_signal_new ("thumbnail_created",
                        GTK_RUN_FIRST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, thumbnail_created),
                        gtk_marshal_NONE__POINTER,
                        GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

   imageview_signals[IMAGE_PRESSED_SIGNAL]
      = gtk_signal_new ("image_pressed",
                        GTK_RUN_LAST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, image_pressed),
                        gtk_marshal_BOOL__POINTER,
                        GTK_TYPE_BOOL, 1, GTK_TYPE_POINTER);

   imageview_signals[IMAGE_RELEASED_SIGNAL]
      = gtk_signal_new ("image_released",
                        GTK_RUN_LAST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, image_released),
                        gtk_marshal_BOOL__POINTER,
                        GTK_TYPE_BOOL, 1, GTK_TYPE_POINTER);

   imageview_signals[IMAGE_CLICKED_SIGNAL]
      = gtk_signal_new ("image_clicked",
                        GTK_RUN_LAST,
                        GTK_CLASS_TYPE(object_class),
                        GTK_SIGNAL_OFFSET (ImageViewClass, image_clicked),
                        gtk_marshal_BOOL__POINTER,
                        GTK_TYPE_BOOL, 1, GTK_TYPE_POINTER);

   gtk_object_class_add_signals (object_class, imageview_signals, LAST_SIGNAL);

   object_class->destroy  = imageview_destroy;

   klass->image_changed     = imageview_image_changed;
   klass->load_start        = NULL;
   klass->load_end          = NULL;
   klass->set_list          = NULL;
   klass->unset_list        = NULL;
   klass->rendered          = NULL;
   klass->toggle_aspect     = NULL;
   klass->toggle_buffer     = NULL;
   klass->thumbnail_created = NULL;
}


static void
imageview_init (ImageView *iv)
{
   /* initialize struct data based on config */
   iv->loader     = NULL;
   iv->image      = NULL;

   iv->show_scrollbar   = FALSE;
   iv->hadj = iv->vadj  = NULL;

   iv->draw_area        = NULL;
   iv->draw_area_funcs  = NULL;

   iv->progressbar      = NULL;

   iv->fit_to_frame_when_change_file = conf.imgview_fit_image_to_win;
   iv->fit_to_frame                  = conf.imgview_fit_image_to_win;
   iv->fit_to_frame_zoom_out_only    = conf.imgview_fit_image_zoom_out_only;

   iv->x_scale          = conf.imgview_scale;
   iv->y_scale          = conf.imgview_scale;
   iv->keep_aspect      = conf.imgview_keep_aspect;
   iv->ignore_alpha     = FALSE;
   iv->rotate           = 0;
   iv->buffer           = conf.imgview_buffer;

   iv->show_scrollbar   = conf.imgview_scrollbar;
   iv->continuance_play = conf.imgview_movie_continuance;

   iv->bg_color = NULL;

   iv->cursor           = NULL;
   iv->imageview_popup  = NULL;
   iv->zoom_menu        = NULL;
   iv->rotate_menu      = NULL;
   iv->movie_menu       = NULL;
   iv->view_modes_menu  = NULL;

   iv->player_flags     = 0;
   iv->player_visible   = ImageViewPlayerVisibleAuto;

   /* init private data */
   iv->priv = g_new0(ImageViewPrivate, 1);

   iv->priv->pixmap     = NULL;
   iv->priv->mask       = NULL;

   iv->priv->x_pos      = 0;
   iv->priv->y_pos      = 0;
   iv->priv->width      = 0;
   iv->priv->height     = 0;

   /* list */
   iv->priv->image_list = NULL;

   iv->priv->load_end_signal_id               = 0;
   iv->priv->loader_progress_update_signal_id = 0;
   iv->priv->loader_load_end_signal_id        = 0;

   iv->priv->fullscreen = NULL;
}


GtkWidget*
imageview_new (ImageInfo *info)
{
   ImageView *iv = gtk_type_new (imageview_get_type ());
   GtkWidget *player, *event_box;

   /* create widgets */
   iv->loader = image_loader_new ();

   iv->table = gtk_table_new (2, 2, FALSE);
   gtk_widget_show (iv->table);
   gtk_box_pack_start (GTK_BOX (iv), iv->table, TRUE, TRUE, 0);

   player = imageview_create_player_toolbar (iv);
   gtk_box_pack_start (GTK_BOX (iv), player, FALSE, FALSE, 2);

   imageview_change_draw_widget (iv, 0);

   iv->hadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 10.0, 10.0, 0.0));
   iv->hscrollbar = gtk_hscrollbar_new (iv->hadj);
   gtk_widget_show (iv->hscrollbar);

   iv->vadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 10.0, 10.0, 0.0));
   iv->vscrollbar = gtk_vscrollbar_new (iv->vadj);
   gtk_widget_show (iv->vscrollbar);

   event_box = gtk_event_box_new ();
   gtk_widget_set_name (event_box, "NavWinButton");
   gtk_widget_show (event_box);

   iv->nav_button = gimv_icon_stock_get_widget ("nav-button");
   gtk_container_add (GTK_CONTAINER (event_box), iv->nav_button);
   gtk_widget_show (iv->nav_button);

   gtk_table_attach (GTK_TABLE (iv->table), iv->vscrollbar,
                     1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
   gtk_table_attach (GTK_TABLE (iv->table), iv->hscrollbar,
                     0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
   gtk_table_attach (GTK_TABLE (iv->table), event_box,
                     1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);

   /* set signals */
   gtk_signal_connect (GTK_OBJECT (iv->hadj), "value_changed",
                       GTK_SIGNAL_FUNC (cb_scrollbar_value_changed), iv);

   gtk_signal_connect (GTK_OBJECT (iv->vadj), "value_changed",
                       GTK_SIGNAL_FUNC (cb_scrollbar_value_changed), iv);

   gtk_signal_connect (GTK_OBJECT (event_box), "button_press_event",
                       GTK_SIGNAL_FUNC (cb_nav_button), iv);

   /* add to list */
   ImageViewList = g_list_append (ImageViewList, iv);

   if (!iv->show_scrollbar) {
      imageview_hide_scrollbar (iv);
   }

   if (info) {
      iv->info = image_info_ref (info);
   } else {
      iv->info = NULL;
   }

   /* load image */
   if (iv->info)
      gtk_idle_add (idle_imageview_change_image_info, iv);

   return GTK_WIDGET (iv);
}


static void
imageview_destroy (GtkObject *object)
{
   ImageView *iv = IMAGEVIEW (object);

   if (iv->priv) {
      if (iv->priv->pixmap) {
         gimv_image_free_pixmap_and_mask (iv->priv->pixmap, iv->priv->mask);
         iv->priv->pixmap = NULL;
         iv->priv->mask = NULL;
      }

      if (iv->priv->image_list)
         imageview_remove_list (iv, iv->priv->image_list->owner);
      iv->priv->image_list = NULL;

      g_free(iv->priv);
      iv->priv = NULL;
   }

   if (iv->loader) {
      if (image_loader_is_loading (iv->loader)) {
         imageview_cancel_loading (iv);
         gtk_signal_connect (GTK_OBJECT (iv->loader), "load_end",
                             GTK_SIGNAL_FUNC (cb_destroy_loader),
                             iv);
      } else {
         image_loader_unref (iv->loader);
      }
   }
   iv->loader = NULL;

   if (iv->image)
      gimv_image_unref (iv->image);
   iv->image = NULL;

   if (iv->bg_color)
      g_free (iv->bg_color);
   iv->bg_color = NULL;

   if (iv->cursor)
      gdk_cursor_destroy (iv->cursor);
   iv->cursor = NULL;

   if (iv->info)
      image_info_unref (iv->info);
   iv->info = NULL;

   if (GTK_OBJECT_CLASS (parent_class)->destroy)
      GTK_OBJECT_CLASS (parent_class)->destroy (object);
}


static void
imageview_image_changed (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs) return;

   playable = iv->draw_area_funcs->playable;
   if (!playable) return;
   if (!playable->is_playable_fn) return;

   if (imageview_is_playable (iv)) {
      ImageViewPlayableStatus status;

      if (iv->player_visible == ImageViewPlayerVisibleAuto)
         gtk_widget_show (iv->player_container);
      status = imageview_playable_get_status (iv);
      imageview_playable_set_status (iv, status);

   } else {
      if (iv->player_visible == ImageViewPlayerVisibleAuto)
         gtk_widget_hide (iv->player_container);
      imageview_playable_set_status (iv, ImageViewPlayableDisable);
   }
}



/*****************************************************************************
 *
 *   Callback functions for reference popup menu.
 *
 *****************************************************************************/
static void
cb_ignore_alpha (ImageView *iv, guint action, GtkWidget *widget)
{
   iv->ignore_alpha = GTK_CHECK_MENU_ITEM(widget)->active;
   imageview_show_image (iv);
}


static void
cb_keep_aspect (ImageView *iv, guint action, GtkWidget *widget)
{
   iv->keep_aspect = GTK_CHECK_MENU_ITEM(widget)->active;

   gtk_signal_emit (GTK_OBJECT(iv),
                    imageview_signals[TOGGLE_ASPECT_SIGNAL],
                    iv->keep_aspect);
}


static void
cb_zoom (ImageView *iv, ImageViewZoomType zoom, GtkWidget *widget)
{
   imageview_zoom_image (iv, zoom, 0, 0);
}


static void
cb_rotate (ImageView *iv, ImageViewAngle rotate, GtkWidget *widget)
{
   guint angle;

   /* convert to absolute angle */
   angle = (iv->rotate + rotate) % 4;

   imageview_rotate_image (iv, angle);
}


static void
cb_toggle_scrollbar (ImageView *iv, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      imageview_show_scrollbar (iv);
   } else {
      imageview_hide_scrollbar (iv);
   }
}


static void
cb_create_thumbnail (ImageView *iv, guint action, GtkWidget *widget)
{
   g_return_if_fail (iv);

   imageview_create_thumbnail (iv);
}


static void
cb_toggle_buffer (ImageView *iv, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      imageview_load_image_buf (iv);
      iv->buffer = TRUE;
   }
   else {
      iv->buffer = FALSE;
      imageview_free_image_buf (iv);
   }

   gtk_signal_emit (GTK_OBJECT(iv),
                    imageview_signals[TOGGLE_BUFFER_SIGNAL],
                    iv->buffer);
}



/*****************************************************************************
 *
 *   callback functions for movie menu.
 *
 *****************************************************************************/
static void
cb_movie_menu (ImageView *iv, ImageViewMovieMenu operation, GtkWidget *widget)
{
   g_return_if_fail (iv);

   switch (operation) {
   case MOVIE_PLAY:
      imageview_playable_play (iv);
      break;
   case MOVIE_STOP:
      imageview_playable_stop (iv);
      break;
   case MOVIE_PAUSE:
      imageview_playable_pause (iv);
      break;
   case MOVIE_FORWARD:
      imageview_playable_forward (iv);
      break;
   case MOVIE_REVERSE:
      imageview_playable_reverse (iv);
      break;
   case MOVIE_EJECT:
      imageview_playable_eject (iv);
      break;
   default:
      break;
   }
}


static void
cb_movie_menu_destroy (GtkWidget *widget, ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));
   iv->movie_menu = NULL;
}


static void
cb_view_modes_menu_destroy (GtkWidget *widget, ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));
   iv->view_modes_menu = NULL;
}


static void
cb_continuance (ImageView *iv, guint action, GtkWidget *widget)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));
   g_return_if_fail (GTK_IS_CHECK_MENU_ITEM (widget));

   iv->continuance_play = GTK_CHECK_MENU_ITEM (widget)->active;
}


static void
cb_change_view_mode (GtkWidget *widget, ImageView *iv)
{
   const gchar *label;

   label = gtk_object_get_data (GTK_OBJECT (widget), "ImageView::ViewMode");
   imageview_change_view_mode (iv, label);
}



/*****************************************************************************
 *
 *   callback functions for player toolbar.
 *
 *****************************************************************************/
static void
cb_imageview_play (GtkButton *button, ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   imageview_playable_play (iv);
}


static void
cb_imageview_stop (GtkButton *button, ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   imageview_playable_stop (iv);
}


static void
cb_imageview_fw (GtkButton *button, ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   imageview_playable_forward (iv);
}


static void
cb_imageview_eject (GtkButton *button, ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   imageview_playable_eject (iv);
}


static void
cb_imageview_rw (GtkButton *button, ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   imageview_playable_reverse (iv);
}


static gboolean
cb_seekbar_pressed (GtkWidget *widget,
                    GdkEventButton *event,
                    ImageView *iv)
{
   g_return_val_if_fail (IS_IMAGEVIEW (iv), FALSE);

   iv->player_flags |= ImageViewSeekBarDraggingFlag;

   return FALSE;
}


static gboolean
cb_seekbar_released (GtkWidget *widget,
                     GdkEventButton *event,
                     ImageView *iv)
{
   GtkAdjustment *adj;
   g_return_val_if_fail (IS_IMAGEVIEW (iv), FALSE);

   adj = gtk_range_get_adjustment (GTK_RANGE (iv->player.seekbar));
   imageview_playable_seek (iv, adj->value);

   iv->player_flags &= ~ImageViewSeekBarDraggingFlag;

   return FALSE;
}



/*****************************************************************************
 *
 *   other callback functions.
 *
 *****************************************************************************/
static void
cb_destroy_loader (ImageLoader *loader, gpointer data)
{
   gtk_signal_disconnect_by_func (GTK_OBJECT (loader),
                                  GTK_SIGNAL_FUNC (cb_destroy_loader),
                                  data);
   image_loader_unref (loader);
}


static gboolean
cb_image_key_press (GtkWidget *widget, GdkEventKey *event, ImageView *iv)
{
   guint keyval, popup_key;
   GdkModifierType modval, popup_mod;
   gboolean move = FALSE;
   gint mx, my;

   g_return_val_if_fail (iv, FALSE);

   imageview_get_view_position (iv, &mx, &my);

   keyval = event->keyval;
   modval = event->state;

   if (akey.common_popup_menu || *akey.common_popup_menu)
      gtk_accelerator_parse (akey.common_popup_menu, &popup_key, &popup_mod);
   else
      return FALSE;

   if (keyval == popup_key && (!popup_mod || (modval & popup_mod))) {
      imageview_popup_menu (iv, NULL);
   } else {
      switch (keyval) {
      case GDK_Left:
      case GDK_KP_Left:
      case GDK_KP_4:
         mx -= 20;
         move = TRUE;
         break;
      case GDK_Right:
      case GDK_KP_Right:
      case GDK_KP_6:
         mx += 20;
         move = TRUE;
         break;
      case GDK_Up:
      case GDK_KP_Up:
      case GDK_KP_8:
         my -= 20;
         move = TRUE;
         break;
      case GDK_Down:
      case GDK_KP_Down:
      case GDK_KP_2:
         my += 20;
         move = TRUE;
         break;
      case GDK_KP_5:
         mx = 0;
         my = 0;
         move = TRUE;
         break;
      case GDK_Page_Up:
      case GDK_KP_Page_Up:
      case GDK_KP_9:
         imageview_prev (iv);
         return TRUE;
      case GDK_space:
      case GDK_Page_Down:
      case GDK_KP_Space:
      case GDK_KP_Page_Down:
      case GDK_KP_3:
      case GDK_KP_0:
         imageview_next (iv);
         return TRUE;
      case GDK_Home:
      case GDK_KP_Home:
      case GDK_KP_7:
        imageview_nth (iv, 0);
         return TRUE;
      case GDK_End:
      case GDK_KP_End:
      case GDK_KP_1:
      {
         gint last;
         if (iv->priv->image_list) {
            last = g_list_length (iv->priv->image_list->list) - 1;
            if (last > 0)
               imageview_nth (iv, last);
         }
         return TRUE;
      }
      case GDK_KP_Add:
      case GDK_plus:
         imageview_zoom_image (iv, IMAGEVIEW_ZOOM_IN, 0, 0);
         return FALSE;
         break;
      case GDK_KP_Subtract:
      case GDK_minus:
         imageview_zoom_image (iv, IMAGEVIEW_ZOOM_OUT, 0, 0);
         return FALSE;
         break;
      case GDK_equal:
      case GDK_KP_Equal:
      case GDK_KP_Enter:
         imageview_zoom_image (iv, IMAGEVIEW_ZOOM_100, 0, 0);
         return FALSE;
         break;
      case GDK_KP_Divide:
         imageview_zoom_image (iv, IMAGEVIEW_ZOOM_FIT, 0, 0);
         return FALSE;
         break;
      }
   }

   if (move) {
      imageview_moveto (iv, mx, my);
      return TRUE;
   }

   return FALSE;
}


static gboolean
cb_image_button_press (GtkWidget *widget, GdkEventButton *event,
                       ImageView *iv)
{
   GdkCursor *cursor;
   gint retval = FALSE;

   g_return_val_if_fail (iv, FALSE);

   iv->priv->pressed = TRUE;
   iv->priv->button  = event->button;

   gtk_widget_grab_focus (widget);

   gtk_signal_emit (GTK_OBJECT (iv),
                    imageview_signals[IMAGE_PRESSED_SIGNAL],
                    event, &retval);

   if (iv->priv->dragging)
      return FALSE;

   if (event->button == 1) {   /* scroll image */
      if (!iv->priv->pixmap)
         return FALSE;

      cursor = cursor_get (widget->window, CURSOR_HAND_CLOSED);
      retval = gdk_pointer_grab (widget->window, FALSE,
                                 (GDK_POINTER_MOTION_MASK
                                  | GDK_POINTER_MOTION_HINT_MASK
                                  | GDK_BUTTON_RELEASE_MASK),
                                 NULL, cursor, event->time);
      gdk_cursor_destroy (cursor);

      if (retval != 0)
         return FALSE;

      iv->priv->drag_startx = event->x - iv->priv->x_pos;
      iv->priv->drag_starty = event->y - iv->priv->y_pos;
      iv->priv->x_pos_drag_start = iv->priv->x_pos;
      iv->priv->y_pos_drag_start = iv->priv->y_pos;

      allocate_draw_buffer (iv);

      return TRUE;

   }

   return FALSE;
}


static gboolean
cb_image_button_release  (GtkWidget *widget, GdkEventButton *event,
                          ImageView *iv)
{
   gint retval = FALSE;

   /*
   if (event->button != 1)
      return FALSE;
   */

   gtk_signal_emit (GTK_OBJECT (iv),
                    imageview_signals[IMAGE_RELEASED_SIGNAL],
                    event, &retval);

   if(iv->priv->pressed && !iv->priv->dragging)
      gtk_signal_emit (GTK_OBJECT (iv),
                       imageview_signals[IMAGE_CLICKED_SIGNAL],
                       event, &retval);

   if (buffer) {
      gdk_pixmap_unref (buffer);
      buffer = NULL;
   }

   iv->priv->button   = 0;
   iv->priv->pressed  = FALSE;
   iv->priv->dragging = FALSE;

   gdk_pointer_ungrab (event->time);

   return retval;
}


static gboolean
cb_image_motion_notify (GtkWidget *widget, GdkEventMotion *event,
                        ImageView *iv)
{
   GdkModifierType mods;
   gint x, y, x_pos, y_pos, dx, dy;

   if (!iv->priv->pressed)
      return FALSE;

   gdk_window_get_pointer (widget->window, &x, &y, &mods);

   x_pos = x - iv->priv->drag_startx;
   y_pos = y - iv->priv->drag_starty;
   dx = x_pos - iv->priv->x_pos_drag_start;
   dy = y_pos - iv->priv->y_pos_drag_start;

   if (!iv->priv->dragging && (abs(dx) > 2 || abs (dy) > 2))
      iv->priv->dragging = TRUE;

   /* scroll image */
   if (iv->priv->button == 1) {
      if (event->is_hint)
         gdk_window_get_pointer (widget->window, &x, &y, &mods);

      iv->priv->x_pos = x_pos;
      iv->priv->y_pos = y_pos;

      imageview_adjust_pos_in_frame (iv, FALSE);
   }

   imageview_draw_image (iv);

   return TRUE;
}


static void
cb_drag_data_received (GtkWidget *widget,
                       GdkDragContext *context,
                       gint x, gint y,
                       GtkSelectionData *seldata,
                       guint info,
                       guint32 time,
                       gpointer data)
{
   ImageView *iv = data;
   FilesLoader *files;
   GList *filelist;
   gchar *tmpstr;

   g_return_if_fail (iv || widget);

   filelist = dnd_get_file_list (seldata->data, seldata->length);

   if (filelist) {
      ImageInfo *info = image_info_get ((gchar *) filelist->data);

      if (!info) goto ERROR;

      imageview_change_image (iv, info);
      tmpstr = filelist->data;
      filelist = g_list_remove (filelist, tmpstr);
      g_free (tmpstr);

      if (filelist) {
         files = files_loader_new ();
         files->filelist = filelist;
         open_image_files_in_image_view (files);
         files->filelist = NULL;
         files_loader_delete (files);
      }
   }

ERROR:
   g_list_foreach (filelist, (GFunc) g_free, NULL);
   g_list_free (filelist);
}


static void
cb_scrollbar_value_changed (GtkAdjustment *adj, ImageView *iv)
{
   g_return_if_fail (iv);

   if (!move_scrollbar_by_user) return;

   if (iv->priv->width > iv->draw_area->allocation.width)
      iv->priv->x_pos = 0 - iv->hadj->value;
   if (iv->priv->height > iv->draw_area->allocation.height)
      iv->priv->y_pos = 0 - iv->vadj->value;

   imageview_draw_image (iv);
}


static gboolean
cb_nav_button (GtkWidget *widget, GdkEventButton *event, ImageView *iv)
{
   g_return_val_if_fail (iv, FALSE);

   imageview_open_navwin (iv, event->x_root, event->y_root);

   return FALSE;
}


/*****************************************************************************
 *
 *   Private functions.
 *
 *****************************************************************************/
static void
allocate_draw_buffer (ImageView *iv)
{
   gint fwidth, fheight;

   if (!iv) return;

   if (buffer) gdk_pixmap_unref (buffer);
   imageview_get_image_frame_size (iv, &fwidth, &fheight);
   buffer = gdk_pixmap_new (iv->draw_area->window, fwidth, fheight, -1);
}



static void
imageview_calc_image_size (ImageView *iv)
{
   gint orig_width, orig_height;
   gint width, height;
   gfloat x_scale, y_scale;

   /* image scale */
   if (iv->rotate == 0 || iv->rotate == 2) {
      x_scale = iv->x_scale;
      y_scale = iv->y_scale;
   } else {
      x_scale = iv->y_scale;
      y_scale = iv->x_scale;
   }

   if (iv->rotate == 0 || iv->rotate == 2) {
      orig_width  = iv->info->width;
      orig_height = iv->info->height;
   } else {
      orig_width  = iv->info->height;
      orig_height = iv->info->width;
   }

   /* calculate image size */
   if (iv->fit_to_frame) {
      gint fwidth, fheight;

      width  = orig_width;
      height = orig_height;
      imageview_get_image_frame_size (iv, &fwidth, &fheight);

      if (width <= fwidth && height <= fheight
          && iv->fit_to_frame_zoom_out_only)
      {
         iv->x_scale = 100.0;
         iv->y_scale = 100.0;
      } else {
         iv->x_scale = (gfloat) fwidth  / (gfloat) width  * 100.0;
         iv->y_scale = (gfloat) fheight / (gfloat) height * 100.0;

         if (iv->keep_aspect) {
            if (iv->x_scale > iv->y_scale) {
               iv->x_scale = iv->y_scale;
               width  = orig_width * iv->x_scale / 100.0;
               height = fheight;
            } else {
               iv->y_scale = iv->x_scale;
               width  = fwidth;
               height = orig_height * iv->y_scale / 100.0;
            }
         }
      }

      iv->fit_to_frame = FALSE;

   } else {
      width  = orig_width  * x_scale / 100.0;
      height = orig_height * y_scale / 100.0;
   }

   /*
    * zoom out if image is too big & window is not scrollable on
    * fullscreen mode
    */
   /*
     win_width  = hadj->page_size;
     win_height = vadj->page_size;

     if (im->fullscreen && (width > win_width || height > win_height)) {
     gfloat width_ratio, height_ratio;

     width_ratio = (gdouble) width / hadj->page_size;
     height_ratio = (gdouble) height / vadj->page_size;
     if (width_ratio > height_ratio) {
     width = win_width;
     height = (gdouble) height / (gdouble) width_ratio;
     } else {
     width = (gdouble) width / (gdouble) height_ratio;
     height = win_height;
     }
     }
   */

   if (width > MIN_IMAGE_WIDTH)
      iv->priv->width = width;
   else
      iv->priv->width = MIN_IMAGE_WIDTH;
   if (height > MIN_IMAGE_HEIGHT)
      iv->priv->height = height;
   else
      iv->priv->height = MIN_IMAGE_HEIGHT;
}


static void
imageview_change_draw_widget (ImageView *iv, const gchar *label)
{
   GList *node;
   ImageViewPlugin *vftable = NULL;
   const gchar *view_mode;

   g_return_if_fail (iv);

   if (iv->draw_area_funcs
       && iv->draw_area_funcs->is_supported_fn
       && iv->draw_area_funcs->is_supported_fn (iv, iv->info))
   {
      return;
   }

   /* if default movie view mode is specified */
   /* FIXME!! this code is ad-hoc */
   view_mode = conf.movie_default_view_mode;
   if (iv->info
       && (image_info_is_movie (iv->info) || image_info_is_audio (iv->info))
       && view_mode && *view_mode)
   {
      for (node = imageview_plugin_get_list(); node; node = g_list_next (node)) {
         ImageViewPlugin *table = node->data;

         if (!table) continue;

         if (!strcmp (view_mode, table->label)
             && table->is_supported_fn
             && table->is_supported_fn (iv, iv->info))
         {
            vftable = table;
            break;
         }
      }
   }

   /* fall down... */
   for (node = imageview_plugin_get_list();
        node && !vftable;
        node = g_list_next (node))
   {
      ImageViewPlugin *table = node->data;

      if (!table) continue;

      if (label && *label) {
         if (!strcmp (label, table->label)
             || !strcmp (IMAGEVIEW_DEFAULT_VIEW_MODE, table->label))
         {
            vftable = table;
            break;
         }
      } else {
         if (!table->is_supported_fn
             || table->is_supported_fn (iv, iv->info))
         {
            vftable = table;
            break;
         }
      }
   }

   if (!vftable)
      vftable = &imageview_draw_vfunc_table;
   g_return_if_fail (vftable->create_fn);
   if (iv->draw_area_funcs && !strcmp (iv->draw_area_funcs->label, vftable->label))
      return;
   if (iv->draw_area)
      gtk_widget_destroy (iv->draw_area);

   iv->draw_area = vftable->create_fn (iv);
   iv->draw_area_funcs = vftable;

   gtk_widget_show (iv->draw_area);

   gtk_signal_connect_after (GTK_OBJECT (iv->draw_area), "key-press-event",
                             GTK_SIGNAL_FUNC(cb_image_key_press), iv);
   gtk_signal_connect (GTK_OBJECT (iv->draw_area), "button_press_event",
                       GTK_SIGNAL_FUNC (cb_image_button_press), iv);
   gtk_signal_connect (GTK_OBJECT (iv->draw_area), "button_release_event",
                       GTK_SIGNAL_FUNC (cb_image_button_release), iv);
   gtk_signal_connect (GTK_OBJECT (iv->draw_area), "motion_notify_event",
                       GTK_SIGNAL_FUNC (cb_image_motion_notify), iv);
   SIGNAL_CONNECT_TRANSRATE_SCROLL(iv->draw_area);

   if (iv->priv->fullscreen) {
      gtk_container_add (GTK_CONTAINER (iv->priv->fullscreen), iv->draw_area);
   } else {
      gtk_table_attach (GTK_TABLE (iv->table), iv->draw_area,
                        0, 1, 0, 1,
                        GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
   }

   /* for droping file list */
   gtk_signal_connect(GTK_OBJECT (iv->draw_area), "drag_data_received",
                      GTK_SIGNAL_FUNC (cb_drag_data_received), iv);

   dnd_dest_set (iv->draw_area, dnd_types_archive, dnd_types_archive_num);

   /* set flags */
   GTK_WIDGET_SET_FLAGS (iv->draw_area, GTK_CAN_FOCUS);
}


static void
imageview_adjust_pos_in_frame (ImageView *iv, gboolean center)
{
   gint fwidth, fheight;

   if (center) {
      imageview_get_image_frame_size (iv, &fwidth, &fheight);

      if (fwidth < iv->priv->width) {
         if (conf.imgview_scroll_nolimit) {
            if (iv->priv->x_pos > fwidth)
               iv->priv->x_pos = 0;
            else if (iv->priv->x_pos < 0 - iv->priv->width)
               iv->priv->x_pos = 0 - iv->priv->width + fwidth;
         } else {
            if (iv->priv->x_pos > 0)
               iv->priv->x_pos = 0;
            else if (iv->priv->x_pos < 0 - iv->priv->width + fwidth)
               iv->priv->x_pos = 0 - iv->priv->width + fwidth;
         }
      } else {
         iv->priv->x_pos = (fwidth - iv->priv->width) / 2;
      }

      if (fheight < iv->priv->height) {
         if (conf.imgview_scroll_nolimit) {
            if (iv->priv->y_pos > fheight)
               iv->priv->y_pos = 0;
            else if (iv->priv->y_pos < 0 - iv->priv->height)
               iv->priv->y_pos = 0 - iv->priv->height + fheight;
         } else {
            if (iv->priv->y_pos > 0)
               iv->priv->y_pos = 0;
            else if (iv->priv->y_pos < 0 - iv->priv->height + fheight)
               iv->priv->y_pos = 0 - iv->priv->height + fheight;
         }
      } else {
         iv->priv->y_pos = (fheight - iv->priv->height) / 2;
      }
   } else {
      if (!conf.imgview_scroll_nolimit) {
         imageview_get_image_frame_size (iv, &fwidth, &fheight);

         if (iv->priv->width <= fwidth) {
            if (iv->priv->x_pos < 0)
               iv->priv->x_pos = 0;
            if (iv->priv->x_pos > fwidth - iv->priv->width)
               iv->priv->x_pos = fwidth - iv->priv->width;
         } else if (iv->priv->x_pos < fwidth - iv->priv->width) {
            iv->priv->x_pos = fwidth - iv->priv->width;
         } else if (iv->priv->x_pos > 0) {
            iv->priv->x_pos = 0;
         }

         if (iv->priv->height <= fheight) {
            if (iv->priv->y_pos < 0)
               iv->priv->y_pos = 0;
            if (iv->priv->y_pos > fheight - iv->priv->height)
               iv->priv->y_pos = fheight - iv->priv->height;
         } else if (iv->priv->y_pos < fheight - iv->priv->height) {
            iv->priv->y_pos = fheight - iv->priv->height;
         } else if (iv->priv->y_pos > 0) {
            iv->priv->y_pos = 0;
         }
      }
   }
}


/*****************************************************************************
 *
 *   Public functions.
 *
 *****************************************************************************/
GList *
imageview_get_list (void)
{
   return ImageViewList;
}


void
imageview_change_image (ImageView *iv, ImageInfo *info)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   imageview_playable_stop (iv);

   imageview_change_image_info (iv, info);

   imageview_change_draw_widget (iv, NULL);
   if (!g_list_find (ImageViewList, iv)) return;

   imageview_show_image (iv);
   if (!g_list_find (ImageViewList, iv)) return;

   imageview_playable_play (iv);

   gtk_signal_emit (GTK_OBJECT(iv), imageview_signals[IMAGE_CHANGED_SIGNAL]);
}


void
imageview_change_image_info (ImageView *iv, ImageInfo *info)
{
   g_return_if_fail (iv);

   if (!g_list_find (ImageViewList, iv)) return;

   /* free old image */
   if (iv->priv->pixmap)
      gimv_image_free_pixmap_and_mask (iv->priv->pixmap, iv->priv->mask);
   iv->priv->pixmap = NULL;
   iv->priv->mask = NULL;

   if (iv->image)
      gimv_image_unref (iv->image);
   iv->image = NULL;

   /* free old image info */
   if (iv->info)
      image_info_unref (iv->info);

   /* reset fit_to_widnow */
   iv->fit_to_frame = iv->fit_to_frame_when_change_file;

   /* suggestion from sheepman <sheepman@tcn.zaq.ne.jp> */
   iv->priv->x_pos = iv->priv->y_pos = 0;

   /* suggestion from Thorsten Schnier <thorsten@schnier.net> */
   iv->rotate = 0;

   /* suggestion from me :D <ashie@homa.ne.jp> */
   iv->priv->width = iv->priv->height = 0;

   /* allocate memory for new image info*/
   if (info)
      iv->info = image_info_ref (info);
   else
      iv->info = NULL;
}


static gint
idle_imageview_change_image_info (gpointer data)
{
   ImageView *iv = data;
   GList *node;

   g_return_val_if_fail (data, FALSE);

   node = g_list_find (ImageViewList, iv);
   if (!node) return FALSE;

   image_info_ref (iv->info);
   imageview_change_image (iv, iv->info);

   return FALSE;
}


void
imageview_draw_image (ImageView *iv)
{
   GdkGC *bg_gc;
   gboolean free = FALSE;

   /* allocate buffer */
   if (!buffer) {
      allocate_draw_buffer (iv);
      free = TRUE;
   }

   /* fill background by default bg color */
   bg_gc = iv->draw_area->style->bg_gc[GTK_WIDGET_STATE (iv->draw_area)];
   gdk_draw_rectangle (buffer, bg_gc, TRUE, 0, 0, -1, -1);

   /* draw image to buffer */
   if (iv->priv->pixmap) {
      if (iv->priv->mask) {
         gdk_gc_set_clip_mask (iv->draw_area->style->black_gc, iv->priv->mask);
         gdk_gc_set_clip_origin (iv->draw_area->style->black_gc,
                                 iv->priv->x_pos, iv->priv->y_pos);
      }

      gdk_draw_pixmap (buffer,
                       iv->draw_area->style->black_gc,
                       iv->priv->pixmap,
                       0, 0,
                       iv->priv->x_pos, iv->priv->y_pos,
                       -1, -1);

      if (iv->priv->mask) {
         gdk_gc_set_clip_mask (iv->draw_area->style->black_gc, NULL);
         gdk_gc_set_clip_origin (iv->draw_area->style->black_gc, 0, 0);
      }
   }

   /* draw from buffer to foreground */
   gdk_draw_pixmap (iv->draw_area->window,
                    iv->draw_area->style->fg_gc[GTK_WIDGET_STATE (iv->draw_area)],
                    buffer, 0, 0, 0, 0, -1, -1);

   /* free buffer */
   if (free) {
      gdk_pixmap_unref (buffer);
      buffer = NULL;
   }

   imageview_reset_scrollbar (iv);
}


void
imageview_show_image (ImageView *iv)
{
   imageview_rotate_image (iv, iv->rotate);
}


static gboolean
check_can_draw_image (ImageView *iv)
{
   gboolean is_movie = FALSE;
   const gchar *newfile = NULL;

   g_return_val_if_fail (IS_IMAGEVIEW (iv), FALSE);
   g_return_val_if_fail (iv->draw_area, FALSE);
   g_return_val_if_fail (GTK_IS_DRAWING_AREA (iv->draw_area), FALSE);

   /* if (!GTK_WIDGET_MAPPED (iv->draw_area)) return FALSE; */

#warning FIXME!!
   if (iv->info && image_info_is_movie (iv->info))
      is_movie = TRUE;

   if (is_movie) return TRUE;
   if (iv->image) return TRUE;

   if (iv->info)
      newfile = image_info_get_path (iv->info);

   if (newfile && *newfile) {
      if (!file_exists (newfile))
         g_print (_("File not exist: %s\n"), newfile);
      else
         g_print(_("Not an image (or unsupported) file: %s\n"), newfile);
   }

   /* clear draw area */
   imageview_draw_image (iv);

   return FALSE;
}




void
imageview_create_thumbnail (ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->info) return;
   if (iv->info->flags & IMAGE_INFO_MRL_FLAG) return;   /* Dose it need? */

   if (!iv->draw_area_funcs) return;
   if (!iv->draw_area_funcs->create_thumbnail_fn) return;

   iv->draw_area_funcs->create_thumbnail_fn (iv, conf.cache_write_type);
}


/*****************************************************************************
 *
 *   functions for creating/show/hide/setting status child widgets
 *
 *****************************************************************************/
void
imageview_change_view_mode (ImageView *iv,
                            const gchar *label)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   imageview_change_image (iv, NULL);
   imageview_change_draw_widget (iv, label);
}


static GtkWidget *
imageview_create_player_toolbar (ImageView *iv)
{
   GtkWidget *hbox, *toolbar;
   GtkWidget *button, *seekbar;
   GtkWidget *iconw;
   GtkObject *adj;

   hbox = gtk_hbox_new (FALSE, 0);
   toolbar = gtkutil_create_toolbar ();
   gtk_toolbar_set_style (GTK_TOOLBAR(toolbar), GTK_TOOLBAR_ICONS);
   gtk_box_pack_start (GTK_BOX (hbox), toolbar, FALSE, FALSE, 0);
   gtk_widget_show (toolbar);

   iv->player_container = hbox;
   iv->player_toolbar = toolbar;

   /* Reverse button */
   iconw = gimv_icon_stock_get_widget ("rw");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("RW"),
                                     _("Reverse"), _("Reverse"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_imageview_rw), iv);
   gtk_widget_set_sensitive (button, FALSE);
   iv->player.rw = button;

   /* play button */
   iconw = gimv_icon_stock_get_widget ("play");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Play"),
                                     _("Play"), _("Play"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_imageview_play), iv);
   gtk_widget_set_sensitive (button, FALSE);
   iv->player.play = button;
   iv->player.play_icon = iconw;

   /* stop button */
   iconw = gimv_icon_stock_get_widget ("stop2");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Stop"),
                                     _("Stop"), _("Stop"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_imageview_stop), iv);
   gtk_widget_set_sensitive (button, FALSE);
   iv->player.stop = button;

   /* Forward button */
   iconw = gimv_icon_stock_get_widget ("ff");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("FF"),
                                     _("Forward"), _("Forward"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_imageview_fw), iv);
   gtk_widget_set_sensitive (button, FALSE);
   iv->player.fw = button;

   /* Eject button */
   iconw = gimv_icon_stock_get_widget ("eject");
   button = gtk_toolbar_append_item (GTK_TOOLBAR (toolbar),
                                     _("Eject"),
                                     _("Eject"), _("Eject"),
                                     iconw,
                                     GTK_SIGNAL_FUNC (cb_imageview_eject), iv);
   iv->player.eject = button;
   gtk_widget_set_sensitive (button, FALSE);

   adj = gtk_adjustment_new (0.0, 0.0, 100.0, 0.1, 1.0, 1.0);
   seekbar = gtk_hscale_new (GTK_ADJUSTMENT (adj));
   gtk_scale_set_draw_value (GTK_SCALE (seekbar), FALSE);
   gtk_box_pack_start (GTK_BOX (hbox), seekbar, TRUE, TRUE, 0);
   gtk_widget_show (seekbar);
   iv->player.seekbar = seekbar;

   gtk_signal_connect (GTK_OBJECT (iv->player.seekbar), "button_press_event",
                       GTK_SIGNAL_FUNC (cb_seekbar_pressed), iv);
   gtk_signal_connect (GTK_OBJECT (iv->player.seekbar), "button_release_event",
                       GTK_SIGNAL_FUNC (cb_seekbar_released), iv);

   return hbox;
}


GtkWidget *
imageview_create_zoom_menu (GtkWidget *window,
                            ImageView *iv,
                            const gchar *path)
{
   GtkWidget *menu;
   guint n_menu_items;

   g_return_val_if_fail (window, NULL);
   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (path && *path, NULL);

   n_menu_items = sizeof(imageview_zoom_items)
      / sizeof(imageview_zoom_items[0]) - 1;
   menu = menu_create_items(window, imageview_zoom_items,
                            n_menu_items, path, iv);
   iv->zoom_menu = menu;
   menu_check_item_set_active (menu, "/Keep aspect ratio", iv->keep_aspect);

   return menu;
}


GtkWidget *
imageview_create_rotate_menu (GtkWidget *window,
                              ImageView *iv,
                              const gchar *path)
{
   GtkWidget *menu;
   guint n_menu_items;

   g_return_val_if_fail (window, NULL);
   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (path && *path, NULL);

   n_menu_items = sizeof(imageview_rotate_items)
      / sizeof(imageview_rotate_items[0]) - 1;
   menu = menu_create_items(window, imageview_rotate_items,
                            n_menu_items, path, iv);
   iv->rotate_menu = menu;

   return menu;
}


GtkWidget *
imageview_create_movie_menu (GtkWidget *window,
                             ImageView *iv,
                             const gchar *path)
{
   GtkWidget *menu;
   guint n_menu_items;
   ImageViewPlayableStatus status;

   n_menu_items = sizeof(imageview_playable_items)
      / sizeof(imageview_playable_items[0]) - 1;
   menu = menu_create_items(window, imageview_playable_items,
                            n_menu_items, path, iv);
   iv->movie_menu = menu;

   menu_check_item_set_active (iv->movie_menu, "/Continuance",
                               iv->continuance_play);
   status = imageview_playable_get_status (iv);
   imageview_playable_set_status (iv, status);

   gtk_signal_connect (GTK_OBJECT (menu), "destroy",
                       GTK_SIGNAL_FUNC (cb_movie_menu_destroy), iv);

   return menu;
}


GtkWidget *
imageview_create_view_modes_menu (GtkWidget *window,
                                  ImageView *iv,
                                  const gchar *path)
{
   GtkWidget *menu;
   GList *node;

   menu = gtk_menu_new();
   iv->view_modes_menu = menu;

   for (node = imageview_plugin_get_list(); node; node = g_list_next (node)) {
      GtkWidget *menu_item;
      ImageViewPlugin *vftable = node->data;

      if (!vftable) continue;

      menu_item = gtk_menu_item_new_with_label (_(vftable->label));
      gtk_object_set_data (GTK_OBJECT (menu_item),
                           "ImageView::ViewMode",
                           (gpointer) vftable->label);
      gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                          GTK_SIGNAL_FUNC (cb_change_view_mode), iv);
      gtk_menu_append (GTK_MENU (menu), menu_item);
      gtk_widget_show (menu_item);
   }

   gtk_signal_connect (GTK_OBJECT (menu), "destroy",
                       GTK_SIGNAL_FUNC (cb_view_modes_menu_destroy), iv);

   return menu;
}


GtkWidget *
imageview_create_popup_menu (GtkWidget *window,
                             ImageView *iv,
                             const gchar *path)
{
   guint n_menu_items;

   g_return_val_if_fail (window, NULL);
   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (path && *path, NULL);

   n_menu_items = sizeof(imageview_popup_items)
      / sizeof(imageview_popup_items[0]) - 1;
   iv->imageview_popup = menu_create_items(window, imageview_popup_items,
                                           n_menu_items, path, iv);

   imageview_create_zoom_menu (window, iv, path);
   imageview_create_rotate_menu (window, iv, path);

   menu_set_submenu (iv->imageview_popup, "/Zoom",   iv->zoom_menu);
   menu_set_submenu (iv->imageview_popup, "/Rotate", iv->rotate_menu);

   menu_check_item_set_active (iv->imageview_popup, "/Show Scrollbar",
                               iv->show_scrollbar);
   menu_check_item_set_active (iv->imageview_popup, "/Memory Buffer",
                               iv->buffer);

   imageview_create_movie_menu (window, iv, path);
   menu_set_submenu (iv->imageview_popup, "/Movie",  iv->movie_menu);

   imageview_create_view_modes_menu (window, iv, path);
   menu_set_submenu (iv->imageview_popup, "/View modes",  iv->view_modes_menu);

   return iv->imageview_popup;
}


void
imageview_popup_menu (ImageView *iv, GdkEventButton *event)
{
   guint button;
   guint32 time;
   GtkMenuPositionFunc pos_fn = NULL;

   g_return_if_fail (iv);
   /* g_return_if_fail (event); */

   if (event) {
      button = event->button;
      time = gdk_event_get_time ((GdkEvent *) event);
   } else {
      button = 0;
      time = GDK_CURRENT_TIME;
      pos_fn = menu_calc_popup_position;
   }

   if (iv->imageview_popup)
      gtk_menu_popup (GTK_MENU (iv->imageview_popup),
                      NULL, NULL,
                      pos_fn, iv->draw_area->window,
                      button, time);
}


void
imageview_set_bg_color (ImageView *iv, gint red, gint green, gint brue)
{
   g_return_if_fail (iv);
   g_return_if_fail (iv->draw_area);

   if (!iv->bg_color) {
      iv->bg_color = g_new0 (GdkColor, 1);
      iv->bg_color->pixel = 0;
   }
   iv->bg_color->red   = red;
   iv->bg_color->green = green;
   iv->bg_color->blue  = brue;

   if (GTK_WIDGET_MAPPED (iv->draw_area)) {
      GdkColormap *colormap;
      GtkStyle *style;
      colormap = gdk_window_get_colormap(iv->draw_area->window);
      gdk_colormap_alloc_color (colormap, iv->bg_color, FALSE, TRUE);
      style = gtk_style_copy (gtk_widget_get_style (iv->draw_area));
      style->bg[GTK_STATE_NORMAL] = *iv->bg_color;
      gtk_widget_set_style (iv->draw_area, style);
      gtk_style_unref (style);
      /* imageview_show_image (iv); */
   }
}


void
imageview_show_scrollbar (ImageView *iv)
{
   g_return_if_fail (iv);
   g_return_if_fail (iv->hscrollbar);
   g_return_if_fail (iv->vscrollbar);

   gtk_widget_show (iv->hscrollbar);
   gtk_widget_show (iv->vscrollbar);
   gtk_widget_show (iv->nav_button);

   iv->show_scrollbar = TRUE;
}


void
imageview_hide_scrollbar (ImageView *iv)
{
   g_return_if_fail (iv);
   g_return_if_fail (iv->hscrollbar);
   g_return_if_fail (iv->vscrollbar);

   gtk_widget_hide (iv->hscrollbar);
   gtk_widget_hide (iv->vscrollbar);
   gtk_widget_hide (iv->nav_button);

   iv->show_scrollbar = FALSE;
}


void
imageview_set_progressbar (ImageView *iv, GtkWidget *progressbar)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   iv->progressbar = progressbar;
}


void
imageview_set_player_visible (ImageView *iv,
                              ImageViewPlayerVisibleType type)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   switch (type) {
   case ImageViewPlayerVisibleHide:
      gtk_widget_hide (iv->player_container);
      iv->player_visible = ImageViewPlayerVisibleHide;
      break;
   case ImageViewPlayerVisibleShow:
      gtk_widget_show (iv->player_container);
      iv->player_visible = ImageViewPlayerVisibleShow;
      break;
   case ImageViewPlayerVisibleAuto:
   default:
      if (imageview_is_playable (iv)) {
         gtk_widget_show (iv->player_container);
      } else {
         gtk_widget_hide (iv->player_container);
      }
      iv->player_visible = ImageViewPlayerVisibleAuto;
      break;
   }
}


void
imageview_open_navwin (ImageView *iv, gfloat x_root, gfloat y_root)
{
   GimvImage *image;
   GdkPixmap *pixmap;
   GdkBitmap *mask;
   gint src_width, src_height, dest_width, dest_height;

   g_return_if_fail (iv);
   if (!iv->priv->pixmap) return;

   /* get pixmap for navigator */
   gdk_window_get_size (iv->priv->pixmap, &src_width, &src_height);
   image = gimv_image_create_from_drawable (iv->priv->pixmap, 0, 0,
                                            src_width, src_height);
   g_return_if_fail (image);

   if (src_width > src_height) {
      dest_width  = NAV_WIN_SIZE;
      dest_height = src_height * NAV_WIN_SIZE / src_width;
   } else {
      dest_height = NAV_WIN_SIZE;
      dest_width  = src_width * NAV_WIN_SIZE / src_height;
   }

   gimv_image_scale_get_pixmap (image, dest_width, dest_height,
                                &pixmap, &mask);
   if (!pixmap) goto ERROR;

   /* open navigate window */
   navwin_create (iv, x_root, y_root, pixmap, mask);

   /* free */
   gdk_pixmap_unref (pixmap);

ERROR:
   gimv_image_unref (image);
}


void
imageview_set_fullscreen (ImageView *iv, GtkWindow *fullscreen)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));
   g_return_if_fail (GTK_IS_WINDOW (fullscreen));

   if (iv->priv->fullscreen) return;
   iv->priv->fullscreen = fullscreen;
   gtk_widget_reparent (iv->draw_area, GTK_WIDGET (iv->priv->fullscreen));
}


void
imageview_unset_fullscreen (ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->priv->fullscreen) return;

   gtk_widget_reparent (iv->draw_area, iv->table);
   iv->priv->fullscreen = NULL;
}



/*****************************************************************************
 *
 *  loading functions
 *
 *****************************************************************************/
void
imageview_free_image_buf (ImageView *iv)
{
   if (!iv)
      return;

   if (!iv->image) return;
   if (iv->buffer) return;

#warning FIXME!!
   if (image_info_is_animation (iv->info)
       || image_info_is_movie (iv->info)
       || image_info_is_audio (iv->info))
   {
      return;
   }

   gimv_image_unref (iv->image);
   iv->image = NULL;
}
static gint
progress_timeout (gpointer data)
{
   gfloat new_val;
   GtkAdjustment *adj;

   adj = GTK_PROGRESS (data)->adjustment;

   new_val = adj->value + 1;
   if (new_val > adj->upper)
      new_val = adj->lower;

   gtk_progress_set_value (GTK_PROGRESS (data), new_val);

   return (TRUE);
}


static void
cb_loader_progress_update (ImageLoader *loader, ImageView *iv)
{
   while (gtk_events_pending()) gtk_main_iteration();
}


static void
cb_loader_load_end (ImageLoader *loader, ImageView *iv)
{
   GimvImage *image, *rgb_image;

   g_return_if_fail (IS_IMAGE_LOADER (loader));
   g_return_if_fail (IS_IMAGEVIEW (iv));

   image = image_loader_get_image (loader);
   if (!image) goto ERROR;
   gimv_image_ref (image);
   image_loader_unref_image (loader);

   /* FIXME */
   if (gimv_image_has_alpha (image) && !gimv_image_is_anim (image)) {
      gint bg_r = 255, bg_g = 255, bg_b = 255;
      GtkStyle *style;

      style = gtk_widget_get_style (iv->draw_area);
      bg_r = style->bg[GTK_STATE_NORMAL].red   / 256;
      bg_g = style->bg[GTK_STATE_NORMAL].green / 256;
      bg_b = style->bg[GTK_STATE_NORMAL].blue  / 256;
      
      rgb_image = gimv_image_rgba2rgb (image,
                                       bg_r, bg_g, bg_b,
                                       iv->ignore_alpha);
      if (rgb_image) {
         gimv_image_unref (image);
         image = rgb_image;
      }
   }

   if (!g_list_find (ImageViewList, iv)) {
      gimv_image_unref (image);
      goto ERROR;
   } else {
      iv->image = image;
   }

   imageview_rotate_render (iv, iv->rotate);

ERROR:
   gtk_signal_disconnect_by_func (GTK_OBJECT (iv->loader),
                                  GTK_SIGNAL_FUNC (cb_loader_progress_update),
                                  iv);
   gtk_signal_disconnect_by_func (GTK_OBJECT (iv->loader),
                                  GTK_SIGNAL_FUNC (cb_loader_load_end),
                                  iv);

   iv->priv->loader_progress_update_signal_id = 0;
   iv->priv->loader_load_end_signal_id        = 0;

   gtk_signal_emit (GTK_OBJECT(iv),
                    imageview_signals[LOAD_END_SIGNAL],
                    iv->info, FALSE);
}


static gboolean
imageview_need_load (ImageView *iv)
{
   gint width, height;

   if (!iv->image) return TRUE;

#if IMAGE_VIEW_ENABLE_SCALABLE_LOAD

   if (iv->rotate == 0 || iv->rotate == 2) {
      width  = gimv_image_width (iv->image);
      height = gimv_image_height (iv->image);
   } else {
      width  = gimv_image_height (iv->image);
      height = gimv_image_width  (iv->image);
   }

   if ( width == iv->info->width
       &&  height == iv->info->height)
   {
      /* full scale image was already loaded */
      return FALSE;

   } else {
      gint req_width, req_height;

      imageview_get_request_size (iv, &req_width, &req_height);

      if (req_width >= 0 && req_height >= 0
          && width  >= req_width
          && height >= req_height)
      {
         /* the image is bigger than request size */
         return FALSE;
      }
   }

   return TRUE;

#endif /* IMAGE_VIEW_ENABLE_SCALABLE_LOAD */

   return FALSE;
}


static void
imageview_load_image_buf_start (ImageView *iv)
{
   const gchar *filename;
   gint req_width, req_height;

   g_return_if_fail (iv);

   if (!iv->info) return;
   if (!g_list_find (ImageViewList, iv)) return;

   if (!imageview_need_load (iv)) return;

#warning FIXME!!
   if (image_info_is_archive (iv->info)
       || image_info_is_movie (iv->info)
       || image_info_is_audio (iv->info))
   {
      return;
   }

   filename = image_info_get_path (iv->info);
   if (!filename || !*filename) return;

   gtk_signal_emit (GTK_OBJECT(iv),
                    imageview_signals[LOAD_START_SIGNAL],
                    iv->info);

   if (image_info_is_in_archive (iv->info)) {
      guint timer = 0;

      /* set progress bar */
      if (iv->progressbar) {
         gtk_progress_set_activity_mode (GTK_PROGRESS (iv->progressbar), TRUE);
         timer = gtk_timeout_add (50,
                                  (GtkFunction) progress_timeout,
                                  iv->progressbar);
         gtk_grab_add (iv->progressbar);
      }

      /* extract */
      image_info_extract_archive (iv->info);

      /* unset progress bar */
      if (iv->progressbar) {
         gtk_timeout_remove (timer);
         gtk_progress_set_activity_mode (GTK_PROGRESS (iv->progressbar), FALSE);
         gtk_progress_bar_update (GTK_PROGRESS_BAR(iv->progressbar), 0.0);
         gtk_grab_remove (iv->progressbar);
      }
   }

   /* load image buf */
   /* iv->loader->flags |= IMAGE_LOADER_DEBUG_FLAG; */
   if (iv->priv->loader_progress_update_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv->loader),
                             iv->priv->loader_progress_update_signal_id);
   iv->priv->loader_progress_update_signal_id = 
      gtk_signal_connect (GTK_OBJECT (iv->loader), "progress_update",
                          GTK_SIGNAL_FUNC (cb_loader_progress_update),
                             iv);
   if (iv->priv->loader_load_end_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv->loader),
                             iv->priv->loader_load_end_signal_id);
   iv->priv->loader_load_end_signal_id = 
      gtk_signal_connect (GTK_OBJECT (iv->loader), "load_end",
                          GTK_SIGNAL_FUNC (cb_loader_load_end),
                          iv);

   image_loader_set_image_info (iv->loader, iv->info);
   image_loader_set_as_animation (iv->loader, TRUE);

#if IMAGE_VIEW_ENABLE_SCALABLE_LOAD
   imageview_get_request_size (iv, &req_width, &req_height);
   image_loader_set_size_request (iv->loader, req_width, req_height);
#endif

   image_loader_load_start (iv->loader);
}


static void
cb_loader_load_restart (ImageLoader *loader, ImageView *iv)
{
   if (iv->priv->loader_load_end_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv->loader),
                             iv->priv->loader_load_end_signal_id);
   iv->priv->loader_load_end_signal_id = 0;
   imageview_load_image_buf_start (iv);
}


void
imageview_load_image_buf (ImageView *iv)
{
   if (image_loader_is_loading (iv->loader)) {
      if (iv->info == iv->loader->info) return;

      imageview_cancel_loading (iv);

      iv->priv->loader_load_end_signal_id
         = gtk_signal_connect (GTK_OBJECT (iv->loader), "load_end",
                               GTK_SIGNAL_FUNC (cb_loader_load_restart),
                               iv);
   } else {
      imageview_load_image_buf_start (iv);
   }
}


gboolean
imageview_is_loading (ImageView *iv)
{
   g_return_val_if_fail (IS_IMAGEVIEW (iv), FALSE);
   g_return_val_if_fail (iv->loader, FALSE);

   return image_loader_is_loading (iv->loader);
}


void
imageview_cancel_loading (ImageView *iv)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!image_loader_is_loading (iv->loader)) return;

   image_loader_load_stop (iv->loader);

   if (iv->priv->loader_progress_update_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv->loader),
                             iv->priv->loader_progress_update_signal_id);
   iv->priv->loader_progress_update_signal_id = 0;

   if (iv->priv->loader_load_end_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv->loader),
                             iv->priv->loader_load_end_signal_id);
   iv->priv->loader_load_end_signal_id = 0;

   /*
   gtk_signal_emit (GTK_OBJECT(iv),
                    imageview_signals[LOAD_END_SIGNAL],
                    iv->info, TRUE);
   */
}



/*****************************************************************************
 *
 *   scalable interface functions
 *
 *****************************************************************************/
void
imageview_zoom_image (ImageView *iv, ImageViewZoomType zoom,
                      gfloat x_scale, gfloat y_scale)
{
   gint cx_pos, cy_pos, fwidth, fheight;
   gfloat src_x_scale, src_y_scale;

   g_return_if_fail (iv);

   src_x_scale = iv->x_scale;
   src_y_scale = iv->y_scale;

   switch (zoom) {
   case IMAGEVIEW_ZOOM_IN:
      if (iv->x_scale < IMG_MAX_SCALE && iv->y_scale < IMG_MAX_SCALE) {
         iv->x_scale = iv->x_scale + IMG_MIN_SCALE;
         iv->y_scale = iv->y_scale + IMG_MIN_SCALE;
      }
      break;
   case IMAGEVIEW_ZOOM_OUT:
      if (iv->x_scale > IMG_MIN_SCALE && iv->y_scale > IMG_MIN_SCALE) {
         iv->x_scale = iv->x_scale - IMG_MIN_SCALE;
         iv->y_scale = iv->y_scale - IMG_MIN_SCALE;
      }
      break;
   case IMAGEVIEW_ZOOM_FIT:
      iv->fit_to_frame = TRUE;
      break;
   case IMAGEVIEW_ZOOM_10:
      iv->x_scale = iv->y_scale =  10;
      break;
   case IMAGEVIEW_ZOOM_25:
      iv->x_scale = iv->y_scale =  25;
      break;
   case IMAGEVIEW_ZOOM_50:
      iv->x_scale = iv->y_scale =  50;
      break;
   case IMAGEVIEW_ZOOM_75:
      iv->x_scale = iv->y_scale =  75;
      break;
   case IMAGEVIEW_ZOOM_100:
      iv->x_scale = iv->y_scale = 100;
      break;
   case IMAGEVIEW_ZOOM_125:
      iv->x_scale = iv->y_scale = 125;
      break;
   case IMAGEVIEW_ZOOM_150:
      iv->x_scale = iv->y_scale = 150;
      break;
   case IMAGEVIEW_ZOOM_175:
      iv->x_scale = iv->y_scale = 175;
      break;
   case IMAGEVIEW_ZOOM_200:
      iv->x_scale = iv->y_scale = 200;
      break;	 
   case IMAGEVIEW_ZOOM_300:
      iv->x_scale = iv->y_scale = 300;
      break;	 
   case IMAGEVIEW_ZOOM_FREE:
      iv->x_scale = x_scale;
      iv->y_scale = y_scale;
      break;
   default:
      break;
   }

   imageview_get_image_frame_size (iv, &fwidth, &fheight);

   cx_pos = (iv->priv->x_pos - fwidth  / 2) * iv->x_scale / src_x_scale;
   cy_pos = (iv->priv->y_pos - fheight / 2) * iv->y_scale / src_y_scale;

   iv->priv->x_pos = cx_pos + fwidth  / 2;
   iv->priv->y_pos = cy_pos + fheight / 2;

   if (zoom != IMAGEVIEW_ZOOM_FIT)
      iv->fit_to_frame = FALSE;

   imageview_show_image (iv);
}


gboolean
imageview_get_image_size (ImageView   *iv,
                          gint        *width,
                          gint        *height)
{
   g_return_val_if_fail (IS_IMAGEVIEW (iv), FALSE);

   if (width)
      *width = iv->priv->width;
   if (height)
      *height = iv->priv->height;

   return TRUE;
}


static void
imageview_get_request_size (ImageView *iv, gint *width_ret, gint *height_ret)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));
   g_return_if_fail (width_ret && height_ret);

   *width_ret  = -1;
   *height_ret = -1;

   if (iv->fit_to_frame) {
      gint fwidth, fheight;

      imageview_get_image_frame_size (iv, &fheight, &fwidth);

      if (iv->rotate == 0 && iv->rotate == 2) {
         *width_ret  = fwidth;
         *height_ret = fheight;
      } else {
         *width_ret  = fheight;
         *height_ret = fwidth;
      }
      return;

   } else if (iv->priv->width > MIN_IMAGE_WIDTH && iv->priv->height > MIN_IMAGE_HEIGHT) {
      *width_ret  = iv->info->width  * iv->x_scale / 100.0;
      *height_ret = iv->info->height * iv->y_scale / 100.0;
      return;
   } else {
      /*
       * the image was changed, so the image size may be unknown yet.
       * (iv->info will know it, but (currently) we don't trust it yet. )
       */
   }
}



/*****************************************************************************
 *
 *   rotatable interface functions
 *
 *****************************************************************************/
static void
imageview_rotate_render (ImageView *iv, ImageViewAngle angle)
{
   switch (angle) {
   case IMAGEVIEW_ROTATE_90:
      iv->image = gimv_image_rotate_90 (iv->image, TRUE);
      break;
   case IMAGEVIEW_ROTATE_180:
      iv->image = gimv_image_rotate_180 (iv->image);
      break;
   case IMAGEVIEW_ROTATE_270:
      iv->image = gimv_image_rotate_90 (iv->image, FALSE);
      break;
   default:
      break;
   }
}


typedef struct RotateData_Tag {
   ImageViewAngle angle;
} RotateData;


static void
cb_imageview_rotate_load_end (ImageView *iv, ImageInfo *info,
                              gboolean cancel, gpointer user_data)
{
   RotateData *data = user_data;
   ImageViewAngle angle = data->angle, rotate_angle;
   gboolean can_draw;
   gchar *path, *cache;

   if (iv->priv->load_end_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv), iv->priv->load_end_signal_id);
   iv->priv->load_end_signal_id = 0;

   if (cancel) return;

   can_draw = check_can_draw_image (iv);
   if (!can_draw) goto func_end;

   /* rotate image */
   rotate_angle = (angle - iv->rotate) % 4;
   if (rotate_angle < 0)
      rotate_angle = rotate_angle + 4;

   iv->rotate = angle;
   imageview_rotate_render (iv, rotate_angle);

   /* set image size */
   imageview_calc_image_size (iv);

   /* image rendering */
   gimv_image_free_pixmap_and_mask (iv->priv->pixmap, iv->priv->mask);
   gimv_image_scale_get_pixmap (iv->image,
                                iv->priv->width, iv->priv->height,
                                &iv->priv->pixmap, &iv->priv->mask);

   /* reset image geometory */
   imageview_adjust_pos_in_frame (iv, TRUE);

   /* draw image */
   imageview_draw_image (iv);

   /* create thumbnail if not exist */
   path  = image_info_get_path_with_archive (iv->info);
   cache = thumbnail_find_thumbcache(path, NULL);
   if (path && *path && !cache) {
      imageview_create_thumbnail (iv);
   }
   g_free (path);
   g_free (cache);
   path  = NULL;
   cache = NULL;

   imageview_free_image_buf (iv);

func_end:
   if (iv->priv->load_end_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv), iv->priv->load_end_signal_id);
   iv->priv->load_end_signal_id = 0;

   gtk_signal_emit (GTK_OBJECT(iv), imageview_signals[RENDERED_SIGNAL]);
}


void
imageview_rotate_image (ImageView *iv, ImageViewAngle angle)
{
   RotateData *data;

   if (iv->priv->load_end_signal_id)
      gtk_signal_disconnect (GTK_OBJECT (iv), iv->priv->load_end_signal_id);

   data = g_new0 (RotateData, 1);
   data->angle = angle;

   if (imageview_need_load (iv)) {
      iv->priv->load_end_signal_id
         = gtk_signal_connect_full (GTK_OBJECT (iv), "load_end",
                                    GTK_SIGNAL_FUNC (cb_imageview_rotate_load_end),
                                    NULL,
                                    data, (GtkDestroyNotify) g_free,
                                    FALSE, FALSE);

      imageview_load_image_buf (iv);
   } else {
      cb_imageview_rotate_load_end (iv, iv->info, FALSE, data);
      g_free (data);
   }
}


void
imageview_rotate_ccw (ImageView *iv)
{
   guint angle;

   g_return_if_fail (iv);

   /* convert to absolute angle */
   angle = (iv->rotate + 1) % 4;

   imageview_rotate_image (iv, angle);
}


void
imageview_rotate_cw (ImageView *iv)
{
   guint angle;

   g_return_if_fail (iv);

   /* convert to absolute angle */
   angle = (iv->rotate + 3) % 4;

   imageview_rotate_image (iv, angle);
}



/*****************************************************************************
 *
 *   scrollable interface functions
 *
 *****************************************************************************/
void
imageview_get_image_frame_size (ImageView *iv, gint *width, gint *height)
{
   g_return_if_fail (width && height);

   *width = *height = 0;

   g_return_if_fail (iv);

   *width  = iv->draw_area->allocation.width;
   *height = iv->draw_area->allocation.height;
}


gboolean
imageview_get_view_position (ImageView *iv, gint *x, gint *y)
{
   g_return_val_if_fail (iv, FALSE);
   g_return_val_if_fail (x && y, FALSE);

   *x = 0 - iv->priv->x_pos;
   *y = 0 - iv->priv->y_pos;

   return TRUE;
}


void
imageview_moveto (ImageView *iv, gint x, gint y)
{
   g_return_if_fail (iv);

   iv->priv->x_pos = 0 - x;
   iv->priv->y_pos = 0 - y;

   imageview_adjust_pos_in_frame (iv, TRUE);

   imageview_draw_image (iv);
}


void
imageview_reset_scrollbar (ImageView *iv)
{
   g_return_if_fail (iv);
   g_return_if_fail (iv->draw_area);
   g_return_if_fail (iv->hadj);
   g_return_if_fail (iv->vadj);

   /* horizontal */
   if (iv->priv->x_pos < 0)
      iv->hadj->value = 0 - iv->priv->x_pos;
   else
      iv->hadj->value = 0;

   if (iv->priv->width > iv->draw_area->allocation.width)
      iv->hadj->upper = iv->priv->width;
   else
      iv->hadj->upper = iv->draw_area->allocation.width;
   iv->hadj->page_size = iv->draw_area->allocation.width;

   /* vertical */
   if (iv->priv->y_pos < 0)
      iv->vadj->value = 0 - iv->priv->y_pos;
   else
      iv->vadj->value = 0;

   if (iv->priv->height > iv->draw_area->allocation.height)
      iv->vadj->upper = iv->priv->height;
   else
      iv->vadj->upper = iv->draw_area->allocation.height;
   iv->vadj->page_size = iv->draw_area->allocation.height;

   move_scrollbar_by_user = FALSE;

   gtk_signal_emit_by_name (GTK_OBJECT(iv->hadj), "changed");
   gtk_signal_emit_by_name (GTK_OBJECT(iv->vadj), "changed");

   move_scrollbar_by_user = TRUE;
}



/*****************************************************************************
 *
 *   playable interface functions
 *
 *****************************************************************************/
gboolean
imageview_is_playable (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_val_if_fail (IS_IMAGEVIEW (iv), FALSE);

   if (!iv->info) return FALSE;
   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return FALSE;

   playable = iv->draw_area_funcs->playable;

   if (!playable->is_playable_fn) return FALSE;

   return playable->is_playable_fn (iv, iv->info);
}


void
imageview_playable_play (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;
   playable = iv->draw_area_funcs->playable;

   if (imageview_is_playable (iv)) {
      g_return_if_fail (playable->play_fn);
      playable->play_fn (iv);
   }
}


void
imageview_playable_stop (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;

   playable = iv->draw_area_funcs->playable;

   if (!playable->stop_fn) return;
   playable->stop_fn (iv);
}


void
imageview_playable_pause (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;

   playable = iv->draw_area_funcs->playable;

   if (!playable->pause_fn) return;
   playable->pause_fn (iv);
}


void
imageview_playable_forward (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;

   playable = iv->draw_area_funcs->playable;

   if (!playable->forward_fn) return;
   playable->forward_fn (iv);
}


void
imageview_playable_reverse (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;

   playable = iv->draw_area_funcs->playable;

   if (!playable->reverse_fn) return;
   playable->reverse_fn (iv);
}


void
imageview_playable_seek (ImageView *iv, guint pos)
{
   ImageViewPlayableInterFace *playable;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;

   playable = iv->draw_area_funcs->playable;

   if (!playable->seek_fn) return;
   playable->seek_fn (iv, pos);
}


void
imageview_playable_eject (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;


   g_return_if_fail (IS_IMAGEVIEW (iv));

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;

   playable = iv->draw_area_funcs->playable;

   if (!playable->eject_fn) return;
   playable->eject_fn (iv);
}


ImageViewPlayableStatus
imageview_playable_get_status (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_val_if_fail (IS_IMAGEVIEW (iv), ImageViewPlayableDisable);

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable)
      return ImageViewPlayableDisable;

   playable = iv->draw_area_funcs->playable;

   if (!playable->get_status_fn) return ImageViewPlayableDisable;
   return playable->get_status_fn (iv);
}


guint
imageview_playable_get_length (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_val_if_fail (IS_IMAGEVIEW (iv), 0);

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return 0;

   playable = iv->draw_area_funcs->playable;

   if (!playable->get_length_fn) return 0;
   return playable->get_length_fn (iv);
}


guint
imageview_playable_get_position (ImageView *iv)
{
   ImageViewPlayableInterFace *playable;

   g_return_val_if_fail (IS_IMAGEVIEW (iv), 0);

   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return 0;

   playable = iv->draw_area_funcs->playable;

   if (!playable->get_position_fn) return 0;
   return playable->get_position_fn (iv);
}



/*****************************************************************************
 *
 *   list interface functions
 *
 *****************************************************************************/
void
imageview_set_list (ImageView       *iv,
                    GList           *list,
                    GList           *current,
                    gpointer         list_owner,
                    ImageViewNextFn  next_fn,
                    ImageViewPrevFn  prev_fn,
                    ImageViewNthFn   nth_fn,
                    ImageViewRemoveListFn remove_list_fn,
                    gpointer         list_fn_user_data)
{
   g_return_if_fail (iv);
   g_return_if_fail (list);
   g_return_if_fail (current);

   if (!g_list_find (ImageViewList, iv)) return;

   if (iv->priv->image_list)
      imageview_remove_list (iv, iv->priv->image_list->owner);

   iv->priv->image_list = g_new0 (ImageViewImageList, 1);

   iv->priv->image_list->list = list;
   if (!current)
      iv->priv->image_list->current = list;
   else
      iv->priv->image_list->current = current;

   iv->priv->image_list->owner             = list_owner;
   iv->priv->image_list->next_fn           = next_fn;
   iv->priv->image_list->prev_fn           = prev_fn;
   iv->priv->image_list->nth_fn            = nth_fn;
   iv->priv->image_list->remove_list_fn    = remove_list_fn;
   iv->priv->image_list->list_fn_user_data = list_fn_user_data;

   gtk_signal_emit (GTK_OBJECT(iv), imageview_signals[SET_LIST_SIGNAL]);
}


void
imageview_remove_list (ImageView *iv, gpointer list_owner)
{
   g_return_if_fail (iv);

   if (!iv->priv->image_list) return;

   if (iv->priv->image_list->remove_list_fn
       && iv->priv->image_list->owner == list_owner)
   {
      iv->priv->image_list->remove_list_fn
         (iv,
          iv->priv->image_list->owner,
          iv->priv->image_list->list_fn_user_data);
   }

   g_free (iv->priv->image_list);
   iv->priv->image_list = NULL;

   gtk_signal_emit (GTK_OBJECT(iv), imageview_signals[UNSET_LIST_SIGNAL]);
}


gint
imageview_image_list_length (ImageView *iv)
{
   g_return_val_if_fail (IS_IMAGEVIEW (iv), 0);
   if (!iv->priv->image_list) return 0;

   return g_list_length (iv->priv->image_list->list);
}


gint
imageview_image_list_position (ImageView *iv)
{
   g_return_val_if_fail (IS_IMAGEVIEW (iv), 0);
   if (!iv->priv->image_list) return 0;

   return g_list_position (iv->priv->image_list->list,
                           iv->priv->image_list->current);
}


GList *
imageview_image_list_current (ImageView *iv)
{
   g_return_val_if_fail (IS_IMAGEVIEW (iv), NULL);
   if (!iv->priv->image_list) return NULL;

   return iv->priv->image_list->current;
}


static gint
idle_imageview_next (gpointer data)
{
   ImageView *iv = data;

   g_return_val_if_fail (iv, FALSE);
   if (!iv->priv->image_list) return FALSE;
   if (!iv->priv->image_list->next_fn) return FALSE;

   iv->priv->image_list->current
      = iv->priv->image_list->next_fn (iv,
                                       iv->priv->image_list->owner,
                                       iv->priv->image_list->current,
                                       iv->priv->image_list->list_fn_user_data);

   return FALSE;
}


void
imageview_next (ImageView *iv)
{
   g_return_if_fail (iv);
   gtk_idle_add (idle_imageview_next, iv);
}


static gint
idle_imageview_prev (gpointer data)
{
   ImageView *iv = data;

   g_return_val_if_fail (iv, FALSE);
   if (!iv->priv->image_list) return FALSE;
   if (!iv->priv->image_list->prev_fn) return FALSE;

   iv->priv->image_list->current
      = iv->priv->image_list->prev_fn (iv,
                                       iv->priv->image_list->owner,
                                       iv->priv->image_list->current,
                                       iv->priv->image_list->list_fn_user_data);

   return FALSE;
}


void
imageview_prev (ImageView *iv)
{
   g_return_if_fail (iv);

   gtk_idle_add (idle_imageview_prev, iv);
}


typedef struct NthFnData_Tag {
   ImageView *iv;
   gint       nth;
} NthFnData;


static gint
idle_imageview_nth (gpointer data)
{
   NthFnData *nth_fn_data = data;
   ImageView *iv = data;
   gint nth;

   g_return_val_if_fail (nth_fn_data, FALSE);

   iv = nth_fn_data->iv;
   nth = nth_fn_data->nth;

   g_free (nth_fn_data);
   nth_fn_data = NULL;
   data = NULL;

   g_return_val_if_fail (iv, FALSE);
   if (!iv->priv->image_list) return FALSE;
   if (!iv->priv->image_list->nth_fn) return FALSE;

   iv->priv->image_list->current
      = iv->priv->image_list->nth_fn (iv,
                                      iv->priv->image_list->owner,
                                      iv->priv->image_list->list,
                                      nth,
                                      iv->priv->image_list->list_fn_user_data);

   return FALSE;
}


void
imageview_nth (ImageView *iv, guint nth)
{
   NthFnData *nth_fn_data;

   g_return_if_fail (iv);

   nth_fn_data = g_new (NthFnData, 1);
   nth_fn_data->iv  = iv;
   nth_fn_data->nth = nth;

   gtk_idle_add (idle_imageview_nth, nth_fn_data);
}


static GList *
next_image (ImageView *iv,
            gpointer list_owner,
            GList *current,
            gpointer data)
{
   GList *next, *node;

   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (iv == list_owner, NULL);
   g_return_val_if_fail (current, NULL);

   if (!iv->priv->image_list) return NULL;

   node = g_list_find (iv->priv->image_list->list, current->data);
   g_return_val_if_fail (node, NULL);

   next = g_list_next (current);
   if (!next)
      next = current;
   g_return_val_if_fail (next, NULL);

   if (next != current) {
      imageview_change_image (iv, next->data);
   }

   return next;
}


static GList *
prev_image (ImageView *iv,
            gpointer list_owner,
            GList *current,
            gpointer data)
{
   GList *prev, *node;

   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (iv == list_owner, NULL);
   g_return_val_if_fail (current, NULL);

   if (!iv->priv->image_list) return NULL;

   node = g_list_find (iv->priv->image_list->list, current->data);
   g_return_val_if_fail (node, NULL);

   prev = g_list_previous (current);
   if (!prev)
      prev = current;
   g_return_val_if_fail (prev, NULL);

   if (prev != current) {
      imageview_change_image (iv, prev->data);
   }

   return prev;
}


static GList *
nth_image (ImageView *iv,
           gpointer list_owner,
           GList *list,
           guint nth,
           gpointer data)
{
   GList *node;

   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (iv == list_owner, NULL);

   if (!iv->priv->image_list) return NULL;

   node = g_list_nth (iv->priv->image_list->list, nth);
   g_return_val_if_fail (node, NULL);

   imageview_change_image (iv, node->data);

   return node;
}


static void
remove_list (ImageView *iv, gpointer list_owner, gpointer data)
{
   g_return_if_fail (iv == list_owner);

   if (!iv->priv->image_list) return;

   g_list_foreach (iv->priv->image_list->list, (GFunc) image_info_unref, NULL);
   g_list_free (iv->priv->image_list->list);
   iv->priv->image_list->list = NULL;
   iv->priv->image_list->current = NULL;
}


void
imageview_set_list_self (ImageView *iv, GList *list, GList *current)
{
   g_return_if_fail (IS_IMAGEVIEW (iv));
   g_return_if_fail (list);
 
   if (!current || !g_list_find (list, current->data))
      current = list;

   imageview_set_list (iv, list, current, iv,
                       next_image,
                       prev_image,
                       nth_image,
                       remove_list,
                       iv);
}


gboolean
imageview_has_list (ImageView *iv)
{
   if (iv->priv->image_list) {
      return TRUE;
   } else {
      return FALSE;
   }
}

void
imageview_playable_set_status (ImageView *iv, ImageViewPlayableStatus status)
{
   ImageViewPlayableInterFace *playable;
   GtkWidget *play, *stop, *pause, *forward, *reverse, *eject;
   GtkItemFactory *ifactory;

   g_return_if_fail (IS_IMAGEVIEW (iv));
   if (!iv->draw_area_funcs || !iv->draw_area_funcs->playable) return;

   playable = iv->draw_area_funcs->playable;

   if (iv->movie_menu && GTK_IS_WIDGET (iv->movie_menu)) {
      ifactory = gtk_item_factory_from_widget (iv->movie_menu);
      play     = gtk_item_factory_get_item (ifactory, "/Play");
      stop     = gtk_item_factory_get_item (ifactory, "/Stop");
      pause    = gtk_item_factory_get_item (ifactory, "/Pause");
      forward  = gtk_item_factory_get_item (ifactory, "/Forward");
      reverse  = gtk_item_factory_get_item (ifactory, "/Reverse");
      eject    = gtk_item_factory_get_item (ifactory, "/Eject");
   } else {
      play = stop = pause = forward = reverse = eject = NULL;
   }

   if (status == ImageViewPlayableDisable) {
      gimv_icon_stock_change_widget_icon  (iv->player.play_icon, "play");
      gtk_widget_set_sensitive (iv->player.play,    FALSE);
      gtk_widget_set_sensitive (iv->player.stop,    FALSE);
      gtk_widget_set_sensitive (iv->player.fw,      FALSE);
      gtk_widget_set_sensitive (iv->player.rw,      FALSE);
      gtk_widget_set_sensitive (iv->player.eject,   FALSE);
      gtk_widget_set_sensitive (iv->player.seekbar, FALSE);
      if (play)    gtk_widget_set_sensitive (play,    FALSE);
      if (stop)    gtk_widget_set_sensitive (stop,    FALSE);
      if (pause)   gtk_widget_set_sensitive (pause,   FALSE);
      if (forward) gtk_widget_set_sensitive (forward, FALSE);
      if (reverse) gtk_widget_set_sensitive (reverse, FALSE);
      if (eject)   gtk_widget_set_sensitive (eject, FALSE);
      imageview_playable_set_position (iv, 0.0);
      return;

   } else {
      gboolean enable;

      enable = playable->play_fn ? TRUE : FALSE;
      gtk_widget_set_sensitive (iv->player.play,  enable);
      if (play) gtk_widget_set_sensitive (play, enable);

      enable = playable->pause_fn ? TRUE : FALSE;
      if (pause) gtk_widget_set_sensitive (pause, enable);

      enable = playable->stop_fn ? TRUE : FALSE;
      gtk_widget_set_sensitive (iv->player.stop,  enable);
      if (stop) gtk_widget_set_sensitive (stop, enable);

      enable = playable->forward_fn ? TRUE : FALSE;
      gtk_widget_set_sensitive (iv->player.fw,    enable);
      if (forward) gtk_widget_set_sensitive (forward, enable);

      enable = playable->reverse_fn ? TRUE : FALSE;
      gtk_widget_set_sensitive (iv->player.rw,    enable);
      if (reverse) gtk_widget_set_sensitive (reverse, enable);

      enable = playable->eject_fn ? TRUE : FALSE;
      gtk_widget_set_sensitive (iv->player.eject, enable);
      if (eject) gtk_widget_set_sensitive (eject, enable);

      if (playable->is_seekable_fn
          && playable->is_seekable_fn (iv)
          && playable->seek_fn)
      {
         gtk_widget_set_sensitive (iv->player.seekbar, TRUE);
      }
   }

   switch (status) {
   case ImageViewPlayableStop:
      imageview_playable_set_position (iv, 0.0);
      gtk_widget_set_sensitive (iv->player.stop,    FALSE);
      gtk_widget_set_sensitive (iv->player.fw,      FALSE);
      gtk_widget_set_sensitive (iv->player.rw,      FALSE);
      if (stop)    gtk_widget_set_sensitive (stop,    FALSE);
      if (pause)   gtk_widget_set_sensitive (pause,   FALSE);
      if (forward) gtk_widget_set_sensitive (forward, FALSE);
      if (reverse) gtk_widget_set_sensitive (reverse, FALSE);
      break;
   case ImageViewPlayableForward:
   case ImageViewPlayableReverse:
      gtk_widget_set_sensitive (iv->player.fw,      FALSE);
      gtk_widget_set_sensitive (iv->player.rw,      FALSE);
      if (pause)   gtk_widget_set_sensitive (pause,   FALSE);
      if (forward) gtk_widget_set_sensitive (forward, FALSE);
      if (reverse) gtk_widget_set_sensitive (reverse, FALSE);
      break;
   case ImageViewPlayablePlay:
      if (!playable->pause_fn) {
         gtk_widget_set_sensitive (iv->player.play, FALSE);
         if (pause) gtk_widget_set_sensitive (pause, FALSE);
      }
      break;
   case ImageViewPlayablePause:
   default:
      break;
   }

   if (status == ImageViewPlayablePlay && playable->pause_fn) {
      gimv_icon_stock_change_widget_icon (iv->player.play_icon, "pause");
   } else {
      gimv_icon_stock_change_widget_icon (iv->player.play_icon, "play");
   }
}


void
imageview_playable_set_position (ImageView *iv, gfloat pos)
{
   GtkAdjustment *adj;

   g_return_if_fail (IS_IMAGEVIEW (iv));

   adj = gtk_range_get_adjustment (GTK_RANGE (iv->player.seekbar));

   if (!(iv->player_flags & ImageViewSeekBarDraggingFlag)) {
      adj->value = pos;
      gtk_signal_emit_by_name (GTK_OBJECT(adj), "value_changed"); 
   }

}
