/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: thumbnail.c,v 1.32.2.8 2003/05/18 15:10:37 makeinu Exp $
 */

#include <string.h>

#include "gimageview.h"
#include "gimv_image.h"
#include "fileload.h"
#include "fileutil.h"
#include "fr-archive.h"
#include "gfileutil.h"
#include "gimv_icon_stock.h"
#include "image_info.h"
#include "image_loader.h"
#include "prefs.h"
#include "thumbnail.h"
#include "thumbnail_support.h"
#include "thumbnail_view.h"


static GHashTable *loader_table = NULL;


/* private functions */
static void       store_thumbnail                (GimvImage      *image,
                                                  gint            thumbsize,
                                                  GdkPixmap     **pixmap,
                                                  GdkBitmap     **mask,
                                                  gint           *width,
                                                  gint           *height);
static void       create_thumbnail               (Thumbnail      *thumb,
                                                  gint            thumbsize,
                                                  gchar          *cache_type,
                                                  gboolean        save_cache);
static void       load_thumbnail_cache           (Thumbnail      *thumb,
                                                  gint            thumbsize);
static void       thumbnail_remove_mode_data     (Thumbnail *thumb);


static GHashTable *cache_type_labels;
static gchar *config_cache_read_string = NULL;
static GList *cache_read_list = NULL;


/******************************************************************************
 *
 *   Private Functions.
 *
 ******************************************************************************/
/* defined and initialized in thumbnail_view.c */
extern ThumbViewPlugin thumb_view_modes[];

static void
thumbnail_remove_mode_data (Thumbnail *thumb)
{
   GList *list;

   g_return_if_fail (thumb);

   list = thumbview_get_disp_mode_list ();
   for (;list; list = g_list_next (list)) {
      ThumbViewPlugin *view = list->data;

      if (view && view->thumbnail_data_remove_func)
         view->thumbnail_data_remove_func (thumb);
   }
}


static void
store_thumbnail (GimvImage *image,   gint thumbsize,
                 GdkPixmap **pixmap, GdkBitmap **mask,
                 gint *width_ret, gint *height_ret)
{
   gfloat scale;
   gint width = 0, height = 0;

   if (!image || !thumbsize) return;
   g_return_if_fail (pixmap && mask);

   if (width_ret)  *width_ret = -1;
   if (height_ret) *height_ret = -1;

   if (gimv_image_width (image) >= gimv_image_height (image)) {
      scale = (gfloat) thumbsize / (gfloat) gimv_image_width (image);
      width = thumbsize;
      height = gimv_image_height (image) * scale;
   } else {
      scale = (gfloat) thumbsize / (gfloat) gimv_image_height (image);
      width = gimv_image_width (image) * scale;
      height = thumbsize;
   }

   if (width > 0 && height > 0) {
      gimv_image_scale_get_pixmap (image, width, height, pixmap, mask);
      if (width_ret)  *width_ret = width;
      if (height_ret) *height_ret = height;
   }
}


static void
thumbnail_apply_cache_read_prefs ()
{
   gchar **labels;
   gint i;

   if (cache_read_list) {
      g_list_foreach (cache_read_list, (GFunc)g_free, NULL);
      g_list_free (cache_read_list);
      cache_read_list = NULL;
   }

   config_cache_read_string = conf.cache_read_list;
   if (!conf.cache_read_list) return;

   labels = g_strsplit (conf.cache_read_list, ",", -1);

   for (i = 0; labels[i]; i++) {
      cache_read_list = g_list_append (cache_read_list, g_strdup (labels[i]));
   }

   g_strfreev (labels);
}


static void
cb_loader_progress_update (ImageLoader *loader, Thumbnail *thumb)
{
   while (gtk_events_pending()) gtk_main_iteration();

   /* FIXME!! */
   if (thumb && thumb->thumb_view && thumb->thumb_view->progress
       && thumb->thumb_view->progress->status >= CANCEL)
   {
      thumbnail_load_stop (thumb);
   }
}


/*
 *  create_thumbnail:
 *     @ Render thumbnail from original image or thumbnail cache,
 *       and create GtkPixmap Widget.
 *     @ If save_cache equal TRUE and specified thumbnail type,
 *       call save_thumbnail_cache () to save thumbnail to disk.
 *
 *  thumb     : Pointer to Thumbnail struct.
 *  thumbsize : Size of thumbnail. Positive value means thumbnail size is
 *              determined by longer axis, and negative value means determined
 *              by longer axis.
 *  type      : Thumbnail cache type.
 *  Return    : Pointer to GtkPixmap widget(thumbnail).
 */
static void
create_thumbnail (Thumbnail *thumb, gint thumbsize,
                  gchar *cache_type, gboolean save_cache)
{
   ImageInfo *info;
   GimvImage *imdisp = NULL;
   gboolean has_cache = FALSE;
   ThumbCacheSaveInfo *save_info;
   gchar *filename;

   g_return_if_fail (thumb);
   g_return_if_fail (thumb->info);

   info = thumb->info;

   /* get filename */
   filename = image_info_get_path_with_archive (info);

   /* load cache if cache type is specified */
   if (cache_type && !save_cache) {
      GimvImage *imcache = NULL;
      gchar *cache_path;
      struct stat cache_st;

      cache_path = thumbsupport_get_thumb_cache_path (filename, cache_type);
      if (!cache_path) goto ERROR;
      if (!stat (cache_path, &cache_st)) {
         if (info->st.st_mtime > cache_st.st_mtime) {
            g_free (cache_path);
            goto ERROR;
         }
      } else {
         g_free (cache_path);
         goto ERROR;
      }
      g_free (cache_path);

      imcache = thumbsupport_load_thumbnail_cache (filename, cache_type, &save_info);
      if (imcache) {
         has_cache = TRUE;
         imdisp = imcache;
         if (save_info) {
            image_info_set_size (info,
                                 save_info->image_width,
                                 save_info->image_height);
            thumbsupport_save_info_delete (save_info);
            save_info = NULL;
         }
      }

   /* load original image */
   } else {
      ImageLoader *loader;
      GimvImage *image = NULL, *imcache = NULL;

      if (!loader_table)
         loader_table = g_hash_table_new (g_direct_hash, g_direct_equal);

      /* load image */
      loader = image_loader_new_with_image_info (info);
      if (loader) {
         g_hash_table_insert (loader_table, thumb, loader);

         gtk_signal_connect (GTK_OBJECT (loader), "progress_update",
                             GTK_SIGNAL_FUNC (cb_loader_progress_update),
                             thumb);
         image_loader_set_load_type (loader, IMAGE_LOADER_LOAD_THUMBNAIL);
         image_loader_set_size_request (loader, thumbsize, thumbsize);
         image_loader_set_as_animation (loader, FALSE);
         image_loader_load (loader);
         image = image_loader_get_image (loader);
         if (image)
            gimv_image_ref (image);
         image_loader_unref (loader);

         g_hash_table_remove (loader_table, thumb);
      }

      /* write cache to disk */
      if (image) {
         if (save_cache && cache_type
             && (!image_info_is_in_archive (info)
                 || !strcmp (cache_type, "GImageView")))
         {
            save_info = thumbsupport_save_info_new (info, NULL);
            imcache = thumbsupport_save_thumbnail_cache (filename, cache_type,
                                                         image, save_info);
            thumbsupport_save_info_delete (save_info);
         }
      }

      if (imcache) {
         has_cache = TRUE;
         imdisp = imcache;
         gimv_image_unref (image);
         image = NULL;
      } else {
         imdisp = image;
      }
   }

   if (imdisp) {
      /* resize to display thumbnail size */
      store_thumbnail (imdisp, thumbsize,
                       &thumb->thumbnail,
                       &thumb->thumbnail_mask,
                       &thumb->thumb_width,
                       &thumb->thumb_height);
      store_thumbnail (imdisp, ICON_SIZE,
                       &thumb->icon,
                       &thumb->icon_mask,
                       NULL,
                       NULL);

      gimv_image_unref  (imdisp);
      imdisp = NULL;
   }

   if (thumb->thumbnail && has_cache && cache_type) {
      gchar *label;

      if (!cache_type_labels)
         cache_type_labels = g_hash_table_new (g_str_hash, g_str_equal);
      label = g_hash_table_lookup (cache_type_labels, cache_type);
      if (!label) {
         label = g_strdup (cache_type);
         g_hash_table_insert (cache_type_labels, label, label);
      }
      thumb->cache_type = label;
   }

ERROR:
   g_free (filename);
}


/*
 *  load_thumbnail_cache:
 *     @ Create cache from thumbnail disk cache. 
 *
 *  thumb     : Pointer to Thumbnail struct.
 *  thumbsize : Size of thumbnail.
 *  Return    : Pointer to GtkPixmap widget(thumbnail).
 */
static void
load_thumbnail_cache (Thumbnail *thumb, gint thumbsize)
{
   GList *node;

   g_return_if_fail (thumb);

   if (!cache_read_list || config_cache_read_string != conf.cache_read_list)
      thumbnail_apply_cache_read_prefs ();

   node = cache_read_list;
   while (node) {
      gchar *cache_type = node->data;
      create_thumbnail (thumb, thumbsize, cache_type, FALSE);
      if (thumb->thumbnail) return;
      node = g_list_next (node);
   }
}


/******************************************************************************
 *
 *   Public Functions.
 *
 ******************************************************************************/
Thumbnail *
thumbnail_new (ImageInfo *info)
{
   Thumbnail *thumb;

   thumb = g_new0 (Thumbnail, 1);
   g_return_val_if_fail (thumb, NULL);
   g_return_val_if_fail (info,  NULL);

   thumb->info = image_info_ref (info);
   thumb->thumb_view = NULL;
   thumb->selected = FALSE;
   thumb->sync = FALSE;
   thumb->mode_data = g_hash_table_new (g_str_hash, g_str_equal);
   thumb->ref_count = 0;

   thumb->image          = NULL;
   thumb->thumbnail      = NULL;
   thumb->thumbnail_mask = NULL;
   thumb->icon           = NULL;
   thumb->icon_mask      = NULL;
   thumb->thumb_width    = -1;
   thumb->thumb_height   = -1;
   thumb->cache_type     = NULL;

   thumb = thumbnail_ref (thumb);

   /* image_info_add_thumbnail (info, thumb); */

   return thumb;
}


ThumbView *
thumbnail_get_parent_thumbview (Thumbnail *thumb)
{
   g_return_val_if_fail (thumb, NULL);

   return thumb->thumb_view;
}


void
thumbnail_set_parent_thumbview (Thumbnail *thumb, ThumbView *tv)
{
   g_return_if_fail (thumb);

   thumb->thumb_view = tv;
}


void
thumbnail_finalize (Thumbnail *thumb)
{
   g_return_if_fail (thumb);

   /* image_info_remove_thumbnail (thumb->info, thumb); */

   /* free thumbnail */
   if (thumb->image) {
      gimv_image_unref (thumb->image);
      thumb->image = NULL;
   }

   if (thumb->thumbnail) {
      gdk_pixmap_unref (thumb->thumbnail);
      thumb->thumbnail = NULL;
   }
   if (thumb->thumbnail_mask) {
      gdk_bitmap_unref (thumb->thumbnail_mask);
      thumb->thumbnail_mask = NULL;
   }
   if (thumb->icon) {
      gdk_pixmap_unref (thumb->icon);
      thumb->icon = NULL;
   }
   if (thumb->icon_mask) {
      gdk_bitmap_unref (thumb->icon_mask);
      thumb->icon_mask = NULL;
   }

   image_info_unref (thumb->info);
   thumb->info = NULL;
   thumbnail_remove_mode_data (thumb);
   g_hash_table_destroy (thumb->mode_data);
   g_free (thumb);
}


Thumbnail *
thumbnail_ref (Thumbnail *thumb)
{
   g_return_val_if_fail (thumb, NULL);

   thumb->ref_count++;

   return thumb;
}


void
thumbnail_unref (Thumbnail *thumb)
{
   g_return_if_fail (thumb);

   thumb->ref_count--;
   if (thumb->ref_count < 1)
      thumbnail_finalize (thumb);
}


/*
 *  thumbnail_create:
 *     @ Create thumbnail. If disk cache is not exist, create it automaticary.
 *
 *  thumb     : Pointer to Thumbnail struct.
 *  type      : Thumbnail load type
 *              (from cache or original image (or other method...?)).
 *  Return    : Pointer to GtkPixmap widget(thumbnail).
 */
gboolean
thumbnail_create (Thumbnail    *thumb,
                  gint          thumb_size,
                  ThumbLoadType type)
{
   /* free old thumbnail */
   if (thumb->thumbnail) {
      gdk_pixmap_unref (thumb->thumbnail);
      thumb->thumbnail = NULL; thumb->thumbnail_mask = NULL;
   }
   if (thumb->icon) {
      gdk_pixmap_unref (thumb->icon);
      thumb->icon = NULL; thumb->icon_mask = NULL;
   }

   /* if file is directory */
   if (image_info_is_dir (thumb->info))
   {
      GimvIcon *icon;

      icon = gimv_icon_stock_get_icon ("folder48");
      thumb->thumbnail      = gdk_pixmap_ref (icon->pixmap);
      thumb->thumbnail_mask = gdk_bitmap_ref (icon->mask);

      icon = gimv_icon_stock_get_icon ("folder");
      thumb->icon      = gdk_pixmap_ref (icon->pixmap);
      thumb->icon_mask = gdk_bitmap_ref (icon->mask);
   }

   /* if file is archive */
   if (!thumb->thumbnail /* && conf.detect_filetype_by_ext */
       &&  image_info_is_archive (thumb->info))
   {
      GimvIcon *icon;

      icon = gimv_icon_stock_get_icon ("archive");
      thumb->thumbnail      = gdk_pixmap_ref (icon->pixmap);
      thumb->thumbnail_mask = gdk_bitmap_ref (icon->mask);

      icon = gimv_icon_stock_get_icon ("small_archive");
      thumb->icon      = gdk_pixmap_ref (icon->pixmap);
      thumb->icon_mask = gdk_bitmap_ref (icon->mask);
   }

   /* load chache  */
   if (!thumb->thumbnail && type != CREATE_THUMB) {
      load_thumbnail_cache (thumb, thumb_size);
   }

#if 0
   if (!thumb->thumbnail  /* && conf.detect_filetype_by_ext */
       && image_info_is_movie (thumb->info))
   {
      return FALSE;
   }
#endif
   if (image_info_is_audio (thumb->info))
      return FALSE;

   /* create cache if not exist */
   if (!thumb->thumbnail
       && conf.cache_write_type && conf.cache_write_type[0] != '\0')
   {
      create_thumbnail (thumb, thumb_size,
                        conf.cache_write_type, TRUE);
   }

   /* create thumbnail from original image */
   if (!thumb->thumbnail) {
      create_thumbnail (thumb, thumb_size, NULL, FALSE);
   }

   if (!thumb->thumbnail)
      return FALSE;
   else
      return TRUE;
}


gboolean
thumbnail_is_loading (Thumbnail *thumb)
{
   if (!loader_table) return FALSE;

   if (g_hash_table_lookup (loader_table, thumb))
      return TRUE;

   return FALSE;
}


void
thumbnail_load_stop (Thumbnail *thumb)
{
   ImageLoader *loader;

   if (!loader_table) return;

   loader = g_hash_table_lookup (loader_table, thumb);

   if (loader)
      image_loader_load_stop (loader);
}


void
thumbnail_get_thumb (Thumbnail *thumb, GdkPixmap **pixmap, GdkBitmap **mask)
{
   g_return_if_fail (thumb);
   g_return_if_fail (pixmap && mask);

   *pixmap = thumb->thumbnail;
   *mask   = thumb->thumbnail_mask;
}


GtkWidget *
thumbnail_get_thumb_by_widget (Thumbnail *thumb)
{
   g_return_val_if_fail (thumb, NULL);
   if (!thumb->thumbnail) return NULL;

#ifdef USE_GTK2
   return gtk_image_new_from_pixmap (thumb->thumbnail, thumb->thumbnail_mask);
#else /* USE_GTK2 */
   return gtk_pixmap_new (thumb->thumbnail, thumb->thumbnail_mask);
#endif /* USE_GTK2 */
}


void
thumbnail_get_icon (Thumbnail *thumb, GdkPixmap **pixmap, GdkBitmap **mask)
{
   g_return_if_fail (thumb);
   g_return_if_fail (pixmap && mask);

   *pixmap = thumb->icon;
   *mask   = thumb->icon_mask;
}


GtkWidget *
thumbnail_get_icon_by_widget (Thumbnail *thumb)
{
   g_return_val_if_fail (thumb, NULL);
   if (!thumb->icon) return NULL;

#ifdef USE_GTK2
   return gtk_image_new_from_pixmap (thumb->thumbnail, thumb->thumbnail_mask);
#else /* USE_GTK2 */
   return gtk_pixmap_new (thumb->icon, thumb->icon_mask);
#endif /* USE_GTK2 */
}


gchar *
thumbnail_get_cache_type (Thumbnail *thumb)
{
   g_return_val_if_fail (thumb, NULL);

   return thumb->cache_type;
}


gboolean
thumbnail_has_thumbnail (Thumbnail *thumb)
{
   g_return_val_if_fail (thumb, FALSE);

   if (isdir (image_info_get_path (thumb->info)))
      return FALSE;

   if (fr_archive_utils_get_file_name_ext (image_info_get_path (thumb->info)))
      return FALSE;

   if (thumb->thumbnail)
      return TRUE;
   else
      return FALSE;
}


gchar *
thumbnail_find_thumbcache (const gchar *filename, gchar **type)
{
   gchar *thumb_file = NULL;
   gchar *cache_type;
   GList *node;

   if (!cache_read_list || config_cache_read_string != conf.cache_read_list)
      thumbnail_apply_cache_read_prefs ();

   for (node = cache_read_list; node; node = g_list_next (node)) {
      cache_type = node->data;
      thumb_file = thumbsupport_get_thumb_cache_path (filename, cache_type);
      if (file_exists (thumb_file)) {
         if (type)
            *type = cache_type;
         return thumb_file;
      }
      g_free (thumb_file);
      thumb_file = NULL;
   }

   if (type)
      *type = NULL;

   return NULL;
}
