/* -*- Mode: C; tab-width: 3; indent-tabs-mode: nil; c-basic-offset: 3 -*- */

/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: thumbnail_support.c,v 1.30.2.4 2003/05/21 14:39:21 makeinu Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <string.h>

#include "gimageview.h"
#include "gimv_image.h"
#include "gtkutils.h"
#include "image_info.h"
#include "thumbnail_support.h"
#include "fileutil.h"
#include "prefs.h"


GHashTable *thumbnail_loaders = NULL;


/*******************************************************************************
 *
 *  Private functions
 *
 ******************************************************************************/
static void
store_loader_label (gpointer key, gpointer value, gpointer data)
{
   ThumbCacheLoader *loader = value;
   GList **list = data;

   g_return_if_fail (key);
   g_return_if_fail (loader);
   g_return_if_fail (data);

   *list = g_list_append (*list, loader->label);
}


static void
store_saver_label (gpointer key, gpointer value, gpointer data)
{
   ThumbCacheLoader *loader = value;
   GList **list = data;

   g_return_if_fail (key);
   g_return_if_fail (loader);
   g_return_if_fail (data);

   if (loader->save)
      *list = g_list_append (*list, loader->label);
}


/*******************************************************************************
 *
 *  Public functions
 *
 ******************************************************************************/
gboolean
thumbsupport_plugin_regist (const gchar *plugin_name,
                            const gchar *module_name,
                            gpointer impl,
                            gint     size)
{
   ThumbCacheLoader *loader = impl;

   g_return_val_if_fail (module_name, FALSE);
   g_return_val_if_fail (loader, FALSE);
   g_return_val_if_fail (size > 0, FALSE);
   g_return_val_if_fail (loader->if_version == GIMV_THUMB_CACHE_LOADER_IF_VERSION, FALSE);
   g_return_val_if_fail (loader->load && loader->get_path, FALSE);

   if (!thumbnail_loaders)
      thumbnail_loaders = g_hash_table_new (g_str_hash, g_str_equal);

   g_hash_table_insert (thumbnail_loaders, loader->label, loader);

   return TRUE;
}


/*
 *  thumbsupport_load_thumbnail_cache:
 *     @ load thumbnail cache to GimvImage struct. 
 *
 *  filename : File path of image.
 *  type     : Thuymbnail cache type.
 *  Return   : GimvImage struct.
 */
GimvImage *
thumbsupport_load_thumbnail_cache (const gchar *filename,
                                   const gchar *type,
                                   ThumbCacheSaveInfo **info)
{
   ThumbCacheLoader *loader;
   GimvImage *retval;

   if (info)
      *info = NULL;

   g_return_val_if_fail (filename, NULL);
   g_return_val_if_fail (type, NULL);

   if (!thumbnail_loaders) return NULL;

   if (!g_strcasecmp (type, "none")) return FALSE;

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return NULL;

   retval = loader->load (filename, type, info);

   return retval;
}


/*
 *  thumbsupport_save_thumbnail_cache:
 *     @ Save thumbnail cache to disk. 
 *
 *  filename : File path of image.
 *  type     : Thuymbnail cache type.
 *  image    : Pointer to GimvImage of original image.
 *             If this value is NULL, create from original image.
 *  info     : 
 *  Return   : TRUE if success.
 */
GimvImage *
thumbsupport_save_thumbnail_cache (const gchar *filename,
                                   const gchar *type,
                                   GimvImage   *image,
                                   ThumbCacheSaveInfo *info)
{
   ThumbCacheLoader *loader;
   GimvImage *retval;

   g_return_val_if_fail (filename, FALSE);
   g_return_val_if_fail (type, FALSE);

   if (!thumbnail_loaders) return FALSE;

   if (!g_strcasecmp (type, "none")) return FALSE;

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return FALSE;

   retval = loader->save (filename, type, image, info);

   return retval;
}


/*
 *  thumbsupport_get_thumb_cache_path:
 *     @ Return thumbnail path for specified image file and thumbnail type.
 *
 *  filename : Image file name.
 *  type     : Thumbnail cache type.
 *  Return   : Thumbnail path.
 */
gchar *
thumbsupport_get_thumb_cache_path (const gchar *filename, const gchar *type)
{
   ThumbCacheLoader *loader;

   g_return_val_if_fail (filename, NULL);
   g_return_val_if_fail (type, NULL);

   if (!thumbnail_loaders) return NULL;

   if (!g_strcasecmp (type, "none")) return NULL;

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return NULL;

   return loader->get_path (filename, type);
}


gboolean
thumbsupport_has_thumbnail (const gchar *filename, const gchar *type)
{
   gchar *cache = thumbsupport_get_thumb_cache_path (filename, type);
   gboolean exist;

   if (cache) {
      exist = file_exists (cache);
      g_free (cache);
      return exist;
   }

   return FALSE;
}


/*
 *  thumbsupport_get_thumb_cache_size:
 *     @ Return thumbnail size for various thumnail type.
 *
 *  type       : Thumbnail Cache type.
 *  width_ret  : Return thumbnail width.
 *  height_ret : Return thumbnail height.
 *  type_ret   : Which axis to fix.
 *  Return     : Thumbnail size [pixel]
 *               @ Zero means not allow (or not support) for writing cache
 */
gboolean
thumbsupport_get_thumb_cache_size (gint width, gint height, const gchar *type,
                                   gint *width_ret, gint *height_ret)
{
   ThumbCacheLoader *loader;

   g_return_val_if_fail (type, FALSE);

   if (!thumbnail_loaders) return FALSE;

   if (!g_strcasecmp (type, "none")) return FALSE;

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return FALSE;

   if (!loader->get_size) return FALSE;

   return loader->get_size (width, height, type, width_ret, height_ret);
}


gboolean
thumbsupport_has_load_prefs (const gchar *type)
{
   ThumbCacheLoader *loader;

   g_return_val_if_fail (type, FALSE);

   if (!thumbnail_loaders) return FALSE;

   if (!g_strcasecmp (type, "none")) return FALSE;

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return FALSE;

   if (!loader->prefs_load)
      return FALSE;
   else
      return TRUE;
}


gboolean
thumbsupport_has_save_prefs (const gchar *type)
{
   ThumbCacheLoader *loader;

   g_return_val_if_fail (type, FALSE);

   if (!thumbnail_loaders) return FALSE;

   if (!g_strcasecmp (type, "none")) return FALSE;

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return FALSE;

   if (!loader->prefs_save)
      return FALSE;
   else
      return TRUE;
}


GtkWidget *
thumbsupport_get_load_prefs (const gchar *type, gpointer data)
{
   ThumbCacheLoader *loader;

   g_return_val_if_fail (type, NULL);

   if (!thumbnail_loaders) return NULL;

   if (!g_strcasecmp (type, "none")) return NULL;

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return NULL;

   if (!loader->prefs_load) return NULL;

   return loader->prefs_load (data);
}


GtkWidget *
thumbsupport_get_save_prefs (const gchar *type, gpointer data)
{
   ThumbCacheLoader *loader;

   if (!g_strcasecmp (type, "none")) return NULL;

   if (!thumbnail_loaders) return NULL;

   g_return_val_if_fail (type, NULL);

   loader = g_hash_table_lookup (thumbnail_loaders, type);
   if (!loader) return NULL;

   if (!loader->prefs_save) return NULL;

   return loader->prefs_save (data);
}


ThumbCacheClearStatus
thumbsupport_clear_cache (const gchar *path,
                          const gchar *cache_type,
                          ThumbCacheClearFlags flags,
                          gpointer unused_yet)
{
   ThumbCacheLoader *loader;

   g_return_val_if_fail (cache_type && *cache_type,
                         THUMB_CACHE_CLEAR_STATUS_UNKNOWN_ERROR);

   loader = g_hash_table_lookup (thumbnail_loaders, cache_type);
   if (!loader) return THUMB_CACHE_CLEAR_STATUS_UNKNOWN_ERROR;
   if (!loader->clear) return THUMB_CACHE_CLEAR_STATUS_NORMAL;

   return loader->clear(path, cache_type, flags, unused_yet);
}


gboolean
thumbsupport_can_clear_cache (const gchar *path, const gchar *cache_type)
{
   ThumbCacheLoader *loader;

   g_return_val_if_fail (cache_type && *cache_type,
                         THUMB_CACHE_CLEAR_STATUS_UNKNOWN_ERROR);

   loader = g_hash_table_lookup (thumbnail_loaders, cache_type);
   if (!loader) return FALSE;
   if (!loader->clear) return FALSE;

   return TRUE;
}


/* list data should not be free */
GList *
thumbsupport_get_loader_list (void)
{
   GList *list = NULL;

   if (thumbnail_loaders)
      g_hash_table_foreach (thumbnail_loaders, (GHFunc) store_loader_label,
                            &list);

   if (list)
      list = g_list_sort (list, gtkutil_comp_spel);

   return list;
}


/* list data should not be free */
GList *
thumbsupport_get_saver_list (void)
{
   GList *list = NULL;

   if (thumbnail_loaders)
      g_hash_table_foreach (thumbnail_loaders, (GHFunc) store_saver_label,
                            &list);

   if (list)
      list = g_list_sort (list, gtkutil_comp_spel);

   list = g_list_prepend (list, "NONE");

   return list;
}


ThumbCacheSaveInfo *
thumbsupport_save_info_new_prot (void)
{
   ThumbCacheSaveInfo *save_info;

   save_info = g_new0 (ThumbCacheSaveInfo, 1);

   save_info->image_width  = -1;
   save_info->image_height = -1;
   save_info->image_depth  = 0;
   save_info->image_size   = 0;
   save_info->image_mtime  = 0;
   save_info->comment      = NULL;
   save_info->software     = NULL;
   save_info->data         = NULL;

   return save_info;
}


ThumbCacheSaveInfo *
thumbsupport_save_info_new (ImageInfo *info, gpointer data)
{
   ThumbCacheSaveInfo *save_info;

   g_return_val_if_fail (info, NULL);

   save_info = thumbsupport_save_info_new_prot ();

   save_info->image_width  = info->width;
   save_info->image_height = info->height;
   save_info->image_depth  = info->depth;
   save_info->image_size   = info->st.st_size;
   save_info->image_mtime  = info->st.st_mtime;
   save_info->software     = g_strdup (GIMV_PROG_NAME);
   save_info->data         = data;

   return save_info;
}


void
thumbsupport_save_info_delete (ThumbCacheSaveInfo *save_info)
{
   g_return_if_fail (save_info);

   if (save_info->software)
      g_free (save_info->software);

   if (save_info->comment)
      g_free (save_info->comment);

   g_free (save_info);
}
