/**********************************************************************
 
	Copyright (C) 2008- Hirohisa MORI <joshua@globalbase.org>
 
	This program is free software; you can redistribute it 
	and/or modify it under the terms of the GLOBALBASE 
	Library General Public License (G-LGPL) as published by 

	http://www.globalbase.org/
 
	This program is distributed in the hope that it will be 
	useful, but WITHOUT ANY WARRANTY; without even the 
	implied warranty of MERCHANTABILITY or FITNESS FOR A 
	PARTICULAR PURPOSE.

**********************************************************************/


#include	"memory_debug.h"
#include	"xl.h"
#include	"xlerror.h"
#include	"utils.h"
#include	"spidering.h"
#include	"recordlist64.h"
#include	"filespace64.h"
#include	"change_endian.h"
#include	"pdb64_basic.h"
#include	"lock_level.h"

#define SPA_HASH_SIZE		23

SEM sp_lock;
SPIDERING_WORK_AREA * wa_list;
SPIDERING_AGENT * spa_hash[SPA_HASH_SIZE];
SPIDERING_AGENT * spa_iid_hash[SPA_HASH_SIZE];
XLISP_ENV * sp_env;

void
init_spcache()
{
	sp_lock = new_lock(LL_SP);
	sp_env = new_env(gblisp_top_env0);
	set_env(gblisp_top_env1,l_string(std_cm,"SPenv"),get_env(sp_env));
}


void
spidering_refresh_favt_endian(SPIDERING_REFRESH_FAVT * ix)
{
	change_endian(ix->refresh_time);
	change_endian(ix->que_head);
	change_endian(ix->que_tail);
}


SPIDERING_WORK_AREA *
open_spcache(int * errp,L_CHAR * filename)
{
SPIDERING_WORK_AREA * wa;
L_CHAR * encoding;
PN64_SPIDERING_HEADER * sph;
INTEGER64 fofs;
int i;

	if ( errp )
		*errp = 0;
	wa = d_alloc(sizeof(*wa));
	memset(wa,0,sizeof(*wa));
	encoding = l_string(std_cm,"utf8");
	wa->pdb_fd = open_filespace64(
		n_string(std_cm,filename),
		O_RDWR|O_CREAT,0644,
		PF_USEFREELIST,FT_SPCACHE,encoding);
	if ( wa->pdb_fd == 0 ) {
		d_f_ree(wa);
		if ( errp )
			*errp = -1;
		return 0;
	}

	sph = get_file_record64((U_INTEGER64*)&wa->header_fofs,
		wa->pdb_fd,PNT_SP_HEADER,0);
	if ( sph == 0 ) {
		l_close_filespace64(wa->pdb_fd);
		wa->pdb_fd = open_filespace64(
			n_string(std_cm,filename),
			O_RDWR|O_CREAT|O_TRUNC,0644,
			PF_USEFREELIST,FT_SPCACHE,encoding);
		if ( wa->pdb_fd == 0 ) {
			d_f_ree(wa);
			if ( errp )
				*errp = -2;
			return 0;
		}
		for ( i = 0 ; i < SP_REFRESH_QUE_MAX ; i ++ ) {
			wa->rt_refresh_que[i] = l_favt64_alloc_root(wa->pdb_fd,FAT_SP_NODE,spidering_refresh_favt_endian);
			wa->cache_header.refresh_que[i] = wa->rt_refresh_que[i]->h.fofs;
		}
		wa->rt_data_que = l_favt64_alloc_root(wa->pdb_fd,FAT_SP_NODE,favt64_string_endian);
		wa->cache_header.data_que = wa->rt_data_que->h.fofs;
		wa->cache_header.version = SP_VERSION;

		wa->cache_header.h.type = PNT_SP_HEADER;
		wa->cache_header.h.size = sizeof(wa->cache_header);
		wa->cache_header.dummy = 0;

		fofs = l_alloc_filespace64(wa->pdb_fd,&wa->cache_header.h);
		
		sph = d_alloc(sizeof(*sph));
		*sph = wa->cache_header;
		
		change_endian_header64(&sph->h);
		change_endian(sph->version);
		change_endian(sph->data_que);
		for ( i = 0 ; i < SP_REFRESH_QUE_MAX ; i ++ ) {
			change_endian(sph->refresh_que[i]);
		}

		l_write_filespace64f(wa->pdb_fd,fofs,sph);
		
		create_sp_path(wa,l_string(std_cm,"root"),NT_SYSTEM);
		create_sp_path(wa,l_string(std_cm,"root.addr"),NT_SYSTEM);
		create_sp_path(wa,l_string(std_cm,"root.addr.ipv4"),NT_SYSTEM);
		create_sp_path(wa,l_string(std_cm,"root.addr.ipv6"),NT_SYSTEM);
		create_sp_path(wa,l_string(std_cm,"root.queue"),NT_SYSTEM);
		create_sp_path(wa,l_string(std_cm,"root.queue.myurl"),NT_SYSTEM);
		create_sp_path(wa,l_string(std_cm,"root.queue.url"),NT_SYSTEM);
		create_sp_path(wa,l_string(std_cm,"root.data"),NT_SYSTEM);
	}
	else {
		wa->cache_header = *sph;
		
		change_endian_header64(&wa->cache_header.h);
		change_endian(wa->cache_header.version);
		change_endian(wa->cache_header.data_que);
		for ( i = 0 ; i < SP_REFRESH_QUE_MAX ; i ++ ) {
			change_endian(wa->cache_header.refresh_que[i]);
		}
		
		wa->rt_data_que = l_get_root64(&wa->favt64_err,
					wa->pdb_fd,
					wa->cache_header.data_que,
					favt64_string_endian);
		for ( i = 0 ; i < SP_REFRESH_QUE_MAX ; i ++ ) {
		
			wa->rt_refresh_que[i] = l_get_root64(&wa->favt64_err,
						wa->pdb_fd,
						wa->cache_header.refresh_que[i],
						spidering_refresh_favt_endian);
		}
	}
	
	d_f_ree(sph);
	
	lock_task(sp_lock);
	wa->next = wa_list;
	wa_list = wa;
	unlock_task(sp_lock,"");
	
	if ( errp )
		*errp = 0;

	return wa;
}

void
close_spcache(SPIDERING_WORK_AREA * wa)
{
SPIDERING_WORK_AREA ** wap;
	lock_task(sp_lock);
	for ( ; wa->op_cnt ; ) {
		sleep_task((int)wa,sp_lock);
		lock_task(sp_lock);
	}
	for ( wap = &wa_list ; *wap != wa ; wap = &(*wap)->next );
	if ( *wap )
		*wap = wa->next;
	if ( wa->pdb_fd )
		l_close_filespace64(wa->pdb_fd);
	d_f_ree(wa);
	unlock_task(sp_lock,"");
}


void
close_all_spcache()
{
SPIDERING_WORK_AREA * wa;
	for ( ; ; ) {
		wa = wa_list;
		if ( wa == 0 )
			break;
		close_spcache(wa);
	}
}

void
_set_wa_op_cnt_write(SPIDERING_WORK_AREA * wa)
{
	for ( ; wa->op_cnt > 0 ; ) {
		sleep_task((int)wa,sp_lock);
		lock_task(sp_lock);
	}
	wa->op_cnt = -1;
}

void
set_wa_op_cnt_write(SPIDERING_WORK_AREA * wa)
{
	lock_task(sp_lock);
	_set_wa_op_cnt_write(wa);
	unlock_task(sp_lock,"");
}


void
_set_wa_op_cnt(SPIDERING_WORK_AREA * wa)
{
	for ( ; wa->op_cnt < 0 ; ) {
		sleep_task((int)wa,sp_lock);
		lock_task(sp_lock);
	}
	wa->op_cnt ++;
}

void
set_wa_op_cnt(SPIDERING_WORK_AREA * wa)
{
	lock_task(sp_lock);
	_set_wa_op_cnt(wa);
	unlock_task(sp_lock,"");
}


void
_reset_wa_op_cnt(SPIDERING_WORK_AREA * wa)
{
	if ( wa->op_cnt > 0 )
		wa->op_cnt --;
	else if ( wa->op_cnt < 0 )
		wa->op_cnt ++;
	else	er_panic("_reset_wa_op_cnt");
	if ( wa->op_cnt == 0 )
		wakeup_task((int)wa);
}

void
reset_wa_op_cnt(SPIDERING_WORK_AREA * wa)
{
	lock_task(sp_lock);
	_reset_wa_op_cnt(wa);
	unlock_task(sp_lock,"");
}



void
change_endian_spidering_fundamental_core(SPIDERING_FUNDAMENTAL_CORE * c)
{
	change_endian(c->create_time);
	change_endian(c->modify_time);
	change_endian(c->next_refresh_time);
	change_endian(c->life_expire_time);
	change_endian(c->refresh_interval);
	change_endian(c->life_interval);
	change_endian(c->type);
}

void
change_endian_pn64_search_fundamental_info(PN64_SEARCH_FUNDAMENTAL_INFO* info)
{
	change_endian_spidering_fundamental_core(&info->c);
	change_endian_header64(&info->h);
}

void
change_endian_spidering_refresh_queue(SPIDERING_REFRESH_QUEUE* q)
{
L_CHAR * str;
	change_endian_header64(&q->h);
	change_endian(q->next);
	for ( str = &q->d[0] ; *str ; str ++ ) {
		change_endian(*str);
	}
}

int
spidering_refresh_cmp(SPIDERING_REFRESH_FAVT*srf1,SPIDERING_REFRESH_FAVT*srf2)
{
	if ( srf1->refresh_time < srf2->refresh_time )
		return -1;
	if ( srf1->refresh_time > srf2->refresh_time )
		return 1;
	return 0;
}

void
_insert_sp_refresh_que(SPIDERING_WORK_AREA * wa,int pri,L_CHAR * path,PN64_SEARCH_FUNDAMENTAL_INFO * info)
{
SPIDERING_REFRESH_FAVT * srf, * srf2;
SPIDERING_REFRESH_QUEUE * srq,* srq2;
int len;
int size;
INTEGER64 fofs;
FAVT64_NODE * fn,*fn2;


	_set_wa_op_cnt(wa);
	len = l_strlen(path);
	srq = d_alloc(size = sizeof(*srq) + sizeof(L_CHAR)*(len));
	srq->h.size = size;
	srq->h.type = PNT_SP_REFRESH_QUE;
	l_strcpy(&srq->d[0],path);
	srq->next = 0;

	fofs = l_alloc_filespace64(
				wa->pdb_fd,&srq->h);
	l_write_filespace64f(wa->pdb_fd,fofs,srq);

	srf = d_alloc(sizeof(*srf));
	srf->refresh_time = info->c.next_refresh_time;
	srf->que_head = 0;
	srf->que_tail = 0;

	fn = l_favt64_search(&wa->favt64_err,
			wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_refresh_que[pri]),
			srf,spidering_refresh_cmp,0);
	if ( fn ) {
		srf2 = fn->data;
		srq2 = l_read_filespace64f(wa->pdb_fd,srf2->que_tail);
		change_endian_spidering_refresh_queue(srq2);
		srq2->next = fofs;
		
		srf2->que_tail = fofs;
		fn->h.flags |= FAF_DIRTY;
	}
	else {
		srf->que_head = srf->que_tail = fofs;
		fn = l_favt64_alloc_node(
				wa->rt_refresh_que[pri],
				srf,len);
		fn2 = l_favt64_insert(&wa->favt64_err,
				wa->rt_refresh_que[pri],
				&wa->rt_refresh_que[pri]->node,fn,
				spidering_refresh_cmp,0);
	}
	d_f_ree(srf);
	
	_reset_wa_op_cnt(wa);
	return;
}

void
insert_sp_refresh_que(SPIDERING_WORK_AREA * wa,int pri,L_CHAR * path,PN64_SEARCH_FUNDAMENTAL_INFO * info)
{
	lock_task(sp_lock);
	insert_sp_refresh_que(wa,pri,path,info);
	unlock_task(sp_lock,"");
}

L_CHAR *
_delete_sp_refresh_que(SPIDERING_WORK_AREA * wa,int pri,INTEGER64 tim)
{
FAVT64_NODE * fn;
SPIDERING_REFRESH_FAVT srf,*srf2;
SPIDERING_REFRESH_QUEUE * srq2;
L_CHAR * ret;
INTEGER64 fofs;

	_set_wa_op_cnt(wa);
	srf.refresh_time = tim;
	srf.que_head = 0;
	srf.que_tail = 0;
	fn = l_favt64_search(&wa->favt64_err,
			wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_refresh_que[pri]),
			&srf,spidering_refresh_cmp,0);
	if ( fn == 0 ) {
		_reset_wa_op_cnt(wa);
		return 0;
	}
	srf2 = fn->data;
	srq2 = l_read_filespace64f(wa->pdb_fd,fofs = srf2->que_head);
	change_endian_spidering_refresh_queue(srq2);
	ret = ll_copy_str(&srq2->d[0]);

	if ( srq2->next ) {
		srf2->que_head = srq2->next;
		fn->h.flags |= FAF_DIRTY;
	}
	else {
		fn = l_favt64_delete(&wa->favt64_err,wa->rt_data_que,&wa->rt_data_que->node,&srf,spidering_refresh_cmp,0);
		if ( fn == 0 )
			er_panic("create_sp_path");
		l_favt64_free_node(fn);
	}
	l_free_filespace64(wa->pdb_fd,fofs);
	
	d_f_ree(srq2);
	
	_reset_wa_op_cnt(wa);
	return ret;
}

L_CHAR *
delete_sp_refresh_que(SPIDERING_WORK_AREA * wa,int pri,INTEGER64 tim)
{
L_CHAR * ret;
	lock_task(sp_lock);
	ret = delete_sp_refresh_que(wa,pri,tim);
	unlock_task(sp_lock,"");
	return ret;
}


int
_delete_sp_data(SPIDERING_WORK_AREA * wa,L_CHAR * path,L_CHAR * ext)
{
L_CHAR* _path;
int len1,len2,_len;
STRING_INDEX64 * ix,*ixp;
FAVT64_NODE * fn;
INTEGER64 fofs;
int ret;
	ret = 0;
	if ( ext[0] != LC_SECC_HEADER && ext[0] != LC_SECC_TERMINATOR )
		return -1;
	_set_wa_op_cnt_write(wa);
	len1 = l_strlen(path);
	len2 = l_strlen(ext);
	_path = ll_copy_str(path);
	_path = d_re_alloc(_path,sizeof(L_CHAR)*(len1+len2+1));
	l_strcpy(&_path[len1],ext);
	ix = d_alloc(_len=STRING_IX64_LENGTH_1(len1+len2));
	l_strcpy(&ix->d[0],_path);
	fn = l_favt64_delete(&wa->favt64_err,wa->rt_data_que,&wa->rt_data_que->node,ix,favt64_cmp_string_1,0);
	if ( fn == 0 ) {
		ret = -2;
		goto end;
	}

	ixp = fn->data;
	fofs = ixp->fofs;

	l_favt64_free_node(fn);

	if ( fofs )
		l_free_filespace64(wa->pdb_fd,fofs);
end:
	d_f_ree(ix);
	d_f_ree(_path);
	_reset_wa_op_cnt(wa);
	return 0;
}

int
delete_sp_data(SPIDERING_WORK_AREA * wa,L_CHAR * path,L_CHAR * ext)
{
int ret;
	lock_task(sp_lock);
	ret = _delete_sp_data(wa,path,ext);
	unlock_task(sp_lock,"");
	return ret;
}

void *
_get_sp_data(INTEGER64 * fofs,SPIDERING_WORK_AREA * wa,L_CHAR * path,L_CHAR * ext)
{
int len1,len2,_len;
L_CHAR * _path;
STRING_INDEX64 * ix,*ixp;
FAVT64_NODE * fn;

void * ret;
	_set_wa_op_cnt(wa);
	len1 = l_strlen(path);
	len2 = l_strlen(ext);
	_path = ll_copy_str(path);
	_path = d_re_alloc(_path,sizeof(L_CHAR)*(len1+len2+1));
	l_strcpy(&_path[len1],ext);
	ix = d_alloc(_len=STRING_IX64_LENGTH_1(len1+len2));
	l_strcpy(&ix->d[0],_path);

	fn = l_favt64_search(&wa->favt64_err,
			wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_data_que),
			ix,favt64_cmp_string_1,0);
	if ( fn == 0 ) {
		d_f_ree(_path);
		d_f_ree(ix);
		_reset_wa_op_cnt(wa);
		return 0;
	}

	ixp = fn->data;
	ret = l_read_filespace64f(wa->pdb_fd,ixp->fofs);
	if ( fofs )
		*fofs = ixp->fofs;

	d_f_ree(_path);
	d_f_ree(ix);
	_reset_wa_op_cnt(wa);
	return ret;
}

void *
get_sp_data(INTEGER64 * fofs,SPIDERING_WORK_AREA * wa,L_CHAR * path,L_CHAR * ext)
{
void * ret;
	lock_task(sp_lock);
	ret = _get_sp_data(fofs,wa,path,ext);
	unlock_task(sp_lock,"");
	return ret;
}

L_CHAR *
_get_sp_parent_path(PN64_SEARCH_FUNDAMENTAL_INFO ** infp,SPIDERING_WORK_AREA * wa,L_CHAR * path)
{
L_CHAR * _path;
int len;
PN64_SEARCH_FUNDAMENTAL_INFO * info;
L_CHAR ext[3] = {LC_SECC_HEADER,LC_SECC_INFO_START,0};

	_set_wa_op_cnt(wa);
	_path = ll_copy_str(path);
	len = l_strlen(path);
	_path = d_re_alloc(_path,sizeof(L_CHAR)*(len+3));

	len --;
	for ( ; len > 0 ; len -- ) {
		_path[len] = 0;
		info = _get_sp_data(0,wa,_path,&ext[0]);
		if ( info ) {
			change_endian_pn64_search_fundamental_info(info);
			*infp = info;
			_reset_wa_op_cnt(wa);
			return _path;
		}
	}
	*infp = 0;
	d_f_ree(_path);
	_reset_wa_op_cnt(wa);
	return 0;
}

L_CHAR *
get_sp_parent_path(PN64_SEARCH_FUNDAMENTAL_INFO ** infp,SPIDERING_WORK_AREA * wa,L_CHAR * path)
{
L_CHAR * ret;
	lock_task(sp_lock);
	ret = _get_sp_parent_path(infp,wa,path);
	unlock_task(sp_lock,"");
	return ret;
}


int
_set_sp_lock_list(SPIDERING_WORK_AREA * wa,L_CHAR * path,int type,int wait_flag)
{
SP_LOCK_LIST * spl, * target_spl;
int len1,len2;
PN64_SEARCH_FUNDAMENTAL_INFO * info;
L_CHAR * parent,* p_parent;

retry:
	len1 = l_strlen(path);
	target_spl = 0;
	p_parent = 0;
	for ( spl = wa->sp_lock_list ; spl ; spl = spl->next ) {
		len2 = l_strlen(spl->path);
		if ( len2 == len1 && l_strcmp(spl->path,path) == 0 )
			target_spl = spl;
		switch ( type ) {
		case SPLT_INSERT:
			if ( len2 > len1 )
				continue;
			if ( memcmp(spl->path,path,sizeof(L_CHAR)*len2) == 0 ) {
				if ( spl->cnt[SPLT_DELETE] )
					goto wait_mode;
				if ( l_strcmp(path,spl->path) == 0 ) {
					if ( spl->cnt[SPLT_READDIR] )
						goto wait_mode;
					continue;
				}
				if ( p_parent )
					parent = ll_copy_str(p_parent);
				else	parent = _get_sp_parent_path(&info,wa,path);
				if ( parent == 0 )
					continue;
				if ( p_parent == 0 )
					p_parent = ll_copy_str(parent);
				d_f_ree(info);
				if ( l_strcmp(parent,spl->path) ) {
					d_f_ree(parent);
					continue;
				}
				d_f_ree(parent);
				if ( spl->cnt[SPLT_READDIR] )
					goto wait_mode;
				if ( spl->cnt[SPLT_INSERT] )
					goto wait_mode;
				continue;
			}
			continue;
		case SPLT_DELETE:
			if ( len2 >= len1 ) {
				if ( memcmp(spl->path,path,sizeof(L_CHAR)*len1) == 0 )
					goto wait_mode;
				continue;
			}
			if ( memcmp(spl->path,path,sizeof(L_CHAR)*len2) == 0 ) {
				if ( spl->cnt[SPLT_DELETE] )
					goto wait_mode;
				continue;
			}
			if ( p_parent )
				parent = ll_copy_str(p_parent);
			else	parent = _get_sp_parent_path(&info,wa,path);
			if ( parent == 0 )
				continue;
			if ( p_parent == 0 )
				p_parent = ll_copy_str(parent);
			d_f_ree(info);
			if ( l_strcmp(parent,spl->path) ) {
				d_f_ree(parent);
			}
			d_f_ree(parent);
			if ( spl->cnt[SPLT_READDIR] )
				goto wait_mode;
			continue;
		case SPLT_READDIR:
			if ( len2 > len1 ) {
				if ( memcmp(spl->path,path,sizeof(L_CHAR)*len1) )
					continue;
				if ( spl->parent )
					parent = ll_copy_str(spl->parent);
				else	parent = _get_sp_parent_path(&info,wa,spl->path);
				if ( parent == 0 )
					continue;
				if ( spl->parent == 0 )
					spl->parent = ll_copy_str(parent);
				d_f_ree(info);
				if ( l_strcmp(parent,path) ) {
					d_f_ree(parent);
					continue;
				}
				d_f_ree(parent);
				if ( spl->cnt[SPLT_INSERT] )
					goto wait_mode;
				if ( spl->cnt[SPLT_DELETE] )
					goto wait_mode;
				continue;
			}
			if ( l_strcmp(spl->path,path) == 0 ) {
				if ( spl->cnt[SPLT_INSERT] )
					goto wait_mode;
			}
			if ( memcmp(spl->path,path,sizeof(L_CHAR)*len2) == 0 ) {
				if ( spl->cnt[SPLT_DELETE] )
					goto wait_mode;
				continue;
			}
			continue;
		case SPLT_OP:
			if ( len2 > len1 )
				continue;
			if ( len2 == len1 ) {
				if ( l_strcmp(spl->path,path) == 0 &&
						spl->cnt[SPLT_INSERT] )
					goto wait_mode;
			}
			if ( memcmp(spl->path,path,sizeof(L_CHAR)*len2) == 0 ) {
				if ( spl->cnt[SPLT_DELETE] )
					goto wait_mode;
				continue;
			}
			continue;
		default:
			er_panic("_set_sp_lock_list");
		}
	}
	if ( target_spl == 0 ) {
		target_spl = d_alloc(sizeof(*target_spl));
		memset(target_spl,0,sizeof(*target_spl));
		target_spl->path = ll_copy_str(path);
	}
	target_spl->cnt[type] ++;
	if ( p_parent ) {
		if ( target_spl->parent == 0 )
			target_spl->parent = p_parent;
		else	d_f_ree(p_parent);
	}
	target_spl->next = wa->sp_lock_list;
	wa->sp_lock_list = target_spl;
	return 0;
wait_mode:
	if ( p_parent )
		d_f_ree(p_parent);
	p_parent = 0;
	if ( wait_flag )
		return -1;
	sleep_task((int)spl,sp_lock);
	lock_task(sp_lock);
	goto retry;
}

int
set_sp_lock_list(SPIDERING_WORK_AREA * wa,L_CHAR * path,int type,int wait_flag)
{
int ret;
	lock_task(sp_lock);
	ret = _set_sp_lock_list(wa,path,type,wait_flag);
	unlock_task(sp_lock,"");
	return ret;
}

int
_reset_sp_lock_list(SPIDERING_WORK_AREA * wa,L_CHAR * path,int type)
{
SP_LOCK_LIST * spl,**splp;
int i;
	for ( splp = &wa->sp_lock_list ; *splp ; splp = &spl->next ) {
		spl = *splp;
		if ( l_strcmp(spl->path,path) )
			continue;
		if ( spl->cnt[type] <= 0 )
			er_panic("_reset_sp_lock_list");
		spl->cnt[type] --;
		if ( spl->cnt[type] == 0 )
			wakeup_task((int)spl);
		for ( i = 0 ; i < SPLT_MAX ; i ++ )
			if ( spl->cnt[i] )
				return 0;
		*splp = spl->next;
		d_f_ree(spl->path);
		d_f_ree(spl);
		return 0;
	}
	er_panic("_reset_sp_lock_list(2)");
	return 0;
}


int
reset_sp_lock_list(SPIDERING_WORK_AREA * wa,L_CHAR * path,int type)
{
int ret;
	lock_task(sp_lock);
	ret = _reset_sp_lock_list(wa,path,type);
	unlock_task(sp_lock,"");
	return ret;
}

int
create_sp_path(SPIDERING_WORK_AREA * wa,L_CHAR* path,int type)
{
FAVT64_NODE * fn,*fn2;
STRING_INDEX64 * ix,*ix_large,*ixp,*ixp2;
int len,_len;
PN64_SEARCH_FUNDAMENTAL_INFO * info;
L_CHAR * _path,*p;
BOUND64_LIST *bp, * blp;
int ret;
int plen,len3;
L_CHAR * parent;

	info = 0;
	set_wa_op_cnt(wa);
	_path = ll_copy_str(path);
	len = l_strlen(path);

	set_sp_lock_list(wa,path,SPLT_INSERT,1);

	_path = d_re_alloc(_path,sizeof(L_CHAR)*(len+3));
	_path[len] = LC_SECC_HEADER;
	_path[len+1] = LC_SECC_INFO_START;
	_path[len+2] = 0;
	ix = d_alloc(_len=STRING_IX64_LENGTH_1(len+2));
	l_strcpy(&ix->d[0],_path);

	fn = l_favt64_search(&wa->favt64_err,
			wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_data_que),
			ix,favt64_cmp_string_1,0);
	if ( fn ) {
		d_f_ree(_path);
		d_f_ree(ix);
		ret =  -1;
		goto end;
	}

	info = d_alloc(sizeof(PN64_SEARCH_FUNDAMENTAL_INFO));
	info->h.type = PNT_SP_INFO;
	info->h.size = sizeof(PN64_SEARCH_FUNDAMENTAL_INFO);
	info->c.create_time = get_xltime();
	info->c.modify_time = info->c.create_time;
	info->c.next_refresh_time = info->c.create_time + DEFAULT_REFRESH_INTERVAL;
	info->c.life_expire_time = info->c.create_time + DEFAULT_LIFE_INTERVAL;
	info->c.type = type;
	info->c.refresh_interval = DEFAULT_REFRESH_INTERVAL;
	info->c.life_interval = DEFAULT_LIFE_INTERVAL;
	
	change_endian_pn64_search_fundamental_info(info);

	ix->fofs = l_alloc_filespace64(
				wa->pdb_fd,&info->h);
	l_write_filespace64f(wa->pdb_fd,ix->fofs,info);

	fn = l_favt64_alloc_node(
			wa->rt_data_que,
			ix,_len);
	fn2 = l_favt64_insert(&wa->favt64_err,
			wa->rt_data_que,
			&wa->rt_data_que->node,fn,
			favt64_cmp_string_1,0);
	if ( fn != fn2 )
		er_panic("create_sp_path");
	
	_path[len] = LC_SECC_TERMINATOR;
	_path[len+1] = LC_SECC_INFO_END;
	l_strcpy(&ix->d[0],_path);
	ix->fofs = 0;
	fn = l_favt64_alloc_node(
			wa->rt_data_que,
			ix,_len);
	fn2 = l_favt64_insert(&wa->favt64_err,
			wa->rt_data_que,
			&wa->rt_data_que->node,fn,
			favt64_cmp_string_1,0);
	if ( fn != fn2 )
		er_panic("create_sp_path");
	
	if ( type != NT_SYSTEM ) {
		_path[len] = 0;
		insert_sp_refresh_que(wa,0,_path,info);
	}


	/* EMPTY TAG CHECK */



	_path[len] = LC_SECC_HEADER;
	_path[len+1] = 0;
	ix_large = d_alloc(_len=STRING_IX64_LENGTH_1(len+2));
	l_strcpy(&ix_large->d[0],_path);
	ix_large->fofs = 0;
	
	ix->d[0] = 0;
	ix->fofs = 0;
	
	bp = l_favt64_bound_search_reverse(
		&wa->favt64_err,wa->rt_data_que,
		l_root_node64(&wa->favt64_err,wa->rt_data_que),
		ix,ix_large,2,
		favt64_cmp_string_1,0);
	
	for ( blp = bp; blp ; blp = blp->next ) {
		ixp = blp->data;
		for ( p = &ixp->d[0] ; *p && *p != LC_SECC_EMPTY_FROM ; p ++ );
		if ( *p == 0 )
			break;
		fn = l_favt64_delete(&wa->favt64_err,wa->rt_data_que,&wa->rt_data_que->node,ixp,favt64_cmp_string_1,0);
		if ( fn == 0 )
			er_panic("create_sp_path");
		l_favt64_free_node(fn);
		goto in_empty;
	}
	free_bound_list64(bp);
	goto next;
in_empty:
	free_bound_list64(bp);
	_path[len] = LC_SECC_TERMINATOR;
	_path[len+1] = LC_SECC_EMPTY_FROM;
	_path[len+2] = 0;
	l_strcpy(&ix->d[0],_path);
	ix->fofs = 0;
	
	ix_large->d[0] = LC_SECC_TERMINATOR;
	ix_large->d[1] = 0;
	ix_large = 0;
	
	bp = l_favt64_bound_search(
		&wa->favt64_err,wa->rt_data_que,
		l_root_node64(&wa->favt64_err,wa->rt_data_que),
		ix,ix_large,2,
		favt64_cmp_string_1,0);

	for ( blp = bp ; blp ; blp = blp->next ) {
		ixp = blp->data;
		for ( p = &ixp->d[0] ; *p && *p != LC_SECC_EMPTY_TO ; p ++ );
		if ( *p == 0 ) {
			free_bound_list64(bp);
			goto next;
		}
		fn = l_favt64_delete(&wa->favt64_err,wa->rt_data_que,&wa->rt_data_que->node,ixp,favt64_cmp_string_1,0);
		if ( fn == 0 )
			er_panic("create_sp_path");
		l_favt64_free_node(fn);
		break;
	}
	free_bound_list64(bp);
next:

	/* make readdir data */

	_path[len] = 0;
	if ( info )
		d_f_ree(info);
	info = 0;
	parent = get_sp_parent_path(&info,wa,_path);
	if ( parent == 0 )
		goto next2;
	plen = l_strlen(parent);
	parent = d_re_alloc(parent,sizeof(L_CHAR)*(len+3));
	memcpy(&parent[plen+2],&_path[plen],sizeof(L_CHAR)*(len-plen));
	parent[len+2] = 0;
	parent[plen] = LC_SECC_HEADER;
	parent[plen+1] = LC_SECC_READDIR;
	ix = d_re_alloc(ix,_len=STRING_IX64_LENGTH_1(len+2));
	l_strcpy(&ix->d[0],parent);

	fn = l_favt64_alloc_node(
			wa->rt_data_que,
			ix,_len);

	fn2 = l_favt64_insert(&wa->favt64_err,
			wa->rt_data_que,
			&wa->rt_data_que->node,fn,
			favt64_cmp_string_1,0);
	if ( fn != fn2 )
		er_panic("create_sp_path");


	for ( ; ; ) {
		bp = l_favt64_bound_search(
			&wa->favt64_err,wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_data_que),
			ix,ix_large,10,
			favt64_cmp_string_1,0);
		blp = bp->next;
		if ( blp == 0 )
			break;
		for ( ; blp ; blp = blp->next ) {
			ixp = blp->data;
			len3 = l_strlen(&ixp->d[0]);
			if ( len3 < len+2 )
				goto mkreaddirdata_end;
			if ( memcmp(&ixp->d[0],parent,sizeof(L_CHAR)*(len+2)) )
				goto mkreaddirdata_end;
			fn = l_favt64_delete(&wa->favt64_err,wa->rt_data_que,&wa->rt_data_que->node,ixp,favt64_cmp_string_1,0);
			if ( fn == 0 )
				er_panic("create_sp_path");
			ixp2 = fn->data;
			if ( ixp2->fofs )
				l_free_filespace64(wa->pdb_fd,ixp2->fofs);
			l_favt64_free_node(fn);
			memcpy(&ixp->d[plen],&ixp->d[plen+2],sizeof(L_CHAR)*(len+2-plen));
			ixp->d[len] = LC_SECC_HEADER;
			ixp->d[len+1] = LC_SECC_READDIR;
			fn = l_favt64_alloc_node(
					wa->rt_data_que,
					ixp,STRING_IX64_LENGTH_1(len3));

			fn2 = l_favt64_insert(&wa->favt64_err,
					wa->rt_data_que,
					&wa->rt_data_que->node,fn,
					favt64_cmp_string_1,0);
			if ( fn != fn2 )
				er_panic("create_sp_path");
		}
		free_bound_list64(bp);
	}
mkreaddirdata_end:
	free_bound_list64(bp);
next2:

	d_f_ree(ix_large);
	d_f_ree(ix);
	d_f_ree(_path);
	if ( info )
		d_f_ree(info);
	ret = 0;
end:
	reset_sp_lock_list(wa,path,SPLT_INSERT);
	reset_wa_op_cnt(wa);
	return ret;
}


int
sp_agent_auth(SPIDERING_WORK_AREA * wa,L_CHAR* path)
{
XL_INTERPRETER * xli;
SPIDERING_AGENT * a;
L_CHAR * parent,*parent2;
PN64_SEARCH_FUNDAMENTAL_INFO * info;
int pf;
INTEGER64 fofs;
int ret;
L_CHAR * ext;
int len;
	xli = get_my_xli();
	if ( xli == 0 )
		return -1;
	a = search_sp_agent_by_iid(xli->id);
	if ( a == 0 )
		return -1;
	parent = path;
	pf = 0;
	ret = -1;
	len = l_strlen(a->name);
	ext = d_alloc(sizeof(L_CHAR)*(len+3));
	ext[0] = LC_SECC_HEADER;
	ext[1] = LC_SECC_AGENT;
	l_strcpy(ext,a->name);
	for ( ; ; ) {
		info = get_sp_data(&fofs,wa,parent,ext);
		if ( info ) {
			ret = 0;
			break;
		}
		d_f_ree(info);
		info = 0;
		parent2 = get_sp_parent_path(&info,wa,parent);
		if ( info == 0 )
			d_f_ree(info);
		if ( parent2 == 0 )
			break;
		if ( pf )
			d_f_ree(parent);
		parent = parent2;
		pf = 1;
	}
	if ( pf )
		d_f_ree(parent);
	d_f_ree(ext);
	return ret;
}


int
create_sp_path_auth(SPIDERING_WORK_AREA * wa,L_CHAR* path,int type)
{
int ret;
	ret = sp_agent_auth(wa,path);
	if ( ret < 0 )
		return ret;
	return create_sp_path_auth(wa,path,type);
}

int
delete_sp_path(SPIDERING_WORK_AREA * wa,L_CHAR* path)
{
L_CHAR * _path,*parent;
int len,_len,plen;
int ret;
FAVT64_NODE * fn;
STRING_INDEX64 * ix,*ix_large,*ixp;
BOUND64_LIST *bp, * blp;
PN64_SEARCH_FUNDAMENTAL_INFO * info;

	set_wa_op_cnt(wa);
	ret = 0;
	_path = ll_copy_str(path);
	len = l_strlen(_path);
	
	set_sp_lock_list(wa,path,SPLT_DELETE,1);

	_path = d_re_alloc(_path,sizeof(L_CHAR)*(len+3));
	_path[len] = LC_SECC_HEADER;
	_path[len+1] = LC_SECC_INFO_START;
	_path[len+2] = 0;
	ix = d_alloc(_len=STRING_IX64_LENGTH_1(len+2));
	l_strcpy(&ix->d[0],_path);
	ix->fofs = 0;
	
	_path[len+1] = LC_SECC_TERMINATOR;
	_path[len+2] = LC_SECC_INFO_END;
	ix_large = d_alloc(_len);
	l_strcpy(&ix->d[0],_path);
	ix_large->fofs = 0;

	for ( ; ; ) {
		bp = l_favt64_bound_search(
			&wa->favt64_err,wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_data_que),
			ix,ix_large,10,
			favt64_cmp_string_1,0);
		if ( bp == 0 )
			break;
		for ( blp = bp ; blp ; blp = blp->next ) {
			ixp = blp->data;
			if ( ixp->fofs )
				l_free_filespace64(wa->pdb_fd,ixp->fofs);
			fn = l_favt64_delete(&wa->favt64_err,wa->rt_data_que,&wa->rt_data_que->node,ixp,favt64_cmp_string_1,0);
			if ( fn == 0 )
				er_panic("create_sp_path");
			l_favt64_free_node(fn);
		}
		free_bound_list64(bp);
	}

	parent = get_sp_parent_path(&info,wa,_path);
	if ( parent == 0 )
		goto end;
	d_f_ree(info);
	plen = l_strlen(parent);
	parent = d_re_alloc(parent,sizeof(L_CHAR)*(len+3));
	memcpy(&parent[plen+2],&_path[plen],sizeof(L_CHAR)*(len-plen));
	parent[len+2] = 0;
	parent[plen] = LC_SECC_HEADER;
	parent[plen+1] = LC_SECC_READDIR;
	memcpy(&ix->d[0],parent,sizeof(L_CHAR)*(len+3));
	fn = l_favt64_delete(&wa->favt64_err,wa->rt_data_que,&wa->rt_data_que->node,ix,favt64_cmp_string_1,0);
	if ( fn == 0 )
		er_panic("create_sp_path");
	ixp = fn->data;
	if ( ixp->fofs )
		l_free_filespace64(wa->pdb_fd,ixp->fofs);
	l_favt64_free_node(fn);

end:
	d_f_ree(ix);
	d_f_ree(ix_large);
	d_f_ree(_path);
	reset_sp_lock_list(wa,path,SPLT_DELETE);
	reset_wa_op_cnt(wa);
	return ret;
}

int
delete_sp_path_auth(SPIDERING_WORK_AREA * wa,L_CHAR* path)
{
int ret;
	ret = sp_agent_auth(wa,path);
	if ( ret < 0 )
		return ret;
	return delete_sp_path_auth(wa,path);
}

void *
start_sp_operation(int * err_p,SPIDERING_WORK_AREA * wa,L_CHAR* path,L_CHAR * ext,int wait_flag)
{
void * ret;
SP_LOCK_OP_LIST * s;
INTEGER64 fofs;
PN64_HEADER * hp;
int err;
	ret = 0;
	err = 0;
	lock_task(sp_lock);
	_set_wa_op_cnt(wa);
	if ( _set_sp_lock_list(wa,path,SPLT_OP,wait_flag) < 0 ) {
		err = -1;
		goto end;
	}
retry:
	for ( s = wa->sp_lock_op_list ; s ; s = s->next ) {
		if ( l_strcmp(s->path,path) )
			continue;
		if ( l_strcmp(s->ext,ext) )
			continue;
		if ( wait_flag ) {
			err = -1;
			goto end;
		}
		sleep_task((int)wa,sp_lock);
		lock_task(sp_lock);
		goto retry;
	}
	ret = _get_sp_data(&fofs,wa,path,ext);
	s = d_alloc(sizeof(*s));
	s->path = ll_copy_str(path);
	s->ext = ll_copy_str(ext);
	s->fofs = fofs;
	if ( ret ) {
		hp = ret;
		s->h = *hp;
	}
	else {
		s->h.size = 0;
	}
	change_endian_header64(&s->h);
	s->next = wa->sp_lock_op_list;
	wa->sp_lock_op_list = s;
end:
	if ( err_p )
		*err_p = err;
	unlock_task(sp_lock,"");
	return ret;
}

int
delete_sp_operation_auth(SPIDERING_WORK_AREA * wa,L_CHAR * path,L_CHAR * ext,int wait_flag)
{
int ret;
	ret = sp_agent_auth(wa,path);
	if ( ret < 0 )
		return ret;
	set_sp_lock_list(wa,path,SPLT_OP,wait_flag);
	ret = delete_sp_data(wa,path,ext);
	reset_sp_lock_list(wa,path,SPLT_OP);
	return ret;
}

void *
start_sp_operation_auth(int * err_p,SPIDERING_WORK_AREA * wa,L_CHAR* path,L_CHAR * ext,int wait_flag)
{
int ret;
	ret = sp_agent_auth(wa,path);
	if ( ret < 0 )
		return 0;
	return start_sp_operation(err_p,wa,path,ext,wait_flag);
}


int
finish_sp_operation(SPIDERING_WORK_AREA * wa,L_CHAR* path,L_CHAR * ext,void * data)
{
SP_LOCK_OP_LIST * s,**sp;
PN64_HEADER * hp,h;
INTEGER64 fofs;
STRING_INDEX64 * ix;
int len1,len2,_len;
L_CHAR * _path;
FAVT64_NODE * fn,* fn2;
	lock_task(sp_lock);
	for ( sp = &wa->sp_lock_op_list ; *sp ; sp = &(*sp)->next ) {
		s = *sp;
		if ( l_strcmp(s->path,path) )
			continue;
		if ( l_strcmp(s->ext,ext) )
			continue;
		goto ok;
	}
	return -1;
ok:
	if ( data == 0 )
		goto end;
	hp = data;
	h = *hp;
	change_endian_header64(&h);
	if ( s->h.size == 0 ) {
		goto write;
	}
	else if ( s->h.size != h.size ) {
		l_free_filespace64(wa->pdb_fd,s->fofs);
	write:
		fofs = l_alloc_filespace64(wa->pdb_fd,&h);
		l_write_filespace64f(wa->pdb_fd,fofs,data);
		
		_path = ll_copy_str(path);
		len1 = l_strlen(_path);
		len2 = l_strlen(ext);
		_path = d_re_alloc(_path,sizeof(L_CHAR)*(len1+len2+1));
		l_strcpy(&_path[len1],ext);
		ix = d_alloc(_len=STRING_IX64_LENGTH_1(len1+len2));
		ix->fofs = 0;
		l_strcpy(&ix->d[0],_path);
		
		fn = l_favt64_search(&wa->favt64_err,
				wa->rt_data_que,
				l_root_node64(&wa->favt64_err,wa->rt_data_que),
				ix,favt64_cmp_string_1,0);
		if ( fn == 0 ) {
			ix->fofs = fofs;
			fn = l_favt64_alloc_node(
					wa->rt_data_que,
					ix,_len);
			fn2 = l_favt64_insert(&wa->favt64_err,
					wa->rt_data_que,
					&wa->rt_data_que->node,fn,
					favt64_cmp_string_1,0);
			if ( fn2 != fn )
				er_panic("finish_op");
			
			d_f_ree(ix);
		}
		else {
		
			d_f_ree(ix);
			
			ix = fn->data;
			ix->fofs = fofs;
			fn->h.flags |= FAF_DIRTY;
		}
	}
	else {
		l_write_filespace64f(wa->pdb_fd,s->fofs,data);
	}
end:
	*sp = s->next;
	d_f_ree(s->path);
	d_f_ree(s->ext);
	d_f_ree(s);
	wakeup_task((int)s);
	_reset_sp_lock_list(wa,path,SPLT_OP);
	_reset_wa_op_cnt(wa);
	unlock_task(sp_lock,"");
	return 0;
}

int
finish_sp_operation_auth(SPIDERING_WORK_AREA * wa,L_CHAR* path,L_CHAR * ext,void * data)
{
int ret;
	ret = sp_agent_auth(wa,path);
	if ( ret < 0 )
		return 0;
	return finish_sp_operation(wa,path,ext,data);
}



SP_READDIR_LIST *
readdir_sp_list(SPIDERING_WORK_AREA * wa,L_CHAR * path)
{
L_CHAR * _path;
int len1,_len;
STRING_INDEX64 * ix,*ix_large,*ixp;
BOUND64_LIST *bp, * blp, * last_bp;

int flag;
SP_READDIR_LIST * ret,**retp,*r;
int len2;

	set_wa_op_cnt(wa);
	set_sp_lock_list(wa,path,SPLT_READDIR,1);
	
	_path = ll_copy_str(path);
	len1 = l_strlen(_path);
	
	_path = d_re_alloc(_path,sizeof(L_CHAR)*(len1+4));
	
	_path[len1] = LC_SECC_HEADER;
	_path[len1+1] = LC_SECC_READDIR;
	_path[len1+2] = 0;
	
	ix = d_alloc(_len=STRING_IX64_LENGTH_1(len1+2));
	ix->fofs = 0;
	l_strcpy(&ix->d[0],_path);
	
	_path[len1+1] = LC_SECC_TERMINATOR;
	_path[len1+2] = 0;

	ix_large = d_alloc(_len=STRING_IX64_LENGTH_1(len1+3));
	ix_large->fofs = 0;
	l_strcpy(&ix_large->d[0],_path);

	ret = 0;
	retp = &ret;

	flag = 1;
	for ( ; ; ) {
		bp = l_favt64_bound_search(
			&wa->favt64_err,wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_data_que),
			ix,ix_large,10,
			favt64_cmp_string_1,0);
		if ( bp == 0 )
			break;
		if ( flag ) {
			ixp = bp->data;
			r = d_alloc(sizeof(*r));
			r->path = ll_copy_str(&ixp->d[0]);
			r->next = 0;
			*retp = r;
			retp = &r->next;
		}
		flag = 0;
		if ( bp->next == 0 ) {
			free_bound_list64(bp);
			break;
		}
		for ( blp = bp->next ; blp ; blp = blp->next ) {
			ixp = blp->data;
			r = d_alloc(sizeof(*r));
			r->path = ll_copy_str(&ixp->d[0]);
			r->next = 0;
			*retp = r;
			retp = &r->next;
			last_bp = blp;
		}
		ixp = last_bp->data;
		len2 = l_strlen(&ixp->d[0]);
		_len = STRING_IX64_LENGTH_1(len2);
		ix = d_re_alloc(ix,_len);
		memcpy(ix,ixp,_len);
		
		free_bound_list64(bp);
	}
	
	d_f_ree(ix);
	d_f_ree(ix_large);
	d_f_ree(_path);
	
	reset_sp_lock_list(wa,path,SPLT_READDIR);
	reset_wa_op_cnt(wa);
	return ret;
}

SP_READDIR_LIST *
readdir_sp_list_auth(SPIDERING_WORK_AREA * wa,L_CHAR * path)
{
int ret;
	ret = sp_agent_auth(wa,path);
	if ( ret < 0 )
		return 0;
	return readdir_sp_list(wa,path);
}





void
free_sp_readdir_list(SP_READDIR_LIST * lst)
{
SP_READDIR_LIST * lst2;
	for ( ; lst ; ) {
		lst2 = lst->next;
		d_f_ree(lst->path);
		d_f_ree(lst);
		lst = lst2;
	}
}



SP_READDIR_LIST *
readdir_sp_list_all(SPIDERING_WORK_AREA * wa)
{
L_CHAR * _path;
int _len;
STRING_INDEX64 * ix,*ix_large,*ixp;
BOUND64_LIST *bp, * blp, * last_bp;
int len2;
int flag;
SP_READDIR_LIST * ret,**retp,*r;

	set_wa_op_cnt(wa);
	
	
	_path = d_alloc(sizeof(L_CHAR)*(4));
	
	_path[0] = LC_SECC_HEADER;
	_path[1] = 0;
	
	ix = d_alloc(_len=STRING_IX64_LENGTH_1(1));
	ix->fofs = 0;
	l_strcpy(&ix->d[0],_path);
	
	_path[0] = LC_SECC_TERMINATOR;
	_path[1] = 0;

	ix_large = d_alloc(_len=STRING_IX64_LENGTH_1(1));
	ix_large->fofs = 0;
	l_strcpy(&ix_large->d[0],_path);

	ret = 0;
	retp = &ret;

	flag = 1;
	for ( ; ; ) {
		bp = l_favt64_bound_search(
			&wa->favt64_err,wa->rt_data_que,
			l_root_node64(&wa->favt64_err,wa->rt_data_que),
			ix,ix_large,10,
			favt64_cmp_string_1,0);
		if ( bp == 0 )
			break;
		if ( flag ) {
			ixp = bp->data;
			r = d_alloc(sizeof(*r));
			r->path = ll_copy_str(&ixp->d[0]);
			r->next = 0;
			*retp = r;
			retp = &r->next;
		}
		flag = 0;
		if ( bp->next == 0 ) {
			free_bound_list64(bp);
			break;
		}
		for ( blp = bp->next ; blp ; blp = blp->next ) {
			ixp = blp->data;
			r = d_alloc(sizeof(*r));
			r->path = ll_copy_str(&ixp->d[0]);
			r->next = 0;
			*retp = r;
			retp = &r->next;
			last_bp = blp;
		}
		ixp = last_bp->data;
		len2 = l_strlen(&ixp->d[0]);
		_len = STRING_IX64_LENGTH_1(len2);
		ix = d_re_alloc(ix,_len);
		memcpy(ix,ixp,_len);
		
		free_bound_list64(bp);
	}
	
	d_f_ree(ix);
	d_f_ree(ix_large);
	d_f_ree(_path);
	
	reset_wa_op_cnt(wa);
	return ret;
}

unsigned int 
get_sp_agent_hash_key(L_CHAR * name)
{
unsigned int ret;
	ret = 0;
	for ( ; *name ; name ++ , ret += *name );
	return ret % SPA_HASH_SIZE;
}


SPIDERING_AGENT *
_search_sp_agent(unsigned int * keyp,L_CHAR * name)
{
unsigned int key;
SPIDERING_AGENT * ret;
	key = get_sp_agent_hash_key(name);
	*keyp = key;
	for ( ret = spa_hash[key] ; ret ; ret = ret->next )
		if ( l_strcmp(ret->name,name) == 0 )
			return ret;
	return 0;
}


SPIDERING_AGENT *
_search_sp_agent_by_iid(int iid)
{
unsigned int key;
SPIDERING_AGENT * ret;
	key = iid % SPA_HASH_SIZE;
	for ( ret = spa_hash[key] ; ret ; ret = ret->iid_next )
		if ( ret->d.outer.iid == iid )
			return ret;
	return 0;
}

SPIDERING_AGENT *
search_sp_agent_by_iid(int iid)
{
SPIDERING_AGENT * ret;
	lock_task(sp_lock);
	ret = _search_sp_agent_by_iid(iid);
	unlock_task(sp_lock,"");
	return ret;
}




SPIDERING_AGENT*
_new_sp_agent(L_CHAR * name)
{
SPIDERING_AGENT * ret;
unsigned int key;
	ret = _search_sp_agent(&key,name);
	if ( ret )
		return 0;
	ret = d_alloc(sizeof(*ret));
	memset(ret,0,sizeof(*ret));
	ret->name = ll_copy_str(name);
	ret->next = spa_hash[key];
	spa_hash[key] = ret;
	return ret;
}

SPIDERING_AGENT*
new_sp_agent(L_CHAR * name)
{
SPIDERING_AGENT * ret;	
	lock_task(sp_lock);
	ret = _new_sp_agent(name);
	unlock_task(sp_lock,"");
	return ret;
}

SPIDERING_AGENT*
new_outer_sp_agent(L_CHAR * name,L_CHAR * core,L_CHAR * loading)
{
SPIDERING_AGENT * ret;
	lock_task(sp_lock);
	ret = _new_sp_agent(name);
	if ( ret == 0 )
		return ret;
	ret->d.outer.agent_core = ll_copy_str(core);
	ret->d.outer.loading_file = ll_copy_str(loading);
	ret->type = SPAT_OUTER;
	unlock_task(sp_lock,"");
	return ret;
}

void
_sp_agent_write_lock(SPIDERING_AGENT * spa)
{
	for ( ; spa->access_lock ; ) {
		sleep_task((int)spa,sp_lock);
		lock_task(sp_lock);
	}
	spa->access_lock = -1;
}

void
_sp_agent_read_lock(SPIDERING_AGENT * spa)
{
	for ( ; spa->access_lock < 0 ; ) {
		sleep_task((int)spa,sp_lock);
		lock_task(sp_lock);
	}
	spa->access_lock ++;
}

void
_sp_agent_unlock(SPIDERING_AGENT * spa)
{
	if ( spa->access_lock > 0 )
		spa->access_lock --;
	else if ( spa->access_lock < 0 )
		spa->access_lock = 0;
	else	er_panic("_sp_agent_unlock");
	wakeup_task((int)spa);
}


XL_SEXP *
_call_sp_agent(SPIDERING_AGENT * spa,XL_SEXP * cmd)
{
XL_SEXP * ret;
XL_SEXP * loading,*sym;
int iid;

retry_null:
	switch ( spa->type ) {
	case SPAT_NULL:
		for ( ; spa->type == SPAT_NULL ; ) {
			sleep_task((int)spa,sp_lock);
			lock_task(sp_lock);
		}
		goto retry_null;
/*
		return get_error(
			0,
			0,
			XLE_PROTO_INV_RESOURCE,
			l_string(std_cm,"call_sp_agent"),
			n_get_string("spa is not initialized"));
*/
	case SPAT_INNER:
		gc_push(0,0,"call_sp_agent");
		
		_sp_agent_read_lock(spa);
		unlock_task(sp_lock,"");
		
		ret = eval(spa->d.inner.env,cmd);
		
		lock_task(sp_lock);
		_sp_agent_unlock(spa);
		gc_pop(ret,gc_gb_sexp);
		break;
	case SPAT_OUTER:
		gc_push(0,0,"call_sp_agent");
	retry:
		iid = spa->d.outer.iid;
		if ( iid < 0 ) {
			_sp_agent_write_lock(spa);
			iid = spa->d.outer.iid;
			if ( iid >= 0 ) {
				_sp_agent_unlock(spa);
				goto ok;
			}
			unlock_task(sp_lock,"");
		
			ret = Shell_target(spa->d.outer.agent_core,0,sp_env,0);
			
			lock_task(sp_lock);

			switch ( get_type(ret) ) {
			case XLT_INTEGER:
				spa->d.outer.iid = iid = ret->integer.data;
				break;
			case XLT_ERROR:
				spa->err = -1;
				ret = get_error(
					0,
					0,
					XLE_PROTO_INV_RESOURCE,
					l_string(std_cm,"call_sp_agent"),
					n_get_string("invoke agent"));
				_sp_agent_unlock(spa);
				goto err;
			default:
				er_panic("_call_sp_agent");
			}
			unlock_task(sp_lock,"");
			
			sym = n_get_symbol("Load");
			set_attribute(sym,l_string(std_cm,"option"),l_string(std_cm,"exec"));
			
			loading = List(sym,get_string(spa->d.outer.loading_file));
			
			ret = remote_query(spa->d.outer.iid,sp_env,0,loading);
			
			lock_task(sp_lock);

			switch ( get_type(ret) ) {
			case XLT_ERROR:
				close_interpreter(spa->d.outer.iid);
				spa->d.outer.iid = -1;
				spa->err = -1;
				ret = get_error(
					0,
					0,
					XLE_PROTO_INV_RESOURCE,
					l_string(std_cm,"call_sp_agent"),
					n_get_string("invoke agent"));
				_sp_agent_unlock(spa);
				goto err;
			default:
				break;
			}
			lock_task(sp_lock);
			_sp_agent_unlock(spa);
		}
	ok:
		_sp_agent_read_lock(spa);
		unlock_task(sp_lock,"");

		ret = remote_query(iid,sp_env,0,cmd);
		if ( ret && ret->h.type == XLT_ERROR ) {
			switch ( ret->err.code ) {
			case XLE_PROTO_ACCESS_STREAM:
			case XLE_PROTO_INV_IID:
				lock_task(sp_lock);
				spa->d.outer.iid = -1;
				goto retry;
			}
		}

		lock_task(sp_lock);
		_sp_agent_unlock(spa);

		spa->err = 0;
		gc_pop(ret,gc_gb_sexp);
		break;
	default:
		er_panic("_call_sp_agent");
	}
err:
	return ret;
}

XL_SEXP *
call_sp_agent(SPIDERING_AGENT * spa,XL_SEXP * cmd)
{
XL_SEXP * ret;
	lock_task(sp_lock);
	ret = _call_sp_agent(spa,cmd);
	unlock_task(sp_lock,"");
	return ret;
}


void
set_sp_agent_type(SPIDERING_AGENT * spa,int type)
{
	lock_task(sp_lock);
	spa->type = type;
	wakeup_task((int)spa);
	unlock_task(sp_lock,"");
}



