// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prorate;

import java.io.Serializable;

import jp.sourceforge.gnp.prorate.ProrateAuditImpl;
import jp.sourceforge.gnp.prorate.ProrateTax;
import jp.sourceforge.gnp.prorate.database.ProrateTaxTable;
import jp.sourceforge.gnp.prorate.export.ProrateAudit;
import jp.sourceforge.gnp.prorate.export.ProrateTaxData;
import jp.sourceforge.gnp.prorate.export.ProrateSector;


/* TAX ʬ󥸥 */

public class ProrateTaxImpl implements ProrateTax, Serializable {

  /**
   * 
   */
  private static final long serialVersionUID = 1L;
  static final int	ABORT = -1;
  static final int	NO_HITS = 1;
  static final int	ALL_HITS = 2;
  static final int	FIRST_HIT = 4;
  static final int	OK = 0;

  /**
   * variable <code>audit</code> 	audit information
   *
   */
  ProrateAudit	audit;
  /**
   * variable <code>engine</code>	audit proration engine
   *
   */
  ProrateAuditImpl engine;
  /**
   * variable <code>informError</code>	flag whether inform error or not
   *
   */
  boolean	informError;

  /**
   * Creates a new <code>ProrateTaxImpl</code> instance.
   *
   */
  public ProrateTaxImpl() {
  }

  /**
   * <code>divide</code> method
   *
   * @param audit a <code>ProrateAudit</code> value
   * @return a <code>boolean</code> value
   */
  public boolean	divide(ProrateAudit audit) {
    this.audit = audit;
    if (audit.getTax() == null) {
      return true;
    }
    
    informError = true;
    if (audit.getErrorFlag() != 0) {
      informError = false;
    }
    if (informError) {
      for (int i = 0; i < audit.getSectors().length; i++) {
	ProrateSector	sector = (ProrateSector)audit.getSectors()[i];
	if (sector.getErrorFlag() != 0) {
	  informError = false;
	}
      }
    }
    
    if (!audit.isOwnAudit() && !audit.isInward()) {
      return false;
    }
    if (audit.getSectors().length == 0) {
      return false;
    }

    boolean	taxDivided = false;
    for (int i = 0; i < audit.getSectors().length; i++) {
      ProrateSector	sector = (ProrateSector)audit.getSectors()[i];
      if (sector.getTax() > (double)0) {	/* TAXʬѤ */
	return true;	/* TAXʬԤʤʤ */
      }
      if (sector.getTaxIndex() > 0) {	/* TAXʬѤ */
	taxDivided = true;
      }
    }
    
    if (!taxDivided) {
      double	totalTax3 = (double)0;
      for (int i = 0; i < audit.getTax().length; i++) {
	ProrateTaxData	tax = (ProrateTaxData)audit.getTax()[i];
	totalTax3 += tax.getAmount();
      }
      double	totalTax10 = (double)0;
      for (int i = 0; i < audit.getTaxMisc().length; i++) {
	ProrateTaxData	tax = (ProrateTaxData)audit.getTaxMisc()[i];
	totalTax10 += tax.getAmount();
      }

      for (int i = 0; i < audit.getTaxMisc().length; i++) {
	ProrateTaxData	tax = (ProrateTaxData)audit.getTaxMisc()[i];
	if (tax.getAmount() == (double)0) {
	  continue;
	}
	String	type = tax.getType();
	if (type.length() > 2 && type.substring(0, 2).equals("XF")) {
	  continue;
	}
	if (type.length() < 2) {
	  continue;
	}
	int	ret = NO_HITS;
	ProrateTaxTable	taxTable =
	  getEngine().database.getTaxTable(type.substring(0, 2),
					   audit.getIssueDate());
	if (taxTable == null) {
	  if (getEngine().database.getResult() > 1) {
	    if (informError) {
	      getEngine().DBError((ProrateSector)null,
				  "getTaxTable", type.substring(0, 2),
				  audit.getIssueDate());
	    }
	    return false;
	  }
	  if (informError) {
	    getEngine().error(ProrateAudit.ERA_TAXNOTFOUND,
			      type.substring(0, 2));
	  }
	  continue;
	}

	for (int j = 0;
	     j < 3 && j < taxTable.id.length && taxTable.id[j] > 0; j++) {
	  for (int k = 0; k < audit.getSectors().length; k++) {
	    int	retval = NO_HITS;
	    ProrateSector sector = (ProrateSector)audit.getSectors()[k];
	    boolean isStpo = !(sector.getStopOver() == '\0'
			       || sector.getStopOver() == ' '
			       || sector.getStopOver() == '0');
	    retval = checkDivide(taxTable, i, j,
				 ((type.equals("UH") || type.equals("TH")) ?
				  sector.getDepAirport() :
				  sector.getDepCode()),
				 ((type.equals("UH") || type.equals("TH")) ?
				  sector.getDestAirport() :
				  sector.getDestCode()),
				 isStpo, k);
	    if (retval != NO_HITS) {
	      ret = retval;
	    }
	    if (ret == ABORT) {
	      return false;
	    }
	    if (ret == FIRST_HIT) {
	      break;
	    }
	  }
	  if (ret != NO_HITS) {
	    break;
	  }
	}
	if (ret == NO_HITS) {
	  ret = checkNotApplicable(i);
	}
	if (ret == ABORT) {
	  return false;
	}

      }

    }

    if (!audit.isNotDivideTax()) {
      doDivide();
    }

    return true;
  }

  /* not implemented yet */
  /**
   * <code>checkDivide</code> method
   *	check if the tax is devided for the sector
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param level an <code>int</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDivide(ProrateTaxTable taxTable, int index, int level,
		    String from, String to, boolean isStpo, int sectorIndex) {
    ProrateTaxData
      tax = (ProrateTaxData)audit.getTaxMisc()[index];
    if (tax.getType().equals("XY") || tax.getType().equals("XF")) {
      return
	checkDivideXYXF(taxTable, index, level, from, to, isStpo, sectorIndex);
    }
    String	fromCountry = getEngine().getCountryName(from);
    if (fromCountry.equals("")) {
      if (getEngine().database.getResult() > 1) {
	if (informError) {
	  getEngine().DBError((ProrateSector)null, "getCountryName", from);
	}
	return ABORT;
      }
      if (informError) {
	getEngine().error(ProrateAudit.ASK_COUNTRY, from);
      }
      return ABORT;
    }
    String	toCountry = getEngine().getCountryName(to);
    if (toCountry.equals("")) {
      if (getEngine().database.getResult() > 1) {
	if (informError) {
	  getEngine().DBError((ProrateSector)null, "getCountryName", to);
	}
	return ABORT;
      }
      if (informError) {
	getEngine().error(ProrateAudit.ASK_COUNTRY, to);
      }
      return ABORT;
    }

    int	retval = NO_HITS;

    /* not implemented yet */
    switch (taxTable.id[level]) {
    case 1:	/* ȯݶʬ */
      retval = checkDepFirstIntl(taxTable, index, fromCountry, toCountry,
				 from, to, isStpo, sectorIndex);
      break;
    case 2:	/* ݶʬ */
      retval = checkDestFirstIntl(taxTable, index, fromCountry, toCountry,
				  from, to, isStpo, sectorIndex);
      break;
    case 3:	/* ȯݶʬ */
      retval = checkDepAllIntl(taxTable, index, fromCountry, toCountry,
			       from, to, isStpo, sectorIndex);
      break;
    case 4:	/* ݶʬ */
      retval = checkDestAllIntl(taxTable, index, fromCountry, toCountry,
				from, to, isStpo, sectorIndex);
      break;
    case 5:	/* ȯݶʬ(̾ʤ) */
      retval = checkDepFirstIntlNoPort(taxTable, index,
				       fromCountry, toCountry,
				       from, to, isStpo, sectorIndex);
      break;
    case 6:	/* ݶʬ(̾ʤ) */
      retval = checkDestFirstIntlNoPort(taxTable, index,
					fromCountry, toCountry,
					from, to, isStpo, sectorIndex);
      break;
    case 7:	/* ȯݶʬ(̾ʤ) */
      retval = checkDepAllIntlNoPort(taxTable, index, fromCountry, toCountry,
				     from, to, isStpo, sectorIndex);
      break;
    case 8:	/* ݶʬ(̾ʤ) */
      retval = checkDestAllIntlNoPort(taxTable, index,
				      fromCountry, toCountry,
				      from, to, isStpo, sectorIndex);
      break;
    case 9:	/* ȯʬ */
      retval = checkDepDestFirstDomestic(taxTable, index,
					 fromCountry, toCountry,
					 from, to, isStpo, sectorIndex);
      break;
    case 10:	/* ȯʬ */
      retval = checkDepDestAllDomestic(taxTable, index,
				       fromCountry, toCountry,
				       from, to, isStpo, sectorIndex);
      break;
    case 11:	/* ȯʬ */
      retval = checkDepFirstDomestic(taxTable, index, fromCountry, toCountry,
				     from, to, isStpo, sectorIndex);
      break;
    case 12:	/* ȯʬ */
      retval = checkDepAllDomestic(taxTable, index, fromCountry, toCountry,
				   from, to, isStpo, sectorIndex);
      break;
    case 13:	/* ʬ */
      retval = checkDestFirstDomestic(taxTable, index,
				      fromCountry, toCountry,
				      from, to, isStpo, sectorIndex);
      break;
    case 14:	/* ʬ */
      retval = checkDestAllDomestic(taxTable, index, fromCountry, toCountry,
				    from, to, isStpo, sectorIndex);
      break;
    case 15:	/* ȯʬ(̾ʤ) */
      retval = checkDepDestFirstDomesticNoPort(taxTable, index,
					       fromCountry, toCountry,
					       from, to, isStpo, sectorIndex);
      break;
    case 16:	/* ȯʬ(̾ʤ) */
      retval = checkDepDestAllDomesticNoPort(taxTable, index,
					     fromCountry, toCountry,
					     from, to, isStpo, sectorIndex);
      break;
    default:
      retval = NO_HITS;
      break;
    }

    return retval;
  }

  /* ȯݶTAXʬ(NO1) */
  /**
   * <code>checkDepFirstIntl</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepFirstIntl(ProrateTaxTable taxTable, int index,
			  String fromCountry, String toCountry,
			  String from, String to, boolean isStpo,
			  int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (from.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* ݶTAXʬ(NO2) */
  /**
   * <code>checkDestFirstIntl</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDestFirstIntl(ProrateTaxTable taxTable, int index,
			   String fromCountry, String toCountry,
			   String from, String to, boolean isStpo,
			   int sectorIndex) {
    if (!toCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* ȯݶTAXʬ(NO3) */
  /**
   * <code>checkDepAllIntl</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepAllIntl(ProrateTaxTable taxTable, int index,
			String fromCountry, String toCountry,
			String from, String to, boolean isStpo,
			int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (from.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /* ݶTAXʬ(NO4) */
  /**
   * <code>checkDestAllIntl</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDestAllIntl(ProrateTaxTable taxTable, int index,
			 String fromCountry, String toCountry,
			 String from, String to, boolean isStpo,
			 int sectorIndex) {
    if (!toCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /* ȯݶTAXʬ(̾ʤ)(NO5) */
  /**
   * <code>checkDepFirstIntlNoPort</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepFirstIntlNoPort(ProrateTaxTable taxTable, int index,
				String fromCountry, String toCountry,
				String from, String to, boolean isStpo,
				int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* ݶTAXʬ(̾ʤ)(NO6) */
  /**
   * <code>checkDestFirstIntlNoPort</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDestFirstIntlNoPort(ProrateTaxTable taxTable, int index,
				 String fromCountry, String toCountry,
				 String from, String to, boolean isStpo,
				 int sectorIndex) {
    if (!toCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* ȯݶTAXʬ(̾ʤ)(NO7) */
  /**
   * <code>checkDepAllIntlNoPort</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepAllIntlNoPort(ProrateTaxTable taxTable, int index,
			      String fromCountry, String toCountry,
			      String from, String to, boolean isStpo,
			      int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /* ݶTAXʬ(̾ʤ)(NO8) */
  /**
   * <code>checkDestAllIntlNoPort</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDestAllIntlNoPort(ProrateTaxTable taxTable, int index,
			       String fromCountry, String toCountry,
			       String from, String to, boolean isStpo,
			       int sectorIndex) {
    if (!toCountry.equals(taxTable.country)
	|| fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /* ȯTAXʬ(NO9) */
  /**
   * <code>checkDepDestFirstDomestic</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepDestFirstDomestic(ProrateTaxTable taxTable, int index,
				  String fromCountry, String toCountry,
				  String from, String to, boolean isStpo,
				  int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* ȯTAXʬ(NO10) */
  /**
   * <code>checkDepDestAllDomestic</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepDestAllDomestic(ProrateTaxTable taxTable, int index,
				String fromCountry, String toCountry,
				String from, String to, boolean isStpo,
				int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /* ȯTAXʬ(NO11) */
  /**
   * <code>checkDepFirstDomestic</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepFirstDomestic(ProrateTaxTable taxTable, int index,
			      String fromCountry, String toCountry,
			      String from, String to, boolean isStpo,
			      int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* ȯTAXʬ(NO12) */
  /**
   * <code>checkDepAllDomestic</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepAllDomestic(ProrateTaxTable taxTable, int index,
			    String fromCountry, String toCountry,
			    String from, String to, boolean isStpo,
			    int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /* TAXʬ(NO13) */
  /**
   * <code>checkDestFirstDomestic</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDestFirstDomestic(ProrateTaxTable taxTable, int index,
			       String fromCountry, String toCountry,
			       String from, String to, boolean isStpo,
			       int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* TAXʬ(NO14) */
  /**
   * <code>checkDestAllDomestic</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDestAllDomestic(ProrateTaxTable taxTable, int index,
			     String fromCountry, String toCountry,
			     String from, String to, boolean isStpo,
			     int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    if (taxTable.city.size() > 0) {
      boolean	cityMatch = false;
      for (int i = 0; i < taxTable.city.size(); i++) {
	if (to.equals(taxTable.city.get(i))) {
	  cityMatch = true;
	  break;
	}
      }
      if (!cityMatch) {
	return NO_HITS;
      }
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /* ȯTAXʬ(̾ʤ)(NO15) */
  /**
   * <code>checkDepDestFirstDomesticNoPort</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepDestFirstDomesticNoPort(ProrateTaxTable taxTable, int index,
					String fromCountry, String toCountry,
					String from, String to, boolean isStpo,
					int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return FIRST_HIT;
  }
  
  /* ȯTAXʬ(̾ʤ)(NO16) */
  /**
   * <code>checkDepDestAllDomesticNoPort</code> method
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDepDestAllDomesticNoPort(ProrateTaxTable taxTable, int index,
				      String fromCountry, String toCountry,
				      String from, String to, boolean isStpo,
				      int sectorIndex) {
    if (!fromCountry.equals(taxTable.country)
	|| !fromCountry.equals(toCountry)) {
      return NO_HITS;
    }
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    return ALL_HITS;
  }
  
  /**
   * <code>checkDivideXYXF</code> method
   *	check if the tax XY or XF is devided for the sector
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param level an <code>int</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkDivideXYXF(ProrateTaxTable taxTable, int index, int level,
			String from, String to, boolean isStpo,
			int sectorIndex) {
    String	fromCountry = getEngine().getCountryName(from);
    if (fromCountry.equals("")) {
      if (getEngine().database.getResult() > 1) {
	if (informError) {
	  getEngine().DBError((ProrateSector)null, "getCountryName", from);
	}
	return ABORT;
      }
      if (informError) {
	getEngine().error(ProrateAudit.ASK_COUNTRY, from);
      }
      return ABORT;
    }
    String	toCountry = getEngine().getCountryName(to);
    if (toCountry.equals("")) {
      if (getEngine().database.getResult() > 1) {
	if (informError) {
	  getEngine().DBError((ProrateSector)null, "getCountryName", to);
	}
	return ABORT;
      }
      if (informError) {
	getEngine().error(ProrateAudit.ASK_COUNTRY, to);
      }
      return ABORT;
    }

    int	retval = NO_HITS;
    if (taxTable.id[level] == 98) {	/* ¾ʬXY */
      retval = checkXY(taxTable, index, level, fromCountry, toCountry,
		       from, to, isStpo, sectorIndex);
    }
    if (taxTable.id[level] == 99) {	/* ¾ʬXF */
      retval = checkXF(taxTable, index, level, fromCountry, toCountry,
		       from, to, isStpo, sectorIndex);
    }
    return retval;
  }

  /* XYTAXʬ */
  /**
   * <code>checkXY</code> method
   *	check if the tax XY is devided for the sector
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param level an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkXY(ProrateTaxTable taxTable, int index, int level,
		String fromCountry, String toCountry,
		String from, String to, boolean isStpo, int sectorIndex) {
    ProrateSector sector = (ProrateSector)audit.getSectors()[sectorIndex];
    switch (level) {
    case 0:
      if (fromCountry.equals(toCountry)) {
	return NO_HITS;
      }
      if (!toCountry.equals("USA")) {
	return NO_HITS;
      }
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
      return FIRST_HIT;
    case 1:
      if (!toCountry.equals(taxTable.country)) {
	return NO_HITS;
      }
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
      return FIRST_HIT;
    default:
      break;
    }
    return NO_HITS;
  }

  /* XFTAXʬ */
  /**
   * <code>checkXF</code> method
   *	check if the tax XF is devided for the sector
   *
   * @param taxTable a <code>ProrateTaxTable</code> value
   * @param index an <code>int</code> value
   * @param level an <code>int</code> value
   * @param fromCountry a <code>String</code> value
   * @param toCountry a <code>String</code> value
   * @param from a <code>String</code> value
   * @param to a <code>String</code> value
   * @param isStpo a <code>boolean</code> value
   * @param sectorIndex an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkXF(ProrateTaxTable taxTable, int index, int level,
		String fromCountry, String toCountry,
		String from, String to, boolean isStpo, int sectorIndex) {
    ProrateSector sector = (ProrateSector)audit.getSectors()[index];
    switch (level) {
    case 0:
      if (fromCountry.equals(toCountry)) {
	return NO_HITS;
      }
      if (!fromCountry.equals("USA")) {
	return NO_HITS;
      }
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
      return FIRST_HIT;
    case 1:
      if (!fromCountry.equals(taxTable.country)) {
	return NO_HITS;
      }
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
      return FIRST_HIT;
    case 2:
      if (!fromCountry.equals("USA") || !toCountry.equals("USA")) {
	return NO_HITS;
      }
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
      return FIRST_HIT;
    default:
      break;
    }
    return NO_HITS;
  }

  /**
   * <code>checkNotApplicable</code> method	check if the tax is applicable.
   *
   * @param index an <code>int</code> value
   * @return an <code>int</code> value
   */
  int	checkNotApplicable(int index) {
    int	foreignIndex = -1;
    int	ownIndex = -1;
    for (int i = 0; i < audit.getSectors().length; i++) {
      ProrateSector	sector = (ProrateSector)audit.getSectors()[i];
      String	fromCountry = getEngine().getCountryName(sector.getDepCode());
      if (fromCountry.equals("")) {
	if (getEngine().database.getResult() > 1) {
	  if (informError) {
	    getEngine().DBError((ProrateSector)null, "getCountryName",
				sector.getDepCode());
	  }
	  return ABORT;
	}
	if (informError) {
	  getEngine().error(ProrateAudit.ASK_COUNTRY, sector.getDepCode());
	}
	return ABORT;
      }
      String	toCountry = getEngine().getCountryName(sector.getDestCode());
      if (toCountry.equals("")) {
	if (getEngine().database.getResult() > 1) {
	  if (informError) {
	    getEngine().DBError((ProrateSector)null, "getCountryName",
				sector.getDestCode());
	  }
	  return ABORT;
	}
	if (informError) {
	  getEngine().error(ProrateAudit.ASK_COUNTRY, sector.getDestCode());
	}
	return ABORT;
      }
      if (!fromCountry.equals(toCountry)) {
	if (foreignIndex < 0) {
	  foreignIndex = i;
	}
	if (sector.getCarrier().equals(audit.getAirwayId()) && ownIndex < 0) {
	  ownIndex = i;
	}
	if (foreignIndex >= 0 && ownIndex >= 0) {
	  break;
	}
      }
    }
    if (ownIndex >= 0) {
      ProrateSector sector = (ProrateSector)audit.getSectors()[ownIndex];
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    }
    else if (foreignIndex >= 0) {
      ProrateSector
	sector = (ProrateSector)audit.getSectors()[foreignIndex];
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    }
    else {
      ProrateSector sector = (ProrateSector)audit.getSectors()[0];
      sector.setTaxIndex(sector.getTaxIndex() | (1 << index));
    }
    return OK;
  }

  /**
   * <code>doDivide</code> method	do tax dividing
   *
   */
  void	doDivide() {
    for (int i = 0; i < audit.getTaxMisc().length; i++) {

      ProrateTaxData	tax = (ProrateTaxData)audit.getTaxMisc()[i];
      if (tax.getAmount() == (double)0) {
	continue;
      }

      int	sectors = 0;
      for (int j = 0; j < audit.getSectors().length; j++) {
	ProrateSector sector = (ProrateSector)audit.getSectors()[j];
	if ((sector.getTaxIndex() & (1 << i)) > 0) {
	  sectors++;
	}
      }

      int	apply_cnt = 0;
      double	apply_error =
	tax.getAmount() - (tax.getAmount() / sectors) * sectors;
      for (int j = 0; j < audit.getSectors().length; j++) {
	ProrateSector sector = (ProrateSector)audit.getSectors()[j];
	if ((sector.getTaxIndex() & (1 << i)) > 0) {
	  double taxValue = (tax.getAmount() / sectors) / audit.getDay5Rate();
	  sector.setTaxInLocal(sector.getTaxInLocal()
			       + (tax.getAmount() / sectors));
	  apply_cnt ++;
	  if (apply_cnt == sectors) {	/* add error to last sector */
	    sector.setTaxInLocal(sector.getTaxInLocal() + apply_error);
	    sector.setTax(sector.getTaxInLocal() / audit.getDay5Rate());
	    getEngine().trace.traceTax(audit, taxValue,
				       tax.getType(),
				       tax.getAmount(), sectors,
				       apply_error, j);
	    break;
	  }
	  sector.setTax(sector.getTaxInLocal() / audit.getDay5Rate());
	  getEngine().trace.traceTax(audit, taxValue,
				     tax.getType(),
				     tax.getAmount(), sectors,
				     (double)0, j);
	}
      }

    }
  }

  /**
   * <code>getEngine</code> method	return audit proration engine
   *
   * @return a <code>ProrateAuditImpl</code> value
   */
  public ProrateAuditImpl getEngine() {
    return engine;
  }

  /**
   * <code>setEngine</code> method	set audit proration engine
   *
   * @param engine a <code>ProrateAuditImpl</code> value
   */
  public void setEngine(ProrateAuditImpl engine) {
    this.engine = engine;
  }
}
