// Copyright (c) 2002-2009  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prorate.struts.service;


import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.gnp.prorate.export.ProrateAudit;
import jp.sourceforge.gnp.prorate.export.ProrateTaxData;
import jp.sourceforge.gnp.prorate.export.ProrateSector;
import jp.sourceforge.gnp.prorate.struts.view.ProrateView;

/**
 * class <code>ProrateService</code>
 *	struts proration service using remote prorate client
 *
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public abstract class ProrateService implements IProrateService {

  // Constructor
  /**
   * Creates a new <code>ProrateService</code> instance.
   * @throws Exception 
   *
   */
  public ProrateService() throws Exception {
    super();
  }

  // Operations
  /**
   * <code>prorate</code> method
   *	requests a prorate calculation to the proration service program.
   *
   * @param audit a <code>ProrateAudit</code> value
   * @return a <code>Prorate</code> value
   * @exception Exception if an error occurs
   */
  abstract public ProrateView prorate(ProrateAudit audit);
  
  /**
   * <code>createView</code> method
   *	creates proration result view.
   *
   * @param audit a <code>ProrateAudit</code> value
   * @return a <code>ProrateView</code> value
   */
  public ProrateView createView(ProrateAudit audit) {

    ProrateView	view = new ProrateView();

    /* ;;; set audit data into view */
    view.setOwnAirwayId(audit.getOwnAirwayId());
    view.setAirwayNumber(audit.getAirwayNumber());
    view.setInwardNumber(audit.getInwardNumber());
    view.setEndorsement(audit.getEndorsement());
    view.setIssueDate(audit.getIssueDate());
    view.setIssuePlace(audit.getIssuePlace());
    view.setInvoiceMonth(audit.getInvoiceMonth());
    view.setOrigin(audit.getOrigin());
    view.setDestination(audit.getDestination());
    view.setCurrency(audit.getCurrency());
    view.setTicketFare(Double.toString(audit.getTicketFare()));
    view.setSalesCurrency(audit.getSalesCurrency());
    view.setFareCalculation(audit.getFareCalculation());
    view.setSalesFare(Double.toString(audit.getSalesFare()));
    view.setTotalNuc(Double.toString(audit.getTotalNuc()));
    view.setLessAmt(Double.toString(audit.getLessAmt()));
    view.setStopOverCharge(Double.toString(audit.getStopOverCharge()));
    view.setTourCode(audit.getTourCode());
    view.setAgentCode(audit.getAgentCode());
    view.setErrorFlag(Integer.toString(audit.getErrorFlag()));
    view.setFatalError(false);
    if (audit.getErrorFlag() > 0 || audit.getErrorString().length() > 0) {
      String	errorString = view.getErrorString();
      if (errorString.length() > 0) {
	errorString += "\n";
      }
      String	auditError = "";
      switch (audit.getErrorFlag()) {
      case ProrateAudit.ERA_SIDETRIP:
	auditError = "Value for a Side Trip not defined ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_FCOMP:
	auditError = "Value for the last Fare Component not defined ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_LASTSIDE:
	auditError = "Value for the last Side Trip not defined ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_TOTALNUC:
	auditError = "Amount Mismatch in Total Nuc and Fare ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_FARECALC:
	auditError = "Error in Fare Calculation ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_MEMORY:
	auditError = "Memory Exhausted ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_STRHEAP:
	auditError = "String Heap Block Memory Exhausted ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_NOFARE:
	auditError = "No Fare on Coupon Audit ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_FCOMPEMPTY:
	auditError = "Empty Fare Component ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_TAXNOTMATCH:
	auditError = "Tax amount total not match ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_TAXNOTFOUND:
	auditError = "Tax code is not found ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_OS:
	auditError = "OS error ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ERA_DATABASE:
	auditError = "database error ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ASK_AIRWAYID:
	auditError = "cannot get AirwayId of the ticket ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ASK_FACTOR:
	auditError = "Prorate Factor not found ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ASK_ROERATE:
	auditError = "cannot get RoeRate for ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ASK_5DAYRATE:
      case ProrateAudit.ASK_5DAYSPA:
      case ProrateAudit.ASK_5DAYAPDP:
	auditError = "cannot get 5DayRate for ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ASK_COUNTRY:
	auditError = "cannot get CountryName for ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ASK_AREA:
	auditError = "cannot get AreaName for ";
	view.setFatalError(true);
	break;
      case ProrateAudit.ASK_FIXEDFARE:
	auditError = "cannot get FixedFare for ";
	view.setFatalError(true);
	break;
      }
      auditError += audit.getErrorString();
      errorString += auditError;
      view.setErrorString(errorString);
    }

    if (audit.getTraceStrings() == null) {
      view.setTraceRows(0);
      view.setTraceString("");
    }
    else {
      view.setTraceRows(audit.getTraceStrings().length);
      StringBuffer	traceString = new StringBuffer();
      for (int i = 0; i < audit.getTraceStrings().length; i++) {
	if (i > 0) {
	  traceString.append("\n");
	}
	traceString.append(audit.getTraceStrings()[i]);
      }
      view.setTraceString(traceString.toString());
    }
    
    if (audit.getTax() != null) {
      for (int i = 0; i < audit.getTax().length; i++) {
	ProrateTaxData	tax = (ProrateTaxData)audit.getTax()[i];
	view.setTaxType(i, tax.getType());
	view.setTaxAmount(i, Double.toString(tax.getAmount()));
      }
    }
    if (audit.getTaxMisc() != null) {
      for (int i = 0; i < audit.getTaxMisc().length; i++) {
	ProrateTaxData	tax = (ProrateTaxData)audit.getTaxMisc()[i];
	view.setTaxMiscType(i, tax.getType());
	view.setTaxMiscAmount(i, Double.toString(tax.getAmount()));
      }
    }

    if (audit.getSectors().length <= 0) {
      return view;
    }

    /* ;;; set input data from iform into sectors */
    ArrayList<List>	sectorsList = new ArrayList<List>();
    for (int i = 0; i < audit.getSectors().length; i++) {
      ProrateSector	sector = (ProrateSector)audit.getSectors()[i];

      ArrayList<String>	list = new ArrayList<String>();

      list.add(sector.getStopOver() == 0 ? " " : "X/");
      list.add(sector.getDepCode());
      list.add(sector.getDestCode());
      if (sector.getFareBasisFullStr() != null
	  && sector.getFareBasisFullStr().length() > 0) {
	list.add(sector.getFareBasisFullStr());
      }
      else {
	list.add(sector.getFareBasis());
      }
      list.add(sector.getCarrier());
      list.add(sector.getClassOfService());
      list.add(sector.getFareComponent() < 0 ?
	       "" : Double.toString(sector.getFareComponent()));
      list.add(sector.getFlightNo().equals("-1") ? "" : sector.getFlightNo());
      list.add(sector.getFlightDate());
      list.add(Double.toString(sector.getProrateFactor()));

      String	prorationTypeStr = "";
      int	prorationType = sector.getProrationType();
      if ((prorationType & ProrateAudit.PRT_SPA) != 0) {
	prorationTypeStr = "Spa";
	if ((prorationType & ProrateAudit.PRT_PRVS) != 0) {
	  prorationTypeStr += "Pv";
	}
      }
      else if ((prorationType & ProrateAudit.PRT_APDP) != 0) {
	prorationTypeStr = "Apd";
      }
      else if ((prorationType & ProrateAudit.PRT_SRP) != 0) {
	prorationTypeStr = "Srp";
      }
      if ((prorationType & ProrateAudit.PRT_PFM) != 0) {
	prorationTypeStr += "Pf";
      }
      else if ((prorationType & ProrateAudit.PRT_FIX) != 0) {
	prorationTypeStr += "Fx";
      }
      else if ((prorationType & ProrateAudit.PRT_FIXSRP) != 0) {
	prorationTypeStr += "Fs";
      }
      else if ((prorationType & ProrateAudit.PRT_NP) != 0) {
	prorationTypeStr += "Np";
      }
      if ((prorationType & ProrateAudit.PRT_HIGHSPA) != 0) {
	prorationTypeStr += "Hig";
      }
      else if ((prorationType & ProrateAudit.PRT_LOWSPA) != 0) {
	prorationTypeStr += "Low";
      }
      else if ((prorationType & ProrateAudit.PRT_HIGHSPAFIX) != 0) {
	prorationTypeStr += "Hif";
      }
      else if ((prorationType & ProrateAudit.PRT_LOWSPAFIX) != 0) {
	prorationTypeStr += "Lof";
      }
      if ((prorationType & ProrateAudit.PRT_MULSPA) != 0) {
	prorationTypeStr += "Ml";
      }
      else if ((prorationType & ProrateAudit.PRT_MULAPDP) != 0) {
	prorationTypeStr += "Ml";
      }
      list.add(prorationTypeStr);

      list.add(Double.toString(sector.getProrateValue()));
      list.add(Double.toString(sector.getNpPvalues()));
      list.add(Double.toString(sector.getInvoiceValue()));

      list.add(sector.getSpaBaseAmtType());
      list.add(Double.toString(sector.getSpaBaseAmt()));
      list.add(Double.toString(sector.getSpaDay5Rate()));
      list.add(Double.toString(sector.getSpaDiscountRate()));
      list.add(Double.toString(sector.getSpaClassDiff()));
      list.add(Double.toString(sector.getSpaNuc()));
      list.add(sector.getApdpBaseAmtType());
      list.add(Double.toString(sector.getApdpBaseAmt()));
      list.add(Double.toString(sector.getApdpDay5Rate()));
      list.add(Double.toString(sector.getApdpDiscountRate()));
      list.add(Double.toString(sector.getApdpClassDiff()));
      list.add(Double.toString(sector.getApdpNuc()));
      list.add(Double.toString(sector.getSrpNuc()));
      list.add(Double.toString(sector.getProratedClassDiff()));
      list.add(Double.toString(sector.getExstPlus()));
      list.add(Double.toString(sector.getProratedExstPlus()));
      list.add(Double.toString(sector.getTax()));
      list.add(Double.toString(sector.getTaxInLocal()));
      list.add(Double.toString(sector.getCommission()));
      list.add(Double.toString(sector.getClassDiffPlus()));
      list.add(Double.toString(sector.getSecureCharge()));
      list.add(Double.toString(sector.getSideTripPlus() < 0 ?
			       0 : sector.getSideTripPlus()));
      list.add(Double.toString(sector.getStopOverPlus()));
      list.add(Integer.toString(sector.getProrateError()));
      list.add(Double.toString(sector.getAmountInLocal()));
      list.add(Integer.toString(sector.getSideTripIndex()));
      list.add(Integer.toString(sector.getClassDiffIndex()));
      list.add(Integer.toString(sector.getSecureIndex()));
      list.add(Double.toString(sector.getFixedFareCheck()));
      list.add(Double.toString(sector.getFixedFareDiscount()));
      list.add(Integer.toString(sector.getComponentIndex()));
      list.add(Integer.toString(sector.getComponentKind()));
      list.add(sector.getViaRouting());
      list.add(Integer.toString(sector.getFareType()));
      list.add(Integer.toString(sector.getSequenceNo()));
      list.add(sector.getOpCarrier());
      list.add(Double.toString(sector.getNpClassDiff()));
      list.add(Double.toString(sector.getSecureChargeSaved()));
      list.add(Double.toString(sector.getProratedSecureCharge()));
      list.add(Double.toString(sector.getErrorFlag()));
      list.add(sector.getErrorString());

      if (sector.getErrorFlag() > 0 || sector.getErrorString().length() > 0) {
	String	errorString = view.getErrorString();
	if (errorString.length() > 0) {
	  errorString += "\n";
	}
	String	sectorError = "";
	switch (sector.getErrorFlag()) {
	case ProrateAudit.ERR_NUMARG:
	  sectorError = "wrong number of arguments ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_ILLARG:
	  sectorError = "illegal argument ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_FUNC:
	  sectorError = "function not defined or illegal definition ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_UNBOUND:
	  sectorError = "unbound variable ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_MULTI:
	  sectorError = "set multi path amount failed ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_PFM:
	  sectorError = "baseamount data not found ";
	  break;
	case ProrateAudit.ERR_VALUE:
	  sectorError = "SET: illegal value $DISCOUNT";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_ERROR:
	  sectorError = "";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_MEMORY:
	  sectorError = "Memory Exhausted";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_CURRENCY:
	  sectorError = "Currency not match ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ERR_DATABASE:
	  sectorError = "";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ASK_AIRWAYID:
	  sectorError = "cannot get AirwayId of the ticket ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ASK_FACTOR:
	  sectorError = "Prorate Factor not found ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ASK_ROERATE:
	  sectorError = "cannot get RoeRate for ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ASK_5DAYRATE:
	case ProrateAudit.ASK_5DAYSPA:
	case ProrateAudit.ASK_5DAYAPDP:
	  sectorError = "cannot get 5DayRate for ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ASK_COUNTRY:
	  sectorError = "cannot get CountryName for ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ASK_AREA:
	  sectorError = "cannot get AreaName for ";
	  view.setFatalError(true);
	  break;
	case ProrateAudit.ASK_FIXEDFARE:
	  sectorError = "cannot get FixedFare for ";
	  view.setFatalError(true);
	  break;
	}

	sectorError += sector.getErrorString();
	errorString += "sector " + i + " : " + sectorError;
	view.setErrorString(errorString);
      }

      sectorsList.add(list);
    }
    view.setSectorsList(sectorsList);
    
    if (audit.isOwnAudit() && audit.isTicketing()) {
      view.setProrateType("issued by own carrier");
    }
    else if (audit.isOwnAudit() && audit.isInward()) {
      view.setProrateType("inward");
    }
    else {
      view.setProrateType("outward");
    }
    
    if (audit.getTraceLevel() == 10000) {
      view.setTraceLevel("detail");
    }

    return view;
  }

  /**
   * <code>createErrorView</code> method
   *	creates proration error view.
   *
   * @param audit a <code>ProrateAudit</code> value
   * @return a <code>ProrateView</code> value
   */
  public ProrateView createErrorView(ProrateAudit audit) {

    ProrateView	view = new ProrateView();

    /* ;;; set audit data into view */
    view.setAirwayNumber(audit.getAirwayNumber());
    view.setInwardNumber(audit.getInwardNumber());
    view.setEndorsement(audit.getEndorsement());
    view.setIssueDate(audit.getIssueDate());
    view.setIssuePlace(audit.getIssuePlace());
    view.setInvoiceMonth(audit.getInvoiceMonth());
    view.setOrigin(audit.getOrigin());
    view.setDestination(audit.getDestination());
    view.setCurrency(audit.getCurrency());
    view.setSalesCurrency(audit.getSalesCurrency());
    view.setFareCalculation(audit.getFareCalculation());
    view.setTicketFare(Double.toString(audit.getTicketFare()));
    view.setSalesFare(Double.toString(audit.getSalesFare()));
    view.setTotalNuc(Double.toString(audit.getTotalNuc()));
    view.setLessAmt(Double.toString(audit.getLessAmt()));
    view.setStopOverCharge(Double.toString(audit.getStopOverCharge()));
    view.setErrorFlag(Integer.toString(audit.getErrorFlag()));
    view.setErrorString(audit.getErrorString());
    if (audit.getTraceStrings() == null) {
      view.setTraceRows(0);
      view.setTraceString("");
    }
    else {
      view.setTraceRows(audit.getTraceStrings().length);
      StringBuffer	traceString = new StringBuffer();
      for (int i = 0; i < audit.getTraceStrings().length; i++) {
	if (i > 0) {
	  traceString.append("\n");
	}
	traceString.append(audit.getTraceStrings()[i]);
      }
      view.setTraceString(traceString.toString());
    }

    if (audit.getSectors().length <= 0) {
      return view;
    }

    /* ;;; set input data from iform into sectors */
    for (int i = 0; i < (int)audit.getSectors().length; i++) {
      ProrateSector	sector = (ProrateSector)audit.getSectors()[i];
      if (sector.getDepCode().length() > 0
	  && sector.getDestCode().length() > 0
	  && sector.getCarrier().length() > 0) {
	view.setStopOver(i, (sector.getStopOver() == (byte)0 ?
			     "" : Byte.toString(sector.getStopOver())));
	view.setDepCode(i, sector.getDepCode());
	view.setDestCode(i, sector.getDestCode());
	if (sector.getFareBasisFullStr() != null
	      && sector.getFareBasisFullStr().length() > 0) {
	  view.setFareBasis(i, sector.getFareBasisFullStr());
	}
	else {
	  view.setFareBasis(i, sector.getFareBasis());
	}
	view.setCarrier(i, sector.getCarrier());
	view.setClassOfService(i, sector.getClassOfService());
	view.setFlightNo(i, sector.getFlightNo());
	view.setFlightDate(i, sector.getFlightDate());
	try {
	  view
	    .setFareComponent(i, (sector.getFareComponent() < (double)0 ?
				  "" :
				  Double.toString(sector.getFareComponent())));
	  view
	    .setClassDiffPlus(i, (sector.getClassDiffPlus() < (double)0 ?
				  "" :
				  Double.toString(sector.getClassDiffPlus())));
	  view.setSecureCharge(i, (sector.getSecureCharge() < (double)0 ?
				   "" :
				   Double.toString(sector.getSecureCharge())));
	  view.setSideTripPlus(i, (sector.getSideTripPlus() < (double)0 ?
				   "" :
				   Double.toString(sector.getSideTripPlus())));
	  view.setStopOverPlus(i, (sector.getStopOverPlus() < (double)0 ?
				   "" :
				   Double.toString(sector.getStopOverPlus())));
	}
	catch (NumberFormatException e) {
	}
      }
    }

    return view;
  }
}
