// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import gnu.lists.LList;
import gnu.math.IntNum;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Vector;
import java.util.Properties;
import javax.swing.BoxLayout;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import jp.sourceforge.glj.lisp.Lisp;
import jp.sourceforge.gnp.prorate.export.Prorate;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseException;
import jp.sourceforge.gnp.rulebase.xml.PrubaeReaderXml;
import jp.sourceforge.gnp.rulebase.xml.PrubaeWriterXml;



/**
 * class <code>Prubae</code> is the top class of ProrationRUleBAseEditor.
 *	this class handles top editor panel, tree editor panel and tree model.
 *	also handles file command menu, about menu.
 *
 * @author   <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version   1.0
 */

public class Prubae implements ActionListener {

  /**
   * constant <code>propertyFile</code>
   *	property file name for Prubae application in gnp.prubae package.
   *
   */
  public static final
    String propertyFile = "jp/sourceforge/gnp/prubae.properties";

  /**
   * constant <code>PrubaeEditFile</code> defines that the edit type is file.
   */
  public static final String	PrubaeEditFile = "file";
  /**
   * constant <code>PrubaeEditUrl</code> defines that the edit type is url.
   */
  public static final String	PrubaeEditUrl = "url";
  /**
   * constant <code>prubaercFile</code>	prubaerc configuration file name
   */
  public static final String
    prubaercFile = "jp/sourceforge/gnp/prubae/prubaerc";

  // Attributes  
  /**
   * variable <code>rootPanel</code> is root panel of prubae window.
   *	on this panel, the editor panel(mainPanel) and the menubar are placed.
   *
   * @uml.property  name="rootPanel"
   */
  private JPanel rootPanel = null;  
  /**
   * variable <code>mainPanel</code> is main panel of prubae editor window.
   *	on this panel, in the left side, tree editing panel window is placed.
   *	on the right side, panel window for editing details is placed.
   *
   * @uml.property  name="mainPanel"
   */
  private JPanel mainPanel = null;  
  /**
   * variable <code>treePanel</code> is panel window of editing rule tree.
   * @uml.property  name="treePanel"
   */
  private JPanel treePanel = null;  
  /**
   * variable <code>partPanel</code> is panel window of editing detail rules.
   *	on this panel, you can edit the detail rules of each statement.
   *
   * @uml.property  name="partPanel"
   */
  private JPanel partPanel = null;
  /**
   * variable <code>fileMenu</code> is a file menu.
   *	new spa,apdp,extf,part items,
   *	open and save items are placed on this menu.
   *
   * @uml.property  name="fileMenu"
   */
  private JMenu	fileMenu = null;
  /**
   * variable <code>editMenu</code> is a file menu.
   *	cut, copy, paste items are placed on this menu.
   *
   * @uml.property  name="editMenu"
   */
  private JMenu	editMenu = null;
  /**
   * variable <code>helpMenu</code> is a help menu.
   *	now, about item is placed on this menu.
   *
   * @uml.property  name="helpMenu"
   */
  private JMenu	helpMenu = null;
  /**
   * variable <code>scrollPane</code> is a scroll pane of tree editing panel.
   *
   * @uml.property  name="scrollPane"
   */
  private JScrollPane	scrollPane = null;
  /**
   * variable <code>tree</code> is a Swing JTree tree used as rule tree.
   *
   * @uml.property  name="tree"
   */
  private JTree tree = null;
  /**
   * variable <code>editType</code> defines whether edit file or url.
   *	the value is whether PrubaeEditFile or PrubaeEditUrl.
   *
   * @uml.property  name="editType"
   */
  private String	editType = null;
  /**
   * variable <code>filename</code>
   *	is a filename from which to read rules, or to which save rules.
   *
   * @uml.property  name="filename"
   */
  private String filename = null;  
  /**
   * variable <code>urlname</code>
   *	is a url String from which to read rules, or to which save rules.
   *
   * @uml.property  name="urlname"
   */
  private String urlname = null;
  /**
   * variable <code>dtdDir</code>
   *	is a directory String of the dtd reference file.
   *
   * @uml.property  name="dtdDir"
   */
  /* ;;; deBug
  private String dtdDir = null;
  */
  /**
   * variable <code>rulebaseDirectory</code>
   *	is a directory String of the rulebase directory.
   *
   */
  private String rulebaseDirectory;
  /**
   * variable <code>additionalPropertyFilename</code>
   *	is a additional property filename to be loaded by rulebase object etc.
   *
   */
  private String additionalPropertyFilename = null;
  /**
   * variable <code>treeModel</code> is a Swing JTree tree model.
   *
   * @uml.property  name="treeModel"
   */
  private DefaultTreeModel treeModel = null;  
  /**
   * variable <code>chooser</code> is a Swing File Chooser Dialog.
   *	used when open menu or save menu is executed.
   *
   * @uml.property  name="chooser"
   */
  private JFileChooser chooser = null;

  /**
   * variable <code>configList</code> is a configure attribute list.
   *	not used in this version. described in prubaerc file.
   *
   * @uml.property  name="configList"
   */
  private LList	configList = (LList)Lisp.nil;
  /**
   * variable <code>doList</code> is a Do menu configuration list.
   *	this is used to define Do menu ComboBox items.
   *	described in prubaerc file.
   *	configuration lists are in Scheme Lisp format.
   *
   * @uml.property  name="doList"
   */
  private LList	doList = (LList)Lisp.nil;
  /**
   * variable <code>judgeList</code> is a Judge menu configuration list.
   *	this is used to define Judge menu ComboBox items.
   *	described in prubaerc file.
   *	configuration lists are in Scheme Lisp format.
   *
   * @uml.property  name="judgeList"
   */
  private LList	judgeList = (LList)Lisp.nil;
  /**
   * variable <code>valueList</code> is a Value menu configuraton list.
   *	this is used to define Value menu ComboBox items.
   *	described in prubaerc file.
   *	configuration lists are in Scheme Lisp format.
   *
   * @uml.property  name="valueList"
   */
  private LList	valueList = (LList)Lisp.nil;
  /**
   * variable <code>varList</code> is a Variable menu configuration list.
   *	this is used to define Value menu ComboBox items.
   *	described in prubaerc file.
   *	configuration lists are in Scheme Lisp format.
   *
   * @uml.property  name="varList"
   */
  private LList	varList = (LList)Lisp.nil;
  /**
   * variable <code>constList</code> is a Constant menu configuration list.
   *	this is used to define Value menu ComboBox items.
   *	described in prubaerc file.
   *	configuration lists are in Scheme Lisp format.
   *
   * @uml.property  name="constList"
   */
  private LList	constList = (LList)Lisp.nil;
  /**
   * variable <code>settablevarList</code> is a settable variable config list.
   *	not used in this version. described in prubaerc file.
   *	configuration lists are in Scheme Lisp format.
   *
   * @uml.property  name="settablevarList"
   */
  private LList	settablevarList = (LList)Lisp.nil;
  /**
   * variable <code>savedVarList</code>
   *	is a saved Variable menu configuration list.
   *
   */
  private LList	savedVarList = null;

  // Associations  
  /**
   * variable <code>model</code> is a model element object.
   *	which describes top model element of the rule under edit.
   *	null if no rule file is opened.
   * 
   * @uml.property name="model"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeModel model = null;

  /**
   * variable <code>view</code> is a view element object on right pane,
   *	which is combined to the model which you are editing the details.
   *	note that the model and the view is not paired normally.
   *	but it is always paired with the controller below.
   * 
   * @uml.property name="view"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeView view = null;

  /**
   * variable <code>controller</code> is a controller element object,
   *	which is combined to the model which you are editing the details.
   * 
   * @uml.property name="controller"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeController controller = null;

  /**
   * variable <code>reader</code> is the object to use to read rule from file.
   * 
   * @uml.property name="reader"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeReader reader = null;

  /**
   * variable <code>writer</code> is the object to use to write rule to file.
   * 
   * @uml.property name="writer"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeWriter writer = null;

  /**
   * variable <code>yank</code>	is the object which is in the yank buffer.
   *
   */
  protected PrubaeModel	yank = null;

  // Constructor  
  /**
   * Creates a new <code>Prubae</code> instance.
   * this creates all GUI elements on main panel and menu.
   * also reads configuration lists from prubaerc file(method initialize()).
   */
  public Prubae() {
    super();
    System.err.println("new Prubae()");
  }

  // Operations
  /**
   * <code>initialize</code> method 
   * reads configuration lists from prubaerc file.
   * configuration lists are in Scheme Lisp format.
   * this method uses GNU Kawa library(Java Scheme system).
   */
  public void initialize() {
    setAdditionalPropertyFilename(System
				  .getProperty("PRUBAE_PROPERTIES_FILE"));
    if (getAdditionalPropertyFilename() == null) {
      String	propPath = System.getProperty("user.dir");
      propPath += System.getProperty("file.separator");
      propPath += ".prubae.properties";
      File	propFile = new File(propPath);
      if (propFile.exists()) {
	setAdditionalPropertyFilename(propPath);
      }
    }
    if (getAdditionalPropertyFilename() == null) {
      String	propPath = System.getProperty("user.home");
      propPath += System.getProperty("file.separator");
      propPath += ".prubae.properties";
      File	propFile = new File(propPath);
      if (propFile.exists()) {
	setAdditionalPropertyFilename(propPath);
      }
    }

    String      ruleDir = null;
    ruleDir = System.getProperty("GNP_RULE_DIR");
    System.err.println("Prubae.initialize(): GNP_RULE_DIR, ruleDir = "
		       + ruleDir);
    Properties	properties = new Properties();
    if (ruleDir == null) {
      String	propertyFileName = System.getProperty("PRORATE_PROPERTY_FILE");
      if (propertyFileName == null) {
	propertyFileName = System.getProperty("user.home");
	propertyFileName += System.getProperty("file.separator");
	propertyFileName += ".prorate.properties";
	/* ;;; deBug for Windoze */
	String	separator = System.getProperty("file.separator");
	if (separator != null && separator.equals("\\")) {
	  String	homeDir = System.getProperty("user.home");
	  System.err.println("user.home = " + homeDir);
	  System.err.println("file.separator = " + separator);
	  propertyFileName = homeDir + separator + "prorate.properties";
	  System.err.println("WIN: propertyFileName = " + propertyFileName);
	}
	/* ;;; deBug for Windoze end */
      }
      InputStream	propertyFile = null;
      try {
	propertyFile = new FileInputStream(propertyFileName);
      }
      catch (FileNotFoundException e) {
	propertyFile = null;
      }
      System.err.println("propertyFileName = " + propertyFileName
			 + ", propertyFile = " + propertyFile);
      InputStream	is = null;
      if (propertyFile == null) {
	ClassLoader	loader = Prorate.class.getClassLoader();
	is = loader.getResourceAsStream(Prorate.propertyFile);
      }
      else {
	is = propertyFile;
      }
      try {
	properties.load(is);
      } catch (IOException eProperty) {
	/* ;;; deBug */
	System.err.println("IOException in Prubae.initialize() "
			   + "in reading " + Prorate.propertyFile
			   + ": " + eProperty.getMessage());
      }
      ruleDir =
	properties.getProperty("GNP_RULE_DIR", "/usr/local/gnp/rulebase");
    }
    System.err.println("Prubae.initialize(): ruleDir = " + ruleDir);
    if (getAdditionalPropertyFilename() != null) {
      String	propFile = getAdditionalPropertyFilename();
      InputStream	propIn = null;
      if (!propFile.startsWith("http:")) {
	try {
	  propIn = new FileInputStream(propFile);
	}
	catch (FileNotFoundException e) {
	  propIn = null;
	}
	if (propIn != null) {
	  try {
	    properties.load(propIn);
	  } catch (IOException eAdditionalProperty) {
	    System.err.println("IOException in Prubae.initialize() in reading "
			       + getAdditionalPropertyFilename()
			       + " : " + eAdditionalProperty.getMessage());
	  }
	}
      }
    }
    setRulebaseDirectory(ruleDir);

    if (getWriter() == null) {
      setWriter(new PrubaeWriterXml());
      getWriter().setEditor(this);
    }
    System.err.println("Prubae(): before getReader() and new PrubaeReaderXml()");
    if (getReader() == null) {
      System.err.println("Prubae(): getReader()==" + getReader() + ", before new PrubaeReaderXml()");
      setReader(new PrubaeReaderXml());
      getReader().setEditor(this);
      getReader().initialize();
      System.err.println("Prubae(): getReader()==" + getReader() + ", after new PrubaeReaderXml()");
      getReader().setEditor(this);
    }

    String	confFileName = System.getProperty("PRUBAERC");
    if (confFileName == null) {
      confFileName = System.getProperty("user.home");
      confFileName += System.getProperty("file.separator");
      confFileName += ".prubaerc";
    }
    FileReader	confFile = null;
    try {
      confFile = new FileReader(confFileName);
    }
    catch (FileNotFoundException e) {
      confFile = null;
    }
    System.err.println("confFileName = " + confFileName
		       + ", confFile = " + confFile);
    LList	tree = null;
    try {
      if (confFile != null) {
	tree = Lisp.read(confFile);
      }
      else {
	ClassLoader	loader = Prubae.class.getClassLoader();
	InputStream	is = loader.getResourceAsStream(Prubae.prubaercFile);
	tree = Lisp.read(is);
      }
    }
    catch (Exception e) {
      System.err.println("configuration read error : " + e.getMessage());
      return;
    }
    initConfTree(tree);
  }

  /**
   * <code>initConfTree</code> method
   *	initialize configuration lists from prubaerc tree list
   *
   * @param tree a <code>LList</code> value
   */
  public void initConfTree(LList tree) {
    configList = (LList)Lisp.car(Lisp.cdr(Lisp.assoc("prubaeConfig", tree)));
    doList = (LList)Lisp.car(Lisp.cdr(Lisp.assoc("prubaeDoFunctions", tree)));
    judgeList =
      (LList)Lisp.car(Lisp.cdr(Lisp.assoc("prubaeJudgeFunctions", tree)));
    valueList =
      (LList)Lisp.car(Lisp.cdr(Lisp.assoc("prubaeValueFunctions", tree)));
    varList = (LList)Lisp.car(Lisp.cdr(Lisp.assoc("prubaeVariables", tree)));
    constList = (LList)Lisp.cdr(Lisp.assoc("prubaeConstantValues", tree));
    settablevarList =
      (LList)Lisp.car(Lisp.cdr(Lisp.assoc("prubaeSettableVariables", tree)));

    List	extfDoList = getReader().getAllActionExtfs();
    System.err.println("initConfTree: extfDoList.size() = "
		       + extfDoList.size());
    for (int i = 0; i < extfDoList.size(); i++) {
      List	objs = (List)extfDoList.get(i);
      Object	obj = objs.get(0);
      int	numArgs = objs.size() - 1;
      Object	argsList = Lisp.nil;
      for (int j = 0; j < numArgs; j++) {
	argsList = Lisp.cons("any", argsList);
      }
      IntNum	regist = new IntNum(ProrateRulebaseElement.EXTERNAL_FUNCTION);
      IntNum	numArgsInt = new IntNum(numArgs);
      LList
	insList =
	Lisp.cons(regist,
		  Lisp.cons(obj,
			    Lisp.cons(obj,
				      Lisp.cons(Lisp.nil,
						Lisp.cons(numArgsInt,
							  argsList)))));
      doList = Lisp.append(doList, Lisp.cons(insList, Lisp.nil));
    }
    List	partDoList = getReader().getAllParts();
    System.err.println("initConfTree: partDoList.size() = "
		       + partDoList.size());
    for (int i = 0; i < partDoList.size(); i++) {
      Object	obj = partDoList.get(i);
      IntNum	regist = new IntNum(ProrateRulebaseElement.PARTCALL_CODE);
      LList
	insList =
	Lisp.cons(regist,
		  Lisp.cons(obj,
			    Lisp.cons(obj,
				      Lisp.cons(Lisp.nil,
						Lisp.cons(0, Lisp.nil)))));
      doList = Lisp.append(doList, Lisp.cons(insList, Lisp.nil));
    }
    List	extfJudgeList = getReader().getAllJudgeExtfs();
    System.err.println("initConfTree: extfJudgeList.size() = "
		       + extfJudgeList.size());
    for (int i = 0; i < extfJudgeList.size(); i++) {
      List	objs = (List)extfJudgeList.get(i);
      Object	obj = objs.get(0);
      int	numArgs = objs.size() - 1;
      Object	argsList = Lisp.nil;
      for (int j = 0; j < numArgs; j++) {
	argsList = Lisp.cons("any", argsList);
      }
      IntNum	regist = new IntNum(ProrateRulebaseElement.EXTERNAL_FUNCTION);
      IntNum	numArgsInt = new IntNum(numArgs);
      LList
	insList = Lisp.cons(regist,
			    Lisp.cons(obj,
				      Lisp.cons(obj,
						Lisp.cons(numArgsInt,
							  argsList))));
      judgeList = Lisp.append(judgeList, Lisp.cons(insList, Lisp.nil));
    }
    List	extfValueList = getReader().getAllValueExtfs();
    System.err.println("initConfTree: extfValueList.size() = "
		       + extfValueList.size());
    for (int i = 0; i < extfValueList.size(); i++) {
      List	objs = (List)extfValueList.get(i);
      Object	obj = objs.get(0);
      int	numArgs = objs.size() - 1;
      Object	argsList = Lisp.nil;
      for (int j = 0; j < numArgs; j++) {
	argsList = Lisp.cons("any", argsList);
      }
      IntNum	regist = new IntNum(ProrateRulebaseElement.EXTERNAL_FUNCTION);
      IntNum	numArgsInt = new IntNum(numArgs);
      LList
	insList =
	Lisp.cons(regist,
		  Lisp.cons(obj,
			    Lisp.cons(obj,
				      Lisp.cons("any",
						Lisp.cons(numArgsInt,
							  argsList)))));
      valueList = Lisp.append(valueList, Lisp.cons(insList, Lisp.nil));
    }
  }

  /**
   * <code>initPanel</code> method
   * creates editor panel and menu and menu items.
   */
  public void initPanel() {
    setRootPanel(new JPanel());
    getRootPanel().setLayout(new BorderLayout());

    JMenuBar	menuBar = new JMenuBar();
    setFileMenu(new JMenu("File"));
    setEditMenu(new JMenu("Edit"));
    setHelpMenu(new JMenu("Help"));
    JMenuItem	spaItem = new JMenuItem("Spa");
    spaItem.setActionCommand("spa");
    spaItem.addActionListener(this);
    getFileMenu().add(spaItem);
    JMenuItem	apdpItem = new JMenuItem("Apdp");
    apdpItem.setActionCommand("apdp");
    apdpItem.addActionListener(this);
    getFileMenu().add(apdpItem);
    JMenuItem	extfItem = new JMenuItem("Extf");
    extfItem.setActionCommand("extf");
    extfItem.addActionListener(this);
    getFileMenu().add(extfItem);
    JMenuItem	partItem = new JMenuItem("Part");
    partItem.setActionCommand("part");
    partItem.addActionListener(this);
    getFileMenu().add(partItem);
    JMenuItem	openItem = new JMenuItem("Open");
    openItem.setActionCommand("open");
    openItem.addActionListener(this);
    getFileMenu().add(openItem);
    JMenuItem	saveItem = new JMenuItem("Save");
    saveItem.setActionCommand("save");
    saveItem.addActionListener(this);
    getFileMenu().add(saveItem);
    JMenuItem	closeItem = new JMenuItem("Close");
    closeItem.setActionCommand("close");
    closeItem.addActionListener(this);
    getFileMenu().add(closeItem);
    menuBar.add(getFileMenu());
    JMenuItem	undoItem = new JMenuItem("Undo");
    undoItem.setActionCommand("undo");
    undoItem.addActionListener(this);
    undoItem.setEnabled(false);	/* ;;; not available yet  */
    getEditMenu().add(undoItem);
    JMenuItem	cutItem = new JMenuItem("Cut");
    cutItem.setActionCommand("cut");
    cutItem.addActionListener(this);
    getEditMenu().add(cutItem);
    JMenuItem	copyItem = new JMenuItem("Copy");
    copyItem.setActionCommand("copy");
    copyItem.addActionListener(this);
    /* copyItem.setEnabled(false);	/* ;;; not available yet by Bugs */
    getEditMenu().add(copyItem);
    JMenuItem	pasteItem = new JMenuItem("Paste");
    pasteItem.setActionCommand("paste");
    pasteItem.addActionListener(this);
    getEditMenu().add(pasteItem);
    menuBar.add(getEditMenu());
    JMenuItem	aboutItem = new JMenuItem("About");
    aboutItem.setActionCommand("about");
    aboutItem.addActionListener(this);
    getHelpMenu().add(aboutItem);
    menuBar.add(getHelpMenu());
    if (getEditType() != null && getEditType().equals(PrubaeEditUrl)) {
      System.err.println("editType = PrubaeEditUrl, disable file menus");
      spaItem.setEnabled(false);
      apdpItem.setEnabled(false);
      extfItem.setEnabled(false);
      partItem.setEnabled(false);
      openItem.setEnabled(false);
      closeItem.setEnabled(false);
    }

    setMainPanel(new JPanel());
    getMainPanel().setLayout(new BoxLayout(getMainPanel(), BoxLayout.X_AXIS));
    setTreePanel(new JPanel());
    getTreePanel().setPreferredSize(new Dimension(300, 500));
    getTreePanel().setLayout(new BorderLayout());
    setPartPanel(new JPanel());
    /* getPartPanel().setPreferredSize(new Dimension(300, 500)); */
    getMainPanel().add(getTreePanel());
    /*
      getMainPanel().add(getPartPanel());
    */
    JPanel	partBorderPane = new JPanel();
    partBorderPane.setLayout(new BorderLayout());
    JScrollPane	partScrollPane = new JScrollPane();
    partScrollPane.setPreferredSize(new Dimension(300, 500));
    partScrollPane.getViewport().add(getPartPanel());
    // getMainPanel().add(partScrollPane);
    partBorderPane.add("Center", partScrollPane);
    getMainPanel().add(partBorderPane);
    /**/

    getRootPanel().add("North", menuBar);
    getRootPanel().add("Center", getMainPanel());
  }

  /**
   * <code>setFunction</code> method
   * set the rule model element to edit.
   * when another model is being edited, just close it first. *FIXME*
   * set the view and controller to the top element model,
   * create and display the tree panel and part panel.
   * @param model a <code>PrubaeModel</code> value is the model to edit.
   * @return a <code>PrubaeModel</code> value is a model when succeeded,
   * null otherwise.
   */
  public PrubaeModel setFunction(PrubaeModel model) {
    if (model == null) {
      return null;
    }
    if (getModel() != null) {
      /* ;;; 2006.09.13 */
      if (getModel().equals(model)) {
	return getModel();
      }
      /* ;;; 2006.09.12 */
      int confirm = JOptionPane.showConfirmDialog(null, "Save editing rule?");
      if (confirm == JOptionPane.CANCEL_OPTION) {
	return getModel();
      }
      if (confirm == JOptionPane.OK_OPTION) {
	if (prepareSaveFunction()) {
	  saveFunction();
	}
      }
      closeFunction();
      setModel(null);
    }
    setModel(model);
    getModel().initialize(this, null, null);
    setController(getModel().getController());
    setView(getModel().getView());
    DefaultMutableTreeNode	root = getController().getTreeNode();
    if (root == null) {
      closeFunction();
      return null;
    }
    setTree(new JTree(root));
    if (getTree() == null) {
      closeFunction();
      return null;
    }
    setTreeModel((DefaultTreeModel)getTree().getModel());
    getTree().getSelectionModel().setSelectionMode(TreeSelectionModel
						   .SINGLE_TREE_SELECTION);
    getTree().setRootVisible(true);
    getTree().addTreeSelectionListener(new TreeSelectionListener() {
	public void valueChanged(TreeSelectionEvent evt) {
	  System.err.println("selectionListener.valueChanged");
	  TreePath	path = evt.getPath();
	  for (int i = path.getPathCount()-1; i >= 0; i--) {
	    DefaultMutableTreeNode
	      node = (DefaultMutableTreeNode)path.getPathComponent(i);
	    Object	obj = node.getUserObject();
	    if (obj instanceof PrubaeController) {
	      System.err.println("selection class " + i + ":"
				 + obj.getClass());
	      if (getController() != null) {
		getController().close();
	      }
	      else {
		getPartPanel().removeAll();
	      }
	      setController((PrubaeController)obj);
	      setView(getController().getView());
	      getController().open();
	      return;
	    }
	    System.err.println("selection ignore class " + i + ":"
			       + obj.getClass());
	  }
	}
      });

    MouseListener ml = null;
    ml = new MouseAdapter() {
	public void mousePressed(MouseEvent evt) {
	  if (evt.isPopupTrigger()) {
	    int selRow = getTree().getRowForLocation(evt.getX(), evt.getY());
	    if(selRow != -1) {
	      getTree().setSelectionRow(selRow);
	      TreePath	treePath = getTree().getPathForRow(selRow);
	      if (treePath != null) {
		Object[]	objs = treePath.getPath();
		/* ;;; deBug
		System.err.print("Prubae.mousePressed() : treePath.getPath() ");
		for (int i = 0; i < objs.length; i++) {
		  System.err.print("objs[" + i + "] = " + objs[i]);
		}
		System.err.println("");
		*/
		for (int i = 0; i < objs.length; i++) {
		  if (objs[i] instanceof DefaultMutableTreeNode) {
		    DefaultMutableTreeNode
		      node = (DefaultMutableTreeNode)objs[i];
		    Object	obj = node.getUserObject();
		    /* ;;; deBug
		    System.err.println("Prubae.mousePressed() : objs[" + i + "] = " + objs[i] + ", node = " + node + ", userObject = " + obj);
		    */
		    if (obj instanceof PrubaeController) {
		      PrubaeController	cntl = (PrubaeController)obj;
		      setController(cntl);
		      setView(cntl.getView());
		      JPopupMenu	popup = cntl.getPopupMenu();
		      /* ;;; deBug
		      System.err.println("Prubae.mousePressed() : cntl = " + cntl + ", model = " + cntl.getModel() + ", view = " + cntl.getView() + ", popup = " + popup);
		      */
		      if (popup != null) {
			popup.show(evt.getComponent(), evt.getX(), evt.getY());
			/* ;;; deBug
			System.err.println("Prubae.mousePressed() : popup.show() cntl = " + cntl + ", model = " + cntl.getModel() + ", view = " + cntl.getView() + ", popup = " + popup);
			*/
		      }
		    }
		  }
		}
	      }
	    }
	  }
	}
	public void mouseReleased(MouseEvent evt) {
	  if (evt.isPopupTrigger()) {
	    int selRow = getTree().getRowForLocation(evt.getX(), evt.getY());
	    if(selRow != -1) {
	      getTree().setSelectionRow(selRow);
	      TreePath	treePath = getTree().getPathForRow(selRow);
	      if (treePath != null) {
		Object[]	objs = treePath.getPath();
		/* ;;; deBug
		System.err.print("Prubae.mouseReleased() : treePath.getPath() ");
		for (int i = 0; i < objs.length; i++) {
		  System.err.print("objs[" + i + "] = " + objs[i]);
		}
		System.err.println("");
		*/
		for (int i = 0; i < objs.length; i++) {
		  if (objs[i] instanceof DefaultMutableTreeNode) {
		    DefaultMutableTreeNode
		      node = (DefaultMutableTreeNode)objs[i];
		    Object	obj = node.getUserObject();
		    /* ;;; deBug
		    System.err.println("Prubae.mouseReleased() : objs[" + i + "] = " + objs[i] + ", node = " + node + ", userObject = " + obj);
		    */
		    if (obj instanceof PrubaeController) {
		      PrubaeController	cntl = (PrubaeController)obj;
		      setController(cntl);
		      setView(cntl.getView());
		      JPopupMenu	popup = cntl.getPopupMenu();
		      /* ;;; deBug
		      System.err.println("Prubae.mouseReleased() : cntl = " + cntl + ", model = " + cntl.getModel() + ", view = " + cntl.getView() + ", popup = " + popup);
		      */
		      if (popup != null) {
			popup.show(evt.getComponent(), evt.getX(), evt.getY());
			/* ;;; deBug
			System.err.println("Prubae.mouseReleased() : popup.show() cntl = " + cntl + ", model = " + cntl.getModel() + ", view = " + cntl.getView() + ", popup = " + popup);
			*/
		      }
		    }
		  }
		}
	      }
	    }
	  }
	}
	public void mouseClicked(MouseEvent evt) {
	  if (evt.isPopupTrigger()) {
	    int selRow = getTree().getRowForLocation(evt.getX(), evt.getY());
	    if(selRow != -1) {
	      getTree().setSelectionRow(selRow);
	      TreePath	treePath = getTree().getPathForRow(selRow);
	      if (treePath != null) {
		Object[]	objs = treePath.getPath();
		/* ;;; deBug
		System.err.print("Prubae.mouseClicked() : treePath.getPath() ");
		for (int i = 0; i < objs.length; i++) {
		  System.err.print("objs[" + i + "] = " + objs[i]);
		}
		System.err.println("");
		*/
		for (int i = 0; i < objs.length; i++) {
		  if (objs[i] instanceof DefaultMutableTreeNode) {
		    DefaultMutableTreeNode
		      node = (DefaultMutableTreeNode)objs[i];
		    Object	obj = node.getUserObject();
		    /* ;;; deBug
		    System.err.println("Prubae.mouseClicked() : objs[" + i + "] = " + objs[i] + ", node = " + node + ", userObject = " + obj);
		    */
		    if (obj instanceof PrubaeController) {
		      PrubaeController	cntl = (PrubaeController)obj;
		      setController(cntl);
		      setView(cntl.getView());
		      JPopupMenu	popup = cntl.getPopupMenu();
		      /* ;;; deBug
		      System.err.println("Prubae.mouseClicked() : cntl = " + cntl + ", model = " + cntl.getModel() + ", view = " + cntl.getView() + ", popup = " + popup);
		      */
		      if (popup != null) {
			popup.show(evt.getComponent(), evt.getX(), evt.getY());
			/* ;;; deBug
			System.err.println("Prubae.mouseClicked() : popup.show() cntl = " + cntl + ", model = " + cntl.getModel() + ", view = " + cntl.getView() + ", popup = " + popup);
			*/
		      }
		    }
		  }
		}
	      }
	    }
	  }
	}
      };
    getTree().addMouseListener(ml);

    if (getScrollPane() == null) {
      setScrollPane(new JScrollPane());
      getScrollPane().setPreferredSize(new Dimension(300, 500));
    }
    getScrollPane().getViewport().add(tree);
    getTreePanel().add("Center", getScrollPane());
    getTreePanel().setVisible(true);

    getController().open();
    setTreeSelection(root);

    /* save varList of Prubae */
    System.err.println("prubae.setFunction() : nthCdr 35 savedVarList = " + Lisp.nthCdr(35, getSavedVarList()));
    if (getSavedVarList() == null) {
      setSavedVarList(getVarList());
    }
    System.err.println("prubae.setFunction() : after nthCdr 35 savedVarList = " + Lisp.nthCdr(35, getSavedVarList()));

    return getModel();
  }

  /**
   * <code>setTreeSelection</code> method
   * selects the tree node on the tree panel.
   * @param node a <code>DefaultMutableTreeNode</code> value
   * is the tree node to select
   */
  public void setTreeSelection(DefaultMutableTreeNode node) {
    TreePath	treePath = new TreePath(node.getPath());
    getTree().expandPath(treePath);
    getTree().setExpandsSelectedPaths(true);	/* ;;; add 2009.07.18 */
    getTree().setSelectionPath(treePath);
  }

  /**
   * <code>newFunction</code> method 
   * when another model is being edited, just close it first. *FIXME*
   * create top rule element(PrubaeModelFunction),
   * and set it to the model element to edit.
   * @param type an <code>int</code> value is either SPA,APDP,EXTF or PART,
   * which is defined in class PrubaeModelFunction.
   * @return a <code>PrubaeModel</code> value is a model to edit.
   */
  public PrubaeModel newFunction(int type) {
    if (getModel() != null) {
      /* ;;; 2006.09.12 */
      int confirm = JOptionPane.showConfirmDialog(null, "Save editing rule?");
      if (confirm == JOptionPane.CANCEL_OPTION) {
	return getModel();
      }
      if (confirm == JOptionPane.OK_OPTION) {
	if (prepareSaveFunction()) {
	  saveFunction();
	}
      }
      closeFunction();
      setModel(null);
    }
    if (getEditType() != null && getEditType().equals(PrubaeEditFile)
	&& getFilename() != null && !getFilename().equals("")) {
      setUrlname(null);
      setFilename(null);
    }
    else if (getEditType() != null && getEditType().equals(PrubaeEditUrl)
	     && getUrlname() != null && !getUrlname().equals("")) {
      setEditType(PrubaeEditFile);
      setUrlname(null);
      setFilename(null);
    }
    PrubaeModel	model = new PrubaeModelFunction();
    ((PrubaeModelFunction)model).setFunctionType(type);
    setFunction(model);
    return getModel();
  }

  /**
   * <code>closeFunction</code> method
   * closes the all current editing rule element models,
   * clean up part panel.
   * remove JTree tree from tree panel and clear tree panel.
   */
  public void closeFunction() {
    if (getModel() == null) {
      return;
    }
    /* restore varList of Prubae from PrubaeControllerFunction.varList */
    System.err.println("Prubae.closeFunction() : prubae.varList nthCdr 35 = "
		       + Lisp.nthCdr(35, getVarList()));
    setVarList(getSavedVarList());
    setSavedVarList(null);
    System.err.println("Prubae.closeFunction() : closed prubae.varList nthCdr 35 = "
		       + Lisp.nthCdr(35, getVarList()));
    if (getEditType() != null && getEditType().equals(PrubaeEditFile)
	&& getFilename() != null && !getFilename().equals("")) {
      setUrlname(null);
      setFilename(null);
    }
    else if (getEditType() != null && getEditType().equals(PrubaeEditUrl)
	     && getUrlname() != null && !getUrlname().equals("")) {
      setEditType(PrubaeEditFile);
      setUrlname(null);
      setFilename(null);
    }
    getModel().close();
    setModel(null);
    setController(null);
    setView(null);
    if (getScrollPane() != null) {
      getScrollPane().getViewport().remove(tree);
      getTreePanel().remove(getScrollPane());
    }
    getTreePanel().setVisible(false);
    setTreeModel(null);
    setTree(null);
  }  

  /**
   * <code>readFunction</code> method
   * creates reader PrubaeReader when it is not created yet.
   * get the rule file name to read from.
   * read rule model elements from the rule file using the reader.
   * set the top rule model element read to edit.
   */
  public void readFunction() {
    PrubaeModel	modelSaved = getModel();
    PrubaeModel	model = null;
    getReader().read(getFilename());
    model = getModel();
    setModel(modelSaved);
    setFunction(model);
  }  

  /**
   * <code>saveFunction</code> method
   * creates writer PrubaeWriter when it is not created yet.
   * get the rule file name to write to.
   * write rule model elements to the rule file using the writer.
   */
  public void saveFunction() {
    if (getEditType() != null && getEditType().equals(PrubaeEditFile)
	&& getFilename() != null && !getFilename().equals("")) {
      int	index = getFilename().lastIndexOf('/');
      String	path = (index >= 0 ? getFilename().substring(0, index)
			: getFilename());
      setFilename(path + "/" + getModel().toFilename());
      try {
        getWriter().write(getFilename());
      } catch (ProrateRulebaseException e) {
	JOptionPane.showMessageDialog(null, e.getMessage());
      }
    }
    else if (getEditType() != null && getEditType().equals(PrubaeEditUrl)
	     && getUrlname() != null && !getUrlname().equals("")) {
      int	index = getUrlname().lastIndexOf('/');
      String	path = (index >= 0 ? getUrlname().substring(0, index)
			: getUrlname());
      setUrlname(path + "/" + getModel().toFilename());
      getWriter().upload(getUrlname());
    }
  }
  
  /* ;;; 2006.09.12 */
  /**
   * <code>prepareSaveFunction</code> method
   * check if any model is been editing.
   * updates the controller editing status to the model first.
   * 
   * @return a <code>boolean</code> value
   */
  public boolean prepareSaveFunction() {
    if (getModel() == null) {
      return false;
    }
    if (getController() != null) {
      getController().update();
    }
    List	errors = new Vector();
    if (!getModel().check(errors)) {
      StringBuffer	errorMessage = new StringBuffer();
      for (int i = 0; i < errors.size(); i++) {
	if (i > 0) {
	  errorMessage.append("\n");
	}
	errorMessage.append(errors.get(i));
      }
      JOptionPane.showMessageDialog(null, errorMessage.toString());
      return false;
    }
    return true;
  }

  /**
   * <code>inputFilename</code> method
   * creates the file chooser when it is not created yet.
   * display the chooser's open dialog, and when it returned by OK button,
   * set the rule filename from the chooser's selection.
   * @return a <code>boolean</code> value
   * is true when the chooser returned by OK button, false otherwise.
   */
  public boolean inputFilename() {
    if (getChooser() == null) {
      setChooser(new JFileChooser());
    }
    File	selectedDirectory = new File(getRulebaseDirectory());
    getChooser().setCurrentDirectory(selectedDirectory);
    int ret = getChooser().showOpenDialog(getRootPanel());
    if (ret == JFileChooser.APPROVE_OPTION) {
      setEditType(Prubae.PrubaeEditFile);
      setFilename(getChooser().getSelectedFile().getAbsolutePath());
      setUrlname(null);
      return true;
    }
    return false;
  }

  /**
   * <code>outputFilename</code> method
   * creates the file chooser when it is not created yet.
   * set the filename of the model as the filename of the chooser.
   * display the chooser's save dialog, and when it returned by OK button,
   * set the rule filename from the chooser's selection.
   * @return a <code>boolean</code> value
   */
  public boolean outputFilename() {
    if (getChooser() == null) {
      setChooser(new JFileChooser());
    }
    String
      selectedFilePath = (getRulebaseDirectory()
			  + System.getProperty("file.separator")
			  + ((PrubaeModelFunction)getModel()).toRuleSubdir()
			  + System.getProperty("file.separator")
			  + getModel().toFilename());
    File	selectedFile = new File(selectedFilePath);
    getChooser().setSelectedFile(selectedFile);
    int ret = getChooser().showSaveDialog(getRootPanel());
    if (ret == JFileChooser.APPROVE_OPTION) {
      setEditType(Prubae.PrubaeEditFile);
      setFilename(getChooser().getSelectedFile().getAbsolutePath());
      setUrlname(null);
      return true;
    }
    return false;
  }

  /**
   * <code>cutFunction</code> method	cut the node from the editor tree.
   */
  public void cutFunction() {
    PrubaeController	controller = getController();
    if (controller == null) {
      return;
    }
    PrubaeModel	model = controller.getModel();
    PrubaeModel	parent = model.getParent();
    if (parent == null) {
      return;
    }
    /* ;;; deBug
    System.err.println("Prubae.cutFunction() : model = " + model + ", view = " + model.getView() + ", cntl = " + controller);
    */
    setYank((PrubaeModel)model.clone());
    /* ;;; deBug
    System.err.println("Prubae.cutFunction() : yank model = " + getYank() + ", view = " + getYank().getView() + ", cntl = " + getYank().getController());
    */
    if (!(model instanceof PrubaeModelDo)) {
      return;	/* JudgeλcopyưԤʤ(JudgeϺǤʤΤ) */
    }
    model.delete();
    DefaultMutableTreeNode
      node = (DefaultMutableTreeNode)parent.getController().getTreeNode();
    setTreeSelection(node);
    parent.getController().open();
  }

  /**
   * <code>copyFunction</code> method	copy the node in the editor tree.
   */
  public void copyFunction() {
    PrubaeController	controller = getController();
    if (controller == null) {
      return;
    }
    PrubaeModel	model = (PrubaeModel)controller.getModel();
    if (model.getParent() == null) {
      return;
    }
    /* ;;; deBug
    System.err.println("Prubae.copyFunction() : model = " + model + ", view = " + model.getView() + ", cntl = " + controller);
    */
    setYank((PrubaeModel)model.clone());
    /* ;;; deBug
    System.err.println("Prubae.copyFunction() : yank model = " + getYank() + ", view = " + getYank().getView() + ", cntl = " + getYank().getController());
    */
  }

  /**
   * <code>pasteFunction</code> method
   *	paste the node to exchange selected node on the editor tree.
   */
  public void pasteFunction() {
    PrubaeController	controller = getController();
    if (controller == null) {
      return;
    }
    PrubaeModel	model = controller.getModel();
    PrubaeModel	parent = model.getParent();
    if (parent == null) {
      return;
    }
    if (!(model instanceof PrubaeModelDo)
	&& !(model instanceof PrubaeModelJudge)) {
      return;
    }
    if ((model instanceof PrubaeModelDo
	 && !(getYank() instanceof PrubaeModelDo))
	|| (model instanceof PrubaeModelJudge
	    && !(getYank() instanceof PrubaeModelJudge)) ) {
      return;
    }
    /* ;;; deBug
    System.err.println("Prubae.pasteFunction() : yank model = " + getYank() + ", view = " + getYank().getView() + ", cntl = " + getYank().getController());
    */
    PrubaeModel	paste = (PrubaeModel)getYank().clone();
    /* ;;; deBug
    System.err.println("Prubae.pasteFunction() : paste model = " + paste + ", view = " + paste.getView() + ", cntl = " + paste.getController());
    */
    paste.setParent(null);
    paste.setParentList(null);
    PrubaeController	pasteController = paste.getController();
    DefaultMutableTreeNode
      node = (DefaultMutableTreeNode)controller.getTreeNode().getParent();
    pasteController.swapTreeNode(node, controller.getTreeNode());
    paste.swap(model);
    setTreeSelection((DefaultMutableTreeNode)pasteController.getTreeNode());
    pasteController.close();
    pasteController.open();
  }

  /**
   * <code>aboutFunction</code> method	view About dialog
   *
   */
  protected void aboutFunction() {
    String	title = "About";
    String	message =
      "This is Prubae Proration Rulebase Editor,\nwhich is an Editing Facility of Proration Rulebase Rule Files\nfor the GNP Proration System.\n\nThe Java classes (with related files and documentation) in these packages\nare copyright (C) 2002 - 2009  Hitoshi Guutara Maruyama and GNP Ltd.\n\nThese classes are distributed in the hope that they will be useful,\nbut WITHOUT ANY WARRANTY; without even the implied warranty of\nMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\nGNU General Public License [included in the source package] for more details.\n\nThese classes are free software.  You can use and re-distribute a class\nwithout restriction (in source or binary form) as long as you use a\nversion has not been modified in any way from a version released\nby Hitoshi Guutara Maruyama, or GNP Ltd.\nYou may make and distribute a modified version, provided you follow\nthe terms of the GNU General Public License; either version 3,\nor (at your option) any later version.\n  (GPL Version 3 is included in the source package.)\n\nIf you have questions, contact Hitoshi Guutara Maruyama\nBy email gnp@sourceforge.jp.";
    JOptionPane.showMessageDialog(getRootPanel(), message, title,
				  JOptionPane.PLAIN_MESSAGE);
  }

  /**
   * <code>updateUrlname</code> method	updates urlname String.
   *
   * @param regist a <code>short</code> value
   * @param carrier a <code>String</code> value
   * @param tkCarrier a <code>String</code> value
   * @param name a <code>String</code> value
   * @param fromDate an <code>int</code> value
   * @param toDate an <code>int</code> value
   */
  void	updateUrlname(short regist,
		      String carrier, String tkCarrier, String name,
		      int fromDate, int toDate) {
    if (getEditType() != null && getEditType().equals(PrubaeEditUrl)
	&& getUrlname() != null && !getUrlname().equals("")) {
      StringBuffer	newString = new StringBuffer();
      newString
	.append(regist == (short)ProrateRulebaseElement.SPA_CODE ? "SPA"
		: (regist == (short)ProrateRulebaseElement.APD_P_CODE ? "APDP"
		   : (regist == (short)ProrateRulebaseElement.EXTF_CODE
		      ? "EXTF"
		      : (regist == (short)ProrateRulebaseElement.PART_CODE
			 ? "PART" : ""))));
      if ((name != null && !name.equals(""))
	  || (carrier != null && !carrier.equals(""))) {
	newString.append("(");
	newString.append(name);
	newString.append(carrier);
	if (tkCarrier != null && !tkCarrier.equals("")) {
	  newString.append(",");
	  newString.append(tkCarrier);
	}
	newString.append(")");
	newString.append(Integer.toString(fromDate));
	newString.append(Integer.toString(toDate));
	if (newString.length() > 0) {
	  int	index = getUrlname().lastIndexOf('/');
	  setUrlname(getUrlname().substring(0, index) + newString.toString());
	}
      }
    }
  }
  
  /* ;;; 2006.09.06 */
  /**
   * <code>refresh</code> method	refreshes mainPanel
   *
   */
  void refresh() {
    PrubaeController	cntl = getController();
    PrubaeView	view = cntl.getView();
    cntl.close();
    cntl.open();
    view.display();
  }

  /**
   * <code>actionPerformed</code> method
   * handles the action events returned from menu items of the editor.
   * @param e an <code>ActionEvent</code> value is the action event.
   */
  public void actionPerformed(ActionEvent e) {
    if (e.getActionCommand().equals("spa")) {
      newFunction(PrubaeModelFunction.SPA);
    }
    else if (e.getActionCommand().equals("apdp")) {
      newFunction(PrubaeModelFunction.APDP);
    }
    else if (e.getActionCommand().equals("extf")) {
      newFunction(PrubaeModelFunction.EXTF);
    }
    else if (e.getActionCommand().equals("part")) {
      newFunction(PrubaeModelFunction.PART);
    }
    else if (e.getActionCommand().equals("open")) {
      if (inputFilename()) {
	if (getFilename() != null && !getFilename().equals("")) {
	  readFunction();
	}
      }
    }
    else if (e.getActionCommand().equals("save")) {
      if (getEditType() != null && getEditType().equals(PrubaeEditUrl)) {
	if (getUrlname() != null && !getUrlname().equals("")) {
	  if (prepareSaveFunction()) {
	    saveFunction();
	  }
	}
      }
      else {
	if (prepareSaveFunction()) {
	  if (outputFilename()) {
	    if (getFilename() != null && !getFilename().equals("")) {
	      saveFunction();
	    }
	  }
	}
      }
    }
    else if (e.getActionCommand().equals("close")) {
      if (getModel() != null) {
	/* ;;; 2006.09.12 */
	int confirm =
	  JOptionPane.showConfirmDialog(null, "Save editing rule?");
	if (confirm == JOptionPane.CANCEL_OPTION) {
	  return;
	}
	if (confirm == JOptionPane.OK_OPTION) {
	  if (prepareSaveFunction()) {
	    saveFunction();
	  }
	}
	closeFunction();
	setModel(null);
      }
    }
    else if (e.getActionCommand().equals("cut")) {
      cutFunction();
    }
    else if (e.getActionCommand().equals("copy")) {
      copyFunction();
    }
    else if (e.getActionCommand().equals("paste")) {
      pasteFunction();
    }
    else if (e.getActionCommand().equals("about")) {
      aboutFunction();
    }
  }

  // Setters and Getters  
  /**
   * <code>setRootPanel</code> method
   * sets the value of instance variable rootPanel.
   * @param theRootPanel a <code>JPanel</code> value to set the variable
   * 
   * @uml.property name="rootPanel"
   */
  public void setRootPanel(JPanel theRootPanel) {
    rootPanel = theRootPanel;
  }

  /**
   * <code>getRootPanel</code> method
   * gets the value of instance variable rootPanel.
   * @return a <code>JPanel</code> value of the variable
   * 
   * @uml.property name="rootPanel"
   */
  public JPanel getRootPanel() {
    return rootPanel;
  }

  /**
   * <code>setMainPanel</code> method
   * sets the value of instance variable mainPanel.
   * @param theMainPanel a <code>JPanel</code> value to set the variable
   * 
   * @uml.property name="mainPanel"
   */
  public void setMainPanel(JPanel theMainPanel) {
    mainPanel = theMainPanel;
  }

  /**
   * <code>getMainPanel</code> method
   * gets the value of instance variable mainPanel.
   * @return a <code>JPanel</code> value of the variable
   * 
   * @uml.property name="mainPanel"
   */
  public JPanel getMainPanel() {
    return mainPanel;
  }

  /**
   * <code>setTreePanel</code> method
   * sets the value of instance variable treePanel.
   * @param theTreePanel a <code>JPanel</code> value to set the variable
   * 
   * @uml.property name="treePanel"
   */
  public void setTreePanel(JPanel theTreePanel) {
    treePanel = theTreePanel;
  }

  /**
   * <code>getTreePanel</code> method
   * gets the value of instance variable treePanel.
   * @return a <code>JPanel</code> value of the variable
   * 
   * @uml.property name="treePanel"
   */
  public JPanel getTreePanel() {
    return treePanel;
  }

  /**
   * <code>setPartPanel</code> method
   * sets the value of instance variable partPanel.
   * @param thePartPanel a <code>JPanel</code> value to set the variable
   * 
   * @uml.property name="partPanel"
   */
  public void setPartPanel(JPanel thePartPanel) {
    partPanel = thePartPanel;
  }

  /**
   * <code>getPartPanel</code> method
   * gets the value of instance variable partPanel.
   * @return a <code>JPanel</code> value of the variable
   * 
   * @uml.property name="partPanel"
   */
  public JPanel getPartPanel() {
    return partPanel;
  }

  /**
   * <code>setFileMenu</code> method
   * sets the value of instance variable fileMenu
   * @param theFileMenu a <code>JMenu</code> value to set the variable
   * 
   * @uml.property name="fileMenu"
   */
  public void setFileMenu(JMenu theFileMenu) {
    fileMenu = theFileMenu;
  }

  /**
   * <code>getFileMenu</code> method
   * gets the value of instance variable fileMenu
   * @return a <code>JMenu</code> value of the variable
   * 
   * @uml.property name="fileMenu"
   */
  public JMenu getFileMenu() {
    return fileMenu;
  }

  /**
   * <code>getEditMenu</code> method gets the value of instance variable editMenu
   * @return  a <code>JMenu</code> value
   * @uml.property  name="editMenu"
   */
  public JMenu getEditMenu() {
    return editMenu;
  }

  /**
   * <code>setEditMenu</code> method sets the value of instance variable editMenu
   * @param editMenu  a <code>JMenu</code> value
   * @uml.property  name="editMenu"
   */
  public void setEditMenu(JMenu editMenu) {
    this.editMenu = editMenu;
  }

  /**
   * <code>setHelpMenu</code> method
   * sets the value of instance variable helpMenu
   * @param theHelpMenu a <code>JMenu</code> value to set the variable
   * 
   * @uml.property name="helpMenu"
   */
  public void setHelpMenu(JMenu theHelpMenu) {
    helpMenu = theHelpMenu;
  }

  /**
   * <code>getHelpMenu</code> method
   * gets the value of instance variable helpMenu
   * @return a <code>JMenu</code> value of the variable
   * 
   * @uml.property name="helpMenu"
   */
  public JMenu getHelpMenu() {
    return helpMenu;
  }

  /**
   * <code>setTree</code> method
   * sets the value of instance variable tree
   * @param theTree a <code>JTree</code> value to set the variable
   * 
   * @uml.property name="tree"
   */
  public void setTree(JTree theTree) {
    tree = theTree;
  }

  /**
   * <code>getTree</code> method
   * gets the value of instance variable tree
   * @return a <code>JTree</code> value of the variable
   * 
   * @uml.property name="tree"
   */
  public JTree getTree() {
    return tree;
  }

  /**
   * <code>setScrollPane</code> method
   * sets the value of instance variable scrollPane
   * @param theScrollPane a <code>JScrollPane</code> value to set the variable.
   * 
   * @uml.property name="scrollPane"
   */
  public void setScrollPane(JScrollPane theScrollPane) {
    scrollPane = theScrollPane;
  }

  /**
   * <code>getScrollPane</code> method
   * gets the value of instance variable scrollPane
   * @return a <code>JScrollPane</code> value of the variable
   * 
   * @uml.property name="scrollPane"
   */
  public JScrollPane getScrollPane() {
    return scrollPane;
  }

  /**
   * <code>setFilename</code> method
   * sets the value of instance variable filename
   * @param theFilename a <code>String</code> value to set the variable
   * 
   * @uml.property name="filename"
   */
  public void setFilename(String theFilename) {
    filename = theFilename;
  }

  /**
   * <code>getFilename</code> method
   * gets the value of instance variable scrollPane
   * @return a <code>String</code> value of the variable
   * 
   * @uml.property name="filename"
   */
  public String getFilename() {
    return filename;
  }

  /**
   * <code>setTreeModel</code> method
   * sets the value of instance variable treeModel
   * @param theTreeModel a <code>DefaultTreeModel</code> value
   * to set the varriable
   * 
   * @uml.property name="treeModel"
   */
  public void setTreeModel(DefaultTreeModel theTreeModel) {
    treeModel = theTreeModel;
  }

  /**
   * <code>getTreeModel</code> method
   * gets the value of instance variable scrollPane.
   * @return a <code>DefaultTreeModel</code> value of the variable
   * 
   * @uml.property name="treeModel"
   */
  public DefaultTreeModel getTreeModel() {
    return treeModel;
  }

  /**
   * <code>setModel</code> method
   * sets the value of instance variable model.
   * @param theModel a <code>PrubaeModel</code> value to set the variable
   * 
   * @uml.property name="model"
   */
  public void setModel(PrubaeModel theModel) {
    model = theModel;
  }

  /**
   * <code>getModel</code> method
   * gets the value of instance variable model.
   * @return a <code>PrubaeModel</code> value of the variable
   * 
   * @uml.property name="model"
   */
  public PrubaeModel getModel() {
    return model;
  }

  /**
   * <code>setView</code> method
   * sets the value of instance variable view.
   * @param theView a <code>PrubaeView</code> value to set the variable
   * 
   * @uml.property name="view"
   */
  public void setView(PrubaeView theView) {
    view = theView;
  }

  /**
   * <code>getView</code> method
   * gets the value of instance variable view.
   * @return a <code>PrubaeView</code> value of the variable.
   * 
   * @uml.property name="view"
   */
  public PrubaeView getView() {
    return view;
  }

  /**
   * <code>setController</code> method
   * sets the value of instance variable controller.
   * @param theController a <code>PrubaeController</code> value
   * to set the variable.
   * 
   * @uml.property name="controller"
   */
  public void setController(PrubaeController theController) {
    controller = theController;
  }

  /**
   * <code>getController</code> method
   * gets the value of instance variable controller.
   * @return a <code>PrubaeController</code> value of the variable.
   * 
   * @uml.property name="controller"
   */
  public PrubaeController getController() {
    return controller;
  }

  /**
   * <code>setReader</code> method
   * sets the value of instance variable reader.
   * @param theReader a <code>PrubaeReader</code> value to set the variable
   * 
   * @uml.property name="reader"
   */
  public void setReader(PrubaeReader theReader) {
    reader = theReader;
  }

  /**
   * <code>getReader</code> method
   * gets the value of instance variable reader
   * @return a <code>PrubaeReader</code> value of the variable
   * 
   * @uml.property name="reader"
   */
  public PrubaeReader getReader() {
    return reader;
  }

  /**
   * <code>setWriter</code> method
   * sets the value of instance variable writer.
   * @param theWriter a <code>PrubaeWriter</code> value to set the variable.
   * 
   * @uml.property name="writer"
   */
  public void setWriter(PrubaeWriter theWriter) {
    writer = theWriter;
  }

  /**
   * <code>getWriter</code> method
   * gets the value of instance variable writer.
   * @return a <code>PrubaeWriter</code> value of the variable.
   * 
   * @uml.property name="writer"
   */
  public PrubaeWriter getWriter() {
    return writer;
  }

  /**
   * <code>setChooser</code> method
   * sets the value of instance variable chooser.
   * @param theChooser a <code>JFileChooser</code> value to set the variable.
   * 
   * @uml.property name="chooser"
   */
  public void setChooser(JFileChooser theChooser) {
    chooser = theChooser;
  }

  /**
   * <code>getChooser</code> method
   * gets the value of instance variable chooser.
   * @return a <code>JFileChooser</code> value of the variable.
   * 
   * @uml.property name="chooser"
   */
  public JFileChooser getChooser() {
    return chooser;
  }

  /**
   * <code>setConfigList</code> method
   * sets the value of instance variable configList.
   * @param theConfigList a <code>LList</code> value to set the variable.
   * 
   * @uml.property name="configList"
   */
  public void setConfigList(LList theConfigList) {
    configList = theConfigList;
  }

  /**
   * <code>getConfigList</code> method
   * gets the value of instance variable configList.
   * @return a <code>LList</code> value of the variable
   * 
   * @uml.property name="configList"
   */
  public LList getConfigList() {
    return configList;
  }

  /**
   * <code>setDoList</code> method
   * sets the value of instance variable doList.
   * @param theDoList a <code>LList</code> value to set the variable.
   * 
   * @uml.property name="doList"
   */
  public void setDoList(LList theDoList) {
    doList = theDoList;
  }

  /**
   * <code>getDoList</code> method
   * gets the value of instance variable doList.
   * @return a <code>LList</code> value of the variable
   * 
   * @uml.property name="doList"
   */
  public LList getDoList() {
    return doList;
  }

  /**
   * <code>setJudgeList</code> method
   * sets the value of instance variable judgeList.
   * @param theJudgeList a <code>LList</code> value to set the variable.
   * 
   * @uml.property name="judgeList"
   */
  public void setJudgeList(LList theJudgeList) {
    judgeList = theJudgeList;
  }

  /**
   * <code>getJudgeList</code> method
   * gets the value of instance variable judgeList.
   * @return a <code>LList</code> value of the variable
   * 
   * @uml.property name="judgeList"
   */
  public LList getJudgeList() {
    return judgeList;
  }

  /**
   * <code>setValueList</code> method
   * sets the value of instance variable valueList.
   * @param theValueList a <code>LList</code> value to set the variable.
   * 
   * @uml.property name="valueList"
   */
  public void setValueList(LList theValueList) {
    valueList = theValueList;
  }

  /**
   * <code>getValueList</code> method
   * gets the value of instance variable valueList.
   * @return a <code>LList</code> value of the variable
   * 
   * @uml.property name="valueList"
   */
  public LList getValueList() {
    return valueList;
  }

  /**
   * <code>setVarList</code> method
   * sets the value of instance variable varList.
   * @param theVarList a <code>LList</code> value to set the variable.
   * 
   * @uml.property name="varList"
   */
  public void setVarList(LList theVarList) {
    varList = theVarList;
  }

  /**
   * <code>getVarList</code> method
   * gets the value of instance variable varList.
   * @return a <code>LList</code> value of the variable
   * 
   * @uml.property name="varList"
   */
  public LList getVarList() {
    return varList;
  }

  /**
   * <code>setConstList</code> method
   * sets the value of instance variable constList.
   * @param theConstList a <code>LList</code> value to set the variable.
   * 
   * @uml.property name="constList"
   */
  public void setConstList(LList theConstList) {
    constList = theConstList;
  }

  /**
   * <code>getConstList</code> method
   * gets the value of instance variable constList.
   * @return a <code>LList</code> value of the variable
   * 
   * @uml.property name="constList"
   */
  public LList getConstList() {
    return constList;
  }

  /**
   * <code>setSettablevarList</code> method
   * sets the value of instance variable settablevarList.
   * @param theSettablevarList a <code>LList</code> value to set the variable.
   * 
   * @uml.property name="settablevarList"
   */
  public void setSettablevarList(LList theSettablevarList) {
    settablevarList = theSettablevarList;
  }

  /**
   * <code>getSettablevarList</code> method
   * gets the value of instance variable settablevarList.
   * @return a <code>LList</code> value of the variable
   * 
   * @uml.property name="settablevarList"
   */
  public LList getSettablevarList() {
    return settablevarList;
  }

  /**
   * @return  urlname
   * @uml.property  name="urlname"
   */
  public String getUrlname() {
    return urlname;
  }

  /**
   * @param urlname  	set urlname
   * @uml.property  name="urlname"
   */
  public void setUrlname(String urlname) {
    this.urlname = urlname;
  }

  /**
   * @return  dtdDir
   * @uml.property  name="dtdDir"
   */
  /* ;;; deBug
  public String getDtdDir() {
    return dtdDir;
  }
  */

  /**
   * @param dtdDir  	set dtdDir
   * @uml.property  name="dtdDir"
   */
  /* ;;; deBug
  public void setDtdDir(String dtdDir) {
    this.dtdDir = dtdDir;
  }
  */

  /**
   * @return  a <code>String</code> value
   * @uml.property  name="editType"
   */
  public String getEditType() {
    return editType;
  }

  /**
   * @param editType  a <code>String</code> value
   * @uml.property  name="editType"
   */
  public void setEditType(String editType) {
    this.editType = editType;
  }

  /**
   * @return  a <code>PrubaeModel</code> value
   * @uml.property  name="yank"
   */
  public PrubaeModel getYank() {
    return yank;
  }

  /**
   * @param yank  a <code>PrubaeModel</code> value
   * @uml.property  name="yank"
   */
  public void setYank(PrubaeModel yank) {
    this.yank = yank;
  }

  public String getAdditionalPropertyFilename() {
    return additionalPropertyFilename;
  }

  public void setAdditionalPropertyFilename(String additionalPropertyFilename) {
    this.additionalPropertyFilename = additionalPropertyFilename;
  }

  public String getRulebaseDirectory() {
    return rulebaseDirectory;
  }

  public void setRulebaseDirectory(String rulebaseDirectory) {
    this.rulebaseDirectory = rulebaseDirectory;
  }

  public LList getSavedVarList() {
    return savedVarList;
  }

  public void setSavedVarList(LList savedVarList) {
    this.savedVarList = savedVarList;
  }

} /* end class Prubae */
