// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;

/**
 * class <code>PrubaeController</code>
 * is the abstract class of prubae rule element model controller utility.
 * this class has relationship with its model class and view class,
 * designed according to the Model View Controller design pattern.
 * this class handles the common behavior of the rule element controlling.
 * this class also has model view controller associations
 * and has an association to the editor top class instance.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */

public abstract class PrubaeController
  implements ActionListener, Cloneable {

  // Attributes  
  /**
   * variable <code>treeNode</code> is the TreeNode instance of the JTree on editor tree panel, which is related to the model related to this controller.
   * @uml.property  name="treeNode"
   */
  protected DefaultMutableTreeNode treeNode = null;  
  /**
   * variable <code>panel</code> is the panel on which to display the details of the model related to this controller. the controlling buttons are also placed on the panel. the panel will be placed on the part panel of the editor, when the controller is active(the model is opened to display details).
   * @uml.property  name="panel"
   */
  protected JPanel panel;

  // Associations  
  /**
   * variable <code>model</code> is the reference to the model instance.
   * 
   * @uml.property name="model"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeModel model = null;

  /**
   * variable <code>view</code> is the reference to the view instance.
   * 
   * @uml.property name="view"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeView view = null;

  /**
   * variable <code>editor</code>
   * is the reference to the editor top class instance.
   * 
   * @uml.property name="editor"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected Prubae editor = null;

  /**
   * variable <code>commentField</code> is the JTextField to input/display the comment string
   * @uml.property  name="commentField"
   */
  private JTextField commentField = null;

  /**
   * variable <code>popupMenu</code> is the PopupMenu of the controller item.
   *
   */
  private JPopupMenu popupMenu = null;

  // Constructor  
  /**
   * Creates a new <code>PrubaeController</code> instance.
   * this creates all GUI elements on the panel
   * to controll/display the details of this model.
   */
  public PrubaeController() {  
    super();
  }

  // Operations  
  /**
   * <code>initialize</code> method creates treeNode related to this model.
   *
   */
  public void initialize() {
    if (getTreeNode() == null) {
      setTreeNode(new DefaultMutableTreeNode(this));
      getTreeNode().setUserObject(this);
      createPopupMenu();
    }
  }

  /**
   * <code>createPopupMenu</code> method	create popup menu and show.
   *
   */
  abstract void createPopupMenu();

  /**
   * <code>open</code> method
   * opens and displays the panel related to this model.
   * set this controller to the current editing target of the editor instance.
   * if there was already the current editing target controller, close it.
   * send view display to refresh the view of the related model.
   */
  public void open() {
    PrubaeController	old = getEditor().getController();
    if (old != null && old != this) {
      old.close();
    }
    getEditor().setController(this);
    getEditor().setView(getView());
    if (getPanel() == null) {
      createPanel();
    }
    getEditor().getPartPanel().add(getPanel());
    getView().display();
  }

  /**
   * <code>close</code> method closes the current editing controller.
   * first updates the model with the input values on the panel.
   * then closes the GUI controller (closeUI()) if required by the subclass.
   * remove this from the current editing target controller of the editor.
   * remove the panel related to this model from the editors part panel.
   */
  public void close() {
    if (getPanel() != null) {
      update();
      getModel().closeUI();
      getEditor().setController(null);
      getEditor().setView(null);
      getEditor().getPartPanel().remove(getPanel());
    }
    destroyPanel();
  }

  /**
   * <code>update</code> method
   * updates the model with the input values on the panel.
   * this updates the comment only.(common for all model subclasses).
   * send view display to refresh the view of the related model.
   */
  public void update() {
    getModel().setComment(getCommentField().getText());
    getView().display();
  }

  /**
   * <code>createPanel</code> method
   *
   */
  public void createPanel() {
    setPanel(new JPanel());
    getPanel().setLayout(new BoxLayout(getPanel(), BoxLayout.Y_AXIS));
    setCommentField(new JTextField());
    getCommentField().setEditable(true);
    JPanel	commentPanel = new JPanel();
    commentPanel.setLayout(new BoxLayout(commentPanel, BoxLayout.X_AXIS));
    JLabel	commentLabel = new JLabel("C : ");
    commentPanel.add(commentLabel);
    JScrollPane	commentScrollPane = new JScrollPane();
    commentScrollPane.getViewport().add(getCommentField());
    commentScrollPane
      .setHorizontalScrollBarPolicy(JScrollPane
				    .HORIZONTAL_SCROLLBAR_ALWAYS);
    commentPanel.add(commentScrollPane);
    getPanel().add(commentPanel);
  }

  /**
   * <code>destroyPanel</code> method
   *
   */
  public void destroyPanel() {
    if (getPanel() != null) {
      getPanel().removeAll();
    }
    setPanel(null);
    setCommentField(null);
  }

  /**
   * <code>insertTreeNode</code> method
   * insert tree node related to this model under the specified parent,
   * at the index of the specified child of the parent.
   * then sends reload to the tree model to notifify
   * that the tree structure may have changed.
   * then set the tree selection to the child before which the insertion done.
   * @param parent a <code>DefaultMutableTreeNode</code> value
   * is the specified tree parent.
   * @param child a <code>TreeNode</code> value is the specified tree child.
   */
  public void insertTreeNode(DefaultMutableTreeNode parent, TreeNode child) {
    int	index = parent.getIndex(child);
    System.err.println("PrubaeController.insertTreeNode() : index = " + index + ", insertNode = " + getTreeNode());
    parent.insert(getTreeNode(), index);
    if (getEditor().getTreeModel() != null) {
      getEditor().getTreeModel().reload(parent);
    }
    getEditor().setTreeSelection((DefaultMutableTreeNode)child);
  }

  /**
   * <code>addTreeNode</code> method
   * add tree node related to this model under the specified parent.
   * then sends reload to the tree model to notifify
   * that the tree structure may have changed.
   * @param parent a <code>DefaultMutableTreeNode</code> value
   * is the specified tree parent.
   */
  public void addTreeNode(DefaultMutableTreeNode parent) {  
    parent.add(getTreeNode());
    if (getEditor().getTreeModel() != null) {
      getEditor().getTreeModel().reload(parent);
    }
  }

  /**
   * <code>swapTreeNode</code> method
   * swap tree node related to this model with the specified child tree node,
   * under the specified parent.
   * then sends reload to the tree model to notifify
   * that the tree structure may have changed.
   * then set the tree selection to this tree node.
   * @param parent a <code>DefaultMutableTreeNode</code> value
   * @param child a <code>TreeNode</code> value
   */
  public void swapTreeNode(DefaultMutableTreeNode parent, TreeNode child) {  
    int	index = parent.getIndex(child);
    System.err.println("PrubaeController.swapTreeNode() : index = " + index + ", insertNode = " + getTreeNode());
    parent.remove(index);
    parent.insert(getTreeNode(), index);
    if (getEditor().getTreeModel() != null) {
      getEditor().getTreeModel().reload(parent);
    }
    getEditor().setTreeSelection(getTreeNode());
  }

  /**
   * <code>deleteTreeNode</code> method
   * delete the tree node related this model from its parent tree node.
   * then set null to the tree node reference of this controller.
   * then sends reload to the tree model to notifify
   * that the tree structure may have changed.
   * then set the tree selection to the parent of the tree node deleted.
   */
  public void deleteTreeNode() {
    DefaultMutableTreeNode	child = getTreeNode();
    if (child == null) {
      return;
    }
    child.removeFromParent();
    setTreeNode(null);
    if (getEditor().getTreeModel() != null) {
      getEditor().getTreeModel().reload(child);
    }
    /* ;;; deleted
    getEditor().setTreeSelection(child);
    */
  }

  /**
   * <code>actionPerformed</code> method
   * handles the action events returned from GUI items on the panel
   * @param e an <code>ActionEvent</code> value is the action event.
   */
  public void actionPerformed(ActionEvent e) {  
  }

  /**
   * <code>toString</code> method
   * returns the String returned by toString from the model related.
   * @return a <code>String</code> value
   */
  public String	toString() {
    return model.toString();
  }

  /**
   * <code>deleteAndUp</code> method	delete this model and move to parent
   *
   */
  protected void deleteAndUp() {
    System.err.println("PrubaeController.deleteAndUp() : cntl = " + this);
    PrubaeController	parent = getModel().getParent().getController();
    DefaultMutableTreeNode
      node = (DefaultMutableTreeNode)getTreeNode().getParent();
    getModel().delete();
    parent.open();
    if (node != null) {
      parent.getEditor().setTreeSelection(node);
    }
  }
  
  /**
   * <code>delete</code> method	delete this model and move to current model
   *
   */
  protected void delete() {
    System.err.println("PrubaeController.delete() : cntl = " + this);
    Prubae	editor = getEditor();
    PrubaeController	currentCntl = editor.getController();
    getModel().delete();
    editor.setController(currentCntl);
    editor.setView(currentCntl.getView());
    DefaultMutableTreeNode
      node = (DefaultMutableTreeNode)currentCntl.getTreeNode();
    if (node != null) {
      editor.setTreeSelection(node);
    }
  }
  
  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeController	clone = null;
    try {
      clone = (PrubaeController)super.clone();
    }
    catch (CloneNotSupportedException e) {
      return null;
    }
    if (getTreeNode() != null) {
      clone.setTreeNode(new DefaultMutableTreeNode(clone));
      clone.getTreeNode().setUserObject(clone);
      clone.createPopupMenu();
    }
    if (getPanel() != null) {
      clone.setPanel(new JPanel());
    }
    if (getCommentField() != null) {
      clone.setCommentField(new JTextField());
    }
    return clone;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="treeNode"
   */
  public void setTreeNode(DefaultMutableTreeNode theTreeNode) {
    treeNode = theTreeNode;
  }

  /**
   * 
   * @uml.property name="treeNode"
   */
  public DefaultMutableTreeNode getTreeNode() {
    return treeNode;
  }

  /**
   * 
   * @uml.property name="panel"
   */
  public void setPanel(JPanel thePanel) {
    panel = thePanel;
  }

  /**
   * 
   * @uml.property name="panel"
   */
  public JPanel getPanel() {
    return panel;
  }

  /**
   * 
   * @uml.property name="model"
   */
  public void setModel(PrubaeModel theModel) {
    model = theModel;
  }

  /**
   * 
   * @uml.property name="model"
   */
  public PrubaeModel getModel() {
    return model;
  }

  /**
   * 
   * @uml.property name="view"
   */
  public void setView(PrubaeView theView) {
    view = theView;
  }

  /**
   * 
   * @uml.property name="view"
   */
  public PrubaeView getView() {
    return view;
  }

  /**
   * 
   * @uml.property name="editor"
   */
  public void setEditor(Prubae theEditor) {
    editor = theEditor;
  }

  /**
   * 
   * @uml.property name="editor"
   */
  public Prubae getEditor() {
    return editor;
  }

  /**
   * 
   * @uml.property name="commentField"
   */
  public void setCommentField(JTextField theCommentField) {
    commentField = theCommentField;
  }

  /**
   * 
   * @uml.property name="commentField"
   */
  public JTextField getCommentField() {
    return commentField;
  }

  public JPopupMenu getPopupMenu() {
    return popupMenu;
  }

  public void setPopupMenu(JPopupMenu popupMenu) {
    this.popupMenu = popupMenu;
  }

} /* end class PrubaeController */
