// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import gnu.lists.LList;
import gnu.math.IntNum;
import java.awt.event.ActionEvent;
import java.util.List;
import java.util.Vector;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.tree.DefaultMutableTreeNode;
import jp.sourceforge.glj.lisp.Lisp;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;


/**
 * Describe class <code>PrubaeControllerCase</code> here.
 * is the controller of the prubae rule case-branch element model.
 * this class controls the details input/display panel on the part panel.
 * this class also controls the tree node related to the case-branch model.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public class PrubaeControllerCase extends PrubaeController {

  // Attributes  
  /**
   * variable <code>argsCountPanel</code> is the panel which contains the args count field to change visibility.
   * @uml.property  name="argsCountPanel"
   */
  private JPanel argsCountPanel = null;
  /**
   * variable <code>argsCountLabel</code> is the label before args count field.
   * @uml.property  name="argsCountLabel"
   */
  private JLabel argsCountLabel = null;
  /**
   * variable <code>argsCountField</code> is the field to display/input the args coount value
   * @uml.property  name="argsCountField"
   */
  private JTextField argsCountField = null;
  /**
   * variable <code>argsCountButton</code> is the button to refresh args coount and the args panels.
   * @uml.property  name="argsCountButton"
   */
  private JButton argsCountButton = null;
  /**
   * variable <code>argsCount</code> is the number of args.
   * @uml.property  name="argsCount"
   */
  private int	argsCount = 0;
  /**
   * variable <code>argsPanels</code> is the list of panels which to display/input details of the args.
   * @uml.property  name="argsPanels"
   */
  private List argsPanels = null;

  // Constructor  
  /**
   * Creates a new <code>PrubaeControllerCase</code> instance.
   * creates delete button to delete this case-branch from the parent list,
   * insert button to insert new case-branch before this in the parent list,
   * and actions button to add action into the actions list in this model.
   * it also creates args count field to input the number of cases.
   */
  public PrubaeControllerCase() {  
    super();
  }

  // Operations  
  /**
   * <code>initialize</code> method
   * first creates treeNode related to this model using super.initialize().
   * then sets tree nodes related to the actions in the actions list,
   * if they exist, as the child nodes of the treeNode related to this.
   */
  public void initialize() {
    super.initialize();

    /* setActions(new DefaultMutableTreeNode("actions")); */
    List	actionList = ((PrubaeModelCase)getModel()).getActions();
    if (actionList != null) {
      for (int i = 0; i < actionList.size(); i++) {
	DefaultMutableTreeNode	node =
	  ((PrubaeModel)actionList.get(i)).getController().getTreeNode();
	getTreeNode().add(node);
      }
    }
    setArgsCount(0);
  }

  /**
   * <code>update</code> method
   * send updateCases() to the model to update all values in cases,
   * and updates comment by using super.update().
   */
  public void update() {
    ((PrubaeModelCase)getModel()).updateCases();
    super.update();
  }

  /**
   * <code>createPanel</code> method
   *
   */
  public void createPanel() {
    super.createPanel();
    getPanel().add(new JLabel("CASE"));
    JButton	deleteButton = new JButton("delete this case");
    deleteButton.setActionCommand("delete");
    deleteButton.addActionListener(this);
    JButton	insertButton = new JButton("insert case before this");
    insertButton.setActionCommand("insert");
    insertButton.addActionListener(this);
    JButton	actionButton = new JButton("add action");
    actionButton.setActionCommand("action");
    actionButton.addActionListener(this);
    JPanel	commandPanel = new JPanel();
    commandPanel.setLayout(new BoxLayout(commandPanel, BoxLayout.Y_AXIS));
    commandPanel.add(deleteButton);
    commandPanel.add(insertButton);
    commandPanel.add(actionButton);
    getPanel().add(commandPanel);

    setArgsCountPanel(new JPanel());
    setArgsCountLabel(new JLabel("COUNT"));
    setArgsCountField(new JTextField("", 4));
    setArgsCountButton(new JButton("INPUT"));
    getArgsCountButton().setActionCommand("set cases");
    getArgsCountButton().addActionListener(this);
    getArgsCountPanel().setLayout(new BoxLayout(getArgsCountPanel(),
						BoxLayout.X_AXIS));
    getArgsCountPanel().add(argsCountLabel);
    getArgsCountPanel().add(getArgsCountField());
    getArgsCountPanel().add(argsCountButton);
    getPanel().add(getArgsCountPanel());
  }

  /**
   * <code>destroyPanel</code> method
   *
   */
  public void destroyPanel() {
    setArgsCountLabel(null);
    setArgsCountField(null);
    setArgsCountButton(null);
    setArgsCountPanel(null);
    setArgsPanels(null);
    super.destroyPanel();
  }

  /**
   * <code>insert</code> method
   * creates new case-branch model and inserts it before this case-branch.
   * then send insertTreeNode() to the controller of the new case-branch,
   * to insert the tree node related to the new case-branch
   * as the child of the parent tree node.
   */
  protected void insert() {
    PrubaeModel	newModel = new PrubaeModelCase();
    newModel.insertToParent(getModel().getParentList(),
			    getModel().getParent(),
			    getModel());
    DefaultMutableTreeNode
      treeParent = (DefaultMutableTreeNode)getTreeNode().getParent();
    newModel.getController().insertTreeNode(treeParent, getTreeNode());
  }

  /**
   * <code>addActions</code> method
   * add action rule element model to the actions list of this model.
   */
  private void addActions() {
    List	actionsList = ((PrubaeModelCase)getModel()).getActions();
    if (actionsList != null) {
      PrubaeModel	last = null;
      if (actionsList.size() > 0) {
	last = (PrubaeModel)actionsList.get(actionsList.size()-1);
      }
      if (last != null) {
	System.err.println("addActions() last class name " +
			   last.getClass().getName());
      }
      if (last != null
	  && last.getClass().getName().endsWith("PrubaeModelDo")) {
	return;
      }
      last = new PrubaeModelDo();
      last.addToParent(actionsList, getModel());
      last.getController().addTreeNode(getTreeNode());
    }
  }

  /**
   * <code>updateArgsCount</code> method
   * first send updateCases() to the model to update all values in cases.
   * get and save old args count and set the new args count from the input.
   * if new args count > old one, increase arguments using addArguments(),
   * if new args count < old one, decrease arguments using removeArguments().
   */
  private void updateArgsCount() {
    ((PrubaeModelCase)getModel()).updateCases();
    int	numArgs = 0;
    try {
      numArgs = Integer.parseInt(getArgsCountField().getText());
    }
    catch (NumberFormatException e) {
      return;
    }
    int	oldArgsCount = getArgsCount();
    setArgsCount(numArgs);
    if (numArgs > oldArgsCount) {
      addArguments(oldArgsCount, numArgs);
    }
    else if (numArgs < oldArgsCount) {
      removeArguments(numArgs, oldArgsCount);
    }
  }

  /**
   * <code>addArguments</code> method
   * creates new case models for increased case-branches,
   * and add it to the values list of this model.
   * and then, for every new cases,
   * creates all value statement models in the new case,
   * creates view and controller and opens the GUI by sending openUI to them,
   * and add them to the values list of the new case.
   * @param minArgs an <code>int</code> value
   * is the number of arguments before increment.
   * @param numArgs an <code>int</code> value
   * is the number of arguments to increase to.
   */
  private void addArguments(int minArgs, int numArgs) {
    if (((PrubaeModelCase)getModel()).getValues() == null) {
      ((PrubaeModelCase)getModel()).setValues(new Vector());
    }
    PrubaeModelSwitch	parent = (PrubaeModelSwitch)getModel().getParent();
    /* ;;; add 2009.07.10 start */
    LList
      list = Lisp.append(Lisp.append(Lisp.append(getEditor().getValueList(),
						 getEditor().getVarList()),
				     getEditor().getConstList()),
			 getEditor().getSettablevarList());
    /* ;;; add 2009.07.10 end */
    for (int i = minArgs; i < numArgs; i++) {
      PrubaeModelValue	branch = new PrubaeModelValue();
      branch.setRegist((short)ProrateRulebaseElement.CASE_BRANCH_BEGIN);
      branch.setComment("");
      branch.setStatement("");
      branch.initialize(getEditor(), ((PrubaeModelCase)getModel()),
			((PrubaeModelCase)getModel()).getValues());
      ((PrubaeModelCase)getModel()).getValues().add(branch);
      for (int j = 0; j < parent.getVars().size(); j++) {
	/* ;;; add 2009.07.10 start */
	PrubaeModel	var = (PrubaeModel)parent.getVars().get(i);
	LList	valueList = Lisp.assoc(new IntNum(var.getRegist()), list);
	Object	type = Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.cdr(valueList))));
	/* ;;; add 2009.07.10 end
	Object	type = ((PrubaeModelValue)parent.getVars().get(j)).getType();
	*/
	PrubaeModelValue	child = new PrubaeModelValueStatement();
	child.setType(type);
	child.setComment("");
	child.setStatement("");
	child.initialize(getEditor(), (PrubaeModelValue)branch,
		       ((PrubaeModelValue)branch).getValues());
	/* ;;; add 2009.07.10 start */
	child.setRequiredType(type);
	/* ;;; add 2009.07.10 end */
	child.openUI();
	((PrubaeModelValue)branch).getValues().add(child);
      }
    }
  }

  /**
   * <code>removeArguments</code> method
   * for all decreased case-branches,
   * delete all view and controller and delete GUI items for all values in it,
   * and remove the case-branch from the values list of this model.
   * @param minArgs an <code>int</code> value
   * is the number of arguments to decrease to.
   * @param numArgs an <code>int</code> value
   * is the number of arguments before decrement.
   */
  private void removeArguments(int minArgs, int numArgs) {
    for (int i = numArgs-1; i >= minArgs; i--) {
      PrubaeModelValue	branch =
	(PrubaeModelValue)((PrubaeModelCase)getModel()).getValues().get(i);
      for (int j = ((PrubaeModelValue)branch).getValues().size()-1;
	   j >= 0; j--) {
	PrubaeModelValue	child
	  = (PrubaeModelValue)((PrubaeModelValue)branch).getValues().get(j);
	child.closeUI();
      }
      ((PrubaeModelCase)getModel()).getValues().remove(branch);
    }
  }

  /**
   * <code>actionPerformed</code> method
   * handles the action events returned from GUI items on the panel
   * send the model delete when the action command is "delete",
   * call insert() method when the action command is "insert",
   * call addActions() method when the action command is "action",
   * call updateArgsCount() method when the action command is "set cases",
   * and refresh the screen by sending display to the view.
   * @param e an <code>ActionEvent</code> value is the action event.
   */
  public void actionPerformed(ActionEvent e) {  
    if (e.getActionCommand().equals("delete")) {
      PrubaeController	cntl = getEditor().getController();
      if (this == cntl) {
	deleteAndUp();
      }
      else {
	delete();
      }
    }
    else if (e.getActionCommand().equals("insert")) {
      insert();
    }
    else if (e.getActionCommand().equals("action")) {
      addActions();
    }
    else {
      System.err.println("action = " + e.getActionCommand());
      if (e.getActionCommand().equals("set cases")) {
	updateArgsCount();
      }
      getView().display();
    }
  }

  /**
   * <code>createPopupMenu</code> method create popup menu for this node.
   *
   */
  void createPopupMenu() {
    JMenuItem	mi = null;
    setPopupMenu(new JPopupMenu());
    mi = new JMenuItem("delete this case");
    mi.setActionCommand("delete");
    mi.addActionListener(this);
    getPopupMenu().add(mi);
    mi = new JMenuItem("insert case before this");
    mi.setActionCommand("insert");
    mi.addActionListener(this);
    getPopupMenu().add(mi);
    mi = new JMenuItem("add action");
    mi.setActionCommand("action");
    mi.addActionListener(this);
    getPopupMenu().add(mi);
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeControllerCase	clone = null;
    clone = (PrubaeControllerCase)super.clone();
    if (getArgsCountPanel() != null) {
      clone.setArgsCountPanel(new JPanel());
    }
    if (getArgsCountField() != null) {
      clone.setArgsCountField(new JTextField());
    }
    clone.setArgsPanels((List)((Vector)getArgsPanels()).clone());
    return clone;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="argsCountPanel"
   */
  public void setArgsCountPanel(JPanel theArgsCountPanel) {
    argsCountPanel = theArgsCountPanel;
  }

  /**
   * 
   * @uml.property name="argsCountPanel"
   */
  public JPanel getArgsCountPanel() {
    return argsCountPanel;
  }

  /**
   * 
   * @uml.property name="argsCountLabel"
   */
  public void setArgsCountLabel(JLabel theArgsCountLabel) {
    argsCountLabel = theArgsCountLabel;
  }

  /**
   * 
   * @uml.property name="argsCountLabel"
   */
  public JLabel getArgsCountLabel() {
    return argsCountLabel;
  }

  /**
   * 
   * @uml.property name="argsCountField"
   */
  public void setArgsCountField(JTextField theArgsCountField) {
    argsCountField = theArgsCountField;
  }

  /**
   * 
   * @uml.property name="argsCountField"
   */
  public JTextField getArgsCountField() {
    return argsCountField;
  }

  /**
   * 
   * @uml.property name="argsCountButton"
   */
  public void setArgsCountButton(JButton theArgsCountButton) {
    argsCountButton = theArgsCountButton;
  }

  /**
   * 
   * @uml.property name="argsCountButton"
   */
  public JButton getArgsCountButton() {
    return argsCountButton;
  }

  /**
   * 
   * @uml.property name="argsCount"
   */
  public void setArgsCount(int theArgsCount) {
    argsCount = theArgsCount;
  }

  /**
   * 
   * @uml.property name="argsCount"
   */
  public int getArgsCount() {
    return argsCount;
  }

  /**
   * 
   * @uml.property name="argsPanels"
   */
  public void setArgsPanels(List theArgsPanels) {
    argsPanels = theArgsPanels;
  }

  /**
   * 
   * @uml.property name="argsPanels"
   */
  public List getArgsPanels() {
    return argsPanels;
  }

} /* end class PrubaeControllerCase */
