// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.util.List;
import java.util.Vector;

import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.tree.DefaultMutableTreeNode;

import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;

/**
 * class <code>PrubaeControllerFunction</code>
 * is the controller of the prubae rule function element model.
 * this class controls the details input/display panel on the part panel.
 * this class also controls the tree node related to the function model,
 * and tree nodes related to [common] and [actions] lists of the model.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public class PrubaeControllerFunction extends PrubaeController {

  // Attributes  
  /**
   * variable <code>firstField</code> is the carrier input/display field.
   * @uml.property  name="firstField"
   */
  private JTextField firstField = null;
  /**
   * variable <code>secondField</code> is the tkcarrier input/display field.
   * @uml.property  name="secondField"
   */
  private JTextField secondField = null;
  /**
   * variable <code>nameField</code> is the EXTF function name input/display field.
   * @uml.property  name="nameField"
   */
  private JTextField nameField = null;
  /**
   * variable <code>fromDate</code> is the valid from date input/display field
   * @uml.property  name="fromDate"
   */
  private JTextField fromDate = null;
  /**
   * variable <code>toDate</code> is the valid until date input/display field.
   * @uml.property  name="toDate"
   */
  private JTextField toDate = null;
  /**
   * variable <code>common</code> is the tree node which contains the ]common] action statement elements. this node is added as the child of the treeNode related to this model.
   * @uml.property  name="common"
   */
  private DefaultMutableTreeNode common = null;
  /**
   * variable <code>actions</code> is the tree node which contains the ]common] action statement elements. this node is added as the child of the treeNode related to this model.
   * @uml.property  name="actions"
   */
  private DefaultMutableTreeNode actions = null;
  /**
   * variable <code>judgeButton</code> is the radio button to select the extf returns the judgement boolean value.
   * @uml.property  name="judgeButton"
   */
  private JRadioButton judgeButton;
  /**
   * variable <code>actionButton</code> is the radio button to select the extf does action and does not return value.
   * @uml.property  name="actionButton"
   */
  private JRadioButton actionButton;
  /**
   * variable <code>valueButton</code> is the radio button to select the extf returns the object value.
   * @uml.property  name="valueButton"
   */
  private JRadioButton valueButton;
  /**
   * variable <code>argumentPanel</code> is the panel which contains panels which sets the arguments of the EXTF.
   *
   */
  private JPanel	argumentPanel;
  /**
   * variable <code>argsCountPanel</code> is the panel which contains the arguments count field to change visibility.
   *
   */
  private JPanel	argsCountPanel;
  /**
   * variable <code>argsCountLabel</code> is the label before arguments count field.
   *
   */
  private JLabel	argsCountLabel;
  /**
   * variable <code>argsCountField</code> is the field to display/input the arguments count value
   *
   */
  private JTextField	argsCountField;
  /**
   * variable <code>argsCountButton</code> is the button to refresh arguments count and the args panels.
   *
   */
  private JButton	argsCountButton;
  /**
   * variable <code>argFields</code> is the list of JTextField to insert the name of argument of EXTF.
   *
   */
  private List<JTextField>	argFields;

  // Constructor  
  /**
   * Creates a new <code>PrubaeControllerFunction</code> instance.
   *
   */
  public PrubaeControllerFunction() {
    super();
  }

  // Operations  
  /**
   * <code>initialize</code> method
   * first creates treeNode related to this model using super.initialize().
   * and then creates fields for carrier, tkcarrier, valid from/until date
   * when the rule is SPA or APDP,
   * creates fields for EXTF or PART name when the rule is EXTF or PART.
   * then creates common button and action button,
   * which to add an action to the [common]/[action] action lists.
   * then creates and set common and action tree nodes,
   * and also creates and set tree nodes related to the actions, if exists,
   * as their child nodes
   * then set common and action tree nodes to the treeNode as its child node.
   */
  public void initialize() {
    super.initialize();
    super.createPanel();

    if ((((PrubaeModelFunction)getModel()).getFunctionType()
	 == PrubaeModelFunction.SPA)
	|| (((PrubaeModelFunction)getModel()).getFunctionType()
	    == PrubaeModelFunction.APDP)) {
      JPanel carrierPane = new JPanel();
      JLabel	carrierLabel = new JLabel("carrier : ");
      setFirstField(new JTextField("", 2));
      carrierPane.setLayout(new FlowLayout(FlowLayout.LEADING));
      carrierPane.add(carrierLabel);
      carrierPane.add(getFirstField());
      if (((PrubaeModelFunction)getModel()).getFunctionType()
	  == PrubaeModelFunction.SPA) {
	setSecondField(new JTextField("", 2));
	carrierPane.add(getSecondField());
      }
      getPanel().add(carrierPane);
      JPanel datePane = new JPanel();
      JLabel	dateLabel = new JLabel("date : ");
      setFromDate(new JTextField(8));
      JLabel	dateLabel2 = new JLabel(" - ");
      setToDate(new JTextField(8));
      datePane.setLayout(new FlowLayout(FlowLayout.LEADING));
      datePane.add(dateLabel);
      datePane.add(getFromDate());
      datePane.add(dateLabel2);
      datePane.add(getToDate());
      getPanel().add(datePane);
    }

    if (((PrubaeModelFunction)getModel()).getFunctionType()
	== PrubaeModelFunction.EXTF
	|| (((PrubaeModelFunction)getModel()).getFunctionType()
	    == PrubaeModelFunction.PART)) {
      JPanel functionPane = new JPanel();
      String
	label = ((((PrubaeModelFunction)getModel()).getFunctionType()
		  == PrubaeModelFunction.PART) ? "part name : "
		 : "function name : ");
      JLabel	functionLabel = new JLabel(label);
      setNameField(new JTextField(16));
      functionPane.setLayout(new FlowLayout(FlowLayout.LEADING));
      functionPane.add(functionLabel);
      functionPane.add(getNameField());
      getPanel().add(functionPane);
    }

    if (((PrubaeModelFunction)getModel()).getFunctionType()
	== PrubaeModelFunction.EXTF) {
      setJudgeButton(new JRadioButton("judge"));
      setActionButton(new JRadioButton("action"));
      setValueButton(new JRadioButton("value"));
      ButtonGroup	group = new ButtonGroup();
      group.add(getJudgeButton());
      group.add(getActionButton());
      group.add(getValueButton());
      getJudgeButton().setActionCommand("judge");
      getJudgeButton().addActionListener(this);
      getActionButton().setActionCommand("action");
      getActionButton().addActionListener(this);
      getValueButton().setActionCommand("value");
      getValueButton().addActionListener(this);
      getJudgeButton().setSelected(false);
      getActionButton().setSelected(false);
      getValueButton().setSelected(true);
      JPanel	extrReturnTypePanel = new JPanel();
      extrReturnTypePanel.setLayout(new BoxLayout(extrReturnTypePanel,
						  BoxLayout.X_AXIS));
      extrReturnTypePanel.add(new JLabel("Value Type"));
      extrReturnTypePanel.add(getJudgeButton());
      extrReturnTypePanel.add(getActionButton());
      extrReturnTypePanel.add(getValueButton());
      getPanel().add(extrReturnTypePanel);

      setArgumentPanel(new JPanel());
      getArgumentPanel().setLayout(new BoxLayout(getArgumentPanel(),
						 BoxLayout.Y_AXIS));
      setArgsCountPanel(new JPanel());
      setArgsCountLabel(new JLabel("COUNT"));
      setArgsCountField(new JTextField("", 4));
      setArgsCountButton(new JButton("INPUT"));
      getArgsCountButton().setActionCommand("set args");
      getArgsCountButton().addActionListener(this);
      getArgsCountPanel().setLayout(new BoxLayout(getArgsCountPanel(),
						  BoxLayout.X_AXIS));
      getArgsCountPanel().add(getArgsCountLabel());
      getArgsCountPanel().add(getArgsCountField());
      getArgsCountPanel().add(getArgsCountButton());
      getArgumentPanel().add(getArgsCountPanel());
      getPanel().add(getArgumentPanel());
    }

    JPanel	commandPanel = new JPanel();
    commandPanel.setLayout(new BoxLayout(commandPanel, BoxLayout.Y_AXIS));
    if ((((PrubaeModelFunction)getModel()).getFunctionType()
	 == PrubaeModelFunction.SPA)
	|| (((PrubaeModelFunction)getModel()).getFunctionType()
	    == PrubaeModelFunction.APDP)) {
      JButton	commonButton = new JButton("add common");
      commonButton.setActionCommand("common");
      commonButton.addActionListener(this);
      commandPanel.add(commonButton);
    }
    JButton	actionButton = new JButton("add actions");
    actionButton.setActionCommand("actions");
    actionButton.addActionListener(this);
    commandPanel.add(actionButton);
    getPanel().add(commandPanel);

    List	commonList = ((PrubaeModelFunction)getModel()).getCommon();
    if ((((PrubaeModelFunction)getModel()).getFunctionType()
	 == PrubaeModelFunction.SPA)
	|| (((PrubaeModelFunction)getModel()).getFunctionType()
	    == PrubaeModelFunction.APDP)) {
      setCommon(new DefaultMutableTreeNode("common"));
    }
    else if (commonList != null && commonList.size() > 0) {
      setCommon(new DefaultMutableTreeNode("common"));
    }
    if (commonList != null) {
      for (int i = 0; i < commonList.size(); i++) {
	DefaultMutableTreeNode
	  node = ((PrubaeModel)commonList.get(i)).getController().getTreeNode();
	getCommon().add(node);
      }
    }
    setActions(new DefaultMutableTreeNode("actions"));
    List	actionList = ((PrubaeModelFunction)getModel()).getActions();
    if (actionList != null) {
      for (int i = 0; i < actionList.size(); i++) {
	DefaultMutableTreeNode	node =
	  ((PrubaeModel)actionList.get(i)).getController().getTreeNode();
	getActions().add(node);
      }
    }
    if (getCommon() != null) {
      getTreeNode().add(getCommon());
    }
    getTreeNode().add(getActions());
  }

  /**
   * <code>createPopupMenu</code> method	do nothing.
   *
   */
  void createPopupMenu() {
  }

  /**
   * <code>update</code> method
   * updates the model with the input values on the panel.
   * this updates the carrier, tkcarrier, valid from/until dates,
   * or the EXTF function name.
   * then updates the comment and refresh the view using super.update().
   */
  public void update() {
    PrubaeModelFunction	model = (PrubaeModelFunction)getModel();
    if (model.getFunctionType() == PrubaeModelFunction.SPA
	|| model.getFunctionType() == PrubaeModelFunction.APDP) {
      model.setCarrier(getFirstField().getText());
      if (model.getFunctionType() == PrubaeModelFunction.SPA) {
	model.setTkCarrier(getSecondField().getText());
      }
      int	value = 0;
      try {
	value = Integer.parseInt(getFromDate().getText());
      }
      catch (NumberFormatException e) {
	value = 0;
      }
      model.setFromDate(value);
      try {
	value = Integer.parseInt(getToDate().getText());
      }
      catch (NumberFormatException e) {
	value = 0;
      }
      model.setToDate(value);
    }
    else if (model.getFunctionType() == PrubaeModelFunction.EXTF) {
      model.setName(getNameField().getText());
      if (model.getArgs() != null) {
	for (int i = 0; i < model.getArgs().size(); i++) {
	  JTextField	argField = getArgFields().get(i);
	  model.getArgs().set(i, argField.getText());
	}
      }
      model.updateArgs();
    }
    else if (model.getFunctionType() == PrubaeModelFunction.PART) {
      model.setName(getNameField().getText());
    }

    super.update();
  }

  /**
   * <code>createPanel</code> method
   *
   */
  public void createPanel() {
  }

  /**
   * <code>destroyPanel</code> method
   *
   */
  public void destroyPanel() {
  }

  /**
   * <code>addCommon</code> method
   * adds an action rule element to the [common] list of the model.
   */
  private void addCommon() {
    update();
    List	commonList = ((PrubaeModelFunction)getModel()).getCommon();
    if (commonList != null) {
      PrubaeModel	last = null;
      if (commonList.size() > 0) {
	last = (PrubaeModel)commonList.get(commonList.size()-1);
      }
      if (last != null) {
	System.err.println("addCommon() last class name " +
			   last.getClass().getName());
      }
      if (last != null
	  && last.getClass().getName().endsWith("PrubaeModelDo")) {
	return;
      }
      last = new PrubaeModelDo();
      last.addToParent(commonList, getModel());
      last.getController().addTreeNode(getCommon());
    }
  }

  /**
   * <code>addAction</code> method
   * adds an action rule element to the [actions] list of the model.
   */
  private void addAction() {  
    update();
    List	actionList = ((PrubaeModelFunction)getModel()).getActions();
    if (actionList != null) {
      PrubaeModel	last = null;
      if (actionList.size() > 0) {
	last = (PrubaeModel)actionList.get(actionList.size()-1);
      }
      if (last != null
	  && last.getClass().getName().endsWith("PrubaeModelDo")) {
	return;
      }
      last = new PrubaeModelDo();
      last.addToParent(actionList, getModel());
      last.getController().addTreeNode(getActions());
    }
  }

  /**
   * <code>updateArgsCount</code> method
   * get and save old args count and set the new args count from the input.
   * if new args count > old one, increase args using addArgs() of the model,
   * new args count < old one, decrease args using removeArgs() of the model.
   */
  void updateArgsCount() {
    int	numArgs = 0;
    try {
      numArgs = Integer.parseInt(getArgsCountField().getText());
    }
    catch (NumberFormatException e) {
      return;
    }
    PrubaeModelFunction	model = (PrubaeModelFunction)getModel();
    model.setName(getNameField().getText());
    model.updateArgs();
    int	oldArgsCount = 0;
    if (model.getArgs() != null) {
      oldArgsCount = model.getArgs().size();
    }
    if (numArgs > oldArgsCount) {
      addArgs(oldArgsCount, numArgs);
    }
    else if (numArgs < oldArgsCount) {
      removeArgs(numArgs, oldArgsCount);
    }
  }

  /**
   * <code>addArgs</code> method
   * creates new arg value models for increased arg values,
   * creates view and controller and opens the GUI by sending openUI to them,
   * and add them to the arg values list of this model.
   * @param minArgs an <code>int</code> value
   * is the number of arguments before increment.
   * @param numArgs an <code>int</code> value
   * is the number of arguments to increase to.
   */
  public void addArgs(int minArgs, int numArgs) {
    PrubaeModelFunction	model = (PrubaeModelFunction)getModel();
    if (model.getArgs() == null) {
      model.setArgs(new Vector());
    }
    if (getArgFields() == null) {
      setArgFields(new Vector());
    }
    for (int i = minArgs; i < numArgs; i++) {
      String	arg = new String();
      model.getArgs().add(arg);
      JTextField	argField = new JTextField();
      getArgFields().add(argField);
      getArgumentPanel().add(argField);
    }
  }

  /**
   * <code>removeArgs</code> method
   * for all decreased arg values,
   * delete all view and controller and delete GUI items of them,
   * and remove the arg value from the arg values list of this model.
   * @param minArgs an <code>int</code> value
   * is the number of arguments to decrease to.
   * @param numArgs an <code>int</code> value
   * is the number of arguments before decrement.
   */
  public void removeArgs(int minArgs, int numArgs) {
    PrubaeModelFunction	model = (PrubaeModelFunction)getModel();
    for (int i = numArgs-1; i >= minArgs; i--) {
      model.getArgs().remove(i);
      JTextField	argField = (JTextField)getArgFields().get(i);
      getArgFields().remove(argField);
      getArgumentPanel().remove(argField);
    }
  }

  /**
   * <code>actionPerformed</code> method
   * handles the action events returned from GUI items on the panel
   * call addCommon() method when the action command is "common",
   * call addAction() method when the action command is "actions".
   * after that, refresh the screen by sending display to the view.
   * @param e an <code>ActionEvent</code> value is the action event.
   */
  public void actionPerformed(ActionEvent e) {
    System.err.println("action = " + e.getActionCommand());
    if (e.getActionCommand().equals("common")) {
      addCommon();
    }
    else if (e.getActionCommand().equals("actions")) {
      addAction();
    }
    else if (e.getActionCommand().equals("judge")) {
      PrubaeModelFunction	model = (PrubaeModelFunction)getModel();
      model.setName(getNameField().getText());
      model.setExtfReturnType(PrubaeModelFunction.ExtfReturnJudgement);
    }
    else if (e.getActionCommand().equals("action")) {
      PrubaeModelFunction	model = (PrubaeModelFunction)getModel();
      model.setName(getNameField().getText());
      model.setExtfReturnType(PrubaeModelFunction.ExtfReturnAction);
    }
    else if (e.getActionCommand().equals("value")) {
      PrubaeModelFunction	model = (PrubaeModelFunction)getModel();
      model.setName(getNameField().getText());
      model.setExtfReturnType(PrubaeModelFunction.ExtfReturnValue);
    }
    else if (e.getActionCommand().equals("set args")) {
      updateArgsCount();
    }
    getView().display();
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeControllerFunction	clone = null;
    clone = (PrubaeControllerFunction)super.clone();
    if (getFirstField() != null) {
      clone.setFirstField(new JTextField());
    }
    if (getSecondField() != null) {
      clone.setSecondField(new JTextField());
    }
    if (getNameField() != null) {
      clone.setNameField(new JTextField());
    }
    if (getFromDate() != null) {
      clone.setFromDate(new JTextField());
    }
    if (getToDate() != null) {
      clone.setToDate(new JTextField());
    }
    clone.setCommon((DefaultMutableTreeNode)getCommon().clone());
    clone.setActions((DefaultMutableTreeNode)getActions().clone());
    return clone;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="firstField"
   */
  public void setFirstField(JTextField theFirstField) {
    firstField = theFirstField;
  }

  /**
   * 
   * @uml.property name="firstField"
   */
  public JTextField getFirstField() {
    return firstField;
  }

  /**
   * 
   * @uml.property name="secondField"
   */
  public void setSecondField(JTextField theSecondField) {
    secondField = theSecondField;
  }

  /**
   * 
   * @uml.property name="secondField"
   */
  public JTextField getSecondField() {
    return secondField;
  }

  /**
   * 
   * @uml.property name="nameField"
   */
  public void setNameField(JTextField theNameField) {
    nameField = theNameField;
  }

  /**
   * 
   * @uml.property name="nameField"
   */
  public JTextField getNameField() {
    return nameField;
  }

  /**
   * 
   * @uml.property name="common"
   */
  public void setCommon(DefaultMutableTreeNode theCommon) {
    common = theCommon;
  }

  /**
   * 
   * @uml.property name="common"
   */
  public DefaultMutableTreeNode getCommon() {
    return common;
  }

  /**
   * 
   * @uml.property name="fromDate"
   */
  public void setFromDate(JTextField theFromDate) {
    fromDate = theFromDate;
  }

  /**
   * 
   * @uml.property name="fromDate"
   */
  public JTextField getFromDate() {
    return fromDate;
  }

  /**
   * 
   * @uml.property name="toDate"
   */
  public void setToDate(JTextField theToDate) {
    toDate = theToDate;
  }

  /**
   * 
   * @uml.property name="toDate"
   */
  public JTextField getToDate() {
    return toDate;
  }

  /**
   * 
   * @uml.property name="actions"
   */
  public void setActions(DefaultMutableTreeNode theActions) {
    actions = theActions;
  }

  /**
   * 
   * @uml.property name="actions"
   */
  public DefaultMutableTreeNode getActions() {
    return actions;
  }

  public JRadioButton getJudgeButton() {
    return judgeButton;
  }

  public void setJudgeButton(JRadioButton judgeButton) {
    this.judgeButton = judgeButton;
  }

  public JRadioButton getActionButton() {
    return actionButton;
  }

  public void setActionButton(JRadioButton actionButton) {
    this.actionButton = actionButton;
  }

  public JRadioButton getValueButton() {
    return valueButton;
  }

  public void setValueButton(JRadioButton valueButton) {
    this.valueButton = valueButton;
  }

  public JPanel getArgumentPanel() {
    return argumentPanel;
  }

  public void setArgumentPanel(JPanel argumentPanel) {
    this.argumentPanel = argumentPanel;
  }

  public JPanel getArgsCountPanel() {
    return argsCountPanel;
  }

  public void setArgsCountPanel(JPanel argsCountPanel) {
    this.argsCountPanel = argsCountPanel;
  }

  public JLabel getArgsCountLabel() {
    return argsCountLabel;
  }

  public void setArgsCountLabel(JLabel argsCountLabel) {
    this.argsCountLabel = argsCountLabel;
  }

  public JTextField getArgsCountField() {
    return argsCountField;
  }

  public void setArgsCountField(JTextField argsCountField) {
    this.argsCountField = argsCountField;
  }

  public JButton getArgsCountButton() {
    return argsCountButton;
  }

  public void setArgsCountButton(JButton argsCountButton) {
    this.argsCountButton = argsCountButton;
  }

  public List<JTextField> getArgFields() {
    return argFields;
  }

  public void setArgFields(List<JTextField> argFields) {
    this.argFields = argFields;
  }

} /* end class PrubaeControllerFunction */
