// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import java.awt.event.ActionEvent;
import java.awt.Dimension;

import java.util.List;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.tree.DefaultMutableTreeNode;

/**
 * class <code>PrubaeControllerIf</code>
 * is the controller of the prubae rule if-statement element model.
 * this class controls the details input/display panel on the part panel.
 * this class also controls the tree node related to the if-statement model,
 * and tree nodes related to thens and elses lists of the model.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public class PrubaeControllerIf extends PrubaeControllerDo {

  // Attributes  
  /**
   * variable <code>thens</code> is the tree node which contains the thens action statement elements this node is added as the child of the treeNode related to this model.
   * @uml.property  name="thens"
   */
  private DefaultMutableTreeNode thens = null;  
  /**
   * variable <code>thens</code> is the tree node which contains the elses action statement elements this node is added as the child of the treeNode related to this model.
   * @uml.property  name="elses"
   */
  private DefaultMutableTreeNode elses = null;  

  // Constructor  
  /**
   * Creates a new <code>PrubaeControllerIf</code> instance.
   *
   */
  public PrubaeControllerIf() {  
    super();  
  }

  // Operations  
  /**
   * <code>createButtons</code> method
   * creates thens button and elses button,
   * which is to add an action to the thens/elses action lists.
   */
  protected void createButtons() {
    getPanel().add(new JLabel("IF"));
    JButton	thenButton = new JButton("add then");
    thenButton.setPreferredSize(new Dimension(150, 50));
    thenButton.setActionCommand("then");
    thenButton.addActionListener(this);
    JButton	elseButton = new JButton("add else");
    elseButton.setActionCommand("else");
    elseButton.addActionListener(this);
    elseButton.setPreferredSize(new Dimension(150, 50));
    JPanel	opPanel = new JPanel();
    opPanel.setLayout(new BoxLayout(opPanel, BoxLayout.Y_AXIS));
    opPanel.add(thenButton);
    opPanel.add(elseButton);
    getPanel().add(opPanel);
  }

  /**
   * <code>initialize</code> method
   * first creates treeNode related to this model using super.initialize().
   * then creates and sets thens and elses tree nodes,
   * and also sets tree nodes related to the actions, if they exist,
   * as the child nodes of either thens/elses tree node.
   * then set thens and elses tree nodes to the treeNode as its child node.
   */
  public void initialize() {
    super.initialize();

    PrubaeModel	condition = ((PrubaeModelIf)getModel()).getCondition();
    setThens(new DefaultMutableTreeNode("thens"));
    List	thensList = ((PrubaeModelIf)getModel()).getThens();
    if (thensList != null) {
      for (int i = 0; i < thensList.size(); i++) {
	DefaultMutableTreeNode
	  node = ((PrubaeModel)thensList.get(i)).getController().getTreeNode();
	getThens().add(node);
      }
    }
    setElses(new DefaultMutableTreeNode("elses"));
    List	elsesList = ((PrubaeModelIf)getModel()).getElses();
    if (elsesList != null) {
      for (int i = 0; i < elsesList.size(); i++) {
	DefaultMutableTreeNode
	  node = ((PrubaeModel)elsesList.get(i)).getController().getTreeNode();
	getElses().add(node);
      }
    }
    getTreeNode().add(condition.getController().getTreeNode());
    getTreeNode().add(getThens());
    getTreeNode().add(getElses());
  }

  /**
   * <code>destroyPanel</code> method
   *
   */
  public void destroyPanel() {
    super.destroyPanel();
  }

  /**
   * <code>addThens</code> method
   * adds an action rule element to the thens list of the model
   */
  private void addThens() {
    List	thensList = ((PrubaeModelIf)getModel()).getThens();
    if (thensList != null) {
      PrubaeModel	last = null;
      if (thensList.size() > 0) {
	last = (PrubaeModel)thensList.get(thensList.size()-1);
      }
      if (last != null) {
	System.err.println("addThens() last class name " +
			   last.getClass().getName());
      }
      if (last != null
	  && last.getClass().getName().endsWith("PrubaeModelDo")) {
	return;
      }
      last = new PrubaeModelDo();
      last.addToParent(thensList, getModel());
      last.getController().addTreeNode(getThens());
    }
  }

  /**
   * <code>addElses</code> method
   * adds an action rule element to the elses list of the model
   */
  private void addElses() {
    List	elsesList = ((PrubaeModelIf)getModel()).getElses();
    if (elsesList != null) {
      PrubaeModel	last = null;
      if (elsesList.size() > 0) {
	last = (PrubaeModel)elsesList.get(elsesList.size()-1);
      }
      if (last != null
	  && last.getClass().getName().endsWith("PrubaeModelDo")) {
	return;
      }
      last = new PrubaeModelDo();
      last.addToParent(elsesList, getModel());
      last.getController().addTreeNode(getElses());
    }
  }

  /**
   * <code>actionPerformed</code> method
   * handles the action events returned from GUI items on the panel
   * send the model delete when the action command is "delete",
   * call insert() method when the action command is "insert",
   * call addThens() method when the action command is "then",
   * call addElses() method when the action command is "elses".
   * after addthen/else, refresh the screen by sending display to the view.
   * @param e an <code>ActionEvent</code> value is the action event.
   */
  public void actionPerformed(ActionEvent e) {
    if (e.getActionCommand().equals("delete")) {
      PrubaeController	cntl = getEditor().getController();
      if (this == cntl) {
	deleteAndUp();
      }
      else {
	delete();
      }
    }
    else if (e.getActionCommand().equals("insert")) {
      insert();
    }
    else {
      System.err.println("action = " + e.getActionCommand());
      if (e.getActionCommand().equals("then")) {
	addThens();
      }
      else if (e.getActionCommand().equals("else")) {
	addElses();
      }
      getView().display();
    }
  }

  /**
   * <code>createPopupMenu</code> method create popup menu for this node.
   *
   */
  void createPopupMenu() {
    JMenuItem	mi = null;
    setPopupMenu(new JPopupMenu());
    mi = new JMenuItem("delete this action");
    mi.setActionCommand("delete");
    mi.addActionListener(this);
    getPopupMenu().add(mi);
    mi = new JMenuItem("insert action before this");
    mi.setActionCommand("insert");
    mi.addActionListener(this);
    getPopupMenu().add(mi);
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeControllerIf	clone = null;
    clone = (PrubaeControllerIf)super.clone();
    clone.setThens((DefaultMutableTreeNode)getThens().clone());
    clone.setElses((DefaultMutableTreeNode)getElses().clone());
    return clone;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="thens"
   */
  public void setThens(DefaultMutableTreeNode theThens) {
    thens = theThens;
  }

  /**
   * 
   * @uml.property name="thens"
   */
  public DefaultMutableTreeNode getThens() {
    return thens;
  }

  /**
   * 
   * @uml.property name="elses"
   */
  public void setElses(DefaultMutableTreeNode theElses) {
    elses = theElses;
  }

  /**
   * 
   * @uml.property name="elses"
   */
  public DefaultMutableTreeNode getElses() {
    return elses;
  }

} /* end class PrubaeControllerIf */
