// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import java.awt.event.ActionEvent;
import java.util.List;
import java.util.Vector;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.tree.DefaultMutableTreeNode;

/**
 * class <code>PrubaeControllerSwitch</code>
 * is the controller of the prubae rule if-statement element model.
 * this class controls the details input/display panel on the part panel.
 * this class also controls the tree node related to the if-statement model,
 *
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public class PrubaeControllerSwitch extends PrubaeControllerDo {

  // Attributes  
  /**
   * variable <code>argsCountPanel</code> is the panel which contains the args count field to change visibility.
   * @uml.property  name="argsCountPanel"
   */
  private JPanel argsCountPanel;
  /**
   * variable <code>argsCountLabel</code> is the label before args count field.
   * @uml.property  name="argsCountLabel"
   */
  private JLabel argsCountLabel;
  /**
   * variable <code>argsCountField</code> is the field to display/input the args count value
   * @uml.property  name="argsCountField"
   */
  private JTextField argsCountField;
  /**
   * variable <code>argsCountButton</code> is the button to refresh args count and the args panels.
   * @uml.property  name="argsCountButton"
   */
  private JButton argsCountButton;
  /**
   * variable <code>argsCount</code> is the number of args.
   * @uml.property  name="argsCount"
   */
  private int	argsCount = 0;

  // Constructor  
  /**
   * Creates a new <code>PrubaeControllerSwitch</code> instance.
   *
   */
  public PrubaeControllerSwitch() {  
    super();  
  }


  // Operations  
  /**
   * <code>createButtons</code> method
   * creates case button to add new case-branch into the cases list,
   * it also creates args count field to input the number of case-branches.
   */
  protected void createButtons() {  
    getPanel().add(new JLabel("SWITCH"));
    JButton	caseButton = new JButton("add case");
    caseButton.setActionCommand("case");
    caseButton.addActionListener(this);
    JPanel	opPanel = new JPanel();
    opPanel.setLayout(new BoxLayout(opPanel, BoxLayout.Y_AXIS));
    opPanel.add(caseButton);
    getPanel().add(opPanel);

    setArgsCountPanel(new JPanel());
    setArgsCountLabel(new JLabel("COUNT"));
    setArgsCountField(new JTextField("", 4));
    setArgsCountButton(new JButton("INPUT"));
    getArgsCountButton().setActionCommand("set vars");
    getArgsCountButton().addActionListener(this);
    getArgsCountPanel().setLayout(new BoxLayout(getArgsCountPanel(),
						BoxLayout.X_AXIS));
    getArgsCountPanel().add(argsCountLabel);
    getArgsCountPanel().add(getArgsCountField());
    getArgsCountPanel().add(argsCountButton);
    getPanel().add(getArgsCountPanel());
  }

  /**
   * <code>initialize</code> method
   * first creates treeNode related to this model using super.initialize().
   * then sets tree nodes related to the case-branches in the cases list,
   * if they exist, as the child nodes of the treeNode related to this.
   */
  public void initialize() {
    super.initialize();

    List	caseList = ((PrubaeModelSwitch)getModel()).getCases();
    if (caseList != null) {
      for (int i = 0; i < caseList.size(); i++) {
	DefaultMutableTreeNode
	  node = ((PrubaeModel)caseList.get(i)).getController().getTreeNode();
	getTreeNode().add(node);
      }
    }
    setArgsCount(0);
  }

  /**
   * <code>update</code> method
   * send updateVars() to the model to update all values in args,
   * and updates comment by using super.update().
   */
  public void update() {
    ((PrubaeModelSwitch)getModel()).updateVars();
    super.update();
  }

  /**
   * <code>destroyPanel</code> method
   *
   */
  public void destroyPanel() {
    setArgsCountPanel(null);
    setArgsCountLabel(null);
    setArgsCountField(null);
    setArgsCountButton(null);
    super.destroyPanel();
  }

  /**
   * <code>addCases</code> method
   * creates new case-branch model and add it at the end of cases list.
   * then send addTreeNode to thhe controller of the new case-branch,
   * to add the tree node related to the new case-branch
   * as the child of the parent tree noce.
   */
  private void addCases() {
    List	casesList = ((PrubaeModelSwitch)getModel()).getCases();
    if (casesList != null) {
      PrubaeModel	last = new PrubaeModelCase();
      last.addToParent(casesList, getModel());
      last.getController().addTreeNode(getTreeNode());
    }
  }

  /**
   * <code>updateArgsCount</code> method
   * get and save old args count and set the new args count from the input.
   * if new args count > old one, increase arguments using addArguments(),
   * if new args count < old one, decrease arguments using removeArguments().
   */
  private void updateArgsCount() {
    ((PrubaeModelSwitch)getModel()).updateVars();
    int	numArgs = 0;
    try {
      numArgs = Integer.parseInt(getArgsCountField().getText());
    }
    catch (NumberFormatException e) {
      return;
    }
    int	oldArgsCount = getArgsCount();
    setArgsCount(numArgs);
    if (numArgs > oldArgsCount) {
      addArguments(oldArgsCount, numArgs);
    }
    else if (numArgs < oldArgsCount) {
      removeArguments(numArgs, oldArgsCount);
    }
  }

  /**
   * <code>addArguments</code> method
   * creates new case models for increased case-branches,
   * creates view and controller and opens the GUI by sending openUI to them,
   * and add them to the values list of this model.
   * @param minArgs an <code>int</code> value
   * is the number of arguments before increment.
   * @param numArgs an <code>int</code> value
   * is the number of arguments to increase to.
   */
  private void addArguments(int minArgs, int numArgs) {
    PrubaeModelSwitch	model = (PrubaeModelSwitch)getModel();
    if (model.getVars() == null) {
      model.setVars(new Vector());
    }
    for (int i = minArgs; i < numArgs; i++) {
      PrubaeModelValue	ret = new PrubaeModelValueStatement();
      ret.setComment("");
      ret.setStatement("");
      ret.initialize(getEditor(), model, model.getVars());
      ret.openUI();
      model.getVars().add(ret);
      getPanel().add(ret.getController().getPanel());
    }
    /* ;;; add start 2009.07.06 */
    for (int i = 0; i < model.getCases().size(); i++) {
      PrubaeModelCase	caseModel = (PrubaeModelCase)model.getCases().get(i);
      for (int j = 0; j < caseModel.getValues().size(); j++) {
	PrubaeModelValue
	  caseBranch = ((PrubaeModelValue)caseModel.getValues().get(j));
	for (int k = minArgs; k < numArgs; k++) {
	  PrubaeModelValue	value = new PrubaeModelValueStatement();
	  value.setComment("");
	  value.setStatement("");
	  value.initialize(getEditor(), caseBranch, caseBranch.getValues());
	  caseBranch.getValues().add(value);
	}
      }
    }
    /* ;;; add end 2009.07.06 */
  }

  /**
   * <code>removeArguments</code> method
   * for all decreased case-branches,
   * delete all view and controller and delete GUI items of them,
   * and remove the case-branch from the values list of this model.
   * @param minArgs an <code>int</code> value
   * is the number of arguments to decrease to.
   * @param numArgs an <code>int</code> value
   * is the number of arguments before decrement.
   */
  private void removeArguments(int minArgs, int numArgs) {
    PrubaeModelSwitch	model = (PrubaeModelSwitch)getModel();
    for (int i = numArgs-1; i >= minArgs; i--) {
      PrubaeModelValue	ret = (PrubaeModelValue)model.getVars().get(i);
      getPanel().remove(ret.getController().getPanel());
      model.getVars().remove(ret);
      ret.closeUI();
    }
    /* ;;; add start 2009.07.06 */
    for (int i = 0; i < model.getCases().size(); i++) {
      PrubaeModelCase	caseModel = (PrubaeModelCase)model.getCases().get(i);
      for (int j = 0; j < caseModel.getValues().size(); j++) {
	PrubaeModelValue
	  caseBranch = ((PrubaeModelValue)caseModel.getValues().get(j));
	for (int k = numArgs-1; k >= minArgs; k--) {
	  PrubaeModelValue
	    value = (PrubaeModelValueStatement)caseBranch.getValues().get(k);
	  caseBranch.getValues().remove(value);
	}
      }
    }
    /* ;;; add end 2009.07.06 */
  }

  /**
   * <code>actionPerformed</code> method
   * handles the action events returned from GUI items on the panel
   * send the model delete when the action command is "delete",
   * call insert() method when the action command is "insert",
   * call addCases() method when the action command is "case",
   * call updateArgsCount() method when the action command is "set cases",
   * @param e an <code>ActionEvent</code> value
   */
  public void actionPerformed(ActionEvent e) {  
    if (e.getActionCommand().equals("delete")) {
      PrubaeController	cntl = getEditor().getController();
      if (this == cntl) {
	deleteAndUp();
      }
      else {
	delete();
      }
    }
    else if (e.getActionCommand().equals("insert")) {
      insert();
    }
    else {
      System.err.println("action = " + e.getActionCommand());
      if (e.getActionCommand().equals("case")) {
	addCases();
      }
      else if (e.getActionCommand().equals("set vars")) {
	updateArgsCount();
      }
      getView().display();
    }
  }

  /**
   * <code>createPopupMenu</code> method create popup menu for this node.
   *
   */
  void createPopupMenu() {
    JMenuItem	mi = null;
    setPopupMenu(new JPopupMenu());
    mi = new JMenuItem("delete this action");
    mi.setActionCommand("delete");
    mi.addActionListener(this);
    getPopupMenu().add(mi);
    mi = new JMenuItem("insert action before this");
    mi.setActionCommand("insert");
    mi.addActionListener(this);
    getPopupMenu().add(mi);
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeControllerSwitch	clone = null;
    clone = (PrubaeControllerSwitch)super.clone();
    if (getArgsCountPanel() != null) {
      clone.setArgsCountPanel(new JPanel());
    }
    if (getArgsCountField() != null) {
      clone.setArgsCountField(new JTextField());
    }
    return clone;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="argsCountPanel"
   */
  public void setArgsCountPanel(JPanel theArgsCountPanel) {
    argsCountPanel = theArgsCountPanel;
  }

  /**
   * 
   * @uml.property name="argsCountPanel"
   */
  public JPanel getArgsCountPanel() {
    return argsCountPanel;
  }

  /**
   * 
   * @uml.property name="argsCountLabel"
   */
  public void setArgsCountLabel(JLabel theArgsCountLabel) {
    argsCountLabel = theArgsCountLabel;
  }

  /**
   * 
   * @uml.property name="argsCountLabel"
   */
  public JLabel getArgsCountLabel() {
    return argsCountLabel;
  }

  /**
   * 
   * @uml.property name="argsCountField"
   */
  public void setArgsCountField(JTextField theArgsCountField) {
    argsCountField = theArgsCountField;
  }

  /**
   * 
   * @uml.property name="argsCountField"
   */
  public JTextField getArgsCountField() {
    return argsCountField;
  }

  /**
   * 
   * @uml.property name="argsCountButton"
   */
  public void setArgsCountButton(JButton theArgsCountButton) {
    argsCountButton = theArgsCountButton;
  }

  /**
   * 
   * @uml.property name="argsCountButton"
   */
  public JButton getArgsCountButton() {
    return argsCountButton;
  }

  /**
   * 
   * @uml.property name="argsCount"
   */
  public void setArgsCount(int theArgsCount) {
    argsCount = theArgsCount;
  }

  /**
   * 
   * @uml.property name="argsCount"
   */
  public int getArgsCount() {
    return argsCount;
  }

} /* end class PrubaeControllerSwitch */
