// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import gnu.lists.LList;
import gnu.lists.Pair;
import gnu.math.IntNum;
import java.awt.event.ActionEvent;
import java.util.List;
import java.util.Vector;
import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import jp.sourceforge.glj.lisp.Lisp;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;


/**
 * class <code>PrubaeControllerValueStatement</code>
 * is the controller of the prubae rule value statement element model.
 * this class controls the details input/display subpanel on the part panel.
 *
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public class PrubaeControllerValueStatement extends PrubaeController {

  // Attributes  
  /**
   * attribute variable <code>statementField</code> is the JTextField to display the judge statement string. not editable.
   * @uml.property  name="statementField"
   */
  private JTextField statementField;
  /**
   * attribute variable <code>functionBox</code> is the JComboBox to select the function type of the judge statement.
   * @uml.property  name="functionBox"
   */
  private JComboBox functionBox;
  /**
   * variable <code>numberPanel</code> is the panel which contains number field for changing visibility.
   * @uml.property  name="numberPanel"
   */
  private JPanel numberPanel;
  /**
   * variable <code>numberField</code> is the JTextField to display/input the number value.
   * @uml.property  name="numberField"
   */
  private JTextField numberField;
  /**
   * variable <code>datePanel</code> is the panel which contains date fields for changing visibility.
   * @uml.property  name="datePanel"
   */
  private JPanel datePanel;
  /**
   * variable <code>monthField</code> is the JTextField to display/input the month value.
   * @uml.property  name="monthField"
   */
  private JTextField monthField;
  /**
   * variable <code>dayField</code> is the JTextField to display/input the day value.
   * @uml.property  name="dayField"
   */
  private JTextField dayField;
  /**
   * variable <code>stringPanel</code> is the panel which contains string field for changing visibility.
   * @uml.property  name="stringPanel"
   */
  private JPanel stringPanel;
  /**
   * variable <code>stringField</code> is the JTextField to display/input the string value.
   * @uml.property  name="stringField"
   */
  private JTextField stringField;
  /**
   * variable <code>amountPanel</code> is the panel which contains amount fields for changing visibility.
   * @uml.property  name="amountPanel"
   */
  private JPanel amountPanel;
  /**
   * variable <code>currencyField</code> is the JTextField to display/input the currency string.
   * @uml.property  name="currencyField"
   */
  private JTextField currencyField;
  /**
   * variable <code>amountField</code> is the JTextField to display/input the amount value.
   * @uml.property  name="amountField"
   */
  private JTextField amountField;

  /**
   * variable <code>argsCountPanel</code> is the panel which contains the args count field to change visibility.
   * @uml.property  name="argsCountPanel"
   */
  private JPanel argsCountPanel;
  /**
   * variable <code>argsCountLabel</code> is the label before args count field.
   * @uml.property  name="argsCountLabel"
   */
  private JLabel argsCountLabel;
  /**
   * variable <code>argsCountField</code> is the field to display/input the args coount value
   * @uml.property  name="argsCountField"
   */
  private JTextField argsCountField;
  /**
   * variable <code>argsCountButton</code> is the button to refresh args coount and the args panels.
   * @uml.property  name="argsCountButton"
   */
  private JButton argsCountButton;
  /**
   * variable <code>argsCount</code> is the number of args.
   * @uml.property  name="argsCount"
   */
  private int	argsCount = 0;

  /**
   * variable <code>comboItems</code> is the List of PrubaeComboItems.
   *
   */
  private List<PrubaeComboItem>	comboItems;

  // Constructor
  /**
   * Creates a new <code>PrubaeControllerValueStatement</code> instance.
   * creates GUI fields and buttons on the subpanel using createButtons().
   */
  public PrubaeControllerValueStatement() {
    super();
  }

  // Operations
  /**
   * <code>createButtons</code> method
   * creates statement field, function box, number field, date fields,
   * string field, amount fields and args count field and button,
   * on the subpanel, and draw thick border around it.
   */
  protected void createButtons() {
    setStatementField(new JTextField("", 16));
    getStatementField().setEditable(false);
    JPanel	valuePanel = new JPanel();
    valuePanel.setLayout(new BoxLayout(valuePanel, BoxLayout.X_AXIS));
    JLabel	valueLabel = new JLabel("Value : ");
    valuePanel.add(valueLabel);
    valuePanel.add(getStatementField());
    getPanel().add(valuePanel);

    setFunctionBox(new JComboBox());
    getFunctionBox().setMaximumRowCount(4);
    getFunctionBox().setActionCommand("select function");
    getPanel().add(getFunctionBox());

    setNumberPanel(new JPanel());
    setNumberField(new JTextField("", 4));
    JLabel	numberLabel = new JLabel("NUMBER");
    getNumberPanel().setLayout(new BoxLayout(getNumberPanel(),
					     BoxLayout.X_AXIS));
    getNumberPanel().add(numberLabel);
    getNumberPanel().add(getNumberField());
    getNumberPanel().setVisible(false);
    getPanel().add(getNumberPanel());

    setDatePanel(new JPanel());
    setMonthField(new JTextField("", 4));
    setDayField(new JTextField("", 4));
    JLabel	dateLabel = new JLabel("DATE");
    JLabel	date2Label = new JLabel("/");
    getDatePanel().setLayout(new BoxLayout(getDatePanel(),
					   BoxLayout.X_AXIS));
    getDatePanel().add(dateLabel);
    getDatePanel().add(getMonthField());
    getDatePanel().add(date2Label);
    getDatePanel().add(getDayField());
    getDatePanel().setVisible(false);
    getPanel().add(getDatePanel());

    setStringPanel(new JPanel());
    setStringField(new JTextField("", 4));
    JLabel	stringLabel = new JLabel("STRING");
    getStringPanel().setLayout(new BoxLayout(getStringPanel(),
					     BoxLayout.X_AXIS));
    getStringPanel().add(stringLabel);
    getStringPanel().add(getStringField());
    getStringPanel().setVisible(false);
    getPanel().add(getStringPanel());

    setAmountPanel(new JPanel());
    setCurrencyField(new JTextField("", 4));
    setAmountField(new JTextField("", 4));
    JLabel	amountLabel = new JLabel("AMOUNT");
    getAmountPanel().setLayout(new BoxLayout(getAmountPanel(),
					     BoxLayout.X_AXIS));
    getAmountPanel().add(amountLabel);
    getAmountPanel().add(getCurrencyField());
    getAmountPanel().add(getAmountField());
    getAmountPanel().setVisible(false);
    getPanel().add(getAmountPanel());

    setArgsCountPanel(new JPanel());
    setArgsCountLabel(new JLabel("COUNT"));
    setArgsCountField(new JTextField("", 4));
    setArgsCountButton(new JButton("INPUT"));
    getArgsCountButton().setActionCommand("set args");
    getArgsCountButton().addActionListener(this);
    getArgsCountPanel().setLayout(new BoxLayout(getArgsCountPanel(),
						BoxLayout.X_AXIS));
    getArgsCountPanel().add(argsCountLabel);
    getArgsCountPanel().add(getArgsCountField());
    getArgsCountPanel().add(argsCountButton);
    getArgsCountPanel().setVisible(false);
    getPanel().add(getArgsCountPanel());

    getPanel().setBorder(BorderFactory.createEmptyBorder(5, 20, 0, 0));
  }

  /**
   * <code>initialize</code> method	do nothing
   *
   */
  public void initialize() {
  }

  /**
   * <code>update</code> method
   * updates the model with the input values on the panel.
   * updates number value from the input in number field.
   * updates date values(month and day) from the input in date fields.
   * updates string value from the input in string field.
   * updates currency/amount value from the input in currency/amount fields.
   * this also updates the statements of the model with updateStatement(),
   * and updates comment string using super.update().
   */
  public void update() {
    PrubaeModelValueStatement	model = (PrubaeModelValueStatement)getModel();
    if (getNumberPanel().isVisible() && getNumberField().isVisible()) {
      double	value = (double)0;
      try {
	value = Double.parseDouble(getNumberField().getText());
      }
      catch (NumberFormatException e) {
	value = 0;
      }
      model.getValues().set(0, new Double(value));
    }
    if (getDatePanel().isVisible()
	&& getMonthField().isVisible() && getDayField().isVisible()) {
      short	month = (short)0;
      try {
	month = Short.parseShort(getMonthField().getText());
      }
      catch (NumberFormatException e) {
	month = 0;
      }
      short	day = (short)0;
      try {
	day = Short.parseShort(getDayField().getText());
      }
      catch (NumberFormatException e) {
	day = 0;
      }
      model.getValues().set(0, new Short(month));
      model.getValues().set(1, new Short(day));
    }
    if (getStringPanel().isVisible() && getStringField().isVisible()) {
      String	string = getStringField().getText();
      model.getValues().set(0, string);
    }
    if (getAmountPanel().isVisible()
	&& getCurrencyField().isVisible() && getAmountField().isVisible()) {
      String	currency = getCurrencyField().getText();
      double	value = (double)0;
      try {
	value = Double.parseDouble(getAmountField().getText());
      }
      catch (NumberFormatException e) {
	value = 0;
      }
      model.getValues().set(0, currency);
      model.getValues().set(1, new Double(value));
    }
    model.updateStatement();
    super.update();
  }

  /**
   * <code>createPanel</code> method
   * set functionBox items from the list valueList, varList, constList
   * of the editor.
   *
   */
  public void createPanel() {
    super.createPanel();
    createButtons();

    PrubaeModelValueStatement	model = (PrubaeModelValueStatement)getModel();
    System.err.println("prubae.varList nthCdr 35 = "
		       + Lisp.nthCdr(35, getEditor().getVarList()));
    LList
      list = Lisp.append(Lisp.append(Lisp.append(getEditor().getValueList(),
						 getEditor().getVarList()),
				     getEditor().getConstList()),
			 getEditor().getSettablevarList());
    if (model.getValues() != null) {
      int	numArgs = model.getValues().size();
      LList	valueList = null;
      if ((model.getRegist()
	   & ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION)
	  == ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION) {
	LList	top = Lisp.assoc(new IntNum(model.getRegist()), list);
	LList	autoList = Lisp.member(top, list);
	for ( ; !Lisp.isNil(autoList); autoList = (LList)Lisp.cdr(autoList)) {
	  IntNum	registNum = (IntNum)Lisp.car(Lisp.car(autoList));
	  int	regist = registNum.intValue();
	  if ((regist & ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION)
	      != ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION) {
	    break;
	  }
	  if (Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.car(autoList))))
	      .equals(model.getStatement())) {
	    valueList = (LList)Lisp.car(autoList);
	    break;
	  }
	}
      }
      else if (model.getRegist()
	       == (short) ProrateRulebaseElement.EXTERNAL_FUNCTION) {
	LList	top = Lisp.assoc(new IntNum(model.getRegist()), list);
	LList	extfList = Lisp.member(top, list);
	for (int i = 0; !Lisp.isNil(extfList);
	     extfList = (LList)Lisp.cdr(extfList), i++) {
	  IntNum	registNum = (IntNum)Lisp.car(Lisp.car(extfList));
	  int	regist = registNum.intValue();
	  if (regist != ProrateRulebaseElement.EXTERNAL_FUNCTION) {
	    break;
	  }
	  String	item = model.getName();
	  int	index = item.indexOf("(");
	  if (index >= 0) {
	    item = item.substring(0, index);
	  }
	  String	itemStr =
	    (Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.car(extfList))))).toString();
	  if (itemStr.equals(item)) {
	    valueList = (LList)Lisp.car(extfList);
	    break;
	  }
	}
      }
      else {
	valueList = Lisp.assoc(new IntNum(model.getRegist()), list);
      }
      if (valueList != null) {
	Object	listArgs
	  = Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(valueList)))));
	Object[]	typeArgs = new Object[numArgs];
	for (int i = 0; i < numArgs; i++) {
	  typeArgs[i] = Lisp.car(listArgs);
	  listArgs = Lisp.cdr(listArgs);
	  //System.err.println("PrubaeControllerValueStatement.createPanel(): typeArgs[" + i + "] = " + typeArgs[i]);
	}
	for (int i = 0; i < model.getValues().size(); i++) {
	  if (model.getValues().get(i) instanceof PrubaeModel) {
	    PrubaeModelValue
	      child = (PrubaeModelValue)model.getValues().get(i);
	    /* ;;; 2009.07.10 add start */
	    if (child instanceof PrubaeModelValueStatement) {
	      if (model.getRegist() == ProrateRulebaseElement.MULTIVAL_CODE
		  && (model.getRequiredType()
		      != null)) {
		//System.err.println("PrubaeControllerValueStatement.createPanel(): model = " + model + ", requiredType = " + model.getType());
		child.setRequiredType(model.getRequiredType());
	      }
	      else if (model.getRegist()
		       == ProrateRulebaseElement.MULTIVAL_CODE
		       && model.getType() != null) {
		//System.err.println("PrubaeControllerValueStatement.createPanel(): model = " + model + ", requiredType = " + model.getType());
		child.setRequiredType(model.getType());
	      }
	      /* ;;; 2009.07.15 add start */
	      else if (model.getRegist()
		       == ProrateRulebaseElement.INTERVAL_CODE
		       && (model.getRequiredType() != null)) {
		//System.err.println("PrubaeControllerValueStatement.createPanel(): model = " + model + ", requiredType = " + model.getType());
		child.setRequiredType(model.getRequiredType());
	      }
	      else if (model.getRegist()
		       == ProrateRulebaseElement.INTERVAL_CODE
		       && model.getType() != null) {
		//System.err.println("PrubaeControllerValueStatement.createPanel(): model = " + model + ", requiredType = " + model.getType());
		child.setRequiredType(model.getType());
	      }
	      /* ;;; 2009.07.15 add end */
	      else if (i > 0
		       && typeArgs[i] != null && typeArgs[i].equals("first")) {
		child
		  .setRequiredType(((PrubaeModelValue)model.getValues().get(0))
				   .getType());
	      }
	      else if (typeArgs[i] != null) {
		child.setRequiredType(typeArgs[i]);
	      }
	    }
	    /* ;;; 2009.07.10 add end */
	    if (child.getController() == null) {
	      child.openUI();
	    }
	    //System.err.println("PrubaeControllerValueStatement.createPanel() : child = " + child + ", childCntl = " + child.getController());
	    child.getController().createPanel();
	  }
	}
      }
    }

    if (getFunctionBox().getItemCount() > 0) {
      return;	/* initialized already */
    }
    setComboItems(new Vector());
    PrubaeComboItem item = new PrubaeComboItem("", "");
    getComboItems().add(item);
    getFunctionBox().addItem(item);
    Object requiredType = model.getRequiredType();
    if (requiredType == null) {
      requiredType = model.getType();
    }
    for ( ; !Lisp.isNil(list); list = (LList)Lisp.cdr(list)) {
      LList	valueList = (LList)Lisp.car(list);
      String	dispStr = (Lisp.car(Lisp.cdr(valueList))).toString();
      String	itemStr = (Lisp.car(Lisp.cdr(Lisp.cdr(valueList)))).toString();
      Object	type = Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.cdr(valueList))));
      if (matchType(requiredType, type)) {
	item = new PrubaeComboItem(dispStr, itemStr);
	getComboItems().add(item);
	getFunctionBox().addItem(item);
      }
    }
    getFunctionBox().addActionListener(this);
    /* ;;; 2006.09.07 */
    if (getPanel() != null && getFunctionBox() != null
	&& !getPanel().isAncestorOf(getFunctionBox())) {
      getPanel().add(getFunctionBox());
    }
  }

  /**
   * <code>destroyPanel</code> method
   *
   */
  public void destroyPanel() {
    setStatementField(null);
    setFunctionBox(null);
    setNumberPanel(null);
    setNumberField(null);
    setDatePanel(null);
    setMonthField(null);
    setDayField(null);
    setStringPanel(null);
    setStringField(null);
    setAmountPanel(null);
    setCurrencyField(null);
    setAmountField(null);
    setArgsCountPanel(null);
    setArgsCountLabel(null);
    setArgsCountField(null);
    setArgsCountButton(null);
    super.destroyPanel();
  }

  /**
   * <code>removePanel</code> method
   * remove the subpanel related this controller from the parent panel.
   */
  public void removePanel() {
    PrubaeModel	parent = getModel().getParent();
    if (parent instanceof PrubaeModelValue
	&& !(parent instanceof PrubaeModelValueStatement)
	&& (parent.getParent() instanceof PrubaeModelValueStatement
	    || parent.getParent() instanceof PrubaeModelDoStatement
	    || parent.getParent() instanceof PrubaeModelJudgeStatement)) {
      parent = parent.getParent();
    }
    PrubaeController
      parentCtl = parent.getController();
    if (parentCtl != null && parentCtl.getPanel() != null
    		&& parentCtl.getPanel().isAncestorOf(getPanel())) {
      parentCtl.getPanel().remove(getPanel());
    }
  }
  
  /**
   * <code>matchType</code> method	check type matching.
   *
   * @param requiredType an <code>Object</code> value
   * @param valueType an <code>Object</code> value
   * @return a <code>boolean</code> value
   */
  protected boolean	matchType(Object requiredType, Object valueType) {
   if (requiredType instanceof Pair) {
      LList	list = (LList)requiredType;
      for ( ; !Lisp.isNil(list); list = (LList)Lisp.cdr(list)) {
	String	type = (String)Lisp.car(list);
	if (valueType instanceof String
	    && ((String)valueType).startsWith("set")
	    && matchTypeRequired(type, ((String)valueType).substring(3))) {
	  // System.err.println("matchType(" + requiredType + ", " + valueType + ") = true");
	  return true;
	}
	if (!matchTypeRequired(type, valueType)) {
	  // System.err.println("matchType(" + requiredType + ", " + valueType + ") = false");
	  return false;
	}
      }
      // System.err.println("matchType(" + requiredType + ", " + valueType + ") = true");
      return true;
    }
    if (valueType instanceof String
	&& ((String)valueType).startsWith("set")
	&& requiredType instanceof String
	&& matchTypeRequired((String)requiredType,
			     ((String)valueType).substring(3))) {
      // System.err.println("matchType(" + requiredType + ", " + valueType + ") = true");
      return true;
    }
    if (requiredType instanceof String) {
      // System.err.println("matchType(" + requiredType + ", " + valueType + ") = " + matchTypeRequired((String)requiredType, valueType));
      return matchTypeRequired((String)requiredType, valueType);
    }
    // System.err.println("matchType(" + requiredType + ", " + valueType + ") = true");
    return true;
  }

  /**
   * <code>matchTypeRequired</code> method
   *	check type matching for each required type.
   *
   * @param requiredType a <code>String</code> value
   * @param valueType an <code>Object</code> value
   * @return a <code>boolean</code> value
   */
  protected boolean matchTypeRequired(String requiredType, Object valueType) {
    if (requiredType.equals("any")) {
      return true;
    }
    if (valueType instanceof Pair) {
      LList	list = (LList)valueType;
      for ( ; !Lisp.isNil(list); list = (LList)Lisp.cdr(list)) {
	String	type = (String)Lisp.car(list);
	if (type.equals("any")) {
	  return true;
	}
	if (type.equals(requiredType)) {
	  return true;
	}
      }
      return false;
    }
    if (requiredType instanceof String) {
      if (valueType.equals("any")) {
	return true;
      }
      if (valueType.equals(requiredType)) {
	return true;
      }
      return false;
    }
    return false;
  }

  /**
   * <code>select</code> method
   * handles the selection in the function box.
   * first, get regist code of selected item
   * using valueList, varList and constList of the editor.
   * if the regist code is changed by selection,
   * delete view and controller and close GUI items
   * by sending closeUI() method to the models in the values list.
   * and clear the values list.
   * set the regist code to the model.
   * handle const set fields using constSet().
   * handle GUI panels of values using arguments().
   * update statement of the model by sending updateStatement() to the model.
   * @param selected a <code>String</code> value is the selected item string.
   */
  protected void select(String selected) {
    System.err.println("PrubaeControllerValueStatement().select([" + selected + "])");
    PrubaeModelValueStatement	model = (PrubaeModelValueStatement)getModel();
    short	oldRegist = model.getRegist();	/* ;;; add 2009.07.16 */
    System.err.println("PrubaeControllerValueStatement().select() : before updateStatement() 1");
    model.updateStatement();
    System.err.println("PrubaeControllerValueStatement().select() : after updateStatement() 1");
    LList
      list = Lisp.append(Lisp.append(getEditor().getValueList(),
				     getEditor().getVarList()),
			 getEditor().getConstList());
    for ( ; !Lisp.isNil(list); list = (LList)Lisp.cdr(list)) {
      LList	valueList = (LList)Lisp.car(list);
      String	itemStr = (Lisp.car(Lisp.cdr(Lisp.cdr(valueList)))).toString();
      if (selected.equals(itemStr)) {
	System.err.println("PrubaeControllerValueStatement().select() : selected = " + selected + ", valueList = " + valueList);
	short	regist = (short)((IntNum)Lisp.car(valueList)).intValue();
	Object	type = (Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.cdr(valueList)))));
	System.err.println("PrubaeControllerValueStatement().select() : regist = " + regist + ", model.regist = " + model.getRegist());
	if (model.getRegist() != regist
	    || (((regist & ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION)
		 == ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION)
		&& !selected.equals(model.getStatement()))
	    || ((regist == ProrateRulebaseElement.EXTERNAL_FUNCTION)
		&& !selected.equals(model.getName()))) {
	  System.err.println("PrubaeControllerValueStatement().select() change : regist = " + regist + ", model.regist = " + model.getRegist());
	  if (model.getValues() != null) {
	    for (int i = 0; i < model.getValues().size(); i++) {
	      if (model.getValues().get(i) instanceof PrubaeModel) {
		((PrubaeModel)model.getValues().get(i)).closeUI();
	      }
	    }
	    model.getValues().clear();
	  }
	  setArgsCount(0);
	  System.err.println("PrubaeControllerValueStatement().select() : before change regist = " + regist + ", model.regist = " + model.getRegist() + ", model.name = " + model.getName());
	  model.setRegist(regist);
	  if ((regist & ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION)
	      == ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION) {
	    model.setStatement(selected);
	  }
	  else if (regist == ProrateRulebaseElement.EXTERNAL_FUNCTION) {
	    model.setName(selected);
	  }
	  System.err.println("PrubaeControllerValueStatement().select() : after change regist = " + regist + ", model.regist = " + model.getRegist() + ", model.statement = " + model.getStatement() + ", model.name = " + model.getName());
	  if (type.equals("any")) {
	    type = null;
	  }
	  else if (type instanceof String && ((String)type).length() > 3
		   && ((String)type).substring(0, 3).equals("set")) {
	    type = ((String)type).substring(3);
	  }
	  if (type != null) {
	    model.setType(type);
	  }
	  PrubaeModel	parent = model.getParent();
	  if (type != null
	      && parent instanceof PrubaeModelValue
	      && ((PrubaeModelValue)parent).getType() == null
	      && ((PrubaeModelValue)parent).getValues().get(0) == model) {
	    ((PrubaeModelValue)parent).setType(type);
	  }
	  /* ;;; add 2009.07.16 start */
	  if (parent instanceof PrubaeModelTable) {
	    if (model == ((PrubaeModelTable)parent).getVarRow()
		&& model.getRegist() != oldRegist) {
	      ((PrubaeControllerTable)parent.getController()).resetRows();
	    }
	    if (model == ((PrubaeModelTable)parent).getVarCol()
		&& model.getRegist() != oldRegist) {
	      ((PrubaeControllerTable)parent.getController()).resetCols();
	    }
	  }
	  /* ;;; add 2009.07.16 end */
	  /* constset */
	  constset(type);
	  /* ;;; 2006.09.10 */
	  System.err.println("PrubaeControllerValueStatement().select() : before updateStatement() 2");
	  model.updateStatement();
	  System.err.println("PrubaeControllerValueStatement().select() : after updateStatement() 2");
	  /* */
	  /* ;;; 2006.09.05 */
	  getEditor().refresh();
	  /* ;;; 2006.09.10
	  model.updateStatement();
	  */
	  /* ;;; 2006.09.05
	  if (parent instanceof PrubaeModelValue) {
	  }
	  else if (model.getRegist() == ProrateRulebaseElement.MULTIVAL_CODE) {
	  }
	  else {
	    parent.getController().close();
	    parent.openUI();
	    parent.getController().open();
	  }
	  */
	}
	break;	/* ;;; 2006.09.10 */
      }
    }
  }

  /**
   * <code>constset</code> method
   * adds constant values to the values list of the model.
   * @param valueList a <code>LList</code> value
   * is the associated list element matched to current regist code
   * in the valueList, varList or constList of the editor.
   */
  private void constset(Object type) {
    PrubaeModelValueStatement	model = (PrubaeModelValueStatement)getModel();
    getNumberPanel().setVisible(false);
    getDatePanel().setVisible(false);
    getStringPanel().setVisible(false);
    getAmountPanel().setVisible(false);
    if (type == null || type instanceof LList) {
      return;
    }
    if (type.equals("setnumber")) {
      /* set number statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add(new Double(0));
      getNumberPanel().setVisible(true);
    }
    else if (type.equals("setdate")) {
      /* set date statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add(new Short((short)0));
      model.getValues().add(new Short((short)0));
      getDatePanel().setVisible(true);
    }
    else if (type.equals("setstring")) {
      /* set date statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add("");
      getStringPanel().setVisible(true);
    }
    else if (type.equals("setamount")) {
      /* set amount statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add("");
      model.getValues().add(new Double(0));
      getAmountPanel().setVisible(true);
    }
    else if (type.equals("number")) {
      /* set number statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add(new Double(0));
      getNumberPanel().setVisible(true);
    }
    else if (type.equals("date")) {
      /* set date statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add(new Short((short)0));
      model.getValues().add(new Short((short)0));
      getDatePanel().setVisible(true);
    }
    else if (type.equals("string")) {
      /* set date statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add("");
      getStringPanel().setVisible(true);
    }
    else if (type.equals("amount")) {
      /* set amount statement */
      if (model.getValues() == null) {
	model.setValues(new Vector());
      }
      model.getValues().add("");
      model.getValues().add(new Double(0));
      getAmountPanel().setVisible(true);
    }
  }

  /**
   * <code>arguments</code> method
   * get and save old args count and set the new args count
   * from the valueList association list.
   * set the args count field visible if arguments are required,
   * set it invisible otherwise.
   * if new args count > old one, increase arguments using addArguments(),
   * if new args count < old one, decrease arguments using removeArguments().
   * @param valueList a <code>LList</code> value
   * is the associated list element matched to current regist code
   * in the valueList, varList or constList of the editor.
   */
  /* ;;; 2009.07.10 not used
  int arguments(LList valueList) {
    Object
      isArgs = Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(valueList)))));
    int	numArgs = 0;
    int	oldArgsCount = getArgsCount();
    if (isArgs == null) {
      numArgs = 0;
    }
    else if (!(isArgs instanceof IntNum)) {
      numArgs = 0;
    }
    else {
      numArgs = ((IntNum)isArgs).intValue();
    }
    if (numArgs < 0) {
      System.err.println("numArgs -1 :"
			 + Lisp.car(Lisp.cdr(Lisp.cdr(valueList))));
      getArgsCountPanel().setVisible(true);
      numArgs = getArgsCount();
    }
    else {
      setArgsCount(0);
      getArgsCountPanel().setVisible(false);
    }
    if (numArgs > oldArgsCount) {
      addArguments(valueList, oldArgsCount, numArgs);
    }
    else if (numArgs < oldArgsCount) {
      removeArguments(valueList, numArgs, oldArgsCount);
    }
    return numArgs;
  }
  */

  /**
   * <code>addArguments</code> method
   * creates new value statement models for increased arguments,
   * creates view and controller and opens the GUI by sending openUI to them,
   * and add them to the values list of this model.
   * and add their subpanels to subpanel related to this controller.
   * @param valueList a <code>LList</code> value
   * is the associated list element matched to current regist code
   * in the valueList, varList or constList of the editor.
   * @param minArgs an <code>int</code> value
   * is the number of arguments before increment.
   * @param numArgs an <code>int</code> value
   * is the number of arguments to increase to.
   */
  private void addArguments(LList valueList, int minArgs, int numArgs) {
    PrubaeModelValueStatement	model = (PrubaeModelValueStatement)getModel();
    Object
      listArgs = Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(valueList)))));
    Object[]	typeArgs = new Object[numArgs];
    for (int i = minArgs; i < numArgs; i++) {
      if (!Lisp.isNil(listArgs)) {
	typeArgs[i] = Lisp.car(listArgs);
      }
      else {
	typeArgs[i] = null;
      }
      if (!Lisp.isNil(Lisp.cdr(listArgs))) {
	listArgs = Lisp.cdr(listArgs);
      }
    }
    if (model.getValues() == null) {
      model.setValues(new Vector());
    }
    for (int i = minArgs; i < numArgs; i++) {
      String	type;
      if (typeArgs[i] == null) {
	type = (String)"any";
      }
      else if (typeArgs[i] instanceof Pair) {
	type = (String)Lisp.car(typeArgs[i]);
      }
      else if (Lisp.isNil(typeArgs[i])) {
	type = null;
      }
      else {
	type = (String)typeArgs[i];
      }
      //System.err.println("Arguments(" + i + ") = " + type);
      if (type.equals("setnumber")) {
	PrubaeModelValue	child = new PrubaeModelValueStatement();
	child.setRegist((short)ProrateRulebaseElement.NUMBER_CODE);
	child.setComment("");
	child.setStatement("" + 0);
	child.getValues().add(new Double((double)0));
	child.setType(typeArgs[i]);
	child.initialize(getEditor(), model, model.getValues());
	child.openUI();
	model.getValues().add(child);
      }
      else if (type.equals("setdate")) {
	PrubaeModelValue	child = new PrubaeModelValueStatement();
	child.setRegist((short)ProrateRulebaseElement.DATE_CODE);
	child.setComment("");
	child.setStatement("" + 0 + "/" + 0);
	child.getValues().add(new Short((short)0));
	child.getValues().add(new Short((short)0));
	child.setType(typeArgs[i]);
	child.initialize(getEditor(), model, model.getValues());
	child.openUI();
	model.getValues().add(child);
      }
      else if (type.equals("setstring")) {
	PrubaeModelValue	child = new PrubaeModelValueStatement();
	child.setRegist((short)ProrateRulebaseElement.STRING_CODE);
	child.setComment("");
	child.setStatement("\"\"");
	child.getValues().add("");
	child.setType(typeArgs[i]);
	child.initialize(getEditor(), model, model.getValues());
	child.openUI();
	model.getValues().add(child);
      }
      else if (type.equals("setamount")) {
	PrubaeModelValue	child = new PrubaeModelValueStatement();
	child.setRegist((short)ProrateRulebaseElement.AMOUNT_CODE);
	child.setComment("");
	child.getValues().add("");
	child.getValues().add(new Double((double)0));
	child.setType(typeArgs[i]);
	child.initialize(getEditor(), model, model.getValues());
	child.openUI();
	model.getValues().add(child);
      }
      else {
	PrubaeModelValue	child = new PrubaeModelValueStatement();
	child.setComment("");
	child.setStatement("");
	child.setType(typeArgs[i]);
	child.initialize(getEditor(), model, model.getValues());
	/* ;;; 2009.07.10 add start */
	if (model.getRegist() == ProrateRulebaseElement.MULTIVAL_CODE
	    && model.getType() != null) {
	  //System.err.println("PrubaeControllerValueStatement.addArguments(): model = " + model + ", type = " + model.getType());
	  child.setRequiredType(model
				.getType());
	}
	else if (model.getRegist() == ProrateRulebaseElement.MULTIVAL_CODE
		 && (model.getRequiredType()
		     != null)) {
	  //System.err.println("PrubaeControllerValueStatement.addArguments(): model = " + model + ", requiredType = " + model.getRequiredType());
	  child.setRequiredType(model
				.getRequiredType());
	}
	/* ;;; 2009.07.15 add start */
	else if (model.getRegist() == ProrateRulebaseElement.INTERVAL_CODE
		 && model.getType() != null) {
	  //System.err.println("PrubaeControllerValueStatement.addArguments(): model = " + model + ", type = " + model.getType());
	  child.setRequiredType(model
				.getType());
	}
	else if (model.getRegist() == ProrateRulebaseElement.INTERVAL_CODE
		 && (model.getRequiredType()
		     != null)) {
	  //System.err.println("PrubaeControllerValueStatement.addArguments(): model = " + model + ", requiredType = " + model.getRequiredType());
	  child.setRequiredType(model
				.getRequiredType());
	}
	/* ;;; 2009.07.15 add end */
	else if (typeArgs[i] != null) {
	  child.setRequiredType(typeArgs[i]);
	}
	/* ;;; 2009.07.10 add end */
	child.openUI();
	/* constset */
	((PrubaeControllerValueStatement)child.getController()).constset(type);
	model.getValues().add(child);
      }
    }
  }

  /**
   * <code>updateArgsCount</code> method
   * first send updateStatement() to the model to update all values.
   * if the variable number arguments are required
   * according to the valueList association list,
   * from valueList, varList or constList from the editor,
   * get and save old args count and set the new args count from the input.
   * if new args count > old one, increase arguments using addArguments(),
   * if new args count < old one, decrease arguments using removeArguments().
   */
  private void updateArgsCount() {
    PrubaeModelValueStatement	model = (PrubaeModelValueStatement)getModel();
    model.updateStatement();
    LList
      list = Lisp.append(Lisp.append(getEditor().getValueList(),
				     getEditor().getVarList()),
			 getEditor().getConstList());
    LList	valueList = null;
    if ((model.getRegist()
	 & ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION)
	== ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION) {
      LList	top = Lisp.assoc(new IntNum(model.getRegist()), list);
      LList	autoList = Lisp.member(top, list);
      for ( ; !Lisp.isNil(autoList); autoList = (LList)Lisp.cdr(autoList)) {
	IntNum	registNum = (IntNum)Lisp.car(Lisp.car(autoList));
	int	regist = registNum.intValue();
	if ((regist & ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION)
	    != ProrateRulebaseElement.AUTO_VARIABLE_SPECIFICATION) {
	  break;
	}
	if (Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.car(autoList))))
	    .equals(model.getStatement())) {
	  valueList = (LList)Lisp.car(autoList);
	  break;
	}
      }
    }
    else if (model.getRegist()
	     == (short) ProrateRulebaseElement.EXTERNAL_FUNCTION) {
      LList	top = Lisp.assoc(new IntNum(model.getRegist()), list);
      LList	extfList = Lisp.member(top, list);
      for (int i = 0; !Lisp.isNil(extfList);
	   extfList = (LList)Lisp.cdr(extfList), i++) {
	IntNum	registNum = (IntNum)Lisp.car(Lisp.car(extfList));
	int	regist = registNum.intValue();
	if (regist != ProrateRulebaseElement.EXTERNAL_FUNCTION) {
	  break;
	}
	String	item = model.getName();
	int	index = item.indexOf("(");
	if (index >= 0) {
	  item = item.substring(0, index);
	}
	String	itemStr =
	  (Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.car(extfList))))).toString();
	if (itemStr.equals(item)) {
	  valueList = (LList)Lisp.car(extfList);
	  break;
	}
      }
    }
    else {
      valueList = Lisp.assoc(new IntNum(model.getRegist()), list);
    }
    if (valueList == null) {
      return;
    }
    Object
      isArgs = Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(valueList)))));
    if (isArgs == null) {
      return;
    }
    else if (!(isArgs instanceof IntNum)) {
      return;
    }
    int	numArgs = ((IntNum)isArgs).intValue();
    if (numArgs >= 0) {
      return;
    }
    try {
      numArgs = Integer.parseInt(getArgsCountField().getText());
    }
    catch (NumberFormatException e) {
      return;
    }
    int	oldArgsCount = getArgsCount();
    setArgsCount(numArgs);
    if (numArgs > oldArgsCount) {
      addArguments(valueList, oldArgsCount, numArgs);
      /* ;;; 2006.09.05 */
      getEditor().refresh();
    }
    else if (numArgs < oldArgsCount) {
      removeArguments(valueList, numArgs, oldArgsCount);
      /* ;;; 2006.09.05 */
      getEditor().refresh();
    }
  }

  /**
   * <code>removeArguments</code> method
   * for all decreased values,
   * delete all view and controller and delete GUI items of them,
   * and remove the value from the values list of this model.
   * @param minArgs an <code>int</code> value
   * is the number of arguments to decrease to.
   * @param numArgs an <code>int</code> value
   * is the number of arguments before decrement.
   */
  private void removeArguments(LList valueList, int minArgs, int numArgs) {
    PrubaeModelValueStatement	model = (PrubaeModelValueStatement)getModel();
    for (int i = numArgs-1; i >= minArgs; i--) {
      PrubaeModelValue	ret = (PrubaeModelValue)model.getValues().get(i);
      getPanel().remove(ret.getController().getPanel());
      ret.closeUI();
      model.getValues().remove(ret);
    }
  }
  
  /**
   * <code>activateSelection</code> method
   *	activates JComboBox selection action.
   *
   */
  public void	activateSelection() {
    getFunctionBox().setActionCommand("select function");
  }
  
  /**
   * <code>inactivateSelection</code> method
   *	inactivates JComboBox selection action.
   *
   */
  public void	inactivateSelection() {
    getFunctionBox().setActionCommand("nop");
  }

  /**
   * <code>actionPerformed</code> method
   * handles the action events returned from GUI items on the panel
   * call select() when the action command is "select function",
   * call updateArgsCount() when the action command is "set args".
   * @param e an <code>ActionEvent</code> value is the action event.
   */
  public void actionPerformed(ActionEvent e) {
    if (e.getActionCommand().equals("select function")) {
      PrubaeComboItem
	selected =
	(PrubaeComboItem)((JComboBox)e.getSource()).getSelectedItem();
      select(selected.getItemString());
    }
    else if (e.getActionCommand().equals("set args")) {
      updateArgsCount();
    }
    else if (e.getActionCommand().equals("nop")) {
    }
    else {
    }
  }

  /**
   * <code>createPopupMenu</code> method create popup menu for this node.
   *
   */
  void createPopupMenu() {
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeControllerValueStatement	clone = null;
    clone = (PrubaeControllerValueStatement)super.clone();
    if (getStatementField() != null) {
      clone.setStatementField(new JTextField());
    }
    if (getFunctionBox() != null) {
      clone.setFunctionBox(new JComboBox());
      clone.getFunctionBox()
	.setSelectedItem(getFunctionBox().getSelectedItem());
    }
    if (getNumberPanel() != null) {
      clone.setNumberPanel(new JPanel());
    }
    if (getNumberField() != null) {
      clone.setNumberField(new JTextField());
    }
    if (getDatePanel() != null) {
      clone.setDatePanel(new JPanel());
    }
    if (getMonthField() != null) {
      clone.setMonthField(new JTextField());
    }
    if (getDayField() != null) {
      clone.setDayField(new JTextField());
    }
    if (getStringPanel() != null) {
      clone.setStringPanel(new JPanel());
    }
    if (getStringField() != null) {
      clone.setStringField(new JTextField());
    }
    if (getAmountPanel() != null) {
      clone.setAmountPanel(new JPanel());
    }
    if (getCurrencyField() != null) {
      clone.setCurrencyField(new JTextField());
    }
    if (getAmountField() != null) {
      clone.setAmountField(new JTextField());
    }
    if (getArgsCountPanel() != null) {
      clone.setArgsCountPanel(new JPanel());
    }
    if (getArgsCountField() != null) {
      clone.setArgsCountField(new JTextField());
    }
    return clone;
  }

  // Setters and Getters  
  /**
   * <code>setStatementField</code> method
   * sets theStatementField to statementField attribute 
   * @param theStatementField a <code>JTextField</code> value
   * 
   * @uml.property name="statementField"
   */
  public void setStatementField(JTextField theStatementField) {
    statementField = theStatementField;
  }

  /**
   * <code>getStatementField</code> method
   * returns statementField attribute value
   * @return a <code>JTextField</code> value
   * 
   * @uml.property name="statementField"
   */
  public JTextField getStatementField() {
    return statementField;
  }

  /**
   * <code>setFunctionBox</code> method here.
   * sets theFunctionBox to functionBox attribute
   * @param theFunctionBox a <code>JComboBox</code> value
   * 
   * @uml.property name="functionBox"
   */
  public void setFunctionBox(JComboBox theFunctionBox) {
    functionBox = theFunctionBox;
  }

  /**
   * <code>getFunctionBox</code> method
   * returns functionBox attribute
   * @return a <code>JComboBox</code> value
   * 
   * @uml.property name="functionBox"
   */
  public JComboBox getFunctionBox() {
    return functionBox;
  }

  /**
   * 
   * @uml.property name="numberPanel"
   */
  public void setNumberPanel(JPanel theNumberPanel) {
    numberPanel = theNumberPanel;
  }

  /**
   * 
   * @uml.property name="numberPanel"
   */
  public JPanel getNumberPanel() {
    return numberPanel;
  }

  /**
   * 
   * @uml.property name="numberField"
   */
  public void setNumberField(JTextField theNumberField) {
    numberField = theNumberField;
  }

  /**
   * 
   * @uml.property name="numberField"
   */
  public JTextField getNumberField() {
    return numberField;
  }

  /**
   * 
   * @uml.property name="datePanel"
   */
  public void setDatePanel(JPanel theDatePanel) {
    datePanel = theDatePanel;
  }

  /**
   * 
   * @uml.property name="datePanel"
   */
  public JPanel getDatePanel() {
    return datePanel;
  }

  /**
   * 
   * @uml.property name="monthField"
   */
  public void setMonthField(JTextField theMonthField) {
    monthField = theMonthField;
  }

  /**
   * 
   * @uml.property name="monthField"
   */
  public JTextField getMonthField() {
    return monthField;
  }

  /**
   * 
   * @uml.property name="dayField"
   */
  public void setDayField(JTextField theDayField) {
    dayField = theDayField;
  }

  /**
   * 
   * @uml.property name="dayField"
   */
  public JTextField getDayField() {
    return dayField;
  }

  /**
   * 
   * @uml.property name="stringPanel"
   */
  public void setStringPanel(JPanel theStringPanel) {
    stringPanel = theStringPanel;
  }

  /**
   * 
   * @uml.property name="stringPanel"
   */
  public JPanel getStringPanel() {
    return stringPanel;
  }

  /**
   * 
   * @uml.property name="stringField"
   */
  public void setStringField(JTextField theStringField) {
    stringField = theStringField;
  }

  /**
   * 
   * @uml.property name="stringField"
   */
  public JTextField getStringField() {
    return stringField;
  }

  /**
   * 
   * @uml.property name="amountPanel"
   */
  public void setAmountPanel(JPanel theAmountPanel) {
    amountPanel = theAmountPanel;
  }

  /**
   * 
   * @uml.property name="amountPanel"
   */
  public JPanel getAmountPanel() {
    return amountPanel;
  }

  /**
   * 
   * @uml.property name="currencyField"
   */
  public void setCurrencyField(JTextField theCurrencyField) {
    currencyField = theCurrencyField;
  }

  /**
   * 
   * @uml.property name="currencyField"
   */
  public JTextField getCurrencyField() {
    return currencyField;
  }

  /**
   * 
   * @uml.property name="amountField"
   */
  public void setAmountField(JTextField theAmountField) {
    amountField = theAmountField;
  }

  /**
   * 
   * @uml.property name="amountField"
   */
  public JTextField getAmountField() {
    return amountField;
  }

  /**
   * 
   * @uml.property name="argsCountPanel"
   */
  public void setArgsCountPanel(JPanel theArgsCountPanel) {
    argsCountPanel = theArgsCountPanel;
  }

  /**
   * 
   * @uml.property name="argsCountPanel"
   */
  public JPanel getArgsCountPanel() {
    return argsCountPanel;
  }

  /**
   * 
   * @uml.property name="argsCountLabel"
   */
  public void setArgsCountLabel(JLabel theArgsCountLabel) {
    argsCountLabel = theArgsCountLabel;
  }

  /**
   * 
   * @uml.property name="argsCountLabel"
   */
  public JLabel getArgsCountLabel() {
    return argsCountLabel;
  }

  /**
   * 
   * @uml.property name="argsCountField"
   */
  public void setArgsCountField(JTextField theArgsCountField) {
    argsCountField = theArgsCountField;
  }

  /**
   * 
   * @uml.property name="argsCountField"
   */
  public JTextField getArgsCountField() {
    return argsCountField;
  }

  /**
   * 
   * @uml.property name="argsCountButton"
   */
  public void setArgsCountButton(JButton theArgsCountButton) {
    argsCountButton = theArgsCountButton;
  }

  /**
   * 
   * @uml.property name="argsCountButton"
   */
  public JButton getArgsCountButton() {
    return argsCountButton;
  }

  /**
   * 
   * @uml.property name="argsCount"
   */
  public void setArgsCount(int theArgsCount) {
    argsCount = theArgsCount;
  }

  /**
   * 
   * @uml.property name="argsCount"
   */
  public int getArgsCount() {
    return argsCount;
  }

  public List getComboItems() {
    return comboItems;
  }

  public void setComboItems(List comboItems) {
    this.comboItems = comboItems;
  }
  
  public PrubaeComboItem	getComboItem(String itemStr) {
    for (int i = 0; i < getComboItems().size(); i++) {
      PrubaeComboItem	ret = (PrubaeComboItem)getComboItems().get(i);
      //System.err.println("PrubaeControllerValueStatement.getComboItem(" + itemStr + ") : comboItems.get(" + i + ") = " + ret.getItemString());
      if (itemStr.equals(ret.getItemString())) {
	return ret;
      }
    }
    return (PrubaeComboItem)getComboItems().get(0);
  }

} /* end class PrubaeControllerValueStatement */
