// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;


import java.util.List;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseWriter;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseWriterElement;

/**
 * class <code>PrubaeModel</code>
 * is the abstract class of prubae rule element model.
 * this class has relationship with its view class and controller class,
 * designed according to the Model View Controller design pattern.
 * this class handles the common behavior of the rule element model instances.
 * this class also has model view controller associations
 * and has an association to the editor's top class instance.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */

public abstract class PrubaeModel
  implements ProrateRulebaseElement, ProrateRulebaseWriterElement, Cloneable {

  // Attributes  
  /**
   * variable <code>regist</code> is the regist code of Rulebase.
   * @uml.property  name="regist"
   */
  protected short regist = (short)0;

  /**
   * variable <code>parent</code> is the reference to the parent model.
   * 
   * @uml.property name="parent"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeModel parent = null;

  /**
   * variable <code>parentList</code> is the reference to the list of the parent model, in which this model belongs.
   * @uml.property  name="parentList"
   */
  protected List parentList = null;
  /**
   * variable <code>comment</code> is the comment string.
   * @uml.property  name="comment"
   */
  protected String comment = null;

  // Associations  
  /**
   * variable <code>view</code> is the reference to the view instance.
   * 
   * @uml.property name="view"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeView view = null;

  /**
   * variable <code>controller</code>
   * is the reference to the controller instance.
   * 
   * @uml.property name="controller"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected PrubaeController controller = null;

  /**
   * variable <code>editor</code>
   * is the reference to the editor's top class instance.
   * 
   * @uml.property name="editor"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected Prubae editor = null;


  // Constructor  
  /**
   * Creates a new <code>PrubaeModel</code> instance.
   *
   */
  public PrubaeModel() {  
    super();  
  }

  // Operations  
  /**
   * <code>initialize</code> method
   * sets the reference to the editor's top class instance,
   * also sets the reference of the view and controller as well.
   * sends initialize to controller, which makes controller to
   * create panel GUI objects and tree node/nodes.
   * sets parent and parentList references.
   * @param editor a <code>Prubae</code> value
   * is the reference to the editor's top class instance.
   * @param parent a <code>PrubaeModel</code> value
   * is the reference to the parent model instance.
   * @param parentList a <code>List</code> value
   * is the reference to the parent list in which this model belongs.
   */
  public void initialize(Prubae editor, PrubaeModel parent,
			 List parentList) {
    System.err.println("model initialize() " + this + ":" + getRegist());
    setEditor(editor);
    getView().setEditor(editor);
    getController().setEditor(editor);
    getController().initialize();
    setParent(parent);
    setParentList(parentList);
  }

  /**
   * <code>close</code> method
   * closes rule element model and destroys all view and controller elements,
   * destroys all GUI elements related to this model,
   * using delete method below.
   */
  public void close() {
    delete();
  }

  /**
   * <code>delete</code> method
   * deletes treenode, close and destroy all GUI elements on the part panel
   * related to the controller of this model.
   * removes this model from parent list, and put the control to the parent.
   * removes all references which this model, view, and controller hava.
   */
  public void delete() {
    /* ;;; deBug
    System.err.println("PrubaeModel.delete() : model = " + this + ", view = " + getView() + ", cntl = " + getController());
    */
    if (getController() != null) {
      getController().deleteTreeNode();
      getController().close();
    }
    if (getParentList() != null) {
      getParentList().remove(this);
    }
    getController().setEditor(null);
    getController().setModel(null);
    getController().setView(null);
    getController().setTreeNode(null);
    getController().setPanel(null);
    getView().setEditor(null);
    getView().setModel(null);
    getView().setController(null);
    setEditor(null);
    setView(null);
    setController(null);
    setParent(null);
    setParentList(null);
  }

  /**
   * <code>openUI</code> method
   * constructs view and controller, and opens GUI elements dynamically.
   * this is to be overrided in class PrubaeModelValue and its subclasses.
   */
  public void openUI() {
  }

  /**
   * <code>closeUI</code> method
   * closes GUI elements and destructs view and controller dynamically.
   * this is to be overrided in class PrubaeModelValue and its subclasses.
   */
  public void closeUI() {
  }

  /**
   * <code>addToParent</code> method
   * adds this model as a child of the specified parent
   * at the end of the specified list.
   * @param list a <code>List</code> value of a list to add to.
   * @param parent a <code>PrubaeModel</code> value to be the parent.
   */
  public void addToParent(List list, PrubaeModel parent) {  
    list.add(this);
    initialize(parent.getEditor(), parent, list);
  }

  /**
   * <code>insertToParent</code> method
   * inserts this model as a child of the specified parent
   * before the specified child in the specified list.
   * @param list a <code>List</code> value of a list to add to.
   * @param parent a <code>PrubaeModel</code> value to be the parent.
   * @param child a <code>PrubaeModel</code> value
   * is the child model before which this model to be inserted.
   */
  public void insertToParent(List list,
			     PrubaeModel parent, PrubaeModel child) {
    int	index = list.indexOf(child);
    list.add(index, this);
    initialize(parent.getEditor(), parent, list);
  }

  /**
   * <code>swap</code> method
   * swaps this model and the old model in the parent list,
   * as the child of the parent of the old model.
   * @param old a <code>PrubaeModel</code> value
   * is the old child model with which this model to be swapped.
   */
  public void swap(PrubaeModel old) {  
    initialize(old.getEditor(), old.getParent(), old.getParentList());
    List	vec = old.getParentList();
    int	index = vec.indexOf(old);
    vec.set(index, this);
  }

  /**
   * <code>toString</code> method
   * returns comment string if comment is not null, returns null otherwise.
   * this is to be overrided in prubae model subclasses.
   * @return a <code>String</code> value is a returned string.
   */
  public String toString() {  
    if (getComment() != null && !getComment().equals("")) {
      return getComment();
    }
    return describe();
  }

  /**
   * <code>toFilename</code> method
   * returns xml rule filename string.
   * this is to be overrided in prubae model subclasses.
   * @return a <code>String</code> value is a returned string.
   */
  public String toFilename() {  
    return null;
  }

  /**
   * <code>describe</code> method
   * this is to be overrided in prubae model subclasses.
   * @return a <code>String</code> value is a returned string.
   */
  public String describe() {  
    return null;
  }

  /**
   * <code>getName</code> method
   * is to be overrided in prubae statement model subclasses,
   * such as PrubaeModelDoStatement, PrubaeModelValueStatement.
   * @return a <code>String</code> value
   */
  public String getName() {  
    return toString();  
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeModel	clone = null;
    PrubaeView	cloneView = null;
    PrubaeController	cloneCntl = null;
    try {
      clone = (PrubaeModel)super.clone();
    } catch (CloneNotSupportedException e) {
      // TODO ư줿 catch ֥å
      e.printStackTrace();
    }
    clone.setRegist(getRegist());
    clone.setParent(null);
    clone.setParentList(null);
    if (getView() != null) {
      cloneView = (PrubaeView)getView().clone();
      clone.setView(cloneView);
    }
    if (getController() != null) {
      cloneCntl = (PrubaeController)getController().clone();
      clone.setController(cloneCntl);
    }
    if (cloneView != null) {
      cloneView.setModel(clone);
      cloneView.setController(cloneCntl);
    }
    if (cloneCntl != null) {
      cloneCntl.setModel(clone);
      cloneCntl.setView(cloneView);
    }
    System.err.println("PrubaeModel.clone() : model = " + this + ", view = " + getView() + ", cntl = " + getController());
    System.err.println("PrubaeModel.clone() : cloneModel = " + clone + ", cloneView = " + cloneView + ", cloneCntl = " + cloneCntl);
    return clone;
  }

  /**
   * <code>getStatement</code> method
   * is to be overrided in prubae statement model subclasses,
   * such as PrubaeModelDoStatement, PrubaeModelValueStatement.
   * @return a <code>String</code> value
   */
  public String getStatement() {
    return "";
  }

  /**
   * <code>setStatement</code> method
   * is to be overrided in prubae statement model subclasses,
   * such as PrubaeModelDoStatement, PrubaeModelValueStatement.
   * @return a <code>String</code> value
   */
  public void setStatement(String theStatement) {
  }

  /**
   * <code>writeModel</code> method writes the model using the writer.
   * this is to be overrided in prubae model subclasses.
   */
  public abstract void writeModel(ProrateRulebaseWriter rulebase);

  /**
   * <code>getSize</code> method counts size of the model and its children.
   * this is to be overrided in prubae model subclasses.
   * @param rulebase a <code>ProrateRulebaseWriter</code> value
   * @return an <code>int</code> value
   */
  public abstract int getSize(ProrateRulebaseWriter rulebase);

  /* ;;; 2006.09.12 */
  /**
   * <code>check</code> method	check validity of the prubae model.
   *
   * @param errors a <code>List</code> value
   * @return a <code>boolean</code> value
   */
  public abstract boolean check(List errors);

  /* ;;; 2006.09.13 */
  /**
   * <code>equals</code> method	return if this is equal to the other.
   *
   * @param other a <code>PrubaeModel</code> value
   * @return a <code>boolean</code> value
   */
  public boolean equals(PrubaeModel other) {
    return (this == other);
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="regist"
   */
  public void setRegist(short theRegist) {
    regist = theRegist;
  }

  /**
   * 
   * @uml.property name="regist"
   */
  public short getRegist() {
    return (short) regist;
  }

  /**
   * 
   * @uml.property name="parent"
   */
  public void setParent(PrubaeModel theParent) {
    parent = theParent;
  }

  /**
   * 
   * @uml.property name="parent"
   */
  public PrubaeModel getParent() {
    return parent;
  }

  /**
   * 
   * @uml.property name="parentList"
   */
  public void setParentList(List theParentList) {
    parentList = theParentList;
  }

  /**
   * 
   * @uml.property name="parentList"
   */
  public List getParentList() {
    return parentList;
  }

  /**
   * 
   * @uml.property name="comment"
   */
  public void setComment(String theComment) {
    comment = theComment;
  }

  /**
   * 
   * @uml.property name="comment"
   */
  public String getComment() {
    return comment;
  }

  /**
   * 
   * @uml.property name="editor"
   */
  public void setEditor(Prubae theEditor) {
    editor = theEditor;
  }

  /**
   * 
   * @uml.property name="editor"
   */
  public Prubae getEditor() {
    return editor;
  }

  /**
   * 
   * @uml.property name="controller"
   */
  public void setController(PrubaeController theController) {
    controller = theController;
  }

  /**
   * 
   * @uml.property name="controller"
   */
  public PrubaeController getController() {
    return controller;
  }

  /**
   * 
   * @uml.property name="view"
   */
  public void setView(PrubaeView theView) {
    view = theView;
  }

  /**
   * 
   * @uml.property name="view"
   */
  public PrubaeView getView() {
    return view;
  }

} /* end class PrubaeModel */
