// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import gnu.lists.LList;
import gnu.math.IntNum;
import java.util.List;
import java.util.Vector;
import jp.sourceforge.glj.lisp.Lisp;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseWriter;


/**
 * class <code>PrubaeModelDoStatement</code>
 * handles action statement rule element models such as DONT_APPLY, Pfm() etc.
 * this class has values rule model list and statement String,
 * and percent value and function name string as the attributes of the model.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public class PrubaeModelDoStatement extends PrubaeModelDo
  implements PrubaeModelStatement {

  // Attributes  
  /**
   * variable <code>statement</code> is the statement String to describe the action statement of this model.
   * @uml.property  name="statement"
   */
  private String statement = null;
  /**
   * variable <code>values</code> is a list of the rule element models of the statements functions.
   * @uml.property  name="values"
   */
  private List values = null;
  /**
   * variable <code>percent</code> is the discount percent value when this statement is the amount setting such as "60% of USD 1000", "100% of Pfm($F-BASIS)" or "90% of Np()" etc.
   * @uml.property  name="percent"
   */
  private double percent = (double)0;
  /**
   * variable <code>name</code> is the function name of this statement
   * @uml.property  name="name"
   */
  private String name = null;

  // Constructor  
  /**
   * Creates a new <code>PrubaeModelDoStatement</code> instance.
   * this creates all GUI elements on part panel of the function,
   * also sets default values to the percent and name variables.
   */
  public PrubaeModelDoStatement() {  
    setView(new PrubaeViewDoStatement());
    getView().setModel(this);
    setController(new PrubaeControllerDoStatement());
    getController().setModel(this);
    getView().setController(getController());
    getController().setView(getView());
    setPercent((double)0);
    setName("");
  }  

  // Operations  
  /**
   * <code>closeUI</code> method
   * sends closeUI() method to the all models in the values list.
   */
  public void closeUI() {
    if (getValues() != null) {
      for (int i = 0; i < getValues().size(); i++) {
	if (getValues().get(i) instanceof PrubaeModel) {
	  ((PrubaeModel)getValues().get(i)).closeUI();
	}
      }
    }
  }  

  /**
   * <code>toString</code> method
   * returns comment string if comment is not null nor "",
   * returns the statement string when it is not null nor "",
   * returns "DoStatement" string otherwise.
   * @return a <code>String</code> value is a returned string.
   */
  public String toString() {
    if (getComment() != null && !getComment().equals("")) {
      return getComment();
    }
    if (getStatement() == null || getStatement().equals("")) {
      return "DoStatement";
    }
    else {
      return getStatement();
    }
  }

  /**
   * <code>updateStatement</code> method
   * creates statement string to describe this action statement element model.
   * for all models in the values list, if the model is the value statement,
   * sends update() method to the controller of the model in the list.
   */
  public void updateStatement() {
    String	funcName = null;
    LList	list = getEditor().getDoList();
    LList	item = Lisp.assoc(new IntNum(getRegist()), list);
    if (item != null) {
      if (Lisp.car(Lisp.cdr(Lisp.cdr(item))) == null
	  || Lisp.isNil(Lisp.car(Lisp.cdr(Lisp.cdr(item))))) {
	funcName = "";
      }
      else if (getRegist() == (short) ProrateRulebaseElement.EXTERNAL_FUNCTION
	       || (getRegist() ==
		   (short) ProrateRulebaseElement.PARTCALL_CODE)) {
	funcName = "";
      }
      else {
	funcName = (Lisp.car(Lisp.cdr(Lisp.cdr(item)))).toString();
      }
    }
    if (funcName == null) {
      funcName = "";
    }
    if (funcName != null && funcName.length() > 0) {
      setName(funcName);
    }
    else if (getName() != null && getName().length() > 0) {
      funcName = getName();
    }

    if (getValues() == null || getValues().size() <= 0) {
      if ((getRegist() & ProrateRulebaseElement.FUNCTION_SPECIFICATION) != 0) {
	setStatement(funcName + "()");
      }
      else {
	setStatement(funcName);
      }
      if (getPercent() != (double)0) {
	setStatement("" + getPercent() + "% of " + getStatement());
      }
      return;
    }
    setStatement(funcName + "(");
    for (int i = 0; i < getValues().size(); i++) {
      if (i > 0) {
	setStatement(getStatement() + " ");
      }
      if (getValues().get(i) instanceof PrubaeModelValue) {
	if (getValues().get(i) instanceof PrubaeModelValueStatement) {
	  PrubaeController
	    cntl =
	    (PrubaeController)
	    ((PrubaeModel)getValues().get(i)).getController();
	  if (cntl != null) {
	    cntl.update();
	  }
	}
	((PrubaeModelValue)getValues().get(i)).updateStatement();
	setStatement(getStatement() +
		     ((PrubaeModelValue)getValues().get(i)).getStatement());
      }
      else {
	setStatement(getStatement() + getValues().get(i).toString());
      }
    }
    setStatement(getStatement() + ")");
    if (getPercent() != (double)0) {
      setStatement("" + getPercent() + "% of " + getStatement());
    }
    return;
  }

  /**
   * <code>writeModel</code> method
   * writes the PERCENT_CODE and percent value if the percent value is not 0,
   * writes the regist code of the model and attributes with it,
   * writes all model values in the values list,
   * using the writer(PrubaeWriter) instance.
   */
  public void writeModel(ProrateRulebaseWriter rulebase) {
    rulebase.writeDoStatement(this);
  }

  /**
   * <code>getSize</code> method
   * counts the size of this rule and returns the size.
   *
   * @param rulebase a <code>ProrateRulebaseWriter</code> value
   * @return an <code>int</code> value of size bytes
   */
  public int getSize(ProrateRulebaseWriter rulebase) {
    return rulebase.getDoStatementSize(this);
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeModelDoStatement	clone = null;
    clone = (PrubaeModelDoStatement)super.clone();
    if (getValues() != null) {
      clone.setValues((List)((Vector)getValues()).clone());
      for (int i = 0; i < getValues().size(); i++) {
	clone.getValues().set(i, ((PrubaeModel)getValues().get(i)).clone());
	((PrubaeModel)clone.getValues().get(i))
	  .initialize(getEditor(), clone, clone.getValues());
      }
    }
    return clone;
  } 

  /* ;;; 2006.09.12 */
  /**
   * <code>check</code> method	check validity of the prubae model.
   *
   * @param errors a <code>List</code> value
   * @return a <code>boolean</code> value
   */
  public boolean check(List errors) {
    LList	list = getEditor().getDoList();
    LList	doList = Lisp.assoc(new IntNum(getRegist()), list);
    if (doList == null) {
      errors.add("illegal function code " + getRegist());
      return false;
    }
    if (getRegist() == (short) ProrateRulebaseElement.EXTERNAL_FUNCTION) {
      String	item = getStatement();
      int	index = item.indexOf("(");
      if (index >= 0) {
	item = item.substring(0, index);
      }
      String	itemStr = (Lisp.car(Lisp.cdr(Lisp.cdr(doList)))).toString();
      if (!itemStr.equals(item)) {
	LList	member = Lisp.member(doList, list);
	for (member = (LList)Lisp.cdr(member);
	     !Lisp.isNil(member);
	     member = (LList)Lisp.cdr(member)) {
	  doList = (LList)Lisp.car(member);
	  itemStr = (Lisp.car(Lisp.cdr(Lisp.cdr(doList)))).toString();
	  if (itemStr.equals(item)) {
	    break;
	  }
	}
      }
    }
    Object
      isArgs = Lisp.car(Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(doList)))));
    LList
      args =
      (LList)Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(Lisp.cdr(doList)))));
    if (isArgs != null && (isArgs instanceof IntNum)) {
      int	numArgs = ((IntNum)isArgs).intValue();
      if (numArgs >= 0) {
	/* ;;; deBug */
	System.err.println("PrubaeModelDoStatement.check() " + getRegist()
			   + " " + getName());
	if (getValues() == null) {
	  if (numArgs != 0) {
	    errors.add("illegal argument count in " + getName() + " function");
	    return false;
	  }
	}
	else if (getValues().size() != numArgs) {
	  errors.add("illegal argument count in " + getName() + " function");
	  return false;
	}
      }
    }
    if (getValues() != null) {
      for (int i = 0; i < getValues().size(); i++) {
	Object	argType = Lisp.car(args);
	args = (LList)Lisp.cdr(args);
	Object	value = getValues().get(i);
	if (argType.equals("const") && !(value instanceof PrubaeModel)) {
	  continue;
	}
	if (!(value instanceof PrubaeModel)) {
	  return false;
	}
	if (!((PrubaeModel)value).check(errors)) {
	  errors.add("in argument #" + (i+1)
		     + " of " + getName() + " function");
	  return false;
	}
      }
    }
    return true;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="statement"
   */
  public void setStatement(String theStatement) {
    statement = theStatement;
  }

  /**
   * 
   * @uml.property name="statement"
   */
  public String getStatement() {
    return statement;
  }

  /**
   * 
   * @uml.property name="values"
   */
  public void setValues(List theValues) {
    values = theValues;
  }

  /**
   * 
   * @uml.property name="values"
   */
  public List getValues() {
    return values;
  }

  /**
   * 
   * @uml.property name="percent"
   */
  public void setPercent(double thePercent) {
    percent = thePercent;
  }

  /**
   * 
   * @uml.property name="percent"
   */
  public double getPercent() {
    return percent;
  }

  /**
   * 
   * @uml.property name="name"
   */
  public void setName(String theName) {
    name = theName;
  }

  /**
   * 
   * @uml.property name="name"
   */
  public String getName() {
    return name;
  }




} /* end class PrubaeModelDoStatement */
