// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;


import java.util.List;
import java.util.Vector;

import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElementFactory;
import jp.sourceforge.gnp.prorate.ProrateRuleObject;

/**
 * class <code>PrubaeModelFactory</code>
 * is the factory class of prubae rule element models.
 * this class is called through the ProrateRulebaseElementFactory interface
 * from XmlRulebase(Xml Rulebase) class groups.
 * this mechanism is designed according to the Abstract Factory design pattern,
 * and the Factory Method design pattern.
 * this class has an association to the editor top class instance.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */

public class PrubaeModelFactory
  implements ProrateRulebaseElementFactory {

  /**
   * variable <code>editor</code>
   * is the reference to the editor top class instance.
   * 
   * @uml.property name="editor"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  Prubae editor = null;


  // Constructor  
  /**
   * Creates a new <code>PrubaeModelFactory</code> instance.
   *
   */
  public PrubaeModelFactory() {  
    super();  
  }  

  // Operations  
  /**
   * <code>makeNopObject</code> method
   * creates PrubaeModelFunction class,
   * with SPA, APDP or EXTF rule element.
   * this also handles COMMON and ACTION rule element,
   * it also creates and returns PrubaeModelFunction with them,
   * these class is stripped in the caller class PrubaeReader,
   * just used to identify the list and divide into [common] and [action].
   * this also handles VALIDITY, ATMARK, ENDTABLE, TABLE_FIELD_END elements,
   * these are simply skipped.
   * @param regist a <code>short</code> value is the regist code.
   * @param first a <code>String</code> value
   * is the [carrier] value of SPA or APDP. function name for EXTF.
   * @param second a <code>String</code> value
   * is the [tkcarrier] value of SPA, "" for APDP and EXTF.
   * @param from a <code>String</code> value
   * is the [validity] valid from value of SPA and APDP,
   * nothing to do with EXTF.
   * @param until a <code>String</code> value
   * is the [validity] valid until value of SPA and APDP,
   * nothing to do with EXTF.
   * @param type a <code>String</code> value
   * is the return type value of EXTF,
   * nothing to do with SPA, APDP and PART.
   * @param comment a <code>String</code> value is the comment string
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeNopObject(short regist, String first, String second,
		  String from, String until, String type, String comment) {
    PrubaeModelFunction	ret = new PrubaeModelFunction();
    ret.setEditor(getEditor());
    switch (regist) {
    case ProrateRulebaseElement.SPA_CODE:
      ret.setRegist(regist);
      ret.setComment(comment);
      ret.setFunctionType((int)PrubaeModelFunction.SPA);
      ret.setCarrier(first);
      ret.setTkCarrier(second);
      ret.setName("");
      ret.setFromDate(Integer.parseInt(from));
      ret.setToDate(Integer.parseInt(until));
      return (ProrateRulebaseElement)ret;
    case ProrateRulebaseElement.APD_P_CODE:
      ret.setRegist(regist);
      ret.setComment(comment);
      ret.setFunctionType((int)PrubaeModelFunction.APDP);
      ret.setCarrier(first);
      ret.setTkCarrier("");
      ret.setName("");
      ret.setFromDate(Integer.parseInt(from));
      ret.setToDate(Integer.parseInt(until));
      return (ProrateRulebaseElement)ret;
    case ProrateRulebaseElement.EXTF_CODE:
      ret.setRegist(regist);
      ret.setComment(comment);
      ret.setFunctionType((int)PrubaeModelFunction.EXTF);
      ret.setCarrier("");
      ret.setTkCarrier("");
      ret.setName(first);
      int	extfReturnType = 0;
      if (type.equals("judge")) {
	extfReturnType = PrubaeModelFunction.ExtfReturnJudgement;
      }
      else if (type.equals("action")) {
	extfReturnType = PrubaeModelFunction.ExtfReturnAction;
      }
      else if (type.equals("value")) {
	extfReturnType = PrubaeModelFunction.ExtfReturnValue;
      }
      ret.setExtfReturnType(extfReturnType);
      return (ProrateRulebaseElement)ret;
    case ProrateRulebaseElement.PART_CODE:
      ret.setRegist(regist);
      ret.setComment(comment);
      ret.setFunctionType((int)PrubaeModelFunction.PART);
      ret.setCarrier("");
      ret.setTkCarrier("");
      ret.setName(first);
      return (ProrateRulebaseElement)ret;
    case ProrateRulebaseElement.COMMON_CODE:
    case ProrateRulebaseElement.ACTIONS_CODE:
      ret.setRegist(regist);
      ret.setComment(comment);
      ret.setFunctionType((int)0);
      ret.setCarrier("");
      ret.setTkCarrier("");
      ret.setName("");
      return (ProrateRulebaseElement)ret;
    case ProrateRulebaseElement.VALIDITY_CODE:
    case ProrateRulebaseElement.ATMARK_CODE:
    case ProrateRulebaseElement.ENDTABLE_CODE:
    case ProrateRulebaseElement.TABLE_FIELD_END:
      return null;
    }
    return null;
  }
  
  /**
   * Describe <code>makePartObject</code> method here.
   * creates PrubaeModelDoStatement class with DONT_APPLY rule element.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makePartObject(String name, List rules,
					       String comment) {
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    ret.setRegist((short)ProrateRulebaseElement.PART_CODE);
    ret.setName(name);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeEndObject</code> method here.
   * creates PrubaeModelDoStatement class with DONT_APPLY rule element.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makeEndObject(String comment) {
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    ret.setRegist((short)ProrateRulebaseElement.END_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeTestObject</code> method here.
   * creates PrubaeModelIf class with If-statement rule element.
   * this sets thens and elses lists as well.
   * @param tests a <code>List</code> value
   * is an list of the rule element models of the judge statements.
   * @param thens a <code>List</code> value
   * is an list of the rule element models of the action statements,
   * to be evaluated when the test is true, of if statement.
   * @param elses a <code>List</code> value
   * is an list of the rule element models of the action statements,
   * to be evaluated when the test is false, of if statement.
   * @param comment a <code>String</code> value is the comment string
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeTestObject(List tests, List thens, List elses, String comment) {
    PrubaeModelIf	ret = new PrubaeModelIf();
    ret.setRegist((short)ProrateRulebaseElement.TEST_BEGIN_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    if (tests== null || tests.size() < 1) {
      PrubaeModelJudge	cond = (PrubaeModelJudge)new PrubaeModelJudge();
      ((PrubaeModel)cond).initialize(getEditor(), ret, null);
      ret.setCondition(cond);
    }
    else {
      PrubaeModelJudge	cond = (PrubaeModelJudge)(tests.get(0));
      ((PrubaeModel)cond).initialize(getEditor(), ret, null);
      ret.setCondition(cond);
    }
    for (int i = 0; i < thens.size(); i++) {
      ((PrubaeModel)thens.get(i)).initialize(getEditor(), ret, ret.getThens());
      ret.getThens().add((PrubaeModel)thens.get(i));
    }
    for (int i = 0; i < elses.size(); i++) {
      ((PrubaeModel)elses.get(i)).initialize(getEditor(), ret, ret.getElses());
      ret.getElses().add((PrubaeModel)elses.get(i));
    }
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeAndObject</code> method here.
   * creates PrubaeModelAndOr class with And judge rule element.
   * @param clauses a <code>List</code> value
   * is a list of the rule element models of the judge statements under this.
   * @param comment a <code>String</code> value is the comment string
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makeAndObject(List clauses, String comment) {
    PrubaeModelAndOr	ret = new PrubaeModelAndOr();
    ret.setRegist((short)ProrateRulebaseElement.AND_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setAnd(true);
    if (clauses == null || clauses.size() < 1) {
      PrubaeModelJudge	cond = (PrubaeModelJudge)new PrubaeModelJudge();
      ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)cond);
      PrubaeModelJudge	next = (PrubaeModelJudge)new PrubaeModelJudge();
      ((PrubaeModel)next).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)next);
      return (ProrateRulebaseElement)ret;
    }
    PrubaeModelJudge	cond = (PrubaeModelJudge)(clauses.get(0));
    if (clauses.size() < 2) {
      ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)cond);
      PrubaeModelJudge	next = (PrubaeModelJudge)new PrubaeModelJudge();
      ((PrubaeModel)next).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)next);
      return (ProrateRulebaseElement)ret;
    }
    PrubaeModelJudge	next = null;
    for (int i = 1; i < clauses.size(); i++) {
      next = (PrubaeModelJudge)(clauses.get(i));
      //       if (next instanceof PrubaeModelAndOr) {
      // 	if (((PrubaeModelAndOr)next).getStatements() == null) {
      // 	  continue;
      // 	}
      // 	List	nexts = new Vector();
      // 	nexts.add(cond);
      // 	nexts.addAll(((PrubaeModelAndOr)next).getStatements());
      // 	for (int j = 0; j < nexts.size(); j++) {
      // 	  ((PrubaeModel)nexts.get(j)).initialize(getEditor(), next, nexts);
      // 	}
      // 	((PrubaeModelAndOr)next).setStatements(nexts);
      // 	cond = next;
      //       }
      //       else {
      // 	((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      // 	ret.getStatements().add((PrubaeModel)cond);
      // 	cond = next;
      //       }
      ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)cond);
      cond = next;
    }
    ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
    ret.getStatements().add((PrubaeModel)cond);
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeOrObject</code> method here.
   * creates PrubaeModelAndOr class with Or judge rule element.
   * @param clauses a <code>List</code> value
   * is a list of the rule element models of the judge statements under this.
   * @param comment a <code>String</code> value is the comment string
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makeOrObject(List clauses, String comment) {
    PrubaeModelAndOr	ret = new PrubaeModelAndOr();
    ret.setRegist((short)ProrateRulebaseElement.OR_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setAnd(false);
    if (clauses == null || clauses.size() < 1) {
      PrubaeModelJudge	cond = (PrubaeModelJudge)new PrubaeModelJudge();
      ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)cond);
      PrubaeModelJudge	next = (PrubaeModelJudge)new PrubaeModelJudge();
      ((PrubaeModel)next).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)next);
      return (ProrateRulebaseElement)ret;
    }
    PrubaeModelJudge	cond = (PrubaeModelJudge)(clauses.get(0));
    if (clauses.size() < 2) {
      ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)cond);
      PrubaeModelJudge	next = (PrubaeModelJudge)new PrubaeModelJudge();
      ((PrubaeModel)next).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)next);
      return (ProrateRulebaseElement)ret;
    }
    PrubaeModelJudge	next = null;
    for (int i = 1; i < clauses.size(); i++) {
      next = (PrubaeModelJudge)(clauses.get(i));
      //       if (next instanceof PrubaeModelAndOr) {
      // 	if (((PrubaeModelAndOr)next).getStatements() == null) {
      // 	  continue;
      // 	}
      // 	List	nexts = new Vector();
      // 	nexts.add(cond);
      // 	nexts.addAll(((PrubaeModelAndOr)next).getStatements());
      // 	for (int j = 0; j < nexts.size(); j++) {
      // 	  ((PrubaeModel)nexts.get(j)).initialize(getEditor(), next, nexts);
      // 	}
      // 	((PrubaeModelAndOr)next).setStatements(nexts);
      // 	cond = next;
      //       }
      //       else {
      // 	((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      // 	ret.getStatements().add((PrubaeModel)cond);
      // 	cond = next;
      //       }
      ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
      ret.getStatements().add((PrubaeModel)cond);
      cond = next;
    }
    ((PrubaeModel)cond).initialize(getEditor(), ret, ret.getStatements());
    ret.getStatements().add((PrubaeModel)cond);
    return (ProrateRulebaseElement)ret;
  }

  public ProrateRulebaseElement	makeJudgeObject(List clauses, String comment) {
    PrubaeModelJudge	ret = new PrubaeModelJudge();
    ret.setRegist((short)0);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeSwitchObject</code> method here.
   * creates PrubaeModelSwitch class with case-statement rule elements.
   * @param args a <code>List</code> value
   * is a list of the rule element models of the case-var values.
   * @param branchacts a <code>List</code> value
   * is a list of the rule element models of the case-branches,
   * which are in the class of PrubaeModelCase.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeSwitchObject(List args, List branchacts, String comment) {
    PrubaeModelSwitch	ret = new PrubaeModelSwitch();
    ret.setRegist((short)ProrateRulebaseElement.CASE_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    for (int i = 0; i < args.size(); i++) {
      /*
	ret.getVars().add(((PrubaeModelValue)args.get(i)).getStatement());
      */
      ((PrubaeModel)args.get(i)).initialize(getEditor(), ret, ret.getVars());
      ret.getVars().add((PrubaeModel)args.get(i));
    }
    for (int i = 0; i < branchacts.size(); i++) {
      ((PrubaeModel)branchacts.get(i)).initialize(getEditor(), ret,
						  ret.getCases());
      ret.getCases().add((PrubaeModel)branchacts.get(i));
    }
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeBranchObject</code> method here.
   * creates PrubaeModelCase class with rule elements.of
   * set of case-branch and related actions.
   * @param branches a <code>List</code> value
   * is a list of the rule element models of the case-branches.
   * @param actions a <code>List</code> value
   * is a list of the rule element models of the actions
   * related to the branch.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeBranchObject(List branches, List actions, String comment) {
    PrubaeModelCase	ret = new PrubaeModelCase();
    ret.setComment(comment);
    ret.setEditor(getEditor());
    for (int i = 0; i < branches.size(); i++) {
      /*
	ret.getValues().add((PrubaeModelValue)branches.get(i));
      */
      ((PrubaeModel)branches.get(i)).initialize(getEditor(), ret,
						ret.getValues());
      ret.getValues().add((PrubaeModel)branches.get(i));
    }
    for (int i = 0; i < actions.size(); i++) {
      ((PrubaeModel)actions.get(i)).initialize(getEditor(), ret,
					       ret.getActions());
      ret.getActions().add((PrubaeModel)actions.get(i));
    }
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeCaseObject</code> method here.
   * creates PrubaeModelValue class with case-branch values rule elements.
   * @param values a <code>List</code> value
   * is a list of the rule element models of the case-branch values.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeCaseObject(List values, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValue();
    ret.setRegist((short)ProrateRulebaseElement.CASE_BRANCH_BEGIN);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement("");
    ret.setValues(new Vector());
    for (int i = 0; i < values.size(); i++) {
      if (i > 0) {
	ret.setStatement(ret.getStatement() + " ");
      }
      ((PrubaeModel)values.get(i)).initialize(getEditor(),
					      ret, ret.getValues());
      ret.getValues().add((PrubaeModel)values.get(i));
      ret.setStatement(ret.getStatement() +
		       ((PrubaeModelValue)values.get(i)).getStatement());
    }
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeBoolObject</code> method here.
   * creates PrubaeModelValue class with boolean value.
   * the returned class is processed in makeReturnObject() method,
   * and the method returns RETTRUE/RETFALSE_CODE PrubaeModelDoStatement.
   * @param bool a <code>boolean</code> value true or false
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makeBoolObject(boolean bool, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValue();
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement(bool ? "true" : "false");
    ret.setValues(new Vector());
    ret.getValues().add(Boolean.valueOf(bool));
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeReturnObject</code> method here.
   * creates PrubaeModelDoStatement class with RETURN rule element.
   * if the value is PrubaeModelValue boolean value made in makeBoolObject(),
   * then set regist code RETTRUE/RETFALSE_CODE according to the boolean value.
   * otherwise, set regist code RETURN_CODE.
   * and add the value rule element model to the value list.
   * @param value a <code>ProrateRulebaseElement</code> value
   * is the value rule element model.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeReturnObject(ProrateRulebaseElement value, String comment) {
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    ret.setComment(comment);
    ret.setEditor(getEditor());
    if ((value instanceof PrubaeModelValue)
	&& (((PrubaeModelValue)value).getValues().size() == 1)
	&& (((PrubaeModelValue)value).getValues().get(0) instanceof Boolean)) {
      Boolean	bool = (Boolean)((PrubaeModelValue)value).getValues().get(0);
      ret.setRegist(bool.booleanValue() ?
		    (short)ProrateRulebaseElement.RETTRUE_CODE
		    : (short)ProrateRulebaseElement.RETFALSE_CODE);
    }
    else {
      ret.setRegist((short)ProrateRulebaseElement.RETURN_CODE);
      ret.setValues(new Vector());
      ((PrubaeModel)value).initialize(getEditor(), ret, ret.getValues());
      ret.getValues().add((PrubaeModel)value);
    }
    //ret.setEditor(getEditor());
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeMultivalObject</code> method here.
   * creates PrubaeModelValueStatement class with MULTIVAL rule elements.
   * @param values a <code>List</code> value
   * is the list of MULTIVAL rule element models.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeMultivalObject(List values, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist((short)ProrateRulebaseElement.MULTIVAL_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement("");
    ret.setValues(new Vector());
    for (int i = 0; i < values.size(); i++) {
      if (i > 0) {
	ret.setStatement(ret.getStatement() + ",");
      }
      ((PrubaeModel)values.get(i)).initialize(getEditor(),
					      ret, ret.getValues());
      ret.getValues().add((PrubaeModel)values.get(i));
      ret.setStatement(ret.getStatement() +
		       ((PrubaeModelValue)values.get(i)).getStatement());
    }
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeIntervalObject</code> method here.
   * creates PrubaeModelValueStatement class with INTERVAL rule elements.
   * @param lowvalue a <code>ProrateRulebaseElement</code> value
   * is the low value rule element model.
   * @param highvalue a <code>ProrateRulebaseElement</code> value
   * is the high value rule element model.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeIntervalObject(ProrateRulebaseElement lowvalue,
		       ProrateRulebaseElement highvalue, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist((short)ProrateRulebaseElement.INTERVAL_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement(((PrubaeModelValue)lowvalue).getStatement() + "-"
		     + ((PrubaeModelValue)highvalue).getStatement());
    ret.setValues(new Vector());
    ((PrubaeModel)lowvalue).initialize(getEditor(), ret, ret.getValues());
    ret.getValues().add((PrubaeModel)lowvalue);
    ((PrubaeModel)highvalue).initialize(getEditor(), ret, ret.getValues());
    ret.getValues().add((PrubaeModel)highvalue);
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeAmountObject</code> method here.
   * creates PrubaeModelDoStatement class with amount rule element.
   * @param currency a <code>String</code> value of currency string.
   * @param value a <code>double</code> value of amount value.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeAmountObject(String currency, double value, String comment) {
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    ret.setRegist((short)ProrateRulebaseElement.AMOUNT_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setValues(new Vector());
    ret.getValues().add(currency);
    ret.getValues().add(new Double(value));
    //ret.setEditor(getEditor());
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makePercentObject</code> method here.
   * creates PrubaeModelDoStatement class with percent rule element.
   * this makes aclone of amount rule element specified as the argument,
   * and set the percent value to the clone, and returns the clone.
   * @param percent a <code>double</code> value is the percent value.
   * @param amount a <code>ProrateRulebaseElement</code> value
   * is the amount rule element model.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makePercentObject(double percent, ProrateRulebaseElement amount,
		      String comment) {
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setRegist(amount.getRegist());
    ret.setName(amount.getName());
    ret.setPercent(percent);
    ret.setValues(((PrubaeModelDoStatement)amount).getValues());
    if (ret.getValues() != null) {
      for (int i = 0; i < ret.getValues().size(); i++) {
	if (!(ret.getValues().get(i) instanceof PrubaeModel)) {
	  continue;
	}
	((PrubaeModel)ret.getValues().get(i)).initialize(getEditor(),
							 ret,
							 ret.getValues());
      }
    }
    //ret.setEditor(getEditor());
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeVarObject</code> method here.
   * creates PrubaeModelValueStatement class with variable rule element.
   * @param regist a <code>short</code> value
   * is the regist code of the variable.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makeVarObject(short regist, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist(regist);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    String	varName;
    /* ;;; 楳˳饹Υ󥹥󥹤 */
    switch (regist) {
    case ProrateRulebaseElement.ENDORSE_VARIABLE:
      varName = "$ENDORSE";
      break;
    case ProrateRulebaseElement.RESTRICT_VARIABLE:
      varName = "$RESTRICT";
      break;
    case ProrateRulebaseElement.ORIGIN_VARIABLE:
      varName = "$ORIGIN";
      break;
    case ProrateRulebaseElement.DEST_VARIABLE:
      varName = "$DEST";
      break;
    case ProrateRulebaseElement.ISSUE_DATE_VARIABLE:
      varName = "$ISSUE-DATE";
      break;
    case ProrateRulebaseElement.ISSUE_BY_VARIABLE:
      varName = "$ISSUE-BY";
      break;
    case ProrateRulebaseElement.ISSUE_PLACE_VARIABLE:
      varName = "$ISSUE-PLACE";
      break;
    case ProrateRulebaseElement.MULTI_PATH_VARIABLE:
      varName = "$MULTI-PATH";
      break;
    case ProrateRulebaseElement.FC_FROM_VARIABLE:
      varName = "$FC-FROM";
      break;
    case ProrateRulebaseElement.FC_TO_VARIABLE:
      varName = "$FC-TO";
      break;
    case ProrateRulebaseElement.FC_PATH_VARIABLE:
      varName = "$FC-PATH";
      break;
    case ProrateRulebaseElement.ISSUE_COUNTRY_VARIABLE:
      varName = "$ISSUE-COUNTRY";
      break;
    case ProrateRulebaseElement.ISSUE_AREA_VARIABLE:
      varName = "$ISSUE-AREA";
      break;
    case ProrateRulebaseElement.ISSUE_AGENT_VARIABLE:
      varName = "$ISSUE-AGENT";
      break;
    case ProrateRulebaseElement.TOURCODE_VARIABLE:
      varName = "$TOURCODE";
      break;
    case ProrateRulebaseElement.FROM_VARIABLE:
      varName = "$FROM";
      break;
    case ProrateRulebaseElement.TO_VARIABLE:
      varName = "$TO";
      break;
    case ProrateRulebaseElement.F_BASIS_VARIABLE:
      varName = "$F-BASIS";
      break;
    case ProrateRulebaseElement.TKDESIGN_VARIABLE:
      varName = "$TKDESIGN";
      break;
    case ProrateRulebaseElement.F_BASIS_ALL_VARIABLE:
      varName = "$F-BASIS-ALL";
      break;
    case ProrateRulebaseElement.CARRIER_VARIABLE:
      varName = "$CARRIER";
      break;
    case ProrateRulebaseElement.FLIGHT_VARIABLE:
      varName = "$FLIGHT";
      break;
    case ProrateRulebaseElement.CLASS_VARIABLE:
      varName = "$CLASS";
      break;
    case ProrateRulebaseElement.DATE_VARIABLE:
      varName = "$DATE";
      break;
    case ProrateRulebaseElement.PATH_VARIABLE:
      varName = "$PATH";
      break;
    case ProrateRulebaseElement.ROUTE_VARIABLE:
      varName = "$ROUTE";
      break;
    case ProrateRulebaseElement.FROM_ANY_VARIABLE:
      varName = "$FROM-A";
      break;
    case ProrateRulebaseElement.TO_ANY_VARIABLE:
      varName = "$TO-A";
      break;
    case ProrateRulebaseElement.F_BASIS_ANY_VARIABLE:
      varName = "$F-BASIS-A";
      break;
    case ProrateRulebaseElement.TKDESIGN_ANY_VARIABLE:
      varName = "$TKDESIGN-A";
      break;
    case ProrateRulebaseElement.F_BASIS_ALL_ANY_VARIABLE:
      varName = "$F-BASIS-ALL-A";
      break;
    case ProrateRulebaseElement.CARRIER_ANY_VARIABLE:
      varName = "$CARRIER-A";
      break;
    case ProrateRulebaseElement.FLIGHT_ANY_VARIABLE:
      varName = "$FLIGHT-A";
      break;
    case ProrateRulebaseElement.CLASS_ANY_VARIABLE:
      varName = "$CLASS-A";
      break;
    case ProrateRulebaseElement.DATE_ANY_VARIABLE:
      varName = "$DATE-A";
      break;
    case ProrateRulebaseElement.PATH_ANY_VARIABLE:
      varName = "$PATH-A";
      break;
    case ProrateRulebaseElement.ROUTE_ANY_VARIABLE:
      varName = "$ROUTE-A";
      break;
    case ProrateRulebaseElement.DISCOUNT_VARIABLE:
      varName = "$DISCOUNT";
      break;
    default:
      varName = "$??";
      break;
    }
    ret.setStatement(varName);
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeExternObject</code> method here.
   * creates PrubaeModelDoStatement class with External Function calling.
   * @param extfName a <code>String</code> value
   * is the external function name string.
   * @param parentName a <code>String</code> value
   * is the name of the parent element.
   * @param args a <code>List</code> value
   * is the list of arguments rule element models.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeExternObject(String extfName, String parentName,
		     List args, String comment) {
    PrubaeModelStatement	ret = null;
    if (parentName.equals("judge-statement")) {
      ret = new PrubaeModelJudgeStatement();
    }
    else if (parentName.equals("action-statement")) {
      ret = new PrubaeModelDoStatement();
    }
    else {
      ret = new PrubaeModelValueStatement();
    }
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setRegist((short)ProrateRulebaseElement.EXTERNAL_FUNCTION);
    ret.setName(extfName);
    if (args.size() <= 0) {
      //ret.setEditor(getEditor());
      ret.updateStatement();
      return (ProrateRulebaseElement)ret;
    }
    ret.setValues(new Vector());
    for (int i = 0; i < args.size(); i++) {
      if (args.get(i) instanceof PrubaeModelDoStatement) {
	PrubaeModelDoStatement	arg = (PrubaeModelDoStatement)args.get(i);
	PrubaeModelValueStatement
	  val = new PrubaeModelValueStatement();
	val.setComment(arg.getComment());
	val.setRegist(arg.getRegist());
	/* val.setStatement(arg.getStatement()); */
	val.setValues(arg.getValues());
	if (val.getValues() != null) {
	  for (int j = 0; j < val.getValues().size(); j++) {
	    if (!(val.getValues().get(j) instanceof PrubaeModel)) {
	      continue;
	    }
	    ((PrubaeModel)val.getValues().get(j)).initialize(getEditor(),
							     val,
							     val.getValues());
	  }
	}
	val.setName(arg.getName());
	args.set(i, val);
      }
      ((PrubaeModel)args.get(i)).initialize(getEditor(), (PrubaeModel)ret,
					    ret.getValues());
      ret.getValues().add((PrubaeModel)args.get(i));
    }
    //ret.setEditor(getEditor());
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /* ;;; not implemented yet */
  /**
   * Describe <code>makeExternObject</code> method here.
   *
   * @param extfName a <code>String</code> value
   * @param args a <code>List</code> value
   * @param comment a <code>String</code> value
   * @return a <code>ProrateRulebaseElement</code> value
   */
  public ProrateRulebaseElement
    makePartcallObject(String name, String comment) {
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setRegist((short)ProrateRulebaseElement.PARTCALL_CODE);
    ret.setName(name);
    //ret.setEditor(getEditor());
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeFuncObject</code> method here.
   * creates PrubaeModelDoStatement, PrubaeModelJudgeStatement,
   * or PrubaeModelValueStatement according to the context,
   * with built-in function calling rule element.
   * @param regist a <code>short</code> value
   * is the regist code of the built-in function
   * @param args a <code>List</code> value
   * is the list of arguments rule element models.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeFuncObject(short regist, List args, String comment) {
    PrubaeModelStatement	ret = null;
    String	funcName;
    switch (regist) {
    case ProrateRulebaseElement.SET_CODE:
      ret = new PrubaeModelDoStatement();
      funcName = "Set";
      break;
    case ProrateRulebaseElement.EQUAL_CODE:
      ret = new PrubaeModelJudgeStatement();
      funcName = "=";
      break;
    case ProrateRulebaseElement.NEQ_CODE:
      ret = new PrubaeModelJudgeStatement();
      funcName = "!=";
      break;
    case ProrateRulebaseElement.STRINC_CODE:
      ret = new PrubaeModelJudgeStatement();
      funcName = ">=";
      break;
    case ProrateRulebaseElement.Pfm_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Pfm";
      break;
    case ProrateRulebaseElement.Pfm_max_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Pfm_max";
      break;
    case ProrateRulebaseElement.Srp_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Srp";
      break;
    case ProrateRulebaseElement.Np_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Np";
      break;
    case ProrateRulebaseElement.Apply_discount_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Apply_discount";
      break;
    case ProrateRulebaseElement.Country_FUNCTION:
      ret = new PrubaeModelValueStatement();
      funcName = "Country";
      break;
    case ProrateRulebaseElement.Area_FUNCTION:
      ret = new PrubaeModelValueStatement();
      funcName = "Area";
      break;
    case ProrateRulebaseElement.Route_FUNCTION:
      ret = new PrubaeModelValueStatement();
      funcName = "Route";
      break;
    case ProrateRulebaseElement.Via_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Via";
      break;
    case ProrateRulebaseElement.Via_country_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Via_country";
      break;
    case ProrateRulebaseElement.Gateway_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Gateway";
      break;
    case ProrateRulebaseElement.Gateway_country_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Gateway_country";
      break;
    case ProrateRulebaseElement.RoundTheWorld_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "RoundTheWorld_p";
      break;
    case ProrateRulebaseElement.Mileage_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Mileage_p";
      break;
    case ProrateRulebaseElement.Route_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Route_p";
      break;
    case ProrateRulebaseElement.Sidetrip_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Sidetrip_p";
      break;
    case ProrateRulebaseElement.No_amount_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "No_amount_p";
      break;
    case ProrateRulebaseElement.Special_fare_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Special_fare_p";
      break;
    case ProrateRulebaseElement.Promotional_fare_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Promotional_fare_p";
      break;
    case ProrateRulebaseElement.Normal_fare_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Normal_fare_p";
      break;
    case ProrateRulebaseElement.Tax_divid_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Tax_divid";
      break;
    case ProrateRulebaseElement.Via_area_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Via_area";
      break;
    case ProrateRulebaseElement.Discount_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Discount";
      break;
    case ProrateRulebaseElement.Bi_proviso_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Bi_proviso";
      break;
    case ProrateRulebaseElement.Multipath_amt_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Multipath_amt";
      break;
    case ProrateRulebaseElement.Uncertain_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Uncertain";
      break;
    case ProrateRulebaseElement.Number_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Number_p";
      break;
    case ProrateRulebaseElement.Date_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Date_p";
      break;
    case ProrateRulebaseElement.String_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "String_p";
      break;
    case ProrateRulebaseElement.Amount_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Amount_p";
      break;
    case ProrateRulebaseElement.Path_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Path_p";
      break;
    case ProrateRulebaseElement.Error_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Error";
      break;
    case ProrateRulebaseElement.Within_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Within";
      break;
    case ProrateRulebaseElement.Within_country_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Within_country";
      break;
    case ProrateRulebaseElement.Within_area_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Within_area";
      break;
    case ProrateRulebaseElement.Intl_sector_p_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Intl_sector_p";
      break;
    case ProrateRulebaseElement.Between_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Between";
      break;
    case ProrateRulebaseElement.Between_country_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Between_country";
      break;
    case ProrateRulebaseElement.Between_area_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Between_area";
      break;
    case ProrateRulebaseElement.Between_PC_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Between_PC";
      break;
    case ProrateRulebaseElement.Between_PA_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Between_PA";
      break;
    case ProrateRulebaseElement.Between_CA_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Between_CA";
      break;
    case ProrateRulebaseElement.Exclusive_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Exclusive";
      break;
    case ProrateRulebaseElement.Another_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Another";
      break;
    case ProrateRulebaseElement.High_spa_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "High_spa";
      break;
    case ProrateRulebaseElement.Low_spa_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Low_spa";
      break;
    case ProrateRulebaseElement.High_spa_fix_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "High_spa_fix";
      break;
    case ProrateRulebaseElement.Low_spa_fix_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Low_spa_fix";
      break;
    case ProrateRulebaseElement.Net_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Net";
      break;
    case ProrateRulebaseElement.Srp_fix_FUNCTION:
      ret = new PrubaeModelDoStatement();
      funcName = "Srp_fix";
      break;
    case ProrateRulebaseElement.Sia_FUNCTION:
      ret = new PrubaeModelJudgeStatement();
      funcName = "Sia";
      break;
    default:
      ret = new PrubaeModelValueStatement();
      funcName = "";
      break;
    }
    ((PrubaeModel)ret).setComment(comment);
    ((PrubaeModel)ret).setRegist(regist);
    ((PrubaeModel)ret).setEditor(getEditor());
    ret.setName(funcName);
    /* ;;; 2007.03.11
    if (!(ret instanceof PrubaeModelDoStatement)) {
      ret.setStatement(funcName + "(");
    }
    */
    ret.setStatement(funcName + "(");
    if (args.size() <= 0) {
      /* ;;; 2007.03.11
      if (ret instanceof PrubaeModelDoStatement) {
	ret.updateStatement();
      }
      else {
	ret.setStatement(ret.getStatement() + ")");
      }
      */
      ret.setStatement(ret.getStatement() + ")");
      return (ProrateRulebaseElement)ret;
    }
    ret.setValues(new Vector());
    for (int i = 0; i < args.size(); i++) {
      if (i > 0) {
	ret.setStatement(ret.getStatement() + " ");
      }
      if (args.get(i) instanceof PrubaeModelDoStatement) {
	PrubaeModelDoStatement	arg = (PrubaeModelDoStatement)args.get(i);
	PrubaeModelValueStatement
	  val = new PrubaeModelValueStatement();
	val.setComment(arg.getComment());
	val.setRegist(arg.getRegist());
	val.setStatement(arg.getStatement());
	val.setValues(arg.getValues());
	if (val.getValues() != null) {
	  for (int j = 0; j < val.getValues().size(); j++) {
	    if (!(val.getValues().get(j) instanceof PrubaeModel)) {
	      continue;
	    }
	    ((PrubaeModel)val.getValues().get(j)).initialize(getEditor(),
							     val,
							     val.getValues());
	  }
	}
	val.setName(arg.getName());
	args.set(i, val);
      }
      ret.setStatement(ret.getStatement() +
		       ((PrubaeModelValue)args.get(i)).getStatement());
      List	vec = null;
      vec = ret.getValues();
      ((PrubaeModel)args.get(i)).initialize(getEditor(),
					    (PrubaeModel)ret, vec);
      vec.add((PrubaeModel)args.get(i));
    }
    ret.setStatement(ret.getStatement() + ")");
    if (ret instanceof PrubaeModelDoStatement) {
      ret.updateStatement();
    }
    return (ProrateRulebaseElement)ret;
  }

  /* ;;; not implemented yet */
  @Override
    public ProrateRulebaseElement
    makeApplyObject(boolean isSpa, String carrier, String tkCarrier,
		    String comment) {
    // TODO Auto-generated method stub
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    int	regist = (isSpa ?
		  ProrateRulebaseElement.Apply_spa_FUNCTION
		  : ProrateRulebaseElement.Apply_apdp_FUNCTION);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setRegist((short)regist);
    String	funcName = null;
    switch (regist) {
    case ProrateRulebaseElement.Apply_spa_FUNCTION:
      funcName = "Apply_spa";
      break;
    case ProrateRulebaseElement.Apply_apdp_FUNCTION:
      funcName = "Apply_apdp";
      break;
    }
    ret.setName(funcName);

    List	val = new Vector();

    PrubaeModelValue	carrierValue = new PrubaeModelValue();
    if (carrier != null && !carrier.equals("")) {
      carrierValue.setRegist((short)ProrateRulebaseElement.STRING_CODE);
      List	carrierVal = new Vector();
      carrierVal.add(carrier);
      carrierValue.setValues(carrierVal);
      val.add(carrierValue);
      /**/
      carrierValue.setComment(comment);
      carrierValue.setEditor(getEditor());
      /**/
      carrierValue.initialize(getEditor(), (PrubaeModel)ret, val);
    }

    PrubaeModelValue	tkCarrierValue = new PrubaeModelValue();
    if (tkCarrier != null && !tkCarrier.equals("")) {
      tkCarrierValue.setRegist((short)ProrateRulebaseElement.STRING_CODE);
      List	carrierVal = new Vector();
      carrierVal.add(tkCarrier);
      tkCarrierValue.setValues(carrierVal);
      val.add(tkCarrierValue);
      /**/
      tkCarrierValue.setComment(comment);
      tkCarrierValue.setEditor(getEditor());
      /**/
      tkCarrierValue.initialize(getEditor(), (PrubaeModel)ret, val);
    }

    ret.setValues(val);
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /* ;;; not implemented yet */
  /**
   * Describe <code>makeApplyObject</code> method here.
   *
   * @param isSpa a <code>boolean</code> value
   * @param tkCarrier a <code>String</code> value
   * @param comment a <code>String</code> value
   * @return a <code>ProrateRulebaseElement</code> value
   */
  @Override
    public ProrateRulebaseElement
    makeApplyObject(boolean isSpa,
		    ProrateRulebaseElement carrierElement,
		    ProrateRulebaseElement tkCarrierElement,
		    String comment) {
    // TODO Auto-generated method stub
    PrubaeModelDoStatement	ret = new PrubaeModelDoStatement();
    int	regist = (isSpa ?
		  ProrateRulebaseElement.Apply_spa_FUNCTION
		  : ProrateRulebaseElement.Apply_apdp_FUNCTION);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setRegist((short)regist);
    String	funcName = null;
    switch (regist) {
    case ProrateRulebaseElement.Apply_spa_FUNCTION:
      funcName = "Apply_spa";
      break;
    case ProrateRulebaseElement.Apply_apdp_FUNCTION:
      funcName = "Apply_apdp";
      break;
    }
    ret.setName(funcName);

    List	val = new Vector();

    PrubaeModelValue	carrierValue = (PrubaeModelValue)carrierElement;
    if (carrierValue != null) {
      val.add(carrierValue);
      /**/
      carrierValue.setComment(comment);
      carrierValue.setEditor(getEditor());
      /**/
      carrierValue.initialize(getEditor(), (PrubaeModel)ret, val);
    }
    PrubaeModelValue	tkCarrierValue = (PrubaeModelValue)tkCarrierElement;
    if (tkCarrierValue != null) {
      val.add(tkCarrierValue);
      /**/
      tkCarrierValue.setComment(comment);
      tkCarrierValue.setEditor(getEditor());
      /**/
      tkCarrierValue.initialize(getEditor(), (PrubaeModel)ret, val);
    }
    ret.setValues(val);
    ret.updateStatement();
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeNumberObject</code> method here.
   * creates PrubaeModelValueStatement class with Number rule element.
   * @param number a <code>double</code> value is the number value.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeNumberObject(double number, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist((short)ProrateRulebaseElement.NUMBER_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement("" + number);
    ret.setValues(new Vector());
    ret.getValues().add(new Double(number));
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeDateObject</code> method here.
   * creates PrubaeModelValueStatement class with Date rule element.
   * @param month a <code>short</code> value is the month value
   * @param day a <code>short</code> value is the day value.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeDateObject(short month, short day, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist((short)ProrateRulebaseElement.DATE_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement("" + month + "/" + day);
    ret.setValues(new Vector());
    ret.getValues().add(new Short(month));
    ret.getValues().add(new Short(day));
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeStringObject</code> method here.
   * creates PrubaeModelValueStatement class with String rule element.
   * @param str a <code>String</code> value is the string value.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makeStringObject(String str, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist((short)ProrateRulebaseElement.STRING_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement("\"" + str + "\"");
    ret.setValues(new Vector());
    ret.getValues().add(str);
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makePathObject</code> method here.
   * creates PrubaeModelValueStatement class with Path rule element.
   * @param value a <code>List</code> value
   * is the list of the string rule element models of the places.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makePathObject(List value, String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist((short)ProrateRulebaseElement.PATH_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement("");
    ret.setValues(new Vector());
    for (int i = 0; i < value.size(); i++) {
      if (i > 0) {
	ret.setStatement(ret.getStatement() + "-");
      }
      ret.setStatement(ret.getStatement() +
		       ((PrubaeModelValue)value.get(i)).getStatement());
      ((PrubaeModel)value.get(i)).initialize(getEditor(),
					     ret, ret.getValues());
      ret.getValues().add(value.get(i));
    }
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeVarAutoObject</code> method here.
   * creates PrubaeModelValueStatement with auto variable rule element.
   * @param regist a <code>short</code> value
   * is the regist code of the auto variable.
   * @param varName a <code>String</code> value
   * is the name of the auto variable.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement	makeVarAutoObject(short regist,
						  String varName,
						  String comment) {
    PrubaeModelValue	ret = new PrubaeModelValueStatement();
    ret.setRegist(regist);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    ret.setStatement(varName);
    ret.setValues(new Vector());
    ret.getValues().add(varName);
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeTableObject</code> method here.
   * creates PrubaeModelTable class with Table-statement rule element,
   * with 2 dimensions table (which have rows and columns).
   * @param prows a <code>ProrateRulebaseElement</code> value
   * is the table row variable rule element model.
   * @param pcols a <code>ProrateRulebaseElement</code> value
   * is the table column variable rule element model.
   * @param rows a <code>List</code> value
   * is the list of the rule element models of table row values.
   * @param cols a <code>List</code> value
   * is the list of the rule element models of table column values.
   * @param tbls a <code>List</code> value
   * is the list of the rule element models of amount table values.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeTableObject(ProrateRulebaseElement prows,
		    ProrateRulebaseElement pcols,
		    List rows, List cols, List tbls,
		    String comment) {
    PrubaeModelTable	ret = new PrubaeModelTable();
    ret.setRegist((short)ProrateRulebaseElement.TABLE_CODE);
    ret.setComment(comment);
    ret.setEditor(getEditor());
    if (prows != null) {
      ((PrubaeModel)prows).initialize(getEditor(), ret, null);
    }
    ret.setVarRow((PrubaeModel)prows);
    if (prows != null) {
      ((PrubaeModel)pcols).initialize(getEditor(), ret, null);
    }
    ret.setVarCol((PrubaeModel)pcols);
    if (rows != null) {
      ret.setRows(new Vector());
      for (int i = 0; i < rows.size(); i++) {
	if (rows.get(i) != null) {
	  ((PrubaeModel)rows.get(i)).initialize(getEditor(),
						ret, ret.getRows());
	}
	ret.getRows().add(rows.get(i));
      }
    }
    if (cols != null) {
      ret.setCols(new Vector());
      for (int i = 0; i < cols.size(); i++) {
	if (cols.get(i) != null) {
	  ((PrubaeModel)cols.get(i)).initialize(getEditor(),
						ret, ret.getCols());
	}
	ret.getCols().add(cols.get(i));
      }
    }
    if (tbls != null) {
      ret.setTbls(new Vector());
      for (int i = 0; i < tbls.size(); i++) {
	if (tbls.get(i) != null) {
	  ((PrubaeModel)tbls.get(i)).initialize(getEditor(),
						ret, ret.getTbls());
	}
	ret.getTbls().add(tbls.get(i));
      }
    }
    return (ProrateRulebaseElement)ret;
  }

  /**
   * Describe <code>makeTableObject</code> method here.
   * creates PrubaeModelTable class with Table-statement rule element,
   * with 1 dimension table (which have columns only).
   * @param pcols a <code>ProrateRulebaseElement</code> value
   * is the table column variable rule element model.
   * @param cols a <code>List</code> value
   * is the list of the rule element models of table column values.
   * @param tbls a <code>List</code> value
   * is the list of the rule element models of amount table values.
   * @param comment a <code>String</code> value is the comment string.
   * @return a <code>ProrateRulebaseElement</code> value
   * is the created PrubaeModel instance.
   */
  public ProrateRulebaseElement
    makeTableObject(ProrateRulebaseElement pcols, List cols, List tbls,
		    String comment) {
    return makeTableObject(null, pcols, null, cols, tbls, comment);
    /*
      PrubaeModelTable	ret = new PrubaeModelTable();
      ret.setRegist((short)ProrateRulebaseElement.TABLE_CODE);
      ret.setComment(comment);
      ret.setVarCol(null);
      ret.setVarCol((PrubaeModel)pcols);
      ret.setRows(null);
      for (int i = 0; i < cols.size(); i++) {
      ret.getCols().add(cols.get(i));
      }
      for (int i = 0; i < tbls.size(); i++) {
      ret.getTbls().add(tbls.get(i));
      }
      return (ProrateRulebaseElement)ret;
    */
  }

  /**
   * <code>makeErrorObject</code> method	NOP
   *
   * @param error_flag an <code>int</code> value
   * @param error_string a <code>String</code> value
   * @param comment a <code>String</code> value
   * @return a <code>ProrateRulebaseElement</code> value
   */
  public ProrateRulebaseElement	makeErrorObject(int error_flag,
						String error_string,
						String comment) {
    return (ProrateRulebaseElement)null;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="editor"
   */
  public void setEditor(Prubae theEditor) {
    editor = theEditor;
  }

  /**
   * 
   * @uml.property name="editor"
   */
  public Prubae getEditor() {
    return editor;
  }

} /* end class PrubaeModelFactory */
