// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;

import gnu.lists.LList;
import gnu.math.IntNum;

import java.util.List;
import java.util.Vector;

import javax.swing.JTextField;

import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseWriter;

import jp.sourceforge.glj.lisp.Lisp;

/**
 * class <code>PrubaeModelFunction</code>
 * handles top rule model element, such as SPA, APDP or EXTF top rule model.
 * this class has top rule attributes such as carrier, validity, or extf name.
 * this class also has [common] and [actiions] rule model lists.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */
public class PrubaeModelFunction extends PrubaeModel {

  /**
   * constant <code>SPA</code> is the functionType value of SPA.
   *
   */
  public static final int SPA = 1;
  /**
   * constant <code>APDP</code> is the functionType value of APDP.
   *
   */
  public static final int APDP = 2;
  /**
   * constant <code>EXTF</code> is the functionType value of EXTF.
   *
   */
  public static final int EXTF = 3;
  /**
   * constant <code>PART</code> is the functionType value of PART.
   *
   */
  public static final int PART = 4;
  /**
   * constant <code>MINFROMDATE</code> is the minimum valid from date value.
   *
   */
  public static final int MINFROMDATE = 19800101;
  /**
   * constant <code>MAXFROMDATE</code> is the maximum valid from date value.
   *
   */
  public static final int MAXFROMDATE = 99991231;
  /**
   * constant <code>MINTODATE</code> is the minimum valid to date value.
   *
   */
  public static final int MINTODATE = 19800101;
  /**
   * constant <code>MAXTODATE</code> is the maximum valid to date value.
   *
   */
  public static final int MAXTODATE = 99991231;
  /**
   * constant <code>ExtfReturnJudgement</code> is the extfReturnType value of Judgement.
   *
   */
  public static final int ExtfReturnJudgement = 1;
  /**
   * constant <code>ExtfReturnAction</code> is the extfReturnType value of Action.
   *
   */
  public static final int ExtfReturnAction = 2;
  /**
   * constant <code>ExtfReturnValue</code> is the extfReturnType value of Value.
   *
   */
  public static final int ExtfReturnValue = 3;

  // Attributes  
  /**
   * variable <code>functionType</code> is the type of rule function, SPA, APDP, or EXTF.
   * @uml.property  name="functionType"
   */
  private int functionType = 0;
  /**
   * variable <code>carrier</code> is the [carrier] value of SPA or APDP. "" for EXTF and PART.
   * @uml.property  name="carrier"
   */
  private String carrier = null;
  /**
   * variable <code>tkCarrier</code>  is the ]tkcarrier] value of SPA, "" for APDP, EXTF and PART.
   * @uml.property  name="tkCarrier"
   */
  private String tkCarrier = null;
  /**
   * variable <code>name</code>  is the function name of EXTF, the part name of PART, "" for SPA, APDP.
   * @uml.property  name="name"
   */
  private String name = null;
  /**
   * variable <code>fromDate</code>  is the [validity] valid from value of SPA and APDP, nothing to do with EXTF and PART.
   * @uml.property  name="fromDate"
   */
  private int fromDate = 0;
  /**
   * variable <code>toDate</code>  is the [validity] valid until value of SPA and APDP, nothing to do with EXTF and PART.
   * @uml.property  name="toDate"
   */
  private int toDate = 0;
  /**
   * variable <code>common</code> is the list of rule models in [common]. nothing to do with EXTF and PART.
   * @uml.property  name="common"
   */
  private List common = null;
  /**
   * variable <code>actiions</code> is the list of rule models in [actiions].
   * @uml.property  name="actions"
   */
  private List actions = null;
  /**
   * variable <code>extfReturnType</code> is whether the EXTF returns the judgement boolean value, object value, or does action and does not return value, nothing to do with SPA, APDP and PART.
   *
   */
  private int	extfReturnType = ExtfReturnValue;
  /**
   * variable <code>args</code> is the list of the name of argument of EXTF, nothing to do with SPA, APDP and PART.
   *
   */
  private List	args;

  // Constructor  
  /**
   * Creates a new <code>PrubaeModelFunction</code> instance.
   * this creates all GUI elements on part panel of the function,
   * also creates common and action empty lists.
   */
  public PrubaeModelFunction() {
    super();

    setView(new PrubaeViewFunction());
    getView().setModel(this);
    setController(new PrubaeControllerFunction());
    getController().setModel(this);
    getView().setController(getController());
    getController().setView(getView());

    setCommon(new Vector());
    setActions(new Vector());
  }

  // Operations  
  /**
   * <code>close</code> method
   * sends close to all the model elements in common and action lists.
   * and then delete all GUI elements and other items related to this model
   * using super.close()
   */
  public void close() {
    getEditor().updateUrlname(getRegist(),
			      getCarrier(), getTkCarrier(), getName(),
			      getFromDate(), getToDate());
    /* ;;; deBug
    for (int i = 0; i < getCommon().size(); i++) {
      ((PrubaeModel)getCommon().get(i)).close();
    }
    */
    while (getCommon() != null && getCommon().size() > 0) {
      ((PrubaeModel)getCommon().get(0)).close();
    }
    /* ;;; deBug
    for (int i = 0; i < getActions().size(); i++) {
      ((PrubaeModel)getActions().get(i)).close();
    }
    */
    while (getActions() != null && getActions().size() > 0) {
      ((PrubaeModel)getActions().get(0)).close();
    }
    super.close();
  }  

  /**
   * <code>toFilename</code> method
   * returns xml rule filename string.
   * @return a <code>String</code> value is a returned string.
   */
  public String toFilename() {  
    if (getFunctionType() == SPA) {
      return (describe() + getFromDate() + getToDate() + ".xml");
    }
    if (getFunctionType() == APDP) {
      return (describe() + getFromDate() + getToDate() + ".xml");
    }
    if (getFunctionType() == EXTF) {
      return (describe() + ".xml");
    }
    if (getFunctionType() == PART) {
      return (describe() + ".xml");
    }
    return "";
  }
  
  /**
   * <code>toRuleSubdir</code> method
   * returns the name string of the subdirectory of the rule.
   *
   * @return a <code>String</code> value
   */
  public String toRuleSubdir() {
    if (getFunctionType() == SPA) {
      return "spa";
    }
    if (getFunctionType() == APDP) {
      return "apdp";
    }
    if (getFunctionType() == EXTF) {
      return "extf";
    }
    if (getFunctionType() == PART) {
      return "part";
    }
    return "";
  }

  /**
   * <code>describe</code> method
   * returns rule description string for each type.
   * @return a <code>String</code> value is a returned string.
   */
  public String describe() {  
    if (getFunctionType() == SPA) {
      return ("SPA" + "(" + getCarrier() + "," + getTkCarrier()
	      /*+ "/" + getFromDate() + "/" getToDate()*/ + ")");
    }
    if (getFunctionType() == APDP) {
      return ("APDP" + "(" + getCarrier()
	      /*+ "/" + getFromDate() + "/" getToDate()*/ + ")");
    }
    if (getFunctionType() == EXTF) {
      return ("EXTF" + "(" + getName() + ")");
    }
    if (getFunctionType() == PART) {
      return ("PART" + "(" + getName() + ")");
    }
    return "Function";
  }

  /**
   * <code>writeModel</code> method
   * writes every attribute such as carrier, validity, or function name
   * using the writer(PrubaeWriter) instance.
   * it also sends writeModel to all model elements
   * in common and action lists.
   */
  public void writeModel(ProrateRulebaseWriter rulebase) {
    rulebase.writeFunction(this);
  }

  /**
   * <code>getSize</code> method
   * counts the size of this rule and returns the size.
   * the size of all model elements in common and action lists are included.
   * @return an <code>int</code> value
   */
  public int getSize(ProrateRulebaseWriter rulebase) {
    return rulebase.getFunctionSize(this);
  }

  /**
   * <code>updateArgs</code> method
   * updates all rule element models in args list,
   * by sending updateStatement method.
   */
  public void updateArgs() {
    if (getArgs() == null) {
      return;
    }
    System.err.println("updateArgs : " + getArgs().size());
    for (int i = 0; i < getArgs().size(); i++) {
      PrubaeControllerFunction
	cntl = (PrubaeControllerFunction)getController();
      JTextField	argField = (JTextField)cntl.getArgFields().get(i);
      String	arg = argField.getText();
      if (arg.startsWith("$$")) {
	getArgs().set(i, arg);
      }
      else if (arg.startsWith("$")) {
	getArgs().set(i, "$"+arg);
      }
      else {
	getArgs().set(i, "$$"+arg);
      }
    }
    /* modify varList of Prubae in below lines */
    System.err.println("updateArgsCount() : nthCdr 35 prubae.savedVarList = " + Lisp.nthCdr(35, getEditor().getSavedVarList()));
    LList	addedVarList = getEditor().getSavedVarList();
    System.err.println("updateArgsCount() : nthCdr 35 addedVarList = " + Lisp.nthCdr(35, addedVarList));
    if (addedVarList == null) {
      return;
    }
    int	registInit = ProrateRulebaseElement.AUTO_VARIABLE_INITIAL_CODE;
    for (int i = 0; i < getArgs().size(); i++) {
      IntNum	regist = new IntNum(registInit+i);
      Object	obj = getArgs().get(i);
      LList
	insList = 
	Lisp.cons(regist,
		  Lisp.cons(obj,
			    Lisp.cons(obj,
				      Lisp.cons("any", Lisp.nil))));
      addedVarList = Lisp.append(addedVarList, Lisp.cons(insList, Lisp.nil));
    }
    System.err.println("updateArgsCount() : nthCdr 35 addedVarList = " + Lisp.nthCdr(35, addedVarList));
    getEditor().setVarList(addedVarList);	/* modify varList of Prubae */
    System.err.println("prubae.varList nthCdr 35 = "
		       + Lisp.nthCdr(35, getEditor().getVarList()));
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeModelFunction	clone = null;
    clone = (PrubaeModelFunction)super.clone();
    if (getCommon() != null) {
      clone.setCommon((List)((Vector)getCommon()).clone());
      for (int i = 0; i < getCommon().size(); i++) {
	clone.getCommon().set(i, ((PrubaeModel)getCommon().get(i)).clone());
	((PrubaeModel)clone.getCommon().get(i))
	  .initialize(getEditor(), clone, clone.getCommon());
	/*
	((PrubaeControllerIf)clone.getController()).getCommon()
	  .add(((PrubaeModel)clone.getCommon().get(i))
	       .getController().getTreeNode());
	*/
      }
    }
    if (getActions() != null) {
      clone.setActions((List)((Vector)getActions()).clone());
      for (int i = 0; i < getActions().size(); i++) {
	clone.getActions().set(i, ((PrubaeModel)getActions().get(i)).clone());
	((PrubaeModel)clone.getActions().get(i))
	  .initialize(getEditor(), clone, clone.getActions());
	/*
	((PrubaeControllerIf)clone.getController()).getActions()
	  .add(((PrubaeModel)clone.getActions().get(i))
	       .getController().getTreeNode());
	*/
      }
    }
    return clone;
  } 

  /* ;;; 2006.09.12 */
  /**
   * <code>check</code> method	check validity of the prubae model.
   *
   * @param errors a <code>List</code> value
   * @return a <code>boolean</code> value
   */
  public boolean check(List errors) {
    switch (getFunctionType()) {
    case SPA:
      if (getCarrier() == null || getCarrier().equals("")) {
	errors.add("carrier not defined");
	return false;
      }
      if (getTkCarrier() == null || getTkCarrier().equals("")) {
	errors.add("tkCarrier not defined");
	return false;
      }
      if (getFromDate() < MINFROMDATE || getFromDate() > MAXFROMDATE) {
	errors.add("validStart is out of range");
	return false;
      }
      if (getToDate() < MINTODATE || getToDate() > MAXTODATE) {
	errors.add("validEnd is out of range");
	return false;
      }
      break;
    case APDP:
      if (getCarrier() == null || getCarrier().equals("")) {
	errors.add("carrier not defined");
	return false;
      }
      if (getFromDate() < MINFROMDATE || getFromDate() > MAXFROMDATE) {
	errors.add("validStart is out of range");
	return false;
      }
      if (getToDate() < MINTODATE || getToDate() > MAXTODATE) {
	errors.add("validEnd is out of range");
	return false;
      }
      break;
    case EXTF:
      if (getExtfReturnType() < ExtfReturnJudgement
	  || getExtfReturnType() > ExtfReturnValue) {
	errors.add("illegal type defined");
	return false;
      }
      if (getArgs() == null) {
	setArgs(new Vector());
      }
    case PART:
      if (getName() == null || getName().equals("")) {
	errors.add("name not defined");
	return false;
      }
      break;
    default:
      errors.add("illegal rule");
      break;
    }
    if (getActions() == null || getActions().size() <= 0) {
      errors.add("no actions are defined in this rule");
      return false;
    }
    for (int i = 0; i < getCommon().size(); i++) {
      if (!((PrubaeModel)getCommon().get(i)).check(errors)) {
	errors.add("in common statement #" + (i+1));
	return false;
      }
    }
    for (int i = 0; i < getActions().size(); i++) {
      if (!((PrubaeModel)getActions().get(i)).check(errors)) {
	errors.add("in action statement #" + (i+1));
	return false;
      }
    }
    return true;
  }

  /* ;;; 2006.09.13 */
  /**
   * <code>equals</code> method	return if this is equal to the other.
   *
   * @param other a <code>PrubaeModel</code> value
   * @return a <code>boolean</code> value
   */
  public boolean equals(PrubaeModel other) {
    if (super.equals(other)) {
      return true;
    }
    if (!(other instanceof PrubaeModelFunction)) {
      return false;
    }
    PrubaeModelFunction	otherFunc = (PrubaeModelFunction)other;
    return (getFunctionType() == otherFunc.getFunctionType()
	    && ((getCarrier() == null && otherFunc.getCarrier() == null)
		|| (getCarrier() != null && otherFunc.getCarrier() != null
		    && getCarrier().equals(otherFunc.getCarrier())))
	    && ((getTkCarrier() == null && otherFunc.getTkCarrier() == null)
		|| (getTkCarrier() != null && otherFunc.getTkCarrier() != null
		    && getTkCarrier().equals(otherFunc.getTkCarrier())))
	    && ((getName() == null && otherFunc.getName() == null)
		|| (getName() != null && otherFunc.getName() != null
		    && getName().equals(otherFunc.getName())))
	    && getFromDate() == otherFunc.getFromDate()
	    && getToDate() == otherFunc.getToDate());
  }

 // Setters and Getters  
  /**
   * <code>setFunctionType</code> method
   * sets the value of instance variable functionType.
   * it also sets the appropriate regist code value to the variable regist.
   * @param theFunctionType an <code>int</code> value to set the variable.
   * 
   * @uml.property name="functionType"
   */
  public void setFunctionType(int theFunctionType) {
    functionType = theFunctionType;
    switch (functionType) {
    case SPA :
      setRegist((short) ProrateRulebaseElement.SPA_CODE);
      break;
    case APDP :
      setRegist((short) ProrateRulebaseElement.APD_P_CODE);
      break;
    case EXTF :
      setRegist((short) ProrateRulebaseElement.EXTF_CODE);
      break;
    case PART :
      setRegist((short) ProrateRulebaseElement.PART_CODE);
      break;
    }
  }

  /**
   * <code>getFunctionType</code> method
   * gets the value of instance variable functionType.
   * @return an <code>int</code> value of the variable.
   * 
   * @uml.property name="functionType"
   */
  public int getFunctionType() {
    return functionType;
  }

  /**
   * 
   * @uml.property name="carrier"
   */
  public void setCarrier(String theCarrier) {
    carrier = theCarrier;
  }

  /**
   * 
   * @uml.property name="carrier"
   */
  public String getCarrier() {
    return carrier;
  }

  /**
   * 
   * @uml.property name="tkCarrier"
   */
  public void setTkCarrier(String theTkCarrier) {
    tkCarrier = theTkCarrier;
  }

  /**
   * 
   * @uml.property name="tkCarrier"
   */
  public String getTkCarrier() {
    return tkCarrier;
  }

  /**
   * 
   * @uml.property name="name"
   */
  public void setName(String theName) {
    name = theName;
  }

  /**
   * 
   * @uml.property name="name"
   */
  public String getName() {
    return name;
  }

  /**
   * 
   * @uml.property name="fromDate"
   */
  public void setFromDate(int theFromDate) {
    fromDate = theFromDate;
  }

  /**
   * 
   * @uml.property name="fromDate"
   */
  public int getFromDate() {
    return fromDate;
  }

  /**
   * 
   * @uml.property name="toDate"
   */
  public void setToDate(int theToDate) {
    toDate = theToDate;
  }

  /**
   * 
   * @uml.property name="toDate"
   */
  public int getToDate() {
    return toDate;
  }

  /**
   * 
   * @uml.property name="common"
   */
  public void setCommon(List theCommon) {
    common = theCommon;
  }

  /**
   * 
   * @uml.property name="common"
   */
  public List getCommon() {
    return common;
  }

  /**
   * 
   * @uml.property name="actions"
   */
  public void setActions(List theActions) {
    actions = theActions;
  }

  /**
   * 
   * @uml.property name="actions"
   */
  public List getActions() {
    return actions;
  }

  public int getExtfReturnType() {
    return extfReturnType;
  }

  public void setExtfReturnType(int extfReturnType) {
    this.extfReturnType = extfReturnType;
  }

  public List getArgs() {
    return args;
  }

  public void setArgs(List args) {
    this.args = args;
  }

} /* end class PrubaeModelFunction */
