// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;


import java.util.List;
import java.util.Vector;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseWriter;

/**
 * class <code>PrubaeModelIf</code> handles if-statement rule model element.
 * this class has condition statement model as the test condition rule element.
 * this class also has thens and elses rule model lists.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */

public class PrubaeModelIf extends PrubaeModelDo {

  // Attributes  
  /**
   * variable <code>condition</code>
   * is the rule element model of the test condition statement of if statement.
   * 
   * @uml.property name="condition"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  private PrubaeModel condition = null;

  /**
   * variable <code>thens</code> is an list of the rule element models of the action statements, to be evaluated when the test is true, of if statement.
   * @uml.property  name="thens"
   */
  private List thens = null;  
  /**
   * variable <code>elses</code>  is an list of the rule element models of the action statements, to be evaluated when the test is false, of if statement.
   * @uml.property  name="elses"
   */
  private List elses = null;  

  // Constructor  
  /**
   * Creates a new <code>PrubaeModelIf</code> instance.
   * this creates all GUI elements on part panel of the function,
   * also creates condition model, creates thens and elses as empty lists.
   */
  public PrubaeModelIf() {  
    setView(new PrubaeViewIf());
    getView().setModel(this);
    setController(new PrubaeControllerIf());
    getController().setModel(this);
    getView().setController(getController());
    getController().setView(getView());

    setCondition(new PrubaeModelJudge());
    setThens(new Vector());
    setElses(new Vector());
  }  

  // Operations  
  /**
   * <code>initialize</code> method
   * sets on the condition the reference to the editor's top class instance.
   * then sets reference to editor's top class, view and controller,
   * and sets parent and parentList, sends initialize to controller,
   * using super.initialize().
   * @param editor a <code>Prubae</code> value
   * is the reference to the editor's top class instance.
   * @param parent a <code>PrubaeModel</code> value
   * is the reference to the parent model instance.
   * @param parentList a <code>List</code> value
   * is the reference to the parent list in which this model belongs.
   */
  public void initialize(Prubae editor, PrubaeModel parent,
			 List parentList) {
    if (getCondition() != null) {
      getCondition().initialize(editor, this, (List)null);
    }
    super.initialize(editor, parent, parentList);
  }

  /**
   * <code>close</code> method sends close to the condition model.
   * and then sends close to all the model elements in thens and elses lists.
   * and then delete all GUI elements and other items related to this model
   * using super.close()
   */
  public void close() {
    condition.close();
    /* ;;; deBug
    for (int i = 0; i < getThens().size(); i++) {
      ((PrubaeModel)getThens().get(i)).close();
    }
    */
    while (getThens() != null && getThens().size() > 0) {
      ((PrubaeModel)getThens().get(0)).close();
    }
    /* ;;; deBug
    for (int i = 0; i < getElses().size(); i++) {
      ((PrubaeModel)getElses().get(i)).close();
    }
    */
    while (getElses() != null && getElses().size() > 0) {
      ((PrubaeModel)getElses().get(0)).close();
    }
    super.close();
  }  

  /**
   * <code>toString</code> method
   * returns comment string if exists, or returns string "If" otherwise.
   * @return a <code>String</code> value is a returned string 
   */
  public String toString() {  
    if (getComment() != null && !getComment().equals("")) {
      return getComment();
    }
    return "If";
  }  

  /**
   * <code>writeModel</code> method
   * writes TEST_CODE, condition statement, and TEST_END_CODE,
   * using the writer(PrubaeWriter) instance.
   * and then writes two skip size bytes integers.
   * and then writes all thens list elements with sending them writeModel.
   * after that, if required, writes END_CODE and one skip size bytes integer.
   * and then writes all elses list elements with sending them writeModel.
   */
  public void writeModel(ProrateRulebaseWriter rulebase) {
    rulebase.writeIf(this);
  }

  /**
   * <code>getSize</code> method
   * counts the size of this rule and returns the size.
   *
   * @param rulebase a <code>ProrateRulebaseWriter</code> value
   * @return an <code>int</code> value
   */
  public int getSize(ProrateRulebaseWriter rulebase) {
    return rulebase.getIfSize(this);
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeModelIf	clone = null;
    clone = (PrubaeModelIf)super.clone();
    if (getCondition() != null) {
      clone.setCondition((PrubaeModel)((PrubaeModel)getCondition()).clone());
      ((PrubaeModel)clone.getCondition())
	.initialize(getEditor(), clone, (List)null);
    }
    if (getThens() != null) {
      clone.setThens((List)((Vector)getThens()).clone());
      for (int i = 0; i < getThens().size(); i++) {
	clone.getThens().set(i, ((PrubaeModel)getThens().get(i)).clone());
	((PrubaeModel)clone.getThens().get(i))
	  .initialize(getEditor(), clone, clone.getThens());
	/*
	((PrubaeControllerIf)clone.getController()).getThens()
	  .add(((PrubaeModel)clone.getThens().get(i))
	       .getController().getTreeNode());
	*/
      }
    }
    if (getElses() != null) {
      clone.setElses((List)((Vector)getElses()).clone());
      for (int i = 0; i < getElses().size(); i++) {
	clone.getElses().set(i, ((PrubaeModel)getElses().get(i)).clone());
	((PrubaeModel)clone.getElses().get(i))
	  .initialize(getEditor(), clone, clone.getElses());
	/*
	((PrubaeControllerIf)clone.getController()).getElses()
	  .add(((PrubaeModel)clone.getElses().get(i))
	       .getController().getTreeNode());
	*/
      }
    }
    return clone;
  } 

  /* ;;; 2006.09.12 */
  /**
   * <code>check</code> method	check validity of the prubae model.
   *
   * @param errors a <code>List</code> value
   * @return a <code>boolean</code> value
   */
  public boolean check(List errors) {
    if (getCondition() == null) {
      errors.add("condition not defined in If Statement");
      return false;
    }
    if (!getCondition().check(errors)) {
      errors.add("in the condition in If Statement");
      return false;
    }
    for (int i = 0; i < getThens().size(); i++) {
      if (!((PrubaeModel)getThens().get(i)).check(errors)) {
	errors.add("in then statement #" + (i+1) + " in If Statement");
	return false;
      }
    }
    for (int i = 0; i < getElses().size(); i++) {
      if (!((PrubaeModel)getElses().get(i)).check(errors)) {
	errors.add("in else statement #" + (i+1) + " in If Statement");
	return false;
      }
    }
    return true;
  }

  // Setters and Getters  
  /**
   * 
   * @uml.property name="condition"
   */
  public void setCondition(PrubaeModel theCondition) {
    condition = theCondition;
  }

  /**
   * 
   * @uml.property name="condition"
   */
  public PrubaeModel getCondition() {
    return condition;
  }

  /**
   * 
   * @uml.property name="thens"
   */
  public void setThens(List theThens) {
    thens = theThens;
  }

  /**
   * 
   * @uml.property name="thens"
   */
  public List getThens() {
    return thens;
  }

  /**
   * 
   * @uml.property name="elses"
   */
  public void setElses(List theElses) {
    elses = theElses;
  }

  /**
   * 
   * @uml.property name="elses"
   */
  public List getElses() {
    return elses;
  }

} /* end class PrubaeModelIf */
