// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.prubae;


import java.util.List;
import java.util.Vector;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseWriter;

/**
 * class <code>PrubaeModelTable</code>
 * handles table-statement rule model element.
 * this class has varRow and varCol rule element models.
 * also has rows, cols and tbls rule element model lists.
 * @author <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version 1.0
 */

public class PrubaeModelTable extends PrubaeModelDo {

  // Attributes  
  /**
   * variable <code>varRow</code> is the row variable element model.
   * 
   * @uml.property name="varRow"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  private PrubaeModel varRow = null;

  /**
   * variable <code>varCol</code> is the column variable element model.
   * 
   * @uml.property name="varCol"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  private PrubaeModel varCol = null;

  /**
   * variable <code>rows</code> is a list of the rule element models of the table row values.
   * @uml.property  name="rows"
   */
  private List rows = null;  
  /**
   * variable <code>cols</code> is a list of the rule element models of the table column values.
   * @uml.property  name="cols"
   */
  private List cols = null;
  /**
   * variable <code>tbls</code> is a list of the rule element models of the table values.
   * @uml.property  name="tbls"
   */
  private List tbls = null;  

  // Constructor  
  /**
   * Creates a new <code>PrubaeModelTable</code> instance.
   * this creates all GUI elements on part panel of the function,
   */
  public PrubaeModelTable() {
    setVarCol(new PrubaeModelValueStatement());
    getVarCol().setComment("");
    getVarCol().setStatement("");
    getVarCol().initialize(getEditor(), this, null);
    setCols(new Vector());
    setView(new PrubaeViewTable());
    getView().setModel(this);
    setController(new PrubaeControllerTable());
    getController().setModel(this);
    getView().setController(getController());
    getController().setView(getView());
  }  

  // Operations  
  /**
   * Describe <code>initialize</code> method here.
   * sets on the var row/col the reference to the editor's top class instance.
   * then sets reference to editor's top class, view and controller,
   * and sets parent and parentList, sends initialize to controller,
   * using super.initialize().
   * @param editor a <code>Prubae</code> value
   * is the reference to the editor's top class instance.
   * @param parent a <code>PrubaeModel</code> value
   * is the reference to the parent model instance.
   * @param parentList a <code>List</code> value
   * is the reference to the parent list in which this model belongs.
   */
  public void initialize(Prubae editor, PrubaeModel parent,
			 List parentList) {
    if (getVarRow() != null) {
      getVarRow().initialize(editor, this, (List)null);
    }
    if (getVarCol() != null) {
      getVarCol().initialize(editor, this, (List)null);
    }
    super.initialize(editor, parent, parentList);
  }

  /**
   * <code>close</code> method
   *
   */
  public void close() {
    if (getVarRow() != null) {
      ((PrubaeModel)getVarRow()).closeUI();
    }
    if (getVarCol() != null) {
      ((PrubaeModel)getVarCol()).closeUI();
    }
    int	rowsCnt = 0;
    if (getRows() != null) {
      for (int i = 0; i < getRows().size(); i++) {
	((PrubaeModel)getRows().get(i)).closeUI();
	if (((PrubaeModel)getRows().get(i)).getRegist() > 0) {
	  rowsCnt++;
	}
      }
    }
    int	colsCnt = 0;
    if (getCols() != null) {
      for (int i = 0; i < getCols().size(); i++) {
	((PrubaeModel)getCols().get(i)).closeUI();
	if (((PrubaeModel)getCols().get(i)).getRegist() > 0) {
	  colsCnt++;
	}
      }
    }
    if (getTbls() != null) {
      int	index = 0;
      int	numRows = ((getRows() == null) ? 1 : getRows().size());
      for (int i = 0; i < numRows; i++) {
	if (i < rowsCnt) {
	  for (int j = 0; j < getCols().size(); j++) {
	    ((PrubaeModel)getTbls().get(index)).closeUI();
	    if (j < colsCnt) {
	      index++;
	    }
	    else {
	      getTbls().remove(((PrubaeModel)getTbls().get(index)));
	    }
	  }
	}
	else {
	  for (int j = 0; j < getCols().size(); j++) {
	    ((PrubaeModel)getTbls().get(index)).closeUI();
	    getTbls().remove(((PrubaeModel)getTbls().get(index)));
	  }
	}
      }
    }
    super.close();
  }

  /**
   * <code>toString</code> method
   * returns comment string if exists, or returns string "Table" otherwise.
   * @return a <code>String</code> value
   */
  public String toString() {  
    if (getComment() != null && !getComment().equals("")) {
      return getComment();
    }
    return "Table";
  }  

  /**
   * <code>updateVarRow</code> method
   * updates varRow model by sending updateStatement method.
   */
  public void updateVarRow() {
    if (getVarRow() != null) {
      if (getVarRow() instanceof PrubaeModelValueStatement) {
	PrubaeController	cntl =
	  (PrubaeController)((PrubaeModel)getVarRow()).getController();
	if (cntl != null) {
	  cntl.update();
	}
      }
    }
  }

  /**
   * <code>updateVarCol</code> method
   * updates varCol model by sending updateStatement method.
   */
  public void updateVarCol() {
    if (getVarCol() != null) {
      if (getVarCol() instanceof PrubaeModelValueStatement) {
	PrubaeController	cntl =
	  (PrubaeController)((PrubaeModel)getVarCol()).getController();
	if (cntl != null) {
	  cntl.update();
	}
      }
    }
  }

  /**
   * <code>updateRows</code> method
   * updates all rule element models in rows list,
   * by sending updateStatement method.
   */
  public void updateRows() {
    if (getRows() == null) {
      return;
    }
    System.err.println("updateRows : " + getRows().size());
    for (int i = 0; i < getRows().size(); i++) {
      if (getRows().get(i) instanceof PrubaeModelValueStatement) {
	PrubaeController	cntl =
	  (PrubaeController)((PrubaeModel)getRows().get(i)).getController();
	if (cntl != null) {
	  cntl.update();
	}
      }
    }
  }

  /**
   * <code>updateCols</code> method
   * updates all rule element models in cols list,
   * by sending updateStatement method.
   */
  public void updateCols() {
    if (getCols() == null) {
      return;
    }
    System.err.println("updateCols : " + getCols().size());
    for (int i = 0; i < getCols().size(); i++) {
      if (getCols().get(i) instanceof PrubaeModelValueStatement) {
	PrubaeController	cntl =
	  (PrubaeController)((PrubaeModel)getCols().get(i)).getController();
	if (cntl != null) {
	  cntl.update();
	}
      }
    }
  }

  /**
   * <code>writeModel</code> method
   * writes TABLE_CODE, number of columns, and all table row column variables,
   * using the writer(PrubaeWriter) instance.
   * and then writes TABLE_FIELD_BEGIN and all table row values
   * and TABLE_FIELD_END if the table has the 2 dimensions.
   * and then writes TABLE_FIELD_BEGIN and all table column values
   * and TABLE_FIELD_END, and writes all table values, and ENDTABLE_CODE.
   */
  public void writeModel(ProrateRulebaseWriter rulebase) {
    rulebase.writeTable(this);
  }

  /**
   * <code>getSize</code> method 
   * counts the size of this rule and returns the size.
   *
   * @param rulebase a <code>ProrateRulebaseWriter</code> value
   * @return an <code>int</code> value
   */
  public int getSize(ProrateRulebaseWriter rulebase) {
    return rulebase.getTableSize(this);
  }

  /**
   * <code>clone</code> method	create clone object and returns it.
   *
   * @return an <code>Object</code> value
   */
  public Object clone() {
    PrubaeModelTable	clone = null;
    clone = (PrubaeModelTable)super.clone();
    if (getVarRow() != null) {
      clone.setVarRow((PrubaeModel)((PrubaeModel)getVarRow()).clone());
      ((PrubaeModel)clone.getVarRow())
	.initialize(getEditor(), clone, (List)null);
    }
    if (getVarCol() != null) {
      clone.setVarCol((PrubaeModel)((PrubaeModel)getVarCol()).clone());
      ((PrubaeModel)clone.getVarCol())
	.initialize(getEditor(), clone, (List)null);
    }
    if (getRows() != null) {
      clone.setRows((List)((Vector)getRows()).clone());
      for (int i = 0; i < getRows().size(); i++) {
	clone.getRows().set(i, ((PrubaeModel)getRows().get(i)).clone());
	((PrubaeModel)clone.getRows().get(i))
	  .initialize(getEditor(), clone, clone.getRows());
      }
    }
    if (getCols() != null) {
      clone.setCols((List)((Vector)getCols()).clone());
      for (int i = 0; i < getCols().size(); i++) {
	clone.getCols().set(i, ((PrubaeModel)getCols().get(i)).clone());
	((PrubaeModel)clone.getCols().get(i))
	  .initialize(getEditor(), clone, clone.getCols());
      }
    }
    if (getTbls() != null) {
      clone.setTbls((List)((Vector)getTbls()).clone());
      for (int i = 0; i < getTbls().size(); i++) {
	clone.getTbls().set(i, ((PrubaeModel)getTbls().get(i)).clone());
	((PrubaeModel)clone.getTbls().get(i))
	  .initialize(getEditor(), clone, clone.getTbls());
      }
    }
    return clone;
  } 
 
  /* ;;; 2006.09.12 */
  /**
   * <code>check</code> method	check validity of the prubae model.
   *
   * @param errors a <code>List</code> value
   * @return a <code>boolean</code> value
   */
  public boolean check(List errors) {
    if (getVarCol() == null) {
      errors.add("variable not defined in Table Statement");
      return false;
    }
    if (getVarRow() != null && !getVarRow().check(errors)) {
      errors.add("in the row variable in Table Statement");
      return false;
    }
    if (!getVarCol().check(errors)) {
      errors.add("in the col variable in Table Statement");
      return false;
    }
    if (getRows() != null) {
      for (int i = 0; i < getRows().size(); i++) {
	if (!((PrubaeModel)getRows().get(i)).check(errors)) {
	  errors.add("in row value #" + (i+1) + " in Table Statement");
	  return false;
	}
      }
    }
    for (int i = 0; i < getCols().size(); i++) {
      if (!((PrubaeModel)getCols().get(i)).check(errors)) {
	errors.add("in column value #" + (i+1) + " in Table Statement");
	return false;
      }
    }
    for (int i = 0; i < getTbls().size(); i++) {
      if (!((PrubaeModel)getTbls().get(i)).check(errors)) {
	if (getRows() != null) {
	  int	numCols = getCols().size();
	  int	rowNum = i / numCols;
	  int	colNum = i % numCols;
	  errors.add("in value table row #" + rowNum + " col #" + colNum
		     + " in Table Statement");
	}
	else {
	  errors.add("in value table col #" + (i+1) + " in Table Statement");
	}
	return false;
      }
    }
    return true;
  }

 // Setters and Getters  
  /**
   * 
   * @uml.property name="varRow"
   */
  public void setVarRow(PrubaeModel theVarRow) {
    varRow = theVarRow;
  }

  /**
   * 
   * @uml.property name="varRow"
   */
  public PrubaeModel getVarRow() {
    return varRow;
  }

  /**
   * 
   * @uml.property name="varCol"
   */
  public void setVarCol(PrubaeModel theVarCol) {
    varCol = theVarCol;
  }

  /**
   * 
   * @uml.property name="varCol"
   */
  public PrubaeModel getVarCol() {
    return varCol;
  }

  /**
   * 
   * @uml.property name="rows"
   */
  public void setRows(List theRows) {
    rows = theRows;
  }

  /**
   * 
   * @uml.property name="rows"
   */
  public List getRows() {
    return rows;
  }

  /**
   * 
   * @uml.property name="cols"
   */
  public void setCols(List theCols) {
    cols = theCols;
  }

  /**
   * 
   * @uml.property name="cols"
   */
  public List getCols() {
    return cols;
  }

  /**
   * 
   * @uml.property name="tbls"
   */
  public void setTbls(List theTbls) {
    tbls = theTbls;
  }

  /**
   * 
   * @uml.property name="tbls"
   */
  public List getTbls() {
    return tbls;
  }

} /* end class PrubaeModelTable */
