// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.rulebase.xml;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import jp.sourceforge.gnp.prorate.export.Prorate;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseException;

/**
 * class <code>XmlRulebaseCache</code>
 * @author  <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version  1.0
 */

public abstract class XmlRulebaseCache extends XmlRulebase {
  /**
   * <code>initialize</code> method	initialize rulebase (static).
   *
   * @exception Exception if an error occurs
   */
  public static synchronized void	initialize() throws Exception {
    if (rulebaseList != null) {
      return;
    }
    String      ruleDir = System.getProperty("GNP_RULE_DIR");
    if (ruleDir == null) {
      String	propertyFileName = System.getProperty("PRORATE_PROPERTY_FILE");
      if (propertyFileName == null) {
	System.err.println("user.home = " + System.getProperty("user.home"));
	propertyFileName = System.getProperty("user.home");
	System.err.println("file.separator = "
			   + System.getProperty("file.separator"));
	propertyFileName += System.getProperty("file.separator");
	propertyFileName += ".prorate.properties";
	  /* ;;; deBug for Windoze */
	  String	separator = System.getProperty("file.separator");
	  if (separator != null && separator.equals("\\")) {
	    String	homeDir = System.getProperty("user.home");
	    System.err.println("user.home = " + homeDir);
	    System.err.println("file.separator = " + separator);
	    propertyFileName = homeDir + separator + "prorate.properties";
	    System.err.println("WIN: propertyFileName = " + propertyFileName);
	  }
	  /* ;;; deBug for Windoze end */
      }
      InputStream	propertyFile = null;
      try {
	propertyFile = new FileInputStream(propertyFileName);
      }
      catch (FileNotFoundException e) {
	propertyFile = null;
      }
      System.err.println("propertyFileName = " + propertyFileName
			 + ", propertyFile = " + propertyFile);
      InputStream	is = null;
      if (propertyFile == null) {
	ClassLoader	loader = Prorate.class.getClassLoader();
	is = loader.getResourceAsStream(Prorate.propertyFile);
      }
      else {
	is = propertyFile;
      }
      Properties	properties = new Properties();
      try {
	properties.load(is);
      } catch (IOException eProperty) {
	/* ;;; deBug */
	System.err.println("IOException in XmlRulebaseCache.initialize():"
			   + eProperty.getMessage());
	throw new
	  ProrateRulebaseException("IOException in XmlRulebase.initialize()"
				   + " in reading " + Prorate.propertyFile
				   + ": " + eProperty.getMessage(),
				   eProperty);
      }
      if (getAdditionalPropertiesInputStream() != null) {
	try {
	  properties.load(getAdditionalPropertiesInputStream());
	} catch (IOException eAdditionalProperty) {
	  throw new
	    ProrateRulebaseException("IOException in "
				     + "XmlRulebase.initialize() in reading "
				     + "additional property file"  + ": "
				     + eAdditionalProperty.getMessage(),
				     eAdditionalProperty);
	}
      }
      ruleDir =
	properties.getProperty("GNP_RULE_DIR", "/usr/local/gnp/rulebase");
    }

    rulebaseList = new Vector();
    if (!ruleInstall("spa", ruleDir, SPA_DIR, INDEX_SPA, SPA_EXTENT,
		     (short)ProrateRulebaseElement.SPA_CODE)) {
      throw new
	Exception("proration service SPA rulebase initialization failed");
    }
    if (!ruleInstall("apdp", ruleDir, APDP_DIR, INDEX_APDP, APDP_EXTENT,
		     (short)ProrateRulebaseElement.APD_P_CODE)) {
      throw new
	Exception("proration service APDP rulebase initialization failed");
    }
    if (!ruleInstall("extf", ruleDir, EXTF_DIR, INDEX_EXTF, EXTF_EXTENT,
		     (short)ProrateRulebaseElement.EXTF_CODE)) {
      throw new
	Exception("proration service EXTF rulebase initialization failed");
    }
    if (!ruleInstall("part", ruleDir, PART_DIR, INDEX_PART, PART_EXTENT,
		     (short)ProrateRulebaseElement.PART_CODE)) {
      throw new
	Exception("proration service PART rulebase initialization failed");
    }
    return;
  }
  
  /**
   * <code>ruleInstall</code> method	install spa/apdp/extf rulebase(static).
   *
   * @param rule a <code>String</code> value
   * @param root a <code>String</code> value
   * @param dir a <code>String</code> value
   * @param index an <code>int</code> value
   * @param extent a <code>String</code> value
   * @param rule a <code>short</code> value
   * @return a <code>boolean</code> value
   * @throws ProrateRulebaseException 
   */
  static boolean	ruleInstall(String rule, String root, String dir,
				    int index, String extent, short regist)
    throws ProrateRulebaseException {
    String	dirName = root + System.getProperty("file.separator") + dir;
    File	directory = new File(dirName);
    String[]	ruleFiles = directory.list();
    if (ruleFiles == null) {
      return false;
    }
    Arrays.sort(ruleFiles);

    List	rules = new Vector();
    for (int i = 0; i < ruleFiles.length; i++) {
      if (!ruleFiles[i].matches("[a-zA-Z0-9(),]*.xml")) {
	continue;	/* not xml rule file, skip */
      }
      String
	fileName = (dirName + System.getProperty("file.separator")
		    + ruleFiles[i]);
      String	name = null;
      String	type = null;
      String	carrier = null;
      String	tkCarrier = null;
      String	validStart = null;
      String	validEnd = null;
      Date	dateFrom = null;
      Date	dateUntil = null;
      final String	ruleName = rule;
      try {
	Reader	reader = new InputStreamReader(new FileInputStream(fileName));
	String	venderParserClass = "org.apache.xerces.parsers.SAXParser";
	XMLReader
	  xmlReader = XMLReaderFactory.createXMLReader(venderParserClass);
	ContentHandler
	  handler = new ContentHandler() {
	      public void startElement(String namespaceURI, String localName,
				       String qName, Attributes atts)
		throws SAXException {
		if (localName.equals(ruleName)) {
		  RuleFoundException	e = new RuleFoundException();
		  for (int i = 0; i < atts.getLength(); i++) {
		    if (atts.getLocalName(i).equals("name")) {
		      e.name = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("type")) {
		      e.type = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("carrier")) {
		      e.carrier = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("tkCarrier")) {
		      e.tkCarrier = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("validStart")) {
		      e.validStart = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("validEnd")) {
		      e.validEnd = atts.getValue(i);
		    }
		    e.extfArgs = null;
		  }
		  if (localName.equals("extf")) {
		    ruleFoundException = e;
		  }
		  else {
		    throw e;
		  }
		}
		else if (localName.equals("extf-args")) {
		  extfArgs = new Vector();
		}
		else if (localName.equals("extf-arg")) {
		  for (int i = 0; i < atts.getLength(); i++) {
		    if (atts.getLocalName(i).equals("name")) {
		      String	name = atts.getValue(i);
		      extfArgs.add(name);
		    }
		  }
		}
	      }
	      public void endElement(String namespaceURI, String localName,
				     String qName) throws SAXException {
		if (localName.equals("extf-args")) {
		  ruleFoundException.extfArgs = extfArgs;
		  throw ruleFoundException;
		}
	      }
	      public void startDocument() throws SAXException {
	      }
	      public void endDocument() throws SAXException {
	      }
	      public void characters(char[] ch, int start, int length)
		throws SAXException {
	      }
	      public void ignorableWhitespace(char[] ch, int start, int length)
		throws SAXException {
	      }
	      public void endPrefixMapping(String prefix) throws SAXException {
	      }
	      public void skippedEntity(String name) throws SAXException {
	      }
	      public void setDocumentLocator(Locator locator) {
	      }
	      public void processingInstruction(String target, String data)
		throws SAXException {
	      }
	      public void startPrefixMapping(String prefix, String uri)
		throws SAXException {
	      }
	    };
	xmlReader.setContentHandler(handler);
	InputSource	inputSource = new InputSource(reader);
	xmlReader.parse(inputSource);
      }
      catch (RuleFoundException e) {
	name = e.name;
	type = e.type;
	carrier = e.carrier;
	tkCarrier = e.tkCarrier;
	validStart = e.validStart;
	validEnd = e.validEnd;
      }
      catch (SAXException e) {
	throw new
	  ProrateRulebaseException("SAXException in ruleInstall() :"
				   + e.getMessage(),
				   e);
      }
      catch (FileNotFoundException e) {
	throw new
	  ProrateRulebaseException(fileName + " : "
				   + "file not found in ruleInstall() :"
				   + e.getMessage(),
				   e);
      }
      catch (IOException e) {
	throw new
	  ProrateRulebaseException("IOException in ruleInstall() :"
				   + e.getMessage(),
				   e);
      }
      
      if (validStart != null && !validStart.equals("")) {
	try {
	  dateFrom = dtFormat.parse(validStart);
	}
	catch (ParseException e) {
	  throw new
	    ProrateRulebaseException(dateFrom + " : " +
				     "ParseException in ruleInstall() :"
				     + e.getMessage(),
				     e);
	}
      }
      if (validEnd != null && !validEnd.equals("")) {
	try {
	  dateUntil = dtFormat.parse(validEnd);
	}
	catch (ParseException e) {
	  throw new
	    ProrateRulebaseException(dateUntil + " : " +
				     "ParseException in ruleInstall() :"
				     + e.getMessage(),
				     e);
	}
      }
      RuleCacheElement
	ruleElem = new RuleCacheElement(rule, name, type, carrier, tkCarrier,
					dateFrom, dateUntil, fileName,
					null, null);
      rules.add(ruleElem);
    }

    Object[]	rulebase = rules.toArray();
    rulebaseList.add(rulebase);
    return true;
  }

  // Attributes
  /**
   * variable <code>selectedRecord</code>	selected rulebase cache record
   *
   */
  RuleCacheElement	selectedRecord;

  /**
   * variable <code>appliedRules</code>	list of applied rules
   *
   */
  List	appliedRules;

  /**
   * Creates a new <code>XmlRulebaseCache</code> instance.
   *
   * @exception Exception if an error occurs
   */
  public XmlRulebaseCache() throws Exception {
    super();
    setAppliedRules(new Vector());
  }

  /**
   * Creates a new <code>XmlRulebaseCache</code> instance.
   *
   * @param in_filename a <code>String</code> value
   */
  public XmlRulebaseCache(String in_filename) {
    super(in_filename);
  }

  // Operations
  /**
   * <code>selectSPA</code> method
   *
   * @param carrier a <code>String</code> value
   * @param tkCarrier a <code>String</code> value
   * @param dateIssue a <code>String</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean	selectSPA(String carrier, String tkCarrier,
				  String dateIssue)
    throws ProrateRulebaseException {
    if (dateIssue == null || dateIssue.equals("")) {
      return false;
    }
    Object[]	rulebase = (Object[])rulebaseList.get(INDEX_SPA);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleCacheElement("spa", "", "",
					       carrier, tkCarrier,
					       null, null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleCacheElement)o1)
				.compareTo((RuleCacheElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleCacheElement)rulebase[i]).match("spa", "",
						 carrier, tkCarrier)) {
	break;
      }
      index = i;
    }
    Date	date = null;
    try {
      date = dtFormat.parse(dateIssue);
    }
    catch (ParseException e) {
      throw new
	ProrateRulebaseException(dateIssue + " : " +
				 "ParseException in selectSPA() :"
				 + e.getMessage(),
				 e);
    }
    for (int i = index; i < rulebase.length; i++) {
      if (!((RuleCacheElement)rulebase[i]).match("spa", "",
						 carrier, tkCarrier)) {
	return false;
      }
      if (((RuleCacheElement)rulebase[i]).match("spa", "",
						carrier, tkCarrier, date)) {
	setSelectedRecord((RuleCacheElement)rulebase[i]);
	getAppliedRules().add(getSelectedRecord());
	setFilename(((RuleCacheElement)rulebase[i]).filename);
	return true;
      }
    }
    return false;
  }

  /**
   * <code>selectAPDP</code> method
   *
   * @param carrier a <code>String</code> value
   * @param dateIssue a <code>String</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean selectAPDP(String carrier, String dateIssue)
    throws ProrateRulebaseException {
    if (dateIssue == null || dateIssue.equals("")) {
      return false;
    }
    Object[]	rulebase = (Object[])rulebaseList.get(INDEX_APDP);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleCacheElement("apdp", "", "", carrier, "",
					       null, null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleCacheElement)o1)
				.compareTo((RuleCacheElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleCacheElement)rulebase[i]).match("apdp", "", carrier, "")) {
	break;
      }
      index = i;
    }
    Date	date = null;
    try {
      date = dtFormat.parse(dateIssue);
    }
    catch (ParseException e) {
      throw new
	ProrateRulebaseException(dateIssue + " : " +
				 "ParseException in selectAPDP() :"
				 + e.getMessage(),
				 e);
    }
    for (int i = index; i < rulebase.length; i++) {
      if (!((RuleCacheElement)rulebase[i]).match("apdp", "", carrier, "")) {
	return false;
      }
      if (((RuleCacheElement)rulebase[i]).match("apdp", "",
						carrier, "", date)) {
	setSelectedRecord((RuleCacheElement)rulebase[i]);
	getAppliedRules().add(getSelectedRecord());
	setFilename(((RuleCacheElement)rulebase[i]).filename);
	return true;
      }
    }
    return false;
  }

  /**
   * <code>selectEXTF</code> method
   *
   * @param extfName a <code>String</code> value
   * @return a <code>boolean</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean selectEXTF(String extfName) throws ProrateRulebaseException {
    Object[]	rulebase = (Object[])rulebaseList.get(INDEX_EXTF);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleCacheElement("extf", extfName, "", "", "",
					       null, null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleCacheElement)o1)
				.compareTo((RuleCacheElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleCacheElement)rulebase[i]).match("extf", extfName, "", "")) {
	break;
      }
      index = i;
    }
    if (!((RuleCacheElement)rulebase[index]).match("extf", extfName, "", "")) {
      throw new
	ProrateRulebaseException(extfName + " : "
				 + "EXTF not found",
				 null);
    }
    setSelectedRecord((RuleCacheElement)rulebase[index]);
    getAppliedRules().add(getSelectedRecord());
    setFilename(((RuleCacheElement)rulebase[index]).filename);
    return true;
  }

  /**
   * <code>selectPART</code> method
   *
   * @param name a <code>String</code> value
   * @return a <code>boolean</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean selectPART(String name) throws ProrateRulebaseException {
    Object[]	rulebase = (Object[])rulebaseList.get(INDEX_PART);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleCacheElement("part", name, "", "", "",
					       null, null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleCacheElement)o1)
				.compareTo((RuleCacheElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleCacheElement)rulebase[i]).match("part", name, "", "")) {
	break;
      }
      index = i;
    }
    if (!((RuleCacheElement)rulebase[index]).match("part", name, "", "")) {
      throw new
	ProrateRulebaseException(name + " : "
				 + "PART not found",
				 null);
    }
    setSelectedRecord((RuleCacheElement)rulebase[index]);
    getAppliedRules().add(getSelectedRecord());
    setFilename(((RuleCacheElement)rulebase[index]).filename);
    return true;
  }

  /**
   * <code>read</code> method
   *
   * @return a <code>List</code> value
   * @throws ProrateRulebaseException 
   */
  public List read() throws ProrateRulebaseException {
    List	rules = null;
    if ((rules = getSelectedRecord().getRules()) != null
	&& rules.size() > 0) {
      getSelectedRecord().setCached(true);
      return rules;
    }
    try {
      rules = read(false);
    }
    catch (Exception e) {
      throw new ProrateRulebaseException("rulebase read error", e);
    }
    getSelectedRecord().setCached(false);
    getSelectedRecord().setRules(rules);
    return rules;
  }

  public RuleCacheElement getSelectedRecord() {
    return selectedRecord;
  }

  public void setSelectedRecord(RuleCacheElement selectedRecord) {
    this.selectedRecord = selectedRecord;
  }

  public List getAppliedRules() {
    return appliedRules;
  }

  public void setAppliedRules(List appliedRules) {
    this.appliedRules = appliedRules;
  }
  
}

class RuleCacheElement extends RuleElement {

  /**
   * variable <code>rules</code>	rules loaded
   *
   */
  List	rules;
  /**
   * variable <code>isCached</code>	whether the rules are cached or not
   *
   */
  boolean	isCached;
  
  /**
   * Creates a new <code>RuleElement</code> instance.
   *
   */
  RuleCacheElement() {
    super();
  }

  /**
   * Creates a new <code>RuleCacheElement</code> instance.
   *
   * @param rule a <code>String</code> value
   * @param name a <code>String</code> value
   * @param type a <code>String</code> value
   * @param carrier a <code>String</code> value
   * @param tkCarrier a <code>String</code> value
   * @param dateFrom a <code>Date</code> value
   * @param dateUntil a <code>Date</code> value
   * @param filename a <code>String</code> value
   * @param extfArgs a <code>List</code> value
   * @param rules a <code>List</code> value
   */
  RuleCacheElement(String rule, String name, String type,
		   String carrier, String tkCarrier,
		   Date dateFrom, Date dateUntil, String filename,
		   List extfArgs, List rules) {
    super(rule, name, type, carrier, tkCarrier, dateFrom, dateUntil, filename,
	  extfArgs);
    this.setRules(rules);
  }

  void	removeRules() {
    if (getRules() != null && getRules().size() > 0) {
      for (int i = 0; i < getRules().size(); i++) {
	getRules().set(i, null);
      }
      setRules(null);
    }
  }

  public List getRules() {
    return rules;
  }

  public void setRules(List rules) {
    this.rules = rules;
  }

  public boolean isCached() {
    return isCached;
  }

  public void setCached(boolean isCached) {
    this.isCached = isCached;
  }
}
