// Copyright (c) 2002  Hitoshi Guutara Maruyama.
// This is free software;  for terms and warranty disclaimer see ./COPYING.

package jp.sourceforge.gnp.rulebase.xml;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.lang.StringUtils;

import java.util.regex.Pattern;
import java.util.regex.Matcher;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.Reader;
import java.security.AccessControlException;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import jp.sourceforge.gnp.prorate.export.Prorate;
import jp.sourceforge.gnp.prorate.export.ProrateAudit;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElement;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseElementFactory;
import jp.sourceforge.gnp.rulebase.ProrateRulebaseException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

/**
 * class <code>XmlRulebase</code>
 * @author  <a href="mailto:gnp@sourceforge.jp">Hitoshi Guutara Maruyama</a>
 * @version  1.0
 */

public class XmlRulebaseUrl extends XmlRulebase {

  /**
   * <code>serialVersionUID</code> Υ
   */
  private static final long serialVersionUID = 1L;

  /**
   * <code>initialize</code> method	initialize rulebase (static).
   *
   * @param ruleUrl a <code>String</code> value
   * @exception Exception if an error occurs
   */
  public static synchronized void initialize(String ruleUrl) throws Exception {
    System.err.println("XmlRulebaseUrl.initialize()");
    if (getRulebaseList() != null) {
      System.err.println("XmlRulebaseUrl.initialize(): rulebaseList != null");
      return;
    }
    if (ruleUrl == null) {
      System.err.println("XmlRulebaseUrl.initialize(): rulebaseList == null");
      try {
	ruleUrl = System.getProperty("GNP_RULE_URL");
	System.err.println("XmlRulebaseUrl.initialize(null): GNP_RULE_URL, ruleUrl = " + ruleUrl);
	if (ruleUrl == null) {
	  String	propertyFileName = System.getProperty("PRORATE_PROPERTY_FILE");
	  if (propertyFileName == null) {
	    System.err.println("user.home = "
			       + System.getProperty("user.home"));
	    propertyFileName = System.getProperty("user.home");
	    System.err.println("file.separator = "
			       + System.getProperty("file.separator"));
	    propertyFileName += System.getProperty("file.separator");
	    propertyFileName += ".prorate.properties";
	    /* ;;; deBug for Windoze */
	    String	separator = System.getProperty("file.separator");
	    if (separator != null && separator.equals("\\")) {
	      String	homeDir = System.getProperty("user.home");
	      System.err.println("user.home = " + homeDir);
	      System.err.println("file.separator = " + separator);
	      propertyFileName = homeDir + separator + "prorate.properties";
	      System.err.println("WIN: propertyFileName = "
				 + propertyFileName);
	  }
	  /* ;;; deBug for Windoze end */
	  }
	  InputStream	propertyFile = null;
	  try {
	    propertyFile = new FileInputStream(propertyFileName);
	  }
	  catch (FileNotFoundException e) {
	    propertyFile = null;
	  }
	  System.err.println("propertyFileName = " + propertyFileName
			     + ", propertyFile = " + propertyFile);
	  InputStream	is = null;
	  if (propertyFile == null) {
	    ClassLoader	loader = Prorate.class.getClassLoader();
	    is = loader.getResourceAsStream(Prorate.propertyFile);
	  }
	  else {
	    is = propertyFile;
	  }
	  Properties	properties = new Properties();
	  try {
	    properties.load(is);
	  } catch (IOException eProperty) {
	    /* ;;; deBug */
	    System.err.println("IOException in XmlRulebaseUrl.initialize(null):"
			       + eProperty.getMessage());
	    throw new
	      ProrateRulebaseException("IOException in "
				       + "XmlRulebaseUrl.initialize(null)"
				       + " in reading " + Prorate.propertyFile
				       + ": " + eProperty.getMessage(),
				       eProperty);
	  }
	  if (getAdditionalPropertiesInputStream() != null) {
	    try {
	      properties.load(getAdditionalPropertiesInputStream());
	    } catch (IOException eAdditionalProperty) {
	      throw new
		ProrateRulebaseException("IOException in "
					 + "XmlRulebaseUrl.initialize(null)"
					 + " in reading "
					 + "additional property file"  + ": "
					 + eAdditionalProperty.getMessage(),
					 eAdditionalProperty);
	    }
	  }
	  ruleUrl =
	    properties.getProperty("GNP_RULE_URL",
				   "http://localhost/gnp/rulebase");
	}
	System.err.println("XmlRulebaseUrl.initialize(null): ruleUrl = "
			   + ruleUrl);
      }
      catch (AccessControlException e) {
	System.err.println("AccessControlException e = " + e.getMessage());
	if (getRulebaseList() == null) {
	  setRulebaseList(new Vector());
	}
      }
    }

    setRulebaseList(new Vector());
    if (getSkipInitializeList().contains(INDEX_SPA)) {
      getRulebaseList().add(null);
    }
    else if (!ruleInstall("spa", ruleUrl, SPA_DIR, INDEX_SPA, SPA_EXTENT,
		     (short)ProrateRulebaseElement.SPA_CODE)) {
      throw new
	Exception("proration service SPA rulebase initialization failed");
    }
    if (getSkipInitializeList().contains(INDEX_APDP)) {
      getRulebaseList().add(null);
    }
    else if (!ruleInstall("apdp", ruleUrl, APDP_DIR, INDEX_APDP, APDP_EXTENT,
			  (short)ProrateRulebaseElement.APD_P_CODE)) {
      throw new
	Exception("proration service APDP rulebase initialization failed");
    }
    if (getSkipInitializeList().contains(INDEX_EXTF)) {
      getRulebaseList().add(null);
    }
    else if (!ruleInstall("extf", ruleUrl, EXTF_DIR, INDEX_EXTF, EXTF_EXTENT,
		     (short)ProrateRulebaseElement.EXTF_CODE)) {
      throw new
	Exception("proration service EXTF rulebase initialization failed");
    }
    if (getSkipInitializeList().contains(INDEX_PART)) {
      getRulebaseList().add(null);
    }
    else if (!ruleInstall("part", ruleUrl, PART_DIR, INDEX_PART, PART_EXTENT,
		     (short)ProrateRulebaseElement.PART_CODE)) {
      throw new
	Exception("proration service PART rulebase initialization failed");
    }
    return;
  }
  
  /**
   * <code>ruleInstall</code> method	install spa/apdp/extf rulebase(static).
   *
   * @param rule a <code>String</code> value
   * @param root a <code>String</code> value
   * @param dir a <code>String</code> value
   * @param index an <code>int</code> value
   * @param extent a <code>String</code> value
   * @param rule a <code>short</code> value
   * @return a <code>boolean</code> value
   * @throws ProrateRulebaseException 
   */
  static boolean	ruleInstall(String rule, String root, String dir,
				    int index, String extent, short regist)
    throws ProrateRulebaseException {
    String	dirName = root + "/" + dir;
    System.err.println("XmlRulebaseUrl.ruleInstall(" + rule + ","
		       + root + "," + dir + "," + index + "," + extent + ","
		       + regist + ")");
    HttpClient client = new HttpClient();
    String	dirUrl = dirName + "/";
    HttpMethod method = new GetMethod( dirUrl );
    String	response = null;
    try {
      client.executeMethod( method );
      response = method.getResponseBodyAsString();
    }
    catch (HttpException e) {
      /* ;;; deBug */
      System.err.println("HttpException in parse() for " + dirUrl + ":"
			 + e.getMessage());
      throw new
	ProrateRulebaseException(dirUrl + " : "
				 + "HttpException in ruleInstall() :"
				 + e.getMessage(),
				 e);
    }
    catch (IOException e) {
      /* ;;; deBug */
      System.err.println("IOException in parse() for " + dirUrl + ":"
			 + e.getMessage());
      throw new
	ProrateRulebaseException("IOException in ruleInstall() :"
				 + e.getMessage(),
				 e);
    }
    finally {
      method.releaseConnection();
    }
    String[]	lines = StringUtils.split(response, "\n");
    String
      patternStr =
      (".*<[Aa] [Hh][Rr][Ee][Ff]=\\\"(" + dir.toUpperCase()
       + "\\(.*\\).*\\.xml)\\\">" + dir.toUpperCase() + ".*</[Aa]>.*");
    Pattern	pattern = Pattern.compile(patternStr);
    List	ruleFilesList = new Vector();
    for (int i = 0; i < lines.length; i++) {
      Matcher	matcher = pattern.matcher(lines[i]);
      boolean	match = false;
      if ((match = matcher.find())) {
	String	ruleFile = matcher.group(1);
	ruleFilesList.add(ruleFile);
      }
    }
    String[]	ruleFiles = new String[ruleFilesList.size()];
    if (ruleFiles == null) {
      return false;
    }
    for (int i = 0; i < ruleFilesList.size(); i++) {
      ruleFiles[i] = (String)ruleFilesList.get(i);
    }
    Arrays.sort(ruleFiles);

    List	rules = new Vector();
    for (int i = 0; i < ruleFiles.length; i++) {
      if (!ruleFiles[i].matches("[a-zA-Z0-9(),]*.xml")) {
	continue;	/* not xml rule file, skip */
      }
      String	fileUrl = (dirName + "/" + ruleFiles[i]);
      String	name = null;
      String	type = null;
      String	carrier = null;
      String	tkCarrier = null;
      String	validStart = null;
      String	validEnd = null;
      Date	dateFrom = null;
      Date	dateUntil = null;
      final String	ruleName = rule;
      method = new GetMethod( fileUrl );
      try {
	client.executeMethod( method );
	String ruleResponse = method.getResponseBodyAsString();
	Reader	reader = new StringReader(ruleResponse);
	String	venderParserClass = "org.apache.xerces.parsers.SAXParser";
	XMLReader
	  xmlReader = XMLReaderFactory.createXMLReader(venderParserClass);
	ContentHandler
	  handler = new ContentHandler() {
	      public void startElement(String namespaceURI, String localName,
				       String qName, Attributes atts)
		throws SAXException {
		if (localName.equals(ruleName)) {
		  RuleFoundException	e = new RuleFoundException();
		  for (int i = 0; i < atts.getLength(); i++) {
		    if (atts.getLocalName(i).equals("name")) {
		      e.name = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("type")) {
		      e.type = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("carrier")) {
		      e.carrier = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("tkCarrier")) {
		      e.tkCarrier = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("validStart")) {
		      e.validStart = atts.getValue(i);
		    }
		    else if (atts.getLocalName(i).equals("validEnd")) {
		      e.validEnd = atts.getValue(i);
		    }
		    e.extfArgs = null;
		  }
		  if (localName.equals("extf")) {
		    ruleFoundException = e;
		  }
		  else {
		    throw e;
		  }
		}
		else if (localName.equals("extf-args")) {
		  extfArgs = new Vector();
		}
		else if (localName.equals("extf-arg")) {
		  for (int i = 0; i < atts.getLength(); i++) {
		    if (atts.getLocalName(i).equals("name")) {
		      String	name = atts.getValue(i);
		      extfArgs.add(name);
		    }
		  }
		}
	      }
	      public void endElement(String namespaceURI, String localName,
				     String qName) throws SAXException {
		if (localName.equals("extf-args")) {
		  ruleFoundException.extfArgs = extfArgs;
		  throw ruleFoundException;
		}
	      }
	      public void startDocument() throws SAXException {
	      }
	      public void endDocument() throws SAXException {
	      }
	      public void characters(char[] ch, int start, int length)
		throws SAXException {
	      }
	      public void ignorableWhitespace(char[] ch, int start, int length)
		throws SAXException {
	      }
	      public void endPrefixMapping(String prefix) throws SAXException {
	      }
	      public void skippedEntity(String name) throws SAXException {
	      }
	      public void setDocumentLocator(Locator locator) {
	      }
	      public void processingInstruction(String target, String data)
		throws SAXException {
	      }
	      public void startPrefixMapping(String prefix, String uri)
		throws SAXException {
	      }
	    };
	xmlReader.setContentHandler(handler);
	InputSource	inputSource = new InputSource(reader);
	xmlReader.parse(inputSource);
      }
      catch (RuleFoundException e) {
	name = e.name;
	type = e.type;
	carrier = e.carrier;
	tkCarrier = e.tkCarrier;
	validStart = e.validStart;
	validEnd = e.validEnd;
	extfArgs = e.extfArgs;
      }
      catch (SAXException e) {
	/* ;;; deBug */
	System.err.println("SAXException in parse() for " + fileUrl + ":"
			   + e.getMessage());
	throw new
	  ProrateRulebaseException("SAXException in ruleInstall() :"
				   + e.getMessage(),
				   e);
      }
      catch (HttpException e) {
	/* ;;; deBug */
	System.err.println("HttpException in parse() for " + fileUrl + ":"
			   + e.getMessage());
	throw new
	  ProrateRulebaseException(fileUrl + " : "
				   + "HttpException in ruleInstall() :"
				   + e.getMessage(),
				   e);
      }
      catch (IOException e) {
	/* ;;; deBug */
	System.err.println("IOException in parse() for " + fileUrl + ":"
			   + e.getMessage());
	throw new
	  ProrateRulebaseException("IOException in ruleInstall() :"
				   + e.getMessage(),
				   e);
      }
      finally {
	method.releaseConnection();
      }
      
      if (validStart != null && !validStart.equals("")) {
	try {
	  dateFrom = dtFormat.parse(validStart);
	}
	catch (ParseException e) {
	  throw new
	    ProrateRulebaseException(dateFrom + " : " +
				     "ParseException in ruleInstall() :"
				     + e.getMessage(),
				     e);
	}
      }
      if (validEnd != null && !validEnd.equals("")) {
	try {
	  dateUntil = dtFormat.parse(validEnd);
	}
	catch (ParseException e) {
	  throw new
	    ProrateRulebaseException(dateUntil + " : " +
				     "ParseException in ruleInstall() :"
				     + e.getMessage(),
				     e);
	}
      }
      RuleElement
	ruleElem = new RuleElement(rule, name, type, carrier, tkCarrier,
				   dateFrom, dateUntil, fileUrl, extfArgs);
      System.err.println("XmlRulebaseUrl.ruleInstall(): new RuleElement("
			 + rule + "," + name + "," + type
			 + "," + carrier + "," + tkCarrier
			 + "," + dateFrom + "," + dateUntil
			 + "," + fileUrl + ")");
      rules.add(ruleElem);
    }

    Object[]	rulebase = rules.toArray();
    getRulebaseList().add(rulebase);
    return true;
  }
  
  public static List	getAllExtfs(String type) {
    List	list = new Vector();
    if (getRulebaseList().size() <= INDEX_EXTF) {
      return list;
    }
    Object[]	rulebase = (Object[])getRulebaseList().get(INDEX_EXTF);
    for (int i = 0; i < rulebase.length; i++) {
      if (((RuleElement)rulebase[i]).type.equals(type)) {
	List	extfInfo = new Vector();
	extfInfo.add(((RuleElement)rulebase[i]).name);
	List	args = ((RuleElement)rulebase[i]).args;
	if (args != null) {
	  for (int j = 0; j < args.size(); j++) {
	    extfInfo.add(args.get(j));
	  }
	}
	list.add(extfInfo);
      }
    }
    return list;
  }
  
  public static List	getAllActionExtfs() {
    return getAllExtfs("action");
  }

  public static List	getAllJudgeExtfs() {
    return getAllExtfs("judge");
  }

  public static List	getAllValueExtfs() {
    return getAllExtfs("value");
  }

  public static List	getAllParts() {
    List	list = new Vector();
    if (getRulebaseList().size() <= INDEX_PART) {
      return list;
    }
    Object[]	rulebase = (Object[])getRulebaseList().get(INDEX_PART);
    for (int i = 0; i < rulebase.length; i++) {
      list.add(((RuleElement)rulebase[i]).name);
    }
    return list;
  }

  /**
   * variable <code>url</code>	selected rulebase file name
   * @uml.property  name="url"
   */
  private String url = "";

  /**
   * variable <code>elementFactory</code>
   * 	롼١Υ롼ιǥ֥󥿡ե
   * 
   * @uml.property name="elementFactory"
   * @uml.associationEnd multiplicity="(0 1)"
   */
  protected ProrateRulebaseElementFactory elementFactory;


  // Constructor
  /**
   * Creates a new <code>XmlRulebaseUrl</code> instance.
   *
   */
  public XmlRulebaseUrl() {
    super();
  }

  /**
   * Creates a new <code>XmlRulebaseUrl</code> instance.
   *
   * @param in_url a <code>String</code> value
   */
  public XmlRulebaseUrl(String in_url) {
    super();
    setUrl(in_url);
  }

  // Operations
  /**
   * <code>selectSPA</code> method
   *
   * @param carrier a <code>String</code> value
   * @param tkCarrier a <code>String</code> value
   * @param dateIssue a <code>String</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean	selectSPA(String carrier, String tkCarrier,
				  String dateIssue)
    throws ProrateRulebaseException {
    if (dateIssue == null || dateIssue.equals("")) {
      return false;
    }
    Object[]	rulebase = (Object[])getRulebaseList().get(INDEX_SPA);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleElement("spa", "", "", carrier, tkCarrier,
					  null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleElement)o1).compareTo((RuleElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleElement)rulebase[i]).match("spa", "", carrier, tkCarrier)) {
	break;
      }
      index = i;
    }
    Date	date = null;
    try {
      date = dtFormat.parse(dateIssue);
    }
    catch (ParseException e) {
      throw new
	ProrateRulebaseException(dateIssue + " : " +
				 "ParseException in selectSPA() :"
				 + e.getMessage(),
				 e);
    }
    for (int i = index; i < rulebase.length; i++) {
      if (!((RuleElement)rulebase[i]).match("spa", "", carrier, tkCarrier)) {
	return false;
      }
      if (((RuleElement)rulebase[i]).match("spa", "",
					   carrier, tkCarrier, date)) {
	setUrl(((RuleElement)rulebase[i]).filename);
	return true;
      }
    }
    return false;
  }

  /**
   * <code>selectAPDP</code> method
   *
   * @param carrier a <code>String</code> value
   * @param dateIssue a <code>String</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean selectAPDP(String carrier, String dateIssue)
    throws ProrateRulebaseException {
    if (dateIssue == null || dateIssue.equals("")) {
      return false;
    }
    Object[]	rulebase = (Object[])getRulebaseList().get(INDEX_APDP);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleElement("apdp", "", "", carrier, "",
					  null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleElement)o1).compareTo((RuleElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleElement)rulebase[i]).match("apdp", "", carrier, "")) {
	break;
      }
      index = i;
    }
    Date	date = null;
    try {
      date = dtFormat.parse(dateIssue);
    }
    catch (ParseException e) {
      throw new
	ProrateRulebaseException(dateIssue + " : " +
				 "ParseException in selectAPDP() :"
				 + e.getMessage(),
				 e);
    }
    for (int i = index; i < rulebase.length; i++) {
      if (!((RuleElement)rulebase[i]).match("apdp", "", carrier, "")) {
	return false;
      }
      if (((RuleElement)rulebase[i]).match("apdp", "", carrier, "", date)) {
	setUrl(((RuleElement)rulebase[i]).filename);
	return true;
      }
    }
    return false;
  }

  /**
   * <code>selectEXTF</code> method
   *
   * @param extfName a <code>String</code> value
   * @return a <code>boolean</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean selectEXTF(String extfName) throws ProrateRulebaseException {
    Object[]	rulebase = (Object[])getRulebaseList().get(INDEX_EXTF);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleElement("extf", extfName, "", "", "",
					  null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleElement)o1).compareTo((RuleElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleElement)rulebase[i]).match("extf", extfName, "", "")) {
	break;
      }
      index = i;
    }
    if (!((RuleElement)rulebase[index]).match("extf", extfName, "", "")) {
      throw new
	ProrateRulebaseException(extfName + " : "
				 + "EXTF not found",
				 null);
    }
    setUrl(((RuleElement)rulebase[index]).filename);
    return true;
  }

  /**
   * <code>selectPART</code> method
   *
   * @param name a <code>String</code> value
   * @return a <code>boolean</code> value
   * @throws ProrateRulebaseException 
   */
  public boolean selectPART(String name) throws ProrateRulebaseException {
    Object[]	rulebase = (Object[])getRulebaseList().get(INDEX_PART);
    int	index =
      Arrays.binarySearch(rulebase,
			  new RuleElement("part", name, "", "", "",
					  null, null, null, null),
			  new Comparator() {
			    public int compare(Object o1, Object o2) {
			      return
				((RuleElement)o1).compareTo((RuleElement)o2);
			    }
			  }
			  );
    if (index < 0) {
      return false;
    }
    for (int i = index; i > 0; i --) {
      if (!((RuleElement)rulebase[i]).match("part", name, "", "")) {
	break;
      }
      index = i;
    }
    if (!((RuleElement)rulebase[index]).match("part", name, "", "")) {
      throw new
	ProrateRulebaseException(name + " : "
				 + "PART not found",
				 null);
    }
    setUrl(((RuleElement)rulebase[index]).filename);
    return true;
  }
  
  /**
   * <code>close</code> method	close proration rulebase selection
   *
   * @param rule a <code>ProrateRule</code> value
   */
  public void	close() {
  }

  /**
   * <code>read</code> method
   *
   * @return a <code>List</code> value
   * @throws ProrateRulebaseException 
   */
  public List read() throws ProrateRulebaseException {
    List	rules = null;
    try {
      rules = read(false);
    }
    catch (Exception e) {
      throw new ProrateRulebaseException("rulebase read error", e);
    }
    return rules;
  }

  /**
   * <code>read</code> method
   *
   * @param skipToActions a <code>boolean</code> value
   * @return a <code>List</code> value
   * @throws ProrateRulebaseException 
   */
  List read(boolean skipToActions) throws ProrateRulebaseException {
    HttpClient client = new HttpClient();
    HttpMethod method = new GetMethod( getUrl() );
    try {
      client.executeMethod( method );
      String	response = method.getResponseBodyAsString();
      Reader	reader = new StringReader(response);
      return read(reader, skipToActions);
    }
    catch (IOException e) {
      throw new
	ProrateRulebaseException("IOException in read() :"
				 + e.getMessage(),
				 e);
    }
  }

  /**
   * <code>read</code> method
   *
   * @param reader a <code>Reader</code> value
   * @return a <code>List</code> value
   * @throws ProrateRulebaseException 
   */
  public List read(Reader reader) throws ProrateRulebaseException {
    return read(reader, false);
  }

  /**
   * <code>read</code> method
   *
   * @param reader a <code>Reader</code> value
   * @param skipToActions a <code>boolean</code> value
   * @return a <code>List</code> value
   * @throws ProrateRulebaseException 
   */
  List read(Reader reader, boolean skipToActions)
    throws ProrateRulebaseException {
    List	ret = null;
    try {
      SAXBuilder	builder = new SAXBuilder();
      Document	doc = builder.build(reader);
      ret = loadFromElement(doc.getRootElement(), skipToActions);
      if (ret == null || ret.size() == 0) {
	return ret;
      }
      ret = (List)ret.get(0);
    }
    catch (IOException e) {
      throw new
	ProrateRulebaseException("IOException in read("
				 + reader.toString() + ") :"
				 + e.getMessage(),
				 e);
    }
    catch (JDOMException e) {
      throw new
	ProrateRulebaseException("JDOMException in read("
				 + reader.toString() + ") :"
				 + e.getMessage(),
				 e);
    }
    catch (Exception e) {
      throw new
	ProrateRulebaseException("Exception in read("
				 + reader.toString() + ") :"
				 + e.getMessage(),
				 e);
    }
    return ret;
  }

  /**
   * <code>read</code> method	read rule from rulebase
   *
   * @param isSPA a <code>boolean</code> value
   * @param carrier a <code>String</code> value
   * @param tkCarrier a <code>String</code> value
   * @param issueDate a <code>String</code> value
   * @param skipToActions a <code>boolean</code> value
   * @return a <code>List</code> value
   * @exception ProrateRulebaseException
   */
  public List	read(boolean isSPA, String carrier, String tkCarrier,
		     String issueDate, boolean skipToActions)
    throws ProrateRulebaseException {
    List	rules = new Vector();

    if (isSPA) {
      if (!selectSPA(carrier, tkCarrier, issueDate)) {
	String	str = "Apply_SPA(";
	str += carrier;
	str += ",";
	str += tkCarrier;
	str += ",";
	str += issueDate;
	str += ")";
	str += ": undefined SPA rule";
	ProrateRulebaseElement
	  pobj = elementFactory.makeErrorObject(ProrateAudit.ERR_ERROR,
						str, "");
	rules.add(pobj);
	return rules;
      }
    }
    else {
      if (!selectAPDP(carrier, issueDate)) {
	String	str = "Apply_APDP(";
	str += carrier;
	str += ",";
	str += issueDate;
	str += ")";
	str += ": undefined APDP rule";
	ProrateRulebaseElement
	  pobj = elementFactory.makeErrorObject(ProrateAudit.ERR_ERROR,
						str, "");
	rules.add(pobj);
	return rules;
      }
    }

    /* skip to [actions] and read rules in [actions] if skipToActions is true*/
    rules = read(skipToActions);

    return rules;
  }

  List	loadFromElement(Element element, boolean skipToActions)
    throws ProrateRulebaseException {
    List	children = new Vector();
    for (Iterator i = element.getChildren().iterator(); i.hasNext(); ) {
      Element	childElement = (Element)i.next();
      String	name = element.getName();
      List	child = null;
      if (name.equals("rule")) {	/* child is <spa/> or <apdp/> etc */
	child = loadFromElement(childElement, skipToActions);
	children.add(child);
	List	ret = createElement(element, children);
	return ret;
      }
      if (name.equals("spa") || name.equals("apdp")
	  || name.equals("part") || name.equals("extf")) {
	if (skipToActions) {
	  String	childName = childElement.getName();
	  if (!childName.equals("actions")) {
	    continue;
	  }
	}
      }
      child = loadFromElement(childElement, false);
      children.add(child);
    }
    List	ret = createElement(element, children);
    return ret;
  }
  
  List	createElement(Element element, List children)
    throws ProrateRulebaseException {
    String	name = element.getName();
    if (name.equals("rule")) {
      return	children;
    }
    else if (name.equals("spa") || name.equals("apdp")
	     || name.equals("extf") || name.equals("part")) {
      return	createNop(name, element, children);
    }
    else if (name.equals("common") || name.equals("actions")) {
      return	createNop(name, element, children);
    }
    else if (name.equals("extf-args")) {
      List	childrenList = new Vector();
      for (int i = 0; i < children.size(); i++) {
	List	child = (List)children.get(i);
	Object	obj = child.get(0);
	childrenList.add(obj);
      }
      return	childrenList;
    }
    else if (name.equals("extf-arg")) {
      /* ;;; not implemented yet */
      List	varsList = createVar(element, children);
      List	childrenList = new Vector();
      childrenList.add(varsList);
      element.setAttribute("name", "Set");
      return	createFunc(element, childrenList);
    }
    else if (name.equals("if")) {
      return	createIf(element, children);
    }
    else if (name.equals("then")) {
      return	children;
    }
    else if (name.equals("else")) {
      return	children;
    }
    else if (name.equals("switch")) {
      return	createSwitch(element, children);
    }
    else if (name.equals("case-vars")) {
      return	children;
    }
    else if (name.equals("branch")) {
      return	createBranch(element, children);
    }
    else if (name.equals("branch-condition")) {
      return	children;
    }
    else if (name.equals("case")) {
      return	createCase(element, children);
    }
    else if (name.equals("table")) {
      return	createTable(element, children);
    }
    else if (name.equals("table-vars")) {
      return	children;
    }
    else if (name.equals("table-conditions")) {
      return	children;
    }
    else if (name.equals("rows")) {
      return	children;
    }
    else if (name.equals("cols")) {
      return	children;
    }
    else if (name.equals("amount-table")) {
      return	children;
    }
    else if (name.equals("action-statement")) {
      return	createStatement(element, children);
    }
    else if (name.equals("judge-statement")) {
      return	createStatement(element, children);
    }
    else if (name.equals("set")) {
      return	createFunc(element, children);
    }
    else if (name.equals("end")) {
      return	createEnd(element, children);
    }
    else if (name.equals("return")) {
      return	createReturn(element, children);
    }
    else if (name.equals("partcall")) {
      return	createPartcall(element, children);
    }
    else if (name.equals("percent")) {
      return	createPercent(element, children);
    }
    else if (name.equals("amount")) {
      return	createAmount(element, children);
    }
    else if (name.equals("funcall")) {
      return	createFunc(element, children);
    }
    else if (name.equals("and")) {
      return	createAnd(element, children);
    }
    else if (name.equals("or")) {
      return	createOr(element, children);
    }
    else if (name.equals("judge")) {
      return	createJudge(element, children);
    }
    else if (name.equals("multival")) {
      return	createMultival(element, children);
    }
    else if (name.equals("interval")) {
      return	createInterval(element, children);
    }
    else if (name.equals("variable")) {
      return	createVar(element, children);
    }
    else if (name.equals("true")) {
      return	createTrue(element, children);
    }
    else if (name.equals("false")) {
      return	createFalse(element, children);
    }
    else if (name.equals("number")) {
      return	createNumber(element, children);
    }
    else if (name.equals("date")) {
      return	createDate(element, children);
    }
    else if (name.equals("string")) {
      return	createString(element, children);
    }
    else if (name.equals("path")) {
      return	createPath(element, children);
    }
    return null;
  }
  
  List	createNop(String name, Element element, List children) {
    ProrateRulebaseElement	ret = null;
    if (name.equals("spa")) {
      ret =
	elementFactory.makeNopObject((short)ProrateRulebaseElement.SPA_CODE,
				     element.getAttributeValue("carrier"),
				     element.getAttributeValue("tkCarrier"),
				     element.getAttributeValue("validStart"),
				     element.getAttributeValue("validEnd"),
				     "",
				     element.getAttributeValue("comment"));
    }
    else if (name.equals("apdp")) {
      ret =
	elementFactory.makeNopObject((short)ProrateRulebaseElement.APD_P_CODE,
				     element.getAttributeValue("carrier"),
				     "",
				     element.getAttributeValue("validStart"),
				     element.getAttributeValue("validEnd"),
				     "",
				     element.getAttributeValue("comment"));
    }
    else if (name.equals("extf")) {
      /* ;;; not implemented yet */
      ret =
	elementFactory.makeNopObject((short)ProrateRulebaseElement.EXTF_CODE,
				     element.getAttributeValue("name"),
				     "", "", "",
				     element.getAttributeValue("type"),
				     element.getAttributeValue("comment"));
    }
    else if (name.equals("part")) {
      ret =
	elementFactory.makeNopObject((short)ProrateRulebaseElement.PART_CODE,
				     element.getAttributeValue("name"),
				     "", "", "", "",
				     element.getAttributeValue("comment"));
    }
    else if (name.equals("common")) {
      ret =
	elementFactory.makeNopObject((short)ProrateRulebaseElement.COMMON_CODE,
				     "", "", "", "", "", "");
    }
    else if (name.equals("actions")) {
      ret =
	elementFactory
	.makeNopObject((short)ProrateRulebaseElement.ACTIONS_CODE,
		       "", "", "", "", "", "");
    }
    List	retList = new Vector();
    if (ret != null) {
      retList.add(ret);
    }
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      List	list = (List)i.next();
      for (Iterator j = list.iterator(); j.hasNext(); ) {
	Object	pobj = j.next();
	retList.add(pobj);
      }
    }
    return	retList;
  }
  
  List	createEnd(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makeEndObject(comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createIf(Element element, List children) {
    List	tests = (List)children.get(0);
    List	thens = new Vector();
    List	thensPre = (List)children.get(1);
    for (Iterator i = thensPre.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      thens.add(pobj);
    }
    List	elses = new Vector();
    List	elsesPre = (List)children.get(2);
    for (Iterator i = elsesPre.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      elses.add(pobj);
    }
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makeTestObject(tests, thens, elses, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createAnd(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    List	args = new Vector();
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      args.add(pobj);
    }
    ProrateRulebaseElement
      ret = elementFactory.makeAndObject(args, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createOr(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    List	args = new Vector();
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      args.add(pobj);
    }
    ProrateRulebaseElement
      ret = elementFactory.makeOrObject(args, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createJudge(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    List	args = new Vector();
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      args.add(pobj);
    }
    ProrateRulebaseElement
      ret = elementFactory.makeJudgeObject(args, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createSwitch(Element element, List children) {
    /* FIXME */
    List	args = new Vector();
    List	argsPre = (List)children.get(0);
    for (Iterator i = argsPre.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      args.add(pobj);
    }
    String	comment = element.getAttributeValue("comment");
    List	branchacts = new Vector();
    boolean	first = true;
    for (Iterator i = children.iterator();
	 i.hasNext(); first = false) {
      List	list = (List)i.next();
      if (list == null || list.size() < 1) {
	continue;
      }
      Object	pobj = list.get(0);
      if (!first) {
	branchacts.add(pobj);
      }
    }
    ProrateRulebaseElement
      ret = elementFactory.makeSwitchObject(args, branchacts, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createBranch(Element element, List children) {
    /* FIXME */
    List	branches = new Vector();
    List	branchesPre = (List)children.get(0);
    for (Iterator i = branchesPre.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      branches.add(pobj);
    }
    String	comment = element.getAttributeValue("comment");
    List	actions = new Vector();
    boolean	first = true;
    for (Iterator i = children.iterator();
	 i.hasNext(); first = false) {
      Object	pobj = ((List)i.next()).get(0);
      if (!first) {
	actions.add(pobj);
      }
    }
    ProrateRulebaseElement
      ret = elementFactory.makeBranchObject(branches, actions, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createCase(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    List	values = new Vector();
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      values.add(pobj);
    }
    ProrateRulebaseElement
      ret = elementFactory.makeCaseObject(values, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createTrue(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makeBoolObject(true, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createFalse(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makeBoolObject(false, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createReturn(Element element, List children) {
    ProrateRulebaseElement
      value = (ProrateRulebaseElement)((List)children.get(0)).get(0);
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makeReturnObject(value, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createPartcall(Element element, List children)
    throws ProrateRulebaseException {
    String	name = element.getAttributeValue("name");
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makePartcallObject(name, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createMultival(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    List	values = new Vector();
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      values.add(pobj);
    }
    ProrateRulebaseElement
      ret = elementFactory.makeMultivalObject(values, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createInterval(Element element, List children) {
    ProrateRulebaseElement
      lowvalue = (ProrateRulebaseElement)((List)children.get(0)).get(0);
    ProrateRulebaseElement
      highvalue = (ProrateRulebaseElement)((List)children.get(1)).get(0);
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makeIntervalObject(lowvalue, highvalue, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createAmount(Element element, List children) {
    String	currency = element.getAttributeValue("currency");
    String	valueStr = element.getAttributeValue("value");
    double	value = Double.parseDouble(valueStr);
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makeAmountObject(currency, value, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createPercent(Element element, List children) {
    ProrateRulebaseElement
      amount = (ProrateRulebaseElement)((List)children.get(0)).get(0);
    String	percentStr = element.getAttributeValue("number");
    double	percent = Double.parseDouble(percentStr);
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = elementFactory.makePercentObject(percent, amount, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createVar(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    String	name = element.getAttributeValue("name");
    short	regist = 0;

    if (name.equals("$ENDORSE")) {
      regist = ProrateRulebaseElement.ENDORSE_VARIABLE;
    }
    else if (name.equals("$RESTRICT")) {
      regist = ProrateRulebaseElement.RESTRICT_VARIABLE;
    }
    else if (name.equals("$ORIGIN")) {
      regist = ProrateRulebaseElement.ORIGIN_VARIABLE;
    }
    else if (name.equals("$DEST")) {
      regist = ProrateRulebaseElement.DEST_VARIABLE;
    }
    else if (name.equals("$ISSUE-DATE")) {
      regist = ProrateRulebaseElement.ISSUE_DATE_VARIABLE;
    }
    else if (name.equals("$ISSUE-BY")) {
      regist = ProrateRulebaseElement.ISSUE_BY_VARIABLE;
    }
    else if (name.equals("$ISSUE-PLACE")) {
      regist = ProrateRulebaseElement.ISSUE_PLACE_VARIABLE;
    }
    else if (name.equals("$MULTI-PATH")) {
      regist = ProrateRulebaseElement.MULTI_PATH_VARIABLE;
    }
    else if (name.equals("$FC-FROM")) {
      regist = ProrateRulebaseElement.FC_FROM_VARIABLE;
    }
    else if (name.equals("$FC-TO")) {
      regist = ProrateRulebaseElement.FC_TO_VARIABLE;
    }
    else if (name.equals("$FC-PATH")) {
      regist = ProrateRulebaseElement.FC_PATH_VARIABLE;
    }
    else if (name.equals("$ISSUE-COUNTRY")) {
      regist = ProrateRulebaseElement.ISSUE_COUNTRY_VARIABLE;
    }
    else if (name.equals("$ISSUE-AREA")) {
      regist = ProrateRulebaseElement.ISSUE_AREA_VARIABLE;
    }
    else if (name.equals("$ISSUE-AGENT")) {
      regist = ProrateRulebaseElement.ISSUE_AGENT_VARIABLE;
    }
    else if (name.equals("$TOURCODE")) {
      regist = ProrateRulebaseElement.TOURCODE_VARIABLE;
    }
    else if (name.equals("$FROM")) {
      regist = ProrateRulebaseElement.FROM_VARIABLE;
    }
    else if (name.equals("$TO")) {
      regist = ProrateRulebaseElement.TO_VARIABLE;
    }
    else if (name.equals("$F-BASIS")) {
      regist = ProrateRulebaseElement.F_BASIS_VARIABLE;
    }
    else if (name.equals("$TKDESIGN")) {
      regist = ProrateRulebaseElement.TKDESIGN_VARIABLE;
    }
    else if (name.equals("$F-BASIS-ALL")) {
      regist = ProrateRulebaseElement.F_BASIS_ALL_VARIABLE;
    }
    else if (name.equals("$CARRIER")) {
      regist = ProrateRulebaseElement.CARRIER_VARIABLE;
    }
    else if (name.equals("$FLIGHT")) {
      regist = ProrateRulebaseElement.FLIGHT_VARIABLE;
    }
    else if (name.equals("$CLASS")) {
      regist = ProrateRulebaseElement.CLASS_VARIABLE;
    }
    else if (name.equals("$DATE")) {
      regist = ProrateRulebaseElement.DATE_VARIABLE;
    }
    else if (name.equals("$PATH")) {
      regist = ProrateRulebaseElement.PATH_VARIABLE;
    }
    else if (name.equals("$ROUTE")) {
      regist = ProrateRulebaseElement.ROUTE_VARIABLE;
    }
    else if (name.equals("$FROM-A")) {
      regist = ProrateRulebaseElement.FROM_ANY_VARIABLE;
    }
    else if (name.equals("$TO-A")) {
      regist = ProrateRulebaseElement.TO_ANY_VARIABLE;
    }
    else if (name.equals("$F-BASIS-A")) {
      regist = ProrateRulebaseElement.F_BASIS_ANY_VARIABLE;
    }
    else if (name.equals("$TKDESIGN-A")) {
      regist = ProrateRulebaseElement.TKDESIGN_ANY_VARIABLE;
    }
    else if (name.equals("$F-BASIS-ALL-A")) {
      regist = ProrateRulebaseElement.F_BASIS_ALL_ANY_VARIABLE;
    }
    else if (name.equals("$CARRIER-A")) {
      regist = ProrateRulebaseElement.CARRIER_ANY_VARIABLE;
    }
    else if (name.equals("$FLIGHT-A")) {
      regist = ProrateRulebaseElement.FLIGHT_ANY_VARIABLE;
    }
    else if (name.equals("$CLASS-A")) {
      regist = ProrateRulebaseElement.CLASS_ANY_VARIABLE;
    }
    else if (name.equals("$DATE-A")) {
      regist = ProrateRulebaseElement.DATE_ANY_VARIABLE;
    }
    else if (name.equals("$PATH-A")) {
      regist = ProrateRulebaseElement.PATH_ANY_VARIABLE;
    }
    else if (name.equals("$ROUTE-A")) {
      regist = ProrateRulebaseElement.ROUTE_ANY_VARIABLE;
    }
    else if (name.equals("$DISCOUNT")) {
      regist = ProrateRulebaseElement.DISCOUNT_VARIABLE;
    }
    else {
      regist = ProrateRulebaseElement.AUTO_VARIABLE_INITIAL_CODE;
    }

    ProrateRulebaseElement	ret = null;
    if (regist == ProrateRulebaseElement.AUTO_VARIABLE_INITIAL_CODE) {
      ret = elementFactory.makeVarAutoObject(regist, name, comment);
    }
    else {
      ret = elementFactory.makeVarObject(regist, comment);
    }
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createFunc(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    String	name = element.getAttributeValue("name");
    short	regist = 0;

    if (name.equals("Set")) {
      regist = ProrateRulebaseElement.SET_CODE;
    }
    else if (name.equals("=")) {
      regist = ProrateRulebaseElement.EQUAL_CODE;
    }
    else if (name.equals("!=")) {
      regist = ProrateRulebaseElement.NEQ_CODE;
    }
    else if (name.equals(">=")) {
      regist = ProrateRulebaseElement.STRINC_CODE;
    }
    else if (name.equals("Pfm")) {
      regist = ProrateRulebaseElement.Pfm_FUNCTION;
    }
    else if (name.equals("Pfm_max")) {
      regist = ProrateRulebaseElement.Pfm_max_FUNCTION;
    }
    else if (name.equals("Srp")) {
      regist = ProrateRulebaseElement.Srp_FUNCTION;
    }
    else if (name.equals("Np")) {
      regist = ProrateRulebaseElement.Np_FUNCTION;
    }
    else if (name.equals("Apply_discount")) {
      regist = ProrateRulebaseElement.Apply_discount_FUNCTION;
    }
    else if (name.equals("Country")) {
      regist = ProrateRulebaseElement.Country_FUNCTION;
    }
    else if (name.equals("Area")) {
      regist = ProrateRulebaseElement.Area_FUNCTION;
    }
    else if (name.equals("Route")) {
      regist = ProrateRulebaseElement.Route_FUNCTION;
    }
    else if (name.equals("Via")) {
      regist = ProrateRulebaseElement.Via_FUNCTION;
    }
    else if (name.equals("Via_country")) {
      regist = ProrateRulebaseElement.Via_country_FUNCTION;
    }
    else if (name.equals("Gateway")) {
      regist = ProrateRulebaseElement.Gateway_FUNCTION;
    }
    else if (name.equals("Gateway_country")) {
      regist = ProrateRulebaseElement.Gateway_country_FUNCTION;
    }
    else if (name.equals("RoundTheWorld_p")) {
      regist = ProrateRulebaseElement.RoundTheWorld_p_FUNCTION;
    }
    else if (name.equals("Mileage_p")) {
      regist = ProrateRulebaseElement.Mileage_p_FUNCTION;
    }
    else if (name.equals("Route_p")) {
      regist = ProrateRulebaseElement.Route_p_FUNCTION;
    }
    else if (name.equals("Sidetrip_p")) {
      regist = ProrateRulebaseElement.Sidetrip_p_FUNCTION;
    }
    else if (name.equals("No_amount_p")) {
      regist = ProrateRulebaseElement.No_amount_p_FUNCTION;
    }
    else if (name.equals("Special_fare_p")) {
      regist = ProrateRulebaseElement.Special_fare_p_FUNCTION;
    }
    else if (name.equals("Promotional_fare_p")) {
      regist = ProrateRulebaseElement.Promotional_fare_p_FUNCTION;
    }
    else if (name.equals("Normal_fare_p")) {
      regist = ProrateRulebaseElement.Normal_fare_p_FUNCTION;
    }
    else if (name.equals("Tax_divid")) {
      regist = ProrateRulebaseElement.Tax_divid_FUNCTION;
    }
    else if (name.equals("Via_area")) {
      regist = ProrateRulebaseElement.Via_area_FUNCTION;
    }
    else if (name.equals("Discount")) {
      regist = ProrateRulebaseElement.Discount_FUNCTION;
    }
    else if (name.equals("Bi_proviso")) {
      regist = ProrateRulebaseElement.Bi_proviso_FUNCTION;
    }
    else if (name.equals("Multipath_amt")) {
      regist = ProrateRulebaseElement.Multipath_amt_FUNCTION;
    }
    else if (name.equals("Uncertain")) {
      regist = ProrateRulebaseElement.Uncertain_FUNCTION;
    }
    else if (name.equals("Number_p")) {
      regist = ProrateRulebaseElement.Number_p_FUNCTION;
    }
    else if (name.equals("Date_p")) {
      regist = ProrateRulebaseElement.Date_p_FUNCTION;
    }
    else if (name.equals("String_p")) {
      regist = ProrateRulebaseElement.String_p_FUNCTION;
    }
    else if (name.equals("Amount_p")) {
      regist = ProrateRulebaseElement.Amount_p_FUNCTION;
    }
    else if (name.equals("Path_p")) {
      regist = ProrateRulebaseElement.Path_p_FUNCTION;
    }
    else if (name.equals("Error")) {
      regist = ProrateRulebaseElement.Error_FUNCTION;
    }
    else if (name.equals("Within")) {
      regist = ProrateRulebaseElement.Within_FUNCTION;
    }
    else if (name.equals("Within_country")) {
      regist = ProrateRulebaseElement.Within_country_FUNCTION;
    }
    else if (name.equals("Within_area")) {
      regist = ProrateRulebaseElement.Within_area_FUNCTION;
    }
    else if (name.equals("Intl_sector_p")) {
      regist = ProrateRulebaseElement.Intl_sector_p_FUNCTION;
    }
    else if (name.equals("Between")) {
      regist = ProrateRulebaseElement.Between_FUNCTION;
    }
    else if (name.equals("Between_country")) {
      regist = ProrateRulebaseElement.Between_country_FUNCTION;
    }
    else if (name.equals("Between_area")) {
      regist = ProrateRulebaseElement.Between_area_FUNCTION;
    }
    else if (name.equals("Between_PC")) {
      regist = ProrateRulebaseElement.Between_PC_FUNCTION;
    }
    else if (name.equals("Between_PA")) {
      regist = ProrateRulebaseElement.Between_PA_FUNCTION;
    }
    else if (name.equals("Between_CA")) {
      regist = ProrateRulebaseElement.Between_CA_FUNCTION;
    }
    else if (name.equals("Exclusive")) {
      regist = ProrateRulebaseElement.Exclusive_FUNCTION;
    }
    else if (name.equals("Another")) {
      regist = ProrateRulebaseElement.Another_FUNCTION;
    }
    else if (name.equals("High_spa")) {
      regist = ProrateRulebaseElement.High_spa_FUNCTION;
    }
    else if (name.equals("Low_spa")) {
      regist = ProrateRulebaseElement.Low_spa_FUNCTION;
    }
    else if (name.equals("High_spa_fix")) {
      regist = ProrateRulebaseElement.High_spa_fix_FUNCTION;
    }
    else if (name.equals("Low_spa_fix")) {
      regist = ProrateRulebaseElement.Low_spa_fix_FUNCTION;
    }
    else if (name.equals("Net")) {
      regist = ProrateRulebaseElement.Net_FUNCTION;
    }
    else if (name.equals("Srp_fix")) {
      regist = ProrateRulebaseElement.Srp_fix_FUNCTION;
    }
    else if (name.equals("Sia")) {
      regist = ProrateRulebaseElement.Sia_FUNCTION;
    }
    /* ;;; deBug 2013.11.19 */
    else if (name.equals("Apply_spa")) {
      return	createApply(true, element, children);
    }
    else if (name.equals("Apply_apdp")) {
      return	createApply(false, element, children);
    }
    else {
      regist = ProrateRulebaseElement.EXTERNAL_FUNCTION;
    }

    List	args = new Vector();
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      args.add(pobj);
    }
    ProrateRulebaseElement	ret = null;
    if (regist == ProrateRulebaseElement.EXTERNAL_FUNCTION) {
      Element	parent = element.getParentElement();
      String	parentName = parent.getName();
      ret = elementFactory.makeExternObject(name, parentName, args, comment);
    }
    else {
      ret = elementFactory.makeFuncObject(regist, args, comment);
    }
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createApply(boolean isSPA, Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    String	carrier = element.getAttributeValue("carrier");
    String	tkCarrier = element.getAttributeValue("tkCarrier");
    /*
    ProrateRulebaseElement
      carrier = (ProrateRulebaseElement)((List)children.get(0)).get(0);
    String	tkCarrier = null;
    */
    ProrateRulebaseElement
      ret = elementFactory.makeApplyObject(isSPA, carrier, tkCarrier, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createNumber(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    String	valueStr = element.getAttributeValue("value");
    double	value = Double.parseDouble(valueStr);
    ProrateRulebaseElement
      ret = elementFactory.makeNumberObject(value, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createDate(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    String	monthStr = element.getAttributeValue("month");
    short	month = (short)Integer.parseInt(monthStr);
    String	dayStr = element.getAttributeValue("day");
    short	day = (short)Integer.parseInt(dayStr);
    ProrateRulebaseElement
      ret = elementFactory.makeDateObject(month, day, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createString(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    String	value = element.getAttributeValue("value");
    ProrateRulebaseElement
      ret = elementFactory.makeStringObject(value, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createPath(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    List	value = new Vector();
    for (Iterator i = children.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      value.add(pobj);
    }
    ProrateRulebaseElement
      ret = elementFactory.makePathObject(value, comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createTable(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    List	vars = new Vector();
    List	varsPre = (List)children.get(0);
    for (Iterator i = varsPre.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      vars.add(pobj);
    }
    List	conds = new Vector();
    List	condsPre = (List)children.get(1);
    for (Iterator i = condsPre.iterator(); i.hasNext(); ) {
      List	list = (List)i.next();
      if (list == null || list.size() < 1) {
	continue;
      }
      List	objList = new Vector();
      for (int j = 0; j < list.size(); j++) {
	Object	pobj = ((List)list.get(j)).get(0);
	objList.add(pobj);
      }
      conds.add(objList);
    }
    List	tbls = new Vector();
    List	tblsPre = (List)children.get(2);
    for (Iterator i = tblsPre.iterator(); i.hasNext(); ) {
      Object	pobj = ((List)i.next()).get(0);
      tbls.add(pobj);
    }
    ProrateRulebaseElement	prows = null;
    ProrateRulebaseElement	pcols = null;
    List	rows = null;
    List	cols = null;
    if (vars.size() > 1) {
      prows = (ProrateRulebaseElement)vars.get(0);
      pcols = (ProrateRulebaseElement)vars.get(1);
    }
    else if (vars.size() == 1) {
      pcols = (ProrateRulebaseElement)vars.get(0);
    }
    if (conds.size() > 1) {
      rows = new Vector();
      cols = new Vector();
      List	rowsPre = (List)conds.get(0);
      for (Iterator i = rowsPre.iterator(); i.hasNext(); ) {
	Object	pobj = i.next();
	rows.add(pobj);
      }
      List	colsPre = (List)conds.get(1);
      for (Iterator i = colsPre.iterator(); i.hasNext(); ) {
	Object	pobj = i.next();
	cols.add(pobj);
      }
    }
    else if (conds.size() == 1) {
      cols = new Vector();
      List	colsPre = (List)conds.get(0);
      for (Iterator i = colsPre.iterator(); i.hasNext(); ) {
	Object	pobj = i.next();
	cols.add(pobj);
      }
    }
    ProrateRulebaseElement
      ret = elementFactory.makeTableObject(prows, pcols, rows, cols, tbls,
					   comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  List	createStatement(Element element, List children) {
    String	comment = element.getAttributeValue("comment");
    ProrateRulebaseElement
      ret = (ProrateRulebaseElement)((List)children.get(0)).get(0);
    ret.setComment(comment);
    List	retList = new Vector();
    retList.add(ret);
    return	retList;
  }

  // Setters and Getters
  /**
   * 
   * @uml.property name="url"
   */
  public void setUrl(String theUrl) {
    url = theUrl;
  }

  /**
   * 
   * @uml.property name="url"
   */
  public String getUrl() {
    return url;
  }

  /**
   * 
   * @uml.property name="elementFactory"
   */
  public void setElementFactory(ProrateRulebaseElementFactory theFactory) {
    elementFactory = theFactory;
  }

  /**
   * 
   * @uml.property name="elementFactory"
   */
  public ProrateRulebaseElementFactory getElementFactory() {
    return elementFactory;
  }

} /* end class XmlRulebaseUrl */
