package jp.haw.grain.transform;

import jp.haw.grain.transform.transformer.AbstractTransformer;
import jp.haw.grain.transform.transformer.BodyDescendants2Inline;
import junit.framework.TestCase;

import org.apache.xerces.dom.DOMImplementationImpl;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

/**
 * @author nakajo
 */
public class BodyDescendants2InlineTest extends TestCase {

	/**
	 * bodyeɂKȃ^OϊeXg
	 *
	 */
	public void testSimpleTag() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element foo = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "foo");
		foo.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(foo);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), true);
		foo2inline.setTargetLocalNames(new String[]{"foo"});
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#operation(jp.haw.grain.transform.transformer.AbstractTransformer)
			 */
			public void operation(AbstractTransformer t) {
				DocumentFragment df = t.transform();
				assertTrue(df.hasChildNodes());
				NodeList nodes = df.getChildNodes();
				assertEquals("size check", nodes.getLength(), 1);
				Node inline = nodes.item(0);
				assertTrue("inline has text check", inline.getTextContent() != null);
				assertTrue("element name check", inline.getNodeName().equals("inline"));
				assertTrue("inline has same text check", inline.getTextContent().equals("قق"));
				assertEquals("current is null check", t.getCurrentElement(), null);
				
			}
		};

		foo2inline.apply(foo, mockOperation);
	}
	/**
	 * bodyeɂKȃ^OϊeXg
	 * ႤO̕ϊeXg
	 *
	 */
	public void testSimpleTag2() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element hoge = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "hoge");
		hoge.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(hoge);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), true);
		foo2inline.setTargetLocalNames(new String[]{"hoge"});
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#operation(jp.haw.grain.transform.transformer.AbstractTransformer)
			 */
			public void operation(AbstractTransformer t) {
				DocumentFragment df = t.transform();
				assertTrue(df.hasChildNodes());
				NodeList nodes = df.getChildNodes();
				assertEquals("size check", nodes.getLength(), 1);
				Node inline = nodes.item(0);
				assertTrue("inline has text check", inline.getTextContent() != null);
				assertTrue("element name check", inline.getNodeName().equals("inline"));
				assertTrue("inline has same text check", inline.getTextContent().equals("قق"));
				assertEquals("current is null check", t.getCurrentElement(), null);
				
			}
		};

		foo2inline.apply(hoge, mockOperation);
	}

	/**
	 * ̃^Owϊ
	 *
	 */
	public void testManyTag() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element hoge = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "hoge");
		hoge.setTextContent("قق");
		Element foo = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "foo");
		foo.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(hoge);
		simpleDoc.getDocumentElement().appendChild(foo);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), true);
		foo2inline.setTargetLocalNames(new String[]{"hoge", "foo"});
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#operation(jp.haw.grain.transform.transformer.AbstractTransformer)
			 */
			public void operation(AbstractTransformer t) {
				DocumentFragment df = t.transform();
				assertTrue(df.hasChildNodes());
				NodeList nodes = df.getChildNodes();
				assertEquals("size check", nodes.getLength(), 1);
				Node inline = nodes.item(0);
				assertTrue("inline has text check", inline.getTextContent() != null);
				assertTrue("element name check", inline.getNodeName().equals("inline"));
				assertTrue("inline has same text check", inline.getTextContent().equals("قق"));
				assertEquals("current is null check", t.getCurrentElement(), null);
				
			}
		};

		foo2inline.apply(hoge, mockOperation);
		foo2inline.apply(foo, mockOperation);
	}

	/**
	 * bodycɂKȃ^OϊeXg
	 *
	 */
	public void testDeepTag() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element p = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "p");
		Element span = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "span");
		Element foo = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "foo");
		foo.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(p);
		p.appendChild(span);
		span.appendChild(foo);

		BodyDescendants2Inline foo2Inline = new MockTransformer(new GudBuilder(simpleDoc), true);
		foo2Inline.setTargetLocalNames(new String[]{"foo"});
		
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#operation(jp.haw.grain.transform.transformer.AbstractTransformer)
			 */
			public void operation(AbstractTransformer t) {
				DocumentFragment df = t.transform();
				assertTrue(df.hasChildNodes());
				NodeList nodes = df.getChildNodes();
				assertEquals("size check", nodes.getLength(), 1);
				Node inline = nodes.item(0);
				assertTrue("inline has more text check", inline.getTextContent() != null);
				assertTrue("inline has same text check", inline.getFirstChild().getNodeValue().equals("قق"));
				assertEquals("current is null check", t.getCurrentElement(), null);
				
			}
		};

		foo2Inline.apply(foo, mockOperation);
	}

	/**
	 * bodyeɎȂ^O͕ϊȂB
	 *
	 */
	public void testParentNotBodyTag() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "head", null);
		Element foo = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "foo");
		Text fooText = simpleDoc.createTextNode("قق");
		simpleDoc.getDocumentElement().appendChild(foo);

		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), false);
		foo2inline.setTargetLocalNames(new String[]{"foo"});
		
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#createChildNodeOperation()
			 */
			public TransformOperation createChildNodeOperation() {
				assertEquals("transformer null check", this.transformer, null);
				// TODO ꂽ\bhEX^u
				return super.createChildNodeOperation();
			}

		};

		foo2inline.apply(foo, mockOperation);
		mockOperation.createChildNodeOperation();
	}

	/**
	 * ^[QbgnameɎw肳ĂȂ^O͕ϊȂB
	 *
	 */
	public void testNonTargetTag() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element foo = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "foo");
		foo.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(foo);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), false);
		foo2inline.setTargetLocalNames(new String[]{"hoge"});
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#createChildNodeOperation()
			 */
			public TransformOperation createChildNodeOperation() {
				assertEquals("transformer null check", this.transformer, null);
				// TODO ꂽ\bhEX^u
				return super.createChildNodeOperation();
			}

		};

		foo2inline.apply(foo, mockOperation);
		mockOperation.createChildNodeOperation();
	}

	/**
	 * eLXgm[hȂ^O͕ϊȂ
	 *
	 */
	public void testNotHaveTextNodeTag() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element foo = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "foo");
		simpleDoc.getDocumentElement().appendChild(foo);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), false);
		foo2inline.setTargetLocalNames(new String[]{"foo"});
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#createChildNodeOperation()
			 */
			public TransformOperation createChildNodeOperation() {
				assertEquals("transformer null check", this.transformer, null);
				// TODO ꂽ\bhEX^u
				return super.createChildNodeOperation();
			}

		};

		foo2inline.apply(foo, mockOperation);
		mockOperation.createChildNodeOperation();
	}
	
	/**
	 * font-stylew肳ĂΓKpB
	 *
	 */
	public void testFontStyle() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element hoge = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "hoge");
		hoge.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(hoge);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), true);
		foo2inline.setTargetLocalNames(new String[]{"hoge"});
		foo2inline.setStyle(GudBuilder.STYLE_BOLD);
		
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#operation(jp.haw.grain.transform.transformer.AbstractTransformer)
			 */
			public void operation(AbstractTransformer t) {
				DocumentFragment df = t.transform();
				assertTrue(df.hasChildNodes());
				NodeList nodes = df.getChildNodes();
				assertEquals("size check", nodes.getLength(), 1);
				Node inline = nodes.item(0);
				assertTrue("inline has text check", inline.getTextContent() != null);
				assertTrue("element name check", inline.getNodeName().equals("inline"));
				assertTrue("inline has same text check", inline.getTextContent().equals("قق"));
				assertEquals("current is null check", t.getCurrentElement(), null);
				
				//fornt-styleB
				NamedNodeMap attrs = inline.getAttributes();
				assertEquals("attr length check", attrs.getLength(), 1);
				Node fontstyle = attrs.getNamedItem("font-style");
				assertEquals("fontstyle check", fontstyle.getNodeValue(), GudBuilder.STYLE_BOLD);
			}
		};

		foo2inline.apply(hoge, mockOperation);
	}

	/**
	 * fontsizew肳ĂΓKp
	 *
	 */
	public void testFontSize() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element hoge = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "hoge");
		hoge.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(hoge);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), true);
		foo2inline.setTargetLocalNames(new String[]{"hoge"});
		foo2inline.setSize(GudBuilder.SIZE_LARGE);
		
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#operation(jp.haw.grain.transform.transformer.AbstractTransformer)
			 */
			public void operation(AbstractTransformer t) {
				DocumentFragment df = t.transform();
				assertTrue(df.hasChildNodes());
				NodeList nodes = df.getChildNodes();
				assertEquals("size check", nodes.getLength(), 1);
				Node inline = nodes.item(0);
				assertTrue("inline has text check", inline.getTextContent() != null);
				assertTrue("element name check", inline.getNodeName().equals("inline"));
				assertTrue("inline has same text check", inline.getTextContent().equals("قق"));
				assertEquals("current is null check", t.getCurrentElement(), null);
				
				//fornt-sizeB
				NamedNodeMap attrs = inline.getAttributes();
				assertEquals("attr length check", attrs.getLength(), 1);
				Node fontsize = attrs.getNamedItem("font-size");
				assertEquals("fontsize check", fontsize.getNodeValue(), GudBuilder.SIZE_LARGE);
			}
		};

		foo2inline.apply(hoge, mockOperation);
	}

	/**
	 * font-sizefont-style̗ݒ
	 *
	 */
	public void testFontSizeAndStyle() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument("http://grain.jp/gud/", "form", null);
		Document simpleDoc = domi.createDocument(GudBuilder.XHTML_NS, "body", null);
		Element hoge = simpleDoc.createElementNS(GudBuilder.XHTML_NS, "hoge");
		hoge.setTextContent("قق");
		simpleDoc.getDocumentElement().appendChild(hoge);
		
		BodyDescendants2Inline foo2inline = new MockTransformer(new GudBuilder(simpleDoc), true);
		foo2inline.setTargetLocalNames(new String[]{"hoge"});
		foo2inline.setSize(GudBuilder.SIZE_MEDIUM);
		foo2inline.setStyle(GudBuilder.STYLE_ITALIC);
		
		TransformOperation mockOperation = new TransformOperation(gudDoc.getDocumentElement()) {
			/* ( Javadoc)
			 * @see jp.haw.grain.transform.TransformOperation#operation(jp.haw.grain.transform.transformer.AbstractTransformer)
			 */
			public void operation(AbstractTransformer t) {
				DocumentFragment df = t.transform();
				assertTrue(df.hasChildNodes());
				NodeList nodes = df.getChildNodes();
				assertEquals("size check", nodes.getLength(), 1);
				Node inline = nodes.item(0);
				assertTrue("inline has text check", inline.getTextContent() != null);
				assertTrue("element name check", inline.getNodeName().equals("inline"));
				assertTrue("inline has same text check", inline.getTextContent().equals("قق"));
				assertEquals("current is null check", t.getCurrentElement(), null);
				
				//fornt-size, styleB
				NamedNodeMap attrs = inline.getAttributes();
				assertEquals("attr length check", attrs.getLength(), 2);
				Node fontsize = attrs.getNamedItem("font-size");
				assertEquals("fontsize check", fontsize.getNodeValue(), GudBuilder.SIZE_MEDIUM);
				Node fontstyle = attrs.getNamedItem("font-style");
				assertEquals("fontstyle check", fontstyle.getNodeValue(), GudBuilder.STYLE_ITALIC);
			}
		};

		foo2inline.apply(hoge, mockOperation);
	}
	
	class MockTransformer extends BodyDescendants2Inline {
		protected boolean targetResult;
		/**
		 * @param builder
		 */
		public MockTransformer(GudBuilder builder, boolean targetResult) {
			super(builder);
			this.targetResult = targetResult;
		}
		
		/* ( Javadoc)
		 * @see jp.haw.grain.transform.transformer.AbstractTransformer#isTarget(org.w3c.dom.Node)
		 */
		protected boolean isTarget(Node node) {
			assertEquals("apply check", super.isTarget(node), this.targetResult);
			return super.isTarget(node);
		}

	}
}
