/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005-2006 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2006/04/02 5:51:24
 * 
 */
package jp.grain.sprout.platform.doja;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Hashtable;

import javax.microedition.io.Connector;

import jp.grain.sprout.util.Formatter;

import com.nttdocomo.io.ConnectionException;
import com.nttdocomo.io.HttpConnection;

/**
 * NOTE: This class is not thread safe. 
 * 
 * @version $Id$
 * @author Go Takahashi
 */
public class DividedHttpConnection implements HttpConnection {

    public static final String HEADER_X_EXPECT_FO = "X-expect-FO";
    public static final String CONTENT_TYPE_APPLICATION_GBXML = "application/gbxml";
    public static final String HEADER_IF_MODIFIED_SINCE = "If-Modified-Since";
    public static final String HEADER_CONTENT_TYPE = "Content-Type";
    public static final String HEADER_X_DIVIDE_COUNT = "X-Divide-Count";
    public static final String HEADER_X_DIVIDE_SESSION_ID = "X-Divide-Session-ID";
    public static final String HEADER_X_DIVIDE_MAX_COUNT = "X-Divide-Max-Count";
    public static final String HEADER_X_DIVIDE_TRANSFER = "X-Divide-Transfer";
    public static final String HEADER_X_DIVIDE_PROTOCOL_VERSION = "X-Divide-Protocol-Version";
    public static final String HEADER_X_CONTENT_TYPE = "X-Content-Type";
    public static final String HEADER_X_IF_MODIFIED_SINCE = "X-If-Modified-Since";
    public static final String HEADER_DELIMITER = ": ";
    public static final String STR_HEADER_X_DIVIDE_PROTOCOL_VERSION = HEADER_X_DIVIDE_PROTOCOL_VERSION + HEADER_DELIMITER;
    public static final String LF = "\n";
    public static final String DIVIDE_PROTOCOL_VERSION = "1.0";
    public static final String NOT_SUPPORTED_OPERATION = "Not supported operation";

    public static final int MAX_WRITE_BYTES_PER_CONN = 10000;
    public static final int WRITE_BYTES_PER_CONN = 9800;

    private String _name;
    private HttpConnection _conn;
    private boolean _connected;
    private boolean _closed;
    private String _sessionId;

    private String _contentType;
    private String _ifModifiedSince;
    private String _realContentType;

    private byte[] _requestBody;
    private String _expectFO;
    
    /**
     * 
     */
    public DividedHttpConnection(String name) {
        _name = name;
    }
    
    /* (non-Javadoc)
     * @see javax.microedition.io.Connection#close()
     */
    public void close() throws IOException {
        _closed = true;
        try {
            if (_conn != null) _conn.close();
        } finally {
            _conn = null;
        }
    }

    /* (non-Javadoc)
     * @see javax.microedition.io.InputConnection#openInputStream()
     */
    public InputStream openInputStream() throws IOException {
        if (!_connected) throw new RuntimeException();
        if (_closed) throw new IOException();
        return new DevidedInputStream();
    }

    /* (non-Javadoc)
     * @see javax.microedition.io.OutputConnection#openOutputStream()
     */
    public OutputStream openOutputStream() throws IOException {
        if (_connected) throw new RuntimeException();
        if (_closed) throw new IOException();
        return new DevidedOutputStream();
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#connect()
     */
    public void connect() throws IOException {
        try {
            if (_connected) return;
            if (_closed) throw new IOException();
            if (_requestBody != null) writeRequestBody();
            _connected = true;
        } catch (ConnectionException e) {
            System.out.println("ConnectionException status=" + e.getStatus());
            throw e;
        }
    }
    
    private void writeRequestBody() throws IOException {
        int count = 1;
        if (_requestBody.length == 0) {
            count = 1;
        } else if (_requestBody.length > MAX_WRITE_BYTES_PER_CONN) {
            count = _requestBody.length / WRITE_BYTES_PER_CONN;
            count += (_requestBody.length % WRITE_BYTES_PER_CONN) > 0 ? 1 : 0;
        }
        int offset = 0;
        _realContentType = count > 1 ? CONTENT_TYPE_APPLICATION_GBXML : _contentType;
        for (int i = 0; i < count; ++i) {
            reopenConnection();
            writeDivision(i + 1, count, offset);
            offset += WRITE_BYTES_PER_CONN;
        }
    }
    
    /**
     * @param index
     * @param offset
     * @param b
     * @throws IOException 
     */
    private void writeDivision(int index, int count, int offset) throws IOException {
        OutputStream os = null;
        try {
            os = _conn.openOutputStream();
            ByteArrayOutputStream header = new ByteArrayOutputStream();
            Writer writer = new OutputStreamWriter(header);
            if (count == 1) {
                if (CONTENT_TYPE_APPLICATION_GBXML.equals(_realContentType)) {
                    if (_expectFO != null) {
                        writer.write(HEADER_X_EXPECT_FO + HEADER_DELIMITER + _expectFO);
                    }
                    writer.write("\n\n");
                }
                os.write(header.toByteArray());
                Formatter.dumpBytes(_requestBody);
                os.write(_requestBody);
                return;
            }
            if (_expectFO != null) {
                writer.write(HEADER_X_EXPECT_FO + HEADER_DELIMITER + _expectFO);
            }
            writer.write(HEADER_X_DIVIDE_PROTOCOL_VERSION + HEADER_DELIMITER + DIVIDE_PROTOCOL_VERSION + "\n");
            if (index == 1) {
                writer.write(HEADER_X_DIVIDE_MAX_COUNT + HEADER_DELIMITER + count + "\n"); 
                if (_contentType != null) {
                    writer.write(HEADER_X_CONTENT_TYPE + HEADER_DELIMITER + _contentType + "\n");                
                }
                if (_ifModifiedSince != null) {
                    writer.write(HEADER_X_IF_MODIFIED_SINCE + HEADER_DELIMITER + _ifModifiedSince + "\n");                
                }
            } else {
                writer.write(HEADER_X_DIVIDE_SESSION_ID + HEADER_DELIMITER + _sessionId + "\n");
            }
            if (index == count) {
                writer.write(HEADER_X_DIVIDE_TRANSFER + HEADER_DELIMITER + "end\n");
            } else if (index == 1) {
                writer.write(HEADER_X_DIVIDE_TRANSFER + HEADER_DELIMITER + "start\n");                
            }
            writer.write(HEADER_X_DIVIDE_COUNT + HEADER_DELIMITER + index + "\n");
            writer.write('\n');
            os.write(header.toByteArray());
            int remaining = (_requestBody.length - offset);
            Formatter.dumpBytes(_requestBody);
            os.write(_requestBody, offset, remaining > WRITE_BYTES_PER_CONN ? WRITE_BYTES_PER_CONN : remaining);
        } finally {
            if (os != null) os.close();
            _conn.connect();
        }
        System.out.println("Response[" + _conn.getResponseCode() + "]=" + _conn.getResponseMessage());
        if (_conn.getResponseCode() != HttpConnection.HTTP_OK) throw new IOException();
        if (index == count) return;
        String version = _conn.getHeaderField(HEADER_X_DIVIDE_PROTOCOL_VERSION);
        System.out.println("Version=" + version);
        if (version == null || !version.equals(DIVIDE_PROTOCOL_VERSION)) throw new IOException(version);
        if (index == 1) {
            _sessionId = _conn.getHeaderField(HEADER_X_DIVIDE_SESSION_ID);
            System.out.println("Session-ID=" + _sessionId);
            if (_sessionId == null) throw new IOException();
        }
    }
    
    /**
     * MsĂꍇɂ́A_ŎMĂubÑTCYԂB
     * 
     * @see javax.microedition.io.ContentConnection#getLength()
     */
    public long getLength() {
        if (!_connected) throw new RuntimeException();
        return _conn.getLength();
    }

    /**
     * MsĂꍇɂ́A_ŎMĂubNContent-TypeԂB
     * 
     */
    public String getType() {
        if (!_connected) throw new RuntimeException();
        return _conn.getType();
    }

    /**
     * MsĂꍇɂ́A_ŎMĂubNEncodingԂB
     */
    public String getEncoding() {
        if (!_connected) throw new RuntimeException();
        return _conn.getEncoding();
    }

    /**
     * MsĂꍇɂ́A_ŎMĂubÑwb_ԂB
     * @see com.nttdocomo.io.HttpConnection#getHeaderField(java.lang.String)
     */
    public String getHeaderField(String name) {
        if (!_connected) throw new RuntimeException();
        return _conn.getHeaderField(name);
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#getURL()
     */
    public String getURL() {
        return _name;
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#setRequestMethod(java.lang.String)
     */
    public void setRequestMethod(String method) throws IOException {
        if (_connected) throw new RuntimeException();
        // Ignored. This connection allows POST only.
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#setRequestProperty(java.lang.String, java.lang.String)
     */
    public void setRequestProperty(String key, String value) throws IOException {
        if (_connected) throw new RuntimeException();
        if (HEADER_CONTENT_TYPE.equals(key)){
            _contentType = value;
        } else if (HEADER_IF_MODIFIED_SINCE.equals(key)) {
            _ifModifiedSince = value;
        } else if (HEADER_X_EXPECT_FO.equals(key)) {
            _expectFO = value;
        }
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#setIfModifiedSince(long)
     */
    public void setIfModifiedSince(long ifmodifiedsince) {
        // TODO Not implemented yet.
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#getResponseCode()
     */
    public int getResponseCode() throws IOException {
        if (!_connected) throw new RuntimeException();
        return _conn.getResponseCode();
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#getResponseMessage()
     */
    public String getResponseMessage() throws IOException {
        if (!_connected) throw new RuntimeException();
        return _conn.getResponseMessage();
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#getExpiration()
     */
    public long getExpiration() {
        if (!_connected) throw new RuntimeException();
        return _conn.getExpiration();
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#getDate()
     */
    public long getDate() {
        if (!_connected) throw new RuntimeException();
        return _conn.getDate();
    }

    /* (non-Javadoc)
     * @see com.nttdocomo.io.HttpConnection#getLastModified()
     */
    public long getLastModified() {
        if (!_connected) throw new RuntimeException();
        return _conn.getLastModified();
    }

    /* (non-Javadoc)
     * @see javax.microedition.io.InputConnection#openDataInputStream()
     */
    public DataInputStream openDataInputStream() throws IOException {
        throw new RuntimeException(NOT_SUPPORTED_OPERATION);
    }

    /* (non-Javadoc)
     * @see javax.microedition.io.OutputConnection#openDataOutputStream()
     */
    public DataOutputStream openDataOutputStream() throws IOException {
        throw new RuntimeException(NOT_SUPPORTED_OPERATION);
    }
    
    
    private void setRequestBody(byte[] buf) {
        _requestBody = buf;
    }
    
    private void reopenConnection() throws IOException {
        try {
            if (_conn != null) _conn.close();
        } catch (IOException e) {
        }
        _conn = (HttpConnection) Connector.open(_name, Connector.READ_WRITE, true);
        _conn.setRequestMethod(POST);
        _conn.setRequestProperty(HEADER_CONTENT_TYPE, _realContentType);
    }
    
    /**
     * Ms߂̓̓Xg[B
     * DevidedHttpConnectionconnectꂽɐOB<br>
     * ŏ̃X|X̃wb_mFāAM̕Kv𔻒fA
     * KvɉāAMNGXg̑MsȂǏsB
     *  
     * @author Go Takahashi
     */
    private class DevidedInputStream extends InputStream {
        
        private int _index = 1;
        private int _count = 1;
        private InputStream _is = null;
        
        private DevidedInputStream() throws IOException {
            String version = _conn.getHeaderField(HEADER_X_DIVIDE_PROTOCOL_VERSION);
            System.out.println("response version=" + version);
            if (version != null) {
                if (!version.equals(DIVIDE_PROTOCOL_VERSION)) throw new IOException(version);
                _sessionId = _conn.getHeaderField(HEADER_X_DIVIDE_SESSION_ID);
                try {
                    _count = Integer.parseInt(_conn.getHeaderField(HEADER_X_DIVIDE_MAX_COUNT));
                } catch (NumberFormatException e) {
                    throw new IOException(e.toString());
                }
            }
            _is = _conn.openInputStream();
        }
        
        /**
         * ̓Xg[1oCgǂށB<br>
         * @see java.io.InputStream#read()
         */
        public int read() throws IOException {
            int b = _is.read();
            
            // Xg[̏IłA݂̕ubN܂ŏIubNłȂꍇA
            // VȃRlNV쐬AMNGXg𑗐MB
            if (b == -1 && _index < _count) {
                int next = _index + 1;
                try {
                    _is.close();
                } catch (IOException e) {
                }
                _realContentType = CONTENT_TYPE_APPLICATION_GBXML;
                reopenConnection();
                requestDivision(next);
                _conn.connect();
                String version = _conn.getHeaderField(HEADER_X_DIVIDE_PROTOCOL_VERSION);
                if (version == null || !version.equals(DIVIDE_PROTOCOL_VERSION)) {
                    throw new IOException(version);
                }
                if (!_conn.getHeaderField(HEADER_X_DIVIDE_COUNT).equals(String.valueOf(next))) {
                    throw new IOException();
                }
                _is = _conn.openInputStream();
                _index = next;
                b = _is.read();
            }
            return b;
        }
        
        public void close() throws IOException {
            if (_is != null) _is.close();
        }
        
        private void requestDivision(int index) throws IOException {
            Writer writer = null;
            try {
                writer = new OutputStreamWriter(_conn.openOutputStream());
                writer.write(HEADER_X_DIVIDE_PROTOCOL_VERSION + HEADER_DELIMITER + DIVIDE_PROTOCOL_VERSION + "\n");
                writer.write(HEADER_X_DIVIDE_SESSION_ID + HEADER_DELIMITER + _sessionId + "\n");
                writer.write(HEADER_X_DIVIDE_TRANSFER + HEADER_DELIMITER + "receive-request\n");
                writer.write(HEADER_X_DIVIDE_COUNT + HEADER_DELIMITER + index  + "\n\n");
            } finally {
                if (writer != null) writer.close();
            }
        }
        
    }
    
    /**
     * Ms߂̏o̓Xg[B
     * ܂obt@Ɋi[ĂAclose()̍ۂɃtbV
     * 
     * @version $Id$
     * @author Go Takahashi
     */
    private class DevidedOutputStream extends OutputStream {

        private ByteArrayOutputStream _buffer = new ByteArrayOutputStream();

        /* (non-Javadoc)
         * @see java.io.OutputStream#write(int)
         */
        public void write(int b) throws IOException {
            if (_buffer == null) throw new IOException();
            _buffer.write(b);
        }
        
        public void write(byte[] b, int offset, int len) throws IOException {
            if (_buffer == null) throw new IOException();
            _buffer.write(b, offset, len);
        }
        
        public void write(byte[] b) throws IOException {
            if (_buffer == null) throw new IOException();
            _buffer.write(b);
        }
        
        public void close() throws IOException {
            if (_buffer == null) return;
            _buffer.close();
            _requestBody = _buffer.toByteArray();
            _buffer = null;
        }
    }
    
}
