package jp.haw.grain.transform.xpath;

import java.io.IOException;
import java.io.StringWriter;

import jp.haw.grain.transform.GudBuilder;
import junit.framework.TestCase;

import org.apache.xerces.dom.DOMImplementationImpl;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class XPathExprCompilerTest extends TestCase {

	public void testSampleCompile() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "/a/b[position()=2]";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(sampleResult, doc2String(gudDoc));
		} catch (Exception e) {
			fail("Exception"+e.toString());
		}
	}

	public void testNamteTestOnly() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "hoge";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(nameTestOnlyResult, doc2String(gudDoc));
		} catch (Exception e) {
			fail("Exception"+e.toString());
		}
	}

	public void testSimplePredicate() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "hoge[2]";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(simplePredicateResult, doc2String(gudDoc));
		} catch (Exception e) {
			fail("Exception"+e.toString());
		}
	}
	
	public void testCalc() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "1 + 2 * 3 - 4 div 5";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(calcResult, doc2String(gudDoc));
		} catch (Exception e) {
			fail("Exception"+e.toString());
		}
	}
	
	public void testFunctionCall() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "concat(output[1], model/scf/d)";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(funcationCallResult, doc2String(gudDoc));
		} catch (Exception e) {
			fail("Exception"+e.toString());
		}
	}

	public void testDeepNode() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "/a/b/c//d/e/f";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(deepNodeResult, doc2String(gudDoc));
		} catch (Exception e) {
			fail("Exception"+e.toString());
		}
	}

	public void testComplicationNode() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "a/b//c[a/b/@hoge=string(/foo//var)]";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(complicationNodeResult, doc2String(gudDoc));
		} catch (Exception e) {
			e.printStackTrace();
			fail("Exception"+e.toString());
		}
	}

	public void testParentNode() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "../method=\'credit\'";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(parentNodeResult, doc2String(gudDoc));
		} catch (Exception e) {
			//e.printStackTrace();
			Throwable h = e.getCause();
			if(h != null) h.printStackTrace();
			fail("Exception"+e.toString());
		}
	}

	public void testFunctionOnly() {
		DOMImplementation domi = new DOMImplementationImpl();
		Document gudDoc = domi.createDocument(GudBuilder.GUD_NS, "exprRoot", null);

		String xpath = "index('hoge')";
		try {
			Element exprRoot = gudDoc.getDocumentElement();
			
			XPathExprCompiler compiler = new XPathExprCompiler();
			exprRoot.appendChild(compiler.compile("test", xpath, gudDoc));

			assertEquals(funcationOnlyResult, doc2String(gudDoc));
		} catch (Exception e) {
			//e.printStackTrace();
			Throwable h = e.getCause();
			if(h != null) h.printStackTrace();
			fail("Exception"+e.toString());
		}
	}

	private String doc2String(Document doc) throws IOException {
		StringWriter out = new StringWriter();
		
		//o̓tH[}bg̎w
		OutputFormat formatter = new OutputFormat();
		//󔒕̕ۑ
		formatter.setPreserveSpace(true);
		formatter.setIndenting(false);
		formatter.setOmitXMLDeclaration(true);
		formatter.setEncoding("Shift_JIS");
		//XMLSerializer ̍쐬
		XMLSerializer serializer = new XMLSerializer(out, formatter);

		serializer.serialize(doc);
		return out.toString();
	}

	public static final String sampleResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:N2/>" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:_absolute/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"\" _1=\"\"/>" +
				"<xpath:step _0=\"a\" _1=\"\"/>" +
				"<xpath:step _0=\"b\" _1=\"\"/>" +
				"<xpath:nfilter/>" +
			"</xpath:expr>" +
		"</exprRoot>";

	public static final String nameTestOnlyResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:_child/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"hoge\" _1=\"\"/>" +
			"</xpath:expr>" +
		"</exprRoot>";

	public static String simplePredicateResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:N2/>" +
				"<xpath:_child/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"hoge\" _1=\"\"/>" +
				"<xpath:nfilter/>" +
			"</xpath:expr>" +
		"</exprRoot>";
	
	public static String calcResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:N1/>" +
				"<xpath:N2/>" +
				"<xpath:N3/>" +
				"<xpath:mult/>" +
				"<xpath:add/>" +
				"<xpath:N4/>" +
				"<xpath:N5/>" +
				"<xpath:div/>" +
				"<xpath:sub/>" +
			"</xpath:expr>" +
		"</exprRoot>";

	public static final String funcationCallResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:N1/>" +
				"<xpath:_child/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"output\" _1=\"\"/>" +
				"<xpath:nfilter/>" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"model\" _1=\"\"/>" +
				"<xpath:step _0=\"scf\" _1=\"\"/>" +
				"<xpath:step _0=\"d\" _1=\"\"/>" +
				"<xpath:fcall _0=\"concat\" _1=\"2\"/>" +
			"</xpath:expr>" +
		"</exprRoot>";

	public static final String deepNodeResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:_descendant-or-self/>" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:_absolute/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"\" _1=\"\"/>" +
				"<xpath:step _0=\"a\" _1=\"\"/>" +
				"<xpath:step _0=\"b\" _1=\"\"/>" +
				"<xpath:step _0=\"c\" _1=\"\"/>" +
				"<xpath:step _0=\"\" _1=\"node\"/>" +
				"<xpath:step _0=\"d\" _1=\"\"/>" +
				"<xpath:step _0=\"e\" _1=\"\"/>" +
				"<xpath:step _0=\"f\" _1=\"\"/>" +
			"</xpath:expr>" +
		"</exprRoot>";

	public static final String complicationNodeResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:clspush>" +
					"<xpath:_attribute/>" +
					"<xpath:_child/>" +
					"<xpath:_child/>" +
					"<xpath:ctxnload/>" +
					"<xpath:step _0=\"a\" _1=\"\"/>" +
					"<xpath:step _0=\"b\" _1=\"\"/>" +
					"<xpath:step _0=\"hoge\" _1=\"\"/>" +
					"<xpath:_child/>" +
					"<xpath:_descendant-or-self/>" +
					"<xpath:_child/>" +
					"<xpath:_absolute/>" +
					"<xpath:ctxnload/>" +
					"<xpath:step _0=\"\" _1=\"\"/>" +
					"<xpath:step _0=\"foo\" _1=\"\"/>" +
					"<xpath:step _0=\"\" _1=\"node\"/>" +
					"<xpath:step _0=\"var\" _1=\"\"/>" +
					"<xpath:fcall _0=\"string\" _1=\"1\"/>" +
					"<xpath:eq/>"+
				"</xpath:clspush>" +
				"<xpath:_child/>" +
				"<xpath:_descendant-or-self/>" +
				"<xpath:_child/>" +
				"<xpath:_child/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"a\" _1=\"\"/>" +
				"<xpath:step _0=\"b\" _1=\"\"/>" +
				"<xpath:step _0=\"\" _1=\"node\"/>" +
				"<xpath:step _0=\"c\" _1=\"\"/>" +
				"<xpath:filter/>" +
			"</xpath:expr>" +
		"</exprRoot>";

	public static final String parentNodeResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:_child/>" +
				"<xpath:_parent/>" +
				"<xpath:ctxnload/>" +
				"<xpath:step _0=\"\" _1=\"\"/>" +
				"<xpath:step _0=\"method\" _1=\"\"/>" +
				"<xpath:spush _0=\"credit\"/>" +
				"<xpath:eq/>" +
			"</xpath:expr>" +
		"</exprRoot>";

	public static final String funcationOnlyResult = 
		"<exprRoot>" +
			"<xpath:expr name=\"test\">" +
				"<xpath:spush _0=\"hoge\"/>" +
				"<xpath:fcall _0=\"index\" _1=\"1\"/>"+
			"</xpath:expr>" +
		"</exprRoot>";
	
}
