/*
 * Grain Core - A XForms processor for mobile terminals.
 * Copyright (C) 2005-2006 HAW International Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Created on 2006/05/19 9:38:14
 * 
 */
package jp.grain.spike.xpath;

import java.util.Stack;
import java.util.Vector;

import jp.grain.spike.Attribute;
import jp.grain.spike.Document;
import jp.grain.spike.Element;
import jp.grain.spike.Node;
import jp.grain.sprout.model.InstanceElement;
import jp.grain.sprout.ui.Form;

public class XPathEvaluator {

	public static final String AXIS_ABSOLUTE = "absolute";
	public static final String AXIS_CHILD = "child";
	public static final String AXIS_ATTRIBUTE = "attribute";
	public static final String AXIS_PARENT = "parent";
	public static final String AXIS_ANCESTOR = "ancestor";
	public static final String AXIS_DESCENDANT = "descendant";
	public static final String AXIS_DESCENDANT_OR_SELF = "descendant-or-self";
	public static final String AXIS_ANCESTOR_OR_SELF = "ancestor-or-self";
	public static final String AXIS_SELF = "self";
	public static final String AXIS_FOLLOWING_SIBLING = "following-sibling";
	public static final String AXIS_PRECEDING_SIBLING = "preceding-sibling";

	public static final String OP_PUSH = "push";
	public static final String OP_CLSPUSH = "clspush";

	public static final String OP_CTXNLOAD = "ctxnload";
	public static final String OP_CTXPLOAD = "ctxpload";

	public static final String OP_STEP = "step";
	public static final String OP_FILTER = "filter";
	public static final String OP_NFILTER = "nfilter";
	public static final String OP_FCALL = "fcall";

	public static final String OP_ADD = "add";
	public static final String OP_SUB = "sub";
	public static final String OP_DIV = "div";
	public static final String OP_MUL = "mul";
	public static final String OP_MOD = "mod";
	public static final String OP_NEG = "neg";

	public static final String OP_EQ = "eq";
	public static final String OP_NE = "ne";
	public static final String OP_LT = "lt";
	public static final String OP_LE = "le";
	public static final String OP_GT = "gt";
	public static final String OP_GE = "ge";

	public static final String OP_AND = "and";
	public static final String OP_OR = "or";

	public static final String NODETYPE_NODE = "node";
	public static final String NODETYPE_TEXT = "text";

	public static XPathEvaluator.Result evaluate(Node contextNode, XPathExpr expr) {
		Frame frame = new Frame(contextNode);
		frame.init(expr);
		exec(frame);
		Result result = new Result(frame._os.peek());
		return result;
	}

	private static XPathFunctionLibrary[] _flibraly = new XPathFunctionLibrary[]{
		new CoreFunctionLibrary(),
		new XFormsFunctionLibrary()
	};
	                                                                           

	/*
	 * ^ꂽt[ŎQƂĂIyhsB @param frame sΏۂ̃t[
	 * 
	 */
	private static void exec(Frame f) {
		try {
			for (f._oc = 0; f._oc < f._operands.length; f._oc++) {
				Object[] op = (Object[]) f._operands[f._oc];
				String opname = (String) op[0];
				if (OP_PUSH.equals(opname)) {
					f._os.push(op[1]);

				} else if (OP_CTXNLOAD.equals(opname)) {
					Node[] param = { f._ctxnode };
					f._os.push(param);

				} else if (OP_CTXPLOAD.equals(opname)) {
					Integer index = CoreFunctionLibrary.number(op[1]);
					f.storeNumber(f._ctxparam[index.intValue()]);

				} else if (OP_STEP.equals(opname)) {
					Node[] nodeset = (Node[]) f._os.pop();
					System.out.println("STEP: nodeset");
					String axis = (String) f._os.pop();
					System.out.println("STEP: axis="+axis);
					f.initBuffer();
					System.out.println("STEP: f.initNodeset()");
					NodeTester nc = new NodeTester(f._buf, (String) op[1], (String) op[2]);
					System.out.println("STEP: created NodeTester name=" + nc._name + ": type=" + nc._type);
					step(nodeset, axis, nc);
					System.out.println("STEP: step(nodeset, axis, nc)");
					f.storeNodeset();
					System.out.println("STEP: f.pushNodeset()");

				} else if (OP_FILTER.equals(opname)) {
					Node[] nodeset = (Node[]) f._os.pop();
					Object[] closure = (Object[]) f._os.pop();
					f.initBuffer();
					for (int i = 0; i < nodeset.length; ++i) {
						System.out.println("*** node=" + i);
						Frame cf = new Frame(nodeset[i], new Integer(nodeset.length), new Integer(i + 1));
						cf.init(closure, f._expr);
						exec(cf);
						Object result = cf._os.peek();
						System.out.println("*** result=" + result);
						if (result instanceof Integer) {
							int index = ((Integer) result).intValue();
							if (index != i + 1) continue;
						} else {
							if (!CoreFunctionLibrary.bool(result)) continue;
						}
						f._buf.addElement(nodeset[i]);
					}
					f.storeNodeset();

				} else if (OP_NFILTER.equals(opname)) {
					Node[] nodeset = (Node[]) f._os.pop();
					int index = CoreFunctionLibrary.number(f._os.pop()).intValue();
					if (index == 0) index = nodeset.length;
					if (index < 1 || index > nodeset.length) {
						f._os.push(new Node[0]);
					} else {
						Node[] filtered = { nodeset[index - 1] };
						f._os.push(filtered);
					}

				} else if (OP_FCALL.equals(opname)) {
					String funcName = (String) op[1];
					int argc = XPathFunctionLibrary.NO_ARGC;
					if (op.length > 2 && op[2] != null) argc = Integer.parseInt((String) op[2]);
					for(int i=0; i<_flibraly.length; i++) {
						if(_flibraly[i].call(funcName, f, argc)) break;
					}
				} else if (OP_ADD.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeNumber(null);
					} else {
						f.storeNumber(f._number[0].intValue() + f._number[1].intValue());
					}
				} else if (OP_SUB.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeNumber(null);
					} else {
						f.storeNumber(f._number[0].intValue() - f._number[1].intValue());
					}
				} else if (OP_MUL.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeNumber(null);
					} else {
						f.storeNumber(f._number[0].intValue() * f._number[1].intValue());
					}
				} else if (OP_DIV.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeNumber(null);
					} else {
						f.storeNumber(f._number[0].intValue() / f._number[1].intValue());
					}
				} else if (OP_MOD.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeNumber(null);
					} else {
						f.storeNumber(f._number[0].intValue() % f._number[1].intValue());
					}
				} else if (OP_NEG.equals(opname)) {
					f.loadUnaryNumber();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeNumber(null);
					} else {
						f.storeNumber(-f._number[0].intValue());
					}

				} else if (OP_LT.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeBoolean(false);
					} else {
						f.storeBoolean(f._number[0].intValue() < f._number[1].intValue());
					}
				} else if (OP_LE.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeBoolean(false);
					} else {
						f.storeBoolean(f._number[0].intValue() <= f._number[1].intValue());
					}
				} else if (OP_GT.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeBoolean(false);
					} else {
						f.storeBoolean(f._number[0].intValue() > f._number[1].intValue());
					}
				} else if (OP_GE.equals(opname)) {
					f.loadBinaryNumbers();
					if (f._number[0] == null || f._number[1] == null) {
						f.storeBoolean(false);
					} else {
						f.storeBoolean(f._number[0].intValue() >= f._number[1].intValue());
					}

				} else if (OP_EQ.equals(opname)) {
					f.loadBinaryStrings();
					if (f._string[0] == null || f._string[1] == null) {
						f.storeBoolean(false);
					} else {
						f.storeBoolean(f._string[0].equals(f._string[1]));
					}
				} else if (OP_NE.equals(opname)) {
					f.loadBinaryStrings();
					if (f._string[0] == null || f._string[1] == null) {
						f.storeBoolean(false);
					} else {
						f.storeBoolean(!(f._string[0].equals(f._string[1])));
					}
				} else if (OP_AND.equals(opname)) {
					f.loadBinaryBooleans();
					f.storeBoolean(f._bool[0] && f._bool[1]);
				} else if (OP_OR.equals(opname)) {
					f.loadBinaryBooleans();
					f.storeBoolean(f._bool[0] || f._bool[1]);
				}
			}
		} catch (Exception e) {
			throw new RuntimeException("operand=" + ((Object[]) f._operands[f._oc])[0] + ": " + e.toString());
		}
	}

	private static void step(Node[] nodeset, String axis, NodeTester nc) {
		for (int i = 0; i < nodeset.length; ++i) {
			Node node = nodeset[i];
			if (AXIS_CHILD.equals(axis)) {
				axisDescendant(node, nc, true);
			} else if (AXIS_DESCENDANT.equals(axis)) {
				axisDescendant(node, nc, false);
			} else if (AXIS_DESCENDANT_OR_SELF.equals(axis)) {
				nc.testAndRecord(node);
				axisDescendant(node, nc, false);
			} else if (AXIS_ABSOLUTE.equals(axis)) {
				nc.testAndRecord(node.getDocument());
			} else if (AXIS_ATTRIBUTE.equals(axis)) {
				Attribute[] attrs = ((Element) node).getAttributes();
				for (int j = 0; j < attrs.length; j++) {
					nc.testAndRecord(attrs[j]);
				}
			} else if (AXIS_PARENT.equals(axis)) {
				axisAncestor(node, nc, true);
			} else if (AXIS_ANCESTOR.equals(axis)) {
				axisAncestor(node, nc, false);
			} else if (AXIS_ANCESTOR_OR_SELF.equals(axis)) {
				nc.testAndRecord(node);
				axisAncestor(node, nc, false);
			} else if (AXIS_SELF.equals(axis)) {
				nc.testAndRecord(node);
			} else if (AXIS_FOLLOWING_SIBLING.equals(axis)) {
				axisFollowingSibling(node, nc);
			} else if (AXIS_PRECEDING_SIBLING.equals(axis)) {
				axisPrecedingSibling(node, nc);
			}
		}
	}

	private static void axisAncestor(Node node, NodeTester nc, boolean directOnly) {
		if (node instanceof Element) {
			Node parent = ((Element) node).getParent();
			nc.testAndRecord(parent);
			if (!directOnly) axisAncestor(parent, nc, false);
		}
	}

	private static void axisDescendant(Node node, NodeTester nc, boolean directOnly) {
		if (node instanceof Element) {
			Element e = (Element) node;
			for (int i = 0; i < e.getChildCount(); ++i) {
				Object child = e.getChild(i);
				nc.testAndRecord(child);
				if (!directOnly && child instanceof Node) {
					axisDescendant((Node) child, nc, false);
				}
			}
		} else if (node instanceof Document) {
			Element root = ((Document) node).getRootElement();
			nc.testAndRecord(root);
			if (!directOnly) axisDescendant(root, nc, false);
		}
	}

	private static void axisFollowingSibling(Node node, NodeTester nc) {
		Element parent = node.getParentElement();
		boolean following = false;
		for (int i = 0; i < parent.getChildCount(); ++i) {
			Object sibling = parent.getChild(i);
			if (sibling == node) {
				following = true;
			} else {
				if (following) nc.testAndRecord(sibling);
			}
		}
	}

	private static void axisPrecedingSibling(Node node, NodeTester nc) {
		Element parent = node.getParentElement();
		boolean preceding = false;
		for (int i = parent.getChildCount() - 1; i >= 0; --i) {
			Object sibling = parent.getChild(i);
			if (sibling == node) {
				preceding = true;
			} else {
				if (preceding) nc.testAndRecord(sibling);
			}
		}
	}

	static class Frame {

		Node _ctxnode;
		Integer[] _ctxparam = new Integer[2];
		Object[] _operands;
		Stack _os = new Stack();
		int _oc; // operand counter
		Vector _buf; // nodeset work area
		Integer[] _number = new Integer[2];
		String[] _string = new String[2];
		boolean[] _bool = new boolean[2];
		Object[] _args;
		XPathExpr _expr;
		
		Frame(Node context) {
			_ctxnode = context;
			_ctxparam[0] = new Integer(1);
			_ctxparam[1] = new Integer(1);
		}

		Frame(Node context, Integer contextSize, Integer contextPos) {
			_ctxnode = context;
			_ctxparam[0] = contextSize;
			_ctxparam[1] = contextPos;
		}

		void init(XPathExpr expr) {
			_expr = expr;
			_operands = expr.getOperands();
		}

		void init(Object[] closure, XPathExpr expr) {
			_operands = closure;
			_expr = expr;
		}

		void initBuffer() {
			if (_buf == null) {
				_buf = new Vector();
			}
			_buf.removeAllElements();
		}

		void loadBinaryNumbers() {
			_number[1] = CoreFunctionLibrary.number(_os.pop());
			_number[0] = CoreFunctionLibrary.number(_os.pop());
			System.out.println("EVALUATOR: binaryNumber: 0=" + _number[0] + " 1=" + _number[1]);
		}

		void loadBinaryStrings() {
			_string[1] = CoreFunctionLibrary.string(_os.pop());
			_string[0] = CoreFunctionLibrary.string(_os.pop());
			System.out.println("EVALUATOR: binaryString: 0=" + _string[0] + " 1=" + _string[1]);
		}

		void loadUnaryNumber() {
			_number[0] = CoreFunctionLibrary.number(_os.pop());
		}

		void loadBinaryBooleans() {
			_bool[1] = CoreFunctionLibrary.bool(_os.pop());
			_bool[0] = CoreFunctionLibrary.bool(_os.pop());
		}

		/*
		 * X^bNobt@Ƀ[h܂B Function̈擾ۂɗp܂B
		 * 
		 * @param size ̐
		 */
		void loadArgs(int size) {
			if (size < 0) size = 0;
			_args = new Object[size];
			for (int i = size - 1; i >= 0; --i) {
				_args[i] = _os.pop();
			}
		}

		/*
		 * obt@ɒlǉ܂B
		 */
		void put(Object obj) {
			_buf.addElement(obj);
		}

		void store(Object obj) {
			_os.push(obj);
		}

		void storeNumber(int number) {
			System.out.println("EVALUATOR: pushInt:" + number);
			_os.push(new Integer(number));
		}

		void storeNumber(Integer number) {
			System.out.println("EVALUATOR: pushInt:" + number);
			_os.push(number);
		}

		void storeBoolean(boolean b) {
			System.out.println("EVALUATOR: pushBoolean:" + b);
			_os.push(new Boolean(b));
		}

		void storeNodeset() {
			Node[] ns = new Node[_buf.size()];
			System.out.println("EVALUATOR: pushNodeset:" + ns.length);
			//_buf.copyInto(ns);
			for (int i = 0; i < ns.length; i++) {
				ns[i] = (Node)_buf.elementAt(i);
				if(ns[i] instanceof InstanceElement) {
					((InstanceElement)ns[i]).addXPathExpr(this._expr);
				}
				System.out.println("EVALUATOR: ns[" + i + "]:" + ns[i]);
			}
			_os.push(ns);
		}

		Object getArg(int pos) {
			return _args[pos];
		}

		Form getForm() {
			return (Form) _expr.getDocument();
		}
	}

	static public class Result {

		private Object result;

		protected Result(Object obj) {
			this.result = obj;
		}

		public String string() {
			return CoreFunctionLibrary.string(result);
		}

		public Integer number() {
			return CoreFunctionLibrary.number(result);
		}

		public boolean bool() {
			return CoreFunctionLibrary.bool(result);
		}

		public Node[] nodeset() {
			return CoreFunctionLibrary.nodeset(result);
		}

		public Class getType() {
			return this.result.getClass();
		}
	}

	static class NodeTester {

		private Vector _nodes;
		private String _name;
		private String _type;

		NodeTester(Vector nodes, String name, String type) {
			_nodes = nodes;
			_name = name;
			_type = type;
		}

		void testAndRecord(Object obj) {
			if (match(obj)) _nodes.addElement(obj);
		}

		boolean match(Object obj) {
			if (_type != null) {
				if (NODETYPE_NODE.equals(_type)) {
					return true;
				} else if (NODETYPE_TEXT.equals(_type) && obj instanceof String) {
					return true;
				}
			} else {
				if (obj instanceof Element) {
					if (_name == null) return true;
					if (_name.equals(((Element) obj).getName())) return true;
				} else if (obj instanceof Attribute) {
					if (_name == null) return true;
					if (_name.equals(((Attribute) obj)._name)) return true;
				} else if (obj instanceof Document) {
					if (_name == null) return true;
				}
			}
			System.out.println("NodeTester[_name=" + _name + ": _type=" + _type + ": obj=" + obj + "]");
			return false;
		}

	}
}
