/* $Id: entryint.cc 53 2004-01-12 19:17:04Z takekawa $
 * Copyright (C) 2003 Takashi Takekawa
 * This file is part of the Grs Library
 *
 * This library is free software; You may redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have recieved a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundtion, Inc., 59 Temple Place, Suite 330, Bostom, MA
 * 02111-1307 USA.
 */

#include "entryint.h"
#include <gdk/gdkkeysyms.h>

namespace {

const int pow_10[] = {
//  0  10  210  3210  43210  543210  6543210  76543210  876543210  9876543210
    1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000
};

Pango::AttrColor
create_attr_bg(Gdk::Color color)
{
    return Pango::Attribute::create_attr_background(
            color.get_red(), color.get_green(), color.get_blue());
}

Pango::AttrColor
create_attr_fg(Gdk::Color color)
{
    return Pango::Attribute::create_attr_foreground(
            color.get_red(), color.get_green(), color.get_blue());
}

}

namespace Grs {

EntryInt::EntryInt(Property<int>& property)
  : Gtk::DrawingArea()
  , m_value(property.get_value())
  , m_position(2)
  , m_layout(create_pango_layout("-undefined-"))
  , m_signal()
{
    property_can_focus() = true;
    add_events(Gdk::KEY_PRESS_MASK);
    add_events(Gdk::BUTTON_PRESS_MASK);

    init_text();
    init_cursol();

    property.connect(sigc::group(sigc::mem_fun(this, &EntryInt::set_value), sigc::_1));
    m_signal.connect(sigc::group(sigc::mem_fun(&property, &Property<int>::set_value), sigc::_1));
}

EntryInt::~EntryInt()
{
}

void
EntryInt::set_position(int pos)
{
    if (pos < 0) pos = 0;
    else if (10 < pos) pos = 10;
    if (pos != m_position) {
        m_position = pos;
        init_cursol();
        queue_draw();
    }
}

void
EntryInt::set_value(int val)
{
    if (val != m_value) {
        m_value = val;
        m_signal(m_value);
        init_text();
        queue_draw();
    }
}

void
EntryInt::on_style_changed(const Glib::RefPtr<Gtk::Style>&)
{
    m_layout->set_font_description(get_style()->get_font());
    int w, h;
    m_layout->get_size(w, h);
    set_size_request(w/Pango::SCALE, h/Pango::SCALE);
}

bool
EntryInt::on_expose_event(GdkEventExpose*)
{
    Glib::RefPtr<Gdk::Window> win = get_window();
    Glib::RefPtr<Gdk::GC> gc = Gdk::GC::create(win);
    win->draw_layout(gc, 0, 0, m_layout);
    return true;
}

int
EntryInt::get_position(double x, double y) const
{
    const int xx = int(x*Pango::SCALE);
    const int yy = int(y*Pango::SCALE);
    int index, trailing;
    m_layout->xy_to_index(xx, yy, index, trailing);
    if ((index <= 0) || (11 <= index))
        return -1;
    return 10 - index;
}

bool
EntryInt::on_scroll_event(GdkEventScroll* event)
{
    grab_focus();
    int pos = get_position(event->x, event->y);
    if (pos < 0) return false;
    set_position(pos);
    switch (event->direction)
    {
    case GDK_SCROLL_UP:
        set_value(m_value+pow_10[pos]);
        return true;
    case GDK_SCROLL_DOWN:
        set_value(m_value-pow_10[pos]);
        return true;
    default:
        break;
    }
    return false;
}

bool
EntryInt::on_button_press_event(GdkEventButton* event)
{
    grab_focus();
    if (event->type != GDK_BUTTON_PRESS) return false;
    int pos = get_position(event->x, event->y);
    if (pos < 0) return false;
    set_position(pos);
    switch (event->button)
    {
    case 1:
        set_value(m_value+pow_10[pos]);
        return true;
    case 3:
        set_value(m_value-pow_10[pos]);
        return true;
    default:
        break;
    }
    return false;
}

bool
EntryInt::on_key_press_event(GdkEventKey* event)
{
    int key = event->keyval;
    switch (key) {
    case GDK_Up:
        set_value(m_value+pow_10[m_position]);
        return true;
    case GDK_Down:
        set_value(m_value-pow_10[m_position]);
        return true;
    case GDK_Left:
        set_position(m_position + 1);
        return true;
    case GDK_Right:
        set_position(m_position - 1);
        return true;
    default:
        break;
    }
    if ('0' <= key && key <= '9') {
        int sign = (m_value < 0) ? -1 : 1;
        int absval = std::abs(m_value);
        int newval = int(key) - int('0');
        int oldval = absval / pow_10[m_position];
        oldval %= 10;
        set_value(sign*(absval+pow_10[m_position]*(newval-oldval)));
        set_position(m_position - 1);
        return true;
    }
    return false;
}

bool
EntryInt::on_focus_in_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_ACTIVE);
    init_cursol();
    queue_draw();
    return true;
}

bool
EntryInt::on_focus_out_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_NORMAL);
    init_cursol();
    queue_draw();
    return true;
}

void
EntryInt::init_text()
{
    //             "01234567890"
    char str[12] = "           ";
    unsigned value;
    if (m_value < 0) {
        str[0] = '-';
        value = -m_value;
    } else {
        str[0] = '+';
        value = m_value;
    }

    for (int i = 10; i > 0; --i) {
        str[i] = '0' + char(value%10);
        value /= 10;
        if (value == 0) break;
    }
    m_layout->set_text(str);
}

void
EntryInt::init_cursol()
{
    Gtk::StateType s2 = (has_focus() ? Gtk::STATE_SELECTED : Gtk::STATE_PRELIGHT);
    int index = (10 - m_position);
    Pango::AttrList list;
    Pango::AttrColor fg = create_attr_fg(get_style()->get_text(get_state()));
    Pango::AttrColor bg = create_attr_bg(get_style()->get_bg(get_state()));
    Pango::AttrColor cfg = create_attr_fg(get_style()->get_text(s2));
    Pango::AttrColor cbg = create_attr_bg(get_style()->get_bg(s2));
    fg.set_start_index(0);
    fg.set_end_index(12);
    bg.set_start_index(0);
    bg.set_end_index(12);
    cfg.set_start_index(index);
    cfg.set_end_index(index+1);
    cbg.set_start_index(index);
    cbg.set_end_index(index+1);
    list.insert(fg);
    list.insert(bg);
    list.insert(cfg);
    list.insert(cbg);
    m_layout->set_attributes(list);
}

}

