/* $Id: entryreal.cc 53 2004-01-12 19:17:04Z takekawa $
 * Copyright (C) 2003 Takashi Takekawa
 * This file is part of the Grs Library
 *
 * This library is free software; You may redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have recieved a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundtion, Inc., 59 Temple Place, Suite 330, Bostom, MA
 * 02111-1307 USA.
 */

#include "entryreal.h"
#include <cmath>
#include <gdk/gdkkeysyms.h>

namespace Grs {

namespace {

Pango::AttrColor
create_attr_bg(Gdk::Color color)
{
    return Pango::Attribute::create_attr_background(
            color.get_red(), color.get_green(), color.get_blue());
}

Pango::AttrColor
create_attr_fg(Gdk::Color color)
{
    return Pango::Attribute::create_attr_foreground(
            color.get_red(), color.get_green(), color.get_blue());
}

}

EntryReal::EntryReal(Property<double>& property, int digits)
  : Gtk::DrawingArea()
  , m_value(property.get_value())
  , m_digits(-1)
  , m_position(0)
  , m_layout(create_pango_layout("-undefined-"))
  , m_signal()
{
    property_can_focus() = true;
    add_events(Gdk::KEY_PRESS_MASK);
    add_events(Gdk::BUTTON_PRESS_MASK);

    set_digits(digits);
    init_text();

    property.connect(sigc::group(sigc::mem_fun(this, &EntryReal::set_value), sigc::_1));
    m_signal.connect(sigc::group(sigc::mem_fun(&property, &Property<double>::set_value), sigc::_1));
}

EntryReal::~EntryReal()
{
}

void
EntryReal::set_digits(int digits)
{
    if (digits < 0)
        digits = 0;
    else if (8 < digits)
        digits = 8;
    if (digits != m_digits) {
        m_digits = digits;
        init_text();
        set_position(m_position);
    }
}

void
EntryReal::set_position(int pos)
{
    if (pos < 0-m_digits)
        pos = 0-m_digits;
    else if (8-m_digits < pos)
        pos = 8-m_digits;
    if (pos != m_position) {
        m_position = pos;
        init_cursol();
        queue_draw();
    }
}

void
EntryReal::set_value(double val)
{
    if (val != m_value) {
        m_value = val;
        init_text();
        queue_draw();
        m_signal(m_value);
    }
}

void
EntryReal::on_style_changed(const Glib::RefPtr<Gtk::Style>&)
{
    m_layout->set_font_description(get_style()->get_font());
    int w, h;
    m_layout->get_size(w, h);
    set_size_request(w/Pango::SCALE, h/Pango::SCALE);
}

bool
EntryReal::on_expose_event(GdkEventExpose*)
{
    Glib::RefPtr<Gdk::Window> win = get_window();
    Glib::RefPtr<Gdk::GC> gc = Gdk::GC::create(win);
    win->draw_layout(gc, 0, 0, m_layout);
    return true;
}

int
EntryReal::get_position(double x, double y) const
{
    const int p = 10 - m_digits;
    const int xx = int(x*Pango::SCALE);
    const int yy = int(y*Pango::SCALE);
    int index, trailing;
    m_layout->xy_to_index(xx, yy, index, trailing);
    if (index <= 0)
        return -10;
    if ((index == p) || (10 < index))
        return -11;
    if (index < p)
        return p - index - 1;
    return p - index;
}

bool
EntryReal::on_scroll_event(GdkEventScroll* event)
{
    grab_focus();
    int pos = get_position(event->x, event->y);
    if (pos < -10) return false;
    if (pos == -10) {
        switch (event->direction)
        {
        case GDK_SCROLL_UP:
            set_digits(m_digits + 1);
            return true;
        case GDK_SCROLL_DOWN:
            set_digits(m_digits - 1);
            return true;
        default:
            break;
        }
    } else {
        set_position(pos);
        switch (event->direction)
        {
        case GDK_SCROLL_UP:
            set_value(m_value+std::pow(10.0, pos));
            return true;
        case GDK_SCROLL_DOWN:
            set_value(m_value-std::pow(10.0, pos));
            return true;
        case GDK_SCROLL_LEFT:
            set_digits(m_digits + 1);
            return true;
        case GDK_SCROLL_RIGHT:
            set_digits(m_digits - 1);
            return true;
        }
    }
    return false;
}

bool
EntryReal::on_button_press_event(GdkEventButton* event)
{
    grab_focus();
    if (event->type != GDK_BUTTON_PRESS) return false;
    int pos = get_position(event->x, event->y);
    if (pos < -10) return false;
    if (pos == -10) {
        switch (event->button)
        {
        case 1:
            set_digits(m_digits + 1);
            return true;
        case 3:
            set_digits(m_digits - 1);
            return true;
        }
    } else {
        set_position(pos);
        switch (event->button)
        {
        case 1:
            set_value(m_value+std::pow(10.0, pos));
            return true;
        case 3:
            set_value(m_value-std::pow(10.0, pos));
            return true;
        }
    }
    return false;
}

bool
EntryReal::on_key_press_event(GdkEventKey* event)
{
    int key = event->keyval;
    switch (key) {
    case GDK_Up:
        set_value(m_value+std::pow(10.0, m_position));
        return true;
    case GDK_Down:
        set_value(m_value-std::pow(10.0, m_position));
        return true;
    case GDK_Left:
        set_position(m_position + 1);
        return true;
    case GDK_Right:
        set_position(m_position - 1);
        return true;
    case GDK_Page_Up:
        set_digits(m_digits + 1);
        return true;
    case GDK_Page_Down:
        set_digits(m_digits - 1);
        return true;
    }
    if ('0' <= key && key <= '9') {
        double absval = std::abs(m_value);
        double newval = double(key) - double('0');
        double oldval = absval * std::pow(10.0, -m_position);
        oldval -= 10.0*std::floor(0.1*oldval);
        oldval = std::floor(oldval);
        double diff = std::pow(10.0, m_position)*(newval-oldval);
        set_value(::copysign(diff, m_value));
        set_position(m_position - 1);
        return true;
    }
    return false;
}

bool
EntryReal::on_focus_in_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_ACTIVE);
    init_cursol();
    queue_draw();
    return true;
}

bool
EntryReal::on_focus_out_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_NORMAL);
    init_cursol();
    queue_draw();
    return true;
}

void
EntryReal::init_text()
{
    double dval = ::rint(m_value*std::pow(10.0, m_digits));
    if (1e9 <= dval) {
        m_layout->set_text("over flow  ");
        return;
    }
    if (dval <= -1e9) {
        m_layout->set_text("under flow ");
        return;
    }

    const int p = 10-m_digits;
    //             "01234567890"
    char str[12] = "           ";
    long value = long(dval);
    if (value < 0) {
        str[0] = '-';
        value = -value;
    } else {
        str[0] = '+';
    }

    for (int i = 10; i > p; --i) {
        str[i] = '0' + char(value%10);
        value /= 10;
    }

    str[p] = '.';

    for (int i = p-1; i > 0; --i) {
        str[i] = '0' + char(value%10);
        value /= 10;
        if (value == 0) break;
    }
    m_layout->set_text(str);
}

void
EntryReal::init_cursol()
{
    Gtk::StateType s2 = (has_focus() ? Gtk::STATE_SELECTED : Gtk::STATE_PRELIGHT);
    int index = (m_position < 0) ? (10 - m_digits - m_position)
                                 : ( 9 - m_digits - m_position); 
    Pango::AttrColor fg = create_attr_fg(get_style()->get_text(get_state()));
    Pango::AttrColor bg = create_attr_bg(get_style()->get_bg(get_state()));
    Pango::AttrColor cfg = create_attr_fg(get_style()->get_text(s2));
    Pango::AttrColor cbg = create_attr_bg(get_style()->get_bg(s2));
    fg.set_start_index(0);
    fg.set_end_index(12);
    bg.set_start_index(0);
    bg.set_end_index(12);
    cfg.set_start_index(index);
    cfg.set_end_index(index+1);
    cbg.set_start_index(index);
    cbg.set_end_index(index+1);

    Pango::AttrList list;
    list.insert(fg);
    list.insert(bg);
    list.insert(cfg);
    list.insert(cbg);
    m_layout->set_attributes(list);
}

}

