/* $Id: graphstream.cc 36 2003-12-20 14:25:00Z takekawa $
 * Copyright (C) 2003 Takashi Takekawa
 * This file is part of the Grs Library
 *
 * This library is free software; You may redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have recieved a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundtion, Inc., 59 Temple Place, Suite 330, Bostom, MA
 * 02111-1307 USA.
 */

#include "graphstream.h"
#include <cmath>
#include <limits>

namespace Grs {

GraphStream::GraphStream()
    : m_data(1)
{
}

Gnome::Canvas::Points
GraphStream::get_points(const Function& fy)
{
    Gnome::Canvas::Points points;
    for (std::size_t i = 0; i < m_data.size()-1; ++i) {
        double x = m_data[i][0];
        double y = fy(&m_data[i][1]);
        if (::finite(x) && ::finite(y))
            points.push_back(Gnome::Art::Point(x, y));
    }
    return points;
}

Gnome::Canvas::Points
GraphStream::get_points(const Function& fx, const Function& fy)
{
    Gnome::Canvas::Points points;
    for (std::size_t i = 0; i < m_data.size()-1; ++i) {
        double x = fx(&m_data[i][1]);
        double y = fy(&m_data[i][1]);
        if (::finite(x) && ::finite(y))
            points.push_back(Gnome::Art::Point(x, y));
    }
    return points;
}

namespace {

double
get_val(double z_min, double dz, unsigned short zlevel, double z)
{
    return std::max(0.0, std::min(1.0, ::floor(dz*(z - z_min))/zlevel));
}

}

GridImage
GraphStream::get_gridimage(Graph& graph,
        double z_min, double z_max, unsigned short zlevel)
{
    return get_gridimage(graph, graph.x1, graph.x2, graph.y1, graph.y2,
            z_min, z_max, zlevel);
}

GridImage
GraphStream::get_gridimage(Gnome::Canvas::Group& group,
        double x_min, double x_max, double y_min, double y_max,
        double z_min, double z_max, unsigned short zlevel)
{
    GridImage image;
    Gdk::Color color;
    const double dz = zlevel/(z_max - z_min);
    const double dy = (y_max - y_min)/(m_data.size()-2);
    for (std::size_t i = 0; i < m_data.size()-1; ++i) {
        const double y1 = y_min + i*dy - 0.5*dy;
        const double y2 = y_min + i*dy + 0.5*dy;
        const double dx = (x_max - x_min)/(m_data[i].size()-1);
        double x1 = x_min - 0.5*dx;
        double val1 = get_val(z_min, dz, zlevel, m_data[i][0]);
        color.set_grey_p(val1);
        for (std::size_t j = 1; j < m_data[i].size()-1; ++j) {
            const double val2 = get_val(z_min, dz, zlevel, m_data[i][j]);
            if (val1 != val2) {
                const double x2 = x_min + j*dx + 0.5*dx;
                image.push_back(Glib::RefPtr<Gnome::Canvas::Rect>(
                    new Gnome::Canvas::Rect(group, x1, y1, x2, y2)));
                image.back()->property_fill_color_gdk().set_value(color);
                x1 = x2;
                val1 = val2;
                color.set_grey_p(val1);
            }
        }
        const double x2 = x_max + 0.5*dx;
        image.push_back(Glib::RefPtr<Gnome::Canvas::Rect>(
            new Gnome::Canvas::Rect(group, x1, y1, x2, y2)));
        image.back()->property_fill_color_gdk().set_value(color);
    }
    return image;
}


Gnome::Canvas::Line*
GraphStream::get_line(Gnome::Canvas::Group& group, const Function& fy)
{
    return manage(new Gnome::Canvas::Line(group, get_points(fy)));
}

Gnome::Canvas::Line*
GraphStream::get_line(Gnome::Canvas::Group& group,
        const Function& fx, const Function& fy)
{
    return manage(new Gnome::Canvas::Line(group, get_points(fx, fy)));
}

void
GraphStream::clear()
{
    m_data.clear();
    m_data.push_back(std::vector<double>());
}

GraphStream&
GraphStream::operator<<(double data)
{
    m_tmp = data;
    return *this;
}

GraphStream&
GraphStream::operator<<(char ch)
{
    switch (ch)
    {
    case '\n':
        m_data.back().push_back(m_tmp);
        m_data.push_back(std::vector<double>());
        break;
    case ' ':
    case '\t':
        m_data.back().push_back(m_tmp);
        break;
    case '-':
        m_data.back().push_back(std::numeric_limits<double>::quiet_NaN());
        break;
    }
    return *this;
}

std::ostream&
operator<<(std::ostream& out, const GraphStream& g)
{
    for (std::size_t i = 0; i < g.m_data.size(); ++i) {
        out << i;
        for (std::size_t j = 0; j < g.m_data[i].size(); ++j)
            out << '\t' << g.m_data[i][j];
        out << '\n';
    }
    return out;
}

}
