/* $Id: entryint.cc 24 2003-12-20 01:38:09Z takekawa $
 * Copyright (C) 2003 Takashi Takekawa
 * This file is part of the Grs Library
 *
 * This library is free software; You may redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have recieved a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundtion, Inc., 59 Temple Place, Suite 330, Bostom, MA
 * 02111-1307 USA.
 */

#include "entryint.h"
#include <gdk/gdkkeysyms.h>

#include <boost/bind.hpp>
using boost::bind;
using boost::ref;

namespace {

const int pow_10[] = {
//  0  10  210  3210  43210  543210  6543210  76543210  876543210  9876543210
    1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000
};

}

namespace Grs {

EntryInt::EntryInt(Property<int>& property)
    : Gtk::DrawingArea(), m_value(property.get_value()),
      m_position(2), m_signal()
{
    property_can_focus() = true;
    add_events(Gdk::KEY_PRESS_MASK);
    add_events(Gdk::BUTTON_PRESS_MASK);
    m_layout = create_pango_layout("-undefined-");
    init_text();
    init_cursol();

    property.connect(bind(&EntryInt::set_value, ref(*this), _1));
    m_signal.connect(bind(&Property<int>::set_value, ref(property), _1));
}

EntryInt::~EntryInt()
{
}

void
EntryInt::set_position(int pos)
{
    if (pos < 0) pos = 0;
    if (10 < pos) pos = 10;
    m_position = pos;
    init_cursol();
    queue_draw();
}

void
EntryInt::set_value(int val)
{
    m_value = val;
    m_signal(m_value);
    init_text();
    queue_draw();
}

void
EntryInt::on_style_changed(const Glib::RefPtr<Gtk::Style>&)
{
    m_layout->set_font_description(get_style()->get_font());
    Pango::Rectangle rect = m_layout->get_pixel_logical_extents();
    set_size_request(rect.get_width(), rect.get_height());
}

bool
EntryInt::on_expose_event(GdkEventExpose*)
{
    Gtk::StateType state = get_state();
    Glib::RefPtr<Gdk::Drawable> win = get_window();
    Glib::RefPtr<Gdk::GC> gc = get_style()->get_base_gc(state);
    gc->set_foreground(get_style()->get_text(state));
    gc->set_background(get_style()->get_bg(state));
    win->draw_layout(gc, 0, 0, m_layout);
    return true;
}

int
EntryInt::get_position(double x, double y) const
{
    const int xx = int(x*Pango::SCALE);
    const int yy = int(y*Pango::SCALE);
    int index, trailing;
    m_layout->xy_to_index(xx, yy, index, trailing);
    if ((index <= 0) || (11 <= index))
        return -1;
    return 10 - index;
}

bool
EntryInt::on_scroll_event(GdkEventScroll* event)
{
    grab_focus();
    int pos = get_position(event->x, event->y);
    if (pos < 0) return false;
    set_position(pos);
    switch (event->direction)
    {
    case GDK_SCROLL_UP:
        set_value(m_value+pow_10[pos]);
        return true;
    case GDK_SCROLL_DOWN:
        set_value(m_value-pow_10[pos]);
        return true;
    default:
        break;
    }
    return false;
}

bool
EntryInt::on_button_press_event(GdkEventButton* event)
{
    grab_focus();
    if (event->type != GDK_BUTTON_PRESS) return false;
    int pos = get_position(event->x, event->y);
    if (pos < 0) return false;
    set_position(pos);
    switch (event->button)
    {
    case 1:
        set_value(m_value+pow_10[pos]);
        return true;
    case 3:
        set_value(m_value-pow_10[pos]);
        return true;
    default:
        break;
    }
    return false;
}

bool
EntryInt::on_key_press_event(GdkEventKey* event)
{
    int key = event->keyval;
    switch (key) {
    case GDK_Up:
        set_value(m_value+pow_10[m_position]);
        return true;
    case GDK_Down:
        set_value(m_value-pow_10[m_position]);
        return true;
    case GDK_Left:
        set_position(m_position + 1);
        return true;
    case GDK_Right:
        set_position(m_position - 1);
        return true;
    default:
        break;
    }
    if ('0' <= key && key <= '9') {
        int sign = (m_value < 0) ? -1 : 1;
        int absval = std::abs(m_value);
        int newval = int(key) - int('0');
        int oldval = absval / pow_10[m_position];
        oldval %= 10;
        set_value(sign*(absval+pow_10[m_position]*(newval-oldval)));
        set_position(m_position - 1);
        return true;
    }
    return false;
}

bool
EntryInt::on_focus_in_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_ACTIVE);
    init_cursol();
    queue_draw();
    return true;
}

bool
EntryInt::on_focus_out_event(GdkEventFocus* event)
{
    set_state(Gtk::STATE_NORMAL);
    init_cursol();
    queue_draw();
    return true;
}

void
EntryInt::init_text()
{
    //             "01234567890"
    char str[12] = "           ";
    unsigned value;
    if (m_value < 0) {
        str[0] = '-';
        value = -m_value;
    } else {
        str[0] = '+';
        value = m_value;
    }

    for (int i = 10; i > 0; --i) {
        str[i] = '0' + char(value%10);
        value /= 10;
        if (value == 0) break;
    }
    m_layout->set_text(str);
}

void
EntryInt::init_cursol()
{
    Pango::AttrList list;
    int index = (10 - m_position);
    Gtk::StateType state = (has_focus() ? Gtk::STATE_SELECTED : get_state());
    {
        Gdk::Color c = get_style()->get_text(state);
        Pango::AttrColor attr
            = Pango::Attribute::create_attr_foreground(
                    c.get_red(), c.get_green(), c.get_blue());
        attr.set_start_index(index);
        attr.set_end_index(index+1);
        list.insert(attr);
    }
    {
        Gdk::Color c = get_style()->get_bg(state);
        Pango::AttrColor attr
            = Pango::Attribute::create_attr_background(
                    c.get_red(), c.get_green(), c.get_blue());
        attr.set_start_index(index);
        attr.set_end_index(index+1);
        list.insert(attr);
    }
    m_layout->set_attributes(list);
}

}

