/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prime-candidate-window.h"

#include <glib/gi18n.h>

typedef struct _PrimeCandidateWindowPrivate	PrimeCandidateWindowPrivate;
struct _PrimeCandidateWindowPrivate
{
	GtkWidget    *candidate_tree;
	GtkWidget    *candidate_box;
	GtkWidget    *numlabel_tree;
	GtkWidget    *preference_button;
	GtkListStore *candidate_list;

	gint	      cell_height;
	gint	      n_candidates;
	gint	      selected_candidate;
};

static const gchar *num[10] =
{ "1", "2", "3", "4", "5", "6", "7", "8", "9", "0" };

#define PRIME_CANDIDATE_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), PRIME_TYPE_CANDIDATE_WINDOW, PrimeCandidateWindowPrivate))


enum {
	COLUMN_TERMINATOR = -1,
	COLUMN_CANDIDATE,
	COLUMN_COMMENT,
	N_COLUMN
};

static void prime_candidate_window_class_init (PrimeCandidateWindowClass *klass);
static void prime_candidate_window_init       (PrimeCandidateWindow *prime_window);
static void prime_candidate_window_dispose    (GObject *object);

static gboolean cb_flag_button_released       (GtkWidget *widget,
					       GdkEventButton *event,
					       PrimeCandidateWindow *window);
static gboolean cb_preference_button_clicked  (GtkWidget *widget,
					       GdkEventButton *event,
					       PrimeCandidateWindow *window);

static GtkWindowClass *parent_class     = NULL;

void
prime_candidate_window_register_type (GTypeModule *module)
{
	if (!prime_type_candidate_window)
	{
		static const GTypeInfo window_info =
		{
			sizeof (PrimeCandidateWindowClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) prime_candidate_window_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (PrimeCandidateWindow),
			0,		/* n_preallocs */
			(GInstanceInitFunc) prime_candidate_window_init,
		};
		prime_type_candidate_window = 
			g_type_module_register_type (module,
						     GTK_TYPE_WINDOW,
						     "PrimeCandidateWindow",
						     &window_info, 0);
	}
}

static void
prime_candidate_window_class_init (PrimeCandidateWindowClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;

	object_class->dispose  = prime_candidate_window_dispose;
	
	g_type_class_add_private (object_class, sizeof(PrimeCandidateWindowPrivate));
}


static void
prime_candidate_window_init (PrimeCandidateWindow *prime_window)
{
	GtkWidget *vbox, *scrwin, *statusbar, *icon, *hbox, *tree;
	GtkWidget *button, *image;
	GtkRcStyle *style;
	GtkTreeView *treeview;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkListStore *list;
	gint cell_height, vertical_separator, i;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (prime_window);

	/* main vbox */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (prime_window), vbox);
	gtk_widget_show (vbox);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	priv->candidate_box = hbox;

	/* number label */
	list = gtk_list_store_new (1, G_TYPE_STRING);
	tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
	priv->numlabel_tree = tree;
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (tree), FALSE);
	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes ("Number",
						        renderer,
						        "text", 0,
						        NULL);
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree), col);
	for (i = 0; i < 10; i++)
	{
		GtkTreeIter iter;
		GtkTreeViewColumn *col;

		gtk_list_store_append (list, &iter);
		gtk_list_store_set (list, &iter,
				    0, num[i],
				    -1);
	}
	g_object_unref (list);
	gtk_widget_show (tree);
	gtk_box_pack_start (GTK_BOX (hbox), tree, FALSE, FALSE, 1);

	/* scroll window for candidate tree */
	scrwin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin),
				        GTK_POLICY_NEVER,
				        GTK_POLICY_ALWAYS);
        gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrwin),
					     GTK_SHADOW_NONE);
	gtk_container_set_border_width (GTK_CONTAINER (scrwin), 0);
	gtk_box_pack_start (GTK_BOX (priv->candidate_box), scrwin, TRUE, TRUE, 0);
	gtk_widget_show (scrwin);

	/* candidate tree */
	priv->candidate_list = gtk_list_store_new (N_COLUMN,
						   G_TYPE_STRING,
						   G_TYPE_STRING);
	priv->candidate_tree = 
		gtk_tree_view_new_with_model (GTK_TREE_MODEL (priv->candidate_list));
	treeview = GTK_TREE_VIEW (priv->candidate_tree);
	gtk_tree_view_set_headers_visible (treeview, FALSE);
	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes ("Candidate",
						        renderer,
						        "text", COLUMN_CANDIDATE,
						        NULL);
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	gtk_tree_view_append_column (treeview, col);
	gtk_cell_renderer_get_size (renderer, priv->candidate_tree,
				    NULL, NULL, NULL,
				    NULL, &cell_height);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes ("Comment",
						        renderer,
						        "text", COLUMN_COMMENT,
						        NULL);
	gtk_tree_view_append_column (treeview, col);

	gtk_container_add (GTK_CONTAINER (scrwin), 
			   priv->candidate_tree);
  	gtk_widget_style_get (priv->candidate_tree,
			      "vertical_separator", &vertical_separator,
			      NULL);
	priv->cell_height = cell_height + vertical_separator * 2;
	gtk_widget_set_size_request (priv->candidate_tree, -1,
				     priv->cell_height * 10);
	gtk_widget_show (priv->candidate_tree);

	/* status bar */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);

	/* hi no maru */
	button = gtk_button_new ();
	image = gtk_image_new_from_file (GTKIMPRIME_DATADIR"/icons/gtkimprime-jp.png");
	gtk_container_add (GTK_CONTAINER (button), image);
	gtk_widget_show (image);
	g_signal_connect (G_OBJECT (button), "released",
			  G_CALLBACK (cb_flag_button_released), prime_window);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 1);
	gtk_widget_show (button);
	style = gtk_rc_style_new ();
	style->xthickness = style->ythickness = 0;
	gtk_widget_modify_style (button, style);
	gtk_rc_style_unref (style);
	
	button = gtk_button_new ();
	image = gtk_image_new_from_file (GTKIMPRIME_DATADIR"/icons/gtkimprime-preference.png");
	gtk_container_add (GTK_CONTAINER (button), image);
	gtk_widget_show (image);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (cb_preference_button_clicked), prime_window);
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	priv->preference_button = button;
	style = gtk_rc_style_new ();
	style->xthickness = style->ythickness = 0;
	gtk_widget_modify_style (button, style);
	gtk_rc_style_unref (style);

	statusbar = gtk_statusbar_new ();
	gtk_statusbar_set_has_resize_grip (GTK_STATUSBAR (statusbar), FALSE);
	gtk_box_pack_start (GTK_BOX (hbox), statusbar, TRUE, TRUE, 0);
	/* gtk_widget_show (statusbar); */

	gtk_window_set_default_size (GTK_WINDOW (prime_window), 20, -1);
	gtk_window_set_resizable (GTK_WINDOW (prime_window), FALSE);
}


static void
prime_candidate_window_dispose (GObject *object)
{
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (object);
	
	if (priv->candidate_list)
	{
		g_object_unref(priv->candidate_list);
		priv->candidate_list = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


PrimeCandidateWindow *
prime_candidate_window_new (void)
{
	return g_object_new (PRIME_TYPE_CANDIDATE_WINDOW,
			     "type", GTK_WINDOW_POPUP,
			     NULL);
}

void
prime_candidate_window_set_candidates (PrimeCandidateWindow *window,
				       gchar **candidates)
{
	guint n = 0;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (window);
	g_return_if_fail (PRIME_IS_CANDIDATE_WINDOW (window));

	gtk_list_store_clear (priv->candidate_list);	

	if (candidates) 
	{
		while (candidates[n])
		{
			GtkTreeIter iter;
			GtkTreeViewColumn *col;

			gtk_list_store_append (priv->candidate_list, &iter);
			gtk_list_store_set (priv->candidate_list, &iter,
					    COLUMN_CANDIDATE, candidates[n],
					    COLUMN_TERMINATOR);
			n++;
		}
		priv->n_candidates = n; 
		if (!GTK_WIDGET_VISIBLE (priv->candidate_box))
		{
			GtkTreePath *path;
    			path = gtk_tree_path_new_first ();
			gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->candidate_tree),
						  path, NULL, FALSE);
    			gtk_tree_path_free (path);
			gtk_widget_show (priv->candidate_box);
			gtk_widget_show (priv->preference_button);

			priv->selected_candidate = 0;
		}
	}
	else
	{
		priv->n_candidates = 0; 
		if (GTK_WIDGET_VISIBLE (priv->candidate_box))
		{
			gtk_widget_hide (priv->candidate_box);
			gtk_widget_hide (priv->preference_button);
		}
	}
}

void   
prime_candidate_window_prev_page (PrimeCandidateWindow *window)
{
	GtkTreePath *path;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (window);

	gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->candidate_tree),
				  &path, NULL);
	if (path)
	{
		GtkTreeIter iter;
		gint *indices;
		gint index;

		indices = gtk_tree_path_get_indices (path);
		if (!indices)
		{
			gtk_tree_path_free (path);
			return;
		}

		index = indices[0];
		if (index == 0)
			index = priv->n_candidates - 1;
		else
		{
			index -= 10;
			if (index < 0)
				index = 0; 
		}
		priv->selected_candidate = index;
		gtk_tree_path_free (path);

		path = gtk_tree_path_new_from_indices (index, -1);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->candidate_tree),
				  	  path, NULL, FALSE);
		gtk_tree_path_free (path);
	}
}

void
prime_candidate_window_next_page (PrimeCandidateWindow *window)
{
	GtkTreePath *path;
	gchar *selected;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (window);

	gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->candidate_tree),
				  &path, NULL);

	if (path)
	{
		GtkTreeIter iter;
		gint *indices;
		gint index;

		indices = gtk_tree_path_get_indices (path);
		
		if (!indices) 
		{
			gtk_tree_path_free (path);
			return;
		}
		index = indices[0];
		if (index == priv->n_candidates -1)
			index = 0;
		else
		{
			index += 10;
			if (index >= priv->n_candidates)
				index = priv->n_candidates - 1;
		}
		priv->selected_candidate = index;
		gtk_tree_path_free (path);

		path = gtk_tree_path_new_from_indices (index, -1);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->candidate_tree),
				  	  path, NULL, FALSE);
		gtk_tree_path_free (path);
	}
}

void
prime_candidate_window_select_prev_candidate (PrimeCandidateWindow *window)
{
	GtkTreePath *path;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (window);

	priv->selected_candidate--;
	if (priv->selected_candidate < 0)
		priv->selected_candidate = priv->n_candidates - 1;

	path = gtk_tree_path_new_from_indices (priv->selected_candidate, -1);
	gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->candidate_tree),
				  path, NULL, FALSE);
	
	gtk_tree_path_free (path);
}


void
prime_candidate_window_select_next_candidate (PrimeCandidateWindow *window)
{
	GtkTreePath *path;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (window);

	priv->selected_candidate++;
	if (priv->selected_candidate == priv->n_candidates)
		priv->selected_candidate = 0;

	path = gtk_tree_path_new_from_indices (priv->selected_candidate, -1);
	gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->candidate_tree),
				  path, NULL, FALSE);
	
	gtk_tree_path_free (path);
}


gchar *
prime_candidate_window_get_candidate (PrimeCandidateWindow *window)
{
	GtkTreePath *path;
	gchar *selected = NULL;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (window);

	gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->candidate_tree),
				  &path, NULL);
	if (path)
	{
		GtkTreeIter iter;
		GValue value = { 0 };

		gtk_tree_model_get_iter (GTK_TREE_MODEL (priv->candidate_list),
					 &iter, path);
		gtk_tree_model_get_value (GTK_TREE_MODEL (priv->candidate_list),
					  &iter, COLUMN_CANDIDATE,
					  &value);
		selected = g_value_dup_string (&value);
		gtk_tree_path_free (path);
	}

	return selected;
}


gchar *
prime_candidate_window_get_candidate_from_index (PrimeCandidateWindow *window,
						 guint index)
{
	GtkTreePath *path;
	gchar *selected;
	gint x, y;
	PrimeCandidateWindowPrivate *priv = PRIME_CANDIDATE_WINDOW_GET_PRIVATE (window);

	x = 0;
	y = priv->cell_height * index + priv->cell_height / 2;
	if (!gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (priv->candidate_tree),
				       x, y, &path,
				       NULL, NULL, NULL))
				       return NULL;

	if (path)
	{
		GtkTreeIter iter;
		GValue value = { 0 };

		gtk_tree_model_get_iter (GTK_TREE_MODEL (priv->candidate_list),
					 &iter, path);
		gtk_tree_model_get_value (GTK_TREE_MODEL (priv->candidate_list),
					  &iter, COLUMN_CANDIDATE,
					  &value);
		selected = g_value_dup_string (&value);
		gtk_tree_path_free (path);
	}

	return selected;
}

static gboolean 
cb_flag_button_released (GtkWidget *widget,
			GdkEventButton *event,
			PrimeCandidateWindow *window)
{
	g_warning (_("flag button released"));
}

static gboolean 
cb_preference_button_clicked (GtkWidget *widget,
			      GdkEventButton *event,
			      PrimeCandidateWindow *window)
{
}

