/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "prime.h"

#include <glib/gi18n.h>

typedef struct _PrimePrivate PrimePrivate;

struct _PrimePrivate
{
	GPid prime_pid;
	gint prime_input;
	gint prime_output;
	gint prime_error;
};

#define PRIME_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), PRIME_TYPE, PrimePrivate))

#define PRIME_LOOKUP_COMPACT	    "lookup_compact\t%s\n"
#define PRIME_LOOKUP_ALL	    "lookup_all\t%s\n"
#define PRIME_LOOKUP_EXACT	    "lookup_exact\t%s\n"
#define PRIME_LOOKUP_HYBRID	    "lookup_hybrid\t%s\n"
#define PRIME_GET_LABEL 	    "get_label\t%s\n"
#define PRIME_PREEDIT_CONVERT_INPUT "preedit_convert_input\t%s\n"
#define PRIME_LEARN_WORD            "learn_word\t%s\t%s\t%s\t%s\t%s\t%s\n"
#define PRIME_CLOSE		    "close\n"

static void	prime_class_init   (PrimeClass *klass);
static void     prime_init         (Prime      *prime);
static void     prime_dispose      (GObject    *object);
static void     prime_finalize     (GObject    *object);

static void     prime_close        (Prime      *prime);

static GObjectClass *parent_class = NULL;


void
prime_register_type (GTypeModule *module)
{
	if (!prime_type)
	{
		static const GTypeInfo prime_info =
		{
			sizeof (PrimeClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) prime_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (Prime),
			0,		/* n_preallocs */
			(GInstanceInitFunc) prime_init,
		};

		prime_type = 
			g_type_module_register_type (module,
						     G_TYPE_OBJECT,
						     "Prime",
						     &prime_info, 0);
	}
}


static void
prime_class_init (PrimeClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
  
	gobject_class->dispose  = prime_dispose;
	gobject_class->finalize = prime_finalize;

	g_type_class_add_private (gobject_class, sizeof(PrimePrivate));
}

static void
prime_init (Prime *prime)
{
	const gchar *prime_command = "prime";
	gint argc;
	gchar **argv = NULL;
	GSpawnFlags flags;
	gboolean ret;
	PrimePrivate *priv = PRIME_GET_PRIVATE (prime);

	g_shell_parse_argv(prime_command,
			   &argc,
			   &argv,
			   NULL);

	flags = G_SPAWN_SEARCH_PATH; 
	ret = g_spawn_async_with_pipes(NULL,
				       argv,
				       NULL,
				       flags,
				       NULL,
				       NULL,
				       &priv->prime_pid,
				       &priv->prime_input,
				       &priv->prime_output,
				       &priv->prime_error,
				       NULL);
	g_strfreev(argv);

}

static void
prime_dispose (GObject *object)
{
	PrimePrivate *priv = PRIME_GET_PRIVATE (object);

	if (priv->prime_pid)
	{
		prime_close (PRIME (object));
		g_spawn_close_pid (priv->prime_pid);
	}

	priv->prime_pid = 0;

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
prime_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

Prime *
prime_new (void)
{
	return g_object_new (PRIME_TYPE, NULL);
}


static GIOStatus
prime_send_command (Prime *prime, const gchar *command)
{
	PrimePrivate *priv;
	GIOChannel *in;
	gsize bytes;
	GIOStatus status;
	
	priv = PRIME_GET_PRIVATE (prime);
	
	in  = g_io_channel_unix_new (priv->prime_input);
	g_io_channel_set_encoding (in, NULL, NULL);
	
	status = g_io_channel_write_chars (in,
				           command,
				           strlen(command),
				           &bytes,
				           NULL);
	g_io_channel_flush (in, NULL);
	g_io_channel_unref (in);

	return status;
}


static void
prime_close (Prime *prime)
{
	prime_send_command (prime, PRIME_CLOSE);
}


static gchar **
prime_process_command (Prime *prime, const gchar *command, const gchar *pattern)
{
	PrimePrivate *priv;
	GIOChannel *in;
	GIOChannel *out;
	gsize bytes;
	GIOStatus status;
	gchar *read_buf = NULL;
	gchar *output_buf;
	GSList *cand_list = NULL, *slist;
	guint n = 0;
	gchar **cand_array = NULL;
	
	priv = PRIME_GET_PRIVATE (prime);
	
	out = g_io_channel_unix_new (priv->prime_output);
	g_io_channel_set_encoding (out, NULL, NULL);

	output_buf = g_strdup_printf (command, pattern);
	status = prime_send_command (prime, output_buf);
	g_free (output_buf);

	while (status == G_IO_STATUS_NORMAL)
	{
		gchar **strings = NULL;
		gchar *utf8;
		status = g_io_channel_read_line (out,
				        	 &read_buf,
				       		 &bytes,
						 NULL,
		        			 NULL);
		if (read_buf && !strcmp (read_buf, "ok\n"))
		{
			g_free(read_buf);
			continue;
		}

		if (read_buf && !strcmp (read_buf, "\n"))
		{
			g_free (read_buf);
			break;
		}
		utf8 = g_convert (read_buf, bytes, "UTF-8", "EUC-JP",
				  NULL, NULL, NULL);

		strings = g_strsplit (utf8, "\t", 3);
		g_free (read_buf);
		g_free (utf8);

		cand_list = g_slist_prepend (cand_list, g_strdup(strings[1]));
		n++;
		g_strfreev (strings);
	}
	g_io_channel_unref (out);

	if (cand_list)
	{
		cand_array = g_new0 (gchar*, n + 1);
		cand_array[n--] = NULL;
		for (slist = cand_list; slist; slist = slist->next)
			cand_array[n--] = slist->data;

		g_slist_free (cand_list);

		return cand_array;
	}
}

gchar **
prime_preedit_convert_input (Prime *prime, const gchar *pattern)
{
	gchar **ret;
	PrimePrivate *priv;
	GIOChannel *out;
	gchar *read_buf = NULL;
	gchar *command;
	gsize bytes;
	GError *e = NULL;
	GIOStatus status;

	priv = PRIME_GET_PRIVATE (prime);

	ret = g_new0 (gchar *, 3); 
	ret[2] = NULL;

	out = g_io_channel_unix_new (priv->prime_output);
	g_io_channel_set_encoding (out, NULL, NULL);
	
	command = g_strdup_printf (PRIME_PREEDIT_CONVERT_INPUT, pattern);
	status = prime_send_command (prime, command);
	g_free (command);

	while (status == G_IO_STATUS_NORMAL)
	{
		gchar **strings = NULL;
		gchar *utf8;
		status = g_io_channel_read_line (out,
				        	 &read_buf,
				       		 &bytes,
						 NULL,
		        			 NULL);
		if (read_buf && !strcmp (read_buf, "ok\n"))
		{
			g_free(read_buf);
			continue;
		}

		if (read_buf && !strcmp (read_buf, "\n"))
		{
			g_free (read_buf);
			break;
		}
		utf8 = g_convert (read_buf, bytes, "UTF-8", "EUC-JP",
				  NULL, NULL, NULL);

		strings = g_strsplit (utf8, "\t", 2);
		g_free (read_buf);
		g_free (utf8);

		ret[0] = g_strdup (g_strchomp(strings[0]));
		if (strings[1])
			ret[1] = g_strdup (g_strchomp(strings[1]));
		else
			ret[1] = g_strdup ("");
		g_strfreev (strings);
		break;
	}
	g_io_channel_unref (out);

	return ret;
}



gchar ** 
prime_lookup (Prime *prime, const gchar *pattern)
{
	return prime_process_command (prime, PRIME_LOOKUP_ALL, pattern);
}


gchar **
prime_lookup_exact (Prime *prime, const gchar *pattern)
{
	return prime_process_command (prime, PRIME_LOOKUP_EXACT, pattern);
}

gchar **
prime_lookup_hybrid (Prime *prime, const gchar *pattern)
{
	return prime_process_command (prime, PRIME_LOOKUP_HYBRID, pattern);
}

gchar **
prime_get_label (Prime *prime, const gchar *pattern)
{
	return prime_process_command (prime, PRIME_GET_LABEL, pattern);
}

gboolean
prime_learn_word (Prime *prime, const gchar *key,
				const gchar *value,
				const gchar *part,
				const gchar *context,
				const gchar *suffix,
				const gchar *rest)
{
	PrimePrivate *priv;
	gboolean ret = FALSE;
	GIOChannel *out;
	gchar *read_buf = NULL;
	gchar *command, *euc_command;
	gsize bytes;
	GError *e = NULL;
	GIOStatus status;

	priv = PRIME_GET_PRIVATE (prime);

	out = g_io_channel_unix_new (priv->prime_output);
	g_io_channel_set_encoding (out, NULL, NULL);
	command = g_strdup_printf (PRIME_LEARN_WORD, key,
						     value,
						     part,
						     context,
						     suffix,
						     rest);
	euc_command = g_convert (command, strlen (command), "EUC-JP", "UTF-8",
			         NULL, NULL, NULL);
	status = prime_send_command (prime, euc_command);
	g_free (command);
	g_free (euc_command);

	while (status == G_IO_STATUS_NORMAL)
	{
		status = g_io_channel_read_line (out,
				        	 &read_buf,
				       		 &bytes,
						 NULL,
		        			 NULL);
		if (read_buf && !strcmp (read_buf, "ok\n"))
		{
			g_free(read_buf);
			ret = TRUE;
			break;
		}
		g_free (read_buf);
	}
	g_io_channel_unref (out);

	return ret;
}
