/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.base.resources;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.clustercontrol.cloud.aws.base.Activator;
import com.clustercontrol.cloud.commons.util.AccountControlUtil;
import com.clustercontrol.cloud.presenter.CloudModelException;
import com.clustercontrol.ws.aws.AWSOptionEndpoint;
import com.clustercontrol.ws.aws.KeyPair;
import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudManagerFault_Exception;
import com.clustercontrol.ws.cloud.CloudRegion;
import com.clustercontrol.ws.cloud.CloudService;
import com.clustercontrol.ws.cloud.InvalidRole_Exception;
import com.clustercontrol.ws.cloud.InvalidUserPass_Exception;
import com.clustercontrol.ws.cloud.Zone;

public class CloudResourceProviderImpl implements CloudResourceProvider {
	private AWSOptionEndpoint awsEndpoint;
	private CloudEndpoint cloudEndpoint;
	private String serviceId;
	private String roleId;

	public CloudResourceProviderImpl(String accountResourceId) {
		this.awsEndpoint = Activator.getEndpointManager().getEndpoint(AWSOptionEndpoint.class);
		this.cloudEndpoint = Activator.getEndpointManager().getEndpoint(CloudEndpoint.class);
		
		try {
			for(CloudService cloudService: cloudEndpoint.getAllCloudServices()){
				if(cloudService.getCloudServiceId().equals("AWS")){
					this.serviceId = cloudService.getCloudServiceId();
				}
			}
		} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
			new CloudModelException(e);
		}
		
		this.roleId = AccountControlUtil.getRoleId(accountResourceId);
	}
	
	@Override
	public List<String> getAllRegion() {
		try {
			List<String> regions = new ArrayList<String>();
			for (CloudRegion r: cloudEndpoint.getCloudRegionsByService(serviceId)) {
				regions.add(r.getRegion());
			}
			return regions;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<AvailabilityZone> getAvailabilityZones(String region) {
		try {
			List<AvailabilityZone> azs = new ArrayList<AvailabilityZone>();
			for (Zone wsAZ: cloudEndpoint.getZones(roleId, region)) {
				AvailabilityZone az = new AvailabilityZone();
				az.setZoneName(wsAZ.getName());
				azs.add(az);
			}
			return azs;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Image> getImages(String region, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.cloud.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.cloud.Filter>();
			for (Filter filter: filters) {
				com.clustercontrol.ws.cloud.Filter wsFilter = new com.clustercontrol.ws.cloud.Filter();
				wsFilter.setName(filter.getName());
				wsFilter.getValues().addAll(filter.getValues());
				wsFilters.add(wsFilter);
			}

			List<Image> images = new ArrayList<Image>();
			for (com.clustercontrol.ws.cloud.Image wsImage: cloudEndpoint.getImagesWithFilter(roleId, region, wsFilters)) {
				Image image = new Image();
				image.setImageId(wsImage.getImageId());
				image.setName(wsImage.getName());
				images.add(image);
			}
			return images;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<GroupIdentifier> getSecurityGroupsByRegion(String regionName) {
		List<Filter> filters = Collections.emptyList();
		return getSecurityGroups(regionName, filters);
	}
	
	@Override
	public List<GroupIdentifier> getSecurityGroups(String region, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.aws.SecurityGroup> wsSGs = Collections.emptyList();
			if (!filters.isEmpty()) {
				List<com.clustercontrol.ws.aws.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.aws.Filter>();
				for (Filter filter: filters) {
					com.clustercontrol.ws.aws.Filter wsFilter = new com.clustercontrol.ws.aws.Filter();
					wsFilter.setName(filter.getName());
					wsFilter.getValues().addAll(filter.getValues());
					wsFilters.add(wsFilter);
				}
				wsSGs = awsEndpoint.getSecurityGroupsWithFilters(roleId, region, wsFilters);
			}
			else {
				wsSGs = awsEndpoint.getSecurityGroups(roleId, region);
			}

			List<GroupIdentifier> sgs = new ArrayList<GroupIdentifier>();
			for (com.clustercontrol.ws.aws.SecurityGroup wsSG: wsSGs) {
				GroupIdentifier gi = new GroupIdentifier();
				gi.setGroupId(wsSG.getGroupId());
				gi.setGroupName(wsSG.getGroupName());
				gi.setVpcId(wsSG.getVpcId());
				sgs.add(gi);
			}
			return sgs;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getAllInstanceType() {
		try {
			return cloudEndpoint.getInstanceFlavorsByService(serviceId);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getKeyNames(String region) {
		try {
			List<String> keyNames = new ArrayList<String>();
			for (KeyPair k: awsEndpoint.getKeyNames(roleId, region)) {
				keyNames.add(k.getKeyName());
			}
			return keyNames;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Snapshot> getSnapshots(String region, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.cloud.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.cloud.Filter>();
			for (Filter filter: filters) {
				com.clustercontrol.ws.cloud.Filter wsFilter = new com.clustercontrol.ws.cloud.Filter();
				wsFilter.setName(filter.getName());
				wsFilter.getValues().addAll(filter.getValues());
				wsFilters.add(wsFilter);
			}

			List<Snapshot> snapshots = new ArrayList<Snapshot>();
			for (com.clustercontrol.ws.cloud.Snapshot ws: cloudEndpoint.getSnapshotsWithFilter(roleId, region, wsFilters)) {
				Snapshot s = new Snapshot();
				s.setSnapshotId(ws.getSnapshotId());
				s.setDescription(ws.getDescription());
//				s.setRegion(ws.getRegion());
				snapshots.add(s);
			}
			return snapshots;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getAllVolumeType() {
		try {
			List<String> volumeTypes = new ArrayList<String>();
			for (String v: cloudEndpoint.getStorageFlavorsByService(serviceId)) {
				volumeTypes.add(v);
			}
			return volumeTypes;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getAllServiceName() {
		try {
			//TODO
			return null;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Subnet> getSubnets(String regionName, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.cloud.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.cloud.Filter>();
			for (Filter filter: filters) {
				com.clustercontrol.ws.cloud.Filter wsFilter = new com.clustercontrol.ws.cloud.Filter();
				wsFilter.setName(filter.getName());
				wsFilter.getValues().addAll(filter.getValues());
				wsFilters.add(wsFilter);
			}

			List<Subnet> subnets = new ArrayList<Subnet>();
			for (com.clustercontrol.ws.aws.Subnet wsSubnet: awsEndpoint.getSubnets(roleId, regionName)) {
				Subnet subnet = new Subnet();
				subnet.setSubnetId(wsSubnet.getSubnetId());
				subnet.setVpcId(wsSubnet.getVpcId());
				subnet.setAvailabilityZone(wsSubnet.getAvailabilityZone());
				subnet.setCidrBlock(wsSubnet.getCidrBlock());
				subnets.add(subnet);
			}
			return subnets;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Subnet> getSubnetsByRegion(String regionName) {
		try {
			List<Subnet> subnets = new ArrayList<Subnet>();
			for (com.clustercontrol.ws.aws.Subnet wsSubnet: awsEndpoint.getSubnets(roleId, regionName)) {
				Subnet subnet = new Subnet();
				subnet.setSubnetId(wsSubnet.getSubnetId());
				subnet.setVpcId(wsSubnet.getVpcId());
				subnet.setAvailabilityZone(wsSubnet.getAvailabilityZone());
				subnet.setCidrBlock(wsSubnet.getCidrBlock());
				subnets.add(subnet);
			}
			return subnets;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}
}