/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.util;

import java.util.ArrayList;
import java.util.Locale;

import com.clustercontrol.accesscontrol.bean.RoleIdConstant;
import com.clustercontrol.accesscontrol.session.AccessControllerBean;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ErrorCode;
import com.clustercontrol.cloud.IMessagesHolder;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.bean.CloudRegion;
import com.clustercontrol.cloud.commons.CloudConstants;
import com.clustercontrol.cloud.factory.AddedEventNotifier;
import com.clustercontrol.cloud.factory.ICloudServiceOperator;
import com.clustercontrol.cloud.registry.IObjectChangedService;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.fault.FacilityDuplicate;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidObjectPrivilege;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.PrivilegeDuplicate;
import com.clustercontrol.fault.UsedObjectPrivilege;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobParamTypeConstant;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.session.JobControllerBean;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

public class AccountResourceUtil {
	public final static String cloudRootScopeId = "Cloud";
	public final static String cloudRootScopeNameId = "cloudRootScopeName";
	
	public final static String cloudAccountResourceListId = "CloudAccountResourceList"; 
	public final static String cloudAccountResourceListNameId = "cloudAccountResourceListName";

	public final static String templateJobnetId = "Template";
	public final static String templateJobnetNameId = "TemplateJobnetName"; 
	public final static String autoControlJobnetId = "AutoControl";
	public final static String autoControlJobnetNameId = "AutoControlJobnetName";
	
	private final static String group_postfix = "_Group";
	private final static String scope_postfix = "_Scope";
	
	
	public static class AccountResourceIdHelper {
		private String accountResourceId;
		private String accountResourceScopeId;
		private String accountResourceGroupId;

		public AccountResourceIdHelper(String accountResourceId) {
			assert accountResourceId != null;
			this.accountResourceId = accountResourceId;
		}

		public String getAccountResourceId() {
			return accountResourceId;
		}

		public String getGroupId() {
			if (accountResourceGroupId == null) {
				accountResourceGroupId = createAccountResourceGroupId(accountResourceId);
			}
			return accountResourceGroupId;
		}

		public String getScopeId() {
			if (accountResourceScopeId == null) {
				accountResourceScopeId = createAccountResourceScopeId(accountResourceId);
			}
			return accountResourceScopeId;
		}

		public String getJobUnitId() {
			return accountResourceId;
		}

		public String decorateZone(String zoneId) {
			return createAccountResourceZoneId(accountResourceId, zoneId);
		}

		public String decorateRegion(String regionId) {
			return createAccountResourceRegionId(accountResourceId, regionId);
		}
	}
	
	private static void createCloudScopeIfNotExist(RepositoryControllerBean repositoryBean, IMessagesHolder messages) throws HinemosUnknown, InvalidRole, FacilityDuplicate, InvalidSetting, CloudManagerFault {
		try {
			repositoryBean.getScope(cloudRootScopeId);
		}
		catch (FacilityNotFound e1) {
			ScopeInfo scopeInfo = HinemosUtil.createScope(cloudRootScopeId, messages.getString(cloudRootScopeNameId), RoleIdConstant.ADMINISTRATORS);
			try (AddedEventNotifier<ScopeInfo> notifier = new AddedEventNotifier<>(CloudConstants.Node_Cloud_Add, ScopeInfo.class, scopeInfo)) {
				repositoryBean.addScope(null, scopeInfo);
				notifier.setCompleted();
			}
		}
	}
	
	private static void createAccountResourceScopeIfNotExist(RepositoryControllerBean repositoryBean, AccountResourceIdHelper idHelper, String accountResourceName, String roleId) throws HinemosUnknown, InvalidRole, FacilityDuplicate, InvalidSetting, CloudManagerFault {
		try {
			repositoryBean.getScope(idHelper.getScopeId());
		}
		catch (FacilityNotFound e1) {
			ScopeInfo scopeInfo = HinemosUtil.createScope(idHelper.getScopeId(), accountResourceName, roleId);
			try (AddedEventNotifier<ScopeInfo> notifier = new AddedEventNotifier<>(CloudConstants.Node_CloudAccountResourceScope_Add, ScopeInfo.class, scopeInfo)) {
				repositoryBean.addScope(cloudRootScopeId, scopeInfo);
				notifier.setCompleted();
			}
		}
	}

	// スコープへ関連付け。
	public static void attachScope(RepositoryControllerBean repositoryBean, String accountResourceId, String accountResourceName, String region, String regionName, String zone, String facilityId, String roleId, IMessagesHolder messages) throws InvalidSetting, InvalidRole, HinemosUnknown, FacilityDuplicate, CloudManagerFault {
		AccountResourceIdHelper idHelper = new AccountResourceIdHelper(accountResourceId);
		try {
			repositoryBean.assignNodeScope(idHelper.decorateZone(zone), new String[]{facilityId});
		}
		catch(InvalidSetting e) {
			createCloudScopeIfNotExist(repositoryBean, messages);
			
			createAccountResourceScopeIfNotExist(repositoryBean, idHelper, accountResourceName, roleId);

			try {
				repositoryBean.getScope(idHelper.decorateRegion(region));
			}
			catch (FacilityNotFound e1) {
				ScopeInfo scopeInfo = HinemosUtil.createScope(idHelper.decorateRegion(region), regionName, roleId);
				repositoryBean.addScope(idHelper.getScopeId(), scopeInfo);
			}
			
			try {
				// request.getAvailabilityZone() は、null の可能性あり。
				repositoryBean.getScope(idHelper.decorateZone(zone));
			}
			catch (FacilityNotFound e1) {
				ScopeInfo scopeInfo = HinemosUtil.createScope(idHelper.decorateZone(zone), zone, roleId);
				repositoryBean.addScope(idHelper.decorateRegion(region), scopeInfo);
			}
			
			repositoryBean.assignNodeScope(idHelper.decorateZone(zone), new String[]{facilityId});
		}
	}
	
	public static void buildAccountResourceHierarchy(
		RepositoryControllerBean repositry,
		String accountResourceId,
		String accountResourceName,
		String cloudId,
		String description,
		String cloudServiceId,
		String ownerId,
		IMessagesHolder messages
		) throws InvalidRole, CloudManagerFault {
		NodeInfo nodeInfo = createAccountResourceNode(
				accountResourceId,
				accountResourceName,
				cloudId,
				description,
				cloudServiceId,
				ownerId);
		
		try {
			ObjectRegistryService.registry().get(IObjectChangedService.class).firePreAddedEvent(CloudConstants.Node_AccountResource, NodeInfo.class, nodeInfo);
		}
		catch (CloudManagerFault e) {
			throw e;
		}
		catch (Exception e) {
			throw new InternalManagerError(e.getMessage(), e);
		}

		AccountResourceIdHelper idHelper = new AccountResourceIdHelper(accountResourceId);
		try {
			registAccountResourceNode(
					repositry,
					idHelper,
					nodeInfo,
					messages
					);
			
			registAccountResourceHierarchy(
					repositry,
					idHelper,
					accountResourceName,
					cloudServiceId,
					ownerId,
					messages
					);
			
			registAccountResourceJobUnit(
					JobControllerBeanWrapper.bean(),
					idHelper,
					accountResourceName,
					description,
					ownerId,
					messages
					);
		}
		catch(InvalidSetting | HinemosUnknown | FacilityDuplicate | NotifyNotFound | JobMasterNotFound | UserNotFound
			 | JobInvalid | InvalidObjectPrivilege e) {
			throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
		}

		try {
			ObjectRegistryService.registry().get(IObjectChangedService.class).firePostAddedEvent(CloudConstants.Node_AccountResource, NodeInfo.class, nodeInfo);
		}
		catch (CloudManagerFault e) {
			throw e;
		}
		catch (Exception e) {
			throw new InternalManagerError(e.getMessage(), e);
		}
	}
	
	public static void removeAccountResourceHierarchy(RepositoryControllerBean repositry, String accountResourceId) throws CloudManagerFault {
//		AccountResourceIdHelper idHelper = new AccountResourceIdHelper(accountResourceId);
//		try {
//			RepositoryControllerBeanWrapper.bean().deleteNode(accountResourceId);
//		}
//		catch (HinemosUnknown e) {
//			// deleteNode は、FacilityNotFound を HinemosUnknown にラップしてしまう。
//			// しょうがないので、HinemosUnknown のメッセージで判断。
//			if (!e.getMessage().startsWith("FacilityEntity.findByPrimaryKey, facilityId = ")) throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
//		}
//		catch(Exception e) {
//			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_Facility(accountResourceId, accountResourceId, e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_FACILITY.cloudManagerFault(accountResourceId, accountResourceId, CloudManagerFault.getMessage(e));
//		}
//		try {
//			RepositoryControllerBeanWrapper.bean().deleteScope(idHelper.getGroupId());
//		}
//		catch (HinemosUnknown e) {
//			// deleteNode は、FacilityNotFound を HinemosUnknown にラップしてしまう。
//			// しょうがないので、HinemosUnknown のメッセージで判断。
//			if (!e.getMessage().startsWith("FacilityEntity.findByPrimaryKey, facilityId = ")) throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
//		}
//		catch(Exception e) {
//			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_Facility(accountResourceId, idHelper.getGroupId(), e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_FACILITY.cloudManagerFault(accountResourceId, idHelper.getGroupId(), CloudManagerFault.getMessage(e));
//		}
//		try {
//			RepositoryControllerBeanWrapper.bean().deleteScope(idHelper.getScopeId());
//		}
//		catch (HinemosUnknown e) {
//			// deleteNode は、FacilityNotFound を HinemosUnknown にラップしてしまう。
//			// しょうがないので、HinemosUnknown のメッセージで判断。
//			if (!e.getMessage().startsWith("FacilityEntity.findByPrimaryKey, facilityId = ")) throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
//		}
//		catch(Exception e) {
//			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_Facility(accountResourceId, idHelper.getScopeId(), e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_FACILITY.cloudManagerFault(accountResourceId, idHelper.getScopeId(), CloudManagerFault.getMessage(e));
//		}
//		try {
//			JobControllerBeanWrapper.bean().deleteJobunit(idHelper.getJobUnitId());
//		}
//		catch(Exception e) {
//			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_JobUnit(accountResourceId, idHelper.getJobUnitId(), e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_JOBUNIT.cloudManagerFault(accountResourceId, idHelper.getJobUnitId(), CloudManagerFault.getMessage(e));
//		}
		AccountResourceIdHelper idHelper = new AccountResourceIdHelper(accountResourceId);
		try {
			RepositoryControllerBeanWrapper.bean().deleteNode(accountResourceId);
		}
//		catch (HinemosUnknown e) {
//			// deleteNode は、FacilityNotFound を HinemosUnknown にラップしてしまう。
//			// しょうがないので、HinemosUnknown のメッセージで判断。
//			if (!e.getMessage().startsWith("FacilityEntity.findByPrimaryKey, facilityId = ")) throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
//		}
		catch(Exception e) {
			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_Facility(accountResourceId, accountResourceId, e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_FACILITY.cloudManagerFault(accountResourceId, accountResourceId, CloudManagerFault.getMessage(e));
		}
		try {
			RepositoryControllerBeanWrapper.bean().deleteScope(idHelper.getGroupId());
		}
//		catch (HinemosUnknown e) {
//			// deleteNode は、FacilityNotFound を HinemosUnknown にラップしてしまう。
//			// しょうがないので、HinemosUnknown のメッセージで判断。
//			if (!e.getMessage().startsWith("FacilityEntity.findByPrimaryKey, facilityId = ")) throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
//		}
		catch(Exception e) {
			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_Facility(accountResourceId, idHelper.getGroupId(), e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_FACILITY.cloudManagerFault(accountResourceId, idHelper.getGroupId(), CloudManagerFault.getMessage(e));
		}
		try {
			RepositoryControllerBeanWrapper.bean().deleteScope(idHelper.getScopeId());
		}
//		catch (HinemosUnknown e) {
//			// deleteNode は、FacilityNotFound を HinemosUnknown にラップしてしまう。
//			// しょうがないので、HinemosUnknown のメッセージで判断。
//			if (!e.getMessage().startsWith("FacilityEntity.findByPrimaryKey, facilityId = ")) throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
//		}
		catch(Exception e) {
			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_Facility(accountResourceId, idHelper.getScopeId(), e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_FACILITY.cloudManagerFault(accountResourceId, idHelper.getScopeId(), CloudManagerFault.getMessage(e));
		}
		try {
			JobControllerBeanWrapper.bean().deleteJobunit(idHelper.getJobUnitId());
		}
		catch(Exception e) {
			CloudMessageUtil.notify_Not_Delete_AccountResource_Not_To_Delete_JobUnit(accountResourceId, idHelper.getJobUnitId(), e);
//			throw ErrorCode.ACCOUNTRESOURCE_NOT_REMOVE_NOT_TO_REMOVE_JOBUNIT.cloudManagerFault(accountResourceId, idHelper.getJobUnitId(), CloudManagerFault.getMessage(e));
		}
	}
	
	public static NodeInfo createAccountResourceNode(
			String accountResourceId,
			String accountResourceName,
			String cloudTypeId,
			String description,
			String cloudServiceId,
			String roleId) {

		return HinemosUtil.createNodeInfo(
				accountResourceId,
				accountResourceName,
				cloudTypeId,
				cloudTypeId,
				accountResourceName,
				description,
				CloudConstants.NT_AccountResource,
				cloudServiceId,
				accountResourceId,
				null,
				null,
				null,
				null,
				roleId);
	}
	
	// スコープへ関連付け。
	public static void registAccountResourceNode(
			RepositoryControllerBean repositoryBean,
			AccountResourceUtil.AccountResourceIdHelper idUtil,
			NodeInfo nodeInfo,
			IMessagesHolder messages) throws InvalidSetting, InvalidRole, HinemosUnknown, FacilityDuplicate, CloudManagerFault {
		repositoryBean.addNode(nodeInfo);
		
		try {
			repositoryBean.assignNodeScope(idUtil.getGroupId(), new String[]{nodeInfo.getFacilityId()});
		}
		catch(InvalidSetting e) {
			createCloudScopeIfNotExist(repositoryBean, messages);

			try {
				repositoryBean.getScope(cloudAccountResourceListId);
			}
			catch (FacilityNotFound e1) {
				ScopeInfo scopeInfo = HinemosUtil.createScope(cloudAccountResourceListId, messages.getString(cloudAccountResourceListNameId), RoleIdConstant.ADMINISTRATORS);
				repositoryBean.addScope(cloudRootScopeId, scopeInfo);
			}

			try {
				repositoryBean.getScope(idUtil.getGroupId());
			}
			catch (FacilityNotFound e1) {
				ScopeInfo scopeInfo = HinemosUtil.createScope(idUtil.getGroupId(), nodeInfo.getFacilityName(), nodeInfo.getOwnerRoleId());
				repositoryBean.addScope(cloudAccountResourceListId, scopeInfo);
			}

			repositoryBean.assignNodeScope(idUtil.getGroupId(), new String[]{nodeInfo.getFacilityId()});
		}
	}
	
	public static void registAccountResourceHierarchy(RepositoryControllerBean repositoryBean, AccountResourceUtil.AccountResourceIdHelper idUtil, String accountResourceName, String serviceId, String roleId, IMessagesHolder messages) throws FacilityDuplicate, InvalidSetting, InvalidRole, HinemosUnknown, FacilityDuplicate, CloudManagerFault {
		createCloudScopeIfNotExist(repositoryBean, messages);
		
		createAccountResourceScopeIfNotExist(repositoryBean, idUtil, accountResourceName, roleId);

		ICloudServiceOperator operator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
		for (CloudRegion r: operator.findCloudRegionsByService(serviceId)) {
			try {
				repositoryBean.getScope(idUtil.decorateRegion(r.getRegion()));
			}
			catch (FacilityNotFound | InvalidRole e1) {
				ScopeInfo scopeInfo = HinemosUtil.createScope(idUtil.decorateRegion(r.getRegion()), r.getRegionName(), roleId);
				repositoryBean.addScope(idUtil.getScopeId(), scopeInfo);
			}
		}
	}
	
	public static void registAccountResourceJobUnit(JobControllerBean jobBean, AccountResourceUtil.AccountResourceIdHelper idUtil, String accountResourceName, String description, String roleId, IMessagesHolder messages) throws CloudManagerFault, NotifyNotFound, HinemosUnknown, JobMasterNotFound, UserNotFound, InvalidRole, JobInvalid, InvalidObjectPrivilege {
		JobTreeItem root = jobBean.getJobTree(roleId, true, Locale.getDefault());
		JobTreeItem item = HinemosUtil.searchJobTreeItem(root, idUtil.getJobUnitId(), idUtil.getJobUnitId());
		if (item != null)
			throw ErrorCode.ACCOUNTRESOURCE_JOBUNIT_ALREADY_EXIST.cloudManagerFault(idUtil.getJobUnitId());
		
		JobInfo info = HinemosUtil.createJobInfo(idUtil.getJobUnitId(), JobConstant.TYPE_JOBUNIT, accountResourceName, description, roleId);
		info.setJobunitId(idUtil.getJobUnitId());

		ArrayList<JobParameterInfo> params = new ArrayList<JobParameterInfo>();
		for (ArrayList<String> paramData : SystemParameterConstant.ALL) {
			JobParameterInfo param = new JobParameterInfo();
			param.setParamId(paramData.get(0));
			param.setType(JobParamTypeConstant.TYPE_SYSTEM);
			param.setValue("auto");
			param.setDescription(paramData.get(1));
			params.add(param);
		}
		info.setParam(params);
		
		JobTreeItem accountResourceUnit = new JobTreeItem();
		accountResourceUnit.setData(info);

		root.addChildren(accountResourceUnit);
		accountResourceUnit.setParent(root);

		HinemosUtil.relateJobTreeItems(accountResourceUnit, HinemosUtil.createJobnet(templateJobnetId, messages.getString(templateJobnetNameId), "", roleId));
		HinemosUtil.relateJobTreeItems(accountResourceUnit, HinemosUtil.createJobnet(autoControlJobnetId, messages.getString(autoControlJobnetNameId), "", roleId));
		
		jobBean.registerJobunit(accountResourceUnit);
	}
	
	// スコープへ関連付け。
	public static void addAccountResourceRightToRole(AccessControllerBean accessBean, String roleId, String accountResourceId) throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
		AccountResourceIdHelper idHelper = new AccountResourceIdHelper(accountResourceId);
		HinemosUtil.addFullRightToObject(accessBean, roleId, HinemosModuleConstant.PLATFORM_REPOSITORY, idHelper.getScopeId());
		HinemosUtil.addFullRightToObject(accessBean, roleId, HinemosModuleConstant.PLATFORM_REPOSITORY, idHelper.getGroupId());
		HinemosUtil.addFullRightToObject(accessBean, roleId, HinemosModuleConstant.JOB, idHelper.getJobUnitId());
	}

	// スコープへ関連付け。
	public static void removeAccountResourceRightToRole(AccessControllerBean accessBean, String roleId, String accountResourceId) throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
		AccountResourceIdHelper idHelper = new AccountResourceIdHelper(accountResourceId);
		HinemosUtil.removeFullRightFromObject(accessBean, roleId, HinemosModuleConstant.PLATFORM_REPOSITORY, idHelper.getScopeId());
		HinemosUtil.removeFullRightFromObject(accessBean, roleId, HinemosModuleConstant.PLATFORM_REPOSITORY, idHelper.getGroupId());
		HinemosUtil.removeFullRightFromObject(accessBean, roleId, HinemosModuleConstant.JOB, idHelper.getJobUnitId());
	}
	
	public static String createAccountResourceGroupId(String accountResourceId) {
		return accountResourceId + group_postfix;
	}

	public static String createAccountResourceScopeId(String accountResourceId) {
		return accountResourceId + scope_postfix;
	}

	public static String createAccountResourceJobUnitId(String accountResourceId) {
		return accountResourceId;
	}

	public static String createAccountResourceZoneId(String accountResourceId, String zone) {
		return accountResourceId + "_" + zone;
	}

	public static String createAccountResourceRegionId(String accountResourceId, String region) {
		return accountResourceId + "_" + region;
	}
}