/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.cloudn.rest.api;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;

import javax.ws.rs.ClientErrorException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;

import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.cloudn.rest.CloudnResponse;
import com.google.common.base.Preconditions;
import com.google.common.reflect.Reflection;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource.Builder;


public class RestApplicationBuilder {
	public static final String COMMON_PARAM_API_KEY = "apikey";
	public static final String COMMON_PARAM_COMMAND = "command";
	public static final String COMMON_PARAM_SIGNAURE = "signature";
	public static final String COMMON_PARAM_RESPONSE = "response";
	public static final String COMMON_PARAM_EXPIRES = "expires";
	public static final String COMMON_PARAM_SIGNATURE_VERSION = "signatureVersion";
	
	private static class RestApplicationInvoker implements InvocationHandler {
		private static Method getSetEndpointMethod() {
			try {
				return CloudnEndpoint.class.getMethod("setEndpoint", new Class[]{String.class});
			} catch (NoSuchMethodException | SecurityException e) {
				throw new InternalManagerError(e);
			}
		}
		
		public static Method setEndpoint = getSetEndpointMethod();
		
		@Nullable
		private String endpointUrl;
		
		private static Client client = Client.create();
		
		private String accessKey;
		private String secretKey;
		private ErrorUnmarshaller emarshaller;
		
		public RestApplicationInvoker(String accessKey, String secretKey, ErrorUnmarshaller emarshaller) {
			Preconditions.checkNotNull(accessKey);
			Preconditions.checkNotNull(secretKey);
			this.accessKey = accessKey;
			this.secretKey = secretKey;
			this.emarshaller = emarshaller;
		}
		
		@Override
		@NonNullByDefault(false)
		public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
			Preconditions.checkNotNull(proxy);
			Preconditions.checkNotNull(method);
			if (setEndpoint.equals(method)) {
				if (args == null) throw new NullPointerException();
				setEndpointUrl(args[0].toString());
				return null;
			}
			else {
				String endpointUrl = getEndpointUrl();
				if (endpointUrl == null) {
					throw CloudnRestfulApplicationError.Endpoint_Unset.exception();
				}
					
				ParamHolder params = ParamHolder.create()
					.with(COMMON_PARAM_API_KEY, accessKey)
					.with(COMMON_PARAM_COMMAND, method.getName())
					.withs(ParamAnnotations.makeRestParams(method, args));
				String requestURL = params.toRequest(endpointUrl, secretKey);
				
				try {
					Builder builder = client.resource(requestURL).type(MediaType.APPLICATION_JSON_TYPE);
					return builder.get(method.getReturnType());
				}
				catch (ClientErrorException e) {
					if (e.getResponse().hasEntity()) {
						ErrorUnmarshaller.Error erorr = emarshaller.unmarshal(method, e.getResponse());
						throw CloudnRestfulApplicationError.Request_ProcessingError.restRequestException(
								e.getResponse().getStatusInfo().getStatusCode(),
								erorr.getDetailCode(),
//								requestURL.replace("&", "&amp;"),
								requestURL,
								erorr.getReason(),
								e);
					}
					else {
						throw CloudnRestfulApplicationError.Request_ProcessingError.exceptionException(e.getMessage(), e);
					}
				}
//				finally {
//					client.close();
//				}
			}
		}

		private @Nullable String getEndpointUrl() {
			return endpointUrl;
		}

		private void setEndpointUrl(@Nullable String endpointUrl) {
			this.endpointUrl = endpointUrl;
		}
	}
	
	public static <T extends CloudnEndpoint> T buildRestApplicationProxy(Class<T> interfaceClass, String accessKey, String secretKey, ErrorUnmarshaller emarshaller) {
		Preconditions.checkNotNull(interfaceClass);
		Preconditions.checkNotNull(accessKey);
		Preconditions.checkNotNull(secretKey);
		return Reflection.newProxy(interfaceClass, new RestApplicationInvoker(accessKey, secretKey, emarshaller));
	}
	
	public static <T extends CloudnEndpoint> T buildRestApplicationProxy(Class<T> interfaceClass, String accessKey, String secretKey) {
		return buildRestApplicationProxy(interfaceClass, accessKey, secretKey, new ErrorUnmarshaller() {
			@Override
			public Error unmarshal(Method m, Response r) throws RestfulApplicationException {
//				CloudnError e = r.readEntity(CloudnError.class);
//				return new Error(e.cserrorcode, e.errortext);
				try {
					if (!r.hasEntity()) {
						return new Error(null, r.getStatusInfo().getReasonPhrase());
					}
					
					CloudnResponse response = (CloudnResponse)r.readEntity(m.getReturnType());
					return new Error(Integer.toString(response.cserrorcode), response.errortext);
				}
				catch (Exception e) {
					throw new InternalManagerError(e);
				}
			}
		});
	}
}
