/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.util;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

import javax.persistence.EntityExistsException;

import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.persistence.TransactionException;
import com.clustercontrol.fault.FacilityDuplicate;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.repository.bean.AgentStatusInfo;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeItem;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.RepositoryTableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.entity.CollectorPlatformMstData;
import com.clustercontrol.repository.entity.CollectorSubPlatformMstData;
import com.clustercontrol.repository.model.CollectorPlatformMstEntity;
import com.clustercontrol.repository.model.CollectorSubPlatformMstEntity;
import com.clustercontrol.repository.model.FacilityEntity;
import com.clustercontrol.repository.model.NodeEntity;
import com.clustercontrol.repository.session.RepositoryControllerBean;

public class RepositoryControllerBeanWrapper extends RepositoryControllerBean {
	private static ThreadLocal<RepositoryControllerBeanWrapper> instance  = new ThreadLocal<RepositoryControllerBeanWrapper>() {
		protected RepositoryControllerBeanWrapper initialValue()
		{
			return null;
		}
	};

	private RepositoryControllerBeanWrapper() {
		super();
	}
	
	@Override
	public void addScope(final String parentFacilityId, final ScopeInfo property) throws FacilityDuplicate, InvalidSetting, InvalidRole, HinemosUnknown {
		synchronized (RepositoryControllerBeanWrapper.class) {
			try {
				execute(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						RepositoryControllerBean bean = new RepositoryControllerBean();
						try {
							if (bean.getScope(property.getFacilityId()) != null) {
								FacilityDuplicate exception = new FacilityDuplicate();
								exception.setFacilityId(property.getFacilityId());
								throw exception;
							}
						}
						catch (FacilityNotFound e) {
						}

						bean.addScope(parentFacilityId, property);
						return null;
					}
				});
				
				SessionService.current().addRollbackAction(
						new SessionService.RolebackAction() {
							@Override
							public void rollback() throws TransactionException {
								try {
									execute(new Callable<Object>() {
										@Override
										public Object call() throws Exception {
											new RepositoryControllerBean().deleteScope(property.getFacilityId());
											return null;
										}
									});
								}
								catch (Exception e) {
									throw new TransactionException(e);
								}
							}
						});
			}
			catch (InterruptedException e) { 
				throw new InternalManagerError(e);
			}
			catch (ExecutionException e) {
				// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
				// ExecutionException 分の例外が途切れる
				if (e.getCause() instanceof InvalidRole) {
					throw new InvalidRole(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof FacilityDuplicate) {
					throw new FacilityDuplicate(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof InvalidSetting) {
					throw new InvalidSetting(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof HinemosUnknown) {
					throw new HinemosUnknown(e.getCause().getMessage(), e);
				}
				else {
					throw new InternalManagerError(e);
				}
			} catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
	}

	@Override
	public void assignNodeScope(final String parentFacilityId, final String[] facilityIds) throws InvalidSetting, InvalidRole, HinemosUnknown {
		synchronized (RepositoryControllerBeanWrapper.class) {
			try {
				execute(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						new RepositoryControllerBean().assignNodeScope(parentFacilityId, facilityIds);
						return null;
					}
				});
				
				SessionService.current().addRollbackAction(
						new SessionService.RolebackAction() {
							@Override
							public void rollback() throws TransactionException {
								try {
									execute(new Callable<Object>() {
										@Override
										public Object call() throws Exception {
											new RepositoryControllerBean().releaseNodeScope(parentFacilityId, facilityIds);
											return null;
										}
									});
								}
								catch (Exception e) {
									throw new TransactionException(e);
								}
							}
						});
	
			}
			catch (InterruptedException e) { 
				throw new InternalManagerError(e);
			}
			catch (ExecutionException e) {
				// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
				// ExecutionException 分の例外が途切れる
				if (e.getCause() instanceof InvalidRole) {
					throw new InvalidRole(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof InvalidSetting) {
					throw new InvalidSetting(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof HinemosUnknown) {
					throw new HinemosUnknown(e.getCause().getMessage(), e);
				}
				else {
					throw new InternalManagerError(e);
				}
			} catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
	}
	
	@Override
	public void addNode(final NodeInfo nodeInfo) throws FacilityDuplicate, InvalidSetting, HinemosUnknown {
		synchronized (RepositoryControllerBeanWrapper.class) {
			try {
				execute(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						RepositoryControllerBean bean = new RepositoryControllerBean();
						try {
							if (bean.getNode(nodeInfo.getFacilityId()) != null) {
								FacilityDuplicate exception = new FacilityDuplicate();
								exception.setFacilityId(nodeInfo.getFacilityId());
								throw exception;
							}
						}
						catch (FacilityNotFound e) {
						}
						bean.addNode(nodeInfo);
						return null;
					}
				});
	
				SessionService.current().addRollbackAction(
						new SessionService.RolebackAction() {
							@Override
							public void rollback() throws TransactionException {
								try {
									execute(new Callable<Object>() {
										@Override
										public Object call() throws Exception {
											new RepositoryControllerBean().deleteNode(nodeInfo.getFacilityId());
											return null;
										}
									});
								}
								catch (Exception e) {
									throw new TransactionException(e);
								}
							}
						});
			}
			catch (InterruptedException e) { 
				throw new InternalManagerError(e);
			}
			catch (ExecutionException e) {
				// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
				// ExecutionException 分の例外が途切れる
				if (e.getCause() instanceof FacilityDuplicate) {
					throw new FacilityDuplicate(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof InvalidSetting) {
					throw new InvalidSetting(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof HinemosUnknown) {
					throw new HinemosUnknown(e.getCause().getMessage(), e);
				}
				else {
					throw new InternalManagerError(e);
				}
			} catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
	}
	
	@Override
	public void modifyNode(final NodeInfo info) throws InvalidSetting, InvalidRole, HinemosUnknown {
		synchronized (RepositoryControllerBeanWrapper.class) {
			try {
				execute(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						new RepositoryControllerBean().modifyNode(info);
						return null;
					}
				});
//				SessionService.current().addRollbackAction(
//						new SessionService.RolebackAction() {
//							@Override
//							public void rollback() throws TransactionException {
//								try {
//									execute(new Callable<Object>() {
//										@Override
//										public Object call() throws Exception {
//											new RepositoryControllerBean().modifyNode(original);
//											return null;
//										}
//									});
//								}
//								catch (Exception e) {
//									throw new TransactionException(e);
//								}
//							}
//						});
			}
			catch (InterruptedException e) { 
				throw new InternalManagerError(e);
			}
			catch (ExecutionException e) {
				// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
				// ExecutionException 分の例外が途切れる
				if (e.getCause() instanceof InvalidRole) {
					throw new InvalidRole(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof InvalidSetting) {
					throw new InvalidSetting(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof HinemosUnknown) {
					throw new HinemosUnknown(e.getCause().getMessage(), e);
				}
				else {
					throw new InternalManagerError(e);
				}
			} catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
	}
	
	@Override
	public void deleteNode(final String facilityId) throws UsedFacility, InvalidRole, HinemosUnknown {
		synchronized (RepositoryControllerBeanWrapper.class) {
			try {
				execute(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						new RepositoryControllerBean().deleteNode(facilityId);
						return null;
					}
				});
//				SessionService.current().addRollbackAction(
//						new SessionService.RolebackAction() {
//							@Override
//							public void rollback() throws TransactionException {
//								try {
//									execute(new Callable<Object>() {
//										@Override
//										public Object call() throws Exception {
//											throw new UnsupportedOperationException();
//										}
//									});
//								}
//								catch (Exception e) {
//									throw new TransactionException(e);
//								}
//							}
//						});
			}
			catch (InterruptedException e) { 
				throw new InternalManagerError(e);
			}
			catch (ExecutionException e) {
				// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
				// ExecutionException 分の例外が途切れる
				if (e.getCause() instanceof InvalidRole) {
					throw new InvalidRole(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof UsedFacility) {
					throw new UsedFacility(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof HinemosUnknown) {
					throw new HinemosUnknown(e.getCause().getMessage(), e);
				}
				else {
					throw new InternalManagerError(e);
				}
			} catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
	}
	
	@Override
	public NodeInfo getNode(final String facilityId) throws FacilityNotFound, HinemosUnknown {
		try {
			return execute(new Callable<NodeInfo>() {
				@Override
				public NodeInfo call() throws Exception {
					return new RepositoryControllerBean().getNode(facilityId);
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof FacilityNotFound) {
				throw new FacilityNotFound(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public void deleteScope(final String facilityId) throws UsedFacility, InvalidRole, HinemosUnknown {
		synchronized (RepositoryControllerBeanWrapper.class) {
			try {
				execute(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						new RepositoryControllerBean().deleteScope(facilityId);
						return null;
					}
				});
//				SessionService.current().addRollbackAction(
//						new SessionService.RolebackAction() {
//							@Override
//							public void rollback() throws TransactionException {
//								try {
//									execute(new Callable<Object>() {
//										@Override
//										public Object call() throws Exception {
//											throw new UnsupportedOperationException();
//										}
//									});
//								}
//								catch (Exception e) {
//									throw new TransactionException(e);
//								}
//							}
//						});
			}
			catch (InterruptedException e) { 
				throw new InternalManagerError(e);
			}
			catch (ExecutionException e) {
				// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
				// ExecutionException 分の例外が途切れる
				if (e.getCause() instanceof UsedFacility) {
					throw new UsedFacility(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof InvalidRole) {
					throw new InvalidRole(e.getCause().getMessage(), e);
				}
				else if (e.getCause() instanceof HinemosUnknown) {
					throw new HinemosUnknown(e.getCause().getMessage(), e);
				}
				else {
					throw new InternalManagerError(e);
				}
			} catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
	}

	private  <T> T execute(Callable<T> task) throws Exception {
		return HinemosUtil.submit(task).get();
	}

	public static RepositoryControllerBeanWrapper bean() {
		RepositoryControllerBeanWrapper bean = instance.get();
		if (bean == null) {
			bean = new RepositoryControllerBeanWrapper();
			instance.set(bean);
		}
		return bean;
	}

	@Override
	public FacilityEntity getFacilityEntityByPK(final String facilityId) throws FacilityNotFound, InvalidRole, HinemosUnknown {
		try {
			return execute(new Callable<FacilityEntity>() {
				@Override
				public FacilityEntity call() throws Exception {
					return new RepositoryControllerBean().getFacilityEntityByPK(facilityId);
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof FacilityNotFound) {
				throw new FacilityNotFound(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public NodeEntity getNodeEntityByPK(String facilityId)
			throws FacilityNotFound, InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void addCollectorPratformMst(CollectorPlatformMstData data)
			throws EntityExistsException, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void addCollectorSubPlatformMst(CollectorSubPlatformMstData data)
			throws EntityExistsException, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void deleteCollectorPratformMst(String platformId)
			throws FacilityNotFound, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void deleteCollectorSubPratformMst(String subPlatformId)
			throws FacilityNotFound, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<CollectorPlatformMstEntity> getCollectorPlatformMstList()
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<CollectorSubPlatformMstEntity> getCollectorSubPlatformMstList()
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public FacilityTreeItem getFacilityTree(final String ownerRoleId, final Locale locale) throws HinemosUnknown {
		try {
			return execute(new Callable<FacilityTreeItem>() {
				@Override
				public FacilityTreeItem call() throws Exception {
					return new RepositoryControllerBean().getFacilityTree(ownerRoleId, locale);
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public FacilityTreeItem getExecTargetFacilityTree(String facilityId,
			String ownerRoleId, Locale locale) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public FacilityTreeItem getNodeFacilityTree(Locale locale,
			String ownerRoleId) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<NodeInfo> getNodeDetailList() throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<NodeInfo> getFilterNodeList(NodeInfo property)
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getExecTargetFacilityIdList(String facilityId,
			String ownerRoleId) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getNodeFacilityIdList() throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getNodeFacilityIdList(boolean sort)
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getFacilityPath(final String facilityId, final String parentFacilityId) throws HinemosUnknown {
		try {
			return execute(new Callable<String>() {
				@Override
				public String call() throws Exception {
					return new RepositoryControllerBean().getFacilityPath(facilityId, parentFacilityId);
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public NodeInfo getNodePropertyBySNMP(String ipAddress, int port,
			String community, String version) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<String> getFacilityIdByIpAddress(InetAddress ipaddr)
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getFacilityIdList(String hostName, String ipAddress)
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void addNode(NodeInfo info, boolean topicSendFlg)
			throws FacilityDuplicate, InvalidSetting, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<FacilityInfo> getFacilityList(final String parentFacilityId) throws HinemosUnknown {
		try {
			return execute(new Callable<ArrayList<FacilityInfo>>() {
				@Override
				public ArrayList<FacilityInfo> call() throws Exception {
					RepositoryControllerBean bean = new RepositoryControllerBean();
					List<NodeInfo> nodes = bean.getNodeList(parentFacilityId, Integer.MAX_VALUE);
					List<FacilityInfo> scopes = bean.getFacilityList(parentFacilityId);
					
					ArrayList<FacilityInfo> facilities = new ArrayList<>();
					facilities.addAll(nodes);
					facilities.addAll(scopes);
					
					return facilities;
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public ScopeInfo getScope(final String facilityId) throws FacilityNotFound, HinemosUnknown, InvalidRole {
		try {
			return execute(new Callable<ScopeInfo>() {
				@Override
				public ScopeInfo call() throws Exception {
					new RepositoryControllerBean().getScope(facilityId);
					return null;
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof FacilityNotFound) {
				throw new FacilityNotFound(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public void addScope(String parentFacilityId, ScopeInfo property,
			int displaySortOrder) throws FacilityDuplicate, InvalidSetting,
			InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void addScope(String parentFacilityId, ScopeInfo info,
			int displaySortOrder, boolean topicSendFlg)
			throws FacilityDuplicate, InvalidSetting, InvalidRole,
			HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void modifyScope(ScopeInfo info) throws InvalidSetting, InvalidRole,
			HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<NodeInfo> getNodeList(String parentFacilityId, int level)
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getNodeScopeList(String facilityId)
			throws InvalidRole, HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getFacilityIdList(String parentFacilityId,
			int level) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getFacilityIdList(String parentFacilityId,
			int level, boolean scopeFlag) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getFacilityIdList(String parentFacilityId,
			int level, boolean sort, boolean scopeFlag) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId,
			String ownerRoleId, int level) throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId,
			String ownerRoleId, int level, boolean sort, Boolean valid)
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void assignNodeScope(String parentFacilityId, String[] facilityIds,
			boolean topicSendFlg) throws InvalidSetting, InvalidRole,
			HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void releaseNodeScope(final String parentFacilityId, final String[] facilityIds) throws InvalidSetting, InvalidRole, HinemosUnknown {
		try {
			execute(new Callable<ScopeInfo>() {
				@Override
				public ScopeInfo call() throws Exception {
					new RepositoryControllerBean().releaseNodeScope(parentFacilityId, facilityIds);
					return null;
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof InvalidSetting) {
				throw new InvalidSetting(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds,
			boolean topicSendFlg) throws InvalidSetting, InvalidRole,
			HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean isNode(final String facilityId) throws FacilityNotFound,
			InvalidRole, HinemosUnknown {
		try {
			return execute(new Callable<Boolean>() {
				@Override
				public Boolean call() throws Exception {
					return new RepositoryControllerBean().isNode(facilityId);
				}
			});
		}
		catch (InterruptedException e) { 
			throw new InternalManagerError(e);
		}
		catch (ExecutionException e) {
			// めんどくさいが、以下のように ExecutionException を含めて例外をラップしないと、
			// ExecutionException 分の例外が途切れる
			if (e.getCause() instanceof FacilityNotFound) {
				throw new FacilityNotFound(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof InvalidRole) {
				throw new InvalidRole(e.getCause().getMessage(), e);
			}
			else if (e.getCause() instanceof HinemosUnknown) {
				throw new HinemosUnknown(e.getCause().getMessage(), e);
			}
			else {
				throw new InternalManagerError(e);
			}
		} catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public String getSeparator() {
		throw new UnsupportedOperationException();
	}

	@Override
	public void sendRepositoryUpdateTopic() {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<RepositoryTableInfo> getPlatformList()
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<RepositoryTableInfo> getHostList() throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<NodeInfo> getOpenFlowSwitchList() throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<RepositoryTableInfo> getCollectorSubPlatformTableInfoList()
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<String> getVmProtocolMstList() throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public Date getLastUpdate() {
		throw new UnsupportedOperationException();
	}

	@Override
	public ArrayList<AgentStatusInfo> getAgentStatusList()
			throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void restartAgent(ArrayList<String> facilityIdList, int agentCommand) {
		throw new UnsupportedOperationException();
	}

	@Override
	public HashMap<String, String> getAgentLibMap() throws HinemosUnknown {
		throw new UnsupportedOperationException();
	}

	@Override
	public void checkIsUseFacility(String facilityId) throws HinemosUnknown,
			UsedFacility {
		throw new UnsupportedOperationException();
	}

	@Override
	public Set<String> getNodeListByNodename(String hostname) {
		throw new UnsupportedOperationException();
	}

	@Override
	public Set<String> getNodeListByIpAddress(InetAddress ipAddress) {
		throw new UnsupportedOperationException();
	}

	@Override
	public Set<String> getNodeListByHostname(String hostname) {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean containsFaciliyId(String scopeFacilityId,
			String nodeFacilityId, String ownerRoleId) {
		throw new UnsupportedOperationException();
	}
}