#ifndef _htlib_h
#define _htlib_h

#ifdef __cplusplus
exern "C" {
#endif

/*
 * Copyright (C) 2009 NISHIMURA Hideki
 * @file: htlib.h
 * @author: NISHIMURA Hideki
 */

/**
 * Basic type definitions
 */	
typedef unsigned long long HTLIB_ULONGLONG;
typedef unsigned long HTLIB_ULONG;
typedef unsigned short HTLIB_USHORT;
typedef unsigned char HTLIB_UCHAR;
typedef int HTLIB_BOOL;

/**
 * Error code declarations
 */
typedef enum {
	HTLIB_E_NO_ERROR = 0,

	HTLIB_E_INVALID_ARG = 1,
	HTLIB_E_INVALID_STATE = 2,
	HTLIB_E_OUT_OF_MEMORY = 3,
	HTLIB_E_TIMEOUT = 4,
	HTLIB_E_INVALID_SYNTAX = 5,
	HTLIB_E_CANCELED = 6,

	HTLIB_E_1XX_RECEIVED = 20,
	HTLIB_E_DISCONNECTED = 21,
	
	HTLIB_E_FATAL = 98,
	HTLIB_E_NOT_IMPLEMENTED = 99,
	
	HTLIB_E_SYSTEM = 100,

} HTLIB_ERROR;

/**
 * Logging level declarations
 * @see HTLIB_SetLogHandler
 */
typedef enum {
	HTLIB_LV_FATAL = 0,
	HTLIB_LV_ERROR = 1,
	HTLIB_LV_WARN = 2,
	HTLIB_LV_INFO = 3,
	HTLIB_LV_DEBUG = 4,

} HTLIB_LOGLEVEL;

/**
 * Structured data definitions
 */

/**
 * Structure for managing socket, transaction, etc.
 */
typedef struct _HTLIB HTLIB, *HTLIB_HANDLE;
struct _HTLIB {
	/* public members */
	int soc;
	int system_errno;
	const struct addrinfo* hints;
	const char* agent_or_server_name;
	void* user;

	/* private members */


	/* for sender */
	char* _send_buffer;
	int _send_buffer_len;
	long long _send_content_length;
	long long _send_content_transfered;

	/* for receiver */
	char* _rec_buffer;
	int _rec_buffer_len;
	int _read_index;
	long long _rec_content_length;
	long long _rec_content_transfered;
	
	char* _header;
	int _body_index;

	/* common */
	volatile HTLIB_BOOL _is_canceled;
	pthread_mutex_t _lock;
	HTLIB_ULONG _flags;
	
};


/**
 * Substructure for describing headers.
 */
typedef struct _HTLIB_Header HTLIB_Header;
struct _HTLIB_Header {
	const char* name;
	const char* value;
	int param_num;
	const HTLIB_Header* param;
};

/**
 * Initializer
 */
HTLIB_BOOL
HTLIB_Init(HTLIB_HANDLE o,
		   char* send_buffer, HTLIB_USHORT send_len,
		   char* rec_buffer, HTLIB_USHORT rec_len,
		   HTLIB_ERROR* err);

/**
 * Uninitializer
 * (closes socket inside if opened yet.)
 */
void
HTLIB_Uninit(HTLIB_HANDLE o);

/**
 * (Optional)
 * (re-)create a socket and (re-)connect to the address explicitly.
 */
HTLIB_BOOL
HTLIB_Open(HTLIB_HANDLE o,
		   const struct sockaddr* addr,
		   socklen_t addr_len,
		   HTLIB_ERROR* err);

/**
 * (Optional for the server)
 * attach a socket which already has a connection
 * before HTLIB_ReceiveRequest().
 */
void
HTLIB_Attach(HTLIB_HANDLE o, int socket);

/**
 * (Optional)
 * close the socket explicitly.
 */
void
HTLIB_Close(HTLIB_HANDLE o);		

/**
 * Send a request to the server identified by 'url'.
 * 'headers' shall not include "Host:" or "Content-Length:"
 * You can include "Connection: close" unless it should keep alive.
 *
 * If the previous request has "Connection: close",
 * it does reset the socket before connect.
 */
HTLIB_BOOL
HTLIB_SendRequest(HTLIB_HANDLE o,
				  int timeout_millis,
				  const char* method,
				  const char* url,
				  const HTLIB_Header* headers,
				  HTLIB_USHORT header_len,
				  const char* body,
				  HTLIB_ULONGLONG body_len,
				  HTLIB_ERROR* err);

HTLIB_BOOL
HTLIB_SendResponse(HTLIB_HANDLE o,
				   int timeout_millis,
				   int status, const char* msg,
				   const HTLIB_Header* headers,
				   HTLIB_USHORT header_len,
				   const char* body,
				   HTLIB_ULONGLONG body_len,
				   HTLIB_ERROR* err);
				   
/**
 *
 */
HTLIB_BOOL
HTLIB_SendBody(HTLIB_HANDLE o,
			   int timeout_millis,
			   const char* body,
			   HTLIB_USHORT body_len,
			   HTLIB_ERROR* err);

/**
 * Optional callback function
 * you can use to eliminate headers you don't want.
 * @see HTLIB_ReceiveResonse.
 */
typedef HTLIB_BOOL (*HTLIB_NEEDS_FUNC)(HTLIB_HANDLE o,
									   const char* header_name);

/**
 * Receive the response for the request sent by HTLIB_SendRequest.
 * This automatically detects "Connection:" inside for the next request.
 * @return URL
 */
const char*
HTLIB_ReceiveRequest(HTLIB_HANDLE o,
					 int timeout_millis,
					 const char** method,
					 HTLIB_Header* header_buffer, HTLIB_USHORT* blen,
					 HTLIB_NEEDS_FUNC needs,
					 HTLIB_ERROR* err);

/**
 * Receive the response for the request sent by HTLIB_SendRequest.
 * This automatically detects "Connection:" inside for the next request.
 */
int
HTLIB_ReceiveResponse(HTLIB_HANDLE o,
					  int timeout_millis,
					  HTLIB_Header* header_buffer, HTLIB_USHORT* blen,
					  HTLIB_NEEDS_FUNC needs,
					  HTLIB_ERROR* err);

/**
 * Find a specified header or parameter in the headers or parameters.
 * 'headers_or_params' can be 'header_buffer' or 'header_buffer[x].param'
 * of HTLIB_ReceiveResponse.
 */
const HTLIB_Header*
HTLIB_Find(const HTLIB_Header* headers_or_params,
		   HTLIB_USHORT len,
		   const char* name);

/**
 * Receive a piece of the body.
 * This should be called repeatedly like 'recv' or 'fread'.
 * And, automatically manages content's length by
 * detecting "Content-Length:" and "Connection: close".
 */
int
HTLIB_ReceiveBody(HTLIB_HANDLE o,
				  int timeout_millis,
				  char* buffer,
				  HTLIB_USHORT buffer_len,
				  HTLIB_ERROR* err);

/**
 * Cancels the undergoing request by this handle immediately.
 * (Of course pthread_join'ing is your job.)
 */
void
HTLIB_Cancel(HTLIB_HANDLE o, HTLIB_BOOL trueIfCancel);


/**
 * Returns a text message corresponding to 'err'.
 */
const char*
HTLIB_GetErrorMessage(HTLIB_ERROR err);


/**
 * Logging functions
 */

/**
 * Prototype for log-handlers:
 * If you want to log messages by your own handler,
 * register yours by SetLogHandler.
 * @see HTLIB_SetLogHandler
 */
typedef void (*HTLIB_LOG_HANDLER)(const char* file,
								  int line,
								  HTLIB_LOGLEVEL level,
								  const char* msg);

/**
 * Registers your own log-handler.
 * default: fprintf(stderr, "%s\n", msg);
 */
void
HTLIB_SetLogHandler(HTLIB_LOG_HANDLER handler);

/*
 * Sets log-level.
 * For example,
 * if you set 1 to levelAndHigher,
 * only 0 and 1-level logs are handed to the handler.
 */
void
HTLIB_SetLogLevel(HTLIB_LOGLEVEL levelAndHigher);


#ifdef __cplusplus
}
#endif

#endif

