#include "kernel.h"
#include "ioport.h"
#include "timer.h"
#include "intmgr.h"

#include "screen.h"

Timer SystemTimer;

class TimerInterruptHandler : public InterruptHandler
{
private: 
    Timer& timer;

public:
    TimerInterruptHandler(Timer& tm) : timer(tm) {}

    void operator()(uint32 intNum) {
        ++timer;
        IntMgr::getInstance()->setEoi(IntMgr::TIMER_IRQ);
    }
};

Timer::Timer(void) : Count(0), JobQueue(JobCmp) {}

void Timer::init(void) {
    IoPortLib::out8(IoPortLib::PIT_CTRL, 0x34);
    IoPortLib::out8(IoPortLib::PIT_CNT0, 0x9C);
    IoPortLib::out8(IoPortLib::PIT_CNT0, 0x2E);

    IntMgr* int_mgr = IntMgr::getInstance();
    int_mgr->connect(IntMgr::TIMER_IRQ, 
                     *(new TimerInterruptHandler(SystemTimer)));
    int_mgr->clearMask(IntMgr::TIMER_MASK);
}

void Timer::add(TimerJob& job) {
    IntMgr* int_mgr = IntMgr::getInstance();
    int_mgr->disable();
    job.Start   = Count;
    job.Timeout = Count + job.Duration;
    JobQueue.enqueue(job);
    int_mgr->enable();
}

void Timer::abort(TimerJob& job) {
    IntMgr* int_mgr = IntMgr::getInstance();
    int_mgr->disable();
    JobQueue.remove(job);
    int_mgr->enable();
}

Timer& Timer::operator++(void) {
    if (++Count > MAX_COUNT) Count = 0;

    TimerJob* job = (TimerJob*)JobQueue.peek(0);
    if (job != 0) {
        checkExpire(*job);
    }
 
    return *this; 
}

void Timer::checkExpire(TimerJob& job) {
    if (job.Start > Count) {
        if (Timer::MAX_COUNT + Count > job.Timeout) {
            JobQueue.remove(job);
            job.onExpire();
        }
    } else {
        if (job.Timeout < Count) {
            JobQueue.remove(job);
            job.onExpire();
        }
    }
}
