package org.arefgard.container.flow;

import java.io.InputStream;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.arefgard.container.Constants;
import org.arefgard.container.ContainerException;
import org.arefgard.container.aop.ClassGenerator;
import org.arefgard.container.aop.MethodInvocation;
import org.arefgard.container.util.MessageUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class UseCaseFlowExecutorImpl implements UseCaseFlowExecutor {
	
	private Log log = LogFactory.getLog(UseCaseFlowExecutorImpl.class);
	
	private static String SEQUENCE_NODE_RECEIVE = "receive";
	private static String SEQUENCE_NODE_INVOKE = "invoke";
	private static String SEQUENCE_NODE_REPLY = "reply";
	private static String SEQUENCE_NODE_THROW = "throw";
	private static String SEQUENCE_NODE_DICISION = "decision";
	private static String SEQUENCE_NODE_WHILE = "while";
	private static String SEQUENCE_NODE_CALL = "call";
	private static String SEQUENCE_NODE_WAIT = "wait";

	private static String XPATH_NODE_INTERFACE = "/usecase/interfaces/interface";
	private static String XPATH_NODE_SEQUENCE = "/usecase/sequence/child::*";
	private static String XPATH_NODE_PARAMETER = "parameter";
	private static String XPATH_NODE_NAVIGATION = "navigation";
	
	private static String XPATH_ATTR_NAME = "string(@name)";
	private static String XPATH_ATTR_TYPE = "string(@type)";
	private static String XPATH_ATTR_METHOD = "string(@method)";
	private static String XPATH_ATTR_CONDITION = "string(@condition)";
	private static String XPATH_ATTR_NEXTTO = "string(@nextTo)";
	private static String XPATH_ATTR_TIME = "string(@time)";
	
	private XPathExpression XPATH_INTERFACE;
	private XPathExpression XPATH_SEQUENCE;
	private XPathExpression XPATH_PARAMETER;
	private XPathExpression XPATH_NAVIGATION;
	
	private XPathExpression XPATH_NAME;
	private XPathExpression XPATH_TYPE;
	private XPathExpression XPATH_METHOD;
	private XPathExpression XPATH_CONDITION;
	private XPathExpression XPATH_NEXTTO;
	private XPathExpression XPATH_TIME;
	
	private Map<String, Interface> input = null;
	private Map<String, Sequence> sequence = null;
	private Map<String, Object> output = null;
	
	private String flowPath = null;
	private String statringPoint = null;
	
	public UseCaseFlowExecutorImpl(String flowPath) throws ContainerException {
		this.flowPath = flowPath;
		input = new java.util.HashMap<String, Interface>();
		sequence = new java.util.HashMap<String, Sequence>();
		output = new java.util.HashMap<String, Object>();
		
		XPathFactory factory = XPathFactory.newInstance();
		XPath xpath = factory.newXPath();
		try {
			XPATH_INTERFACE = xpath.compile(XPATH_NODE_INTERFACE);
			XPATH_SEQUENCE = xpath.compile(XPATH_NODE_SEQUENCE);
			XPATH_PARAMETER = xpath.compile(XPATH_NODE_PARAMETER);
			XPATH_NAVIGATION = xpath.compile(XPATH_NODE_NAVIGATION);
			
			XPATH_NAME = xpath.compile(XPATH_ATTR_NAME);
			XPATH_TYPE = xpath.compile(XPATH_ATTR_TYPE);
			XPATH_METHOD = xpath.compile(XPATH_ATTR_METHOD);
			XPATH_CONDITION = xpath.compile(XPATH_ATTR_CONDITION);
			XPATH_NEXTTO = xpath.compile(XPATH_ATTR_NEXTTO);
			XPATH_TIME = xpath.compile(XPATH_ATTR_TIME);
		}catch(Exception e) {
			e.printStackTrace();
		}
		parseUseCase();
	}
	
	public void execute() throws Exception {
		Receive start = (Receive)sequence.get(this.statringPoint);
		
		for(int i = 0; i < start.getParameterLength(); i++) {
			start.getParameter(i);
		}
		String navi = start.getNextTo();
		Object result = null;
		while(true) {
			Sequence sequenceDef = sequence.get(navi);
			if(sequenceDef instanceof Invoke) {
				Invoke invokeDef = (Invoke)sequenceDef;
				Object obj = input.get(invokeDef.getType()).getValue();
				String method = invokeDef.getMethod();
				List<Object> paramList = new java.util.ArrayList<Object>();
				for(int i = 0; i < invokeDef.getParameterLength(); i++) {
					paramList.add(input.get(invokeDef.getParameter(i)).getValue());
				}
				
				MethodInvocation invokeMethod = new MethodInvocation(obj);
				result = invokeMethod.execute(method, paramList);
				
				navi = invokeDef.getNextTo();
			}else if(sequenceDef instanceof Reply) {
				Reply replyDef = (Reply)sequenceDef;
				for(int i = 0; i < replyDef.getParameterLength(); i++) {
					String param = replyDef.getParameter(i);
					output.put(param, input.get(param).getValue());
				}
				break;
			}else if(sequenceDef instanceof Throw) {
				Throw throwDef = (Throw)sequenceDef;
				throwDef.throwException();
			}else if(sequenceDef instanceof Decision) {
				Decision decisionDef = (Decision)sequenceDef;
				Navigation[] navis = decisionDef.getAllNavigations();
				for(int i = 0, n = navis.length; i < n; i++) {
					String left = navis[i].getLeft();
					if(left.equals("@return")) {
						if(navis[i].evaluate(result)) {
							navi = navis[i].getNextTo();
							break;
						}
					}
				}
			}else if(sequenceDef instanceof While) {
				// FIXME
			}else if(sequenceDef instanceof Call) {
				Call callDef = (Call)sequenceDef;
				String path = callDef.getPath();
				UseCaseFlowExecutor exec = new UseCaseFlowExecutorImpl(path);
				exec.execute();
			}else if(sequenceDef instanceof Wait) {
				Wait waitDef = (Wait)sequenceDef;
				navi = waitDef.execute();
			}
		}
	}

	public Object getOutput(String key) {
		return output.get(key);
	}

	public void setInput(String key, Object obj) {
		Interface interfaceDef = input.get(key);
		interfaceDef.setValue(obj);
		
		input.put(key, interfaceDef);
	}
	
	private void parseUseCase() throws ContainerException {
		InputStream is = ClassLoader.getSystemResourceAsStream(flowPath);
		if(is == null) {
			is = this.getClass().getClassLoader().getResourceAsStream(flowPath);
		}
		if(is == null) {
			log.error(MessageUtil.getMessage(Constants.USECASE_ERROR_NOTFOUND, flowPath));
			throw new UseCaseFlowParseException(MessageUtil.getMessage(Constants.USECASE_ERROR_NOTFOUND, flowPath));
		}
		
		DocumentBuilderFactory factory;
		Document doc = null;
		try {
			factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			doc = builder.parse(is);
		}catch(Exception e) {
			e.printStackTrace();
			log.error(MessageUtil.getMessage(Constants.USECASE_ERROR_PARSE), e);
			throw new UseCaseFlowParseException(MessageUtil.getMessage(Constants.USECASE_ERROR_PARSE), e);
		}
		
		try {
			NodeList interfaces = (NodeList)XPATH_INTERFACE.evaluate(doc, XPathConstants.NODESET);
			
			Node interfaceNode;
			for(int i = 0; i < interfaces.getLength();i++) {
				interfaceNode = interfaces.item(i);
				String name = this.getName(interfaceNode);
				String type = this.getType(interfaceNode);
				Interface interfaceDef = new Interface(name, type);
				input.put(name, interfaceDef);
			}
		}catch(Exception e) {
			e.printStackTrace();
			log.error(MessageUtil.getMessage(Constants.USECASE_ERROR_PARSE), e);
			throw new UseCaseFlowParseException(MessageUtil.getMessage(Constants.USECASE_ERROR_PARSE), e);
		}
		
		try {
			NodeList sequences = (NodeList)XPATH_SEQUENCE.evaluate(doc, XPathConstants.NODESET);
			
			Node sequenceNode;
			for(int i = 0; i < sequences.getLength();i++) {
				sequenceNode = sequences.item(i);
				String nodeName = sequenceNode.getNodeName();
				String name;
				String type;
				String method;
				if(nodeName.endsWith(SEQUENCE_NODE_RECEIVE)) {
					// receive
					name = getName(sequenceNode);
					Receive receiveDef = new Receive(name);
					NodeList parameters = (NodeList)XPATH_PARAMETER.evaluate(sequenceNode, XPathConstants.NODESET);
					for(int j = 0, n = parameters.getLength(); j < n;j++) {
						Node parameterNode = parameters.item(j);
						String param = getName(parameterNode);
						receiveDef.addParameter(param);
					}
					receiveDef.setNextTo(getNextTo(sequenceNode));
					sequence.put(name, receiveDef);
					this.statringPoint = name;
				}else if(nodeName.endsWith(SEQUENCE_NODE_INVOKE)) {
					// invoke
					name = getName(sequenceNode);
					type = getType(sequenceNode);
					method = getMethod(sequenceNode);
					Invoke invokeDef = new Invoke(name, type, method);
					NodeList parameters = (NodeList)XPATH_PARAMETER.evaluate(sequenceNode, XPathConstants.NODESET);
					for(int j = 0, n = parameters.getLength(); j < n;j++) {
						Node parameterNode = parameters.item(j);
						String param = getName(parameterNode);
						invokeDef.addParameter(param);
					}
					invokeDef.setNextTo(getNextTo(sequenceNode));
					sequence.put(name, invokeDef);
				}else if(nodeName.endsWith(SEQUENCE_NODE_REPLY)) {
					// reply
					name = getName(sequenceNode);
					Reply replyDef = new Reply(name);
					NodeList parameters = (NodeList)XPATH_PARAMETER.evaluate(sequenceNode, XPathConstants.NODESET);
					for(int j = 0, n = parameters.getLength(); j < n;j++) {
						Node parameterNode = parameters.item(j);
						String param = getName(parameterNode);
						replyDef.addParameter(param);
					}
					sequence.put(name, replyDef);
				}else if(nodeName.endsWith(SEQUENCE_NODE_THROW)) {
					// throw
					name = getName(sequenceNode);
					type = getType(sequenceNode);
					Throw throwDef = new Throw(name, type);
					sequence.put(name, throwDef);
				}else if(nodeName.endsWith(SEQUENCE_NODE_DICISION)) {
					// decision
					name = getName(sequenceNode);
					Decision decisionDef = new Decision(name);
					NodeList navigations = (NodeList)XPATH_NAVIGATION.evaluate(sequenceNode, XPathConstants.NODESET);
					for(int j = 0, n = navigations.getLength(); j < n; j++) {
						Node navi = navigations.item(j);
						String condition = getCondition(navi);
						String nextTo = getNextTo(navi);
						decisionDef.addNavigation(new Navigation(condition, nextTo));
					}
					sequence.put(name, decisionDef);
				}else if(nodeName.endsWith(SEQUENCE_NODE_WHILE)) {
					// while
					name = getName(sequenceNode);
					While whileDef = new While(name);
					whileDef.setNextTo(getNextTo(sequenceNode));
					Node navi = (Node)XPATH_NAVIGATION.evaluate(sequenceNode, XPathConstants.NODE);
					String condition = getCondition(navi);
					String nextTo	= getNextTo(navi);
					whileDef.setNavigation(new Navigation(condition, nextTo));
					sequence.put(name, whileDef);
				}else if(nodeName.endsWith(SEQUENCE_NODE_CALL)) {
					// call
					name = getName(sequenceNode);
					Call callDef = new Call(name);
					callDef.setType(getType(sequenceNode));
					callDef.setNextTo(getNextTo(sequenceNode));
					sequence.put(name, callDef);
				}else if(nodeName.endsWith(SEQUENCE_NODE_WAIT)) {
					// wait
					name = getName(sequenceNode);
					Wait waitDef = new Wait(name);
					waitDef.setTime(getTime(sequenceNode));
					waitDef.setNextTo(getNextTo(sequenceNode));
					sequence.put(name, waitDef);
				}
			}
		}catch(Exception e) {
			e.printStackTrace();
			log.error(MessageUtil.getMessage(Constants.USECASE_ERROR_PARSE), e);
			throw new UseCaseFlowParseException(MessageUtil.getMessage(Constants.USECASE_ERROR_PARSE), e);
		}
		
	}
	
	private String getName(Object obj) throws XPathExpressionException {
		return (String)XPATH_NAME.evaluate(obj, XPathConstants.STRING);
	}
	
	private String getType(Object obj) throws XPathExpressionException {
		return (String)XPATH_TYPE.evaluate(obj, XPathConstants.STRING);
	}
	
	private String getMethod(Object obj) throws XPathExpressionException {
		return (String)XPATH_METHOD.evaluate(obj, XPathConstants.STRING);
	}
	
	private String getCondition(Object obj) throws XPathExpressionException {
		return (String)XPATH_CONDITION.evaluate(obj, XPathConstants.STRING);
	}
	
	private String getNextTo(Object obj) throws XPathExpressionException {
		return (String)XPATH_NEXTTO.evaluate(obj, XPathConstants.STRING);
	}
	
	private long getTime(Object obj) throws XPathExpressionException {
		return Long.parseLong((String)XPATH_TIME.evaluate(obj, XPathConstants.STRING));
	}
}
