package morpheme;

import double_array.BinaryFileWriter;
import double_array.BinaryFileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import double_array.Searcher;
import java.util.Collections;

public class CharCategory {
    private final Entry[] categorys;
    private final Entry2[] codeCategoryMap;

    public CharCategory (String outputDir) throws IOException {
	categorys = readCategorys(outputDir);
	codeCategoryMap = readCodeCategoryMap(outputDir);
    }

    public Entry category(char code) {
	return categorys[codeCategoryMap[code].id];
    }

    public boolean isCompatible(char code1, char code2) {
	return (codeCategoryMap[code1].mask & codeCategoryMap[code2].mask) != 0;
    }

    private Entry[] readCategorys(String outputDir) throws IOException {
	BinaryFileReader br = new BinaryFileReader(outputDir+"/char.category");
	final int size = (int)br.length()/(4*4);
	Entry[] ary = new Entry[size];

	try{
	    for(int i=0; i < size; i++)
		ary[i] = new Entry(br.readInt(),br.readInt(),br.readInt()==1,br.readInt()==1);
	} finally {
	    br.close();
	}
	return ary;
    }

    private Entry2[] readCodeCategoryMap(String outputDir) throws IOException {
	BinaryFileReader br = new BinaryFileReader(outputDir+"/code->category");
	final int size = (int)br.length()/(4*2);
	Entry2[] ary = new Entry2[size];

	try{
	    for(int i=0; i < size; i++)
		ary[i] = new Entry2(br.readInt(),br.readInt());
	} finally {
	    br.close();
	}
	return ary;
    }

    public static class Entry  implements Comparable { // XXX:
	public Entry(int i, int l, boolean iv, boolean g) {
	    id = i;
	    length = l;
	    invoke = iv;
	    group = g;
	}
 
	public int     id;
	public int     length;
	public boolean invoke;
	public boolean group;

	public int compareTo(Object o) {	
	    Entry e = (Entry)o;
	    return id - e.id;
	}
    }

    public static void build(String inputDir, String outputDir, String encoding) throws IOException {
	// parse character category definition
	final Map<String,Entry> ccmap = parseCharCategoryDef(inputDir,outputDir,encoding);
	
	List<Entry> vals = new ArrayList<Entry>();
	for(Entry e : ccmap.values())
	    vals.add(e);

	// generate ...
	genCharCategoryMap(vals, outputDir);

	// generate code(ucs2) category mapping
	genCodeCategoryMap(ccmap,inputDir,outputDir,encoding);
    }

    // 先頭にタブ文字を付与することで、他の文字よりも上位のIDにくることが保証されている
    // ただし、実際には、タブも時ではなく'\02'などを使用する方が望ましい
    private static void genCharCategoryMap(List<Entry> categorys, String outputDir) throws IOException {
	BinaryFileWriter bw = new BinaryFileWriter(outputDir+"/char.category");
	try {
	    Collections.sort(categorys);
	    for(Entry e : categorys) {
		bw.writeInt(e.id);
		bw.writeInt(e.length);
		bw.writeInt(e.invoke ? 1 : 0);
		bw.writeInt(e.group ? 1 : 0);
	    }
	} finally {
	    bw.close();
	}
    }

    private static Map<String,Entry> parseCharCategoryDef(String inputDir,String outputDir,String encoding) 
	throws IOException {
	ReadLine rl = new ReadLine(inputDir+"/char.def",encoding);
	Searcher srch = new Searcher(outputDir+"/word->id");
	Map<String,Entry> map = new HashMap<String,Entry>();

	String line;
	while((line=rl.read()) != null) {
	    if(line.isEmpty() || line.startsWith("#") || line.startsWith("0"))
		continue;
	    final String[] ss = line.split("\\s+"); // XXX:
	    final String  name = ss[0];
	    final boolean invoke = ss[1].equals("1");
	    final boolean group  = ss[2].equals("1");
	    final int length = Integer.valueOf(ss[3]);
	    final int id = srch.search("\t"+name).id();    // XXX:
	    map.put(name, new Entry(id,length,invoke,group));
	}

	rl.close();
	return map;
    }

    // コードの文字カテゴリIDと、互換ID判定用のビットマスクを持っている
    // これにより、文字カテゴリの上限が32個までになってしまう
    // クラス名は完全に仮のもの
    public static class Entry2 {
	public Entry2(int i) {
	    id = i;
	    mask = 1<<i;
	}
	public Entry2(int i, int m) {
	    id=i;
	    mask=m;
	}
	public void add(int i) {
	    mask |= 1<<i;
	}

	final int id; // 上限は31
	int mask;
    }

    private static void genCodeCategoryMap(Map<String,Entry> ccmap, 
					   String inputDir, String outputDir,
					   String encoding) throws IOException {
	ReadLine rl = new ReadLine(inputDir+"/char.def",encoding);	
	String line;
	Entry2[] map = new Entry2[0x10000];
	final Entry2 def = new Entry2(ccmap.get("DEFAULT").id);
	
	for(int i=0; i < 0x10000; i++)
	    map[i] = def;

	while((line=rl.read()) != null) {
	    if(line.startsWith("0")==false)
		continue;

	    // XXX: 適当
	    final String[] ss = line.split("\\s+");
	    int beg;
	    int end;
	    if(ss[0].indexOf("..") != -1) {
		final String[] ss2 = ss[0].split("\\.\\.");
		beg = Integer.parseInt(ss2[0].substring(2),16);
		end = Integer.parseInt(ss2[1].substring(2),16);
	    } else {
		beg = end = Integer.parseInt(ss[0].substring(2),16);
	    }
	    Entry2 e2 = new Entry2(ccmap.get(ss[1]).id);
	    for(int i=2; i < ss.length; i++) {
		if(ss[i].startsWith("#"))
		    break;
		e2.add(ccmap.get(ss[i]).id);
	    }
	    
	    for(int i=beg; i <= end; i++)
		map[i] = e2;
	}

	BinaryFileWriter bw = new BinaryFileWriter(outputDir+"/code->category");
	for(Entry2 e : map) {
	    bw.writeInt(e.id);
	    bw.writeInt(e.mask);
	}
	bw.close();
	
	rl.close();
    }
}