/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/*
 * gtkimmoduleime
 * Copyright (C) 2003 Takuro Ashie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * $Id: gtkimcontextime.c,v 1.8 2003/05/07 12:42:50 makeinu Exp $
 */

#include "gtkimcontextime.h"

#include <windows.h>
#include <imm.h>

#include <gdk/gdkwin32.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkwidget.h>

/* avoid warning */
#ifdef STRICT
#   undef STRICT
#   include <pango/pangowin32.h>
#   ifndef STRICT
#      define STRICT 1
#   endif
#else /* STRICT */
#   include <pango/pangowin32.h>
#endif /* STRICT */

struct _GtkIMContextIMEPrivate
{
    /* save IME context when the client window is focused out */
    GdkRectangle cursor_location;
    DWORD        conversion_mode;
    DWORD        sentence_mode;

    LPVOID       comp_str;
    DWORD        comp_str_len;
    LPVOID       read_str;
    DWORD        read_str_len;
};


/* #define IMPLEMENT_GET_PREEDIT_STRING 1 */
#define BUFSIZE 4096

#define GET_UTF8_PREEDIT_STRING(himc, idx, str)              \
{                                                            \
    gchar buf[BUFSIZE];                                      \
    glong len;                                               \
    gsize bytes_read, bytes_written;                         \
    GError *error = NULL;                                    \
                                                             \
    len = ImmGetCompositionString (himc, idx,                \
                                   buf, G_N_ELEMENTS (buf)); \
    str = g_locale_to_utf8 (buf, len,                        \
                            &bytes_read, &bytes_written,     \
                            &error);                         \
    if (error) {                                             \
        g_warning ("%s", error->message);                    \
        g_error_free (error);                                \
    }                                                        \
}

#define FREE_PREEDIT_BUFFER(ctx)   \
{                                  \
    g_free((ctx)->priv->comp_str); \
    g_free((ctx)->priv->read_str); \
    (ctx)->priv->comp_str = NULL;  \
    (ctx)->priv->read_str = NULL;  \
    (ctx)->priv->comp_str_len = 0; \
    (ctx)->priv->read_str_len = 0; \
}


static void     gtk_im_context_ime_class_init          (GtkIMContextIMEClass *class);
static void     gtk_im_context_ime_init                (GtkIMContextIME *context_ime);
static void     gtk_im_context_ime_finalize            (GObject *obj);

/* virtual functions */
static void     gtk_im_context_ime_set_client_window   (GtkIMContext   *context,
                                                        GdkWindow      *client_window);
static gboolean gtk_im_context_ime_filter_keypress     (GtkIMContext   *context,
                                                        GdkEventKey    *event);
static void     gtk_im_context_ime_reset               (GtkIMContext   *context);
#ifdef IMPLEMENT_GET_PREEDIT_STRING
static void     gtk_im_context_ime_get_preedit_string  (GtkIMContext   *context,
                                                        gchar         **str,
                                                        PangoAttrList **attrs,
                                                        gint           *cursor_pos);
#endif /* IMPLEMENT_GET_PREEDIT_STRING */
static void     gtk_im_context_ime_focus_in            (GtkIMContext   *context);
static void     gtk_im_context_ime_focus_out           (GtkIMContext   *context);
static void     gtk_im_context_ime_set_cursor_location (GtkIMContext   *context,
                                                        GdkRectangle   *area);
static void     gtk_im_context_ime_set_use_preedit     (GtkIMContext   *context,
                                                        gboolean        use_preedit);

/* private */
static void     gtk_im_context_ime_set_preedit_font    (GtkIMContext   *context,
                                                        PangoFont      *font);


GType gtk_type_im_context_ime = 0;
static GObjectClass *parent_class;
GtkIMContextIME *current_context = NULL;


void
gtk_im_context_ime_register_type (GTypeModule *type_module)
{
    static const GTypeInfo im_context_ime_info = {
        sizeof (GtkIMContextIMEClass),
        (GBaseInitFunc) NULL,
        (GBaseFinalizeFunc) NULL,
        (GClassInitFunc) gtk_im_context_ime_class_init,
        NULL,           /* class_finalize */    
        NULL,           /* class_data */
        sizeof (GtkIMContextIME),
        0,
        (GInstanceInitFunc) gtk_im_context_ime_init,
    };

    gtk_type_im_context_ime = 
        g_type_module_register_type (type_module,
                                     GTK_TYPE_IM_CONTEXT,
                                     "GtkIMContextIME",
                                     &im_context_ime_info, 0);
}

static void
gtk_im_context_ime_class_init (GtkIMContextIMEClass *class)
{
    GtkIMContextClass *im_context_class = GTK_IM_CONTEXT_CLASS (class);
    GObjectClass *gobject_class = G_OBJECT_CLASS (class);

    parent_class = g_type_class_peek_parent (class);

    im_context_class->set_client_window   = gtk_im_context_ime_set_client_window;
    im_context_class->filter_keypress     = gtk_im_context_ime_filter_keypress;
    im_context_class->reset               = gtk_im_context_ime_reset;
#ifdef IMPLEMENT_GET_PREEDIT_STRING
    im_context_class->get_preedit_string  = gtk_im_context_ime_get_preedit_string;
#endif /* IMPLEMENT_GET_PREEDIT_STRING */
    im_context_class->focus_in            = gtk_im_context_ime_focus_in;
    im_context_class->focus_out           = gtk_im_context_ime_focus_out;
    im_context_class->set_cursor_location = gtk_im_context_ime_set_cursor_location;
    im_context_class->set_use_preedit     = gtk_im_context_ime_set_use_preedit;

    gobject_class->finalize               = gtk_im_context_ime_finalize;
}


static void
gtk_im_context_ime_init (GtkIMContextIME *context_ime)
{
    GdkRectangle *area;

    context_ime->client_window = NULL;
    context_ime->preediting    = FALSE;
    context_ime->opened        = FALSE;

    context_ime->priv = g_new0 (GtkIMContextIMEPrivate, 1);

    context_ime->priv->comp_str     = NULL;
    context_ime->priv->comp_str_len = 0;
    context_ime->priv->read_str     = NULL;
    context_ime->priv->read_str_len = 0;
    area = &context_ime->priv->cursor_location;
    area->x = area->y = area->width = area->height = 0;
}


static void
gtk_im_context_ime_finalize (GObject *obj)
{
    GtkIMContext *context = GTK_IM_CONTEXT (obj);
    GtkIMContextIME *context_ime = GTK_IM_CONTEXT_IME (obj);

    gtk_im_context_ime_set_client_window (context, NULL);

    FREE_PREEDIT_BUFFER (context_ime);

    g_free(context_ime->priv);
    context_ime->priv = NULL;
}


GtkIMContext *
gtk_im_context_ime_new (void)
{
    return g_object_new (GTK_TYPE_IM_CONTEXT_IME, NULL);
}


GdkFilterReturn 
ime_message_filter (GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
    GtkIMContext *context;
    GtkIMContextIME *context_ime;
    HWND hwnd;
    HIMC himc;
    MSG *msg = (MSG *) xevent;
    GdkFilterReturn retval = GDK_FILTER_CONTINUE;

    g_return_val_if_fail (GTK_IS_IM_CONTEXT_IME (data), retval);

    context     = GTK_IM_CONTEXT (data);
    context_ime = GTK_IM_CONTEXT_IME (data);
    if (current_context != context_ime)
       return retval;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return retval;
    if (!ImmGetOpenStatus (himc)) goto ERROR_OUT;

    switch (msg->message) {
    case WM_IME_COMPOSITION:
        if (msg->lParam & GCS_COMPSTR) {
            g_signal_emit_by_name (context, "preedit_changed");
        }

        if (msg->lParam & GCS_RESULTSTR) {
            gchar *utf8str;

            GET_UTF8_PREEDIT_STRING (himc, GCS_RESULTSTR, utf8str);
            if (utf8str) {
#ifndef IMPLEMENT_GET_PREEDIT_STRING
                g_signal_emit_by_name (G_OBJECT (context_ime),
                                       "commit", utf8str);
#endif /* IMPLEMENT_GET_PREEDIT_STRING */
                g_free (utf8str);
            }
        }
        break;

    case WM_IME_STARTCOMPOSITION:
        context_ime->preediting = TRUE;
        gtk_im_context_ime_set_cursor_location (context, NULL);
        g_signal_emit_by_name (context, "preedit_start");
        break;

    case WM_IME_ENDCOMPOSITION:
        context_ime->preediting = FALSE;
        g_signal_emit_by_name (context, "preedit_end");
        break;

    case WM_IME_NOTIFY:
        switch (msg->wParam) {
        case IMN_SETOPENSTATUS:
            context_ime->opened = ImmGetOpenStatus (himc);
            gtk_im_context_ime_set_preedit_font (context, NULL);
            break;

        default:
            break;
        }
    default:
        break;
    }

ERROR_OUT:
    ImmReleaseContext (hwnd, himc);
    return retval;
}


static void
gtk_im_context_ime_set_client_window (GtkIMContext *context,
                                      GdkWindow *client_window)
{
    GtkIMContextIME *context_ime;
    GdkWindow *toplevel;
    HWND hwnd = NULL;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));
    context_ime = GTK_IM_CONTEXT_IME (context);

    if (client_window) {
        HIMC himc;

        hwnd = GDK_WINDOW_HWND (client_window);
        toplevel = gdk_window_get_toplevel (client_window);
        gdk_window_add_filter (toplevel, ime_message_filter, context_ime);

        /* get default ime context */
        himc = ImmGetContext (hwnd);
        context_ime->opened = ImmGetOpenStatus (himc);
        ImmGetConversionStatus (himc,
                                &context_ime->priv->conversion_mode,
                                &context_ime->priv->sentence_mode);
        ImmReleaseContext (hwnd, himc);

    } else if (context_ime->client_window) {
        hwnd = GDK_WINDOW_HWND (context_ime->client_window);
        toplevel = gdk_window_get_toplevel (context_ime->client_window);
        gdk_window_remove_filter (toplevel, ime_message_filter, context_ime);
    }

    context_ime->client_window = client_window;
}


static gboolean
gtk_im_context_ime_filter_keypress (GtkIMContext *context,
                                    GdkEventKey  *event)
{
    GtkIMContextIME *context_ime;
    HWND hwnd;
    HIMC himc;
    gboolean retval = FALSE;

    g_return_val_if_fail (GTK_IS_IM_CONTEXT_IME (context), FALSE);
    g_return_val_if_fail (event, FALSE);

    context_ime = GTK_IM_CONTEXT_IME (context);
    if (current_context != context_ime) return FALSE;
    if (!GDK_IS_WINDOW (context_ime->client_window)) return FALSE;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);

    if (!himc || !ImmGetOpenStatus (himc)) {
        if (event->string && *event->string) {
            g_signal_emit_by_name (G_OBJECT (context_ime),
                                   "commit", event->string);
            retval = TRUE;
        }
    }

    ImmReleaseContext (hwnd, himc);

    return retval;
}


static void
gtk_im_context_ime_reset (GtkIMContext *context)
{
    GtkIMContextIME *context_ime = GTK_IM_CONTEXT_IME (context);
    HWND hwnd;
    HIMC himc;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    if (context_ime->preediting && ImmGetOpenStatus (himc))
        ImmNotifyIME (himc, NI_COMPOSITIONSTR, CPS_CANCEL, 0);

    ImmReleaseContext (hwnd, himc);
}


#ifdef IMPLEMENT_GET_PREEDIT_STRING
static void
gtk_im_context_ime_get_preedit_string (GtkIMContext *context,
                                       gchar **str,
                                       PangoAttrList **attrs,
                                       gint *cursor_pos)
{
    GtkIMContextIME *context_ime;
    gchar *utf8str;
    HWND hwnd;
    HIMC himc;
    gint utf8len;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));

    context_ime = GTK_IM_CONTEXT_IME (context);

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);

    if (himc && context_ime->preediting) {
        GET_UTF8_PREEDIT_STRING(himc, GCS_COMPSTR, utf8str);
    } else {
        utf8str = g_strdup("");
    }

    utf8len = g_utf8_strlen (utf8str, -1);

    if (str)
        *str = utf8str;
    else
        g_free (utf8str);

#   warning FIXME
#   if 1 /* FIXME!! */
    if (attrs) {
        PangoAttribute *attr;
        *attrs = pango_attr_list_new ();
        attr = pango_attr_underline_new (PANGO_UNDERLINE_SINGLE);
        attr->start_index = 0;
        attr->end_index = utf8len > 0 ? utf8len - 1 : 0;
        pango_attr_list_change (*attrs, attr);
    }

    if (cursor_pos)
        *cursor_pos = 0;
#   endif /* END FIXME!! */
}
#endif /* IMPLEMENT_GET_PREEDIT_STRING */


static void
gtk_im_context_ime_focus_in (GtkIMContext *context)
{
    GtkIMContextIME *context_ime = GTK_IM_CONTEXT_IME (context);
    HWND hwnd;
    HIMC himc;

    /* swtich current context */
    current_context = context_ime;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    /* restore preedit context */
    ImmSetConversionStatus (himc,
                            context_ime->priv->conversion_mode,
                            context_ime->priv->sentence_mode);

    if (context_ime->opened) {
        if (!ImmGetOpenStatus (himc))
            ImmSetOpenStatus (himc, TRUE);
        if (context_ime->preediting) {
            ImmSetCompositionString (himc,
                                     SCS_SETSTR,
                                     context_ime->priv->comp_str,
                                     context_ime->priv->comp_str_len,
                                     NULL,
                                     0);
            FREE_PREEDIT_BUFFER (context_ime);
        }
    }

    /* clean */
    ImmReleaseContext (hwnd, himc);
}


static void
gtk_im_context_ime_focus_out (GtkIMContext *context)
{
    GtkIMContextIME *context_ime = GTK_IM_CONTEXT_IME (context);
    HWND hwnd;
    HIMC himc;

    current_context = context_ime;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    /* save preedit context */
    ImmGetConversionStatus (himc,
                            &context_ime->priv->conversion_mode,
                            &context_ime->priv->sentence_mode);

    if (himc && ImmGetOpenStatus (himc)) {
        DWORD len;
        gchar buf[BUFSIZE];
        gboolean preediting = context_ime->preediting;

        if (preediting) {
            FREE_PREEDIT_BUFFER (context_ime);

            len = ImmGetCompositionString (himc, GCS_COMPSTR,
                                           buf, G_N_ELEMENTS (buf));
            context_ime->priv->comp_str = g_memdup (buf, len);
            context_ime->priv->comp_str_len = len;

            len = ImmGetCompositionString (himc, GCS_COMPREADSTR,
                                           buf, G_N_ELEMENTS (buf));
            context_ime->priv->read_str = g_memdup (buf, len);
            context_ime->priv->read_str_len = len;
        }

        ImmSetOpenStatus (himc, FALSE);

        context_ime->opened = TRUE;
        context_ime->preediting = preediting;

    } else {
        context_ime->opened = FALSE;
        context_ime->preediting = FALSE;
    }

    /* swtich current context */
    if (current_context == GTK_IM_CONTEXT_IME (context))
        current_context = NULL;

    /* clean */
    ImmReleaseContext (hwnd, himc);
}


/*
 * x and y must be initialized to 0.
 */
static void
get_window_position (GdkWindow *win, gint *x, gint *y)
{
    GdkWindow *parent, *toplevel;
    gint wx, wy;

    g_return_if_fail (GDK_IS_WINDOW (win));
    g_return_if_fail (x && y);

    gdk_window_get_position (win, &wx, &wy);
    *x += wx;
    *y += wy;
    parent = gdk_window_get_parent (win);
    toplevel = gdk_window_get_toplevel (win);

    if (parent && parent != toplevel)
        get_window_position (parent, x, y);
}


static void
gtk_im_context_ime_set_cursor_location (GtkIMContext *context,
                                        GdkRectangle *area)
{
    GtkIMContextIME *context_ime;
    HWND hwnd;
    HIMC himc;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));

    context_ime = GTK_IM_CONTEXT_IME (context);
    if (!area)
        area = &context_ime->priv->cursor_location;
    else
        context_ime->priv->cursor_location = *area;

    if (!context_ime->client_window) return;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    if (ImmGetOpenStatus (himc)) {
        POINT pt;
        COMPOSITIONFORM cf;
        gint wx = 0, wy = 0;

        get_window_position (context_ime->client_window, &wx, &wy);
        pt.x = wx + area->x;
        pt.y = wy + area->y;
        cf.dwStyle = CFS_POINT;
        cf.ptCurrentPos = pt;
        ImmSetCompositionWindow (himc, &cf);
    }

    ImmReleaseContext (hwnd, himc);
}


static void
gtk_im_context_ime_set_use_preedit (GtkIMContext *context,
                                    gboolean      use_preedit)
{
    GtkIMContextIME *context_ime;
    HWND hwnd;
    HIMC himc;
    BOOL success;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));
    context_ime = GTK_IM_CONTEXT_IME (context);

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    success = ImmSetOpenStatus (himc, use_preedit);
    if (!success) {
        const gchar *status_str = use_preedit ? "open" : "close";
        g_warning ("Faild to %s IME!", status_str);
    }

    ImmReleaseContext (hwnd, himc);
}


static void
gtk_im_context_ime_set_preedit_font (GtkIMContext *context,
                                     PangoFont *font)
{
    GtkIMContextIME *context_ime;
    GtkWidget *widget;
    HWND hwnd;
    HIMC himc;
    PangoContext *pango_context;
    LOGFONT *logfont;

    g_return_if_fail (GTK_IS_IM_CONTEXT_IME (context));

    context_ime = GTK_IM_CONTEXT_IME(context);
    if (!context_ime->client_window) return;

    gdk_window_get_user_data (context_ime->client_window,
                              (gpointer *) &widget);
    if (!GTK_IS_WIDGET (widget)) return;

    hwnd = GDK_WINDOW_HWND (context_ime->client_window);
    himc = ImmGetContext (hwnd);
    if (!himc) return;

    /* set font */
    pango_context = gtk_widget_get_pango_context (widget);
    if (!pango_context) goto ERROR_OUT;

    if (!font)
        font = pango_context_load_font (pango_context,
                                        widget->style->font_desc);
    if (!font) goto ERROR_OUT;

    logfont = pango_win32_font_logfont (font);
    if (logfont)
        ImmSetCompositionFont (himc, logfont);

ERROR_OUT:
    /* clean */
    ImmReleaseContext (hwnd, himc);
}
