<?php
/*
 * This file is part of INQMAN
 *
 * Copyright(c) 2008 BULLHEAD,INC. ALL RIGHTS RESERVED.
 *
 * http://www.bullhead.co.jp/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation;
 * either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */


/**
 * ユーザ管理のロジッククラス
 * 
 * 
 *
 */
class Inqman_UserModel extends Inqman_AbstractDao
{
    protected $_name = 'm_user';
    
    /**
     * ユーザ情報を取得する
     * 
     * @param integer $user_id
     * @return array ユーザ情報を連想配列で返す
     */
    public function getOneUser($user_id)
    {
        $db = $this->getAdapter();
        
        $select = $db->select()
                     ->from(array('u'=>'m_user'))
                     ->where('u.id=?', $user_id)
                     ->where('u.deleted<>1')
        ;
        $this->_logger->info('Inqman_UserModel->getOneUser : ' . $select->__toString());
        
        return $db->fetchRow($select);
    }
    
    /**
     * メールアドレスからユーザ情報を取得する
     * 
     */
    public function getOneUserByMailaddress($mailaddress)
    {
        $db = $this->getAdapter();
        $select = $db->select()
                     ->from(array('u'=>'m_user'))
                     ->where('u.mailaddress=?', $mailaddress)
                     ->where('u.deleted<>1')
        ;
        $this->_logger->info('Inqman_UserModel->getOneUserByMailaddress : ' . $select->__toString());
        return $db->fetchRow($select);
    }
    
    /**
     * 検索する
     * 
     * @access public
     * @param mixed $order
     * @param integer $page ページ番号
     * @param integer $rp 1ページあたりの表示件数
     * @param mixed $criteria 検索条件
     */
    public function find($order=null, $page=null, $rp=null, $criteria=null)
    {
        $db = $this->getAdapter();
        
        $select = $db->select()
                     ->from(array('u' => 'm_user'))
                     ->where('u.deleted<>1')
        ;
        
        if ($order !== null) {
            foreach ((array) $order as $str) {
                $select->order($str);
            }
            $select->order($this->_primary[1] . " asc");
        }
        
        if ($page !== null && $rp !== null) {
            $select->limitPage($page, $rp);
        }
        
        $select = $this->_buildFindQuery($select, $criteria);
        
        $this->_logger->info('Inqman_UserModel->find : ' . $select->__toString());
        
        return $db->fetchAll($select);
    }
    
    /**
     * データ件数を得る
     * 
     * @access public
     * @param mixed $criteria 検索条件
     */
    public function getTotalSize($criteria=null)
    {
        $db = $this->getAdapter();
        
        $select = $db->select()
                     ->from(array('u' => 'm_user'), 'count(*)')
                     ->where('u.deleted<>1')
        ;
        
        $select = $this->_buildFindQuery($select, $criteria);
        
        $this->_logger->info('Inqman_UserModel->getTotalSize : ' . $select->__toString());
        
        return $db->fetchOne($select);
    }
    
    /**
     * 検索条件からSQLを生成する
     * 
     * @access protected
     * @param select Zend_DB_Select
     * @param criteria mixed
     * @return string SQLを返す
     */
    protected function _buildFindQuery($select, $criteria)
    {
        //検索条件の設定
        foreach ((array) $criteria as $field => $cond) {
            
            switch ($field) {
                case 'sbj':
                    break;
                default:
                    break;
            }
        }
        return $select;
    }
    
    /**
     * ユーザIDの重複をチェックする
     * 
     * @access public
     * @param string $user_id
     * @param boolean $is_modify 変更時のチェックの場合はTRUEを設定する
     * @return boolean 重複していなければTRUE、重複している場合FALSEを返す
     */
    public function checkDuplicateUserId($user_id, $is_modify=false) 
    {
        $db = $this->getAdapter();
        
        $select = $db->select()
                ->from('m_user', 'count(*)')
                ->where('user_id=?', $user_id)
        ;
        $this->_logger->info($select->__toString());
        
        $result = $db->fetchOne($select);
        
        if ($is_modify == false && $result > 0) {
            return false;
        } elseif ($is_modify == true && $result > 1) {
            return false;
        }
        return true;
    }
    
    /**
     * メールアドレスの重複をチェックする
     * 
     * @access public
     * @param string $mailaddress
     * @param boolean $is_modify 変更時のチェックの場合はTRUEを設定する
     * @return boolean 重複していなければTRUE、重複している場合FALSEを返す
     */
    public function checkDuplicateMailaddress($mailaddress, $is_modify=false) 
    {
        $db = $this->getAdapter();
        
        $select = $db->select()
                ->from('m_user', 'count(*)')
                ->where('mailaddress=?', $mailaddress)
        ;
        $this->_logger->info($select->__toString());
        
        $result = $db->fetchOne($select);
        
        if ($is_modify == false && $result > 0) {
            return false;
        } elseif ($is_modify == true && $result > 1) {
            return false;
        }
        return true;
    }
    
    /**
     * ユーザをDBに登録する
     * 
     * @access public
     * @param mixed $values 登録するユーザ情報
     */
    public function createUser($values)
    {
//        parent::create($values);
        $db = $this->getAdapter();
        $values['create_datetime'] = date("Y-m-d H:i:s", time());
        $db->insert('m_user', $values);
    }
    
    /**
     * ユーザ情報を更新する
     * 
     * @access public 
     * @param integer $user_id
     * @param array $values
     */
    public function updateUser($user_id, $values)
    {
        $db = $this->getAdapter();
        $where = $db->quoteInto('id=?', $user_id);
        $values['update_datetime'] = date("Y-m-d H:i:s", time());
        $db->update('m_user', $values, $where);
    }
    
    /**
     * ユーザ情報を削除する
     * 
     */
    public function deleteUser($user_id)
    {
        $db = $this->getAdapter();
        try {
            $db->beginTransaction();
            
            //ユーザ情報は論理削除
            $where = $db->quoteInto('id=?', $user_id);
            $values = array(
                    'user_id'         => 'DELETED' . $user_id,
                    'mailaddress'     => 'DELETED' . $user_id,
                    'deleted'         => '1',
                    'delete_datetime' => date("Y-m-d H:i:s", time()),
            );
            $db->update('m_user', $values, $where);
            
            //権限情報の削除
            $db->delete('t_acl_user', $db->quoteInto('user_id=?', $user_id));
            
            $db->commit();
        } catch (Exception $e) {
            $db->rollBack();
            $this->_logger->err($e->getMessage());
            throw $e;
        }
    }
    
    /**
     * ユーザ情報をKEY=ID,VALUE=NAMEの連想配列で取得する
     * 
     */
    public function getUserOptions()
    {
        $db = $this->getAdapter();
        $select = $db->select()
                ->from('m_user', array('id', 'name'))
                ->where('enable=1')
                ->order('id asc')
        ;
        return $options = $db->fetchPairs($select);
    }
}