﻿using System;
using System.Collections.Generic;
using System.Text;
using Editor.Parser.IronPythonParser;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Xml;

namespace Editor.Parser
{
    class PythonDictionary : IIntelisenceDictionary, INameResolve
    {
        IronPython.Hosting.PythonEngine _engine;

        internal PythonDictionary(IronPython.Hosting.PythonEngine engine)
        {
            _engine = engine;
        }


        #region IIntelisenceDictionary メンバ

        public string[] GetChild(IParseResult result)
        {
            var parseResult = result as ParseResult;
            if (parseResult == null) { return new string[0]; }

            PyTypeInfo typeInfo = parseResult.IntelisenceExpression.GetTypeInfo(this);

            if (typeInfo.KindOfType == KindOfType.DotNetClass)
            {
                return GetStaticMembers(typeInfo.DotNetType);
            }
            else if (typeInfo.KindOfType == KindOfType.DotNetEvent)
            {
                return GetInstanceMembers(typeInfo.DotNetType);
            }
            else if (typeInfo.KindOfType == KindOfType.DotNetMethod)
            {
                return GetInstanceMembers(typeInfo.DotNetType);
            }
            else if (typeInfo.KindOfType == KindOfType.DotNetNamaSpace)
            {
                return GetPythonDir(typeInfo.ExampleText);
            }
            else if (typeInfo.KindOfType == KindOfType.DotNetVariable)
            {
                return GetInstanceMembers(typeInfo.DotNetType);
            }
            else if (typeInfo.KindOfType == KindOfType.PythonClass)
            {
                return GetPythonDir(typeInfo.ExampleText);
            }
            else if (typeInfo.KindOfType == KindOfType.PythonMethod)
            {
                return new string[0];
            }
            else if (typeInfo.KindOfType == KindOfType.PythonNameSpace)
            {
                return GetPythonDir(typeInfo.ExampleText);
            }
            else if (typeInfo.KindOfType == KindOfType.PythonVariable)
            {
                return GetPythonDir(typeInfo.ExampleText);
            }
            else if (typeInfo.KindOfType == KindOfType.Unknown)
            {
                List<string> list = new List<string>();
                list.AddRange(GetPythonDir(""));
                list.AddRange(GetReserved());
                list.Sort();
                return list.ToArray();
            }
            return new string[0];
        }

        private string[] GetReserved()
        {
            return new string[] { "and", "as", "assert", "break", "class", "continue", "def", "del", "elif", "else", "except", "exec", "finally", "for", "from", "global", "if", "import", "in", "is", "lambda", "not", "or", "pass", "print", "raise", "return", "try", "while", "with", "yield" };
        }

        private string[] GetInstanceMembers(Type type)
        {
            BindingFlags bindingFlags = BindingFlags.Instance | BindingFlags.Public;
            return GetMembers(type, bindingFlags);
        }

        private string[] GetStaticMembers(Type type)
        {
            BindingFlags bindingFlags = BindingFlags.Static | BindingFlags.Public;
            return GetMembers(type, bindingFlags);
        }

        private static string[] GetMembers(Type type, BindingFlags bindingFlags)
        {
            var memberInfos = type.GetMembers(bindingFlags);
            List<string> list = new List<string>();
            foreach (var member in memberInfos)
            {
                if (member.Name.StartsWith("get_") == false 
                    && member.Name.StartsWith("set_") == false 
                    && member.Name.StartsWith(".ctor") == false)
                {
                    if (list.Contains(member.Name) == false)
                    {
                        list.Add(member.Name);
                    }
                }
            }
            list.Sort();
            return list.ToArray();
        }

        public HelpContentCollection GetHelp(IParseResult result)
        {
            var parseResult = result as ParseResult;
            if (parseResult == null) { return new HelpContentCollection(); }

            PyTypeInfo typeInfo = parseResult.HelpExpression.GetTypeInfo(this);
            
            if (typeInfo.KindOfType == KindOfType.DotNetMethod)
            {
                return GetDotNetHelp(typeInfo.MemberInfo as MethodInfo);
            }
            else if (typeInfo.KindOfType == KindOfType.DotNetClass)
            {
                return GetDotNetConstractorHelp(typeInfo.DotNetType);
            }
            else if (typeInfo.KindOfType == KindOfType.PythonMethod)
            {
                return new HelpContentCollection();
            }
            return new HelpContentCollection();
        }

        private HelpContentCollection GetDotNetConstractorHelp(Type type)
        {
            if (type == null)
            {
                return new HelpContentCollection();
            }
            SetCacheIfNonExists(type);
            ConstructorInfo[] constructors = type.GetConstructors();
            
            var result = new HelpContentCollection();
            foreach (var constructor in constructors)
            {
                var help = new HelpContent(constructor, type, _helpCache);
                result.Add(help);
            }
            return result;
        }

        private void SetCacheIfNonExists(Type type)
        {
            if (_helpCache.Contains(type.Assembly) == false)
            {
                var xmlFilePath = GetXmlFilePath(type);
                if (xmlFilePath != "")
                {
                    _helpCache.AddXml(type.Assembly, xmlFilePath);
                }
            }
        }

        XmlHelpCache _helpCache = new XmlHelpCache();
        Regex regex = new Regex("[a-zA-Z0-9]+", RegexOptions.Compiled);
        private HelpContentCollection GetDotNetHelp(MethodInfo method)
        {
            if (method == null)
            {
                return new HelpContentCollection();
            }
            SetCacheIfNonExists(method.DeclaringType);

            BindingFlags flg ;
            if (method.IsStatic)
            {
                flg = BindingFlags.Static | BindingFlags.Public;
            }
            else
            {
                flg = BindingFlags.Instance | BindingFlags.Public;
            }
            MemberInfo[] members = method.DeclaringType.GetMember(method.Name, flg);
            var result = new HelpContentCollection();
            foreach (var member in members)
            {
                var methodTmp = member as MethodInfo;
                if (methodTmp != null)
                {
                    var help = new HelpContent(methodTmp, _helpCache);
                    result.Add(help);
                }
            }
            return result;
        }


        private string GetXmlFilePath(Type type)
        {
            System.Globalization.CultureInfo ci = System.Threading.Thread.CurrentThread.CurrentCulture;

            string location = type.Assembly.Location;
            string dir = System.IO.Path.GetDirectoryName(location);
            string xmlFileName = System.IO.Path.GetFileNameWithoutExtension(location) + ".xml";

            string path = GetXmlFilePathFromDirAndFileName(ci, dir, xmlFileName);
            if (path == "")
            {
                dir = dir.Replace("Framework64", "Framework");
                path = GetXmlFilePathFromDirAndFileName(ci, dir, xmlFileName);
            }
            return path;
        }

        private string GetXmlFilePathFromDirAndFileName(System.Globalization.CultureInfo ci, string dir, string xmlFileName)
        {
            string xmlFilePath;
            xmlFilePath = System.IO.Path.Combine(System.IO.Path.Combine(dir, ci.Name), xmlFileName);
            if (System.IO.File.Exists(xmlFilePath))
            {
                return xmlFilePath;
            }

            MatchCollection matches = regex.Matches(ci.Name);
            foreach (Match match in matches)
            {
                xmlFilePath = System.IO.Path.Combine(System.IO.Path.Combine(dir, match.Value), xmlFileName);
                if (System.IO.File.Exists(xmlFilePath))
                {
                    return xmlFilePath;
                }
            }
            xmlFilePath = System.IO.Path.Combine(dir, xmlFileName);
            if (System.IO.File.Exists(xmlFilePath))
            {
                return xmlFilePath;
            }

            return "";
        }

        #endregion

        #region INameResolve メンバ

        public PyTypeInfo GetIdentifier(string name)
        {
            object obj = null;
            try
            {
                obj = _engine.Evaluate(name);
            }
            catch (IronPython.Runtime.Exceptions.PythonNameErrorException)
            {
            }
            catch (MissingMemberException)
            {
            }
            catch (IronPython.Runtime.Exceptions.PythonSyntaxErrorException)
            {
            }

            if (obj == null)
            {
                return new PyTypeInfo(null, name, KindOfType.Unknown);
            }

            if (obj.GetType() == typeof(IronPython.Runtime.Calls.BuiltinFunction))
            {
                //IronPythonの関数
                return new PyTypeInfo(null, name, IronPythonParser.KindOfType.PythonMethod);
            }
            else if (obj.GetType() == typeof(IronPython.Runtime.PythonModule))
            {
                //ネームスペース
                return new PyTypeInfo(null, name, IronPythonParser.KindOfType.DotNetNamaSpace);
            }
            else if (obj is IronPython.Runtime.Types.DynamicType)
            {
                //クラス
                var dynamicType = (IronPython.Runtime.Types.DynamicType)obj;
                return new PyTypeInfo(dynamicType.type, name, KindOfType.DotNetClass);
            }
            else if (obj.GetType() == typeof(IronPython.Runtime.Calls.BoundBuiltinFunction))
            {
                return new PyTypeInfo(null, name, IronPythonParser.KindOfType.PythonMethod);
            }
            else if (obj.GetType().Assembly == typeof(IronPython.Hosting.PythonEngine).Assembly)
            {
                return new PyTypeInfo(null, name, KindOfType.PythonVariable, null);
            }
            else
            {
                //.NETのインスタンス
                var dotNetType = obj.GetType();
                return new PyTypeInfo(dotNetType, name, KindOfType.DotNetVariable);
            }
        }

        #endregion


        public string[] GetPythonDir(string cls)
        {
            object result;
            string resultStr;
            try
            {
                result = _engine.Evaluate(string.Format("dir({0})", cls));
            }
            catch
            {
                return new string[0];
            }
            resultStr = result.ToString();

            Regex _regex = new Regex("'([a-zA-Z0-9_]+)'");
            MatchCollection parentClassMatches = _regex.Matches(resultStr);
            return CreateArray(parentClassMatches, cls == "");
        }

        private string[] CreateArray(MatchCollection matches, bool isAddReservedAdd)
        {
            List<string> list = new List<string>();
            for (int i = 0; i < matches.Count; i++)
            {
                list.Add(matches[i].Groups[1].Value);
            }
            list.Sort();
            if (isAddReservedAdd == false)
            {
                return list.ToArray();
            }
            //for (int i = 0; i < _reserveWords.Length; i++)
            //{
            //    list.Add(_reserveWords[i]);
            //}
            return list.ToArray();
        }
    }
}
